% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bayesopt_smsego.R
\name{mlr_loop_functions_smsego}
\alias{mlr_loop_functions_smsego}
\alias{bayesopt_smsego}
\title{Sequential Multi-Objective Bayesian Optimization via SMS-EGO}
\usage{
bayesopt_smsego(
  instance,
  init_design_size = NULL,
  surrogate,
  acq_function,
  acq_optimizer,
  random_interleave_iter = 0L
)
}
\arguments{
\item{instance}{(\link[bbotk:OptimInstanceMultiCrit]{bbotk::OptimInstanceMultiCrit})\cr
The \link[bbotk:OptimInstanceMultiCrit]{bbotk::OptimInstanceMultiCrit} to be optimized.}

\item{init_design_size}{(\code{NULL} | \code{integer(1)})\cr
Size of the initial design.
If \code{NULL} and the \link[bbotk:Archive]{bbotk::Archive} contains no evaluations, \code{4 * d} is used with \code{d} being the
dimensionality of the search space.
Points are drawn uniformly at random.}

\item{surrogate}{(\link{SurrogateLearnerCollection})\cr
\link{SurrogateLearnerCollection} to be used as a surrogate.}

\item{acq_function}{(\link{mlr_acqfunctions_smsego})\cr
\link{mlr_acqfunctions_smsego} to be used as acquisition function.}

\item{acq_optimizer}{(\link{AcqOptimizer})\cr
\link{AcqOptimizer} to be used as acquisition function optimizer.}

\item{random_interleave_iter}{(\code{integer(1)})\cr
Every \code{random_interleave_iter} iteration (starting after the initial design), a point is
sampled uniformly at random and evaluated (instead of a model based proposal).
For example, if \code{random_interleave_iter = 2}, random interleaving is performed in the second,
fourth, sixth, ... iteration.
Default is \code{0}, i.e., no random interleaving is performed at all.}
}
\value{
invisible(instance)\cr
The original instance is modified in-place and returned invisible.
}
\description{
Loop function for sequential multi-objective Bayesian Optimization via SMS-EGO.
Normally used inside an \link{OptimizerMbo}.

In each iteration after the initial design, the surrogate and acquisition function (\link{mlr_acqfunctions_smsego}) are
updated and the next candidate is chosen based on optimizing the acquisition function.
}
\note{
\itemize{
\item The \code{acq_function$surrogate}, even if already populated, will always be overwritten by the \code{surrogate}.
\item The \code{acq_optimizer$acq_function}, even if already populated, will always be overwritten by \code{acq_function}.
\item The \code{surrogate$archive}, even if already populated, will always be overwritten by the \link[bbotk:Archive]{bbotk::Archive} of the \link[bbotk:OptimInstanceMultiCrit]{bbotk::OptimInstanceMultiCrit}.
\item Due to the iterative computation of the epsilon within the \link{mlr_acqfunctions_smsego}, requires the \link[bbotk:Terminator]{bbotk::Terminator} of
the \link[bbotk:OptimInstanceMultiCrit]{bbotk::OptimInstanceMultiCrit} to be a \link[bbotk:mlr_terminators_evals]{bbotk::TerminatorEvals}.
}
}
\examples{
\donttest{
if (requireNamespace("mlr3learners") &
    requireNamespace("DiceKriging") &
    requireNamespace("rgenoud")) {

  library(bbotk)
  library(paradox)
  library(mlr3learners)

  fun = function(xs) {
    list(y1 = xs$x^2, y2 = (xs$x - 2) ^ 2)
  }
  domain = ps(x = p_dbl(lower = -10, upper = 10))
  codomain = ps(y1 = p_dbl(tags = "minimize"), y2 = p_dbl(tags = "minimize"))
  objective = ObjectiveRFun$new(fun = fun, domain = domain, codomain = codomain)

  instance = OptimInstanceMultiCrit$new(
    objective = objective,
    terminator = trm("evals", n_evals = 5))

  surrogate = default_surrogate(instance)

  acq_function = acqf("smsego")

  acq_optimizer = acqo(
    optimizer = opt("random_search", batch_size = 100),
    terminator = trm("evals", n_evals = 100))

  optimizer = opt("mbo",
    loop_function = bayesopt_smsego,
    surrogate = surrogate,
    acq_function = acq_function,
    acq_optimizer = acq_optimizer)

  optimizer$optimize(instance)
}
}
}
\references{
\itemize{
\item Beume N, Naujoks B, Emmerich M (2007).
\dQuote{SMS-EMOA: Multiobjective selection based on dominated hypervolume.}
\emph{European Journal of Operational Research}, \bold{181}(3), 1653--1669.
\item Ponweiser, Wolfgang, Wagner, Tobias, Biermann, Dirk, Vincze, Markus (2008).
\dQuote{Multiobjective Optimization on a Limited Budget of Evaluations Using Model-Assisted S-Metric Selection.}
In \emph{Proceedings of the 10th International Conference on Parallel Problem Solving from Nature}, 784--794.
}
}
\seealso{
Other Loop Function: 
\code{\link{loop_function}},
\code{\link{mlr_loop_functions_ego}},
\code{\link{mlr_loop_functions_mpcl}},
\code{\link{mlr_loop_functions_parego}},
\code{\link{mlr_loop_functions}}
}
\concept{Loop Function}
