% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/model_grid.R, R/train_model_grid.R
\name{model_grid}
\alias{model_grid}
\alias{train.model_grid}
\title{Pre-allocate an empty model grid}
\usage{
model_grid()

\method{train}{model_grid}(x, ..., train_all = FALSE, resample_seed = 123)
}
\arguments{
\item{x}{\code{model_grid}}

\item{...}{other arguments passed to methods
across models in order to obtain a fair (and reproducible) comparison of
the models. If set to NULL, seed will not be set (NOT advised).}

\item{train_all}{\code{logical} if set to TRUE, all models will be trained.
If set to FALSE, only models, for which no fit already exists, will be
trained.}

\item{resample_seed}{\code{integer} is used to create identical resamples}
}
\value{
\code{model_grid}

\code{model_grid} equipped with fitted models.
}
\description{
Constructor function that pre-allocates an empty model grid.
The model grid  makes it easy to create, manage and train multiple caret models.
Define the settings that by default are to be shared by all of the models in
the model grid with \code{\link{share_settings}}. Add the individual
specifications for the models you want to investigate with \code{\link{add_model}}.
Train all of the models in the model grid with \code{\link{train}}.

The S3 method of the train function for the 'model_grid' class consolidates
all model (and training) configurations from a model grid and
trains them with the train function from the caret package.
}
\examples{
# Pre-allocate an empty model grid.
model_grid()

\donttest{
library(caret)
library(magrittr)
library(dplyr)
data(GermanCredit)

# Create model grid with two different Random Forest models.
mg <-
  model_grid() \%>\%
  share_settings(
    y = GermanCredit[["Class"]],
    x = GermanCredit \%>\% select(-Class),
    metric = "ROC",
    trControl = trainControl(
      method = "cv",
      number = 2,
      summaryFunction = twoClassSummary,
      classProbs = TRUE
    )
  ) \%>\%
  add_model(
    model_name = "RF",
    method = "rf",
    tuneLength = 3
    ) \%>\%
  add_model(
    model_name = "RF NZV",
    method = "rf",
    preProc = "nzv",
    tuneGrid = data.frame(mtry = c(2, 10))
    )

# Train all model configurations in model grid.
train(mg)
}
}
\seealso{
\code{\link{add_model}} for how to add a model to a model grid,
 \code{\link{edit_model}} for how to edit an existing model within a model grid,
 \code{\link{share_settings}} for how to define the shared settings of models
 within a model grid, \code{\link{consolidate_model}} for how to consolidate
 the shared settings of a model grid and the individual settings of a given
 model into one complete caret model configuration and
 \code{\link{remove_model}} for how to remove a model from a model grid.
}
