#' Well-formedness, Myhill's property, and/or moment of symmetry
#'
#' Tests whether a scale has the property of "well-formedness" or "moment of symmetry."
#'
#' The three concepts of "well-formedness," "Myhill's property," and "moment of symmetry"
#' refer to nearly the same scalar property, generalizing one of the most important features
#' of the familiar diatonic scale. See Clough, Engebretsen, and Kochavi (1999, 77)
#' <doi:10.2307/745921> for a useful discussion of their relationships. In short,
#' except for a few edge cases, a scale possesses these properties if it is generated by copies
#' of a single interval (as the Pythagorean diatonic is generated by the ratio 3:2) and all copies 
#' of the generator belong to the same generic interval (as the 3:2 generator of the diatonic
#' always corresponds to a "fifth" within the scale). Such a structure typically means that
#' all generic intervals come in 2 distinct sizes, which is the definition of "Myhill's property."
#' An exception occurs if the generator manages to produce a perfectly even scale, e.g. when
#' the whole tone scale is generated by 6 copies of `1/6` of the octave. Such a scale lacks
#' Myhill's property and Carey & Clampitt (1989, 200) <doi:10.2307/745935> call such cases
#' "degenerate well-formed." Instead of Myhill's property, such scales have only 1 specific value
#' in each [intervalspectrum()].
#'
#' Clough, Engebretsen, and Kochavi define a related concept, distributionally even scales,
#' which include the hexatonic and octatonic scales (Forte sc6-20 and sc8-28). Such scales are in
#' some sense halfway between "degenerate" and "non-degenerate well-formed" because some of their
#' interval spectra have 1 element while others have 2. From another perspective, distributionally
#' even scales are non-degenerate well formed with a period smaller than the octave (e.g. as the
#' hexatonic scales 1-3 step pattern repeats every third of an octave).
#'
#' The term "moment of symmetry" refers to the non-degenerate well-formed scales and was coined by
#' Erv Wilson 1975 (cited in Clough, Engebretsen, and Kochavi). It tends to be more widely used in 
#' microtonal music theory, e.g. <https://en.xen.wiki/w/MOS_scale>.
#'
#' Scales with this property have considerably interesting voice-leading properties and are 
#' some of the most important landmarks in the geometry of MCT. See "Modal Color Theory," pp. 14, 17,
#' 29, 33-34, and 36-37. A substantial portion of MCT amounts to an attempt to generalize ideas developed
#' for MOS/NDWF scales to all scale structures.
#'
#' @inheritParams tnprime
#' @inheritParams fpunique
#' @param setword A vector representing the ranked step sizes of a scale (e.g.
#'   `c(2, 2, 1, 2, 2, 2, 1)` for the diatonic). The distinct values of the `setword`
#'   should be consecutive integers. If you want to test a step word instead of 
#'   a list of pitch classes, `set` must be entered as `NULL`.
#' @param allow_de Should the function test for degenerate well-formed and distributionally even scales too?
#'   Defaults to `FALSE`.
#' @returns Boolean answering "Is the scale MOS (with equivalence interval equal to
#'   the period)?" (if allow_de=FALSE) or "Is the scale well-formed
#'   in any sense?" (if allow_de=TRUE).
#' @examples
#' iswellformed(sc(7, 35))
#' iswellformed(c(0, 2, 4, 6))
#' iswellformed(c(0, 1, 6, 7))
#' iswellformed(c(0, 1, 6, 7), allow_de=TRUE)
#' iswellformed(NULL, setword=c(2, 2, 1, 2, 1, 2, 1))
#' @export
iswellformed <- function(set, setword=NULL, allow_de=FALSE, edo=12, rounder=10) {
  if (is.null(set)) { 
    set <- realize_setword(setword, edo) 
  }
  if (length(set) < 2) { 
    return(as.logical(allow_de)) 
  }

  speccount <- spectrumcount(set, edo, rounder)
  uniques <- unique(speccount)
  if (toString(uniques)=="2") { 
    return(TRUE) 
  }
  if (toString(uniques)=="1") { 
    return(as.logical(allow_de)) 
  }
  if (toString(sort(uniques))=="1, 2") { 
    return(as.logical(allow_de)) 
  }

  FALSE
}

#' Equivalence two step letters as in the definition of PWF
#'
#' Clampitt's definition of pairwise well formed scales requires that
#' every equivalencing of two letters in the PWF word results int 
#' a well-formed word. This function does that substitution.
#'
#' @param setword A numeric vector: a step word of a scale to test
#' @param lowerbound Integer: the smallest entry in `setword` to equivalence
#' @param windowsize Integer: how many letters above `lowerbound`
#'   (inclusive) are included in the equivalence?
#'
#' @returns A step word (numeric vector) with only two letters.
#'
#' @noRd
equivocate <- function(setword, lowerbound, windowsize) {
  highest <- max(setword)
  toMatch <- lowerbound:(lowerbound+(windowsize-1))
  toMatch <- unique(((toMatch-1)%%highest)+1)
  replacement_positions <- which(setword %in% toMatch)
  result <- replace(setword, replacement_positions, 1)
  result <- replace(result, -replacement_positions, 2)
  result
}

#' Is a scale n-wise well formed?
#'
#' Tests whether a scale has a generalized type of well formedness (pairwise or
#' n-wise well formedness).
#'
#' David Clampitt's 1997 dissertation ("Pairwise Well-Formed Scales: 
#' Structural and Transformational Properties," SUNY Buffalo) offers
#' a generalization of the notion of well-formedness from 1-dimensional
#' structures with a single generator to 2-dimensional structures that 
#' mediate between two well-formed scales. Ongoing research suggests that
#' this can be extended further to "n-wise" or "general" well-formedness,
#' though n-wise well-formed scales are increasingly rare as n grows larger.
#'
#' @inheritParams iswellformed
#' @returns Boolean: is the set n-wise well formed?
#'
#' @examples
#' meantone_diatonic <- c(0, 2, 4, 5, 7, 9, 11)
#' just_diatonic <- j(dia)
#' some_weird_thing <- convert(c(0, 1, 3, 6, 8, 12, 14), 17, 12)
#' example_scales <- cbind(meantone_diatonic, just_diatonic, some_weird_thing)
#'
#' apply(example_scales, 2, howfree)
#' apply(example_scales, 2, isgwf)
#'
#' @export
isgwf <- function(set, setword=NULL, allow_de=FALSE, edo=12, rounder=10) {
  if (is.null(setword)) { 
    setword <- asword(set, edo, rounder) 
  }
  if (anyNA(setword)) { 
    return(FALSE)
  }

  highest <- max(setword)
  equiv_parameters <- expand.grid(1:highest, 1:(highest-1))

  equiv_wrap <- function(params, setword) equivocate(setword, params[1], params[2])
  reduced_words <- apply(equiv_parameters, 1, equiv_wrap, setword=setword)

  iswf_wrap <- function(setword, allow_de, edo, rounder)  {
    iswellformed(NULL, setword, allow_de, edo, rounder)
  }

  tests <- apply(reduced_words,2, iswf_wrap, allow_de=allow_de, edo=edo, rounder=rounder)

  as.logical(prod(tests))
}
