% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/PlotPattern.R
\name{PlotPattern}
\alias{PlotPattern}
\title{Plot the Transcriptome Age Index or Transcriptome Divergence Index}
\usage{
PlotPattern(
  ExpressionSet,
  TestStatistic = "FlatLineTest",
  modules = NULL,
  permutations = 1000,
  lillie.test = FALSE,
  digits.ylab = 4,
  p.value = TRUE,
  shaded.area = FALSE,
  y.ticks = 4,
  custom.perm.matrix = NULL,
  ...
)
}
\arguments{
\item{ExpressionSet}{a standard PhyloExpressionSet or DivergenceExpressionSet object.}

\item{TestStatistic}{a string defining the type of test statistics to be used to quantify the statistical significance the present phylotranscriptomics pattern.
Possible values can be: \code{TestStatistic} = \code{"FlatLineTest"} : Statistical test for the deviation from a flat line.
\code{TestStatistic} = \code{"ReductiveHourglassTest"} : Statistical test for the existence of a hourglass shape (high-low-high pattern).
\code{TestStatistic} = \code{"EarlyConservationTest"} : Statistical test for the existence of a earlyconservation pattern (low-high-high pattern).}

\item{modules}{a list storing three elements for the \code{\link{ReductiveHourglassTest}} or \code{\link{EarlyConservationTest}}: early, mid, and late. 
Each element expects a numeric vector specifying the developmental stages 
or experiments that correspond to each module. For example, 
\code{module} = \code{list(early = 1:2, mid = 3:5, late = 6:7)} devides a dataset storing seven developmental stages into 3 modules.}

\item{permutations}{a numeric value specifying the number of permutations to be performed for the \code{\link{FlatLineTest}}, \code{\link{EarlyConservationTest}} or \code{\link{ReductiveHourglassTest}}.}

\item{lillie.test}{a boolean value specifying whether the Lilliefors Kolmogorov-Smirnov Test shall be performed.}

\item{digits.ylab}{a numeric value specifying the number of digits shown for the TAI or TDI values on the y-axis.}

\item{p.value}{a boolean value specifying whether the p-value of the test statistic shall be printed within the plot area.}

\item{shaded.area}{a boolean value specifying whether a shaded area shall 
be drawn for the developmental stages defined to be the presumptive phylotypic period.}

\item{y.ticks}{a numeric value specifying the number of ticks to be drawn on the y-axis.}

\item{custom.perm.matrix}{a custom \code{\link{bootMatrix}} (permutation matrix) to perform the underlying test statistic visualized by \code{PlotPattern}. Default is \code{custom.perm.matrix = NULL}.}

\item{\dots}{default plot parameters.}
}
\value{
a plot visualizing the phylotranscriptomic pattern of a given PhyloExpressionSet or DivergenceExpressionSet object.

The corresponding \emph{p-value} of the test statistic is named as follows:

\code{p_flt} = p-value of the corresponding \code{\link{FlatLineTest}}

\code{p_rht} = p-value of the corresponding \code{\link{ReductiveHourglassTest}}

\code{p_ect} = p-value of the corresponding \code{\link{EarlyConservationTest}}
}
\description{
Function to plot the TAI or TDI of a given PhyloExpressionSet or DivergenceExpressionSet object.
This function plot the \code{\link{TAI}} or \code{\link{TDI}} of a given PhyloExpressionSet or DivergenceExpressionSet object.
}
\details{
This function computes a permutation test quantifying the statistical significance of the prensent phylotranscriptomics pattern. 
The user can choose between the \code{\link{FlatLineTest}}, \code{\link{ReductiveHourglassTest}}, or \code{\link{EarlyConservationTest}}. 
The \code{\link{FlatLineTest}} tests for any significant deviation from a flat line. 
Each period or stage that significantly deviates from a flat line, might be governed by stronger selective pressure (in terms of natural selection) compared to other stages or periods of development.
The \code{\link{ReductiveHourglassTest}} specificly tests for the statistical significance of a molecular hourglass pattern (high-low-high pattern) with prior biological knowlegde.
The corresponding p-value that is printed within the plot (by default) specifies the statistical significance of the chosen test statistic.

The \code{\link{EarlyConservationTest}} specificly tests for the statistical significance of a low-high-high pattern (monotonically increasing function)
with prior biological knowlegde.
The corresponding p-value that is printed within the plot (by default) specifies the statistical significance of the chosen test statistic.

The x-axis denotes the developmental series (time course / experiments / ontogeny) of the input ExpressionSet and the y-axis 
denotes the corresponding mean transcriptome age value (\code{\link{TAI}} or \code{\link{TDI}}) of the given ExpressionSet. 

Furthermore, the grey lines above and below the actual phylotranscriptomics pattern denotes the standard deviations of \code{\link{TAI}} or \code{\link{TDI}} 
values that have been estimated from the \code{\link{bootMatrix}}.
A low mean transcriptome age value denotes an evolutionary older transcriptome being active during the corresponding periods, 
whereas a high mean transcriptome age value denotes an evolutionary younger transcriptome being active during the corresponding periods.
For mean transcriptome divergence, a low mean transcriptome divergence value denotes a more conserved transcriptome 
being active (between two species), whereas a high mean transcriptome divergence value denotes a more divergent transcriptome 
being active (between two species) - in terms of protein-sequence substitution rates.

This function is useful to fastly plot the \code{\link{TAI}} or \code{\link{TDI}} 
profile of a given PhyloExpressionSet or DivergenceExpressionSet object and 
the statistical significance of the corresponding pattern.
Internally the function calls several graphics functions, 
such as \code{\link{plot}}, \code{\link{axis}}, and \code{\link{legend}}. 
For the ellipsis argument \code{\link{...}} all graphics specific arguments can be defined. 
Internally the function specific arguments for e.g. \code{\link{plot}}, \code{\link{axis}}, 
and \code{\link{legend}} will be detected and are passed to the corresponding function.

Hence, when calling the function \code{PlotPattern}, one can specify arguments 
for \code{\link{plot}} and \code{\link{axis}} and \code{\link{legend}} 
as \code{\link{...}}. 

In case prior biological knowledge is present for a specific period of development,
the \code{shaded.area} argument can be set to \code{TRUE} and the function will use
the values stored in the \code{mid} argument to draw a shaded area within the corresponding period of development.
}
\examples{

# load PhyloExpressionSet
data(PhyloExpressionSetExample)

# only visualize the TAI profile without any test statistics...
# this is equavalent to performing: plot(TAI(PhyloExpressionSetExample), type = "l", lwd = 6)
PlotPattern(ExpressionSet = PhyloExpressionSetExample,
            TestStatistic = NULL,
            type          = "l",
            xlab          = "Ontogeny",
            ylab          = "TAI",
            lwd           = 9)

# the simplest example of plotting the TAI profile of a given PhyloExpressionSet:
# In this case (default) the FlatLineTest will be performed to quantify
# the statistical significance of the present TAI pattern and will be drawn as 'p = ... '
# in the plot

PlotPattern(ExpressionSet = PhyloExpressionSetExample, 
            TestStatistic = "FlatLineTest",
            permutations  = 100, 
            type          = "l", 
            xlab          = "Ontogeny", 
            ylab          = "TAI", 
            lwd           = 9)

# an example performing the ReductiveHourglassTest and printing the p-value
# and shaded area of the presumptive phylotypic period into the plot
# Here the 'p = ...' denotes the p-value that is returned by the ReductiveHourglassTest

PlotPattern(
            ExpressionSet = PhyloExpressionSetExample,
            TestStatistic = "ReductiveHourglassTest",
            modules       = list(early = 1:2,mid = 3:5,late = 6:7), 
            permutations  = 100, 
            p.value       = TRUE, 
            shaded.area   = TRUE, 
            xlab          = "Ontogeny", 
            ylab          = "TAI", 
            type          = "l", 
            lwd           = 9)

# testing for early conservation model 
PlotPattern( ExpressionSet = PhyloExpressionSetExample,
             TestStatistic = "EarlyConservationTest",
            modules        = list(early = 1:2,mid = 3:5,late = 6:7), 
            permutations   = 100,
            p.value        = TRUE, 
            shaded.area    = TRUE, 
            xlab           = "Ontogeny", 
            ylab           = "TAI", 
            type           = "l", 
            lwd            = 9)
            

# use your own permutation matrix
custom_perm_matrix <- bootMatrix(PhyloExpressionSetExample,100)

PlotPattern(ExpressionSet      = PhyloExpressionSetExample, 
            TestStatistic      = "FlatLineTest",
            custom.perm.matrix = custom_perm_matrix, 
            type               = "l", 
            xlab               = "Ontogeny", 
            ylab               = "TAI", 
            lwd                = 9)


}
\references{
Domazet-Loso T and Tautz D. (2010). \emph{A phylogenetically based transcriptome age index mirrors ontogenetic divergence patterns}. Nature (468): 815-818.

Quint M et al. (2012). \emph{A transcriptomic hourglass in plant embryogenesis}. Nature (490): 98-101.

Drost HG et al. (2015) Mol Biol Evol. 32 (5): 1221-1231 doi:10.1093/molbev/msv012.
}
\seealso{
\code{\link{TAI}}, \code{\link{TDI}}, \code{\link{FlatLineTest}},
 \code{\link{ReductiveHourglassTest}}, \code{\link{EarlyConservationTest}},
 \code{\link{PlotSignature}}
}
\author{
Hajk-Georg Drost
}
