% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/get_sites.R
\name{get_sites}
\alias{get_sites}
\title{get_sites}
\usage{
get_sites(x = NA, ...)
}
\arguments{
\item{x}{Use a single integer or vector of integers representing
unique Neotoma site identifiers (siteids) to extract site information.}

\item{...}{accepted arguments, see details for more information.}
}
\value{
The function returns either a single item of class \code{"try-error"}
describing the reason for failure (either misdefined parameters or an error
from the Neotoma API), or a table of sites, with rows corresponding to the
number of individual sites returned by the Neotoma API.
Each "site" object contains 6 parameters that can be accessed as well:
siteid, sitename, location, altitude, description,
limited collection units information.
\itemize{
\item \code{loc} An \code{sf} object that describes site's location.
\item \code{description}
\item \code{collunits} limited information on collunits
}
}
\description{
The get_sites() function is a wrapper for the Neotoma \code{sites} API
endpoint.
The function takes parameters defined by the user and returns a list
of site information supplied by the Neotoma Paleoecological Database.
The user may define all or none of the possible fields.
}
\details{
A \code{site} object in Neotoma is a physical location at which one or more
collection units are located. Each collection unit may have one or more
datasets within it, defined by the dataset type. The \code{get_sites()}
function searches for each site within Neotoma that matches the query
parameters, and returns them as a \code{sites} object, a list of \code{site}
objects. The \code{get_sites()} command wraps the Neotoma API
(\href{https://api.neotomadb.org}{api.neotomadb.org}) call for \code{sites}.
The call itself uses a SQL query which accepts any one of the following
parameters:
\itemize{
\item \code{siteid}  The unique site ID (integer) in Neotoma. Can be passed as a
vector of site IDs.
\item \code{sitename}  The site name, or approximate match using the \% wildcard.
\item \code{database}  The constituent database for the record. See \code{get_table("constituentdatabases")}
\item \code{altmin}  The minimum altitude range for site elevation (in meters).
\item \code{altmax}  The maximum altitude range for site elevation (in meters).
\item \code{datasetid}  The unique dataset ID (integer) in Neotoma. Can be passed
as a vector of dataset IDs.
\item \code{datasettype}  Neotoma contains data for a number of datasettypes.
This returns a subset of data types. For a complete list of available
datasettypes, run \code{neotoma2::get_table('datasettypes')}
\item \code{doi}  The dataset DOI for a dataset contained within a site. Can be
passed as a vector of DOIs.
\item \code{gpid}  The geopolitical name or identifier containing a site. Can be
passed as a vector of names.
\item \code{keywords}  Keywords for samples within a set of sites. For example
"modern" indicates a sample within the record uses the keyword "modern".
\item \code{contacts}  Contact names or IDs associated with a site.
\item \code{ageyoung}  A minimum spanning age for the record, in years before
radiocarbon present (1950).
\item \code{ageold}  A maximum spanning age for the record, in years before
radiocarbon present (1950).
\item \code{ageof}  An age which must be contained within the range of sample ages
for a site.
\item \code{taxa}  The names of taxa which must be present within samples in a
record.
\item \code{all_data}  The API only downloads the first 25 records of the query.
For the complete records, use \code{all_data=TRUE}
This call will then return a data object that contains site metadata for one
or more sites, along with limited metadata describing the collection units
and datasets located at that site.
}
}
\examples{
\donttest{
## Find all sites with a min altitude of 12m and a max altitude of 25m
## By default returns only 25 sites (default limit is 25):
sites_12to25 <- get_sites(altmin=12, altmax=25)
## Return all sites, using a minimum altitude of 2500m (returns >500 sites):
sites_2500 <- get_sites(altmin=2500, all_data = TRUE)
## To find all sites that contain the string "Alex\%"
alex_sites <- get_sites(sitename="Alex\%")
## To find sites in Brazil (again with default 25 records)
brazil <- '{"type": "Polygon",
"coordinates": [[
 [-73.125, -9.102096738726443],
 [-56.953125,-33.137551192346145],
 [-36.5625,-7.710991655433217],
 [-68.203125,13.923403897723347],
 [-73.125,-9.102096738726443]]]}'
brazil_sites <- get_sites(loc = brazil[1])

# Finding all sites with Liliaceae pollen in 1000 year bins:
lilysites <- c()
for (i in seq(0, 10000, by = 1000)) {
  lily <- get_sites(taxa=c("Liliaceae"),
                    ageyoung = i - 500,
                    ageold = i + 500,
                    all_data = TRUE)
  lilysites <- c(lilysites, length(lily))
}
plot(x = seq(0, 10000, by = 1000), y = lilysites, type = 'b')
}
}
\author{
Socorro Dominguez \email{s.dominguez@ht-data.com}
}
