% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dbinomLocal_normalPlateau.R
\name{dbinomLocal_normalPlateau}
\alias{dbinomLocal_normalPlateau}
\alias{rbinomLocal_normalPlateau}
\title{Local evaluation of a binomial SCR observation process}
\usage{
dbinomLocal_normalPlateau(
  x,
  detNums = -999,
  detIndices,
  size,
  p0 = -999,
  p0Traps,
  sigma,
  w = 2,
  s,
  trapCoords,
  localTrapsIndices,
  localTrapsNum,
  resizeFactor = 1,
  habitatGrid,
  indicator,
  lengthYCombined = 0,
  log = 0
)

rbinomLocal_normalPlateau(
  n = 1,
  detNums = -999,
  detIndices,
  size,
  p0 = -999,
  p0Traps,
  sigma,
  w = 2,
  s,
  trapCoords,
  localTrapsIndices,
  localTrapsNum,
  resizeFactor = 1,
  habitatGrid,
  indicator,
  lengthYCombined = 0
)
}
\arguments{
\item{x}{Vector of individual detection frequencies. This argument can be provided in two formats: (i) with the \emph{y} object as returned by \code{\link{getSparseY}}; (ii) with the
\emph{yCombined} object as returned by \code{\link{getSparseY}} Note that when the random generation functionality is used (rbinomLocal_normal), only the yCombined format can be used. 
The  \emph{yCombined}  object combines \emph{detNums}, \emph{x}, and \emph{detIndices} (in that order). When such consolidated 
representation of the detection data x is used, \emph{detIndices} and \emph{detNums} arguments should not be specified.}

\item{detNums}{Number of traps with at least one detection recorded in \emph{x}; from the \emph{detNums} object returned by the \code{\link{getSparseY}} function. 
This argument should not be specified when the \emph{yCombined} object (returned by \code{\link{getSparseY}}) is provided as \emph{x} and when detection data are simulated.}

\item{detIndices}{Vector of indices of traps where the detections in \emph{x} were recorded; from the \emph{detIndices} object returned by the \code{\link{getSparseY}} function. 
This argument should not be specified when x is provided as the  \emph{yCombined} object (returned by \code{\link{getSparseY}} ) and when detection data are simulated.}

\item{size}{Vector of the number of trials (zero or more) for each trap (\emph{trapCoords}).}

\item{p0}{Baseline detection probability (scalar) used in the half-normal detection function. For trap-specific baseline detection probabilities use argument \emph{p0Traps} (vector) instead.}

\item{p0Traps}{Vector of baseline detection probabilities for each trap used in the half-normal detection function. When \emph{p0Traps} is used, \emph{p0} should not be provided.}

\item{sigma}{Scale parameter of the half-normal detection function.}

\item{w}{Length of plateau of the half-normal plateau detection function.}

\item{s}{Individual activity center x- and y-coordinates scaled to the habitat (see (\code{\link{scaleCoordsToHabitatGrid}}).}

\item{trapCoords}{Matrix of x- and y-coordinates of all traps scaled to the habitat (see (\code{\link{scaleCoordsToHabitatGrid}}).}

\item{localTrapsIndices}{Matrix of indices of local traps around each habitat grid cell, as returned by the \code{\link{getLocalObjects}} function.}

\item{localTrapsNum}{Vector of numbers of local traps around all habitat grid cells, as returned by the \code{\link{getLocalObjects}} function.}

\item{resizeFactor}{Aggregation factor used in the \code{\link{getLocalObjects}} function to reduce the number of habitat grid cells to retrieve local traps for.}

\item{habitatGrid}{Matrix of local habitat grid cell indices, from \emph{habitatGrid} returned by the \code{\link{getLocalObjects}} function.}

\item{indicator}{Binary argument specifying whether the individual is available for detection (indicator = 1) or not (indicator = 0).}

\item{lengthYCombined}{The length of the x argument when the (\emph{yCombined}) format of the detection data is provided;  from the \emph{lengthYCombined} object returned by \code{\link{getSparseY}}}

\item{log}{Logical argument, specifying whether to return the log-probability of the distribution.}

\item{n}{Integer specifying the number of realizations to generate.  Only n = 1 is supported.}
}
\value{
The log-likelihood value associated with the vector of detections, given the location of the activity center (s),
 and the half-normal plateau detection function : \eqn{p = p0} when \emph{d < w} and \eqn{p = p0 * exp(-(d-w)^2 / \sigma^2)} when \emph{d >= w}.
}
\description{
The \code{dbinomLocal_normalPlateau} distribution is a NIMBLE custom distribution which can be used to model 
and simulate binomial observations (x) of a single individual over a set of traps defined by their coordinates \emph{trapCoords}
the distribution assumes that an individual’s detection probability at any trap follows a half-normal plateau function of the distance between 
the individual's activity center (s) and the trap location. With the half-normal plateau function, detection probability remains constant with value 
p0 for a plateau of width w before declining with scale sigma.
}
\details{
All coordinates (\code{s} and \code{trapCoords}) should be scaled to the habitat (see \code{\link{scaleCoordsToHabitatGrid}})



The \code{dbinomLocal_normalPlateau} distribution incorporates three features to increase computation efficiency (see Turek et al., 2021 <doi.org/10.1002/ecs2.3385>  for more details):
\enumerate{
\item A local evaluation of the detection probability calculation (see Milleret et al., 2019 <doi:10.1002/ece3.4751> for more details)
\item A sparse matrix representation (\emph{x}, \emph{detIndices} and \emph{detNums}) of the observation data to reduce the size of objects to be processed.
\item An indicator (\emph{indicator}) to shortcut calculations for individuals unavailable for detection.
}

The \code{dbinomLocal_normalPlateau} distribution requires x- and y- detector coordinates (\emph{trapCoords}) to be scaled to the habitat grid (\emph{habitatGrid}) using the (\code{\link{scaleCoordsToHabitatGrid}} function.)

When the aim is to simulate detection data: 
\enumerate{
\item \emph{x} should be provided using the \emph{yCombined} object as returned by \code{\link{getSparseY}}, 
\item arguments \emph{detIndices} and \emph{detNums} should not be provided, 
\item argument \emph{lengthYCombined} should be provided using the \emph{lengthYCombined} object as returned by  \code{\link{getSparseY}}.
}
}
\examples{
# A user friendly vignette is also available on github: 
# https://github.com/nimble-dev/nimbleSCR/blob/master/nimbleSCR/vignettes/
# Vignette name: Fit_with_dbinomLocal_normalPlateau_and_HomeRangeAreaComputation.rmd

# I. DATA SET UP 
coordsHabitatGridCenter <- matrix(c(0.5, 3.5,
                                    1.5, 3.5,
                                    2.5, 3.5,
                                    3.5, 3.5,
                                    0.5, 2.5,
                                    1.5, 2.5,
                                    2.5, 2.5,
                                    3.5, 2.5,
                                    0.5, 1.5,
                                    1.5, 1.5,
                                    2.5, 1.5,
                                    3.5, 1.5,
                                    0.5, 0.5,
                                    1.5, 0.5,
                                    2.5, 0.5,
                                    3.5, 0.5), ncol=2,byrow = TRUE)
colnames(coordsHabitatGridCenter) <- c("x","y")
# CREATE OBSERVATION WINDOWS
trapCoords <- matrix(c(1.5, 1.5, 2.5, 1.5, 1.5, 2.5, 2.5, 2.5), nrow = 4, byrow = TRUE)
colnames(trapCoords) <- c("x","y")
# PLOT CHECK
plot(coordsHabitatGridCenter[,"y"]~coordsHabitatGridCenter[,"x"],pch=16) 
points(trapCoords[,"y"]~trapCoords[,"x"],col="red",pch=16) 

# PARAMETERS
p0 <- 0.25
sigma <- 1
w <- 1.5
indicator <- 1 
# WE CONSIDER 2 INDIVIDUALS
y <- matrix(c(0, 1, 1, 0,
              0, 1, 0, 1),ncol=4,nrow=2)
s <- matrix(c(0.5, 1,
              1.6, 2.3),ncol=2,nrow=2)

# RESCALE COORDINATES 
ScaledtrapCoords <- scaleCoordsToHabitatGrid(coordsData =  trapCoords,
                                             coordsHabitatGridCenter = coordsHabitatGridCenter)
ScaledtrapCoords<- ScaledtrapCoords$coordsDataScaled
habitatMask <- matrix(1, nrow = 4, ncol=4, byrow = TRUE)


# CREATE LOCAL OBJECTS 
TrapLocal <- getLocalObjects(habitatMask = habitatMask,
                                   coords = ScaledtrapCoords,
                                   dmax=2.5,
                                   resizeFactor = 1,
                                   plot.check = TRUE
)

# GET SPARSE MATRIX 
SparseY <- getSparseY(y)

# II. USING THE DENSITY FUNCTION 
 # WE TAKE THE FIRST INDIVIDUAL
i=1
  # OPTION 1: USING THE RANDOM GENERATION FUNCTIONNALITY 
dbinomLocal_normalPlateau(x=SparseY$y[i,,1],
                   detNums=SparseY$detNums[i],
                   detIndices=SparseY$detIndices[i,,1],
                   size=rep(1,4),
                   p0 = p0,
                   sigma= sigma, 
                   w = w,
                   s=s[i,1:2],
                   trapCoords=ScaledtrapCoords,
                   localTrapsIndices=TrapLocal$localIndices,
                   localTrapsNum=TrapLocal$numLocalIndices,
                   resizeFactor=TrapLocal$resizeFactor,
                   habitatGrid=TrapLocal$habitatGrid,
                   indicator=indicator)
                                                                
  # OPTION 2: USING RANDOM GENERATION FUNCTIONNALITY 
  # WE DO NOT PROVIDE THE detNums AND detIndices ARGUMENTS
dbinomLocal_normalPlateau(x=SparseY$yCombined[i,,1],
                   size=rep(1,4),
                   p0 = p0,
                   sigma= sigma, 
                   w = w,
                   s=s[i,1:2],
                   trapCoords=ScaledtrapCoords,
                   localTrapsIndices=TrapLocal$localIndices,
                   localTrapsNum=TrapLocal$numLocalIndices,
                   resizeFactor=TrapLocal$resizeFactor,
                   habitatGrid=TrapLocal$habitatGrid,
                   indicator=indicator,
                   lengthYCombined = SparseY$lengthYCombined)

# III. USING THE RANDOM GENERATION FUNCTION 
rbinomLocal_normalPlateau(n=1,
                   size=rep(1,4),
                   p0 = p0,
                   sigma= sigma, 
                   w = w,
                   s=s[i,1:2],
                   trapCoords=ScaledtrapCoords,
                   localTrapsIndices=TrapLocal$localIndices,
                   localTrapsNum=TrapLocal$numLocalIndices,
                   resizeFactor=TrapLocal$resizeFactor,
                   habitatGrid=TrapLocal$habitatGrid,
                   indicator=indicator,
                   lengthYCombined = SparseY$lengthYCombined)

}
\references{
Dey, S., Bischof, R., Dupont, P. P. A., & Milleret, C. (2022). Does the punishment fit the crime? Consequences and diagnosis of misspecified detection functions in Bayesian spatial capture–recapture modeling. Ecology and Evolution, 12, e8600. https://doi.org/10.1002/ece3.8600
}
\author{
Soumen Dey
}
