\name{latencyhboot}

\alias{latencyhboot}

\title{Compute the Bootstrap Bandwidth for the NP Estimator of the
  Latency
}

\description{This function computes the bootstrap bandwidth for the
  nonparametric estimator of the conditional latency function.
}

\usage{latencyhboot(x, t, d, dataset, x0, bootpars =
npcure::controlpars())
}

\arguments{
  \item{x}{If \code{dataset} is missing, a numeric object giving the
    covariate values. If \code{dataset} is a data frame, it is interpreted
    as the name of the variable corresponding to the covariate in the data
    frame.
  }
  \item{t}{If \code{dataset} is missing, a numeric object giving the
    observed times. If \code{dataset} is a data frame, it is interpreted
    as the name of the variable corresponding to the observed times in the
    data frame.
  }
  \item{d}{If \code{dataset} is missing, an integer object giving the
    values of the uncensoring indicator. Censored observations must be
    coded as 0, uncensored ones as 1. If \code{dataset} is a data frame,
    it is interpreted as the name of the variable corresponding to the
    uncensoring indicator in the data frame.
  }
  \item{dataset}{An optional data frame in which the variables named in
    \code{x}, \code{t} and \code{indicator} are interpreted. If it is
    missing, \code{x}, \code{t} and \code{indicator} must be objects
    of the workspace.
  }
  \item{x0}{A numeric vector of covariate values where the local
    bootstrap bandwidth will be computed.
  }
  \item{bootpars}{A list of parameters controlling the process of
    bandwidth selection. The default is the value returned by the
    \code{controlpars} function called without arguments.
  }
}

\details{The function computes the bootstrap bandwidth selector for the
  nonparametric estimator of the conditional latency function at the
  covariate values given by \code{x0}. The bootstrap bandwidth is the
  minimizer of a bootstrap version of the Mean Integrated Squared Error
  (MISE) of the latency estimator, which is approximated by Monte Carlo
  by simulating a large number of bootstrap resamples, \code{B}. For
  each value of \code{x0}, the bootstrap MISE is the bootstrap
  expectation of the integrated difference between the value of the
  latency estimator computed with the bootstrap sample in a grid of
  bandwidths and its value computed with the original sample and a pilot
  bandwidth. The bootstrap resamples are generated by using the simple
  weighted bootstrap resampling method, fixing the covariate. This
  method is equivalent to the simple weighted bootstrap of Li and Datta
  (2001). All the parameters typically involved in the bootstrap
  bandwidth selection process (number of bootstrap resamples, grid of
  bandwidths, pilot bandwidth, and right boundary of the integration
  interval for computing the MISE) are typically set through the
  \code{controlpars} function, whose output is passed to the
  \code{bootpars} argument. Also, the bootstrap bandwidths can be
  smoothed, and, if so, the smoothed bandwidths are returned as a
  separate component of the output. See the help of \code{controlpars}
  for details.
}

\value{An object of S3 class 'npcure'. Formally, a list of components:
  \item{type}{The constant character string c("Bootstrap bandwidth",
    "latency").
  }
  \item{x0}{Grid of covariate values.
  }
  \item{h}{Selected local bootstrap bandwidths.
  }
  \item{hsmooth}{Smoothed selected local bootstrap bandwidths (optional)
  }
  \item{hgrid}{Grid of bandwidths used (optional).
  }
}

\references{Li, G., Datta, S. (2001). A bootstrap approach to
  nonparametric regression for right censored data.  Annals of the
  Institute of Statistical Mathematics, 53:
  708–729. \url{https://doi.org/10.1023/A:1014644700806}.
  
  López-Cheda, A., Jácome, M. A., Cao, R. (2017). Nonparametric latency
  estimation for mixture cure models. TEST, 26:
  353–376. \url{https://doi.org/10.1007/s11749-016-0515-1}.
}

\author{Ignacio López-de-Ullibarri [aut, cre],
  Ana López-Cheda [aut],
  Maria Amalia Jácome [aut]
}

\seealso{\code{\link{controlpars}}, \code{\link{latency}}
}

\examples{
\donttest{
## Some artificial data
set.seed(123)
n <- 50
x <- runif(n, -2, 2) ## Covariate values
y <- rweibull(n, shape = .5*(x + 4)) ## True lifetimes
c <- rexp(n) ## Censoring values
p <- exp(2*x)/(1 + exp(2*x)) ## Probability of being susceptible
u <- runif(n)
t <- ifelse(u < p, pmin(y, c), c) ## Observed times
d <- ifelse(u < p, ifelse(y < c, 1, 0), 0) ## Uncensoring indicator
data <- data.frame(x = x, t = t, d = d)

## A vector of covariate values
vecx0 <- seq(-1.5, 1.5, by = .1)

## Computation of bootstrap local bandwidths at the values of 'vecx0'...
#### ... with the default control parameters
hb1 <- latencyhboot(x, t, d, data, x0 = vecx0)

#### ... changing the default 'bootpars' with 'controlpars()':
#### (a) 'B = 1999' (1999 bootstrap resamples are generated),
#### (b) 'hbound = c(.2, 4)' and 'hl = 50' (a grid of 50 bandwidths
#### between 0.2 and 4 times the standardized interquartile range of
#### the covariate values is built), and
#### (c) 'hsave = TRUE' (the grid bandwidths are saved), and

hb2 <- latencyhboot(x, t, d, data, x0 = vecx0, bootpars =
controlpars(B = 1999, hbound = c(.2, 4), hl = 50, hsave = TRUE))

## Estimates of the conditional latency at the covariate value x0 = 0
## with the selected bootstrap bandwidths
S1 <- latency(x, t, d, data, x0 = 0, h = hb1$h[hb1$x0 == 0])
S2 <- latency(x, t, d, data, x0 = 0, h = hb2$h[hb2$x0 == 0])

## A plot comparing the estimates with bootstrap bandwidths obtained
## with default and non-default 'bootpars'
plot(S1$testim, S1$S$x0, type = "S", xlab = "Time", ylab = "Latency",
ylim = c(0, 1))
lines(S2$testim, S2$S$x0, type = "S", lty = 2)
lines(S1$testim, pweibull(S1$testim, shape = .5*(0 + 4), lower.tail =
FALSE), lwd = 2)
legend("topright", c("Estimate with 'hb1'", "Estimate with 'hb2'",
"True"), lty = c(1, 2, 1), lwd = c(1, 1, 2))

## Example with the dataset 'bmt' of the 'KMsurv' package
## to study the survival of the uncured patients aged 25 and 40
data("bmt", package = "KMsurv")
x0 <- c(25, 40)
hb <- latencyhboot(z1, t2, d3, bmt, x0 = x0, bootpars = controlpars(B =
1999, hbound = c(.2, 4), hl = 150, hsave = TRUE))
S0 <- latency(z1, t2, d3, bmt, x0 = x0, hb$h, conflevel = .95)
## Plot of predicted latency curves and confidence bands
plot(S0$testim, S0$S$x25, type = "S", xlab = "Time", ylab = "Survival",
ylim = c(0,1))
lines(S0$testim, S0$conf$x25$lower, type = "S", lty = 2)
lines(S0$testim, S0$conf$x25$upper, type = "S", lty = 2)
lines(S0$testim, S0$S$x40, type = "S", lty = 1, lwd = 2)
lines(S0$testim, S0$conf$x40$lower, type = "S", lty = 2, lwd = 2)
lines(S0$testim, S0$conf$x40$upper, type = "S", lty = 2, lwd = 2)
legend("topright", c("25 years", "95\% CI limits", "40 years",
"95\% CI limits"), lty = c(1, 2, 1, 2), lwd = c(1, 1, 2, 2)) 
}
}

\keyword{survival}

\keyword{nonparametric}
