% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/get_cumulative_estimates.R
\name{get_cumulative_estimates}
\alias{get_cumulative_estimates}
\title{Calculate cumulative estimates of a mean/proportion}
\usage{
get_cumulative_estimates(
  survey_design,
  y_var,
  y_var_type = NULL,
  predictor_variable
)
}
\arguments{
\item{survey_design}{A survey design object created with the \code{survey} package.}

\item{y_var}{Name of a variable whose mean or proportion is to be estimated.}

\item{y_var_type}{Either \code{NULL}, \code{"categorical"} or \code{"numeric"}.
For \code{"categorical"}, proportions are estimated. For \code{"numeric"}, means are estimated.
For \code{NULL} (the default), then proportions are estimated if \code{y_var} is a factor or character variable.
Otherwise, means are estimated. The data will be subset to remove any missing values in this variable.}

\item{predictor_variable}{Name of a variable for which cumulative estimates of \code{y_var}
will be calculated. This variable should either be numeric or have categories which when sorted
by their label are arranged in ascending order. The data will be subset to remove
any missing values of the predictor variable.}
}
\value{
A dataframe of cumulative estimates. The first column--whose name matches \code{predictor_variable}--gives
describes the values of \code{predictor_variable} for which a given estimate was computed.
The other columns of the result include the following:
\itemize{
\item{"outcome"}{: The name of the variable for which estimates are computed}
\item{"outcome_category"}{: For a categorical variable, the category of that variable}
\item{"estimate"}{: The estimated mean or proportion.}
\item{"std_error"}{: The estimated standard error}
\item{"respondent_sample_size"}{: The number of cases used to produce the estimate (excluding missing values)}
}
}
\description{
Calculates estimates of a mean/proportion which are cumulative with respect
to a predictor variable, such as week of data collection or number of contact attempts.
This can be useful for examining whether estimates are affected by decisions such as
whether to extend the data collection period or make additional contact attempts.
}
\examples{

# Create an example survey design
# with a variable representing number of contact attempts
library(survey)
data(involvement_survey_srs, package = "nrba")

involvement_survey_srs[["CONTACT_ATTEMPTS"]] <- runif(
  n = nrow(involvement_survey_srs),
  min = 1, max = 8
) |> round()

survey_design <- svydesign(
  weights = ~BASE_WEIGHT,
  id = ~UNIQUE_ID,
  fpc = ~N_STUDENTS,
  data = involvement_survey_srs
)

# Cumulative estimates from respondents for average student age ----
get_cumulative_estimates(
  survey_design = survey_design |>
    subset(RESPONSE_STATUS == "Respondent"),
  y_var = "STUDENT_AGE",
  y_var_type = "numeric",
  predictor_variable = "CONTACT_ATTEMPTS"
)

# Cumulative estimates from respondents for proportions of categorical variable ----
get_cumulative_estimates(
  survey_design = survey_design |>
    subset(RESPONSE_STATUS == "Respondent"),
  y_var = "WHETHER_PARENT_AGREES",
  y_var_type = "categorical",
  predictor_variable = "CONTACT_ATTEMPTS"
)

}
\references{
See Maitland et al. (2017) for an example of a level-of-effort analysis
based on this method.
\itemize{
\item Maitland, A. et al. (2017). \emph{A Nonresponse Bias Analysis
of the Health Information National Trends Survey (HINTS)}.
Journal of Health Communication 22, 545-553.
doi:10.1080/10810730.2017.1324539
}
}
