#' simulate data according to Rasch model
#' 
#' With this function data sets according to the dichotomous 
#' Rasch model (DRM) are simulated
#' 
#' Data are generated with category values 0 and 1. 
#' 
#' Person parameters are generated by a standard normal distribution.
#' 
#' @param itempar a vector with item difficulty parameters
#' @param persons number of persons for the generated data set
#' @param seed a seed for the random number generated can optionally be set
#' @return \item{datmat}{simulated data set} \item{true_itempar}{the fixed item
#' parameters according to the input} \item{true_perspar}{the fixed person
#' parameters}
#' @author Christine Hohensinn
#' @seealso \code{\link{simMPRM}}\code{\link{simCRSM}}
#' @references Fischer, G. H. (1974). Einfuehrung in die Theorie
#' psychologischer Tests [Introduction to test theory]. Bern: Huber.
#' 
#' @keywords Rasch model simulation
#' @examples
#' 
#' #set item parameters
#' item_p <- c(-1.5,-0.3,0,0.3,1.5)
#' 
#' #number of persons
#' pn <- 500
#' 
#' #simulate data set
#' simdatD <- simDRM(item_p, pn)
#' 
#' @export simDRM
#' 
simDRM <-
function(itempar, persons=500, seed=NULL){

ppar <- rnorm(persons, 0,1)

pim <- outer(ppar,itempar, FUN="-")

zahler <- exp(pim)

prob.mat <- 1/(1+zahler)

if (!is.null(seed)) {set.seed(seed)}

tmat <- matrix(runif(length(itempar)*persons, min = 0, max=1), ncol=length(itempar))

datmat <- ifelse(tmat > prob.mat, 1, 0)

return(list(datmat=datmat, true_itempar=itempar, true_perspar=ppar))
}
