% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/partial.R
\name{partial}
\alias{partial}
\alias{partial.default}
\title{Partial Dependence Functions}
\usage{
partial(object, ...)

\method{partial}{default}(object, pred.var, pred.grid, grid.resolution = NULL,
  type, which.class = 1L, plot = FALSE, smooth = FALSE, rug = FALSE,
  chull = FALSE, train, check.class = TRUE, ...)
}
\arguments{
\item{object}{A fitted model object of appropriate class (e.g.,
\code{"gbm"}, \code{"lm"}, \code{"randomForest"}, etc.).}

\item{...}{Additional optional arguments to be passed onto
\code{plyr::aaply}.}

\item{pred.var}{Character string giving the names of the predictor
variables of interest. For reasons of computation/interpretation, this
should include no more than three variables.}

\item{pred.grid}{Data frame containing the joint values of the variables
listed in \code{pred.var}.}

\item{grid.resolution}{Integer giving the number of equally spaced points to
use (only used for the continuous variables listed in \code{pred.var} when
\code{pred.grid} is not supplied). If left \code{NULL}, it will default to
the minimum between \code{51} and the number of unique data points for each
of the continuous independent variables listed in \code{pred.var}.}

\item{type}{Character string specifying the type of supervised learning.
Current options are \code{"regression"} or \code{"classification"}. For
some objects (e.g., tree-based models like \code{"rpart"}), \code{partial}
can usually extract the necessary information from \code{object}.}

\item{which.class}{Integer specifying which column of the matrix of predicted
probabilities to use as the "focus" class. Default is to use the first
class. Only used for classification problems (i.e., when
\code{type = "classification"}).}

\item{plot}{Logical indicating whether to return a data frame containing the
partial dependence values (\code{FALSE}) or plot the partial dependence
function directly (\code{TRUE}). Default is \code{FALSE}.}

\item{smooth}{Logical indicating whether or not to overlay a LOESS smooth.
Default is \code{FALSE}.}

\item{rug}{Logical indicating whether or not to include rug marks on the
predictor axes. Only used when \code{plot = TRUE}. Default is \code{FALSE}.}

\item{chull}{Logical indicating wether or not to restrict the first
two variables in \code{pred.var} to lie within the convex hull of their
data points; this effects \code{pred.grid}. Default is \code{FALSE}.}

\item{train}{An optional data frame containing the original training
data. This may be required depending on the class of \code{object}. For
objects that do not store a copy of the original training data, this
argument is required.}

\item{check.class}{Logical indicating whether or not to make sure each column
in \code{pred.grid} has the correct class, levels, etc. Default is
\code{TRUE}.}
}
\description{
Compute partial dependence functions for various model fitting objects.
}
\examples{
# Fit a random forest to the airquality data
library(randomForest)
data(airquality)
set.seed(101)  # for reproducibility
ozone.rf <- randomForest(Ozone ~ ., data = airquality, importance = TRUE,
                         na.action = na.omit)

# Using randomForest's partialPlot function
partialPlot(ozone.rf, pred.data = airquality, x.var = "Temp")

# Using pdp's partial function
head(partial(ozone.rf, pred.var = "Temp"))  # returns a data frame
partial(ozone.rf, pred.var = "Temp", plot = TRUE, rug = TRUE)

# The partial function allows for multiple predictors
partial(ozone.rf, pred.var = c("Temp", "Wind"), grid.resolution = 20,
        plot = TRUE, chull = TRUE, .progress = "text")

# The plotPartial function offers more flexible plotting
pd <- partial(ozone.rf, pred.var = c("Temp", "Wind"), grid.resolution = 20,
              chull = TRUE)
plotPartial(pd)  # the default
plotPartial(pd, levelplot = FALSE, zlab = "Ozone", drape = TRUE,
            colorkey = FALSE, screen = list(z = 120, x = -60))
}
\references{
J. H. Friedman. Greedy function approximation: A gradient boosting machine.
\emph{Annals of Statistics}, \bold{29}: 1189-1232, 2000.
}

