% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/predict_subgroup.R, R/weighted_svm.R
\name{predict.subgroup_fitted}
\alias{predict.subgroup_fitted}
\alias{predict.wksvm}
\title{Function to predict either benefit scores or treatment recommendations}
\usage{
\method{predict}{subgroup_fitted}(
  object,
  newx,
  type = c("benefit.score", "trt.group"),
  cutpoint = 0,
  ...
)

\method{predict}{wksvm}(object, newx, type = c("class", "linear.predictor"), ...)
}
\arguments{
\item{object}{fitted object returned by \code{validate.subgrp()} function.

For \code{predict.wksvm()}, this should be a fitted \code{wksvm} object from the \code{weighted.ksvm()} function}

\item{newx}{new design matrix for which predictions will be made}

\item{type}{type of prediction. \code{type = "benefit.score"} results in predicted benefit scores and
\code{type = "trt.group"} results in prediction of recommended treatment group.

For \code{predict.wksvm()}, \code{type = 'class'} yields predicted
class and \code{type = 'linear.predictor'} yields estimated function (the sign of which is the estimated class)}

\item{cutpoint}{numeric value for patients with benefit scores above which
(or below which if \code{larger.outcome.better = FALSE})
will be recommended to be in the treatment group. Can also set \code{cutpoint = "median"}, which will
use the median value of the benefit scores as the cutpoint or can set specific quantile values via \code{"quantx"}
where \code{"x"} is a number between 0 and 100 representing the quantile value; e.g. \code{cutpoint = "quant75"}
will use the 75th perent upper quantile of the benefit scores as the quantile.}

\item{...}{not used}
}
\description{
Predicts benefit scores or treatment recommendations based on a fitted subgroup identification model

Function to obtain predictions for weighted ksvm objects
}
\examples{
library(personalized)

set.seed(123)
n.obs  <- 1000
n.vars <- 50
x <- matrix(rnorm(n.obs * n.vars, sd = 3), n.obs, n.vars)


# simulate non-randomized treatment
xbetat   <- 0.5 + 0.5 * x[,21] - 0.5 * x[,41]
trt.prob <- exp(xbetat) / (1 + exp(xbetat))
trt01    <- rbinom(n.obs, 1, prob = trt.prob)

trt      <- 2 * trt01 - 1

# simulate response
delta <- 2 * (0.5 + x[,2] - x[,3] - x[,11] + x[,1] * x[,12])
xbeta <- x[,1] + x[,11] - 2 * x[,12]^2 + x[,13]
xbeta <- xbeta + delta * trt

# continuous outcomes
y <- drop(xbeta) + rnorm(n.obs, sd = 2)

# create function for fitting propensity score model
prop.func <- function(x, trt)
{
    # fit propensity score model
    propens.model <- cv.glmnet(y = trt,
                               x = x, family = "binomial")
    pi.x <- predict(propens.model, s = "lambda.min",
                    newx = x, type = "response")[,1]
    pi.x
}

subgrp.model <- fit.subgroup(x = x, y = y,
                            trt = trt01,
                            propensity.func = prop.func,
                            loss   = "sq_loss_lasso",
                            nfolds = 5)              # option for cv.glmnet

subgrp.model$subgroup.trt.effects
benefit.scores <- predict(subgrp.model, newx = x, type = "benefit.score")

rec.trt.grp <- predict(subgrp.model, newx = x, type = "trt.group")
}
\seealso{
\code{\link[personalized]{fit.subgroup}} for function which fits subgroup identification models.

\code{\link[personalized]{weighted.ksvm}} for fitting \code{weighted.ksvm} objects
}
