% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/hessian.R
\name{Hessian}
\alias{Hessian}
\title{Numerical Hessians}
\usage{
Hessian(
  FUN,
  x,
  side = 0,
  acc.order = 2,
  h = NULL,
  h0 = NULL,
  control = list(),
  f0 = NULL,
  cores = 1,
  preschedule = TRUE,
  cl = NULL,
  func = NULL,
  report = 1L,
  ...
)
}
\arguments{
\item{FUN}{A function returning a numeric scalar.
If the function returns a vector, the output will be is a Jacobian.
If instead of \code{FUN}, \code{func} is passed, as in \code{numDeriv::grad},
it will be reassigned to \code{FUN} with a warning.}

\item{x}{Numeric vector or scalar: point at which the derivative is estimated.
\code{FUN(x)} must return a finite value.}

\item{side}{Integer scalar or vector indicating difference type:
\code{0} for central, \code{1} for forward, and \code{-1} for backward differences.
Central differences are recommended unless computational cost is prohibitive.}

\item{acc.order}{Integer specifying the desired accuracy order.
The error typically scales as \eqn{O(h^{\mathrm{acc.order}})}{O(h^acc.order)}.}

\item{h}{Numeric scalar, vector, or character specifying the step size for the numerical
difference. If character (\code{"CR"}, \code{"CRm"}, \code{"DV"}, or \code{"SW"}),
calls \code{gradstep()} with the appropriate step-selection method.
Must be length 1 or match \code{length(x)}. Matrices of step sizes are not
supported. Suggestions how to handle all pairs of coordinates are welcome.}

\item{h0}{Numeric scalar of vector: initial step size for automatic search with
\code{gradstep()}.Hessian(f, 1:100)}

\item{control}{A named list of tuning parameters passed to \code{gradstep()}.}

\item{f0}{Optional numeric scalar or vector: if provided and applicable, used
where the stencil contains zero (i.e. \code{FUN(x)} is part of the sum)
to save time.
TODO: Currently ignored.}

\item{cores}{Integer specifying the number of CPU cores used for parallel computation.
Recommended to be set to the number of physical cores on the machine minus one.}

\item{preschedule}{Logical: if \code{TRUE}, disables pre-scheduling for \code{mclapply()}
or enables load balancing with \code{parLapplyLB()}. Recommended for functions that
take less than 0.1 s per evaluation.}

\item{cl}{An optional user-supplied \code{cluster} object  (created by \code{makeCluster}
or similar functions). If not \code{NULL}, the code uses \code{parLapply()} (if \code{preschedule}
is \code{TRUE}) or \code{parLapplyLB()} on that cluster on Windows, and \code{mclapply}
(fork cluster) on everything else.}

\item{func}{Deprecated; for \code{numDeriv::grad()} compatibility only.}

\item{report}{Integer: if \code{0}, returns a gradient without any attributes; if \code{1},
attaches the step size and its selection method: \code{2} or higher, attaches the full
diagnostic output (overrides \code{diagnostics = FALSE} in \code{control}).}

\item{...}{Additional arguments passed to \code{FUN}.}
}
\value{
A matrix with as many rows and columns as \code{length(x)}. Unlike the output of
\code{numDeriv::hessian()}, this output preserves the names of \code{x}.
}
\description{
Computes the second derivatives of a function with respect to all combinations
of its input coordinates. Arbitrary accuracies and sides for different coordinates
of the argument vector are supported.
}
\details{
The optimal step size for 2nd-order-accurate central-differences-based Hessians
is of the order Mach.eps^(1/4)
to balance the Taylor series truncation error with the rounding error.
However, selecting the best step size typically requires knowledge
of higher-order cross derivatives and is highly technically involved. Future releases
will allow character arguments to invoke automatic data-driven step-size selection.

The use of \code{f0} can reduce computation time similar to the use of \code{f.lower}
and \code{f.upper} in \code{uniroot()}.

Some numerical packages use the option (or even the default behaviour) of computing
not only the \code{i < j} cross-partials for the Hessian, but all pairs of \code{i}
and \code{j}. The upper and lower triangular matrices are filled, and the matrix is
averaged with its transpose to obtain a Hessian -- this is the behaviour of \code{optimHess()}.
However, it can be shown that \code{H[i, j]} and \code{H[j, i]} use the same evaluation
grid, and with a single parallelisable evaluation of the function on that grid, no
symmetrisation is necessary because the result is mathematically and computationally identical.
In \code{pnd}, only the upper triangular matrix is computed, saving time and ensuring
unambiguous results owing to the interchangeability of summation terms (ignoring the numerical
error in summation as there is nothing that can be done apart from compensation summation, e.g.
via Kahan's algorithm).
}
\examples{
f <- function(x) prod(sin(x))
Hessian(f, 1:4)
# Large matrices
\donttest{
  system.time(Hessian(f, 1:100))
}
}
\seealso{
\code{\link[=Grad]{Grad()}} for gradients, \code{\link[=GenD]{GenD()}} for generalised numerical differences.
}
