% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tfs_lambda&tfsm_lambda.R
\name{tfs_lambda}
\alias{tfs_lambda}
\alias{tfsm_lambda}
\title{Calculate sensitivity using transfer functions}
\usage{
tfs_lambda(A, d=NULL, e=NULL, startval=0.001, tolerance=1e-10, 
           return.fit=FALSE, plot.fit=FALSE)
tfsm_lambda(A, startval=0.001, tolerance=1e-10)
}
\arguments{
\item{A}{a square, nonnegative numeric matrix of any dimension.}

\item{d, e}{numeric vectors that determine the perturbation structure (see
details).}

\item{startval}{\code{tfs_lambda} calculates the limit of the derivative of the
transfer function as lambda of the perturbed matrix approaches the dominant 
eigenvalue of \code{A} (see details). \code{startval} provides a starting 
value for the algorithm: the smaller \code{startval} is, the quicker the 
algorithm should converge.}

\item{tolerance}{the tolerance level for determining convergence (see
details).}

\item{return.fit}{if \code{TRUE} the lambda and sensitivity values obtained
from the convergence algorithm are returned alongside the sensitivity at the
limit.}

\item{plot.fit}{if \code{TRUE} then convergence of the algorithm is plotted
as sensitivity~lambda.}
}
\value{
For \code{tfs_lambda}, the sensitivity of lambda-max to the specified
perturbation structure. If \code{return.fit=TRUE} a list containing
components:
\describe{
\item{sens}{the sensitivity of lambda-max to the specified perturbation 
structure}
\item{lambda.fit}{the lambda values obtained in the fitting process}
\item{sens.fit}{the sensitivity values obtained in the fitting process.}\cr
For \code{tfsm_lambda}, a matrix containing sensitivity of lambda-max 
to each element of \code{A}.
}
}
\description{
Calculate the sensitivity of the dominant eigenvalue of a population matrix 
projection model using differentiation of the transfer function.
}
\details{
\code{tfs_lambda} and \code{tfsm_lambda} differentiate a transfer function to
find sensitivity of the dominant eigenvalue of \code{A} to perturbations. 
This provides an alternative method to using matrix eigenvectors to 
calculate the sensitivity matrix and is useful as it may incorporate a 
greater diversity of perturbation structures.

\code{tfs_lambda} evaluates the transfer function of a specific perturbation 
structure. The perturbation structure is determined by \code{d\%*\%t(e)}. 
Therefore, the rows to be perturbed are determined by \code{d} and the 
columns to be perturbed are determined by \code{e}. The values in d and e 
determine the relative perturbation magnitude. For example, if only entry
[3,2] of a 3 by 3 matrix is to be perturbed, then \code{d = c(0,0,1)} and 
\code{e = c(0,1,0)}. If entries [3,2] and [3,3] are to be perturbed with the 
magnitude of perturbation to [3,2] half that of [3,3] then \code{d = c(0,0,1)} 
and \code{e = c(0,0.5,1)}. \code{d} and \code{e} may also be expressed as 
numeric one-column matrices, e.g. \code{d = matrix(c(0,0,1), ncol=1)}, 
\code{e = matrix(c(0,0.5,1), ncol=1)}. See Hodgson et al. (2006) for more 
information on perturbation structures.

\code{tfsm_lambda} returns a matrix of sensitivity values for observed
transitions (similar to that obtained when using \code{\link{sens}} to
evaluate sensitivity using eigenvectors), where a separate transfer function 
for each nonzero element of \code{A} is calculated (each element perturbed 
independently of the others).

The formula used by \code{tfs_lambda} and \code{tfsm_lambda} cannot be
evaluated at lambda-max, therefore it is necessary to find the limit of the
formula as lambda approaches lambda-max. This is done using a bisection
method, starting at a value of lambda-max + \code{startval}. \code{startval}
should be small, to avoid the potential of false convergence. The algorithm
continues until successive sensitivity calculations are within an accuracy
of one another, determined by \code{tolerance}: a \code{tolerance} of 1e-10
means that the sensitivity calculation should be accurate to 10 decimal
places. However, as the limit approaches lambda-max, matrices are no longer
invertible (singular): if matrices are found to be singular then
\code{tolerance} should be relaxed and made larger.

For \code{tfs_lambda}, there is an extra option to return and/or plot the above
fitting process using \code{return.fit=TRUE} and \code{plot.fit=TRUE}
respectively.
}
\examples{
  # Create a 3x3 matrix
  ( A <- matrix(c(0,1,2,0.5,0.1,0,0,0.6,0.6), byrow=TRUE, ncol=3) )

  # Calculate the sensitivity matrix
  tfsm_lambda(A)

  # Calculate the sensitivity of simultaneous perturbation to 
  # A[1,2] and A[1,3]
  tfs_lambda(A, d=c(1,0,0), e=c(0,1,1))

  # Calculate the sensitivity of simultaneous perturbation to 
  # A[1,2] and A[1,3] and return and plot the fitting process
  tfs_lambda(A, d=c(1,0,0), e=c(0,1,1),
             return.fit=TRUE, plot.fit=TRUE)

}
\references{
\itemize{
 \item Hodgson et al. (2006) J. Theor. Biol., 70, 214-224.
}
}
\seealso{
Other TransferFunctionAnalyses: \code{\link{tfa_inertia}},
  \code{\link{tfa_lambda}}, \code{\link{tfam_inertia}},
  \code{\link{tfam_lambda}}, \code{\link{tfs_inertia}}

Other PerturbationAnalyses: \code{\link{elas}},
  \code{\link{sens}}, \code{\link{tfa_inertia}},
  \code{\link{tfa_lambda}}, \code{\link{tfam_inertia}},
  \code{\link{tfam_lambda}}, \code{\link{tfs_inertia}}
}
\concept{
KEYWORDS
}
