########################################################################################################################################
########################################################################################################################################
# BLSDominance
########################################################################################################################################
########################################################################################################################################
#' @title
#' Computing the BLS dominance matrix of a poset.
#'
#' @description
#' Computes the dominance matrix of the input poset, based on the BLS formula of Brueggemann et al. (2003).
#'
#' @param poset Object of S4 class `POSet` (the input poset).
#' Argument `poset` must be created by using any function contained in the package aimed at building object of S4 class `POSet`
#' (e.g. [POSet()], [LinearPOSet()], [ProductPOSet()], ...) .
#'
#' @return
#' The BLS dominance matrix
#'
#' @references Brueggemann R., Lerche D. B., Sørensen P. B. (2003). First attempts to relate structures of
#' Hasse diagrams with mutual probabilities, in: Sørensen P.B., Brueggemann R., Lerche D.B.,
#' Voigt K.,Welzl G., Simon U., Abs M., Erfmann M., Carlsen L., Gyldenkærne S., Thomsen M.,
#' Fauser P., Mogensen B. B., Pudenz S., Kronvang B. Order Theory in Environmental Sciences
#' Integrative approaches.The 5th workshop held at the National Environmental Research Institute
#' (NERI), Roskilde, Denmark, November 2002. National Environmental Research Institute,
#' Denmark - NERI Technical Report, No. 479.
#'
#' @examples
#'
#' el <- c("a", "b", "c", "d")
#' dom <- matrix(c(
#'   "a", "b",
#'   "c", "b",
#'   "b", "d"
#' ), ncol = 2, byrow = TRUE)
#' pos <- POSet(elements = el, dom = dom)
#'
#' res <- BLSDominance(pos)
#'
#'
#' @name BLSDominance
#' @export BLSDominance
BLSDominance <- function(poset) {
  if (!methods::is(poset, "POSet")) {
    stop("poset must be of class POSet")
  }
  tryCatch({
    ptr <- .Call("_BruggemannLercheSorensenDominance", poset@ptr)
    return (ptr)
  }, error = function(err) {
    err_split <- strsplit(err[[1]], split = ":")
    stop(err_split[[1]][length(err_split[[1]])])
  }) # END tryCatch
}
