#' @title
#' Computing reflexive closure.
#'
#' @description
#' Computes the reflexive closure of the input binary relation.
#'
#' @param set A list of character strings (the names of the elements of the set, on which the binary relation is defined).
#'
#' @param rel A two-columns character matrix, each row comprising an element (pair) of the relation.
#'
#' @return
#' A reflexive binary relation, as a two-columns character matrix
#' (each row comprises an element (pair) of the transitivity closed relation).
#'
#' @examples
#' set<-c("a", "b", "c", "d", "e")
#'
#' rel <- matrix(c(
#'   "a", "b",
#'   "c", "b",
#'   "d", "a",
#'   "c", "a",
#'   "a", "a",
#'   "d", "d"
#' ), ncol = 2, byrow = TRUE)
#'
#' r.clo<-ReflexiveClosure(set, rel)
#'
#' @name ReflexiveClosure
#' @export ReflexiveClosure
ReflexiveClosure <- function(set, rel) {
  if (!is.character(set))
    stop("set must be character array")
  if (length(unique(set)) != length(set))
    stop("set contains duplicated values")
  if (ncol(rel) != 2)
    stop("rel must be two columns character matrix")
  if ((ncol(rel) * nrow(rel) > 0  && !is.character(rel)))
    stop("rel must be two columns character matrix")
  if (!(all(unique(as.vector(rel)) %in% set)))
    stop("rel contains values not belonging to set")

  tryCatch({
    result <- .Call("_ReflexiveClosure", set, rel)
    return (result)
  }, error = function(err) {
    err_split <- strsplit(err[[1]], split = ":")
    stop(err_split[[1]][length(err_split[[1]])])
  }) # END tryCatch
}
