% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/crossv.R
\name{catecvcount}
\alias{catecvcount}
\title{Cross-validation of the conditional average treatment effect (CATE) score
for count outcomes}
\usage{
catecvcount(
  data,
  score.method,
  cate.model,
  ps.model,
  ps.method = "glm",
  initial.predictor.method = "boosting",
  minPS = 0.01,
  maxPS = 0.99,
  higher.y = TRUE,
  prop.cutoff = seq(0.5, 1, length = 6),
  prop.multi = c(0, 1/3, 2/3, 1),
  abc = TRUE,
  train.prop = 3/4,
  cv.n = 10,
  error.max = 0.1,
  max.iter = 5000,
  xvar.smooth = NULL,
  tree.depth = 2,
  n.trees.boosting = 200,
  B = 3,
  Kfold = 5,
  error.maxNR = 0.001,
  max.iterNR = 150,
  tune = c(0.5, 2),
  seed = NULL,
  plot.gbmperf = TRUE,
  verbose = 0,
  ...
)
}
\arguments{
\item{data}{A data frame containing the variables in the outcome and
propensity score model; a data frame with \code{n} rows (1 row per
observation).}

\item{score.method}{A vector of one or multiple methods to estimate the CATE
score. Allowed values are: \code{'boosting'}, \code{'poisson'},
\code{'twoReg'}, \code{'contrastReg'}, and \code{'negBin'}.}

\item{cate.model}{A formula describing the outcome model to be fitted.
The outcome must appear on the left-hand side.}

\item{ps.model}{A formula describing the propensity score model to be fitted.
The treatment must appear on the left-hand side. The treatment must be a
numeric vector coded as 0 or 1. If data are from a randomized trial, specify
\code{ps.model} as an intercept-only model.}

\item{ps.method}{A character value for the method to estimate the propensity
score. Allowed values include one of: \code{'glm'} for logistic regression
with main effects only (default), or \code{'lasso'} for a logistic regression
with main effects and LASSO penalization on two-way interactions (added to
the model if interactions are not specified in \code{ps.model}). Relevant
only when \code{ps.model} has more than one variable.}

\item{initial.predictor.method}{A character vector for the method used to get
initial outcome predictions conditional on the covariates in
\code{cate.model}. Only applies when \code{score.method} includes
\code{'twoReg'} or \code{'contrastReg'}. Allowed values include one of
\code{'poisson'} (fastest), \code{'boosting'} and \code{'gam'}. Default is
\code{'boosting'}.}

\item{minPS}{A numerical value between 0 and 1 below which estimated
propensity scores should be truncated. Default is \code{0.01}.}

\item{maxPS}{A numerical value between 0 and 1 above which estimated
propensity scores should be truncated. Must be strictly greater than
\code{minPS}. Default is \code{0.99}.}

\item{higher.y}{A logical value indicating whether higher (\code{TRUE}) or
lower (\code{FALSE}) values of the outcome are more desirable. Default is
\code{TRUE}.}

\item{prop.cutoff}{A vector of numerical values between 0 and 1 specifying
percentiles of the estimated log CATE scores to define nested subgroups. Each
element represents the cutoff to separate observations in nested subgroups
(below vs above cutoff). The length of \code{prop.cutoff} is the number of
nested subgroups. An equally-spaced sequence of proportions ending with 1 is
recommended. Default is \code{seq(0.5, 1, length = 6)}.}

\item{prop.multi}{A vector of numerical values between 0 and 1 specifying
percentiles of the estimated log CATE scores to define mutually exclusive
subgroups. It should start with 0, end with 1, and be of
\code{length(prop.multi) > 2}. Each element represents the cutoff to separate
the observations into \code{length(prop.multi) - 1} mutually exclusive
subgroups. Default is \code{c(0, 1/3, 2/3, 1)}.}

\item{abc}{A logical value indicating whether the area between curves (ABC)
should be calculated at each cross-validation iterations, for each
\code{score.method}. Default is \code{TRUE}.}

\item{train.prop}{A numerical value between 0 and 1 indicating the proportion
of total data used for training. Default is \code{3/4}.}

\item{cv.n}{A positive integer value indicating the number of
cross-validation iterations. Default is \code{10}.}

\item{error.max}{A numerical value > 0 indicating the tolerance (maximum
value of error) for the largest standardized absolute difference in the
covariate distributions or in the doubly robust estimated rate ratios between
the training and validation sets. This is used to define a balanced
training-validation splitting. Default is \code{0.1}.}

\item{max.iter}{A positive integer value indicating the maximum number of
iterations when searching for a balanced training-validation split. Default
is \code{5,000}.}

\item{xvar.smooth}{A vector of characters indicating the name of the variables used as
the smooth terms if \code{initial.predictor.method = 'gam'}. The variables must be selected
from the variables listed in \code{cate.model}.
Default is \code{NULL}, which uses all variables in \code{cate.model}.}

\item{tree.depth}{A positive integer specifying the depth of individual trees in boosting
(usually 2-3). Used only if \code{score.method = 'boosting'} or
if \code{initial.predictor.method = 'boosting'} with \code{score.method = 'twoReg'} or
\code{'contrastReg'}. Default is 2.}

\item{n.trees.boosting}{A positive integer specifying the maximum number of trees in boosting
(usually 100-1000). Used if \code{score.method = 'boosting'} or
if \code{initial.predictor.method = 'boosting'} with \code{score.method = 'twoReg'} or
\code{'contrastReg'}. Default is \code{200}.}

\item{B}{A positive integer specifying the number of time cross-fitting is repeated in
\code{score.method = 'twoReg'} and \code{'contrastReg'}. Default is \code{3}.}

\item{Kfold}{A positive integer specifying the number of folds used in cross-fitting
to partition the data in \code{score.method = 'twoReg'} and \code{'contrastReg'}.
Default is \code{5}.}

\item{error.maxNR}{A numerical value > 0 indicating the minimum value of the mean absolute
error in Newton Raphson algorithm. Used only if \code{score.method = 'contrastReg'}.
Default is \code{0.001}.}

\item{max.iterNR}{A positive integer indicating the maximum number of iterations in the
Newton Raphson algorithm. Used only if \code{score.method = 'contrastReg'}.
Default is \code{150}.}

\item{tune}{A vector of 2 numerical values > 0 specifying tuning parameters for the
Newton Raphson algorithm. \code{tune[1]} is the step size, \code{tune[2]} specifies a
quantity to be added to diagonal of the slope matrix to prevent singularity.
Used only if \code{score.method = 'contrastReg'}. Default is \code{c(0.5, 2)}.}

\item{seed}{An optional integer specifying an initial randomization seed for reproducibility.
Default is \code{NULL}, corresponding to no seed.}

\item{plot.gbmperf}{A logical value indicating whether to plot the performance measures in
boosting. Used only if \code{score.method = 'boosting'} or if \code{score.method = 'twoReg'}
or \code{'contrastReg'} and \code{initial.predictor.method = 'boosting'}. Default is \code{TRUE}.}

\item{verbose}{An integer value indicating what kind of intermediate progress messages should
be printed. \code{0} means no outputs. \code{1} means only progress bar and run time.
\code{2} means progress bar, run time, and all errors and warnings. Default is \code{0}.}

\item{...}{Additional arguments for \code{gbm()}}
}
\value{
Returns a list containing the following components saved as a \code{"precmed"} object:
\itemize{
 \item{\code{ate.poisson}: }{A list of results output if \code{score.method} includes
 \code{'poisson'}:}
 \itemize{
    \item{\code{ate.est.train.high.cv}: }{A matrix of numerical values with
    \code{length(prop.cutoff)} rows and \code{cv.n} columns.
    The ith row/jth column cell contains the estimated ATE in the nested subgroup of high responders
    defined by CATE score above (if \code{higher.y = TRUE}) or below (if \code{higher.y = FALSE}) the
    \code{prop.cutoff[i]}x100\% percentile of the estimated CATE score in the training set in the jth
    cross-validation iteration.}
    \item{\code{ate.est.train.low.cv}: }{A matrix of numerical values with
    \code{length(prop.cutoff) - 1} rows and \code{cv.n} columns.
    The ith row/jth column cell contains the estimated ATE in the nested subgroup of low responders
    defined by CATE score below (if \code{higher.y = TRUE}) or above (if \code{higher.y = FALSE}) the
    \code{prop.cutoff[i]}x100\% percentile of the estimated CATE score in the training set in the jth
    cross-validation iteration.}
    \item{\code{ate.est.valid.high.cv}: }{Same as \code{ate.est.train.high.cv},
    but in the validation set.}
    \item{\code{ate.est.valid.low.cv}: }{Same as \code{ate.est.train.low.cv},
    but in the validation set.}
    \item{\code{ate.est.train.group.cv}: }{A matrix of numerical values with
    \code{length(prop.multi) - 1} rows and \code{cv.n} columns.
    The jth column contains the estimated ATE in \code{length(prop.multi) - 1}
    mutually exclusive subgroups defined by \code{prop.multi} in the training set in jth
    cross-validation iteration.}
    \item{\code{ate.est.valid.group.cv}: }{Same as \code{ate.est.train.group.cv}, but in the
    validation set.}
    \item{\code{abc.valid}: }{A vector of numerical values of length \code{cv.n}.
    The ith element returns the ABC of the validation curve in the ith cross-validation
    iteration. Only returned if \code{abc = TRUE}.}
 }
 \item{\code{ate.boosting}: }{A list of results similar to \code{ate.poisson} output
 if \code{score.method} includes \code{'boosting'}.}
 \item{\code{ate.twoReg}: }{A list of results similar to \code{ate.poisson} output
 if \code{score.method} includes \code{'twoReg'}.}
 \item{\code{ate.contrastReg}: }{A list of results similar to \code{ate.poisson} output
 if \code{score.method} includes \code{'contrastReg'}.
 This method has an additional element in the list of results:}
 \itemize{
    \item{\code{converge.contrastReg.cv}: }{A vector of logical value of length \code{cv.n}.
    The ith element indicates whether the algorithm converged in the ith cross-validation
    iteration.}
 }
 \item{\code{ate.negBin}: }{A list of results similar to \code{ate.poisson} output
 if \code{score.method} includes \code{'negBin'}.}
 \item{\code{props}: }{A list of 3 elements:}
 \itemize{
    \item{\code{prop.onlyhigh}: }{The original argument \code{prop.cutoff},
    reformatted as necessary.}
    \item{\code{prop.bi}: }{The original argument \code{prop.cutoff},
    similar to \code{prop.onlyhigh} but reformatted to exclude 1.}
    \item{\code{prop.multi}: }{The original argument \code{prop.multi},
    reformatted as necessary to include 0 and 1.}
 }
 \item{\code{overall.ate.valid}: }{A vector of numerical values of length \code{cv.n}.
 The ith element contains the ATE in the validation set of the ith cross-validation
 iteration, estimated with the doubly robust estimator.}
 \item{\code{overall.ate.train}: }{A vector of numerical values of length \code{cv.n}.
 The ith element contains the ATE in the training set of the ith cross-validation
 iteration, estimated with the doubly robust estimator.}
 \item{\code{fgam}: }{The formula used in GAM if \code{initial.predictor.method = 'gam'}.}
 \item{\code{higher.y}: }{The original \code{higher.y} argument.}
 \item{\code{abc}: }{The original \code{abc} argument.}
 \item{\code{cv.n}: }{The original \code{cv.n} argument.}
 \item{\code{response}: }{The type of response. Always 'count' for this function.}
 \item{\code{formulas}:}{A list of 3 elements: (1) \code{cate.model} argument,
 (2) \code{ps.model} argument and (3) original labels of the left-hand side variable in
 \code{ps.model} (treatment) if it was not 0/1.}
}
}
\description{
Provides doubly robust estimation of the average treatment effect (ATE) in
nested and mutually exclusive subgroups of patients defined by an estimated
conditional average treatment effect (CATE) score via cross-validation (CV).
The CATE score can be estimated with up to 5 methods among the following:
Poisson regression, boosting, two regressions, contrast regression, and
negative binomial (see \code{score.method}).
}
\details{
The CATE score represents an individual-level treatment effect expressed as a
rate ratio for count outcomes. It can be estimated with boosting, Poisson regression,
negative binomial regression, and the doubly robust estimator two regressions (Yadlowsky,
2020) applied separately by treatment group or with the other doubly robust estimator
contrast regression (Yadlowsky, 2020) applied to the entire data set.

Internal CV is applied to reduce optimism in choosing the CATE estimation method that
captures the most treatment effect heterogeneity. The CV is applied by repeating the
following steps \code{cv.n} times:

\enumerate{
 \item Split the data into a training and validation set according to \code{train.prop}.
 The training and validation sets must be balanced with respect to covariate distributions
 and doubly robust rate ratio estimates (see \code{error.max}).
 \item Estimate the CATE score in the training set with the specified scoring method.
 \item Predict the CATE score in the validation set using the scoring model fitted from
 the training set.
 \item Build nested subgroups of treatment responders in the training and validation sets,
 separately, and estimate the ATE within each nested subgroup. For each element i of
 \code{prop.cutoff} (e.g., \code{prop.cutoff[i]} = 0.6), take the following steps:
 \enumerate{
   \item Identify high responders as observations with the 60\%
   (i.e., \code{prop.cutoff[i]}x100\%) highest (if \code{higher.y = TRUE}) or
   lowest (if \code{higher.y = FALSE}) estimated CATE scores.
   \item Estimate the ATE in the subgroup of high responders using a doubly robust estimator.
   \item Conversely, identify low responders as observations with the 40\%
   (i.e., 1 - \code{prop.cutoff[i]}x100\%) lowest (if \code{higher.y} = TRUE) or
   highest (if \code{higher.y} = FALSE) estimated CATE scores.
   \item Estimate the ATE in the subgroup of low responders using a doubly robust estimator.
 }
 \item If \code{abc} = TRUE, calculate the area between the ATE and the series of ATEs in
 nested subgroups of high responders in the validation set.
 \item Build mutually exclusive subgroups of treatment responders in the training and
 validation sets, separately, and estimate the ATE within each subgroup. Mutually exclusive
 subgroups are built by splitting the estimated CATE scores according to \code{prop.multi}.
}
}
\examples{
\donttest{
catecv <- catecvcount(data = countExample,
                      score.method = "poisson",
                      cate.model = y ~ age + female + previous_treatment +
                                   previous_cost + previous_number_relapses +
                                   offset(log(years)),
                      ps.model = trt ~ age + previous_treatment,
                      higher.y = FALSE,
                      cv.n = 5,
                      seed = 999,
                      plot.gbmperf = FALSE,
                      verbose = 1)

plot(catecv, ylab = "Rate ratio of drug1 vs drug0 in each subgroup")
boxplot(catecv, ylab = "Rate ratio of drug1 vs drug0 in each subgroup")
abc(catecv)
}
}
\references{
Yadlowsky, S., Pellegrini, F., Lionetto, F., Braune, S., & Tian, L. (2020).
\emph{Estimation and validation of ratio-based conditional average treatment
effects using observational data. Journal of the American Statistical
Association, 1-18.}
\url{https://www.tandfonline.com/doi/full/10.1080/01621459.2020.1772080}
}
\seealso{
\code{\link{plot.precmed}()}, \code{\link{boxplot.precmed}()},
\code{\link{abc}()} methods for \code{"precmed"} objects,
and \code{\link{catefitcount}()} function.
}
