% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lmer_pi_futmat.R
\name{lmer_pi_futmat}
\alias{lmer_pi_futmat}
\title{Prediction intervals for future observations based on linear random effects models}
\usage{
lmer_pi_futmat(
  model,
  newdat = NULL,
  futmat_list = NULL,
  alternative = "both",
  alpha = 0.05,
  nboot = 10000,
  delta_min = 0.01,
  delta_max = 10,
  tolerance = 0.001,
  traceplot = TRUE,
  n_bisec = 30
)
}
\arguments{
\item{model}{a random effects model of class lmerMod}

\item{newdat}{either 1 or a \code{data.frame} with the same column names as the historical data
on which \code{model} depends}

\item{futmat_list}{a list that contains design matrices for each random factor}

\item{alternative}{either "both", "upper" or "lower". \code{alternative} specifies
if a prediction interval or an upper or a lower prediction limit should be computed}

\item{alpha}{defines the level of confidence (1-\code{alpha})}

\item{nboot}{number of bootstraps}

\item{delta_min}{lower start value for bisection}

\item{delta_max}{upper start value for bisection}

\item{tolerance}{tolerance for the coverage probability in the bisection}

\item{traceplot}{plot for visualization of the bisection process}

\item{n_bisec}{maximal number of bisection steps}
}
\value{
If \code{newdat} is a  \code{data.frame}: A \code{data.frame} that contains the future data,
 the historical mean (hist_mean), the calibrated coefficient (quant_calib),
 the prediction standard error (pred_se), the prediction interval (lower and upper)
 and a statement if the prediction interval covers the future observation (cover).

 If \code{newdat=1}: A \code{data.frame} that contains a statement that m=1,
 the historical mean (hist_mean), the calibrated coefficient (quant_calib),
 the prediction standard error (pred_se) and the prediction interval (lower and upper).

 If \code{futmat_list} is defined: A \code{data.frame} that contains the number of future observations (m),
 the historical mean (hist_mean), the calibrated coefficient (quant_calib),
 the prediction standard error (pred_se) and the prediction interval (lower and upper).

 If \code{alternative} is set to "lower": Lower prediction limits are computed instead
 of a prediction interval.

 If \code{alternative} is set to "upper": Upper prediction limits are computed instead
 of a prediction interval.

 If \code{traceplot=TRUE}, a graphical overview about the bisection process is given.
}
\description{
\code{lmer_pi_futmat()} calculates a bootstrap calibrated prediction interval for one or more
future observation(s) based on linear random effects models. With this approach,
the sampling structure of the future data is taken into account (see below).
}
\details{
This function returns a bootstrap calibrated prediction interval
\deqn{[l,u] = \hat{y} \pm q \sqrt{\hat{var}(\hat{y} - y)}}
with \eqn{\hat{y}} as the predicted future observation,
\eqn{y} as the observed future observations, \eqn{\sqrt{\hat{var}(\hat{y} - y)}}
as the prediction standard error and \eqn{q} as the bootstrap calibrated coefficient that
approximates a quantile of the multivariate t-distribution. \cr
Please note that this function relies on linear random effects models that are
fitted with lmer() from the lme4 package. Random effects have to be specified as
\code{(1|random_effect)}.

If \code{newdat} is defined, the bootstrapped future observations used for the calibration
process mimic the structure of the data set provided via \code{newdat}. The
data sampling is based on a list of design matrices (one for each random factor)
that can be obtained if \code{newdat} and the model formula are provided to
\code{lme4::lFormula()}. Hence, each random factor that is part of the initial
model must have at least two replicates in \code{newdat}. \cr
If a random factor in the future data set does not have any replicate, a list
that contains design matrices (one for each random factor) can be provided via \code{futmat_list}.

This function is an implementation of the PI given in Menssen and Schaarschmidt 2021
section 3.2.4 except that the bootstrap calibration values are drawn from
bootstrap samples that mimic the future data as described above.
}
\examples{

# loading lme4
library(lme4)

# Fitting a random effects model based on c2_dat1
\donttest{fit <- lmer(y_ijk~(1|a)+(1|b)+(1|a:b), c2_dat1)}
\donttest{summary(fit)}

#----------------------------------------------------------------------------
### Using newdat

# Prediction interval using c2_dat2 as future data
\donttest{lmer_pi_futmat(model=fit, newdat=c2_dat2, alternative="both", nboot=100)}

# Upper prediction limit for m=1 future observations
\donttest{lmer_pi_futmat(model=fit, newdat=1, alternative="upper", nboot=100)}

#----------------------------------------------------------------------------

### Using futmat_list

# c2_dat4 has no replication for b. Hence the list of design matrices can not be
# generated by lme4::lFormula() and have to be provided by hand via futmat_list.

c2_dat4

# Build a list containing the design matrices

fml <- vector(length=4, "list")

names(fml) <- c("a:b", "b", "a", "Residual")

fml[["a:b"]] <- matrix(nrow=6, ncol=2, data=c(1,1,0,0,0,0, 0,0,1,1,1,1))

fml[["b"]] <- matrix(nrow=6, ncol=1, data=c(1,1,1,1,1,1))

fml[["a"]] <- matrix(nrow=6, ncol=2, data=c(1,1,0,0,0,0, 0,0,1,1,1,1))

fml[["Residual"]] <- diag(6)

fml

# Please note, that the design matrix for the interaction term a:b is also
# provided even there is no replication for b, since it is believed that
# both, the historical and the future data descent from the same data generating
# process.

# Calculate the PI
\donttest{lmer_pi_futmat(model=fit, futmat_list=fml, alternative="both", nboot=100)}

#----------------------------------------------------------------------------

# Please note that nboot was set to 100 in order to decrease computing time
# of the example. For a valid analysis set nboot=10000.

}
\references{
Menssen, M., Schaarschmidt, F.: Prediction intervals for all of M
future observations based on linear random effects models. Statistica Neerlandica.
\doi{10.1111/stan.12260}
}
