% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lmer_pi_futmat.R
\name{lmer_pi_futmat}
\alias{lmer_pi_futmat}
\title{Prediction intervals for future observations based on linear random effects models}
\usage{
lmer_pi_futmat(
  model,
  newdat = NULL,
  futmat_list = NULL,
  alternative = "both",
  alpha = 0.05,
  nboot = 10000,
  delta_min = 0.01,
  delta_max = 10,
  tolerance = 0.001,
  traceplot = TRUE,
  n_bisec = 30,
  algorithm = "MS22"
)
}
\arguments{
\item{model}{a random effects model of class  \code{"lmerMod"}}

\item{newdat}{either 1 or a \code{data.frame} with the same column names as the historical data
on which \code{model} depends}

\item{futmat_list}{a list that contains design matrices for each random factor}

\item{alternative}{either "both", "upper" or "lower". \code{alternative} specifies
if a prediction interval or an upper or a lower prediction limit should be computed}

\item{alpha}{defines the level of confidence (1-\code{alpha})}

\item{nboot}{number of bootstraps}

\item{delta_min}{lower start value for bisection}

\item{delta_max}{upper start value for bisection}

\item{tolerance}{tolerance for the coverage probability in the bisection}

\item{traceplot}{if \code{TRUE}: Plot for visualization of the bisection process}

\item{n_bisec}{maximal number of bisection steps}

\item{algorithm}{either "MS22" or "MS22mod" (see details)}
}
\value{
\code{lmer_pi_futmat()} returns an object of class \code{c("predint", "normalPI")}
with prediction intervals or limits in the first entry (\code{$prediction}).
}
\description{
\code{lmer_pi_futmat()} calculates a bootstrap calibrated prediction interval for one or more
future observation(s) based on linear random effects models. With this approach,
the experimental design of the future data is taken into account (see below).
}
\details{
This function returns bootstrap-calibrated prediction intervals as well as
lower or upper prediction limits.

If \code{algorithm} is set to "MS22", both limits of the prediction interval
are calibrated simultaneously using the algorithm described in Menssen and
Schaarschmidt (2022), section 3.2.4. The calibrated prediction interval is given
as

\deqn{[l,u] = \hat{\mu} \pm q^{calib} \sqrt{\widehat{var}(\hat{\mu}) + \sum_{c=1}^{C+1}
\hat{\sigma}^2_c}}

with \eqn{\hat{\mu}} as the expected future observation (historical mean) and
\eqn{\hat{\sigma}^2_c} as the \eqn{c=1, 2, ..., C} variance components and \eqn{\hat{\sigma}^2_{C+1}}
as the residual variance obtained from the random
effects model fitted with \code{lme4::lmer()} and \eqn{q^{calib}} as the as the bootstrap-calibrated
coefficient used for interval calculation. \cr

If \code{algorithm} is set to "MS22mod", both limits of the prediction interval
are calibrated independently from each other. The resulting prediction interval
is given by

\deqn{[l,u] = \Big[\hat{\mu} - q^{calib}_l \sqrt{\widehat{var}(\hat{\mu}) + \sum_{c=1}^{C+1} \hat{\sigma}^2_c}, \quad
\hat{\mu} + q^{calib}_u \sqrt{\widehat{var}(\hat{\mu}) + \sum_{c=1}^{C+1} \hat{\sigma}^2_c} \Big].}

Please note, that this modification does not affect the calibration procedure, if only
prediction limits are of interest. \cr

If \code{newdat} is defined, the bootstrapped future observations used for the calibration
process mimic the structure of the data set provided via \code{newdat}. The
data sampling is based on a list of design matrices (one for each random factor)
that can be obtained if \code{newdat} and the model formula are provided to
\code{lme4::lFormula()}. Hence, each random factor that is part of the initial
model must have at least two replicates in \code{newdat}. \cr
If a random factor in the future data set does not have any replicate, a list
that contains design matrices (one for each random factor) can be provided via \code{futmat_list}.

This function is an implementation of the PI given in Menssen and Schaarschmidt 2022
section 3.2.4, except, that the bootstrap calibration values are drawn from
bootstrap samples that mimic the future data as described above.
}
\examples{

# loading lme4
library(lme4)

# Fitting a random effects model based on c2_dat1
\donttest{fit <- lmer(y_ijk~(1|a)+(1|b)+(1|a:b), c2_dat1)}
\donttest{summary(fit)}

#----------------------------------------------------------------------------
### Using newdat

# Prediction interval using c2_dat2 as future data
\donttest{pred_int <- lmer_pi_futmat(model=fit, newdat=c2_dat2, alternative="both", nboot=100)
summary(pred_int)}

# Upper prediction limit for m=1 future observations
\donttest{pred_u <- lmer_pi_futmat(model=fit, newdat=1, alternative="upper", nboot=100)
summary(pred_u)}

#----------------------------------------------------------------------------

### Using futmat_list

# c2_dat4 has no replication for b. Hence the list of design matrices can not be
# generated by lme4::lFormula() and have to be provided by hand via futmat_list.

c2_dat4

# Build a list containing the design matrices

fml <- vector(length=4, "list")

names(fml) <- c("a:b", "b", "a", "Residual")

fml[["a:b"]] <- matrix(nrow=6, ncol=2, data=c(1,1,0,0,0,0, 0,0,1,1,1,1))

fml[["b"]] <- matrix(nrow=6, ncol=1, data=c(1,1,1,1,1,1))

fml[["a"]] <- matrix(nrow=6, ncol=2, data=c(1,1,0,0,0,0, 0,0,1,1,1,1))

fml[["Residual"]] <- diag(6)

fml

# Please note, that the design matrix for the interaction term a:b is also
# provided even there is no replication for b, since it is assumed that
# both, the historical and the future data descent from the same data generating
# process.

# Calculate the PI
\donttest{pred_fml <- lmer_pi_futmat(model=fit, futmat_list=fml, alternative="both", nboot=100)
summary(pred_fml)}

#----------------------------------------------------------------------------

# Please note that nboot was set to 100 in order to decrease computing time
# of the example. For a valid analysis set nboot=10000.

}
\references{
Menssen and Schaarschmidt (2022): Prediction intervals for all of M future
observations based on linear random effects models. Statistica Neerlandica,
 \doi{10.1111/stan.12260}
}
