% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/add_linear_penalties.R
\name{add_linear_penalties}
\alias{add_linear_penalties}
\alias{add_linear_penalties,ConservationProblem,ANY,Matrix-method}
\alias{add_linear_penalties,ConservationProblem,ANY,matrix-method}
\alias{add_linear_penalties,ConservationProblem,ANY,dgCMatrix-method}
\alias{add_linear_penalties,ConservationProblem,ANY,character-method}
\alias{add_linear_penalties,ConservationProblem,ANY,numeric-method}
\alias{add_linear_penalties,ConservationProblem,ANY,Raster-method}
\alias{add_linear_penalties,ConservationProblem,ANY,SpatRaster-method}
\title{Add linear penalties}
\usage{
\S4method{add_linear_penalties}{ConservationProblem,ANY,character}(x, penalty, data)

\S4method{add_linear_penalties}{ConservationProblem,ANY,numeric}(x, penalty, data)

\S4method{add_linear_penalties}{ConservationProblem,ANY,matrix}(x, penalty, data)

\S4method{add_linear_penalties}{ConservationProblem,ANY,Matrix}(x, penalty, data)

\S4method{add_linear_penalties}{ConservationProblem,ANY,Raster}(x, penalty, data)

\S4method{add_linear_penalties}{ConservationProblem,ANY,SpatRaster}(x, penalty, data)

\S4method{add_linear_penalties}{ConservationProblem,ANY,dgCMatrix}(x, penalty, data)
}
\arguments{
\item{x}{\code{\link[=problem]{problem()}} object.}

\item{penalty}{\code{numeric} penalty value that is used to scale the
importance not selecting planning units with high \code{data} values.
Higher \code{penalty} values can be used to obtain solutions that
are strongly averse to selecting places with high \code{data}
values, and smaller \code{penalty} values can be used to obtain solutions
that only avoid places with especially high \code{data} values.
Note that negative
\code{penalty} values can be used to obtain solutions that prefer places
with high \code{data} values. Additionally, when adding these
penalties to problems with multiple zones, the argument to \code{penalty}
must have a value for each zone.}

\item{data}{\code{character}, \code{numeric},
\code{\link[terra:rast]{terra::rast()}}, \code{matrix}, or \code{Matrix} object
containing the values used to penalize solutions. Planning units that are
associated with higher data values are penalized more strongly
in the solution. See the Data format section for more information.}
}
\value{
An updated \code{\link[=problem]{problem()}} object with the penalties added to it.
}
\description{
Add penalties to a conservation planning problem to penalize
solutions that select planning units with higher values from a specific
data source (e.g., anthropogenic impact). These penalties assume
a linear trade-off between the penalty values and the primary
objective of the conservation planning problem (e.g.,
solution cost for minimum set problems; \code{\link[=add_min_set_objective]{add_min_set_objective()}}.
}
\details{
This function penalizes solutions that have higher values according
to the sum of the penalty values associated with each planning unit,
weighted by status of each planning unit in the solution.
}
\section{Data format}{


The argument to \code{data} can be specified using the following formats.

\describe{

\item{\code{data} as \code{character} vector}{containing column name(s) that
contain penalty values for planning units. This format is only
compatible if the planning units in the argument to \code{x} are a
\code{\link[sf:sf]{sf::sf()}} or \code{data.frame} object. The column(s) must have \code{numeric}
values, and must not contain any missing (\code{NA}) values.
For problems that contain a single zone, the argument to \code{data} must
contain a single column name. Otherwise, for problems that
contain multiple zones, the argument to \code{data} must
contain a column name for each zone.}

\item{\code{data} as a \code{numeric} vector}{containing values for
planning units. These values must not contain any missing
(\code{NA}) values. Note that this format is only available
for planning units that contain a single zone.}

\item{\code{data} as a \code{matrix}/\code{Matrix} object}{containing \code{numeric} values
that specify data for each planning unit.
Each row corresponds to a planning unit, each column corresponds to a
zone, and each cell indicates the data for penalizing a planning unit
when it is allocated to a given zone.}

\item{\code{data} as a \code{\link[terra:rast]{terra::rast()}} object}{containing values for planning
units. This format is only
compatible if the planning units in the argument to \code{x} are
\code{\link[sf:sf]{sf::sf()}}, or \code{\link[terra:rast]{terra::rast()}} objects.
If the planning unit data are a \code{\link[sf:sf]{sf::sf()}} object,
then the values are calculated by overlaying the
planning units with the argument to \code{data} and calculating the sum of the
values associated with each planning unit.
If the planning unit data are a \code{\link[terra:rast]{terra::rast()}} object, then the values
are calculated by extracting the cell
values (note that the planning unit data and the argument to \code{data} must
have exactly the same dimensionality, extent, and missingness).
For problems involving multiple zones, the argument to \code{data} must
contain a layer for each zone.}

}
}

\section{Mathematical formulation}{

The linear penalties are implemented using the following
equations.
Let \eqn{I} denote the set of planning units
(indexed by \eqn{i}), \eqn{Z} the set of management zones (indexed by
\eqn{z}), and \eqn{X_{iz}}{Xiz} the decision variable for allocating
planning unit \eqn{i} to zone \eqn{z} (e.g., with binary
values indicating if each planning unit is allocated or not). Also, let
\eqn{P_z} represent the penalty scaling value for zones
\eqn{z \in Z}{z in Z} (argument to \code{penalty}), and
\eqn{D_{iz}}{Diz} the penalty data for allocating planning unit
\eqn{i \in I}{i in I} to zones \eqn{z \in Z}{z in Z} (argument to
\code{data}, if supplied as a \code{matrix} object).

\deqn{
\sum_{i}^{I} \sum_{z}^{Z} P_z \times D_{iz} \times X_{iz}
}{
sum_i^I sum_z^Z (Pz * Diz * Xiz)
}

Note that when the problem objective is to maximize some measure of
benefit and not minimize some measure of cost, the term \eqn{P_z} is
replaced with \eqn{-P_z}.
}

\examples{
\dontrun{
# set seed for reproducibility
set.seed(600)

# load data
sim_pu_polygons <- get_sim_pu_polygons()
sim_features <- get_sim_features()
sim_zones_pu_raster <- get_sim_zones_pu_raster()
sim_zones_features <- get_sim_zones_features()

# add a column to contain the penalty data for each planning unit
# e.g., these values could indicate the level of habitat
sim_pu_polygons$penalty_data <- runif(nrow(sim_pu_polygons))

# plot the penalty data to visualise its spatial distribution
plot(sim_pu_polygons[, "penalty_data"], axes = FALSE)

# create minimal problem with minimum set objective,
# this does not use the penalty data
p1 <-
  problem(sim_pu_polygons, sim_features, cost_column = "cost") \%>\%
  add_min_set_objective() \%>\%
  add_relative_targets(0.1) \%>\%
  add_binary_decisions() \%>\%
  add_default_solver(verbose = FALSE)

# print problem
print(p1)

# create an updated version of the previous problem,
# with the penalties added to it
p2 <- p1 \%>\% add_linear_penalties(100, data = "penalty_data")

# print problem
print(p2)

# solve the two problems
s1 <- solve(p1)
s2 <- solve(p2)

# create a new object with both solutions
s3 <- sf::st_sf(
  tibble::tibble(
    s1 = s1$solution_1,
    s2 = s2$solution_1
  ),
  geometry = sf::st_geometry(s1)
)


# plot the solutions and compare them,
# since we supplied a very high penalty value (i.e., 100), relative
# to the range of values in the penalty data and the objective function,
# the solution in s2 is very sensitive to values in the penalty data
plot(s3, axes = FALSE)

# for real conservation planning exercises,
# it would be worth exploring a range of penalty values (e.g., ranging
# from 1 to 100 increments of 5) to explore the trade-offs

# now, let's examine a conservation planning exercise involving multiple
# management zones

# create targets for each feature within each zone,
# these targets indicate that each zone needs to represent 10\% of the
# spatial distribution of each feature
targ <- matrix(
  0.1, ncol = number_of_zones(sim_zones_features),
  nrow = number_of_features(sim_zones_features)
)

# create penalty data for allocating each planning unit to each zone,
# these data will be generated by simulating values
penalty_raster <- simulate_cost(
  sim_zones_pu_raster[[1]],
  n = number_of_zones(sim_zones_features)
)

# plot the penalty data, each layer corresponds to a different zone
plot(penalty_raster, main = "penalty data", axes = FALSE)

# create a multi-zone problem with the minimum set objective
# and penalties for allocating planning units to each zone,
# with a penalty scaling factor of 1 for each zone
p4 <-
  problem(sim_zones_pu_raster, sim_zones_features) \%>\%
  add_min_set_objective() \%>\%
  add_relative_targets(targ) \%>\%
  add_linear_penalties(c(1, 1, 1), penalty_raster) \%>\%
  add_binary_decisions() \%>\%
  add_default_solver(verbose = FALSE)

# print problem
print(p4)

# solve problem
s4 <- solve(p4)

# plot solution
plot(category_layer(s4), main = "multi-zone solution", axes = FALSE)
}
}
\seealso{
See \link{penalties} for an overview of all functions for adding penalties.

Other penalties: 
\code{\link{add_asym_connectivity_penalties}()},
\code{\link{add_boundary_penalties}()},
\code{\link{add_connectivity_penalties}()},
\code{\link{add_feature_weights}()}
}
\concept{penalties}
