% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/add_locked_in_constraints.R
\name{add_locked_in_constraints}
\alias{add_locked_in_constraints}
\alias{add_locked_in_constraints,ConservationProblem,numeric-method}
\alias{add_locked_in_constraints,ConservationProblem,logical-method}
\alias{add_locked_in_constraints,ConservationProblem,matrix-method}
\alias{add_locked_in_constraints,ConservationProblem,character-method}
\alias{add_locked_in_constraints,ConservationProblem,Raster-method}
\alias{add_locked_in_constraints,ConservationProblem,SpatRaster-method}
\alias{add_locked_in_constraints,ConservationProblem,Spatial-method}
\alias{add_locked_in_constraints,ConservationProblem,sf-method}
\title{Add locked in constraints}
\usage{
add_locked_in_constraints(x, locked_in)

\S4method{add_locked_in_constraints}{ConservationProblem,numeric}(x, locked_in)

\S4method{add_locked_in_constraints}{ConservationProblem,logical}(x, locked_in)

\S4method{add_locked_in_constraints}{ConservationProblem,matrix}(x, locked_in)

\S4method{add_locked_in_constraints}{ConservationProblem,character}(x, locked_in)

\S4method{add_locked_in_constraints}{ConservationProblem,Spatial}(x, locked_in)

\S4method{add_locked_in_constraints}{ConservationProblem,sf}(x, locked_in)

\S4method{add_locked_in_constraints}{ConservationProblem,Raster}(x, locked_in)

\S4method{add_locked_in_constraints}{ConservationProblem,SpatRaster}(x, locked_in)
}
\arguments{
\item{x}{\code{\link[=problem]{problem()}} object.}

\item{locked_in}{Object that determines which planning units that should be
locked in. See the Data format section for more information.}
}
\value{
An updated \code{\link[=problem]{problem()}} object with the constraints added to it.
}
\description{
Add constraints to a conservation planning problem to ensure
that specific planning units are selected (or allocated
to a specific zone) in the solution. For example, it may be desirable to
lock in planning units that are inside existing protected areas so that the
solution fills in the gaps in the existing reserve network. If specific
planning units should be locked out of a solution, use
\code{\link[=add_locked_out_constraints]{add_locked_out_constraints()}}. For problems with non-binary
planning unit allocations (e.g., proportions), the
\code{\link[=add_manual_locked_constraints]{add_manual_locked_constraints()}} function can be used to lock
planning unit allocations to a specific value.
}
\section{Data format}{


The locked planning units can be specified using the following formats.
Generally, the locked data should correspond to the planning units
in the argument to \code{x.} To help make working with
\code{\link[terra:rast]{terra::rast()}} planning unit data easier,
the locked data should correspond to cell indices in the
\code{\link[terra:rast]{terra::rast()}} data. For example, \code{integer} arguments
should correspond to cell indices and \code{logical} arguments should have
a value for each cell---regardless of which planning unit cells contain
\code{NA} values.

\describe{

\item{\code{data} as an \code{integer} vector}{containing indices that indicate which
planning units should be locked for the solution. This argument is only
compatible with problems that contain a single zone.}

\item{\code{data} as a \code{logical} vector}{containing \code{TRUE} and/or
\code{FALSE} values that indicate which planning units should be locked
in the solution. This argument is only compatible with problems that
contain a single zone.}

\item{\code{data} as a \code{matrix} object}{containing \code{logical} \code{TRUE} and/or
\code{FALSE} values which indicate if certain planning units are
should be locked to a specific zone in the solution. Each row
corresponds to a planning unit, each column corresponds to a zone, and
each cell indicates if the planning unit should be locked to a given
zone. Thus each row should only contain at most a single \code{TRUE}
value.}

\item{\code{data} as a \code{character} vector}{containing column name(s)
that indicates if planning units should be locked for the solution.
This format is only
compatible if the planning units in the argument to \code{x} are a
\code{\link[sf:sf]{sf::sf()}} or \code{data.frame} object. The columns
must have \code{logical} (i.e., \code{TRUE} or \code{FALSE})
values indicating if the planning unit is to be locked for the solution.
For problems that contain a single zone, the argument to \code{data} must
contain a single column name. Otherwise, for problems that
contain multiple zones, the argument to \code{data} must
contain a column name for each zone.}

\item{\code{data} as a \code{\link[sf:sf]{sf::sf()}} object}{
containing geometries that will be used to lock planning units for
the solution. Specifically, planning units in \code{x} that spatially
intersect with \code{y} will be locked (per \code{\link[=intersecting_units]{intersecting_units()}}).
Note that this option is only available
for problems that contain a single management zone.}

\item{\code{data} as a \code{\link[terra:rast]{terra::rast()}} object}{
containing cells used to lock planning units for the solution.
Specifically, planning units in \code{x}
that intersect with cells that have non-zero and non-\code{NA} values are
locked.
For problems that contain multiple zones, the
\code{data} object must contain a layer
for each zone. Note that for multi-band arguments, each pixel must
only contain a non-zero value in a single band. Additionally, if the
cost data in \code{x} is a \code{\link[terra:rast]{terra::rast()}} object, we
recommend standardizing \code{NA} values in this dataset with the cost
data. In other words, the pixels in \code{x} that have \code{NA} values
should also have \code{NA} values in the locked data.}
}
}

\examples{
\dontrun{
# set seed for reproducibility
set.seed(500)

# load data
sim_pu_polygons <- get_sim_pu_polygons()
sim_features <- get_sim_features()
sim_locked_in_raster <- get_sim_locked_in_raster()
sim_zones_pu_raster <- get_sim_zones_pu_raster()
sim_zones_pu_polygons <- get_sim_zones_pu_polygons()
sim_zones_features <- get_sim_zones_features()

# create minimal problem
p1 <-
  problem(sim_pu_polygons, sim_features, "cost") \%>\%
  add_min_set_objective() \%>\%
  add_relative_targets(0.2) \%>\%
  add_binary_decisions() \%>\%
  add_default_solver(verbose = FALSE)

# create problem with added locked in constraints using integers
p2 <- p1 \%>\% add_locked_in_constraints(which(sim_pu_polygons$locked_in))

# create problem with added locked in constraints using a column name
p3 <- p1 \%>\% add_locked_in_constraints("locked_in")

# create problem with added locked in constraints using raster data
p4 <- p1 \%>\% add_locked_in_constraints(sim_locked_in_raster)

# create problem with added locked in constraints using spatial polygon data
locked_in <- sim_pu_polygons[sim_pu_polygons$locked_in == 1, ]
p5 <- p1 \%>\% add_locked_in_constraints(locked_in)

# solve problems
s1 <- solve(p1)
s2 <- solve(p2)
s3 <- solve(p3)
s4 <- solve(p4)
s5 <- solve(p5)

# create single object with all solutions
s6 <- sf::st_sf(
  tibble::tibble(
    s1 = s1$solution_1,
    s2 = s2$solution_1,
    s3 = s3$solution_1,
    s4 = s4$solution_1,
    s5 = s5$solution_1
  ),
  geometry = sf::st_geometry(s1)
)

# plot solutions
plot(
  s6,
  main = c(
    "none locked in", "locked in (integer input)",
    "locked in (character input)", "locked in (raster input)",
    "locked in (polygon input)"
  )
)

# create minimal multi-zone problem with spatial data
p7 <-
  problem(
    sim_zones_pu_polygons, sim_zones_features,
    cost_column = c("cost_1", "cost_2", "cost_3")
  ) \%>\%
  add_min_set_objective() \%>\%
  add_absolute_targets(matrix(rpois(15, 1), nrow = 5, ncol = 3)) \%>\%
  add_binary_decisions() \%>\%
  add_default_solver(verbose = FALSE)

# create multi-zone problem with locked in constraints using matrix data
locked_matrix <- as.matrix(sf::st_drop_geometry(
  sim_zones_pu_polygons[, c("locked_1", "locked_2", "locked_3")]
))

p8 <- p7 \%>\% add_locked_in_constraints(locked_matrix)

# solve problem
s7 <- solve(p7)

# create new column representing the zone id that each planning unit
# was allocated to in the solution
s7$solution <- category_vector(sf::st_drop_geometry(
  s7[, c("solution_1_zone_1", "solution_1_zone_2", "solution_1_zone_3")]
))
s7$solution <- factor(s7$solution)

# plot solution
plot(s7[ "solution"], axes = FALSE)

# create multi-zone problem with locked in constraints using column names
p9 <- p7 \%>\% add_locked_in_constraints(c("locked_1", "locked_2", "locked_3"))

# solve problem
s9 <- solve(p9)

# create new column representing the zone id that each planning unit
# was allocated to in the solution
s9$solution <- category_vector(sf::st_drop_geometry(
  s9[, c("solution_1_zone_1", "solution_1_zone_2", "solution_1_zone_3")]
))
s9$solution[s9$solution == 1 & s9$solution_1_zone_1 == 0] <- 0
s9$solution <- factor(s9$solution)

# plot solution
plot(s9[, "solution"], axes = FALSE)

# create multi-zone problem with raster planning units
p10 <-
  problem(sim_zones_pu_raster, sim_zones_features) \%>\%
  add_min_set_objective() \%>\%
  add_absolute_targets(matrix(rpois(15, 1), nrow = 5, ncol = 3)) \%>\%
  add_binary_decisions() \%>\%
  add_default_solver(verbose = FALSE)

# create multi-layer raster with locked in units
locked_in_raster <- sim_zones_pu_raster[[1]]
locked_in_raster[!is.na(locked_in_raster)] <- 0
locked_in_raster <- locked_in_raster[[c(1, 1, 1)]]
locked_in_raster[[1]][1] <- 1
locked_in_raster[[2]][2] <- 1
locked_in_raster[[3]][3] <- 1

# plot locked in raster
plot(locked_in_raster)

# add locked in raster units to problem
p10 <- p10 \%>\% add_locked_in_constraints(locked_in_raster)

# solve problem
s10 <- solve(p10)

# plot solution
plot(category_layer(s10), main = "solution", axes = FALSE)
}

}
\seealso{
See \link{constraints} for an overview of all functions for adding constraints.

Other constraints: 
\code{\link{add_contiguity_constraints}()},
\code{\link{add_feature_contiguity_constraints}()},
\code{\link{add_linear_constraints}()},
\code{\link{add_locked_out_constraints}()},
\code{\link{add_mandatory_allocation_constraints}()},
\code{\link{add_manual_bounded_constraints}()},
\code{\link{add_manual_locked_constraints}()},
\code{\link{add_neighbor_constraints}()}
}
\concept{constraints}
