% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cqcheckI.R
\name{cqcheckI}
\alias{cqcheckI}
\title{Interactive visual checks for additive quantile fits}
\usage{
cqcheckI(obj, v, X = NULL, y = NULL, run = TRUE, width = "100\%",
  height = "680px")
}
\arguments{
\item{obj}{the output of a \code{qgam} call.}

\item{v}{if a 1D plot is required, \code{v} should be either a single character or a numeric vector. In the first case
\code{v} should be the names of one of the variables in the dataframe \code{X}. In the second case, the length
of \code{v} should be equal to the number of rows of \code{X}. If a 2D plot is required, \code{v} should be 
either a vector of two characters or a matrix with two columns.}

\item{X}{a dataframe containing the data used to obtain the conditional quantiles. By default it is NULL, in which
case predictions are made using the model matrix in \code{obj$model}.}

\item{y}{vector of responses. Its i-th entry corresponds to the i-th row of X.  By default it is NULL, in which
case it is internally set to \code{obj$y}.}

\item{run}{if TRUE (default) the function produces an interactive plot, otherwise it returns the corresponding shiny app.}

\item{width}{the width of the main plot. Default is "100\%".}

\item{height}{width the width of the main plot. Default is "680px".}
}
\value{
Simply produces an interactive plot.
}
\description{
Given an additive quantile model, fitted using \code{qgam}, \code{cqcheck2DI} provides some interactive
             2D plots that allow to check what proportion of responses, \code{y}, falls below the fitted quantile.
             This is an interactive version of the \code{cqcheck} function.
}
\details{
This is an interactive version of the \code{cqcheck}, see \code{?cqcheck} for details. The main interactive
         feature is that one can select an area by brushing, and then double-click to zoom in. In the 1D case the vertical 
         part of the selected area is not use: we zoom only along the x axis. Double-clicking without brushing zooms out.
}
\examples{
\dontrun{ 
#######
# Example 1: Bivariate additive model y~1+x+x^2+z+x*z/2+e, e~N(0, 1)
#######
library(qgam)
set.seed(15560)
n <- 1000
x <- rnorm(n, 0, 1); z <- rnorm(n)
X <- cbind(1, x, x^2, z, x*z)
beta <- c(0, 1, 1, 1, 0.5)
y <- drop(X \%*\% beta) + rnorm(n) 
dataf <- data.frame(cbind(y, x, z))
names(dataf) <- c("y", "x", "z")

#### Fit a constant model for median
qu <- 0.5
fit <- qgam(y~1, qu = qu, data = dataf)

# Look at what happens along x: clearly there is non linear pattern here
cqcheckI(obj = fit, v = c("x"), X = dataf, y = y) 

#### Add a smooth for x
fit <- qgam(y~s(x), qu = qu, data = dataf)
cqcheckI(obj = fit, v = c("x"), X = dataf, y = y) # Better!

# Lets look across across x and z. As we move along z (x2 in the plot) 
# the colour changes from green to red
cqcheckI(obj = fit, v = c("x", "z"), X = dataf, y = y)

# The effect look pretty linear
cqcheckI(obj = fit, v = c("z"), X = dataf, y = y)

#### Lets add a linear effect for z 
fit <- qgam(y~s(x)+z, qu = qu, data = dataf)

# Looks better!
cqcheckI(obj = fit, v = c("z"))

# Lets look across x and y again: green prevails on the top-left to bottom-right
# diagonal, while the other diagonal is mainly red.
cqcheckI(obj = fit, v = c("x", "z"))

### Maybe adding an interaction would help?
fit <- qgam(y~s(x)+z+I(x*z), qu = qu, data = dataf)

# It does! The real model is: y ~ 1 + x + x^2 + z + x*z/2 + e, e ~ N(0, 1)
cqcheckI(obj = fit, v = c("x", "z"))
}

}
\author{
Matteo Fasiolo <matteo.fasiolo@gmail.com>.
}
