% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tuneLearnFast.R
\name{tuneLearnFast}
\alias{tuneLearnFast}
\title{Fast learning rate calibration for the Gibbs posterior}
\usage{
tuneLearnFast(
  form,
  data,
  qu,
  err = NULL,
  multicore = !is.null(cluster),
  cluster = NULL,
  ncores = detectCores() - 1,
  paropts = list(),
  control = list(),
  argGam = NULL
)
}
\arguments{
\item{form}{A GAM formula, or a list of formulae. See ?mgcv::gam details.}

\item{data}{A data frame or list containing the model response variable and covariates required by the formula.
By default the variables are taken from environment(formula): typically the environment from which gam is called.}

\item{qu}{The quantile of interest. Should be in (0, 1).}

\item{err}{An upper bound on the error of the estimated quantile curve. Should be in (0, 1). 
Since qgam v1.3 it is selected automatically, using the methods of Fasiolo et al. (2017).
The old default was \code{err=0.05}.}

\item{multicore}{If TRUE the calibration will happen in parallel.}

\item{cluster}{An object of class \code{c("SOCKcluster", "cluster")}. This allowes the user to pass her own cluster,
which will be used if \code{multicore == TRUE}. The user has to remember to stop the cluster.}

\item{ncores}{Number of cores used. Relevant if \code{multicore == TRUE}.}

\item{paropts}{a list of additional options passed into the foreach function when parallel computation is enabled. 
This is important if (for example) your code relies on external data or packages: 
use the .export and .packages arguments to supply them so that all cluster nodes 
have the correct environment set up for computing.}

\item{control}{A list of control parameters for \code{tuneLearn} with entries: \itemize{
                  \item{\code{loss} = loss function use to tune log(sigma). If \code{loss=="cal"} is chosen, then log(sigma) is chosen so that
                                      credible intervals for the fitted curve are calibrated. See Fasiolo et al. (2017) for details.
                                      If \code{loss=="pin"} then log(sigma) approximately minimizes the pinball loss on the out-of-sample
                                      data.}
                  \item{\code{sam} = sampling scheme use: \code{sam=="boot"} corresponds to bootstrapping and \code{sam=="kfold"} to k-fold
                                     cross-validation. The second option can be used only if \code{ctrl$loss=="pin"}.}
                  \item{\code{vtype} = type of variance estimator used to standardize the deviation from the main fit in the calibration.
                                       If set to \code{"m"} the variance estimate obtained by the full data fit is used, if set to \code{"b"}
                                       than the variance estimated produced by the bootstrap fits are used. By default \code{vtype="m"}.}
                  \item{\code{epsB} = positive tolerance used to assess convergence when fitting the regression coefficients on bootstrap data.  
                                      In particular, if \code{|dev-dev_old|/(|dev|+0.1)<epsB} then convergence is achieved. 
                                      Default is \code{epsB=1e-5}.}
                  \item{\code{K} = if \code{sam=="boot"} this is the number of boostrap datasets, while if \code{sam=="kfold"} this is the 
                                   number of folds. By default \code{K=50}.}
                  \item{\code{init} = an initial value for the log learning rate (log(sigma)). 
                                      By default \code{init=NULL} and the optimization is initialized by other means.}
                  \item{\code{brac} = initial bracket for Brent method. By default \code{brac=log(c(0.5, 2))}, so the initial 
                                      search range is \code{(init + log(0.5), init + log(2))}.}
                  \item{\code{tol} = tolerance used in the Brent search. By default \code{tol=.Machine$double.eps^0.25}.
                                     See \code{?optimize} for details.}
                  \item{\code{aTol} = Brent search parameter. If the solution to a Brent get closer than 
                                      \code{aTol * abs(diff(brac))} to one of the extremes of the bracket, the optimization is
                                      stop and restarted with an enlarged and shifted bracket. \code{aTol=0.05} should be > 0 and values > 0.1
                                      don't quite make sense. By default \code{aTol=0.05}.}
                  \item{\code{redWd} = parameter which determines when the bracket will be reduced.
                                       If \code{redWd==10} then the bracket is halved if the nearest solution
                                       falls within the central 10\% of the bracket's width. By default \code{redWd = 10}.}
                  \item{\code{b} = offset parameter used by the mgcv::gauslss, which we estimate to initialize the quantile
                                   fit (when a variance model is used). By default \code{b=0}.}
                  \item{\code{link} = Link function to be used. See \code{?elf} and \code{?elflss} for defaults.}
                  \item{\code{verbose} = if TRUE some more details are given. By default \code{verbose=FALSE}.}
                  \item{\code{progress} = if TRUE progress in learning rate estimation is reported via printed text.
                                          \code{TRUE} by default.}
}}

\item{argGam}{A list of parameters to be passed to \code{mgcv::gam}. This list can potentially include all the arguments listed
in \code{?gam}, with the exception of \code{formula}, \code{family} and \code{data}.}
}
\value{
A list with entries: \itemize{
                  \item{\code{lsig} = a vector containing the values of log(sigma) that minimize the loss function, 
                                      for each quantile.}
                  \item{\code{err} = the error bound used for each quantile. Generally each entry is identical to the
                                     argument \code{err}, but in some cases the function increases it to enhance stabily.}
                  \item{\code{ranges} = the search ranges by the Brent algorithm to find log-sigma, for each quantile. }
                  \item{\code{store} = a list, where the i-th entry is a matrix containing all the locations (1st row) at which
                                       the loss function has been evaluated and its value (2nd row), for the i-th quantile.}
}
}
\description{
The learning rate (sigma) of the Gibbs posterior is tuned either by calibrating the credible intervals for the fitted
             curve, or by minimizing the pinball loss on out-of-sample data. This is done by bootrapping or by k-fold cross-validation. 
             Here the loss function is minimized, for each quantile, using a Brent search.
}
\examples{
library(qgam); library(MASS)

###
# Single quantile fit
###
# Calibrate learning rate on a grid
set.seed(5235)
tun <- tuneLearnFast(form = accel~s(times,k=20,bs="ad"), 
                     data = mcycle, 
                     qu = 0.2)

# Fit for quantile 0.2 using the best sigma
fit <- qgam(accel~s(times, k=20, bs="ad"), data = mcycle, qu = 0.2, lsig = tun$lsig)

pred <- predict(fit, se=TRUE)
plot(mcycle$times, mcycle$accel, xlab = "Times", ylab = "Acceleration", 
     ylim = c(-150, 80))
lines(mcycle$times, pred$fit, lwd = 1)
lines(mcycle$times, pred$fit + 2*pred$se.fit, lwd = 1, col = 2)
lines(mcycle$times, pred$fit - 2*pred$se.fit, lwd = 1, col = 2) 

###
# Multiple quantile fits
###
# Calibrate learning rate on a grid
quSeq <- c(0.25, 0.5, 0.75)
set.seed(5235)
tun <- tuneLearnFast(form = accel~s(times, k=20, bs="ad"), 
                     data = mcycle, 
                     qu = quSeq)

# Fit using estimated sigmas
fit <- mqgam(accel~s(times, k=20, bs="ad"), data = mcycle, qu = quSeq, lsig = tun$lsig)

# Plot fitted quantiles
plot(mcycle$times, mcycle$accel, xlab = "Times", ylab = "Acceleration", 
     ylim = c(-150, 80))
for(iq in quSeq){
  pred <- qdo(fit, iq, predict)
  lines(mcycle$times, pred, col = 2)
}  

\dontrun{
# You can get a better fit by letting the learning rate change with "accel"
# For instance
tun <- tuneLearnFast(form = list(accel ~ s(times, k=20, bs="ad"), ~ s(times)), 
                      data = mcycle, 
                      qu = quSeq)

fit <- mqgam(list(accel ~ s(times, k=20, bs="ad"), ~ s(times)),
             data = mcycle, qu = quSeq, lsig = tun$lsig)

# Plot fitted quantiles
plot(mcycle$times, mcycle$accel, xlab = "Times", ylab = "Acceleration", 
     ylim = c(-150, 80))
for(iq in quSeq){
  pred <- qdo(fit, iq, predict)
  lines(mcycle$times, pred, col = 2)
}
} 

}
\references{
Fasiolo, M., Goude, Y., Nedellec, R. and Wood, S. N. (2017). Fast calibrated additive quantile regression. Available at
            \url{https://arxiv.org/abs/1707.03307}.
}
\author{
Matteo Fasiolo <matteo.fasiolo@gmail.com>.
}
