% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/base.R
\name{qgcomp.boot}
\alias{qgcomp.boot}
\title{estimation of quantile g-computation fit, using bootstrap confidence intervals}
\usage{
qgcomp.boot(f, data, expnms = NULL, q = 4, breaks = NULL,
  id = NULL, alpha = 0.05, B = 200, rr = TRUE, degree = 1,
  seed = NULL, ...)
}
\arguments{
\item{f}{R style formula}

\item{data}{data frame}

\item{expnms}{character vector of exposures of interest}

\item{q}{NULL or number of quantiles used to create quantile indicator variables
representing the exposure variables. If NULL, then gcomp proceeds with un-transformed
version of exposures in the input datasets (useful if data are already transformed,
or for performing standard g-computation)}

\item{breaks}{(optional) NULL, or a list of (equal length) numeric vectors that 
characterize the minimum value of each category for which to 
break up the variables named in expnms. This is an alternative to using 'q'
to define cutpoints.}

\item{id}{(optional) NULL, or variable name indexing individual units of 
observation (only needed if analyzing data with multiple observations per 
id/cluster)}

\item{alpha}{alpha level for confidence limit calculation}

\item{B}{integer: number of bootstrap iterations (this should typically be
>=200, though it is set lower in examples to improve run-time).}

\item{rr}{logical: if using binary outcome and rr=TRUE, qgcomp.boot will 
estimate risk ratio rather than odds ratio}

\item{degree}{polynomial basis function for marginal model (e.g. degree = 2
allows that the relationship between the whole exposure mixture and the outcome
is quadratic.}

\item{seed}{integer or NULL: random number seed for replicable bootstrap results}

\item{...}{arguments to glm (e.g. family)}
}
\value{
a qgcompfit object, which contains information about the effect
 measure of interest (psi) and associated variance (var.psi), as well
 as information on the model fit (fit) and information on the 
 marginal structural model (msmfit) used to estimate the final effect
 estimates.
}
\description{
This function yields population average effect estimates for 
  both continuous and binary outcomes
}
\details{
Estimates correspond to the average expected change in the
 (log) outcome per quantile increase in the joint exposure to all exposures 
 in `expnms'. Test statistics and confidence intervals are based on 
 a non-parametric bootstrap, using the standard deviation of the bootstrap
 estimates to estimate the standard error. The bootstrap standard error is 
 then used to estimate Wald-type confidence intervals. Note that no bootstrapping
 is done on estimated quantiles of exposure, so these are treated as fixed
 quantities
}
\examples{
set.seed(30)
# continuous outcome
dat <- data.frame(y=rnorm(100), x1=runif(100), x2=runif(100), z=runif(100))
# Conditional linear slope
qgcomp.noboot(y ~ z + x1 + x2, expnms = c('x1', 'x2'), data=dat, q=4, family=gaussian())
# Marginal linear slope (population average slope, for a purely linear, 
#  additive model this will equal the conditional)
qgcomp.boot(f=y ~ z + x1 + x2, expnms = c('x1', 'x2'), data=dat, q=4, 
  family=gaussian(), B=10) #increase B to at least 200 in actual examples
  
# Population average mixture slope which accounts for non-linearity and interactions
qgcomp.boot(y ~ z + x1 + x2 + I(x1^2) + I(x2*x1), family="gaussian", 
 expnms = c('x1', 'x2'), data=dat, q=4, B=10)
 
# binary outcome
dat <- data.frame(y=rbinom(50,1,0.5), x1=runif(50), x2=runif(50), z=runif(50))

# Conditional mixture OR
qgcomp.noboot(y ~ z + x1 + x2, family="binomial", expnms = c('x1', 'x2'), 
  data=dat, q=2)
  
#Marginal mixture OR (population average OR - in general, this will not equal the 
# conditional mixture OR due to non-collapsibility of the OR)
qgcomp.boot(y ~ z + x1 + x2, family="binomial", expnms = c('x1', 'x2'), 
  data=dat, q=2, B=10)
  
# Population average mixture RR
qgcomp.boot(y ~ z + x1 + x2, family="binomial", expnms = c('x1', 'x2'), 
  data=dat, q=2, rr=TRUE, B=10)
  
# Population average mixture RR, indicator variable representation of x2
# note that I(x==...) operates on the quantile-based category of x,
# rather than the raw value
res = qgcomp.boot(y ~ z + x1 + I(x2==1) + I(x2==2) + I(x2==3), 
  family="binomial", expnms = c('x1', 'x2'), data=dat, q=4, rr=TRUE, B=10)
res$fit  
plot(res)

# now add in a non-linear MSM
res2 = qgcomp.boot(y ~ z + x1 + I(x2==1) + I(x2==2) + I(x2==3), 
  family="binomial", expnms = c('x1', 'x2'), data=dat, q=4, rr=TRUE, B=10, 
  degree=2)
res2$fit  
res2$msmfit  
plot(res2)
# Log risk ratio per one IQR change in all exposures (not on quantile basis)
dat$x1iqr <- dat$x1/with(dat, diff(quantile(x1, c(.25, .75))))
dat$x2iqr <- dat$x2/with(dat, diff(quantile(x2, c(.25, .75))))
# note that I(x>...) nowoperates on the untransformed value of x,
# rather than the raw value
res2 = qgcomp.boot(y ~ z + x1iqr + I(x2iqr>0.1) + I(x2>0.4) + I(x2>0.9), 
  family="binomial", expnms = c('x1iqr', 'x2iqr'), data=dat, q=NULL, rr=TRUE, B=10, 
  degree=2)
res2
}
\seealso{
\code{\link[qgcomp]{qgcomp.noboot}}, and \code{\link[qgcomp]{qgcomp}}
}
\concept{variance mixtures}
