% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lod_int.R
\name{lod_int}
\alias{lod_int}
\title{Calculate LOD support intervals}
\usage{
lod_int(
  scan1_output,
  map,
  chr = NULL,
  lodcolumn = 1,
  threshold = 0,
  peakdrop = Inf,
  drop = 1.5,
  expand2markers = TRUE
)
}
\arguments{
\item{scan1_output}{An object of class \code{"scan1"} as returned by
\code{\link[=scan1]{scan1()}}.}

\item{map}{A list of vectors of marker positions, as produced by
\code{\link[=insert_pseudomarkers]{insert_pseudomarkers()}}.}

\item{chr}{Chromosome ID to consider (must be a single value).}

\item{lodcolumn}{LOD score column to consider (must be a single value).}

\item{threshold}{Minimum LOD score for a peak.}

\item{peakdrop}{Amount that the LOD score must drop between peaks,
if multiple peaks are to be defined on a chromosome.}

\item{drop}{Amount to drop in the support interval.  Must be
\eqn{\le} \code{peakdrop}}

\item{expand2markers}{If TRUE, QTL intervals are expanded so
that their endpoints are at genetic markers.}
}
\value{
A matrix with three columns:
\itemize{
\item \code{ci_lo} - lower bound of interval
\item \code{pos} - peak position
\item \code{ci_hi} - upper bound of interval
}

Each row corresponds to a different peak.
}
\description{
Calculate LOD support intervals for a single LOD curve on a single
chromosome, with the ability to identify intervals for multiple LOD
peaks.
}
\details{
We identify a set of peaks defined as local maxima that
exceed the specified \code{threshold}, with the requirement that
the LOD score must have dropped by at least \code{peakdrop} below
the lowest of any two adjacent peaks.

At a given peak, if there are ties, with multiple positions jointly
achieving the maximum LOD score, we take the average of these
positions as the location of the peak.

The default is to use \code{threshold=0}, \code{peakdrop=Inf}, and
\code{drop=1.5}. We then return results a single peak, no matter the
maximum LOD score, and give a 1.5-LOD support interval.
}
\examples{
# read data
iron <- read_cross2(system.file("extdata", "iron.zip", package="qtl2"))
\dontshow{iron <- iron[,7]}

# insert pseudomarkers into map
map <- insert_pseudomarkers(iron$gmap, step=1)

# calculate genotype probabilities
probs <- calc_genoprob(iron, map, error_prob=0.002)

# grab phenotypes and covariates; ensure that covariates have names attribute
pheno <- iron$pheno
covar <- match(iron$covar$sex, c("f", "m")) # make numeric
names(covar) <- rownames(iron$covar)
Xcovar <- get_x_covar(iron)

# perform genome scan
out <- scan1(probs, pheno, addcovar=covar, Xcovar=Xcovar)

# 1.5-LOD support interval for QTL on chr 7, first phenotype
lod_int(out, map, chr=7, lodcolum=1)
}
\seealso{
\code{\link[=bayes_int]{bayes_int()}}, \code{\link[=find_peaks]{find_peaks()}}, \code{\link[=scan1]{scan1()}}
}
