% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dfm.R
\name{dfm}
\alias{dfm}
\title{create a document-feature matrix}
\usage{
dfm(x, tolower = TRUE, stem = FALSE, select = NULL, remove = NULL,
  thesaurus = NULL, dictionary = NULL, valuetype = c("glob", "regex",
  "fixed"), groups = NULL, verbose = FALSE, ...)
}
\arguments{
\item{x}{character, \link{corpus}, \link{tokens}, or \link{dfm} object}

\item{tolower}{convert all tokens to lowercase}

\item{stem}{if \code{TRUE}, stem words}

\item{select}{a user supplied regular expression defining which features to
keep, while excluding all others.  This can be used in lieu of a dictionary
if there are only specific features that a user wishes to keep. To extract
only Twitter usernames, for example, set \code{select = "@*"} and make
sure that \code{removeTwitter = FALSE} as an additional argument passed to
\link{tokenize}.  Note: \code{select = "^@\\\w+\\\b"} would be the regular
expression version of this matching pattern.  The pattern matching type
will be set by \code{valuetype}.}

\item{remove}{a character vector of user-supplied features to ignore, such as
"stop words".  To access one possible list (from any list you wish), use
\code{\link{stopwords}()}.  The pattern matching type will be set by
\code{valuetype}.  For behaviour of \code{remove} with \code{ngrams > 1},
see Details.}

\item{thesaurus}{A list of character vector "thesaurus" entries, in a 
dictionary list format, which operates as a dictionary but without 
excluding values not matched from the dictionary.  Thesaurus keys are 
converted to upper case to create a feature label in the dfm, as a reminder
that this was not a type found in the text, but rather the label of a 
thesaurus key.  For more fine-grained control over this and other aspects 
of converting features into dictionary/thesaurus keys from pattern matches 
to values, you can use \code{\link{dfm_lookup}} after creating the 
dfm.}

\item{dictionary}{A list of character vector dictionary entries, including 
regular expressions (see examples)}

\item{valuetype}{how to interpret keyword expressions: \code{"glob"} for 
"glob"-style wildcard expressions; \code{"regex"} for regular expressions;
or \code{"fixed"} for exact matching. See \link{valuetype} for details.}

\item{groups}{character vector containing the names of document variables for
aggregating documents; only applies when calling dfm on a corpus object. 
When \code{x} is a \link{dfm} object, \code{groups} provides a convenient
and fast method of combining and refactoring the documents of the dfm
according to the groups.}

\item{verbose}{display messages if \code{TRUE}}

\item{...}{additional arguments passed to \link{tokens}, for character and
corpus}
}
\value{
a \link{dfm-class} object
}
\description{
Construct a sparse document-feature matrix, from a character, \link{corpus}, 
or \link{tokens} object.
}
\details{
The default behavior for \code{remove}/\code{select} when
  constructing ngrams using \code{dfm(x, } \emph{ngrams > 1}\code{)} is to
  remove/select any ngram constructed from a matching feature.  If you wish
  to remove these before constructing ngrams, you will need to first tokenize
  the texts with ngrams, then remove the features to be ignored, and then 
  construct the dfm using this modified tokenization object.  See the code 
  examples for an illustration.
}
\examples{
## for a corpus

corpus_post80inaug <- corpus_subset(data_corpus_inaugural, Year > 1980)
dfm(corpus_post80inaug)
dfm(corpus_post80inaug, tolower = FALSE)

# grouping documents by docvars in a corpus
dfm(corpus_post80inaug, groups = "President", verbose = TRUE)

# with English stopwords and stemming
dfm(corpus_post80inaug, remove = stopwords("english"), stem = TRUE, verbose = TRUE)
# works for both words in ngrams too
dfm("Banking industry", stem = TRUE, ngrams = 2, verbose = FALSE)

# with dictionaries
corpus_post1900inaug <- corpus_subset(data_corpus_inaugural, Year>1900)
mydict <- dictionary(list(christmas = c("Christmas", "Santa", "holiday"),
               opposition = c("Opposition", "reject", "notincorpus"),
               taxing = "taxing",
               taxation = "taxation",
               taxregex = "tax*",
               country = "states"))
dfm(corpus_post1900inaug, dictionary = mydict)

# with the thesaurus feature
mytexts <- c("The new law included a capital gains tax, and an inheritance tax.",
             "New York City has raised a taxes: an income tax and a sales tax.")
mydict <- dictionary(list(tax=c("tax", "income tax", "capital gains tax", "inheritance tax")))
dfm(phrasetotoken(mytexts, mydict), thesaurus = lapply(mydict, function(x) gsub("\\\\s", "_", x)))
# pick up "taxes" with "tax" as a regex
dfm(phrasetotoken(mytexts, mydict), thesaurus = list(anytax = "tax"), valuetype = "regex")

# removing stopwords
testText <- "The quick brown fox named Seamus jumps over the lazy dog also named Seamus, with
             the newspaper from a boy named Seamus, in his mouth."
testCorpus <- corpus(testText)
# note: "also" is not in the default stopwords("english")
featnames(dfm(testCorpus, select = stopwords("english")))
# for ngrams
featnames(dfm(testCorpus, ngrams = 2, select = stopwords("english"), removePunct = TRUE))
featnames(dfm(testCorpus, ngrams = 1:2, select = stopwords("english"), removePunct = TRUE))

## removing stopwords before constructing ngrams
tokensAll <- tokens(char_tolower(testText), removePunct = TRUE)
tokensNoStopwords <- removeFeatures(tokensAll, stopwords("english"))
tokensNgramsNoStopwords <- tokens_ngrams(tokensNoStopwords, 2)
featnames(dfm(tokensNgramsNoStopwords, verbose = FALSE))

# keep only certain words
dfm(testCorpus, select = "*s", verbose = FALSE)  # keep only words ending in "s"
dfm(testCorpus, select = "s$", valuetype = "regex", verbose = FALSE)

# testing Twitter functions
testTweets <- c("My homie @justinbieber #justinbieber shopping in #LA yesterday #beliebers",
                "2all the ha8ers including my bro #justinbieber #emabiggestfansjustinbieber",
                "Justin Bieber #justinbieber #belieber #fetusjustin #EMABiggestFansJustinBieber")
dfm(testTweets, select = "#*", removeTwitter = FALSE)  # keep only hashtags
dfm(testTweets, select = "^#.*$", valuetype = "regex", removeTwitter = FALSE)

# for a dfm
dfm1 <- dfm(data_corpus_irishbudget2010)
dfm2 <- dfm(dfm1, 
            groups = ifelse(docvars(data_corpus_irishbudget2010, "party") \%in\% c("FF", "Green"),
                            "Govt", "Opposition"), 
            tolower = FALSE, verbose = TRUE)

}
\seealso{
\code{\link{dfm_select}}, \link{dfm-class}
}
\keyword{dfm}
