% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/radjust_sym.R
\name{radjust_sym}
\alias{radjust_sym}
\title{Adjust p-values for Replicability across Two Independent Studies with Multiple Endpoints}
\usage{
radjust_sym(pv1, pv2, w1 = 0.5, input_type = c("selected_features",
  "all_features"), general_dependency = FALSE,
  directional_rep_claim = FALSE,
  variant = c("non-adaptive-with-alpha-selection", "adaptive",
  "non-adaptive"), alpha = if (variant == "non-adaptive") NULL else 0.05)
}
\arguments{
\item{pv1, pv2}{numeric vectors of p-values. If \code{directional_rep_claim} is \code{TRUE}, they must be left-sided p-values.
Can be either the p-values for the selected features from each study (the default input type),
or the p-values for all the features from each study.
Can be either of the same length (so the same location in each vector corresponds to the same feature) or with names (so the same name in each vector corresponds to the same feature).}

\item{w1}{fraction between zero and one, of the relative weight for the p-values from study 1. Default value is 0.5
(see Details for other values).}

\item{input_type}{whether \code{pv1} and \code{pv2} contain all the p-values from each study or only the selected ones (the default).}

\item{general_dependency}{\code{TRUE} or \code{FALSE}, indicating whether to correct for general dependency.
The recommended default value is \code{FALSE} (see Details).}

\item{directional_rep_claim}{\code{TRUE} or \code{FALSE}, indicating whether to perform directional replicability analysis. The default value is \code{FALSE}. If \code{TRUE}, \code{pv1} and \code{pv2} should be left-sided p-values (see Details).}

\item{variant}{A character string specifying the chosen variant for a potential increase in the number of discoveries.
Must be one of \code{"non-adaptive-with-alpha-selection"} (default), \code{"adaptive"}, or \code{"non-adaptive"} (see Details).}

\item{alpha}{The threshold on p-values for selecting the features in each study and the significance level for replicability analysis (see Details).}
}
\value{
The function returns a list with the following elements:

  \tabular{ll}{
    \code{call}  \tab  the function call. \cr
    \code{inputs}  \tab  a list with the function's input parameters (except \code{pv1} and \code{pv2}). \cr
    \code{results_table} \tab  a data frame with the features selected in both studies and their r-values (see description below).\cr
    \code{selected1}  \tab the features selected in study 1 (when the variant is either \code{"adaptive"} or \code{"non-adaptive-with-alpha-selection"}). \cr
    \code{selected2}  \tab the features selected in study 2, same as above. \cr
    \code{n_selected1} \tab  the number of selected features in study 1. \cr
    \code{n_selected2} \tab  the number of selected features in study 2. \cr
    \code{pi1}  \tab  the estimate of the true-nulls fraction in the study 1 among the selected in study 2, when \code{variant = "adaptive"}.\cr
    \code{pi2}  \tab  the estimate of the true-nulls fraction in the study 2 among the selected in study 1, when \code{variant = "adaptive"}.
  }

The third element in the list, \code{results_table}, includes the following columns:

  \tabular{lll}{
    \code{name}         \tab char.   \tab the name of the feature as extracted from the named vectors, or the location, if the input vectors are not named. \cr
    \code{p.value.1}    \tab numeric \tab the one-sided p-value from study 1 as inputed (denoted by \code{pv1}). When \code{directional_rep_claim==TRUE} the one-sided p-values in the direction of effect are presented (i.e, \code{min(pv1,1-pv1)}). \cr
    \code{p.value.2}    \tab numeric \tab the one-sided p-value from study 2 as inputed (denoted by \code{pv2}). When \code{directional_rep_claim==TRUE} the one-sided p-values in the direction of effect are presented (i.e, \code{min(pv2,1-pv2)}).  \cr
    \code{r.value}      \tab numeric \tab the replicability adjusted p-value (= r-value). \cr
    \code{Direction}    \tab char.   \tab the direction of the replicability claim, when \code{directional_rep_claim = TRUE}. \cr
    \code{Significant}  \tab char.   \tab indicates for which features replicability claims can be made at level \eqn{\alpha}, when \code{variant} is set to \code{"adaptive"} or \code{"non-adaptive-with-alpha-selection"}.
  }
}
\description{
Given two vectors of p-values from two independent studies, returns the adjusted p-values for
 false discovery rate control on replicability claims.
}
\details{
For FDR control at level \eqn{\alpha} on replicability claims, all features with \strong{\eqn{r}-value} at most \eqn{\alpha} are declared
 as replicated.
 In addition, the discoveries from study 1 among the replicability claims have an FDR control guarantee at level \eqn{w_{1}\alpha}{w1 * \alpha}.
 Similarly, the discoveries from study 2 among the replicability claims have an FDR control guarantee at level \eqn{(1-w_{1})\alpha}{(1-w1) * \alpha}.

 Setting \code{w1} to a value different than half is appropriate for stricter FDR control in one of the studies.
 For example, if study two has a much larger sample size than study one (and both studies examine the same problem), then
 setting \eqn{w_{1} > 0.5}{w1 > 0.5} will provide a stricter FDR control for the larger study and greater power for the replicability analysis,
 see Bogomolov and Heller (2018) for details.

 The theoretical FDR control guarantees assume independence within each vector of p-values. However, empirical
 investigations suggest that the method is robust to deviations from independence. In practice, we recommend using it whenever the
 Benjamini-Hochberg procedure is appropriate for use with single studies, as this procedure can be viewed as a two-dimensional
 Benjamini-Hochberg procedure which enjoys similar robustness properties. For general dependence, we provide the option to apply
 a more conservative procedure with theoretical FDR control guarantee for any type of dependence,
 by setting \code{general_dependency} to TRUE.

 If \code{variant} is \code{"non-adaptive"}, then the non-adaptive  replicability analysis procedure of Bogomolov and Heller (2018)
 is applied on the input p-values \code{pv1} and \code{pv2}.
 If \code{variant} is \code{"non-adaptive-with-alpha-selection"}, then for a user specified \code{alpha} (default 0.05) only p-values from
 study one below \eqn{w_{1}\alpha}{w1 * \alpha} and from study
 two below \eqn{(1-w_{1})\alpha}{(1-w1) * \alpha} are considered for replicability analysis. This additional step prevents
 including in the selected sets features that cannot be discovered as replicability claims at the nominal FDR level
 \eqn{\alpha}, thus reducing the multiplicity adjustment necessary for replicability analysis. If \code{variant} is \code{"adaptive"}, then for a user specified \code{alpha}
 the adaptive replicability analysis procedure is applied on the dataset, see Bogomolov and Heller (2018) for details.

The meaning of the replicability claim for a feature if \code{directional_rep_claim} is \code{FALSE}, is that both null hypotheses  are false (or both alternatives are true).  Setting \code{directional_rep_claim} to
\code{TRUE} is useful if the discoveries of interest are directional but the direction is unknown. For example, a directional replicability
claim for a feature is the claim that both associations examined for it are positive, or both associations examined for it
are negative, but not that one association is positive and the other negative. For  directional replicability analysis,
the input p-values \code{pv1} and \code{pv2} should be the left-sided input p-values
(left-sided is the choice without loss of generality, since we assume the left and right sided p-values sum to one for each null hypothesis).
}
\examples{

data(mice)
## transform the two-sided p-values to one-sided in the same direction (left):
## (we use the direction of the test statistic to do so and assume that it is continuous)

pv1 <- ifelse(mice$dir_is_left1, mice$twosided_pv1/2, 1-mice$twosided_pv1/2)
pv2 <- ifelse(mice$dir_is_left2, mice$twosided_pv2/2, 1-mice$twosided_pv2/2)

## run the examples as in the article:

mice_rv_adaptive <- radjust_sym(pv1, pv2, input_type = "all", directional_rep_claim = TRUE,
                                variant = "adaptive", alpha=0.05)
print(mice_rv_adaptive)

mice_rv_non_adpt_sel <- radjust_sym(pv1, pv2, input_type = "all", directional_rep_claim = TRUE,
                                    variant = "non-adaptive-with-alpha-selection", alpha=0.05)
print(mice_rv_non_adpt_sel)

mice_rv_non_adpt <- radjust_sym(pv1, pv2, input_type = "selected", directional_rep_claim = TRUE,
                                variant = "non-adaptive")
print(mice_rv_non_adpt)

}
\references{
Bogomolov, M. and Heller, R. (2018). Assessing replicability of findings across two studies of multiple
features. Biometrika.
}
\seealso{
\code{\link{radjust_pf}} for replicability analysis in primary and follow-up studies.
}
