% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dataDWD.R
\name{dataDWD}
\alias{dataDWD}
\title{Download data from the DWD CDC FTP Server}
\usage{
dataDWD(file, base = dwdbase, dir = "DWDdata", force = FALSE,
  overwrite = FALSE, sleep = 0, quiet = FALSE, progbar = !quiet,
  browse = FALSE, read = TRUE, meta = grepl(".txt$", file),
  fread = FALSE, format = NA, ntrunc = 2, dfargs = NULL, ...)
}
\arguments{
\item{file}{Char (vector): complete file URL(s) (including base and filename.zip) as returned by
\code{\link{selectDWD}}. Can be a vector with several filenames.}

\item{base}{Single char: base URL that will be removed from output file names.
DEFAULT: \code{\link{dwdbase}}:
\url{ftp://ftp-cdc.dwd.de/pub/CDC/observations_germany/climate}}

\item{dir}{Char: Writeable directory name where to save the downloaded file.
Created if not existent. DEFAULT: "DWDdata" at current \code{\link{getwd}()}}

\item{force}{Logical (vector): always download, even if the file already exists in \code{dir}?
Use NA to force re-downloading files older than 24 hours.
If FALSE, it is still read (or name returned). DEFAULT: FALSE}

\item{overwrite}{Logical (vector): if force=TRUE, overwrite the existing file
rather than append "_1"/"_2" etc to the filename? DEFAULT: FALSE}

\item{sleep}{Number. If not 0, a random number of seconds between 0 and
\code{sleep} is passed to \code{\link{Sys.sleep}} after each download
to avoid getting kicked off the FTP-Server. DEFAULT: 0}

\item{quiet}{Logical: suppress message about directory / filenames? DEFAULT: FALSE}

\item{progbar}{Logical: present a progress bar with estimated remaining time?
If missing and length(file)==1, progbar is internally set to FALSE.
Only works if the R package \code{pbapply} is available. DEFAULT: TRUE (!quiet)}

\item{browse}{Logical: open repository via \code{\link{browseURL}} and
return URL folder path? If TRUE, no data is downloaded.
If file has several values, only unique folders will be opened.
DEFAULT: FALSE}

\item{read}{Logical: read the file(s) with \code{\link{readDWD}}? If FALSE,
only download is performed and the filename(s) returned. DEFAULT: TRUE}

\item{meta}{Logical (vector): is the \code{file} a meta file? Passed to
\code{\link{readDWD}}. DEFAULT: TRUE for each file ending in ".txt"}

\item{fread}{Fast reading? See \code{\link{readDWD}}. DEFAULT: FALSE}

\item{format}{Char (vector): format used in \code{\link{strptime}} to convert date/time column,
see \code{\link{readDWD}}. DEFAULT: NA}

\item{ntrunc}{Single integer: number of filenames printed in messages
before they get truncated with message "(and xx more)". DEFAULT: 2}

\item{dfargs}{Named list of additional arguments passed to \code{\link{download.file}}}

\item{\dots}{Further arguments passed to \code{\link{readDWD}}. 
Were passed to \code{\link{download.file}} prior to rdwd 0.11.7 (2019-02-25)}
}
\value{
Presuming downloading and processing were successful:
        if \code{read=TRUE}, a data.frame of the desired dataset
        (as returned by \code{\link{readDWD}}),
        otherwise the filename as saved on disc
        (may have "_n" appended in name, see \code{\link{newFilename}}).\cr
        If length(file)>1, the output is a list of data.frames / vector of filenames.\cr
        The output is always invisible.
}
\description{
Get climate data from the German Weather Service (DWD) FTP-server.
The desired .zip (or .txt) dataset is downloaded into \code{dir}.
If \code{read=TRUE}, it is also read, processed and returned as a data.frame.
}
\examples{
\dontrun{ ## requires internet connection
# find FTP files for a given station name and file path:
link <- selectDWD("Fuerstenzell", res="hourly", var="wind", per="recent")
# download file:
fname <- dataDWD(link, dir=tempdir(), read=FALSE) ; fname
# dir="DWDdata" is the default directory to store files
# unless force=TRUE, already obtained files will not be downloaded again

# read and plot file:
wind <- readDWD(fname)          ; head(wind)
metafiles <- readMeta(fname)    ; str(metafiles, max.level=1)
column_names <- readVars(fname) ; head(column_names)

plot(wind$MESS_DATUM, wind$F, main="DWD hourly wind Fuerstenzell", col="blue",
     xaxt="n", las=1, type="l", xlab="Date", ylab="Hourly Wind speed  [m/s]")
berryFunctions::monthAxis(1, ym=T)


# current and historical files:
link <- selectDWD("Potsdam", res="daily", var="kl", per="hr"); link
potsdam <- dataDWD(link, dir=tempdir())
potsdam <- do.call(rbind, potsdam) # this will partly overlap in time
plot(TMK~MESS_DATUM, data=tail(potsdam,1500), type="l")
# The straight line marks the jump back in time
# Keep only historical data in the overlap time period:
potsdam <- potsdam[!duplicated(potsdam$MESS_DATUM),]


# With many files (>>50), use sleep to avoid getting kicked off the FTP server
#links <- selectDWD(res="daily", var="solar")
#sol <- dataDWD(links, sleep=20) # random waiting time after download (0 to 20 secs)

# Real life example with data completeness check etc:
browseURL("https://github.com/brry/prectemp/blob/master/Code_analysis.R")

}

}
\seealso{
\code{\link{selectDWD}}. \code{\link{readDWD}}, \code{\link{download.file}}.
         Helpful for plotting: \code{berryFunctions::\link[berryFunctions]{monthAxis}},
         see also \code{berryFunctions::\link[berryFunctions]{climateGraph}}
}
\author{
Berry Boessenkool, \email{berry-b@gmx.de}, Jun-Oct 2016
}
\keyword{data}
\keyword{file}
