% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/flip_values.R
\name{flip_values}
\alias{flip_values}
\alias{flip_values_vec}
\title{Flip the values around an origin}
\usage{
flip_values(
  data,
  cols = NULL,
  origin = NULL,
  origin_fn = create_origin_fn(median),
  suffix = "_flipped",
  keep_original = TRUE,
  origin_col_name = ".origin",
  overwrite = FALSE
)

flip_values_vec(data, origin = NULL, origin_fn = create_origin_fn(median))
}
\arguments{
\item{data}{\code{data.frame} or \code{vector}.}

\item{cols}{Names of columns in \code{`data`} to flip values of.}

\item{origin}{Coordinates of the origin to flip around.
A scalar to use in all dimensions (columns)
or a \code{vector} with one scalar per dimension.

\strong{N.B.} Ignored when \code{`origin_fn`} is not \code{NULL}. Remember to set it to \code{NULL}
when passing origin coordinates manually!}

\item{origin_fn}{Function for finding the origin coordinates.

\strong{Input}: Each column will be passed as a \code{vector} in the order of \code{`cols`}.

\strong{Output}: A \code{vector} with one scalar per dimension.

Can be created with \code{\link[rearrr:create_origin_fn]{create_origin_fn()}} if you want to apply
the same function to each dimension.

E.g. \code{`create_origin_fn(median)`} would find the median of each column.

\strong{Built-in functions} are \code{\link[rearrr:centroid]{centroid()}},
\code{\link[rearrr:most_centered]{most_centered()}},
and \code{\link[rearrr:midrange]{midrange()}}}

\item{suffix}{Suffix to add to the names of the generated columns.

Use an empty string (i.e. \code{""}) to overwrite the original columns.}

\item{keep_original}{Whether to keep the original columns. (Logical)

Some columns may have been overwritten, in which case only the newest versions are returned.}

\item{origin_col_name}{Name of new column with the origin coordinates. If \code{NULL}, no column is added.}

\item{overwrite}{Whether to allow overwriting of existing columns. (Logical)}
}
\description{
\Sexpr[results=rd, stage=render]{lifecycle::badge("experimental")}

The values are flipped with the formula \eqn{`x = 2 * c - x`} where \emph{x} is the value and \emph{c} is
the origin coordinate to flip the values around.

The origin can be supplied as coordinates or as a function that returns coordinates. The
latter can be useful when supplying a grouped \code{data.frame} and flipping around e.g. the centroid
of each group. By default the median value in each dimension is used.

The \code{*_vec()} version take and return a vector.

\strong{Example}:

The column values:

\code{c(5, 2, 7, 4, 3, 1)}

and the \code{origin_fn = create_origin_fn(median)}

Changes the values to :

\code{c(2, 5, 0, 3, 4, 6)}
}
\examples{
# Attach packages
library(rearrr)
library(dplyr)
library(ggplot2)

# Set seed
set.seed(1)

# Create a data frame
df <- data.frame(
  "Index" = 1:10,
  "A" = sample(1:10),
  "B" = runif(10),
  "G" = c(
    1, 1, 1, 2, 2,
    2, 3, 3, 3, 3
  ),
  stringsAsFactors = FALSE
)

# Flip values of the columns
flip_values(df$A)
flip_values(df, cols = "A")
flip_values(df, cols = "B", origin = 0.3, origin_fn = NULL, keep_original = FALSE)
flip_values(df,
  cols = c("A", "B"),
  origin = c(3, 0.3),
  origin_fn = NULL,
  suffix = "",
  keep_original = FALSE,
  overwrite = TRUE
)
flip_values(df, cols = c("A", "B"), origin_fn = create_origin_fn(max))

# Grouped by G
df \%>\%
  dplyr::group_by(G) \%>\%
  flip_values(
    cols = c("A", "B"),
    origin_fn = create_origin_fn(median),
    keep_original = FALSE
  )

# Plot A and flipped A

# First flip A around the median and then around the value 3.
df <- df \%>\%
  flip_values(cols = "A", suffix = "_flip_median", origin_col_name = NULL) \%>\%
  flip_values(cols = "A", suffix = "_flip_3", origin = 3,
              origin_fn = NULL, origin_col_name = NULL)

# Plot A and A flipped around its median
ggplot(df, aes(x = Index, y = A)) +
  geom_line(aes(color = "A")) +
  geom_line(aes(y = A_flip_median, color = "Flipped A (median)")) +
  geom_hline(aes(color = "Median A", yintercept = median(A))) +
  theme_minimal()

# Plot A and A flipped around the value 3
ggplot(df, aes(x = Index, y = A)) +
  geom_line(aes(color = "A")) +
  geom_line(aes(y = A_flip_3, color = "Flipped A (3)")) +
  geom_hline(aes(color = "3", yintercept = 3)) +
  theme_minimal()
}
\seealso{
Other mutate functions: 
\code{\link{apply_transformation_matrix}()},
\code{\link{cluster_groups}()},
\code{\link{dim_values}()},
\code{\link{expand_distances_each}()},
\code{\link{expand_distances}()},
\code{\link{roll_values}()},
\code{\link{rotate_2d}()},
\code{\link{rotate_3d}()},
\code{\link{shear_2d}()},
\code{\link{shear_3d}()},
\code{\link{swirl_2d}()},
\code{\link{swirl_3d}()}
}
\author{
Ludvig Renbo Olsen, \email{r-pkgs@ludvigolsen.dk}
}
\concept{mutate functions}
