#' @name exportNextRecordName
#' 
#' @title Generate Next Record Name from a REDCap Database
#' @description To be used by projects with record auto-numbering enabled, 
#' this method exports the next potential record ID for a project. 
#' 
#' @param rcon A REDCap connection object as generated by \code{redcapConnection}
#' @param ... Arguments to be passed to other methods.
#' @param error_handling An option for how to handle errors returned by the API.
#'   see \code{\link{redcap_error}}
#'   
#' @details 
#' It generates the next record name by determining the current maximum numerical 
#' record ID and then incrementing it by one.
#' 
#' @return 
#' Returns the maximum integer record ID + 1.
#' 
#' @section REDCap API Documentation:
#' 
#' NOTE: This method does not create a new record, but merely determines 
#' what the next record name would be.
#' 
#' @section REDCap Version:
#' 8.1.8+ 
#' 
#' @author Xuefei Jia
#' 
#' @references
#' Please refer to your institution's API documentation.
#'
#' Additional details on API parameters are found on the package wiki at
#' \url{https://github.com/nutterb/redcapAPI/wiki/REDCap-API-Parameters}
#'
#' @export

exportNextRecordName <- function(rcon, ...) UseMethod("exportNextRecordName")

#' @rdname exportNextRecordName
#' @export

exportNextRecordName.redcapDbConnection <- function(rcon, ...){
  message("Please accept my apologies.  The exportNextRecordName method for redcapDbConnection objects\n",
          "has not yet been written.  Please consider using the API.")
}

#' @rdname exportNextRecordName
#' @export


exportNextRecordName.redcapApiConnection <- function(rcon, ...,
                                              error_handling = getOption("redcap_error_handling")){
  coll <- checkmate::makeAssertCollection()
  
  checkmate::assert_class(x = rcon,
                          classes = "redcapApiConnection",
                          add = coll)
  
  error_handling <- checkmate::matchArg(x = error_handling,
                                        choices = c("null", "error"),
                                        add = coll)
  
  checkmate::reportAssertions(coll)
  
  body <- list(token = rcon$token, 
               content = 'generateNextRecordName')
  
  x <- httr::POST(url = rcon$url, 
                  body = body, 
                  config = rcon$config)
  
  if (x$status_code != 200) 
  {
    handled <- redcap_error(x, error_handling)
    #* If the export version API method isn't supported by the REDCap instance,
    #* return "5.12.2".  For convenience, we will treat all pre 6.0.0 
    #* versions the same.  The only inefficiency this will generate is 
    #* in choosing when to run `syncUnderscoreCodings`.
    if (is.null(handled)) return("5.12.2")
  }
  
  as.numeric(rawToChar(x$content))
}



