#' @name exportSurveyParticipants
#' @title Exports details of participants for a given survey
#' @description Retrieving dataframe of survey participants managed in REDCap
#' @param rcon A REDCap connection object as generated by \code{redcapConnection}.
#' @param instrument A string type holding the name of "instrument" or survey the participants are managed under
#' @param event A string type holding the name of the event that the instrument belongs to
#' @param ... additional arguments to pass to other methods.
#' @param error_handling An option for how to handle errors returned by the API.
#'   see \code{\link{redcap_error}}
#' @param config \code{list} Additional configuration parameters to pass to 
#'   \code{\link[httr]{POST}}. These are appended to any parameters in 
#'   \code{rcon$config}.
#' @param api_param \code{list} Additional API parameters to pass into the
#'   body of the API call. This provides users to execute calls with options
#'   that may not otherwise be supported by \code{redcapAPI}.
#' 
#' @details REDCap allows the user to manage a list of participants (if they are known) for each survey. This function 
#' pulls this information into a dataframe. The contents of the dataframe are the contact fields of the participants, 
#' the link to the survey for that participant if the participant hasn't completed the survey yet (otherwise, NA), 
#' the participants record id, and other information.
#' 
#' @author Paddy Tobias
#' @export

exportSurveyParticipants <- function(rcon, 
                                     instrument, 
                                     event, ...){
  UseMethod("exportSurveyParticipants")
}

#' @rdname exportSurveyParticipants
#' @export

exportSurveyParticipants.redcapApiConnection <- function(rcon, 
                                                         instrument     = NULL, 
                                                         event          = NULL,
                                                         ...,
                                                         error_handling = getOption("redcap_error_handling"),
                                                         config         = list(), 
                                                         api_param      = list()){
  
   ##################################################################
  # Argument Validation
  
  coll <- checkmate::makeAssertCollection()
  
  checkmate::assert_class(x = rcon, 
                          classes = "redcapApiConnection", 
                          add = coll)
  
  checkmate::assert_character(x = instrument, 
                              len = 1, 
                              any.miss = FALSE, 
                              null.ok = TRUE, 
                              add = coll)
  
  checkmate::assert_character(x = event, 
                              len = 1, 
                              any.miss = FALSE, 
                              null.ok = TRUE, 
                              add = coll)
  
  error_handling <- checkmate::matchArg(x = error_handling, 
                                        choices = c("null", "error"), 
                                        add = coll)
  
  checkmate::assert_list(x = config, 
                         names = "named", 
                         add = coll)
  
  checkmate::assert_list(x = api_param, 
                         names = "named", 
                         add = coll)
  
  checkmate::reportAssertions(coll)
  
  checkmate::assert_subset(x = instrument, 
                           choices = rcon$instruments()$instrument_name, 
                           add = coll)
  
  checkmate::assert_subset(x = event, 
                           choices = rcon$events()$unique_event_name, 
                           add = coll)
  
  checkmate::reportAssertions(coll)
  
   ##################################################################
  # Make API Body List
  
  body <- list(token=rcon$token, 
               instrument = instrument,
               event = event,
               content = 'participantList',
               format = 'csv', 
               returnFormat = 'csv')
  
  body <- body[lengths(body) > 0]
  
   ##################################################################
  # Call the API
  
  response <- makeApiCall(rcon, 
                          body = c(body, api_param), 
                          config = config)
  
  if (response$status_code != 200) return(redcap_error(response, error_handling))
  
  SurveyParticipant <- utils::read.csv(textConnection(as.character(response)), 
                                       stringsAsFactors=FALSE, 
                                       na.strings="")

  return(SurveyParticipant)
}
