#' @name exportUsers
#' @title Export the Users for a Project
#' 
#' @description Retrieve a data frame giving the users, expiration dates,
#' and data access privileges for each user.
#'
#' @param rcon A REDCap connection object as generated by \code{redcapConnection}.
#' @param dates \code{logical(1)} Indicates if the expiration date is converted to a
#'   \code{POSIXct} object.
#' @param labels \code{logical(1)} Indicates if the data export and form access rights are
#'   converted to factor objects.
#' @param form_rights \code{logical(1)} Indicates if the form rights should be 
#'   transformed to one column per form. The API-provided character string
#'   is always returned with the format [form_name]:[access_code] and a comma separating
#'   each form.
#' @param ... Arguments to be passed to other methods.
#' @param error_handling An option for how to handle errors returned by the API.
#'   see \code{\link{redcap_error}}
#' @param config \code{list} Additional configuration parameters to pass to 
#'   \code{\link[httr]{POST}}. These are appended to any parameters in 
#'   \code{rcon$config}.
#' @param api_param \code{list} Additional API parameters to pass into the
#'   body of the API call. This provides users to execute calls with options
#'   that may not otherwise be supported by \code{redcapAPI}.
#'
#' @details
#' For some reason I have yet to identify, some User Tables do not
#' export correctly. In some situations, the fields are all shifted one
#' column to the left and the form names are not always exported.
#' This seems to be more common in projects still in Development mode.
#' I have seen one instance of a project in Production where one user had
#' one more column given than any other user.  If you notice this behavior,
#' please report it to me as it may help me narrow down the source of the problem
#' 
#' @section REDCap API Documentation (6.5.0):
#' This function allows you to export the users for a project
#' 
#' @section REDCap Version:
#' 5.8.2 (Perhaps earlier) 
#' 
#' @section Known REDCap Limitations:
#' None
#' 
#' @return 
#' Returns a data frame. The number of columns in the data frame will depend on your 
#' version of REDCap.
#' \itemize{
#'   \item{\code{username }}{User name}
#'   \item{\code{email }}{The user's e-mail address}
#'   \item{\code{firstname }}{The user's first name}
#'   \item{\code{lastname }}{The user's last name}
#'   \item{\code{expiration }}{The expiration date of the user's access to the project}
#'   \item{\code{data_access_group }}{The data access group the user is assigned to}
#'   \item{\code{data_export }}{The user's data export rights. 0=no access, 
#'     2=De-Identified, 1=Full Data Set}
#'   \item{\code{mobile_app }}{(6.5.0+) Flag for if the user has permissions for the 
#'     mobile application}
#'   \item{\code{mobile_app_download_data }}{(6.5.0+) Flag for if the user may download
#'     data from the mobile app}
#' }
#' 
#' The data frame will have one additional column for each form giving the user's 
#' form-level permissions in the project.  0=no access, 2=read only, 
#' 1=view records/responses and
#' edit records (survey responses are read-only), 3 = edit survey responses
#'
#' @author Benjamin Nutter
#'
#' @references
#' Please refer to your institution's API documentation.
#'
#' Additional details on API parameters are found on the package wiki at
#' \url{https://github.com/vubiostat/redcapAPI/wiki/REDCap-API-Parameters}
#'
#' @export

exportUsers <- function(rcon, 
                        ...){
  UseMethod("exportUsers")
}

#' @rdname exportUsers
#' @export

exportUsers.redcapApiConnection <- function(rcon, 
                                            dates = TRUE, 
                                            labels = TRUE, 
                                            form_rights = TRUE, 
                                            ...,
                                            error_handling = getOption("redcap_error_handling"), 
                                            config = list(), 
                                            api_param = list()){
  
   ##################################################################
  # Argument Validation
  
  coll <- checkmate::makeAssertCollection()
  
  checkmate::assert_class(x = rcon, 
                          classes = "redcapApiConnection", 
                          add = coll)
  
  checkmate::assert_logical(x = dates, 
                            len = 1, 
                            add = coll)
  
  checkmate::assert_logical(x = labels, 
                            len = 1, 
                            add = coll)
  
  checkmate::assert_logical(x = form_rights, 
                            len = 1, 
                            add = coll)
  
  error_handling <- checkmate::matchArg(x = error_handling,
                                        choices = c("null", "error"), 
                                        .var.name = "error_handling", 
                                        add = coll)
  
  checkmate::assert_list(x = config, 
                         names = "named", 
                         add = coll)
  
  checkmate::assert_list(x = api_param, 
                         names = "named", 
                         add = coll)
  
  checkmate::reportAssertions(coll)
  
   ##################################################################
  # Build the Body List 
  
  body <- list(token = rcon$token, 
               content = 'user', 
               format = 'csv', 
               returnFormat = 'csv')
  
  body <- body[lengths(body) > 0]
  
   ##################################################################
  # API Call 
  
  response <- makeApiCall(rcon, 
                          body = c(body, api_param), 
                          config = config)
  
  if (response$status_code != 200){
    redcap_error(response, 
                 error_handling = error_handling)
  }
  
  Users <- utils::read.csv(text = as.character(response), 
                           stringsAsFactors = FALSE,
                           na.strings = "")
  
  Users$forms_export <- 
    sub(",registration[:]\\d{1}.+$", "", Users$forms_export)
  
   ##################################################################
  # convert expiration date to POSIXct class 
  if (dates){
    Users$expiration <- as.POSIXct(Users$expiration, format="%Y-%m-%d")
  } 
  
   ##################################################################
  # Convert user privileges to labels 
  
  if (labels){
    access_var <- REDCAP_USER_TABLE_ACCESS_VARIABLES # defined in redcapDataStructures.R
    # Just in case the variable names ever change
    access_var <- access_var[access_var %in% names(Users)]
    
    Users[access_var] <- 
      lapply(Users[access_var], 
             .exportUsers_labels, 
             type = "project")
  }
  
   ##################################################################
  # Establish columns for the form rights 
  if (form_rights){
    FormAccess <- .exportUsers_separateFormAccess(rcon = rcon, 
                                                  Users$forms, 
                                                  nrow = nrow(Users),
                                                  export = FALSE)
    ExportAccess <- .exportUsers_separateFormAccess(rcon = rcon, 
                                                    form_access = Users$forms_export, 
                                                    nrow = nrow(Users), 
                                                    export = TRUE)
    Users <- 
      cbind(Users, 
            FormAccess, 
            ExportAccess)
    
    if (labels){
      Users[names(FormAccess)] <- 
        lapply(Users[names(FormAccess)], 
               .exportUsers_labels, 
               type = "form")
      
      Users[names(ExportAccess)] <- 
        lapply(Users[names(ExportAccess)], 
               .exportUsers_labels, 
               type = "form_export")
    }
    
  }
  
  Users
}


#####################################################################
# Unexported 

.exportUsers_separateFormAccess <- function(rcon, form_access, nrow, export = FALSE){
  forms <- unique(rcon$metadata()$form_name)
  
  FormAccess <- replicate(rep(NA_character_, nrow), 
                           n = length(forms), 
                           simplify = FALSE)
  FormAccess <- as.data.frame(FormAccess)
  names(FormAccess) <- sprintf("%s_%s_access", 
                               forms,
                               if (export) "export" else "form")
  
  for (i in seq_along(forms)){
    this_form <- forms[i]
    regex <- sprintf("^(|.+)(%s[:]\\d{1})(|.+)$", 
                     this_form)
    this_access <- sub(regex, "\\2", form_access)
    this_access[!grepl(this_form, this_access)] <- NA_character_
    this_access <- sub(this_form, "", this_access)
    this_access <- sub("[:]", "", this_access)
    this_access <- trimws(this_access)
    FormAccess[[i]] <- as.numeric(this_access)
  }
  
  FormAccess  
}

.exportUsers_labels <- function(x, type = c("project", "form", "form_export")){
  switch(type, 
         "project" = factor(x, 
                            levels = 0:1, 
                            labels = c("No Access", 
                                       "Access")), 
         "form" = factor(x, 
                         levels = c(0, 2, 1, 3), 
                         labels = c("No Access", 
                                    "Read Only", 
                                    "View records/responses and edit records (survey responses are read-only)", 
                                    "Edit survey responses")), 
         "form_export" = factor(x, 
                                levels = c(0, 2, 3, 1), 
                                labels = c("No Access", 
                                           "De-Identified", 
                                           "Remove Identifier Fields", 
                                           "Full Data Set")),
         identity())
}
