% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rposterior.R
\name{binpost}
\alias{binpost}
\title{Random sampling from a binomial posterior distribution}
\usage{
binpost(n, prior, ds_bin, param = c("logit", "p"))
}
\arguments{
\item{n}{A numeric scalar. The size of posterior sample required.}

\item{prior}{A function to evaluate the prior, created by
\code{\link{set_bin_prior}}.}

\item{ds_bin}{A numeric list.  Sufficient statistics for inference
  about a binomial probability \eqn{p}.  Contains
\itemize{
  \item {\code{n_raw} : number of raw observations}
  \item {\code{m} : number of threshold exceedances.}
}}

\item{param}{A character scalar.  Only relevant if \code{prior$prior} is a
  (user-supplied) R function.  \code{param} specifies the parameterization
  of the posterior distribution that \code{\link[rust]{ru}} uses for
  sampling.

  If \code{param = "p"} the original parameterization \eqn{p} is
  used.

  If \code{param = "logit"} (the default) then \code{\link[rust]{ru}}
  samples from the posterior for the logit of \eqn{p}, before transforming
  back to the \eqn{p}-scale.

  The latter tends to make the optimizations involved in the
  ratio-of-uniforms algorithm more stable and to increase the probability
  of acceptance, but at the expense of slower function evaluations.}
}
\value{
An object (list) of class \code{"binpost"} with components
  \itemize{
    \item{\code{bin_sim_vals}:} {An \code{n} by 1 numeric matrix of values
      simulated from the posterior for the binomial
      probability \eqn{p}}
    \item{\code{bin_logf}:} {A function returning the log-posterior for
      \eqn{p}.}
    \item{\code{bin_logf_args}:} {A list of arguments to \code{bin_logf}.}
  }
  If \code{prior$prior} is a (user-supplied) R function then this list
  also contains \code{ru_object} the object of class \code{"ru"}
  returned by \code{\link[rust]{ru}}.
}
\description{
Samples from the posterior distribution of the probability \eqn{p}
of a binomial distribution.
}
\details{
If \code{prior$prior == "bin_beta"} then the posterior for \eqn{p}
  is a beta distribution so \code{\link[stats:Beta]{rbeta}} is used to
  sample from the posterior.

  If \code{prior$prior == "bin_mdi"} then
  rejection sampling is used to sample from the posterior with an envelope
  function equal to the density of a
  beta(\code{ds$m} + 1, \code{ds$n_raw - ds$m} + 1) density.

  If \code{prior$prior} is a (user-supplied) R function then
  \code{\link[rust]{ru}} is used to sample from the posterior using the
  generalised ratio-of-uniforms method.
}
\examples{
data(gom)
u <- quantile(gom, probs = 0.65)
ds_bin <- list()
ds_bin$n_raw <- length(gom)
ds_bin$m <- sum(gom > u)
bp <- set_bin_prior(prior = "jeffreys")
temp <- binpost(n = 1000, prior = bp, ds_bin = ds_bin)
graphics::hist(temp$bin_sim_vals, prob = TRUE)

# Setting a beta prior (Jeffreys in this case) by hand
beta_prior_fn <- function(p, ab) {
  return(stats::dbeta(p, shape1 = ab[1], shape2 = ab[2], log = TRUE))
}
jeffreys <- set_bin_prior(beta_prior_fn, ab = c(1 / 2, 1 / 2))
temp <- binpost(n = 1000, prior = jeffreys, ds_bin = ds_bin)
}
\seealso{
\code{\link{set_bin_prior}} for setting a prior distribution
  for the binomial probability \eqn{p}.
}
