% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ee_image.R
\name{ee_as_stars}
\alias{ee_as_stars}
\title{Convert an Earth Engine (EE) image in a stars object}
\usage{
ee_as_stars(
  image,
  region = NULL,
  dsn = NULL,
  via = "drive",
  container = "rgee_backup",
  scale = NULL,
  maxPixels = 1e+09,
  lazy = FALSE,
  public = TRUE,
  add_metadata = TRUE,
  timePrefix = TRUE,
  quiet = FALSE,
  ...
)
}
\arguments{
\item{image}{ee$Image to be converted into a stars object.}

\item{region}{EE Geometry (ee$Geometry$Polygon) which specifies the region
to export. CRS needs to be the same that the argument \code{image}.
Otherwise, it will be forced. If not specified, image bounds are taken.}

\item{dsn}{Character. Output filename. If missing, a temporary file is
created.}

\item{via}{Character. Method to export the image. Two methods are
implemented: "drive", "gcs". See details.}

\item{container}{Character. Name of the folder ('drive') or bucket ('gcs')
to be exported.}

\item{scale}{Numeric. The resolution in meters per pixel. Defaults
to the native resolution of the image.}

\item{maxPixels}{Numeric. The maximum allowed number of pixels in the
exported image. The task will fail if the exported region covers
more pixels in the specified projection. Defaults to 100,000,000.}

\item{lazy}{Logical. If TRUE, a \code{\link[future:sequential]{
future::sequential}} object is created to evaluate the task in the future.
See details.}

\item{public}{Logical. If TRUE, a public link to the image is created.}

\item{add_metadata}{Add metadata to the stars_proxy object. See details.}

\item{timePrefix}{Logical. Add current date and time (\code{Sys.time()}) as
a prefix to export files. This parameter helps to avoid exported files
with the same name. By default TRUE.}

\item{quiet}{Logical. Suppress info message}

\item{...}{Extra exporting argument. See \link{ee_image_to_drive} and
\link{ee_image_to_gcs}.}
}
\value{
A stars-proxy object
}
\description{
Convert an ee$Image in a stars object.
}
\details{
\code{ee_as_stars} supports the download of \code{ee$Images}
by two different options: "drive"
(\href{https://CRAN.R-project.org/package=googledrive}{Google Drive}) and "gcs"
(\href{https://CRAN.R-project.org/package=googleCloudStorageR}{
Google Cloud Storage}). In both cases, \code{ee_as_stars} works as follow:
\itemize{
\item{1. }{A task is started (i.e. \code{ee$batch$Task$start()}) to
move the \code{ee$Image} from Earth Engine to the intermediate container
specified in the argument \code{via}.}
\item{2. }{If the argument \code{lazy} is TRUE, the task will not be
monitored. This is useful to lunch several tasks simultaneously and
calls them later using \code{\link{ee_utils_future_value}} or
\code{\link[future:value]{future::value}}. At the end of this step,
the \code{ee$Image} is stored on the path specified in the argument
\code{dsn}.}
\item{3. }{Finally, if the argument \code{add_metadata} is TRUE, a list
with the following elements is added to the stars-proxy object.
\itemize{
\item{\bold{if via is "drive":}}
\itemize{
\item{\bold{ee_id: }}{Name of the Earth Engine task.}
\item{\bold{drive_name: }}{Name of the Image in Google Drive.}
\item{\bold{drive_id: }}{Id of the Image in Google Drive.}
\item{\bold{drive_download_link: }}{Download link to the image.}
}
}
\itemize{
\item{\bold{if via is "gcs":}}
\itemize{
\item{\bold{ee_id: }}{Name of the Earth Engine task.}
\item{\bold{gcs_name: }}{Name of the Image in Google Cloud Storage.}
\item{\bold{gcs_bucket: }}{Name of the bucket.}
\item{\bold{gcs_fileFormat: }}{Format of the image.}
\item{\bold{gcs_public_link: }}{Download link to the image.}
\item{\bold{gcs_URI: }}{gs:// link to the image.}
}
}
Run \code{attr(stars, "metadata")} to get the list.
}
}

For getting more information about exporting data from Earth Engine, take
a look at the
\href{https://developers.google.com/earth-engine/guides/exporting}{Google
Earth Engine Guide - Export data}.
}
\examples{
\dontrun{
library(rgee)

ee_Initialize(drive = TRUE, gcs = TRUE)
ee_user_info()

# Define an image.
img <- ee$Image("LANDSAT/LC08/C01/T1_SR/LC08_038029_20180810")$
  select(c("B4", "B3", "B2"))$
  divide(10000)

# OPTIONAL display it using Map
Map$centerObject(eeObject = img)
Map$addLayer(eeObject = img, visParams = list(max = 0.4,gamma=0.1))

# Define an area of interest.
geometry <- ee$Geometry$Rectangle(
  coords = c(-110.8, 44.6, -110.6, 44.7),
  proj = "EPSG:4326",
  geodesic = FALSE
)

## drive - Method 01
# Simple
img_02 <- ee_as_stars(
  image = img,
  region = geometry,
  via = "drive"
)

# Lazy
img_02 <- ee_as_stars(
  image = img,
  region = geometry,
  via = "drive",
  lazy = TRUE
)

img_02_result <- img_02 \%>\% ee_utils_future_value()
attr(img_02_result, "metadata") # metadata

## gcs - Method 02
# Simple
img_03 <- ee_as_stars(
  image = img,
  region = geometry,
  container = "rgee_dev",
  via = "gcs"
)

# Lazy
img_03 <- ee_as_stars(
  image = img,
  region = geometry,
  container = "rgee_dev",
  lazy = TRUE,
  via = "gcs"
)

img_03_result <- img_03 \%>\% ee_utils_future_value()
attr(img_03_result, "metadata") # metadata

# OPTIONAL: clean containers
# ee_clean_container(name = "rgee_backup", type = "drive")
# ee_clean_container(name = "rgee_dev", type = "gcs")
}
}
\seealso{
Other image download functions: 
\code{\link{ee_as_raster}()},
\code{\link{ee_as_thumbnail}()},
\code{\link{ee_imagecollection_to_local}()}
}
\concept{image download functions}
