\name{fitted.sparseLTS}
\alias{fitted.optSparseLTSGrid}
\alias{fitted.sparseLTS}
\alias{fitted.sparseLTSGrid}
\title{Extract fitted values from sparse LTS regression models}
\usage{
  \method{fitted}{sparseLTS} (object,
    fit = c("reweighted", "raw", "both"), ...)

  \method{fitted}{sparseLTSGrid} (object, s = NA,
    fit = c("reweighted", "raw", "both"),
    drop = !is.null(s), ...)

  \method{fitted}{optSparseLTSGrid} (object,
    fit = c("reweighted", "raw", "both"), ...)
}
\arguments{
  \item{object}{the model fit from which to extract fitted
  values.}

  \item{s}{an integer vector giving the indices of the
  models for which to extract fitted values.  If \code{fit}
  is \code{"both"}, this can be a list with two components,
  with the first component giving the indices of the
  reweighted fits and the second the indices of the raw
  fits.  The default is to use the optimal model for each
  of the requested estimators.  Note that the optimal
  models may not correspond to the same value of the
  penalty parameter for the reweighted and the raw
  estimator.}

  \item{fit}{a character string specifying which fitted
  values to extract. Possible values are
  \code{"reweighted"} (the default) for the fitted values
  from the reweighted estimator, \code{"raw"} for the
  fitted values from the raw estimator, or \code{"both"}
  for the fitted values from both estimators.}

  \item{drop}{a logical indicating whether to reduce the
  dimension to a vector in case of only one model.}

  \item{\dots}{currently ignored.}
}
\value{
  A numeric vector or matrix containing the requested
  fitted values.
}
\description{
  Extract fitted values from sparse least trimmed squares
  regression models.
}
\examples{
## generate data
# example is not high-dimensional to keep computation time low
library("mvtnorm")
set.seed(1234)  # for reproducibility
n <- 100  # number of observations
p <- 25   # number of variables
beta <- rep.int(c(1, 0), c(5, p-5))  # coefficients
sigma <- 0.5      # controls signal-to-noise ratio
epsilon <- 0.1    # contamination level
Sigma <- 0.5^t(sapply(1:p, function(i, j) abs(i-j), 1:p))
x <- rmvnorm(n, sigma=Sigma)    # predictor matrix
e <- rnorm(n)                   # error terms
i <- 1:ceiling(epsilon*n)       # observations to be contaminated
e[i] <- e[i] + 5                # vertical outliers
y <- c(x \%*\% beta + sigma * e)  # response
x[i,] <- x[i,] + 5              # bad leverage points

## sparse LTS
# fit model
fit <- sparseLTS(x, y, lambda = 0.05, mode = "fraction")
# extract fitted values
fitted(fit)
fitted(fit, fit = "both")

## sparse LTS over a grid of values for lambda
# fit model
frac <- seq(0.25, 0.05, by = -0.05)
fitGrid <- sparseLTSGrid(x, y, lambda = frac, mode = "fraction")
# extract fitted values
fitted(fitGrid)
fitted(fitGrid, fit = "both")
fitted(fitGrid, s = NULL)
fitted(fitGrid, fit = "both", s = NULL)
}
\author{
  Andreas Alfons
}
\seealso{
  \code{\link[stats:fitted.values]{fitted}},
  \code{\link{sparseLTS}}, \code{\link{sparseLTSGrid}}
}
\keyword{regression}

