% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/unif.R
\name{unif}
\alias{unif}
\alias{d_unif_sphere}
\alias{r_unif_sphere}
\alias{w_p}
\title{Uniform distribution on the hypersphere}
\usage{
d_unif_sphere(x, log = FALSE)

r_unif_sphere(n, p)

w_p(p, log = FALSE)
}
\arguments{
\item{x}{locations in \eqn{S^{p-1}} to evaluate the density. Either a
matrix of size \code{c(nx, p)} or a vector of length \code{p}. Normalized
internally if required (with a \code{warning} message).}

\item{log}{flag to indicate if the logarithm of the density (or the
normalizing constant) is to be computed.}

\item{n}{sample size, a positive integer.}

\item{p}{dimension of the ambient space \eqn{R^p} that contains
\eqn{S^{p-1}}. A positive integer.}
}
\value{
Depending on the function:
\itemize{
\item \code{d_unif_sphere}: a vector of length \code{nx} or \code{1} with
the evaluated density at \code{x}.
\item \code{r_unif_sphere}: a matrix of size \code{c(n, p)} with the
random sample.
\item \code{w_p}: the surface area of \eqn{S^{p-1}}.
}
}
\description{
Density and simulation of the uniform distribution on
\eqn{S^{p-1}:=\{\mathbf{x}\in R^p:||\mathbf{x}||=1\}}{
S^{p-1} := \{x \in R^p : ||x|| = 1\}}, \eqn{p\ge 1}. The density is just the
inverse of the surface area of \eqn{S^{p-1}}, given by
\deqn{\omega_p:=2\pi^{p/2}/\Gamma(p/2).}{
\omega_p := 2\pi^{p/2} / \Gamma(p/2).}
}
\details{
If \eqn{p = 1}, then \eqn{S^{0} = \{-1, 1\}} and the "surface area" is
\eqn{2}. The function \code{w_p} is vectorized on \code{p}.
}
\examples{
## Area of S^{p - 1}

# Areas of S^0, S^1, and S^2
w_p(p = 1:3)

# Area as a function of p
p <- 1:20
plot(p, w_p(p = p), type = "o", pch = 16, xlab = "p", ylab = "Area",
     main = expression("Surface area of " * S^{p - 1}), axes = FALSE)
box()
axis(1, at = p)
axis(2, at = seq(0, 34, by = 2))

## Simulation and density evaluation for p = 1, 2, 3

# p = 1
n <- 500
x <- r_unif_sphere(n = n, p = 1)
barplot(table(x) / n)
head(d_unif_sphere(x))

# p = 2
x <- r_unif_sphere(n = n, p = 3)
plot(x)
head(d_unif_sphere(x))

# p = 3
x <- r_unif_sphere(n = n, p = 3)
rgl::plot3d(x)
head(d_unif_sphere(x))
}
\author{
Eduardo García-Portugués, Davy Paindaveine, and Thomas Verdebout.
}
