#' Deploy a single document
#'
#' @description
#' Deploys a single R Markdown, Quarto document, or other file (e.g. `.html` or
#' `.pdf`).
#'
#' When deploying an `.Rmd`, `.Qmd`, or `.html`, `deployDoc()` will attempt to
#' automatically discover dependencies using [rmarkdown::find_external_resources()],
#' and include an `.Rprofile` if present. If you find that the document is
#' missing dependencies, either specify the dependencies explicitly in the
#' document (see [rmarkdown::find_external_resources()] for details), or call
#' [deployApp()] directly and specify your own file list in `appFiles`.
#'
#' @param doc Path to the document to deploy.
#' @param ... Additional arguments to [deployApp()]. Do not supply `appDir`,
#'   `appFiles`, or `appPrimaryDoc`; these three parameters are automatically
#'   generated by `deployDoc` from the document.
#' @inheritParams deployApp
#' @family Deployment functions
#' @export
#' @examples
#' \dontrun{
#' deployDoc("my-report.Rmd")
#' deployDoc("static-file.html")
#' }
deployDoc <- function(doc, ..., logLevel = c("normal", "quiet", "verbose")) {
  logLevel <- arg_match(logLevel)

  doc <- standardizeSingleDocDeployment(doc, quiet = logLevel == "quiet")
  deployApp(
    appDir = doc$appDir,
    appPrimaryDoc = doc$appPrimaryDoc,
    appFiles = doc$appFiles,
    ...,
    logLevel = logLevel
  )
}

standardizeSingleDocDeployment <- function(path,
                                           quiet = FALSE,
                                           error_call = caller_env(),
                                           error_arg = caller_arg(path)) {
  check_installed(
    "rmarkdown",
    version = "0.5.2",
    reason = "to deploy individual R Markdown documents"
  )
  check_file(path, error_call = error_call, error_arg = error_arg)
  path <- normalizePath(path)

  if (isShinyRmd(path)) {
    # deploy entire directory
    appFiles <- NULL
  } else if (isStaticFile(path)) {
    taskStart(quiet, "Discovering document dependencies...")
    resources <- rmarkdown::find_external_resources(path)
    taskComplete(quiet, "Document dependencies discovered")

    appFiles <- c(basename(path), resources$path)

    if (file.exists(file.path(dirname(path), ".Rprofile"))) {
      appFiles <- c(appFiles, ".Rprofile")
    }
    appFiles
  } else {
    # deploy just the file
    appFiles <- basename(path)
  }

  list(
    appDir = normalizePath(dirname(path)),
    appPrimaryDoc = basename(path),
    appFiles = appFiles
  )
}

isStaticFile <- function(path) {
  ext <- tolower(tools::file_ext(path))
  ext %in% c("rmd", "qmd", "html", "htm")
}
