#' TWIT
#'
#' @description Base function responsible for formulating GET and
#'   POST requests to Twitter API's.
#'
#' @param get Logical with the default, \code{get = TRUE},
#'   indicating whether the provided url should be passed along via
#'   a GET or POST request.
#' @param url Character vector designed to operate like
#'   parse_url and build_url functions in the httr package.
#'   The easiest way to do this is to work through
#'   the call-specific functions as they are designed to simplify
#'   the process. However, if one were interested in reverse-
#'   engingeering such a thing, I would recommend checking out
#'   \code{make_url}.
#' @param \dots Further named parameters, such as config, token,
#'   etc, passed on to modify_url in the httr package.
#' @param timeout Numeric, used only when streaming tweets,
#'   specifying the number of seconds to stream tweets.
#' @param filename Character, used only when streaming tweets,
#'   name of file to save json tweets object.
#' @param catch_error Logical indicating whether to call stop for
#'   status function following GET or POST requests.
#' @note Occasionally Twitter does recommend using POST requests
#'   for data retrieval calls. This is usually the case when requests
#'   can involve long strings (containing up to 100 user_ids). For
#'   the most part, or at least for any function-specific requests
#'   (e.g., \code{get_friends}, take reflect these changes.
#' @examples
#' \dontrun{
#' tokens <- get_tokens()
#'
#' params <- list(q = "rstats", result_type = "recent")
#'
#' url <- make_url(restapi = TRUE,
#'   "search/tweets",
#'   param = params)
#'
#' r <- TWIT(get = TRUE, url,
#'   config = tokens[[1]])
#' }
#' @return json response object
#' @import httr
#' @export
TWIT <- function(get = TRUE, url, ..., timeout = NULL,
                 filename = NULL, catch_error = TRUE) {

  if (get) {
    resp <- GET(url, ...)
  } else {
    if (!is.null(timeout)) {
      if (!is.null(filename)) {
        tryCatch(
          POST(url, ...,
            timeout(timeout),
            write_disk(filename, overwrite = TRUE)),
            error = function(e) return(invisible()))
        return(invisible())
      }
    } else {
      resp <- POST(url, ...)
    }
  }

  if (!exists("resp")) {
    stop(paste0(
        "no response object returned. ",
        "Please try again later or check httr request."),
      call. = FALSE)
  }

  if (catch_error) stop_for_status(resp)

  resp
}

#' make_url
#'
#' @param restapi logical Default \code{restapi = TRUE}
#'   indicates the provided URL components should be
#'   specify Twitter's REST API. Set this to FALSE if you wish
#'   to make a request URL designed for Twitter's streaming api.
#' @param query Twitter's subsetting/topic identifiers.
#'   Although the httr package refers to this as "path",
#'   query is used here to maintain consistency with
#'   Twitter API's excellent documentation.
#' @param param Additional parameters (arguments) passed
#'   along. If none, NULL (default).
#' @param version Twitter API version number. Defaults to most
#'   recent version, which at the current time is
#'   \code{version = "1.1"}. Functions not tested on older
#'   versions.
#' @return URL used in httr call.
#' @export
make_url <- function(restapi = TRUE, query, param = NULL, version = "1.1") {
  if (restapi) {
    hostname <- "api.twitter.com"
  } else {
    hostname <- "stream.twitter.com"
  }

  alst <- structure(
    list(
      scheme = "https",
      hostname = hostname,
      port = NULL,
      path = paste0(version, "/", query, ".json"),
      query = param,
      params = NULL,
      fragment = NULL,
      username = NULL,
      password = NULL),
    class = "url")

  alst
}
