# ========================================================================
# rvRhat; rvneff; rvnchains - convenience functions for some attributes
# ========================================================================
#



#' R-hat Convergence Diagnostic
#' 
#' Retrieves the R-hat convergence diagnostic for each component of the
#' argument
#' 
#' The R-hat values are assumed to be saved as attributes.  If they are not
#' available, \code{NA} will be returned.
#' 
#' R-hat is computed by programs such as Umacs and R2WinBUGS.
#' 
#' @param x an object
#' @return Vector of numbers, \code{NA} if R-hat is not available.
#' @author Jouni Kerman \email{jouni@@kerman.com}
#' @references Kerman, J. and Gelman, A. (2007). Manipulating and Summarizing
#' Posterior Simulations Using Random Variable Objects. Statistics and
#' Computing 17:3, 235-244.
#' 
#' See also \code{vignette("rv")}.
#' @keywords classes
#' @export rvRhat
rvRhat <- function(x) {
  unlist(rvattr(x, "Rhat"))
}



#' Number of Effective Draws in Each Component of a Random Variable
#' 
#' Retrieves the number of effective draws in each component of the argument.
#' 
#' The number of effective draws is supposed to be saved by the simulation
#' generating program (e.g. WinBUGS via R2WinBUGS).
#' 
#' @param x an rv object
#' @return A numeric object of the same length as the argument \code{x}.
#' @author Jouni Kerman \email{jouni@@kerman.com}
#' @references Kerman, J. and Gelman, A. (2007). Manipulating and Summarizing
#' Posterior Simulations Using Random Variable Objects. Statistics and
#' Computing 17:3, 235-244.
#' 
#' See also \code{vignette("rv")}.
#' @keywords classes
#' @examples
#' 
#'   #
#' 
#' @export rvneff
rvneff <- function(x) {
  unlist(rvattr(x, "n.eff"))
}



#' Number of Markov Chains Used to Generate Simulations of a Random Vector
#' 
#' Retrieves the number of mcmc chains in each components of the argument.
#' 
#' Assumes that the rv object was generated by a MCMC process.  Umacs and
#' R2WinBUGS are compatible.
#' 
#' @param x an rv object (supposed to be generated by a MCMC process)
#' @author Jouni Kerman \email{jouni@@kerman.com}
#' @seealso \code{\link{as.rv.bugs}}
#' @references Kerman, J. and Gelman, A. (2007). Manipulating and Summarizing
#' Posterior Simulations Using Random Variable Objects. Statistics and
#' Computing 17:3, 235-244.
#' 
#' See also \code{vignette("rv")}.
#' @keywords classes
#' @examples
#' 
#'   #
#' 
#' @export rvnchains
rvnchains <- function(x) {
  unlist(rvattr(x, "n.chains"))
}

