#---------------------------------------------------------------------------
#
#   This file holds the S4 definition for the constructors of downLogs
#   generic and methods. This is used for a collection or population of
#   "downLogs"
#
#   Constructors include signatures for...
#     1. numeric, matrix: number of logs to generate within an sp bbox
#        matrix--uses method (5) after calling sampleLogs for the data frame
#     2. numeric, missing: sampling from within a bounded area as defined 
#        by the xlim & ylim args--object is the number of logs to be
#        generated--uses method (1) after turning limits into a matrix
#     3. numeric, bufferedTract: generate logs with centers lying inside 
#        the buffer of a bufferedTract object--calls method (1) with
#        the buffer as matrix
#     4. list, missing: passing a list of already created "downLog" 
#        objects--good for an existing population--creates the object directly
#     5. data.frame, missing: allows one to pass a data frame of logs that
#        have been generated by the sampleLogs function--turns the data frame
#        into a list and calls method (4) directly
#
#   From the above, we can see that all methods lead to method (4), the list
#   method; i.e., 2-->1-->5-->4 & 3-->1-->5-->4. Therefore, any real additions
#   on what is finally generated should be to method 4, which will affect all
#   the other constructors.
#
#   Note: In 3. I have put an explicit check in to make sure units desired for
#         logs are the same a those for the bufferedTract; unfortunately, this
#         is not possible for 1. or 2., so the user is on their own.
#
#   Also included at the end is the function .statsDownLogs() for calculating
#   basic statistics on the log collection.
#
#   Returns...
#     a valid "downLogs" object.
#
#Author...									Date: 18-Aug-2010
#	Jeffrey H. Gove
#	USDA Forest Service
#	Northern Research Station
#	271 Mast Road
#	Durham, NH 03824
#	jhgove@unh.edu
#	phone: 603-868-7667	fax: 603-868-7604
#---------------------------------------------------------------------------
#   generic definition...
#
#if (!isGeneric("downLogs")) 
  setGeneric('downLogs',  
             function(object, container, ...) standardGeneric('downLogs'),
             signature = c('object', 'container')
            )







          
#================================================================================
#  1. method for function for synthetic simulation of logs; this takes the number
#     of logs to generate and a bbox matrix from wihtin which the log centers are
#     drawn...
#
setMethod('downLogs',
          signature(object = 'numeric', container='matrix'),
function(object,                            #number of logs to generate
         container,                         #within these bounds--a bbox matrix
         units = 'metric',
         buttDiams = c(8, 40),              #cm for object construction!
         topDiams = c(0, 0.9),              #proportion multiplier
         logLens = c(1,10),
         logAngles = c(0, 2*pi),
         solidTypes = c(1,10),
         species = .StemEnv$species,
         ...
        )
{
#------------------------------------------------------------------------------
#
#   first some checks...
#
    if(object < 1 || length(object)>1 || !is.numeric(object))
      stop('"object" (number of logs) must be a positive numeric scalar!')
    
#   make sure we have a valid "bbox" type matrix...    
    stopifnot(bboxCheck(container))

    
    numLogs = object                #a little easier to understand
    sampleRect = container          #again

#
#   draw the population of logs using the general log sampling function...
#
    slogs = sampleLogs(numLogs, buttDiams=buttDiams, topDiams=topDiams,
                       logLens = logLens, logAngles=logAngles,
                       solidTypes = solidTypes, species=species,
                       sampleRect = sampleRect,
                       ... 
                      )
    dlo = downLogs(slogs, units=units, ...) #call the data.frame method
    
    return(dlo)

}   #downLogs method for synthetic
)   #setMethod
    






          
#================================================================================
#  2. method for function for synthetic simulation of logs within a set of
#     xlim & ylim coordinate--just turn these into a matrix and call the
#     matrix container routine...
#
setMethod('downLogs',
          signature(object = 'numeric', container='missing'),
function(object,                            #number of logs to generate
         xlim = c(0,100),                   #within these bounds
         ylim = c(0,50),
         units = 'metric',
         ...
        )
{
#------------------------------------------------------------------------------
#
    numLogs = object
    
#   create bbox rectangle as a matrix from limits...
    sampleRect = matrix(c(xlim, ylim),
                        nrow = 2, byrow = TRUE,
                        dimnames = list(c('x','y'),c('min','max'))
                       )
    dls = downLogs(numLogs, sampleRect, units=units, ...)

    return(dls)
}   #downLogs method for synthetic with x,y limit vectors bbox
)   #setMethod







          
#================================================================================
#  3. method for function for synthetic simulation of logs from within a
#     bufferedTract internal bbox--just call the matrix container method using
#     the buffer for drawing the log centers...
#
setMethod('downLogs',
          signature(object = 'numeric', container='bufferedTract'),
function(object,                                 #number of logs to generate
         container,                              #within these bounds
         units = 'metric',
         ...
        )
{
#------------------------------------------------------------------------------
#  
    numLogs = object

#
#   quick check to make sure all logs are same units as bufferedTract...
#
    for(i in seq_len(numLogs))
      if(units != container@units)
        stop('downLogs: Units for logs and container do not match!')
    
    dls = downLogs(numLogs, container@bufferRect, units=units, ...)

    return(dls)
}   #downLogs method for synthetic with bufferedtract bbox
)   #setMethod





          
#================================================================================
#  4. method for function for a previously made collection of downLog objects, 
#     stored within a list; note that the constructor will check all the individual
#     logs to make sure they are in fact each valid downLog objects; in addition,
#     the diameters should already have been correctly converted to units of length
#     within these original downLog objects...
#
setMethod('downLogs',
          signature(object = 'list', container='missing'),
function(object,
         description = '',
         ...
        )
{
#------------------------------------------------------------------------------
#
    logs = object
    numLogs = length(logs)
    if(numLogs < 1)
      stop('"object" must be at least one log in the list')

    names(logs) = paste('log',1:numLogs,sep='.')

#
#   group all polygons into one SpatialPolygons to get the overall bbox...
#
    sp = vector('list', numLogs)
    for(i in seq_len(numLogs)) 
      sp[[i]] = logs[[i]]@spLog@polygons$pgsLog
    
    sps = SpatialPolygons(sp)
    bbox = bbox(sps)


    stats = .statsDownLogs(logs)
    
#
#   create the object directly, let the constructor check the rest...
#    
    dlo = new('downLogs', logs=logs, bbox=bbox, units=logs[[1]]@units,
              stats=stats, description=description)
    
    return(dlo)
}   #downLogs method for list
)   #setMethod






          
#================================================================================
#  5. method for function for a previously made collection of logs generated from
#     the sampleLogs() function, which returns a data frame object...
#
setMethod('downLogs',
          signature(object = 'data.frame', container='missing'),
function(object,
         units = 'metric',
         ...
        )
{
#------------------------------------------------------------------------------
#   assuming a data frame in the form returned by sampleLogs() here, a simple
#   check for this follows...
#
    if(!length(na.omit(match(names(object), .StemEnv$sampleLogsNames))) == 9)
      stop('Names of data frame with logs must match those returned from "sampleLogs"!')
    
    logs = object
    numLogs = nrow(logs)

#
#   downLog constructor will convert from in/cm to feet/meters...
#
    dll = vector('list', numLogs)
    for(i in seq_len(numLogs))  
      dll[[i]] = downLog(buttDiam = logs[i,'buttDiam'],    #in cm or inches!
                         topDiam = logs[i,'topDiam'],      #ibid
                         logLen = logs[i, 'logLen'],
                         solidType = logs[i, 'solidType'],
                         logAngle = logs[i, 'logAngle'],
                         centerOffset = c(x=logs[i, 'x'], y=logs[i, 'y']),
                         species = logs[i, 'species'],
                         units = units,
                         ...   #not individualized, applies to all logs
                        ) 
      
    dls = downLogs(dll, units=units, ...)  #call the list method
    
    return(dls)
}   #downLogs method for data.frame
)   #setMethod





#================================================================================
.statsDownLogs = function(logs) 
{
#------------------------------------------------------------------------------
#   calculates summary statistics on the logs in the list...
#------------------------------------------------------------------------------
    attrNames = c('volume', 'length', 'surfaceArea', 'coverageArea','biomass','carbon')
    numLogs = length(logs)
    nvars = length(attrNames)
    logAttr = matrix(NA, nrow=numLogs, ncol=nvars) #log attributes
    colnames(logAttr) = attrNames
    for(i in seq_len(numLogs)) {
      logAttr[i,1] = logs[[i]]@logVol
      logAttr[i,2] = logs[[i]]@logLen
      logAttr[i,3] = logs[[i]]@surfaceArea
      logAttr[i,4] = logs[[i]]@coverageArea
      logAttr[i,5] = logs[[i]]@biomass
      logAttr[i,6] = logs[[i]]@carbon
    }

#
#   carbon and biomass can be all NAs and will throw warnings of no non-NA values when
#   calculated some stats (like min) below; so keep it quiet...
#
    suppressWarnings({
    
    stats = data.frame(matrix(NA, nrow=6, ncol=nvars)) #stats as rows, vars as columns
    colnames(stats) = attrNames
    rownames(stats) = c('mean','total','sd','var', 'min','max')
    #stats[1,] = apply(logAttr, 2, mean, na.rm=TRUE)
    stats[1,] = colMeans(logAttr, na.rm=TRUE)
    stats[2,] = colSums(logAttr, na.rm=TRUE)
    stats[3,] = apply(logAttr, 2, sd, na.rm=TRUE)
    stats[4,] = apply(logAttr, 2, var, na.rm=TRUE)
    stats[5,] = apply(logAttr, 2, min, na.rm=TRUE)
    stats[6,] = apply(logAttr, 2, max, na.rm=TRUE)

    }) #suppressWarnings

#
#   and clean these up if necessary...
#
    if(all(is.na(logAttr[,'biomass'])))
      stats[,'biomass'] = NA
    if(all(is.na(logAttr[,'carbon'])))
      stats[,'carbon'] = NA
 
    return(stats)
}   #statsDownLogs

#showMethods('downLogs')

