
// [[Rcpp::depends(RcppArmadillo)]]
#include <RcppArmadillo.h>
#include <Rmath.h>
#include <R.h>
using namespace Rcpp;

// [[Rcpp::export(name = ".linInterp")]]
arma::vec linInterp(arma::vec x, arma::vec xGrid, arma::vec yGrid, bool equalSpaces = false) {

  // Use Armadillo's interpolation function? Is slower because assumes no equal spacings
  if (equalSpaces) {

    // Length of grid - 1
    int N = xGrid.n_elem - 1;

    // Initial and final values
    double a = xGrid(0), ba = xGrid(N) - a;

    // Closest lower and upper xGrid to x
    arma::uvec lower = arma::conv_to<arma::uvec>::from(floor((x - a) / ba * N));
    arma::uvec upper = lower + 1;

    // Restrict lower and upper indexes
    lower = arma::clamp(lower, 0, N);
    upper = arma::clamp(upper, 0, N);

    // Weighted linear interpolation
    arma::vec output = (yGrid(lower) % (xGrid(upper) - x) + yGrid(upper) % (x - xGrid(lower)));
    output /=  xGrid(upper) - xGrid(lower);

    return output;

  } else {

    // Create output
    arma::vec output(xGrid.n_elem);

    // Armadillo built-in interpolation (assumes no equal spacings)
    interp1(xGrid, yGrid, x, output, "linear");

    return output;

  }

}

// [[Rcpp::export(name = ".besselIExponScaled")]]
arma::vec besselIExponScaled(arma::vec x, int nu = 0, int maxK = 10, bool equalSpaces = false) {

  // Output
  arma::vec output = arma::zeros(x.n_elem);

  // Indexes of regions
  arma::uvec indGrid = find(x < 13);
  arma::uvec indAsymp = find(x >= 13);

  // Grid for [0, 10)
  if (indGrid.n_elem > 0) {

    if (nu == 0) {

      // Obtained from R
      arma::vec xGrid = arma::linspace<arma::vec>(0, 13, 30001);
      arma::vec yGrid = {1,0.999566807466,0.999133896396,0.998701266585,0.998268917832,0.997836849933,0.997405062685,0.996973555886,0.996542329334,0.996111382826,0.99568071616,0.995250329134,0.994820221546,0.994390393195,0.99396084388,0.993531573398,0.993102581548,0.99267386813,0.992245432943,0.991817275785,0.991389396457,0.990961794758,0.990534470487,0.990107423445,0.989680653431,0.989254160246,0.98882794369,0.988402003563,0.987976339666,0.9875509518,0.987125839765,0.986701003364,0.986276442396,0.985852156664,0.985428145969,0.985004410112,0.984580948896,0.984157762122,0.983734849592,0.98331221111,0.982889846476,0.982467755495,0.982045937967,0.981624393698,0.981203122488,0.980782124142,0.980361398463,0.979940945255,0.97952076432,0.979100855464,0.978681218488,0.978261853199,0.977842759399,0.977423936894,0.977005385488,0.976587104985,0.97616909519,0.975751355909,0.975333886946,0.974916688107,0.974499759196,0.97408310002,0.973666710384,0.973250590095,0.972834738957,0.972419156777,0.972003843362,0.971588798518,0.971174022051,0.970759513769,0.970345273477,0.969931300983,0.969517596095,0.969104158619,0.968690988363,0.968278085135,0.967865448742,0.967453078992,0.967040975693,0.966629138653,0.966217567681,0.965806262586,0.965395223175,0.964984449257,0.964573940643,0.964163697139,0.963753718556,0.963344004703,0.96293455539,0.962525370425,0.96211644962,0.961707792783,0.961299399725,0.960891270256,0.960483404187,0.960075801327,0.959668461488,0.959261384479,0.958854570114,0.958448018201,0.958041728552,0.95763570098,0.957229935295,0.956824431308,0.956419188832,0.956014207679,0.95560948766,0.955205028588,0.954800830276,0.954396892534,0.953993215177,0.953589798018,0.953186640868,0.952783743541,0.95238110585,0.951978727608,0.95157660863,0.951174748728,0.950773147717,0.950371805409,0.94997072162,0.949569896164,0.949169328854,0.948769019505,0.948368967932,0.94796917395,0.947569637373,0.947170358016,0.946771335695,0.946372570225,0.945974061422,0.9455758091,0.945177813076,0.944780073165,0.944382589184,0.943985360948,0.943588388274,0.943191670979,0.942795208878,0.942399001789,0.942003049528,0.941607351912,0.941211908759,0.940816719886,0.94042178511,0.940027104248,0.939632677118,0.939238503538,0.938844583327,0.938450916301,0.938057502279,0.93766434108,0.937271432523,0.936878776424,0.936486372604,0.936094220882,0.935702321076,0.935310673005,0.934919276489,0.934528131347,0.934137237399,0.933746594465,0.933356202364,0.932966060916,0.932576169942,0.932186529261,0.931797138695,0.931407998063,0.931019107187,0.930630465887,0.930242073983,0.929853931298,0.929466037652,0.929078392867,0.928690996764,0.928303849165,0.927916949891,0.927530298764,0.927143895607,0.926757740241,0.926371832489,0.925986172174,0.925600759117,0.925215593141,0.92483067407,0.924446001726,0.924061575932,0.923677396512,0.923293463289,0.922909776086,0.922526334727,0.922143139035,0.921760188835,0.921377483951,0.920995024207,0.920612809426,0.920230839435,0.919849114056,0.919467633115,0.919086396436,0.918705403845,0.918324655166,0.917944150225,0.917563888847,0.917183870858,0.916804096082,0.916424564347,0.916045275478,0.9156662293,0.91528742564,0.914908864324,0.914530545179,0.914152468031,0.913774632706,0.913397039032,0.913019686835,0.912642575943,0.912265706183,0.911889077381,0.911512689366,0.911136541965,0.910760635006,0.910384968315,0.910009541723,0.909634355056,0.909259408142,0.908884700811,0.90851023289,0.908136004209,0.907762014595,0.907388263878,0.907014751887,0.906641478451,0.906268443399,0.90589564656,0.905523087765,0.905150766842,0.904778683621,0.904406837933,0.904035229607,0.903663858474,0.903292724363,0.902921827106,0.902551166532,0.902180742473,0.901810554758,0.90144060322,0.901070887689,0.900701407997,0.900332163974,0.899963155451,0.899594382262,0.899225844237,0.898857541207,0.898489473006,0.898121639465,0.897754040416,0.897386675692,0.897019545124,0.896652648546,0.896285985791,0.89591955669,0.895553361078,0.895187398786,0.894821669649,0.8944561735,0.894090910171,0.893725879497,0.893361081312,0.892996515448,0.892632181741,0.892268080024,0.891904210131,0.891540571897,0.891177165156,0.890813989743,0.890451045493,0.890088332239,0.889725849818,0.889363598064,0.889001576812,0.888639785898,0.888278225158,0.887916894426,0.887555793539,0.887194922332,0.886834280641,0.886473868302,0.886113685152,0.885753731027,0.885394005763,0.885034509197,0.884675241165,0.884316201505,0.883957390053,0.883598806645,0.883240451121,0.882882323316,0.882524423068,0.882166750215,0.881809304594,0.881452086044,0.881095094401,0.880738329504,0.880381791192,0.880025479302,0.879669393673,0.879313534143,0.878957900551,0.878602492736,0.878247310537,0.877892353792,0.877537622341,0.877183116023,0.876828834678,0.876474778144,0.876120946262,0.87576733887,0.87541395581,0.875060796921,0.874707862043,0.874355151017,0.874002663682,0.873650399878,0.873298359448,0.872946542231,0.872594948068,0.8722435768,0.871892428269,0.871541502314,0.871190798778,0.870840317502,0.870490058328,0.870140021096,0.86979020565,0.86944061183,0.86909123948,0.86874208844,0.868393158553,0.868044449662,0.867695961609,0.867347694236,0.866999647387,0.866651820903,0.866304214629,0.865956828407,0.86560966208,0.865262715492,0.864915988486,0.864569480906,0.864223192595,0.863877123397,0.863531273156,0.863185641716,0.862840228921,0.862495034615,0.862150058643,0.861805300849,0.861460761077,0.861116439174,0.860772334982,0.860428448348,0.860084779116,0.859741327131,0.85939809224,0.859055074286,0.858712273117,0.858369688576,0.858027320511,0.857685168767,0.857343233189,0.857001513625,0.85666000992,0.856318721921,0.855977649474,0.855636792426,0.855296150622,0.854955723912,0.85461551214,0.854275515154,0.853935732802,0.85359616493,0.853256811386,0.852917672018,0.852578746672,0.852240035197,0.851901537441,0.851563253251,0.851225182476,0.850887324964,0.850549680562,0.85021224912,0.849875030486,0.849538024509,0.849201231037,0.848864649919,0.848528281004,0.848192124142,0.847856179181,0.84752044597,0.84718492436,0.8468496142,0.846514515339,0.846179627627,0.845844950915,0.845510485051,0.845176229886,0.844842185271,0.844508351056,0.84417472709,0.843841313225,0.843508109312,0.843175115201,0.842842330742,0.842509755788,0.842177390189,0.841845233796,0.841513286461,0.841181548035,0.84085001837,0.840518697317,0.840187584728,0.839856680455,0.839525984351,0.839195496266,0.838865216054,0.838535143567,0.838205278657,0.837875621176,0.837546170978,0.837216927915,0.836887891841,0.836559062607,0.836230440068,0.835902024076,0.835573814485,0.835245811148,0.834918013919,0.834590422651,0.834263037199,0.833935857415,0.833608883155,0.833282114272,0.83295555062,0.832629192054,0.832303038428,0.831977089597,0.831651345415,0.831325805738,0.831000470419,0.830675339314,0.830350412278,0.830025689167,0.829701169835,0.829376854138,0.829052741931,0.828728833071,0.828405127412,0.828081624811,0.827758325124,0.827435228206,0.827112333914,0.826789642104,0.826467152633,0.826144865357,0.825822780132,0.825500896816,0.825179215265,0.824857735336,0.824536456886,0.824215379772,0.823894503851,0.823573828982,0.82325335502,0.822933081825,0.822613009253,0.822293137162,0.821973465411,0.821653993856,0.821334722357,0.821015650771,0.820696778958,0.820378106774,0.820059634079,0.819741360731,0.81942328659,0.819105411513,0.81878773536,0.81847025799,0.818152979262,0.817835899036,0.81751901717,0.817202333524,0.816885847958,0.816569560332,0.816253470505,0.815937578337,0.815621883688,0.815306386418,0.814991086388,0.814675983457,0.814361077487,0.814046368338,0.813731855869,0.813417539943,0.81310342042,0.81278949716,0.812475770026,0.812162238877,0.811848903576,0.811535763984,0.811222819961,0.81091007137,0.810597518073,0.810285159931,0.809972996805,0.809661028559,0.809349255054,0.809037676152,0.808726291715,0.808415101607,0.808104105689,0.807793303824,0.807482695875,0.807172281704,0.806862061174,0.80655203415,0.806242200492,0.805932560066,0.805623112733,0.805313858359,0.805004796805,0.804695927936,0.804387251615,0.804078767706,0.803770476074,0.803462376582,0.803154469094,0.802846753475,0.802539229588,0.802231897299,0.801924756472,0.801617806972,0.801311048662,0.801004481409,0.800698105077,0.800391919531,0.800085924636,0.799780120258,0.799474506261,0.799169082512,0.798863848876,0.798558805218,0.798253951405,0.797949287301,0.797644812774,0.797340527689,0.797036431912,0.796732525309,0.796428807748,0.796125279094,0.795821939213,0.795518787974,0.795215825241,0.794913050883,0.794610464766,0.794308066757,0.794005856723,0.793703834532,0.793402000051,0.793100353147,0.792798893688,0.792497621542,0.792196536576,0.791895638658,0.791594927657,0.791294403439,0.790994065874,0.790693914829,0.790393950173,0.790094171775,0.789794579502,0.789495173224,0.789195952809,0.788896918126,0.788598069044,0.788299405432,0.788000927159,0.787702634095,0.787404526108,0.787106603069,0.786808864846,0.786511311309,0.786213942328,0.785916757773,0.785619757514,0.78532294142,0.785026309362,0.78472986121,0.784433596834,0.784137516105,0.783841618893,0.783545905069,0.783250374503,0.782955027066,0.78265986263,0.782364881064,0.78207008224,0.78177546603,0.781481032305,0.781186780935,0.780892711792,0.780598824749,0.780305119676,0.780011596445,0.779718254929,0.779425094998,0.779132116526,0.778839319383,0.778546703443,0.778254268578,0.77796201466,0.777669941562,0.777378049156,0.777086337315,0.776794805912,0.776503454819,0.77621228391,0.775921293058,0.775630482135,0.775339851016,0.775049399573,0.77475912768,0.77446903521,0.774179122038,0.773889388037,0.77359983308,0.773310457042,0.773021259796,0.772732241218,0.77244340118,0.772154739558,0.771866256225,0.771577951057,0.771289823927,0.771001874711,0.770714103283,0.770426509518,0.770139093291,0.769851854477,0.769564792952,0.76927790859,0.768991201267,0.768704670858,0.768418317238,0.768132140284,0.767846139872,0.767560315876,0.767274668173,0.766989196638,0.766703901149,0.76641878158,0.766133837809,0.765849069711,0.765564477163,0.765280060042,0.764995818224,0.764711751586,0.764427860005,0.764144143357,0.76386060152,0.763577234371,0.763294041787,0.763011023645,0.762728179822,0.762445510197,0.762163014646,0.761880693047,0.761598545278,0.761316571217,0.761034770741,0.760753143729,0.760471690059,0.760190409609,0.759909302257,0.759628367881,0.759347606361,0.759067017574,0.758786601399,0.758506357716,0.758226286402,0.757946387337,0.757666660399,0.757387105468,0.757107722423,0.756828511144,0.756549471508,0.756270603397,0.75599190669,0.755713381265,0.755435027004,0.755156843785,0.754878831489,0.754600989996,0.754323319185,0.754045818937,0.753768489132,0.753491329651,0.753214340373,0.75293752118,0.752660871952,0.75238439257,0.752108082914,0.751831942866,0.751555972306,0.751280171115,0.751004539175,0.750729076367,0.750453782572,0.750178657671,0.749903701547,0.74962891408,0.749354295152,0.749079844646,0.748805562443,0.748531448424,0.748257502472,0.74798372447,0.747710114298,0.74743667184,0.747163396978,0.746890289594,0.746617349572,0.746344576792,0.746071971139,0.745799532496,0.745527260744,0.745255155767,0.744983217448,0.744711445671,0.744439840318,0.744168401273,0.743897128419,0.74362602164,0.74335508082,0.743084305841,0.742813696589,0.742543252946,0.742272974797,0.742002862025,0.741732914515,0.741463132152,0.741193514818,0.7409240624,0.74065477478,0.740385651844,0.740116693477,0.739847899563,0.739579269986,0.739310804633,0.739042503387,0.738774366134,0.738506392759,0.738238583147,0.737970937183,0.737703454754,0.737436135744,0.737168980039,0.736901987525,0.736635158087,0.736368491612,0.736101987984,0.735835647091,0.735569468818,0.735303453052,0.735037599678,0.734771908584,0.734506379654,0.734241012777,0.733975807839,0.733710764725,0.733445883324,0.733181163522,0.732916605205,0.732652208261,0.732387972577,0.732123898041,0.731859984538,0.731596231958,0.731332640186,0.731069209112,0.730805938621,0.730542828603,0.730279878944,0.730017089533,0.729754460258,0.729491991006,0.729229681666,0.728967532126,0.728705542273,0.728443711998,0.728182041187,0.72792052973,0.727659177515,0.72739798443,0.727136950365,0.726876075209,0.726615358849,0.726354801177,0.726094402079,0.725834161446,0.725574079167,0.725314155131,0.725054389228,0.724794781347,0.724535331378,0.72427603921,0.724016904734,0.723757927838,0.723499108413,0.72324044635,0.722981941537,0.722723593865,0.722465403225,0.722207369507,0.721949492601,0.721691772398,0.721434208788,0.721176801662,0.72091955091,0.720662456424,0.720405518094,0.720148735812,0.719892109468,0.719635638954,0.71937932416,0.719123164979,0.718867161301,0.718611313017,0.718355620021,0.718100082202,0.717844699454,0.717589471666,0.717334398733,0.717079480545,0.716824716994,0.716570107973,0.716315653373,0.716061353088,0.715807207009,0.715553215028,0.715299377039,0.715045692934,0.714792162605,0.714538785946,0.714285562849,0.714032493207,0.713779576913,0.71352681386,0.713274203941,0.71302174705,0.712769443079,0.712517291923,0.712265293474,0.712013447627,0.711761754274,0.71151021331,0.711258824628,0.711007588122,0.710756503686,0.710505571214,0.710254790601,0.710004161739,0.709753684525,0.709503358851,0.709253184612,0.709003161704,0.708753290019,0.708503569454,0.708253999903,0.70800458126,0.70775531342,0.707506196279,0.707257229731,0.707008413672,0.706759747996,0.7065112326,0.706262867378,0.706014652225,0.705766587038,0.705518671711,0.705270906142,0.705023290224,0.704775823854,0.704528506928,0.704281339342,0.704034320992,0.703787451774,0.703540731585,0.703294160319,0.703047737875,0.702801464148,0.702555339034,0.702309362431,0.702063534235,0.701817854343,0.701572322651,0.701326939057,0.701081703457,0.700836615749,0.700591675829,0.700346883595,0.700102238944,0.699857741774,0.699613391981,0.699369189463,0.699125134118,0.698881225844,0.698637464538,0.698393850098,0.698150382422,0.697907061408,0.697663886954,0.697420858957,0.697177977317,0.696935241932,0.696692652699,0.696450209517,0.696207912285,0.695965760901,0.695723755264,0.695481895272,0.695240180825,0.694998611821,0.694757188159,0.694515909738,0.694274776457,0.694033788216,0.693792944913,0.693552246448,0.69331169272,0.693071283629,0.692831019074,0.692590898956,0.692350923172,0.692111091624,0.691871404211,0.691631860833,0.69139246139,0.691153205782,0.690914093909,0.690675125671,0.690436300969,0.690197619703,0.689959081774,0.689720687081,0.689482435526,0.689244327008,0.68900636143,0.688768538691,0.688530858692,0.688293321335,0.68805592652,0.687818674149,0.687581564122,0.687344596341,0.687107770707,0.686871087122,0.686634545487,0.686398145703,0.686161887672,0.685925771296,0.685689796476,0.685453963114,0.685218271113,0.684982720374,0.684747310799,0.68451204229,0.684276914749,0.684041928079,0.683807082183,0.683572376961,0.683337812317,0.683103388154,0.682869104373,0.682634960879,0.682400957572,0.682167094357,0.681933371136,0.681699787812,0.681466344288,0.681233040467,0.680999876253,0.680766851548,0.680533966256,0.680301220281,0.680068613525,0.679836145893,0.679603817288,0.679371627613,0.679139576773,0.678907664671,0.678675891211,0.678444256297,0.678212759834,0.677981401725,0.677750181874,0.677519100186,0.677288156564,0.677057350915,0.676826683141,0.676596153147,0.676365760838,0.676135506119,0.675905388895,0.67567540907,0.675445566548,0.675215861236,0.674986293038,0.674756861859,0.674527567605,0.67429841018,0.67406938949,0.67384050544,0.673611757936,0.673383146883,0.673154672187,0.672926333754,0.672698131488,0.672470065297,0.672242135086,0.67201434076,0.671786682226,0.67155915939,0.671331772158,0.671104520436,0.670877404131,0.670650423149,0.670423577396,0.670196866778,0.669970291203,0.669743850577,0.669517544807,0.669291373799,0.66906533746,0.668839435698,0.668613668419,0.668388035529,0.668162536937,0.66793717255,0.667711942274,0.667486846018,0.667261883687,0.667037055191,0.666812360436,0.66658779933,0.66636337178,0.666139077695,0.665914916982,0.665690889549,0.665466995304,0.665243234155,0.66501960601,0.664796110776,0.664572748364,0.66434951868,0.664126421633,0.663903457131,0.663680625083,0.663457925397,0.663235357982,0.663012922747,0.662790619601,0.662568448451,0.662346409208,0.662124501779,0.661902726075,0.661681082004,0.661459569475,0.661238188397,0.66101693868,0.660795820233,0.660574832966,0.660353976788,0.660133251608,0.659912657336,0.659692193882,0.659471861156,0.659251659067,0.659031587525,0.658811646441,0.658591835724,0.658372155284,0.658152605031,0.657933184876,0.657713894729,0.657494734501,0.657275704101,0.65705680344,0.656838032429,0.656619390978,0.656400878998,0.6561824964,0.655964243095,0.655746118993,0.655528124005,0.655310258043,0.655092521017,0.654874912838,0.654657433419,0.654440082669,0.654222860501,0.654005766825,0.653788801554,0.653571964598,0.65335525587,0.653138675281,0.652922222742,0.652705898166,0.652489701465,0.652273632549,0.652057691332,0.651841877726,0.651626191641,0.651410632991,0.651195201689,0.650979897645,0.650764720773,0.650549670984,0.650334748192,0.65011995231,0.649905283249,0.649690740922,0.649476325242,0.649262036122,0.649047873475,0.648833837214,0.648619927252,0.648406143501,0.648192485876,0.647978954289,0.647765548653,0.647552268882,0.647339114889,0.647126086588,0.646913183892,0.646700406715,0.646487754971,0.646275228572,0.646062827434,0.645850551469,0.645638400592,0.645426374717,0.645214473757,0.645002697628,0.644791046242,0.644579519514,0.644368117359,0.644156839691,0.643945686424,0.643734657473,0.643523752752,0.643312972176,0.64310231566,0.642891783117,0.642681374464,0.642471089615,0.642260928485,0.642050890988,0.64184097704,0.641631186557,0.641421519452,0.641211975642,0.641002555041,0.640793257566,0.640584083131,0.640375031651,0.640166103044,0.639957297223,0.639748614105,0.639540053605,0.63933161564,0.639123300125,0.638915106976,0.638707036108,0.638499087439,0.638291260884,0.638083556359,0.637875973781,0.637668513065,0.637461174129,0.637253956888,0.637046861259,0.636839887159,0.636633034504,0.63642630321,0.636219693196,0.636013204376,0.635806836668,0.63560058999,0.635394464257,0.635188459388,0.634982575299,0.634776811906,0.634571169128,0.634365646882,0.634160245085,0.633954963655,0.633749802508,0.633544761562,0.633339840736,0.633135039945,0.63293035911,0.632725798146,0.632521356972,0.632317035505,0.632112833665,0.631908751367,0.631704788532,0.631500945076,0.631297220918,0.631093615977,0.630890130169,0.630686763415,0.630483515632,0.630280386738,0.630077376653,0.629874485294,0.629671712581,0.629469058433,0.629266522767,0.629064105503,0.628861806559,0.628659625856,0.62845756331,0.628255618843,0.628053792372,0.627852083818,0.627650493098,0.627449020134,0.627247664843,0.627046427146,0.626845306961,0.626644304209,0.626443418809,0.62624265068,0.626041999743,0.625841465917,0.625641049122,0.625440749278,0.625240566305,0.625040500123,0.624840550652,0.624640717812,0.624441001523,0.624241401705,0.62404191828,0.623842551166,0.623643300286,0.623444165558,0.623245146904,0.623046244245,0.6228474575,0.622648786591,0.622450231438,0.622251791963,0.622053468086,0.621855259728,0.621657166809,0.621459189253,0.621261326978,0.621063579907,0.620865947961,0.62066843106,0.620471029127,0.620273742083,0.620076569849,0.619879512347,0.619682569498,0.619485741225,0.619289027448,0.619092428089,0.618895943071,0.618699572315,0.618503315743,0.618307173277,0.618111144839,0.617915230352,0.617719429736,0.617523742916,0.617328169812,0.617132710348,0.616937364445,0.616742132026,0.616547013014,0.616352007331,0.6161571149,0.615962335643,0.615767669483,0.615573116343,0.615378676146,0.615184348815,0.614990134272,0.614796032441,0.614602043244,0.614408166606,0.614214402448,0.614020750694,0.613827211269,0.613633784094,0.613440469093,0.61324726619,0.613054175308,0.612861196371,0.612668329303,0.612475574027,0.612282930466,0.612090398546,0.611897978189,0.611705669319,0.611513471861,0.611321385738,0.611129410875,0.610937547196,0.610745794624,0.610554153085,0.610362622501,0.610171202799,0.609979893902,0.609788695734,0.609597608221,0.609406631286,0.609215764855,0.609025008852,0.608834363201,0.608643827829,0.608453402658,0.608263087615,0.608072882625,0.607882787611,0.6076928025,0.607502927217,0.607313161686,0.607123505833,0.606933959584,0.606744522863,0.606555195596,0.606365977708,0.606176869126,0.605987869774,0.605798979578,0.605610198464,0.605421526358,0.605232963184,0.60504450887,0.604856163342,0.604667926524,0.604479798343,0.604291778725,0.604103867596,0.603916064883,0.603728370511,0.603540784407,0.603353306498,0.603165936708,0.602978674966,0.602791521197,0.602604475328,0.602417537286,0.602230706997,0.602043984388,0.601857369385,0.601670861916,0.601484461908,0.601298169286,0.601111983979,0.600925905913,0.600739935016,0.600554071214,0.600368314434,0.600182664605,0.599997121652,0.599811685505,0.599626356089,0.599441133332,0.599256017162,0.599071007507,0.598886104294,0.59870130745,0.598516616904,0.598332032583,0.598147554415,0.597963182328,0.597778916249,0.597594756108,0.597410701831,0.597226753347,0.597042910584,0.59685917347,0.596675541934,0.596492015903,0.596308595307,0.596125280073,0.59594207013,0.595758965406,0.59557596583,0.595393071331,0.595210281837,0.595027597277,0.59484501758,0.594662542674,0.594480172489,0.594297906953,0.594115745996,0.593933689545,0.593751737532,0.593569889883,0.59338814653,0.5932065074,0.593024972423,0.592843541529,0.592662214647,0.592480991707,0.592299872637,0.592118857367,0.591937945828,0.591757137948,0.591576433658,0.591395832887,0.591215335564,0.59103494162,0.590854650985,0.590674463589,0.590494379361,0.590314398232,0.590134520131,0.589954744989,0.589775072737,0.589595503304,0.58941603662,0.589236672617,0.589057411224,0.588878252371,0.588699195991,0.588520242012,0.588341390366,0.588162640983,0.587983993794,0.58780544873,0.587627005721,0.587448664699,0.587270425594,0.587092288337,0.58691425286,0.586736319093,0.586558486968,0.586380756415,0.586203127366,0.586025599753,0.585848173505,0.585670848556,0.585493624836,0.585316502277,0.58513948081,0.584962560367,0.584785740879,0.584609022278,0.584432404496,0.584255887465,0.584079471116,0.583903155382,0.583726940194,0.583550825484,0.583374811184,0.583198897226,0.583023083543,0.582847370066,0.582671756729,0.582496243462,0.582320830198,0.582145516871,0.581970303411,0.581795189752,0.581620175826,0.581445261566,0.581270446904,0.581095731773,0.580921116105,0.580746599834,0.580572182892,0.580397865213,0.580223646728,0.580049527371,0.579875507075,0.579701585773,0.579527763398,0.579354039884,0.579180415162,0.579006889168,0.578833461833,0.578660133092,0.578486902877,0.578313771123,0.578140737762,0.577967802729,0.577794965956,0.577622227377,0.577449586927,0.577277044538,0.577104600145,0.576932253682,0.576760005081,0.576587854278,0.576415801207,0.5762438458,0.576071987993,0.575900227719,0.575728564912,0.575556999507,0.575385531439,0.575214160641,0.575042887047,0.574871710593,0.574700631212,0.574529648839,0.574358763409,0.574187974856,0.574017283114,0.57384668812,0.573676189806,0.573505788109,0.573335482963,0.573165274302,0.572995162062,0.572825146178,0.572655226584,0.572485403216,0.57231567601,0.572146044899,0.571976509819,0.571807070706,0.571637727495,0.571468480121,0.571299328519,0.571130272626,0.570961312376,0.570792447705,0.570623678549,0.570455004843,0.570286426523,0.570117943525,0.569949555784,0.569781263236,0.569613065818,0.569444963465,0.569276956113,0.569109043698,0.568941226156,0.568773503423,0.568605875435,0.568438342129,0.568270903441,0.568103559307,0.567936309664,0.567769154447,0.567602093593,0.56743512704,0.567268254722,0.567101476577,0.566934792542,0.566768202552,0.566601706546,0.566435304458,0.566268996228,0.56610278179,0.565936661082,0.565770634042,0.565604700605,0.565438860709,0.565273114291,0.565107461289,0.564941901639,0.564776435278,0.564611062144,0.564445782174,0.564280595306,0.564115501476,0.563950500623,0.563785592683,0.563620777595,0.563456055296,0.563291425723,0.563126888814,0.562962444507,0.562798092739,0.562633833449,0.562469666574,0.562305592052,0.562141609822,0.56197771982,0.561813921985,0.561650216256,0.56148660257,0.561323080865,0.56115965108,0.560996313152,0.560833067021,0.560669912624,0.5605068499,0.560343878788,0.560180999225,0.560018211151,0.559855514503,0.559692909221,0.559530395243,0.559367972507,0.559205640954,0.55904340052,0.558881251146,0.55871919277,0.558557225331,0.558395348768,0.558233563019,0.558071868025,0.557910263724,0.557748750055,0.557587326957,0.55742599437,0.557264752233,0.557103600486,0.556942539067,0.556781567915,0.556620686972,0.556459896175,0.556299195465,0.55613858478,0.555978064062,0.555817633249,0.55565729228,0.555497041097,0.555336879638,0.555176807844,0.555016825654,0.554856933009,0.554697129847,0.55453741611,0.554377791738,0.554218256669,0.554058810846,0.553899454207,0.553740186693,0.553581008244,0.553421918801,0.553262918304,0.553104006694,0.55294518391,0.552786449894,0.552627804585,0.552469247925,0.552310779854,0.552152400313,0.551994109242,0.551835906583,0.551677792275,0.551519766261,0.55136182848,0.551203978874,0.551046217383,0.550888543949,0.550730958513,0.550573461015,0.550416051398,0.550258729602,0.550101495568,0.549944349237,0.549787290552,0.549630319452,0.549473435881,0.549316639778,0.549159931086,0.549003309746,0.5488467757,0.548690328889,0.548533969255,0.54837769674,0.548221511285,0.548065412832,0.547909401323,0.5477534767,0.547597638905,0.547441887879,0.547286223565,0.547130645904,0.54697515484,0.546819750313,0.546664432266,0.546509200641,0.546354055381,0.546198996428,0.546044023723,0.54588913721,0.545734336831,0.545579622528,0.545424994244,0.545270451921,0.545115995502,0.544961624929,0.544807340145,0.544653141094,0.544499027717,0.544344999957,0.544191057757,0.544037201061,0.54388342981,0.543729743948,0.543576143418,0.543422628163,0.543269198126,0.54311585325,0.542962593478,0.542809418753,0.54265632902,0.54250332422,0.542350404297,0.542197569195,0.542044818857,0.541892153226,0.541739572246,0.541587075861,0.541434664014,0.541282336648,0.541130093708,0.540977935136,0.540825860877,0.540673870875,0.540521965072,0.540370143414,0.540218405844,0.540066752306,0.539915182743,0.5397636971,0.539612295322,0.539460977351,0.539309743132,0.53915859261,0.539007525728,0.538856542431,0.538705642663,0.538554826369,0.538404093492,0.538253443977,0.538102877769,0.537952394812,0.537801995051,0.53765167843,0.537501444894,0.537351294387,0.537201226855,0.537051242242,0.536901340492,0.53675152155,0.536601785363,0.536452131873,0.536302561027,0.536153072768,0.536003667043,0.535854343796,0.535705102973,0.535555944518,0.535406868376,0.535257874494,0.535108962815,0.534960133286,0.534811385852,0.534662720458,0.534514137049,0.534365635571,0.534217215969,0.53406887819,0.533920622178,0.533772447879,0.533624355239,0.533476344204,0.533328414718,0.533180566729,0.533032800181,0.532885115021,0.532737511194,0.532589988647,0.532442547325,0.532295187175,0.532147908142,0.532000710172,0.531853593212,0.531706557208,0.531559602105,0.531412727851,0.531265934391,0.531119221672,0.53097258964,0.530826038241,0.530679567423,0.53053317713,0.530386867311,0.530240637911,0.530094488877,0.529948420155,0.529802431693,0.529656523436,0.529510695333,0.529364947329,0.529219279371,0.529073691406,0.528928183381,0.528782755243,0.528637406939,0.528492138416,0.528346949621,0.528201840501,0.528056811003,0.527911861074,0.527766990662,0.527622199713,0.527477488176,0.527332855997,0.527188303123,0.527043829502,0.526899435082,0.52675511981,0.526610883633,0.526466726499,0.526322648356,0.52617864915,0.526034728831,0.525890887344,0.525747124639,0.525603440663,0.525459835364,0.525316308689,0.525172860587,0.525029491005,0.524886199891,0.524742987194,0.524599852861,0.52445679684,0.52431381908,0.524170919528,0.524028098134,0.523885354844,0.523742689607,0.523600102372,0.523457593087,0.5233151617,0.52317280816,0.523030532415,0.522888334413,0.522746214103,0.522604171434,0.522462206354,0.522320318812,0.522178508757,0.522036776136,0.521895120899,0.521753542995,0.521612042373,0.52147061898,0.521329272767,0.521188003682,0.521046811674,0.520905696692,0.520764658685,0.520623697603,0.520482813393,0.520342006006,0.52020127539,0.520060621495,0.51992004427,0.519779543665,0.519639119628,0.519498772109,0.519358501057,0.519218306422,0.519078188153,0.5189381462,0.518798180512,0.518658291039,0.518518477731,0.518378740536,0.518239079406,0.518099494289,0.517959985135,0.517820551894,0.517681194516,0.517541912951,0.517402707148,0.517263577059,0.517124522631,0.516985543816,0.516846640564,0.516707812825,0.516569060548,0.516430383684,0.516291782184,0.516153255997,0.516014805073,0.515876429363,0.515738128818,0.515599903387,0.515461753022,0.515323677671,0.515185677287,0.515047751819,0.514909901218,0.514772125435,0.51463442442,0.514496798123,0.514359246497,0.51422176949,0.514084367055,0.513947039141,0.5138097857,0.513672606682,0.513535502039,0.513398471721,0.51326151568,0.513124633865,0.512987826229,0.512851092723,0.512714433297,0.512577847903,0.512441336491,0.512304899014,0.512168535422,0.512032245667,0.511896029699,0.511759887471,0.511623818933,0.511487824038,0.511351902736,0.51121605498,0.511080280719,0.510944579908,0.510808952495,0.510673398435,0.510537917677,0.510402510174,0.510267175878,0.510131914739,0.509996726711,0.509861611745,0.509726569793,0.509591600807,0.509456704738,0.509321881539,0.509187131161,0.509052453557,0.508917848679,0.508783316479,0.508648856909,0.508514469922,0.508380155469,0.508245913502,0.508111743975,0.507977646839,0.507843622047,0.507709669551,0.507575789303,0.507441981257,0.507308245364,0.507174581577,0.507040989848,0.506907470131,0.506774022378,0.506640646541,0.506507342573,0.506374110428,0.506240950057,0.506107861413,0.50597484445,0.505841899121,0.505709025377,0.505576223172,0.50544349246,0.505310833192,0.505178245322,0.505045728804,0.50491328359,0.504780909633,0.504648606887,0.504516375305,0.50438421484,0.504252125445,0.504120107073,0.503988159679,0.503856283214,0.503724477634,0.503592742891,0.503461078938,0.50332948573,0.503197963219,0.503066511359,0.502935130105,0.502803819409,0.502672579225,0.502541409507,0.502410310208,0.502279281283,0.502148322685,0.502017434369,0.501886616287,0.501755868394,0.501625190644,0.501494582991,0.501364045388,0.50123357779,0.501103180151,0.500972852425,0.500842594567,0.500712406529,0.500582288267,0.500452239735,0.500322260886,0.500192351676,0.500062512059,0.499932741989,0.49980304142,0.499673410306,0.499543848604,0.499414356265,0.499284933247,0.499155579502,0.499026294985,0.498897079652,0.498767933457,0.498638856354,0.498509848298,0.498380909244,0.498252039147,0.498123237961,0.497994505642,0.497865842144,0.497737247423,0.497608721432,0.497480264128,0.497351875465,0.497223555398,0.497095303882,0.496967120873,0.496839006325,0.496710960194,0.496582982436,0.496455073004,0.496327231855,0.496199458943,0.496071754225,0.495944117655,0.495816549189,0.495689048782,0.49556161639,0.495434251968,0.495306955472,0.495179726858,0.49505256608,0.494925473095,0.494798447859,0.494671490326,0.494544600453,0.494417778195,0.494291023508,0.494164336349,0.494037716672,0.493911164434,0.49378467959,0.493658262097,0.493531911911,0.493405628987,0.493279413282,0.493153264751,0.493027183352,0.492901169039,0.492775221769,0.492649341498,0.492523528183,0.492397781779,0.492272102244,0.492146489532,0.492020943602,0.491895464408,0.491770051908,0.491644706058,0.491519426814,0.491394214133,0.491269067971,0.491143988285,0.491018975032,0.490894028168,0.49076914765,0.490644333435,0.490519585478,0.490394903738,0.490270288171,0.490145738733,0.490021255381,0.489896838073,0.489772486765,0.489648201415,0.489523981978,0.489399828413,0.489275740676,0.489151718724,0.489027762514,0.488903872004,0.48878004715,0.488656287911,0.488532594242,0.488408966101,0.488285403446,0.488161906234,0.488038474422,0.487915107967,0.487791806828,0.48766857096,0.487545400323,0.487422294873,0.487299254567,0.487176279364,0.487053369221,0.486930524095,0.486807743945,0.486685028727,0.4865623784,0.486439792922,0.486317272249,0.48619481634,0.486072425153,0.485950098645,0.485827836775,0.4857056395,0.485583506778,0.485461438568,0.485339434826,0.485217495512,0.485095620583,0.484973809998,0.484852063714,0.48473038169,0.484608763883,0.484487210253,0.484365720757,0.484244295353,0.484122934001,0.484001636657,0.483880403281,0.483759233831,0.483638128266,0.483517086543,0.483396108621,0.483275194459,0.483154344016,0.483033557249,0.482912834118,0.482792174581,0.482671578597,0.482551046124,0.482430577121,0.482310171547,0.482189829361,0.482069550522,0.481949334987,0.481829182717,0.48170909367,0.481589067805,0.481469105081,0.481349205457,0.481229368891,0.481109595344,0.480989884773,0.480870237139,0.4807506524,0.480631130515,0.480511671444,0.480392275146,0.480272941579,0.480153670704,0.48003446248,0.479915316865,0.47979623382,0.479677213304,0.479558255275,0.479439359694,0.47932052652,0.479201755712,0.479083047231,0.478964401035,0.478845817084,0.478727295338,0.478608835756,0.478490438298,0.478372102925,0.478253829594,0.478135618267,0.478017468903,0.477899381462,0.477781355904,0.477663392188,0.477545490275,0.477427650124,0.477309871696,0.47719215495,0.477074499847,0.476956906346,0.476839374407,0.476721903991,0.476604495058,0.476487147568,0.47636986148,0.476252636757,0.476135473356,0.47601837124,0.475901330367,0.4757843507,0.475667432196,0.475550574819,0.475433778526,0.47531704328,0.47520036904,0.475083755768,0.474967203423,0.474850711966,0.474734281357,0.474617911558,0.474501602529,0.474385354231,0.474269166624,0.474153039668,0.474036973326,0.473920967557,0.473805022323,0.473689137583,0.4735733133,0.473457549433,0.473341845945,0.473226202795,0.473110619945,0.472995097356,0.472879634988,0.472764232804,0.472648890763,0.472533608828,0.472418386958,0.472303225116,0.472188123262,0.472073081358,0.471958099366,0.471843177245,0.471728314958,0.471613512466,0.47149876973,0.471384086712,0.471269463373,0.471154899675,0.471040395578,0.470925951045,0.470811566037,0.470697240516,0.470582974442,0.470468767779,0.470354620486,0.470240532527,0.470126503863,0.470012534455,0.469898624265,0.469784773255,0.469670981387,0.469557248622,0.469443574923,0.469329960251,0.469216404568,0.469102907836,0.468989470018,0.468876091074,0.468762770968,0.468649509661,0.468536307114,0.468423163292,0.468310078154,0.468197051664,0.468084083784,0.467971174476,0.467858323701,0.467745531424,0.467632797605,0.467520122207,0.467407505192,0.467294946523,0.467182446162,0.467070004071,0.466957620213,0.466845294551,0.466733027047,0.466620817663,0.466508666362,0.466396573106,0.466284537859,0.466172560582,0.466060641239,0.465948779792,0.465836976204,0.465725230437,0.465613542455,0.465501912219,0.465390339694,0.465278824841,0.465167367624,0.465055968005,0.464944625948,0.464833341415,0.464722114369,0.464610944773,0.464499832591,0.464388777786,0.46427778032,0.464166840156,0.464055957258,0.463945131589,0.463834363112,0.46372365179,0.463612997586,0.463502400465,0.463391860388,0.463281377319,0.463170951223,0.463060582061,0.462950269797,0.462840014396,0.46272981582,0.462619674032,0.462509588997,0.462399560677,0.462289589036,0.462179674039,0.462069815648,0.461960013826,0.461850268539,0.461740579749,0.46163094742,0.461521371516,0.461411852,0.461302388837,0.461192981989,0.461083631422,0.460974337099,0.460865098983,0.460755917038,0.46064679123,0.46053772152,0.460428707874,0.460319750256,0.460210848629,0.460102002957,0.459993213205,0.459884479337,0.459775801317,0.459667179109,0.459558612676,0.459450101985,0.459341646998,0.459233247679,0.459124903994,0.459016615907,0.458908383381,0.458800206381,0.458692084872,0.458584018818,0.458476008184,0.458368052933,0.45826015303,0.458152308441,0.458044519129,0.457936785058,0.457829106195,0.457721482502,0.457613913945,0.457506400489,0.457398942098,0.457291538737,0.45718419037,0.457076896963,0.456969658479,0.456862474885,0.456755346145,0.456648272223,0.456541253085,0.456434288695,0.456327379018,0.45622052402,0.456113723665,0.456006977919,0.455900286746,0.455793650111,0.45568706798,0.455580540317,0.455474067088,0.455367648258,0.455261283793,0.455154973656,0.455048717814,0.454942516232,0.454836368874,0.454730275708,0.454624236696,0.454518251806,0.454412321003,0.454306444251,0.454200621516,0.454094852764,0.453989137961,0.453883477071,0.45377787006,0.453672316894,0.453566817538,0.453461371959,0.45335598012,0.453250641989,0.453145357531,0.453040126712,0.452934949496,0.452829825851,0.452724755742,0.452619739134,0.452514775994,0.452409866286,0.452305009978,0.452200207035,0.452095457423,0.451990761108,0.451886118055,0.451781528231,0.451676991602,0.451572508134,0.451468077793,0.451363700545,0.451259376355,0.451155105191,0.451050887019,0.450946721804,0.450842609512,0.450738550111,0.450634543566,0.450530589843,0.450426688909,0.45032284073,0.450219045272,0.450115302503,0.450011612387,0.449907974892,0.449804389984,0.44970085763,0.449597377796,0.449493950448,0.449390575553,0.449287253077,0.449183982988,0.449080765252,0.448977599835,0.448874486704,0.448771425825,0.448668417166,0.448565460694,0.448462556374,0.448359704173,0.448256904059,0.448154155999,0.448051459958,0.447948815905,0.447846223805,0.447743683626,0.447641195334,0.447538758898,0.447436374282,0.447334041456,0.447231760385,0.447129531036,0.447027353378,0.446925227376,0.446823152998,0.446721130211,0.446619158983,0.44651723928,0.446415371069,0.446313554319,0.446211788995,0.446110075066,0.446008412499,0.44590680126,0.445805241318,0.44570373264,0.445602275193,0.445500868945,0.445399513862,0.445298209913,0.445196957065,0.445095755285,0.444994604542,0.444893504801,0.444792456032,0.444691458202,0.444590511278,0.444489615228,0.44438877002,0.444287975621,0.444187231999,0.444086539121,0.443985896957,0.443885305472,0.443784764636,0.443684274416,0.44358383478,0.443483445695,0.44338310713,0.443282819052,0.44318258143,0.443082394231,0.442982257423,0.442882170975,0.442782134855,0.442682149029,0.442582213468,0.442482328138,0.442382493007,0.442282708045,0.442182973219,0.442083288496,0.441983653847,0.441884069238,0.441784534638,0.441685050015,0.441585615338,0.441486230574,0.441386895693,0.441287610662,0.441188375451,0.441089190026,0.440990054358,0.440890968413,0.440791932161,0.440692945571,0.44059400861,0.440495121248,0.440396283452,0.440297495192,0.440198756435,0.440100067152,0.44000142731,0.439902836877,0.439804295824,0.439705804117,0.439607361727,0.439508968622,0.439410624771,0.439312330142,0.439214084704,0.439115888427,0.439017741278,0.438919643228,0.438821594244,0.438723594296,0.438625643353,0.438527741384,0.438429888357,0.438332084242,0.438234329008,0.438136622624,0.438038965059,0.437941356282,0.437843796262,0.437746284969,0.437648822371,0.437551408438,0.437454043138,0.437356726442,0.437259458319,0.437162238737,0.437065067666,0.436967945076,0.436870870935,0.436773845214,0.436676867881,0.436579938906,0.436483058258,0.436386225907,0.436289441823,0.436192705974,0.43609601833,0.435999378862,0.435902787537,0.435806244327,0.4357097492,0.435613302127,0.435516903076,0.435420552018,0.435324248923,0.435227993759,0.435131786497,0.435035627106,0.434939515557,0.434843451819,0.434747435861,0.434651467654,0.434555547168,0.434459674372,0.434363849236,0.434268071731,0.434172341826,0.434076659491,0.433981024696,0.433885437412,0.433789897607,0.433694405253,0.433598960319,0.433503562776,0.433408212593,0.43331290974,0.433217654189,0.433122445908,0.433027284869,0.432932171041,0.432837104395,0.4327420849,0.432647112528,0.432552187248,0.432457309031,0.432362477848,0.432267693667,0.432172956461,0.432078266199,0.431983622852,0.431889026389,0.431794476783,0.431699974002,0.431605518019,0.431511108802,0.431416746323,0.431322430553,0.431228161461,0.431133939019,0.431039763197,0.430945633965,0.430851551295,0.430757515158,0.430663525523,0.430569582361,0.430475685644,0.430381835343,0.430288031427,0.430194273868,0.430100562636,0.430006897703,0.429913279039,0.429819706615,0.429726180403,0.429632700372,0.429539266495,0.429445878742,0.429352537083,0.429259241491,0.429165991935,0.429072788388,0.42897963082,0.428886519203,0.428793453506,0.428700433703,0.428607459763,0.428514531658,0.428421649359,0.428328812837,0.428236022064,0.428143277011,0.428050577649,0.427957923949,0.427865315884,0.427772753423,0.427680236538,0.427587765202,0.427495339385,0.427402959058,0.427310624194,0.427218334763,0.427126090737,0.427033892088,0.426941738787,0.426849630805,0.426757568115,0.426665550687,0.426573578494,0.426481651507,0.426389769697,0.426297933037,0.426206141497,0.426114395051,0.426022693668,0.425931037322,0.425839425984,0.425747859625,0.425656338218,0.425564861734,0.425473430146,0.425382043424,0.425290701541,0.425199404469,0.42510815218,0.425016944645,0.424925781837,0.424834663727,0.424743590288,0.424652561491,0.424561577309,0.424470637714,0.424379742678,0.424288892172,0.424198086169,0.424107324641,0.424016607561,0.4239259349,0.42383530663,0.423744722725,0.423654183155,0.423563687893,0.423473236912,0.423382830184,0.423292467681,0.423202149376,0.42311187524,0.423021645247,0.422931459367,0.422841317575,0.422751219843,0.422661166142,0.422571156445,0.422481190725,0.422391268955,0.422301391106,0.422211557152,0.422121767064,0.422032020816,0.42194231838,0.421852659728,0.421763044834,0.42167347367,0.421583946208,0.421494462422,0.421405022284,0.421315625766,0.421226272842,0.421136963484,0.421047697665,0.420958475358,0.420869296536,0.420780161171,0.420691069237,0.420602020705,0.42051301555,0.420424053744,0.42033513526,0.420246260071,0.42015742815,0.42006863947,0.419979894004,0.419891191725,0.419802532605,0.419713916619,0.419625343739,0.419536813939,0.41944832719,0.419359883468,0.419271482744,0.419183124991,0.419094810184,0.419006538295,0.418918309298,0.418830123166,0.418741979871,0.418653879388,0.41856582169,0.41847780675,0.418389834541,0.418301905036,0.41821401821,0.418126174036,0.418038372486,0.417950613534,0.417862897155,0.417775223321,0.417687592005,0.417600003182,0.417512456825,0.417424952907,0.417337491402,0.417250072284,0.417162695525,0.417075361101,0.416988068984,0.416900819148,0.416813611567,0.416726446214,0.416639323064,0.41655224209,0.416465203265,0.416378206563,0.416291251959,0.416204339426,0.416117468938,0.416030640468,0.415943853991,0.415857109481,0.41577040691,0.415683746255,0.415597127487,0.415510550581,0.415424015512,0.415337522253,0.415251070777,0.41516466106,0.415078293076,0.414991966797,0.414905682199,0.414819439255,0.41473323794,0.414647078227,0.414560960091,0.414474883506,0.414388848447,0.414302854887,0.4142169028,0.414130992161,0.414045122945,0.413959295125,0.413873508675,0.413787763571,0.413702059785,0.413616397294,0.41353077607,0.413445196089,0.413359657325,0.413274159752,0.413188703345,0.413103288077,0.413017913925,0.412932580861,0.412847288861,0.4127620379,0.412676827951,0.412591658989,0.412506530989,0.412421443925,0.412336397773,0.412251392506,0.412166428099,0.412081504527,0.411996621765,0.411911779788,0.411826978569,0.411742218084,0.411657498308,0.411572819215,0.41148818078,0.411403582979,0.411319025784,0.411234509173,0.411150033119,0.411065597597,0.410981202583,0.410896848051,0.410812533975,0.410728260332,0.410644027096,0.410559834241,0.410475681744,0.410391569579,0.41030749772,0.410223466144,0.410139474825,0.410055523738,0.409971612858,0.409887742161,0.409803911621,0.409720121214,0.409636370915,0.409552660699,0.409468990541,0.409385360417,0.409301770302,0.40921822017,0.409134709998,0.40905123976,0.408967809433,0.40888441899,0.408801068408,0.408717757662,0.408634486727,0.408551255579,0.408468064193,0.408384912544,0.408301800608,0.408218728361,0.408135695778,0.408052702834,0.407969749505,0.407886835766,0.407803961594,0.407721126962,0.407638331848,0.407555576227,0.407472860074,0.407390183365,0.407307546076,0.407224948182,0.407142389659,0.407059870482,0.406977390628,0.406894950072,0.40681254879,0.406730186757,0.40664786395,0.406565580344,0.406483335915,0.406401130639,0.406318964491,0.406236837449,0.406154749486,0.40607270058,0.405990690706,0.405908719841,0.405826787959,0.405744895038,0.405663041052,0.405581225979,0.405499449794,0.405417712473,0.405336013992,0.405254354327,0.405172733455,0.405091151351,0.405009607992,0.404928103353,0.404846637411,0.404765210143,0.404683821523,0.404602471529,0.404521160136,0.404439887321,0.40435865306,0.40427745733,0.404196300106,0.404115181365,0.404034101083,0.403953059236,0.403872055802,0.403791090756,0.403710164074,0.403629275734,0.40354842571,0.403467613981,0.403386840522,0.40330610531,0.403225408321,0.403144749531,0.403064128918,0.402983546458,0.402903002127,0.402822495901,0.402742027759,0.402661597675,0.402581205626,0.40250085159,0.402420535543,0.402340257462,0.402260017322,0.402179815102,0.402099650777,0.402019524324,0.40193943572,0.401859384942,0.401779371967,0.401699396771,0.401619459331,0.401539559624,0.401459697627,0.401379873316,0.401300086669,0.401220337663,0.401140626273,0.401060952478,0.400981316253,0.400901717577,0.400822156426,0.400742632777,0.400663146606,0.400583697892,0.40050428661,0.400424912739,0.400345576254,0.400266277134,0.400187015355,0.400107790894,0.400028603729,0.399949453836,0.399870341193,0.399791265777,0.399712227565,0.399633226534,0.399554262661,0.399475335924,0.3993964463,0.399317593767,0.3992387783,0.399159999879,0.399081258479,0.399002554079,0.398923886656,0.398845256186,0.398766662648,0.398688106019,0.398609586276,0.398531103397,0.398452657359,0.398374248139,0.398295875715,0.398217540064,0.398139241165,0.398060978993,0.397982753528,0.397904564746,0.397826412626,0.397748297143,0.397670218277,0.397592176005,0.397514170304,0.397436201153,0.397358268527,0.397280372407,0.397202512768,0.397124689589,0.397046902847,0.39696915252,0.396891438587,0.396813761024,0.396736119809,0.39665851492,0.396580946336,0.396503414033,0.39642591799,0.396348458184,0.396271034594,0.396193647197,0.39611629597,0.396038980893,0.395961701943,0.395884459098,0.395807252335,0.395730081633,0.39565294697,0.395575848324,0.395498785672,0.395421758993,0.395344768265,0.395267813466,0.395190894574,0.395114011567,0.395037164423,0.39496035312,0.394883577637,0.394806837951,0.394730134041,0.394653465885,0.39457683346,0.394500236746,0.394423675721,0.394347150362,0.394270660649,0.394194206558,0.394117788069,0.39404140516,0.393965057809,0.393888745995,0.393812469695,0.393736228889,0.393660023554,0.393583853669,0.393507719212,0.393431620162,0.393355556497,0.393279528196,0.393203535237,0.393127577598,0.393051655259,0.392975768197,0.392899916391,0.39282409982,0.392748318462,0.392672572295,0.392596861299,0.392521185452,0.392445544732,0.392369939119,0.39229436859,0.392218833124,0.392143332701,0.392067867298,0.391992436895,0.39191704147,0.391841681002,0.39176635547,0.391691064851,0.391615809127,0.391540588273,0.391465402271,0.391390251098,0.391315134734,0.391240053157,0.391165006346,0.391089994279,0.391015016937,0.390940074297,0.390865166339,0.390790293042,0.390715454384,0.390640650344,0.390565880902,0.390491146037,0.390416445726,0.390341779951,0.390267148688,0.390192551918,0.39011798962,0.390043461773,0.389968968355,0.389894509346,0.389820084724,0.38974569447,0.389671338562,0.38959701698,0.389522729702,0.389448476708,0.389374257976,0.389300073487,0.38922592322,0.389151807152,0.389077725265,0.389003677537,0.388929663947,0.388855684475,0.3887817391,0.388707827802,0.388633950559,0.388560107351,0.388486298158,0.388412522958,0.388338781732,0.388265074459,0.388191401117,0.388117761687,0.388044156148,0.38797058448,0.387897046661,0.387823542672,0.387750072491,0.3876766361,0.387603233476,0.387529864599,0.38745652945,0.387383228008,0.387309960252,0.387236726161,0.387163525717,0.387090358897,0.387017225682,0.386944126052,0.386871059986,0.386798027464,0.386725028465,0.38665206297,0.386579130958,0.386506232408,0.386433367301,0.386360535617,0.386287737334,0.386214972433,0.386142240894,0.386069542697,0.385996877821,0.385924246246,0.385851647953,0.38577908292,0.385706551129,0.385634052558,0.385561587188,0.385489154998,0.38541675597,0.385344390082,0.385272057314,0.385199757648,0.385127491061,0.385055257536,0.384983057051,0.384910889587,0.384838755124,0.384766653641,0.38469458512,0.384622549539,0.38455054688,0.384478577122,0.384406640246,0.384334736231,0.384262865058,0.384191026707,0.384119221159,0.384047448392,0.383975708389,0.383904001128,0.38383232659,0.383760684756,0.383689075605,0.383617499118,0.383545955276,0.383474444058,0.383402965446,0.383331519418,0.383260105956,0.38318872504,0.383117376651,0.383046060768,0.382974777373,0.382903526445,0.382832307966,0.382761121915,0.382689968273,0.38261884702,0.382547758138,0.382476701606,0.382405677405,0.382334685515,0.382263725918,0.382192798593,0.382121903522,0.382051040685,0.381980210062,0.381909411634,0.381838645382,0.381767911286,0.381697209327,0.381626539485,0.381555901743,0.381485296079,0.381414722475,0.381344180911,0.381273671368,0.381203193828,0.38113274827,0.381062334676,0.380991953025,0.3809216033,0.380851285481,0.380780999549,0.380710745484,0.380640523267,0.38057033288,0.380500174303,0.380430047516,0.380359952502,0.380289889241,0.380219857713,0.3801498579,0.380079889783,0.380009953342,0.379940048559,0.379870175415,0.37980033389,0.379730523966,0.379660745623,0.379590998843,0.379521283607,0.379451599896,0.379381947691,0.379312326973,0.379242737723,0.379173179922,0.379103653551,0.379034158592,0.378964695026,0.378895262834,0.378825861997,0.378756492496,0.378687154312,0.378617847428,0.378548571823,0.378479327479,0.378410114378,0.378340932501,0.378271781828,0.378202662342,0.378133574024,0.378064516854,0.377995490815,0.377926495888,0.377857532053,0.377788599293,0.377719697589,0.377650826922,0.377581987274,0.377513178626,0.377444400959,0.377375654255,0.377306938495,0.377238253662,0.377169599736,0.377100976698,0.377032384532,0.376963823217,0.376895292735,0.376826793069,0.376758324199,0.376689886107,0.376621478775,0.376553102185,0.376484756317,0.376416441155,0.376348156678,0.376279902869,0.37621167971,0.376143487182,0.376075325267,0.376007193947,0.375939093203,0.375871023018,0.375802983372,0.375734974248,0.375666995627,0.375599047491,0.375531129822,0.375463242602,0.375395385813,0.375327559436,0.375259763453,0.375191997846,0.375124262598,0.375056557689,0.374988883101,0.374921238818,0.37485362482,0.374786041089,0.374718487608,0.374650964358,0.374583471322,0.37451600848,0.374448575816,0.374381173312,0.374313800948,0.374246458708,0.374179146574,0.374111864526,0.374044612548,0.373977390622,0.373910198729,0.373843036852,0.373775904973,0.373708803074,0.373641731137,0.373574689144,0.373507677078,0.37344069492,0.373373742653,0.373306820259,0.373239927719,0.373173065018,0.373106232136,0.373039429055,0.372972655759,0.372905912229,0.372839198448,0.372772514398,0.372705860061,0.372639235419,0.372572640456,0.372506075152,0.372439539492,0.372373033456,0.372306557027,0.372240110188,0.372173692921,0.372107305209,0.372040947034,0.371974618378,0.371908319223,0.371842049553,0.37177580935,0.371709598596,0.371643417273,0.371577265365,0.371511142854,0.371445049722,0.371378985951,0.371312951525,0.371246946426,0.371180970636,0.371115024138,0.371049106916,0.37098321895,0.370917360224,0.370851530721,0.370785730423,0.370719959313,0.370654217373,0.370588504587,0.370522820936,0.370457166404,0.370391540973,0.370325944627,0.370260377347,0.370194839117,0.370129329919,0.370063849736,0.369998398551,0.369932976347,0.369867583106,0.369802218811,0.369736883446,0.369671576992,0.369606299434,0.369541050753,0.369475830932,0.369410639955,0.369345477805,0.369280344464,0.369215239915,0.369150164141,0.369085117125,0.36902009885,0.3689551093,0.368890148456,0.368825216302,0.368760312822,0.368695437997,0.368630591812,0.368565774249,0.36850098529,0.36843622492,0.368371493122,0.368306789877,0.368242115171,0.368177468984,0.368112851302,0.368048262106,0.367983701381,0.367919169108,0.367854665272,0.367790189856,0.367725742842,0.367661324214,0.367596933955,0.367532572048,0.367468238477,0.367403933225,0.367339656275,0.36727540761,0.367211187213,0.367146995069,0.36708283116,0.367018695469,0.366954587981,0.366890508677,0.366826457542,0.366762434559,0.366698439711,0.366634472982,0.366570534354,0.366506623812,0.36644274134,0.366378886919,0.366315060534,0.366251262168,0.366187491805,0.366123749428,0.36606003502,0.365996348566,0.365932690048,0.36586905945,0.365805456756,0.365741881949,0.365678335012,0.36561481593,0.365551324686,0.365487861263,0.365424425645,0.365361017816,0.365297637759,0.365234285458,0.365170960896,0.365107664058,0.365044394926,0.364981153485,0.364917939717,0.364854753608,0.36479159514,0.364728464298,0.364665361064,0.364602285423,0.364539237359,0.364476216855,0.364413223894,0.364350258462,0.364287320541,0.364224410116,0.364161527169,0.364098671686,0.364035843649,0.363973043043,0.363910269852,0.363847524059,0.363784805648,0.363722114604,0.363659450909,0.363596814549,0.363534205506,0.363471623765,0.36340906931,0.363346542125,0.363284042194,0.3632215695,0.363159124027,0.363096705761,0.363034314684,0.36297195078,0.362909614034,0.362847304431,0.362785021952,0.362722766584,0.36266053831,0.362598337113,0.362536162979,0.362474015891,0.362411895833,0.36234980279,0.362287736745,0.362225697683,0.362163685587,0.362101700443,0.362039742234,0.361977810945,0.361915906559,0.361854029061,0.361792178435,0.361730354665,0.361668557736,0.361606787631,0.361545044336,0.361483327833,0.361421638109,0.361359975146,0.361298338929,0.361236729443,0.361175146672,0.3611135906,0.361052061212,0.360990558491,0.360929082422,0.360867632991,0.36080621018,0.360744813975,0.360683444359,0.360622101318,0.360560784835,0.360499494895,0.360438231483,0.360376994584,0.36031578418,0.360254600258,0.360193442801,0.360132311794,0.360071207221,0.360010129068,0.359949077318,0.359888051956,0.359827052967,0.359766080335,0.359705134045,0.359644214081,0.359583320429,0.359522453071,0.359461611995,0.359400797182,0.35934000862,0.359279246291,0.359218510182,0.359157800276,0.359097116558,0.359036459012,0.358975827625,0.358915222379,0.358854643261,0.358794090254,0.358733563344,0.358673062515,0.358612587752,0.358552139039,0.358491716362,0.358431319706,0.358370949054,0.358310604392,0.358250285706,0.358189992978,0.358129726195,0.358069485342,0.358009270402,0.357949081362,0.357888918205,0.357828780917,0.357768669483,0.357708583888,0.357648524116,0.357588490152,0.357528481982,0.357468499591,0.357408542962,0.357348612082,0.357288706935,0.357228827507,0.357168973782,0.357109145745,0.357049343381,0.356989566676,0.356929815614,0.356870090181,0.356810390361,0.356750716139,0.356691067502,0.356631444433,0.356571846918,0.356512274942,0.356452728491,0.356393207548,0.3563337121,0.356274242132,0.356214797628,0.356155378574,0.356095984955,0.356036616757,0.355977273964,0.355917956562,0.355858664535,0.35579939787,0.355740156551,0.355680940564,0.355621749894,0.355562584526,0.355503444445,0.355444329637,0.355385240088,0.355326175781,0.355267136704,0.35520812284,0.355149134176,0.355090170696,0.355031232387,0.354972319233,0.35491343122,0.354854568333,0.354795730558,0.35473691788,0.354678130285,0.354619367757,0.354560630282,0.354501917847,0.354443230436,0.354384568034,0.354325930627,0.354267318202,0.354208730742,0.354150168234,0.354091630664,0.354033118016,0.353974630276,0.35391616743,0.353857729464,0.353799316363,0.353740928112,0.353682564697,0.353624226104,0.353565912318,0.353507623325,0.35344935911,0.35339111966,0.35333290496,0.353274714995,0.353216549751,0.353158409214,0.35310029337,0.353042202203,0.352984135701,0.352926093848,0.35286807663,0.352810084034,0.352752116044,0.352694172646,0.352636253827,0.352578359572,0.352520489867,0.352462644698,0.352404824049,0.352347027908,0.35228925626,0.352231509091,0.352173786387,0.352116088133,0.352058414315,0.35200076492,0.351943139933,0.351885539339,0.351827963126,0.351770411278,0.351712883782,0.351655380624,0.351597901789,0.351540447263,0.351483017033,0.351425611084,0.351368229403,0.351310871975,0.351253538786,0.351196229822,0.35113894507,0.351081684515,0.351024448143,0.350967235941,0.350910047894,0.350852883988,0.35079574421,0.350738628545,0.35068153698,0.350624469501,0.350567426093,0.350510406743,0.350453411438,0.350396440162,0.350339492903,0.350282569646,0.350225670378,0.350168795084,0.350111943751,0.350055116366,0.349998312913,0.34994153338,0.349884777753,0.349828046017,0.34977133816,0.349714654167,0.349657994024,0.349601357719,0.349544745236,0.349488156563,0.349431591685,0.349375050589,0.349318533262,0.349262039689,0.349205569856,0.349149123751,0.34909270136,0.349036302668,0.348979927662,0.348923576329,0.348867248655,0.348810944626,0.348754664228,0.348698407449,0.348642174274,0.348585964691,0.348529778684,0.348473616241,0.348417477349,0.348361361993,0.34830527016,0.348249201836,0.348193157009,0.348137135664,0.348081137788,0.348025163368,0.34796921239,0.34791328484,0.347857380705,0.347801499972,0.347745642628,0.347689808657,0.347633998049,0.347578210788,0.347522446861,0.347466706256,0.347410988958,0.347355294955,0.347299624232,0.347243976777,0.347188352576,0.347132751616,0.347077173884,0.347021619365,0.346966088047,0.346910579917,0.346855094961,0.346799633166,0.346744194518,0.346688779005,0.346633386613,0.346578017328,0.346522671138,0.346467348029,0.346412047989,0.346356771003,0.346301517058,0.346246286142,0.346191078241,0.346135893342,0.346080731432,0.346025592497,0.345970476525,0.345915383502,0.345860313416,0.345805266252,0.345750241998,0.345695240641,0.345640262167,0.345585306564,0.345530373818,0.345475463916,0.345420576846,0.345365712594,0.345310871147,0.345256052492,0.345201256616,0.345146483506,0.345091733149,0.345037005531,0.344982300641,0.344927618464,0.344872958989,0.344818322201,0.344763708088,0.344709116637,0.344654547835,0.344600001669,0.344545478127,0.344490977194,0.344436498859,0.344382043107,0.344327609928,0.344273199306,0.344218811231,0.344164445688,0.344110102665,0.344055782149,0.344001484127,0.343947208586,0.343892955514,0.343838724897,0.343784516723,0.34373033098,0.343676167653,0.34362202673,0.3435679082,0.343513812047,0.343459738261,0.343405686829,0.343351657736,0.343297650972,0.343243666522,0.343189704375,0.343135764517,0.343081846936,0.343027951619,0.342974078554,0.342920227727,0.342866399126,0.342812592739,0.342758808552,0.342705046554,0.342651306731,0.34259758907,0.34254389356,0.342490220188,0.34243656894,0.342382939804,0.342329332769,0.34227574782,0.342222184946,0.342168644134,0.342115125371,0.342061628645,0.342008153944,0.341954701254,0.341901270563,0.341847861859,0.341794475129,0.341741110361,0.341687767542,0.34163444666,0.341581147702,0.341527870656,0.341474615509,0.341421382249,0.341368170863,0.341314981339,0.341261813664,0.341208667827,0.341155543814,0.341102441614,0.341049361213,0.3409963026,0.340943265762,0.340890250686,0.340837257361,0.340784285774,0.340731335913,0.340678407764,0.340625501317,0.340572616558,0.340519753476,0.340466912058,0.340414092291,0.340361294163,0.340308517663,0.340255762778,0.340203029495,0.340150317802,0.340097627688,0.340044959139,0.339992312144,0.33993968669,0.339887082765,0.339834500358,0.339781939454,0.339729400044,0.339676882113,0.339624385651,0.339571910645,0.339519457082,0.339467024952,0.33941461424,0.339362224936,0.339309857027,0.339257510502,0.339205185347,0.339152881551,0.339100599102,0.339048337987,0.338996098195,0.338943879714,0.338891682531,0.338839506634,0.338787352012,0.338735218652,0.338683106542,0.338631015671,0.338578946025,0.338526897594,0.338474870365,0.338422864326,0.338370879465,0.338318915771,0.33826697323,0.338215051832,0.338163151564,0.338111272415,0.338059414372,0.338007577423,0.337955761557,0.337903966762,0.337852193025,0.337800440335,0.33774870868,0.337696998048,0.337645308427,0.337593639806,0.337541992172,0.337490365513,0.337438759818,0.337387175075,0.337335611272,0.337284068397,0.337232546439,0.337181045385,0.337129565224,0.337078105944,0.337026667533,0.336975249979,0.336923853271,0.336872477397,0.336821122345,0.336769788103,0.33671847466,0.336667182004,0.336615910123,0.336564659006,0.33651342864,0.336462219015,0.336411030117,0.336359861937,0.336308714461,0.336257587679,0.336206481578,0.336155396147,0.336104331375,0.336053287249,0.336002263758,0.335951260891,0.335900278636,0.335849316981,0.335798375914,0.335747455424,0.3356965555,0.33564567613,0.335594817302,0.335543979005,0.335493161227,0.335442363956,0.335391587182,0.335340830892,0.335290095075,0.335239379719,0.335188684814,0.335138010347,0.335087356307,0.335036722682,0.334986109461,0.334935516633,0.334884944186,0.334834392109,0.33478386039,0.334733349017,0.33468285798,0.334632387266,0.334581936865,0.334531506765,0.334481096955,0.334430707422,0.334380338157,0.334329989147,0.334279660381,0.334229351848,0.334179063535,0.334128795433,0.33407854753,0.334028319814,0.333978112274,0.333927924898,0.333877757676,0.333827610595,0.333777483646,0.333727376815,0.333677290093,0.333627223468,0.333577176928,0.333527150462,0.33347714406,0.333427157709,0.333377191399,0.333327245118,0.333277318855,0.333227412599,0.333177526339,0.333127660063,0.33307781376,0.333027987419,0.332978181029,0.332928394579,0.332878628057,0.332828881453,0.332779154755,0.332729447952,0.332679761032,0.332630093986,0.332580446801,0.332530819466,0.332481211971,0.332431624305,0.332382056455,0.332332508412,0.332282980163,0.332233471699,0.332183983007,0.332134514077,0.332085064898,0.332035635459,0.331986225749,0.331936835756,0.331887465469,0.331838114879,0.331788783972,0.33173947274,0.33169018117,0.331640909251,0.331591656974,0.331542424325,0.331493211296,0.331444017874,0.331394844049,0.33134568981,0.331296555146,0.331247440045,0.331198344498,0.331149268493,0.331100212019,0.331051175065,0.331002157621,0.330953159675,0.330904181216,0.330855222235,0.330806282719,0.330757362659,0.330708462042,0.330659580859,0.330610719098,0.330561876749,0.330513053801,0.330464250243,0.330415466064,0.330366701253,0.3303179558,0.330269229693,0.330220522923,0.330171835478,0.330123167347,0.330074518521,0.330025888987,0.329977278735,0.329928687755,0.329880116035,0.329831563565,0.329783030335,0.329734516333,0.329686021549,0.329637545973,0.329589089592,0.329540652398,0.329492234378,0.329443835523,0.329395455822,0.329347095264,0.329298753838,0.329250431534,0.329202128342,0.32915384425,0.329105579247,0.329057333324,0.32900910647,0.328960898674,0.328912709925,0.328864540214,0.328816389528,0.328768257859,0.328720145194,0.328672051524,0.328623976838,0.328575921126,0.328527884377,0.32847986658,0.328431867725,0.328383887802,0.328335926799,0.328287984707,0.328240061514,0.328192157211,0.328144271787,0.328096405232,0.328048557534,0.328000728684,0.327952918671,0.327905127484,0.327857355114,0.327809601549,0.327761866779,0.327714150795,0.327666453585,0.327618775138,0.327571115446,0.327523474496,0.32747585228,0.327428248786,0.327380664004,0.327333097923,0.327285550534,0.327238021826,0.327190511789,0.327143020411,0.327095547684,0.327048093596,0.327000658138,0.326953241298,0.326905843068,0.326858463435,0.326811102391,0.326763759924,0.326716436025,0.326669130683,0.326621843888,0.32657457563,0.326527325898,0.326480094682,0.326432881973,0.326385687758,0.32633851203,0.326291354776,0.326244215988,0.326197095654,0.326149993765,0.326102910311,0.32605584528,0.326008798664,0.325961770451,0.325914760633,0.325867769197,0.325820796135,0.325773841437,0.325726905091,0.325679987088,0.325633087418,0.325586206071,0.325539343036,0.325492498304,0.325445671864,0.325398863706,0.325352073821,0.325305302197,0.325258548826,0.325211813696,0.325165096799,0.325118398123,0.325071717658,0.325025055396,0.324978411325,0.324931785436,0.324885177718,0.324838588162,0.324792016758,0.324745463495,0.324698928363,0.324652411353,0.324605912455,0.324559431658,0.324512968953,0.32446652433,0.324420097778,0.324373689288,0.32432729885,0.324280926453,0.324234572089,0.324188235746,0.324141917416,0.324095617088,0.324049334751,0.324003070398,0.323956824016,0.323910595597,0.323864385131,0.323818192608,0.323772018017,0.323725861349,0.323679722595,0.323633601744,0.323587498786,0.323541413712,0.323495346512,0.323449297176,0.323403265694,0.323357252056,0.323311256253,0.323265278274,0.323219318111,0.323173375752,0.323127451189,0.323081544412,0.32303565541,0.322989784174,0.322943930695,0.322898094962,0.322852276966,0.322806476697,0.322760694145,0.322714929301,0.322669182155,0.322623452697,0.322577740917,0.322532046806,0.322486370354,0.322440711552,0.322395070389,0.322349446856,0.322303840944,0.322258252642,0.322212681941,0.322167128832,0.322121593304,0.322076075348,0.322030574955,0.321985092115,0.321939626818,0.321894179054,0.321848748815,0.321803336089,0.321757940869,0.321712563144,0.321667202904,0.32162186014,0.321576534843,0.321531227003,0.32148593661,0.321440663655,0.321395408128,0.32135017002,0.321304949321,0.321259746021,0.321214560112,0.321169391583,0.321124240426,0.32107910663,0.321033990186,0.320988891085,0.320943809316,0.320898744872,0.320853697742,0.320808667916,0.320763655385,0.320718660141,0.320673682172,0.320628721471,0.320583778027,0.320538851831,0.320493942874,0.320449051146,0.320404176637,0.320359319339,0.320314479242,0.320269656337,0.320224850614,0.320180062063,0.320135290676,0.320090536443,0.320045799355,0.320001079402,0.319956376575,0.319911690865,0.319867022262,0.319822370757,0.319777736341,0.319733119004,0.319688518737,0.31964393553,0.319599369376,0.319554820263,0.319510288183,0.319465773126,0.319421275084,0.319376794047,0.319332330006,0.31928788295,0.319243452873,0.319199039763,0.319154643611,0.31911026441,0.319065902148,0.319021556818,0.318977228409,0.318932916913,0.31888862232,0.318844344621,0.318800083807,0.318755839869,0.318711612798,0.318667402584,0.318623209218,0.318579032692,0.318534872995,0.318490730119,0.318446604054,0.318402494793,0.318358402324,0.31831432664,0.31827026773,0.318226225587,0.3181822002,0.318138191561,0.318094199661,0.31805022449,0.31800626604,0.317962324301,0.317918399265,0.317874490921,0.317830599262,0.317786724278,0.317742865959,0.317699024298,0.317655199285,0.31761139091,0.317567599166,0.317523824042,0.31748006553,0.317436323621,0.317392598306,0.317348889575,0.31730519742,0.317261521832,0.317217862802,0.31717422032,0.317130594379,0.317086984968,0.317043392079,0.316999815704,0.316956255832,0.316912712455,0.316869185564,0.316825675151,0.316782181206,0.31673870372,0.316695242684,0.31665179809,0.316608369929,0.316564958192,0.316521562869,0.316478183952,0.316434821432,0.3163914753,0.316348145548,0.316304832166,0.316261535145,0.316218254477,0.316174990154,0.316131742165,0.316088510502,0.316045295157,0.31600209612,0.315958913382,0.315915746936,0.315872596772,0.315829462881,0.315786345255,0.315743243884,0.31570015876,0.315657089874,0.315614037217,0.315571000782,0.315527980557,0.315484976536,0.31544198871,0.315399017068,0.315356061604,0.315313122307,0.31527019917,0.315227292184,0.315184401339,0.315141526628,0.315098668041,0.31505582557,0.315012999206,0.31497018894,0.314927394764,0.314884616669,0.314841854646,0.314799108687,0.314756378783,0.314713664925,0.314670967105,0.314628285314,0.314585619543,0.314542969784,0.314500336028,0.314457718267,0.314415116492,0.314372530693,0.314329960864,0.314287406995,0.314244869077,0.314202347101,0.314159841061,0.314117350945,0.314074876747,0.314032418458,0.313989976068,0.31394754957,0.313905138955,0.313862744214,0.313820365339,0.313778002321,0.313735655152,0.313693323823,0.313651008326,0.313608708652,0.313566424792,0.313524156739,0.313481904483,0.313439668016,0.31339744733,0.313355242416,0.313313053266,0.313270879871,0.313228722223,0.313186580313,0.313144454133,0.313102343674,0.313060248928,0.313018169887,0.312976106541,0.312934058884,0.312892026905,0.312850010597,0.312808009952,0.312766024961,0.312724055615,0.312682101906,0.312640163826,0.312598241367,0.312556334519,0.312514443275,0.312472567627,0.312430707565,0.312388863082,0.312347034168,0.312305220817,0.312263423019,0.312221640767,0.312179874051,0.312138122863,0.312096387196,0.312054667041,0.312012962389,0.311971273232,0.311929599563,0.311887941371,0.311846298651,0.311804671392,0.311763059587,0.311721463227,0.311679882305,0.311638316812,0.311596766739,0.311555232079,0.311513712823,0.311472208963,0.31143072049,0.311389247398,0.311347789676,0.311306347317,0.311264920313,0.311223508656,0.311182112337,0.311140731349,0.311099365682,0.311058015329,0.311016680282,0.310975360532,0.310934056071,0.310892766892,0.310851492985,0.310810234343,0.310768990958,0.310727762821,0.310686549924,0.31064535226,0.31060416982,0.310563002595,0.310521850579,0.310480713762,0.310439592136,0.310398485694,0.310357394427,0.310316318328,0.310275257388,0.310234211598,0.310193180952,0.310152165441,0.310111165056,0.31007017979,0.310029209635,0.309988254582,0.309947314624,0.309906389753,0.30986547996,0.309824585237,0.309783705577,0.309742840972,0.309701991412,0.309661156891,0.309620337401,0.309579532932,0.309538743479,0.309497969031,0.309457209582,0.309416465123,0.309375735646,0.309335021144,0.309294321608,0.309253637031,0.309212967404,0.30917231272,0.309131672971,0.309091048148,0.309050438244,0.30900984325,0.308969263159,0.308928697964,0.308888147655,0.308847612226,0.308807091667,0.308766585972,0.308726095132,0.30868561914,0.308645157987,0.308604711666,0.308564280169,0.308523863488,0.308483461614,0.308443074541,0.308402702261,0.308362344765,0.308322002045,0.308281674094,0.308241360905,0.308201062468,0.308160778776,0.308120509823,0.308080255598,0.308040016096,0.307999791307,0.307959581225,0.307919385841,0.307879205148,0.307839039137,0.307798887802,0.307758751134,0.307718629125,0.307678521768,0.307638429055,0.307598350978,0.30755828753,0.307518238702,0.307478204487,0.307438184877,0.307398179865,0.307358189442,0.307318213601,0.307278252335,0.307238305635,0.307198373494,0.307158455904,0.307118552858,0.307078664347,0.307038790364,0.306998930902,0.306959085952,0.306919255507,0.306879439559,0.306839638102,0.306799851126,0.306760078624,0.306720320589,0.306680577013,0.306640847888,0.306601133207,0.306561432962,0.306521747146,0.30648207575,0.306442418768,0.306402776191,0.306363148012,0.306323534223,0.306283934818,0.306244349787,0.306204779124,0.306165222821,0.30612568087,0.306086153264,0.306046639996,0.306007141057,0.30596765644,0.305928186138,0.305888730143,0.305849288447,0.305809861043,0.305770447924,0.305731049081,0.305691664508,0.305652294196,0.305612938139,0.305573596329,0.305534268757,0.305494955418,0.305455656302,0.305416371404,0.305377100715,0.305337844227,0.305298601934,0.305259373827,0.3052201599,0.305180960145,0.305141774554,0.305102603119,0.305063445835,0.305024302692,0.304985173684,0.304946058803,0.304906958042,0.304867871393,0.304828798849,0.304789740403,0.304750696046,0.304711665772,0.304672649573,0.304633647442,0.304594659371,0.304555685354,0.304516725382,0.304477779448,0.304438847545,0.304399929665,0.304361025801,0.304322135947,0.304283260093,0.304244398234,0.304205550361,0.304166716467,0.304127896546,0.304089090589,0.304050298589,0.30401152054,0.303972756433,0.303934006261,0.303895270017,0.303856547694,0.303817839285,0.303779144781,0.303740464176,0.303701797463,0.303663144634,0.303624505682,0.303585880599,0.303547269379,0.303508672014,0.303470088497,0.30343151882,0.303392962977,0.30335442096,0.303315892761,0.303277378374,0.303238877792,0.303200391006,0.303161918011,0.303123458798,0.303085013361,0.303046581692,0.303008163783,0.302969759629,0.302931369221,0.302892992553,0.302854629617,0.302816280406,0.302777944912,0.30273962313,0.302701315051,0.302663020668,0.302624739974,0.302586472962,0.302548219626,0.302509979957,0.302471753948,0.302433541593,0.302395342884,0.302357157814,0.302318986376,0.302280828563,0.302242684368,0.302204553784,0.302166436803,0.302128333418,0.302090243623,0.30205216741,0.302014104772,0.301976055702,0.301938020193,0.301899998238,0.30186198983,0.301823994961,0.301786013625,0.301748045815,0.301710091523,0.301672150743,0.301634223467,0.301596309689,0.3015584094,0.301520522596,0.301482649267,0.301444789408,0.301406943011,0.301369110069,0.301331290575,0.301293484523,0.301255691904,0.301217912713,0.301180146942,0.301142394584,0.301104655633,0.30106693008,0.30102921792,0.300991519146,0.300953833749,0.300916161724,0.300878503063,0.300840857759,0.300803225806,0.300765607197,0.300728001924,0.300690409981,0.30065283136,0.300615266055,0.300577714059,0.300540175365,0.300502649966,0.300465137855,0.300427639025,0.300390153469,0.300352681181,0.300315222153,0.300277776379,0.300240343851,0.300202924563,0.300165518508,0.30012812568,0.30009074607,0.300053379672,0.300016026481,0.299978686487,0.299941359686,0.299904046069,0.29986674563,0.299829458363,0.299792184259,0.299754923314,0.299717675519,0.299680440868,0.299643219354,0.29960601097,0.29956881571,0.299531633566,0.299494464532,0.299457308601,0.299420165766,0.299383036021,0.299345919359,0.299308815772,0.299271725254,0.299234647799,0.299197583399,0.299160532048,0.299123493739,0.299086468465,0.29904945622,0.299012456997,0.298975470788,0.298938497588,0.29890153739,0.298864590186,0.29882765597,0.298790734736,0.298753826476,0.298716931185,0.298680048854,0.298643179478,0.29860632305,0.298569479563,0.29853264901,0.298495831386,0.298459026682,0.298422234892,0.298385456011,0.29834869003,0.298311936944,0.298275196745,0.298238469427,0.298201754984,0.298165053409,0.298128364694,0.298091688834,0.298055025822,0.298018375651,0.297981738314,0.297945113806,0.297908502118,0.297871903245,0.297835317181,0.297798743918,0.297762183449,0.297725635769,0.29768910087,0.297652578747,0.297616069392,0.297579572799,0.297543088961,0.297506617871,0.297470159524,0.297433713913,0.29739728103,0.29736086087,0.297324453426,0.297288058691,0.297251676658,0.297215307322,0.297178950676,0.297142606713,0.297106275426,0.29706995681,0.297033650857,0.296997357561,0.296961076916,0.296924808914,0.296888553551,0.296852310818,0.296816080709,0.296779863219,0.29674365834,0.296707466066,0.296671286391,0.296635119308,0.29659896481,0.296562822892,0.296526693546,0.296490576766,0.296454472546,0.296418380879,0.296382301759,0.296346235179,0.296310181134,0.296274139615,0.296238110618,0.296202094136,0.296166090161,0.296130098688,0.296094119711,0.296058153223,0.296022199217,0.295986257687,0.295950328626,0.29591441203,0.29587850789,0.2958426162,0.295806736955,0.295770870147,0.295735015771,0.295699173819,0.295663344287,0.295627527166,0.295591722451,0.295555930136,0.295520150214,0.295484382679,0.295448627524,0.295412884743,0.29537715433,0.295341436279,0.295305730582,0.295270037235,0.295234356229,0.29519868756,0.295163031221,0.295127387205,0.295091755506,0.295056136118,0.295020529035,0.29498493425,0.294949351757,0.294913781549,0.294878223621,0.294842677966,0.294807144578,0.294771623451,0.294736114577,0.294700617952,0.294665133568,0.29462966142,0.294594201502,0.294558753806,0.294523318327,0.294487895058,0.294452483994,0.294417085128,0.294381698453,0.294346323964,0.294310961654,0.294275611518,0.294240273548,0.294204947739,0.294169634085,0.294134332579,0.294099043214,0.294063765986,0.294028500887,0.293993247912,0.293958007054,0.293922778308,0.293887561666,0.293852357123,0.293817164672,0.293781984308,0.293746816025,0.293711659815,0.293676515673,0.293641383593,0.293606263569,0.293571155594,0.293536059663,0.293500975768,0.293465903905,0.293430844067,0.293395796248,0.293360760441,0.293325736641,0.293290724842,0.293255725036,0.293220737219,0.293185761385,0.293150797526,0.293115845637,0.293080905712,0.293045977745,0.29301106173,0.29297615766,0.29294126553,0.292906385333,0.292871517064,0.292836660716,0.292801816283,0.292766983759,0.292732163139,0.292697354416,0.292662557584,0.292627772636,0.292592999568,0.292558238373,0.292523489045,0.292488751577,0.292454025965,0.292419312201,0.29238461028,0.292349920196,0.292315241943,0.292280575514,0.292245920905,0.292211278108,0.292176647118,0.292142027928,0.292107420534,0.292072824928,0.292038241106,0.29200366906,0.291969108785,0.291934560275,0.291900023524,0.291865498526,0.291830985275,0.291796483765,0.29176199399,0.291727515945,0.291693049622,0.291658595017,0.291624152124,0.291589720936,0.291555301447,0.291520893652,0.291486497545,0.29145211312,0.29141774037,0.29138337929,0.291349029874,0.291314692117,0.291280366011,0.291246051552,0.291211748734,0.29117745755,0.291143177994,0.291108910062,0.291074653746,0.291040409041,0.291006175942,0.290971954442,0.290937744535,0.290903546216,0.290869359478,0.290835184317,0.290801020725,0.290766868697,0.290732728228,0.290698599312,0.290664481942,0.290630376112,0.290596281818,0.290562199053,0.290528127811,0.290494068087,0.290460019875,0.290425983168,0.290391957962,0.29035794425,0.290323942026,0.290289951285,0.290255972021,0.290222004228,0.2901880479,0.290154103032,0.290120169618,0.290086247652,0.290052337128,0.29001843804,0.289984550383,0.289950674151,0.289916809338,0.289882955939,0.289849113947,0.289815283358,0.289781464164,0.289747656361,0.289713859943,0.289680074903,0.289646301237,0.289612538939,0.289578788002,0.289545048421,0.289511320191,0.289477603305,0.289443897759,0.289410203545,0.289376520659,0.289342849095,0.289309188848,0.28927553991,0.289241902278,0.289208275944,0.289174660904,0.289141057152,0.289107464682,0.289073883488,0.289040313565,0.289006754907,0.288973207508,0.288939671363,0.288906146467,0.288872632812,0.288839130395,0.288805639209,0.288772159248,0.288738690507,0.288705232981,0.288671786663,0.288638351548,0.288604927631,0.288571514905,0.288538113365,0.288504723007,0.288471343823,0.288437975808,0.288404618957,0.288371273264,0.288337938724,0.288304615331,0.288271303079,0.288238001963,0.288204711977,0.288171433116,0.288138165374,0.288104908745,0.288071663225,0.288038428806,0.288005205485,0.287971993255,0.28793879211,0.287905602046,0.287872423056,0.287839255135,0.287806098278,0.287772952479,0.287739817732,0.287706694032,0.287673581374,0.287640479751,0.287607389159,0.287574309592,0.287541241044,0.28750818351,0.287475136984,0.287442101461,0.287409076935,0.287376063401,0.287343060854,0.287310069287,0.287277088696,0.287244119074,0.287211160417,0.287178212719,0.287145275974,0.287112350178,0.287079435323,0.287046531406,0.287013638421,0.286980756361,0.286947885223,0.286915024999,0.286882175685,0.286849337276,0.286816509766,0.286783693149,0.28675088742,0.286718092573,0.286685308604,0.286652535507,0.286619773276,0.286587021906,0.286554281391,0.286521551727,0.286488832907,0.286456124927,0.28642342778,0.286390741462,0.286358065967,0.28632540129,0.286292747425,0.286260104368,0.286227472111,0.286194850651,0.286162239982,0.286129640098,0.286097050995,0.286064472666,0.286031905106,0.28599934831,0.285966802273,0.28593426699,0.285901742454,0.285869228661,0.285836725606,0.285804233282,0.285771751685,0.285739280809,0.285706820649,0.285674371199,0.285641932455,0.285609504411,0.285577087062,0.285544680402,0.285512284426,0.285479899129,0.285447524505,0.285415160549,0.285382807257,0.285350464622,0.285318132639,0.285285811303,0.285253500608,0.285221200551,0.285188911124,0.285156632323,0.285124364143,0.285092106578,0.285059859624,0.285027623274,0.284995397523,0.284963182367,0.2849309778,0.284898783817,0.284866600412,0.284834427581,0.284802265318,0.284770113618,0.284737972475,0.284705841885,0.284673721841,0.28464161234,0.284609513376,0.284577424943,0.284545347036,0.28451327965,0.284481222781,0.284449176422,0.284417140568,0.284385115215,0.284353100357,0.284321095989,0.284289102105,0.284257118701,0.284225145771,0.284193183311,0.284161231314,0.284129289777,0.284097358693,0.284065438057,0.284033527865,0.284001628111,0.28396973879,0.283937859897,0.283905991427,0.283874133374,0.283842285733,0.2838104485,0.283778621669,0.283746805235,0.283714999193,0.283683203538,0.283651418264,0.283619643367,0.283587878841,0.283556124682,0.283524380883,0.283492647441,0.28346092435,0.283429211604,0.283397509199,0.28336581713,0.283334135392,0.283302463978,0.283270802886,0.283239152108,0.283207511641,0.283175881479,0.283144261617,0.28311265205,0.283081052772,0.28304946378,0.283017885067,0.28298631663,0.282954758461,0.282923210558,0.282891672914,0.282860145524,0.282828628384,0.282797121489,0.282765624833,0.282734138411,0.282702662218,0.28267119625,0.282639740501,0.282608294966,0.282576859641,0.282545434519,0.282514019597,0.282482614869,0.28245122033,0.282419835976,0.2823884618,0.282357097799,0.282325743967,0.282294400299,0.28226306679,0.282231743435,0.28220043023,0.282169127168,0.282137834246,0.282106551459,0.2820752788,0.282044016266,0.282012763852,0.281981521551,0.28195028936,0.281919067274,0.281887855287,0.281856653395,0.281825461592,0.281794279874,0.281763108236,0.281731946673,0.281700795179,0.28166965375,0.281638522382,0.281607401068,0.281576289805,0.281545188587,0.281514097409,0.281483016266,0.281451945154,0.281420884068,0.281389833002,0.281358791952,0.281327760913,0.281296739879,0.281265728847,0.281234727811,0.281203736767,0.281172755708,0.281141784631,0.281110823531,0.281079872403,0.281048931241,0.281018000042,0.280987078799,0.280956167509,0.280925266167,0.280894374766,0.280863493304,0.280832621774,0.280801760173,0.280770908494,0.280740066734,0.280709234887,0.280678412949,0.280647600915,0.280616798779,0.280586006538,0.280555224186,0.280524451718,0.28049368913,0.280462936417,0.280432193574,0.280401460596,0.280370737478,0.280340024216,0.280309320805,0.28027862724,0.280247943516,0.280217269628,0.280186605572,0.280155951342,0.280125306935,0.280094672345,0.280064047567,0.280033432597,0.28000282743,0.27997223206,0.279941646485,0.279911070697,0.279880504694,0.279849948469,0.279819402019,0.279788865339,0.279758338423,0.279727821267,0.279697313866,0.279666816216,0.279636328312,0.279605850149,0.279575381722,0.279544923026,0.279514474058,0.279484034812,0.279453605283,0.279423185466,0.279392775358,0.279362374953,0.279331984246,0.279301603233,0.279271231909,0.27924087027,0.279210518311,0.279180176026,0.279149843412,0.279119520463,0.279089207175,0.279058903543,0.279028609563,0.27899832523,0.278968050539,0.278937785486,0.278907530065,0.278877284272,0.278847048103,0.278816821553,0.278786604617,0.27875639729,0.278726199569,0.278696011447,0.278665832921,0.278635663986,0.278605504637,0.27857535487,0.278545214679,0.278515084061,0.278484963011,0.278454851524,0.278424749595,0.27839465722,0.278364574394,0.278334501112,0.278304437371,0.278274383165,0.278244338489,0.27821430334,0.278184277712,0.278154261601,0.278124255002,0.278094257911,0.278064270323,0.278034292233,0.278004323638,0.277974364531,0.27794441491,0.277914474768,0.277884544102,0.277854622907,0.277824711179,0.277794808912,0.277764916103,0.277735032746,0.277705158837,0.277675294372,0.277645439346,0.277615593754,0.277585757593,0.277555930856,0.277526113541,0.277496305642,0.277466507154,0.277436718074,0.277406938396,0.277377168116,0.27734740723,0.277317655733,0.27728791362,0.277258180888,0.277228457531,0.277198743545,0.277169038925,0.277139343667,0.277109657767,0.27707998122,0.277050314021,0.277020656166,0.27699100765,0.276961368469,0.276931738619,0.276902118095,0.276872506892,0.276842905006,0.276813312432,0.276783729167,0.276754155205,0.276724590542,0.276695035173,0.276665489095,0.276635952302,0.276606424791,0.276576906556,0.276547397593,0.276517897898,0.276488407467,0.276458926294,0.276429454376,0.276399991708,0.276370538285,0.276341094104,0.276311659159,0.276282233446,0.276252816962,0.2762234097,0.276194011658,0.27616462283,0.276135243212,0.276105872799,0.276076511589,0.276047159574,0.276017816753,0.275988483119,0.275959158669,0.275929843398,0.275900537302,0.275871240376,0.275841952617,0.275812674019,0.275783404578,0.27575414429,0.275724893151,0.275695651155,0.2756664183,0.275637194579,0.27560797999,0.275578774527,0.275549578186,0.275520390964,0.275491212854,0.275462043854,0.275432883959,0.275403733163,0.275374591464,0.275345458857,0.275316335337,0.2752872209,0.275258115542,0.275229019257,0.275199932043,0.275170853895,0.275141784808,0.275112724777,0.2750836738,0.27505463187,0.275025598985,0.274996575139,0.274967560329,0.27493855455,0.274909557797,0.274880570067,0.274851591355,0.274822621656,0.274793660968,0.274764709284,0.274735766601,0.274706832915,0.274677908221,0.274648992515,0.274620085793,0.274591188051,0.274562299283,0.274533419486,0.274504548656,0.274475686789,0.274446833879,0.274417989923,0.274389154917,0.274360328855,0.274331511735,0.274302703552,0.274273904301,0.274245113978,0.274216332579,0.2741875601,0.274158796536,0.274130041884,0.274101296139,0.274072559296,0.274043831352,0.274015112302,0.273986402142,0.273957700868,0.273929008475,0.27390032496,0.273871650318,0.273842984544,0.273814327636,0.273785679588,0.273757040396,0.273728410056,0.273699788564,0.273671175916,0.273642572107,0.273613977134,0.273585390991,0.273556813675,0.273528245182,0.273499685508,0.273471134647,0.273442592597,0.273414059353,0.27338553491,0.273357019265,0.273328512413,0.273300014351,0.273271525073,0.273243044577,0.273214572857,0.273186109909,0.27315765573,0.273129210315,0.27310077366,0.273072345762,0.273043926614,0.273015516215,0.272987114559,0.272958721642,0.27293033746,0.27290196201,0.272873595286,0.272845237285,0.272816888003,0.272788547435,0.272760215578,0.272731892427,0.272703577978,0.272675272227,0.27264697517,0.272618686802,0.272590407121,0.27256213612,0.272533873798,0.272505620149,0.272477375168,0.272449138854,0.2724209112,0.272392692203,0.272364481859,0.272336280164,0.272308087113,0.272279902703,0.272251726929,0.272223559788,0.272195401276,0.272167251387,0.272139110119,0.272110977467,0.272082853427,0.272054737995,0.272026631167,0.271998532939,0.271970443307,0.271942362266,0.271914289813,0.271886225944,0.271858170654,0.27183012394,0.271802085798,0.271774056222,0.271746035211,0.271718022759,0.271690018862,0.271662023516,0.271634036718,0.271606058463,0.271578088747,0.271550127566,0.271522174917,0.271494230795,0.271466295196,0.271438368116,0.271410449551,0.271382539498,0.271354637951,0.271326744908,0.271298860364,0.271270984314,0.271243116756,0.271215257685,0.271187407098,0.271159564989,0.271131731356,0.271103906193,0.271076089498,0.271048281266,0.271020481493,0.270992690176,0.27096490731,0.270937132891,0.270909366915,0.270881609379,0.270853860278,0.270826119609,0.270798387367,0.270770663548,0.27074294815,0.270715241166,0.270687542595,0.270659852431,0.270632170671,0.27060449731,0.270576832346,0.270549175774,0.270521527589,0.270493887789,0.270466256368,0.270438633324,0.270411018652,0.270383412349,0.27035581441,0.270328224831,0.270300643609,0.27027307074,0.27024550622,0.270217950044,0.270190402209,0.270162862712,0.270135331547,0.270107808712,0.270080294202,0.270052788013,0.270025290142,0.269997800584,0.269970319337,0.269942846395,0.269915381755,0.269887925413,0.269860477365,0.269833037608,0.269805606137,0.269778182949,0.269750768039,0.269723361404,0.26969596304,0.269668572944,0.26964119111,0.269613817536,0.269586452217,0.26955909515,0.269531746331,0.269504405756,0.26947707342,0.269449749321,0.269422433455,0.269395125817,0.269367826403,0.269340535211,0.269313252235,0.269285977473,0.26925871092,0.269231452573,0.269204202427,0.269176960479,0.269149726725,0.269122501162,0.269095283785,0.26906807459,0.269040873575,0.269013680734,0.268986496064,0.268959319562,0.268932151223,0.268904991044,0.268877839021,0.26885069515,0.268823559428,0.26879643185,0.268769312413,0.268742201112,0.268715097945,0.268688002907,0.268660915995,0.268633837205,0.268606766533,0.268579703975,0.268552649527,0.268525603187,0.268498564949,0.26847153481,0.268444512767,0.268417498815,0.268390492952,0.268363495172,0.268336505473,0.26830952385,0.268282550301,0.26825558482,0.268228627405,0.268201678051,0.268174736756,0.268147803514,0.268120878323,0.268093961178,0.268067052077,0.268040151014,0.268013257987,0.267986372992,0.267959496025,0.267932627082,0.267905766159,0.267878913254,0.267852068361,0.267825231478,0.2677984026,0.267771581725,0.267744768848,0.267717963965,0.267691167073,0.267664378169,0.267637597247,0.267610824306,0.26758405934,0.267557302347,0.267530553323,0.267503812264,0.267477079165,0.267450354025,0.267423636838,0.267396927602,0.267370226312,0.267343532965,0.267316847557,0.267290170085,0.267263500544,0.267236838932,0.267210185244,0.267183539477,0.267156901627,0.267130271691,0.267103649665,0.267077035545,0.267050429327,0.267023831009,0.266997240585,0.266970658053,0.26694408341,0.26691751665,0.266890957771,0.266864406769,0.266837863641,0.266811328382,0.26678480099,0.26675828146,0.266731769789,0.266705265973,0.266678770009,0.266652281892,0.266625801621,0.26659932919,0.266572864596,0.266546407835,0.266519958905,0.2664935178,0.266467084519,0.266440659057,0.26641424141,0.266387831575,0.266361429548,0.266335035326,0.266308648905,0.266282270281,0.266255899452,0.266229536413,0.26620318116,0.266176833691,0.266150494001,0.266124162087,0.266097837946,0.266071521574,0.266045212966,0.266018912121,0.265992619033,0.2659663337,0.265940056119,0.265913786284,0.265887524193,0.265861269843,0.265835023229,0.265808784349,0.265782553198,0.265756329773,0.265730114071,0.265703906088,0.26567770582,0.265651513264,0.265625328416,0.265599151274,0.265572981832,0.265546820088,0.265520666039,0.26549451968,0.265468381008,0.26544225002,0.265416126712,0.26539001108,0.265363903122,0.265337802833,0.26531171021,0.265285625249,0.265259547948,0.265233478302,0.265207416307,0.265181361962,0.265155315261,0.265129276202,0.26510324478,0.265077220993,0.265051204837,0.265025196309,0.264999195404,0.26497320212,0.264947216453,0.264921238399,0.264895267956,0.264869305119,0.264843349884,0.26481740225,0.264791462211,0.264765529765,0.264739604909,0.264713687638,0.264687777949,0.264661875839,0.264635981304,0.264610094341,0.264584214946,0.264558343116,0.264532478847,0.264506622137,0.264480772981,0.264454931376,0.264429097319,0.264403270805,0.264377451833,0.264351640398,0.264325836496,0.264300040125,0.264274251281,0.264248469961,0.264222696161,0.264196929877,0.264171171106,0.264145419846,0.264119676091,0.26409393984,0.264068211088,0.264042489832,0.264016776069,0.263991069796,0.263965371008,0.263939679702,0.263913995876,0.263888319525,0.263862650647,0.263836989237,0.263811335293,0.26378568881,0.263760049787,0.263734418218,0.263708794102,0.263683177434,0.263657568211,0.263631966429,0.263606372086,0.263580785178,0.263555205701,0.263529633653,0.263504069029,0.263478511826,0.263452962042,0.263427419672,0.263401884714,0.263376357163,0.263350837017,0.263325324272,0.263299818925,0.263274320972,0.26324883041,0.263223347236,0.263197871447,0.263172403038,0.263146942007,0.263121488351,0.263096042065,0.263070603147,0.263045171593,0.2630197474,0.262994330565,0.262968921084,0.262943518954,0.262918124171,0.262892736733,0.262867356636,0.262841983876,0.262816618451,0.262791260356,0.262765909589,0.262740566147,0.262715230025,0.262689901221,0.262664579731,0.262639265553,0.262613958682,0.262588659115,0.26256336685,0.262538081882,0.262512804209,0.262487533827,0.262462270733,0.262437014924,0.262411766395,0.262386525145,0.26236129117,0.262336064466,0.26231084503,0.262285632859,0.262260427949,0.262235230298,0.262210039902,0.262184856757,0.262159680861,0.26213451221,0.262109350801,0.262084196631,0.262059049696,0.262033909993,0.262008777518,0.26198365227,0.261958534243,0.261933423436,0.261908319844,0.261883223465,0.261858134295,0.261833052331,0.26180797757,0.261782910009,0.261757849643,0.261732796471,0.261707750488,0.261682711692,0.261657680079,0.261632655647,0.261607638391,0.261582628308,0.261557625396,0.261532629651,0.26150764107,0.261482659649,0.261457685386,0.261432718277,0.26140775832,0.261382805509,0.261357859844,0.261332921319,0.261307989933,0.261283065682,0.261258148562,0.261233238571,0.261208335705,0.261183439961,0.261158551336,0.261133669827,0.26110879543,0.261083928143,0.261059067961,0.261034214882,0.261009368903,0.260984530021,0.260959698232,0.260934873532,0.26091005592,0.260885245392,0.260860441944,0.260835645573,0.260810856277,0.260786074051,0.260761298894,0.260736530801,0.26071176977,0.260687015797,0.260662268879,0.260637529013,0.260612796196,0.260588070424,0.260563351695,0.260538640006,0.260513935353,0.260489237732,0.260464547142,0.260439863578,0.260415187038,0.260390517519,0.260365855016,0.260341199528,0.260316551051,0.260291909582,0.260267275117,0.260242647654,0.26021802719,0.26019341372,0.260168807243,0.260144207756,0.260119615253,0.260095029734,0.260070451195,0.260045879632,0.260021315042,0.259996757423,0.259972206771,0.259947663083,0.259923126356,0.259898596587,0.259874073773,0.25984955791,0.259825048996,0.259800547027,0.259776052001,0.259751563914,0.259727082763,0.259702608544,0.259678141256,0.259653680895,0.259629227458,0.259604780941,0.259580341341,0.259555908657,0.259531482883,0.259507064018,0.259482652058,0.259458247,0.259433848842,0.259409457579,0.259385073209,0.259360695729,0.259336325136,0.259311961426,0.259287604598,0.259263254646,0.259238911569,0.259214575364,0.259190246027,0.259165923555,0.259141607946,0.259117299196,0.259092997301,0.25906870226,0.259044414069,0.259020132725,0.258995858225,0.258971590566,0.258947329745,0.258923075759,0.258898828604,0.258874588278,0.258850354777,0.2588261281,0.258801908242,0.2587776952,0.258753488973,0.258729289555,0.258705096945,0.25868091114,0.258656732136,0.258632559931,0.258608394521,0.258584235903,0.258560084075,0.258535939033,0.258511800774,0.258487669296,0.258463544595,0.258439426669,0.258415315513,0.258391211126,0.258367113505,0.258343022645,0.258318938545,0.258294861202,0.258270790611,0.258246726771,0.258222669678,0.25819861933,0.258174575723,0.258150538854,0.258126508721,0.25810248532,0.258078468648,0.258054458703,0.258030455481,0.25800645898,0.257982469196,0.257958486127,0.257934509769,0.25791054012,0.257886577176,0.257862620935,0.257838671394,0.257814728549,0.257790792398,0.257766862938,0.257742940166,0.257719024079,0.257695114673,0.257671211947,0.257647315896,0.257623426518,0.257599543811,0.257575667771,0.257551798395,0.25752793568,0.257504079623,0.257480230222,0.257456387473,0.257432551374,0.257408721921,0.257384899112,0.257361082944,0.257337273413,0.257313470517,0.257289674253,0.257265884619,0.25724210161,0.257218325224,0.257194555458,0.25717079231,0.257147035776,0.257123285854,0.25709954254,0.257075805831,0.257052075726,0.25702835222,0.257004635311,0.256980924996,0.256957221272,0.256933524137,0.256909833586,0.256886149618,0.25686247223,0.256838801418,0.25681513718,0.256791479512,0.256767828412,0.256744183878,0.256720545905,0.256696914492,0.256673289635,0.256649671332,0.256626059579,0.256602454374,0.256578855713,0.256555263595,0.256531678016,0.256508098973,0.256484526463,0.256460960483,0.256437401031,0.256413848104,0.256390301699,0.256366761813,0.256343228443,0.256319701586,0.256296181239,0.2562726674,0.256249160066,0.256225659234,0.2562021649,0.256178677063,0.256155195719,0.256131720865,0.256108252499,0.256084790617,0.256061335218,0.256037886297,0.256014443853,0.255991007882,0.255967578382,0.255944155349,0.255920738781,0.255897328676,0.255873925029,0.255850527839,0.255827137102,0.255803752816,0.255780374978,0.255757003585,0.255733638634,0.255710280122,0.255686928048,0.255663582407,0.255640243196,0.255616910414,0.255593584058,0.255570264124,0.255546950609,0.255523643512,0.255500342829,0.255477048557,0.255453760693,0.255430479235,0.25540720418,0.255383935526,0.255360673268,0.255337417405,0.255314167934,0.255290924851,0.255267688155,0.255244457842,0.25522123391,0.255198016355,0.255174805176,0.255151600368,0.25512840193,0.255105209859,0.255082024152,0.255058844805,0.255035671817,0.255012505185,0.254989344905,0.254966190975,0.254943043393,0.254919902155,0.254896767258,0.254873638701,0.25485051648,0.254827400592,0.254804291035,0.254781187805,0.254758090901,0.254735000319,0.254711916057,0.254688838112,0.254665766481,0.254642701161,0.25461964215,0.254596589445,0.254573543043,0.254550502941,0.254527469137,0.254504441628,0.254481420411,0.254458405484,0.254435396843,0.254412394486,0.25438939841,0.254366408613,0.254343425092,0.254320447844,0.254297476866,0.254274512156,0.25425155371,0.254228601527,0.254205655603,0.254182715936,0.254159782523,0.254136855362,0.254113934449,0.254091019781,0.254068111357,0.254045209173,0.254022313228,0.253999423517,0.253976540038,0.253953662789,0.253930791767,0.253907926969,0.253885068393,0.253862216036,0.253839369894,0.253816529966,0.253793696249,0.25377086874,0.253748047437,0.253725232336,0.253702423435,0.253679620731,0.253656824222,0.253634033905,0.253611249777,0.253588471835,0.253565700078,0.253542934502,0.253520175104,0.253497421882,0.253474674834,0.253451933956,0.253429199246,0.253406470701,0.253383748318,0.253361032096,0.25333832203,0.25331561812,0.253292920361,0.253270228751,0.253247543288,0.253224863969,0.253202190791,0.253179523751,0.253156862848,0.253134208078,0.253111559438,0.253088916927,0.253066280541,0.253043650278,0.253021026134,0.252998408109,0.252975796198,0.252953190399,0.252930590709,0.252907997127,0.252885409649,0.252862828272,0.252840252995,0.252817683813,0.252795120726,0.25277256373,0.252750012822,0.252727468,0.252704929261,0.252682396603,0.252659870023,0.252637349518,0.252614835086,0.252592326724,0.25256982443,0.2525473282,0.252524838033,0.252502353926,0.252479875875,0.25245740388,0.252434937936,0.252412478041,0.252390024193,0.252367576389,0.252345134627,0.252322698903,0.252300269216,0.252277845563,0.25225542794,0.252233016346,0.252210610779,0.252188211234,0.25216581771,0.252143430204,0.252121048714,0.252098673237,0.25207630377,0.252053940311,0.252031582857,0.252009231406,0.251986885956,0.251964546502,0.251942213044,0.251919885578,0.251897564101,0.251875248612,0.251852939108,0.251830635586,0.251808338043,0.251786046477,0.251763760886,0.251741481266,0.251719207616,0.251696939933,0.251674678213,0.251652422456,0.251630172657,0.251607928815,0.251585690927,0.25156345899,0.251541233002,0.251519012961,0.251496798863,0.251474590707,0.251452388489,0.251430192207,0.251408001859,0.251385817442,0.251363638954,0.251341466391,0.251319299752,0.251297139034,0.251274984235,0.251252835351,0.251230692381,0.251208555321,0.25118642417,0.251164298924,0.251142179582,0.251120066141,0.251097958597,0.25107585695,0.251053761195,0.251031671331,0.251009587356,0.250987509266,0.250965437059,0.250943370733,0.250921310284,0.250899255712,0.250877207012,0.250855164183,0.250833127222,0.250811096127,0.250789070895,0.250767051523,0.25074503801,0.250723030352,0.250701028547,0.250679032592,0.250657042486,0.250635058225,0.250613079808,0.250591107231,0.250569140492,0.250547179589,0.250525224519,0.25050327528,0.250481331869,0.250459394284,0.250437462522,0.25041553658,0.250393616457,0.25037170215,0.250349793656,0.250327890973,0.250305994099,0.25028410303,0.250262217765,0.250240338301,0.250218464635,0.250196596766,0.25017473469,0.250152878405,0.250131027909,0.250109183199,0.250087344273,0.250065511128,0.250043683762,0.250021862173,0.250000046357,0.249978236313,0.249956432038,0.24993463353,0.249912840786,0.249891053803,0.24986927258,0.249847497114,0.249825727402,0.249803963442,0.249782205231,0.249760452768,0.249738706049,0.249716965072,0.249695229835,0.249673500336,0.249651776571,0.249630058539,0.249608346237,0.249586639662,0.249564938813,0.249543243687,0.249521554281,0.249499870592,0.24947819262,0.24945652036,0.249434853811,0.249413192971,0.249391537836,0.249369888404,0.249348244674,0.249326606642,0.249304974307,0.249283347665,0.249261726715,0.249240111453,0.249218501879,0.249196897988,0.249175299779,0.24915370725,0.249132120397,0.249110539219,0.249088963714,0.249067393878,0.249045829709,0.249024271206,0.249002718365,0.248981171184,0.248959629661,0.248938093794,0.248916563579,0.248895039016,0.2488735201,0.248852006831,0.248830499205,0.24880899722,0.248787500874,0.248766010164,0.248744525089,0.248723045645,0.24870157183,0.248680103642,0.248658641079,0.248637184138,0.248615732816,0.248594287112,0.248572847023,0.248551412547,0.248529983682,0.248508560424,0.248487142771,0.248465730723,0.248444324274,0.248422923425,0.248401528172,0.248380138512,0.248358754444,0.248337375965,0.248316003073,0.248294635766,0.24827327404,0.248251917894,0.248230567326,0.248209222332,0.248187882912,0.248166549061,0.248145220779,0.248123898062,0.248102580909,0.248081269317,0.248059963284,0.248038662806,0.248017367883,0.247996078512,0.24797479469,0.247953516415,0.247932243684,0.247910976496,0.247889714848,0.247868458738,0.247847208164,0.247825963122,0.247804723611,0.247783489628,0.247762261172,0.24774103824,0.247719820829,0.247698608937,0.247677402562,0.247656201702,0.247635006354,0.247613816515,0.247592632185,0.24757145336,0.247550280038,0.247529112216,0.247507949893,0.247486793067,0.247465641734,0.247444495892,0.24742335554,0.247402220675,0.247381091294,0.247359967396,0.247338848978,0.247317736038,0.247296628573,0.247275526581,0.24725443006,0.247233339008,0.247212253422,0.2471911733,0.24717009864,0.24714902944,0.247127965697,0.247106907408,0.247085854573,0.247064807187,0.24704376525,0.247022728759,0.247001697711,0.246980672105,0.246959651937,0.246938637206,0.24691762791,0.246896624046,0.246875625612,0.246854632606,0.246833645025,0.246812662867,0.24679168613,0.246770714812,0.24674974891,0.246728788423,0.246707833347,0.246686883681,0.246665939422,0.246645000569,0.246624067118,0.246603139068,0.246582216416,0.24656129916,0.246540387299,0.246519480829,0.246498579748,0.246477684055,0.246456793746,0.24643590882,0.246415029275,0.246394155108,0.246373286316,0.246352422899,0.246331564853,0.246310712176,0.246289864866,0.246269022922,0.246248186339,0.246227355117,0.246206529254,0.246185708746,0.246164893592,0.246144083789,0.246123279335,0.246102480229,0.246081686467,0.246060898048,0.246040114969,0.246019337229,0.245998564824,0.245977797753,0.245957036014,0.245936279604,0.245915528521,0.245894782763,0.245874042327,0.245853307212,0.245832577416,0.245811852935,0.245791133768,0.245770419913,0.245749711368,0.24572900813,0.245708310196,0.245687617566,0.245666930236,0.245646248205,0.24562557147,0.245604900029,0.24558423388,0.245563573021,0.245542917449,0.245522267163,0.24550162216,0.245480982438,0.245460347994,0.245439718828,0.245419094935,0.245398476315,0.245377862965,0.245357254883,0.245336652067,0.245316054514,0.245295462222,0.24527487519,0.245254293415,0.245233716895,0.245213145627,0.24519257961,0.245172018841,0.245151463318,0.24513091304,0.245110368003,0.245089828206,0.245069293646,0.245048764322,0.245028240231,0.245007721371,0.24498720774,0.244966699336,0.244946196157,0.244925698199,0.244905205463,0.244884717944,0.244864235641,0.244843758552,0.244823286675,0.244802820007,0.244782358547,0.244761902292,0.24474145124,0.244721005389,0.244700564737,0.244680129282,0.244659699021,0.244639273952,0.244618854074,0.244598439384,0.24457802988,0.24455762556,0.244537226422,0.244516832463,0.244496443682,0.244476060076,0.244455681644,0.244435308382,0.24441494029,0.244394577364,0.244374219604,0.244353867005,0.244333519568,0.244313177288,0.244292840165,0.244272508196,0.244252181379,0.244231859713,0.244211543193,0.24419123182,0.24417092559,0.244150624502,0.244130328553,0.244110037741,0.244089752065,0.244069471521,0.244049196109,0.244028925825,0.244008660668,0.243988400636,0.243968145726,0.243947895936,0.243927651265,0.243907411711,0.24388717727,0.243866947942,0.243846723723,0.243826504612,0.243806290608,0.243786081706,0.243765877907,0.243745679207,0.243725485604,0.243705297097,0.243685113683,0.24366493536,0.243644762127,0.24362459398,0.243604430918,0.243584272939,0.243564120041,0.243543972222,0.243523829479,0.243503691811,0.243483559216,0.24346343169,0.243443309233,0.243423191843,0.243403079516,0.243382972252,0.243362870048,0.243342772902,0.243322680812,0.243302593775,0.243282511791,0.243262434856,0.243242362969,0.243222296128,0.24320223433,0.243182177574,0.243162125857,0.243142079178,0.243122037534,0.243102000923,0.243081969344,0.243061942794,0.243041921271,0.243021904773,0.243001893298,0.242981886844,0.24296188541,0.242941888992,0.242921897589,0.242901911199,0.242881929819,0.242861953449,0.242841982085,0.242822015726,0.24280205437,0.242782098014,0.242762146657,0.242742200297,0.242722258931,0.242702322557,0.242682391174,0.242662464779,0.242642543371,0.242622626948,0.242602715506,0.242582809045,0.242562907563,0.242543011056,0.242523119524,0.242503232964,0.242483351375,0.242463474753,0.242443603098,0.242423736407,0.242403874679,0.24238401791,0.2423641661,0.242344319245,0.242324477345,0.242304640397,0.242284808399,0.24226498135,0.242245159246,0.242225342086,0.242205529869,0.242185722592,0.242165920252,0.242146122849,0.24212633038,0.242106542843,0.242086760236,0.242066982558,0.242047209805,0.242027441977,0.24200767907,0.241987921084,0.241968168016,0.241948419864,0.241928676626,0.241908938301,0.241889204885,0.241869476378,0.241849752777,0.24183003408,0.241810320286,0.241790611392,0.241770907396,0.241751208296,0.241731514091,0.241711824778,0.241692140355,0.241672460821,0.241652786174,0.24163311641,0.24161345153,0.241593791529,0.241574136408,0.241554486163,0.241534840792,0.241515200294,0.241495564667,0.241475933909,0.241456308017,0.24143668699,0.241417070826,0.241397459523,0.241377853079,0.241358251491,0.241338654759,0.24131906288,0.241299475851,0.241279893672,0.24126031634,0.241240743854,0.24122117621,0.241201613408,0.241182055445,0.24116250232,0.24114295403,0.241123410574,0.241103871949,0.241084338154,0.241064809186,0.241045285045,0.241025765727,0.241006251231,0.240986741555,0.240967236697,0.240947736655,0.240928241427,0.240908751011,0.240889265406,0.240869784609,0.240850308618,0.240830837432,0.240811371048,0.240791909465,0.24077245268,0.240753000693,0.2407335535,0.2407141111,0.240694673491,0.240675240671,0.240655812638,0.24063638939,0.240616970926,0.240597557243,0.240578148339,0.240558744213,0.240539344863,0.240519950286,0.240500560481,0.240481175446,0.240461795179,0.240442419678,0.240423048942,0.240403682967,0.240384321753,0.240364965297,0.240345613598,0.240326266653,0.240306924461,0.24028758702,0.240268254327,0.240248926382,0.240229603182,0.240210284725,0.240190971009,0.240171662033,0.240152357794,0.240133058291,0.240113763522,0.240094473484,0.240075188177,0.240055907597,0.240036631744,0.240017360615,0.239998094208,0.239978832522,0.239959575554,0.239940323304,0.239921075768,0.239901832945,0.239882594833,0.23986336143,0.239844132735,0.239824908745,0.239805689458,0.239786474873,0.239767264989,0.239748059802,0.239728859311,0.239709663514,0.23969047241,0.239671285996,0.23965210427,0.239632927232,0.239613754878,0.239594587207,0.239575424217,0.239556265907,0.239537112274,0.239517963316,0.239498819032,0.23947967942,0.239460544478,0.239441414204,0.239422288596,0.239403167653,0.239384051372,0.239364939751,0.23934583279,0.239326730485,0.239307632836,0.239288539839,0.239269451494,0.239250367799,0.239231288751,0.23921221435,0.239193144592,0.239174079476,0.239155019001,0.239135963164,0.239116911963,0.239097865398,0.239078823465,0.239059786164,0.239040753491,0.239021725447,0.239002702027,0.238983683232,0.238964669058,0.238945659505,0.23892665457,0.238907654251,0.238888658546,0.238869667455,0.238850680974,0.238831699103,0.238812721838,0.23879374918,0.238774781124,0.238755817671,0.238736858818,0.238717904562,0.238698954903,0.238680009839,0.238661069367,0.238642133486,0.238623202194,0.23860427549,0.23858535337,0.238566435835,0.238547522881,0.238528614507,0.238509710711,0.238490811492,0.238471916847,0.238453026775,0.238434141273,0.238415260341,0.238396383976,0.238377512177,0.238358644941,0.238339782268,0.238320924154,0.238302070598,0.2382832216,0.238264377155,0.238245537264,0.238226701924,0.238207871133,0.238189044889,0.238170223191,0.238151406038,0.238132593426,0.238113785354,0.238094981821,0.238076182825,0.238057388364,0.238038598436,0.238019813039,0.238001032172,0.237982255833,0.237963484019,0.23794471673,0.237925953963,0.237907195717,0.23788844199,0.23786969278,0.237850948085,0.237832207904,0.237813472235,0.237794741075,0.237776014424,0.237757292279,0.237738574639,0.237719861502,0.237701152865,0.237682448729,0.237663749089,0.237645053946,0.237626363296,0.237607677139,0.237588995473,0.237570318295,0.237551645604,0.237532977398,0.237514313676,0.237495654435,0.237476999674,0.237458349392,0.237439703586,0.237421062254,0.237402425396,0.237383793008,0.23736516509,0.23734654164,0.237327922655,0.237309308134,0.237290698076,0.237272092479,0.23725349134,0.237234894658,0.237216302432,0.237197714659,0.237179131339,0.237160552468,0.237141978046,0.23712340807,0.237104842539,0.237086281452,0.237067724805,0.237049172599,0.23703062483,0.237012081498,0.2369935426,0.236975008134,0.2369564781,0.236937952495,0.236919431317,0.236900914565,0.236882402238,0.236863894332,0.236845390847,0.236826891781,0.236808397132,0.236789906899,0.236771421079,0.236752939671,0.236734462673,0.236715990084,0.236697521901,0.236679058124,0.23666059875,0.236642143777,0.236623693205,0.23660524703,0.236586805252,0.236568367869,0.236549934879,0.23653150628,0.23651308207,0.236494662249,0.236476246813,0.236457835762,0.236439429094,0.236421026807,0.236402628899,0.236384235369,0.236365846214,0.236347461434,0.236329081026,0.23631070499,0.236292333322,0.236273966021,0.236255603086,0.236237244516,0.236218890307,0.236200540459,0.23618219497,0.236163853838,0.236145517061,0.236127184639,0.236108856568,0.236090532848,0.236072213476,0.236053898452,0.236035587772,0.236017281437,0.235998979443,0.23598068179,0.235962388475,0.235944099497,0.235925814854,0.235907534545,0.235889258567,0.23587098692,0.235852719601,0.235834456608,0.235816197941,0.235797943597,0.235779693575,0.235761447874,0.23574320649,0.235724969423,0.235706736671,0.235688508232,0.235670284106,0.235652064289,0.23563384878,0.235615637578,0.235597430681,0.235579228087,0.235561029795,0.235542835803,0.235524646109,0.235506460712,0.23548827961,0.235470102801,0.235451930284,0.235433762056,0.235415598117,0.235397438465,0.235379283098,0.235361132013,0.235342985211,0.235324842689,0.235306704445,0.235288570477,0.235270440785,0.235252315366,0.235234194219,0.235216077342,0.235197964733,0.235179856391,0.235161752315,0.235143652501,0.23512555695,0.235107465658,0.235089378626,0.23507129585,0.235053217329,0.235035143062,0.235017073047,0.234999007282,0.234980945766,0.234962888497,0.234944835473,0.234926786693,0.234908742155,0.234890701858,0.234872665799,0.234854633978,0.234836606392,0.23481858304,0.234800563921,0.234782549032,0.234764538372,0.234746531939,0.234728529732,0.23471053175,0.23469253799,0.23467454845,0.23465656313,0.234638582028,0.234620605142,0.23460263247,0.23458466401,0.234566699762,0.234548739723,0.234530783893,0.234512832268,0.234494884848,0.234476941632,0.234459002616,0.234441067801,0.234423137183,0.234405210762,0.234387288536,0.234369370504,0.234351456663,0.234333547012,0.23431564155,0.234297740274,0.234279843184,0.234261950278,0.234244061553,0.234226177009,0.234208296644,0.234190420456,0.234172548444,0.234154680606,0.23413681694,0.234118957445,0.234101102119,0.234083250961,0.234065403969,0.234047561141,0.234029722476,0.234011887973,0.233994057629,0.233976231443,0.233958409413,0.233940591538,0.233922777817,0.233904968247,0.233887162827,0.233869361556,0.233851564432,0.233833771453,0.233815982617,0.233798197924,0.233780417371,0.233762640958,0.233744868681,0.233727100541,0.233709336534,0.23369157666,0.233673820917,0.233656069304,0.233638321819,0.233620578459,0.233602839225,0.233585104114,0.233567373124,0.233549646254,0.233531923502,0.233514204868,0.233496490348,0.233478779943,0.233461073649,0.233443371466,0.233425673392,0.233407979425,0.233390289564,0.233372603808,0.233354922154,0.233337244601,0.233319571148,0.233301901793,0.233284236534,0.23326657537,0.233248918299,0.23323126532,0.233213616431,0.233195971631,0.233178330918,0.23316069429,0.233143061747,0.233125433285,0.233107808905,0.233090188604,0.23307257238,0.233054960233,0.23303735216,0.233019748161,0.233002148232,0.232984552374,0.232966960584,0.232949372861,0.232931789204,0.23291420961,0.232896634078,0.232879062607,0.232861495195,0.23284393184,0.232826372542,0.232808817298,0.232791266107,0.232773718968,0.232756175878,0.232738636837,0.232721101842,0.232703570893,0.232686043987,0.232668521123,0.2326510023,0.232633487517,0.23261597677,0.23259847006,0.232580967384,0.232563468742,0.23254597413,0.232528483549,0.232510996996,0.23249351447,0.232476035969,0.232458561491,0.232441091036,0.232423624602,0.232406162187,0.23238870379,0.232371249409,0.232353799042,0.232336352689,0.232318910347,0.232301472015,0.232284037692,0.232266607376,0.232249181065,0.232231758758,0.232214340454,0.232196926151,0.232179515847,0.232162109541,0.232144707231,0.232127308917,0.232109914595,0.232092524266,0.232075137927,0.232057755577,0.232040377214,0.232023002837,0.232005632444,0.231988266034,0.231970903606,0.231953545157,0.231936190687,0.231918840193,0.231901493675,0.23188415113,0.231866812558,0.231849477956,0.231832147324,0.23181482066,0.231797497962,0.231780179228,0.231762864458,0.23174555365,0.231728246802,0.231710943913,0.231693644981,0.231676350005,0.231659058984,0.231641771915,0.231624488798,0.23160720963,0.231589934411,0.231572663138,0.231555395811,0.231538132428,0.231520872988,0.231503617488,0.231486365928,0.231469118305,0.231451874619,0.231434634869,0.231417399051,0.231400167166,0.231382939211,0.231365715185,0.231348495087,0.231331278915,0.231314066668,0.231296858343,0.231279653941,0.231262453459,0.231245256895,0.231228064248,0.231210875518,0.231193690702,0.231176509798,0.231159332806,0.231142159724,0.23112499055,0.231107825283,0.231090663922,0.231073506464,0.23105635291,0.231039203256,0.231022057502,0.231004915646,0.230987777687,0.230970643623,0.230953513452,0.230936387174,0.230919264787,0.23090214629,0.23088503168,0.230867920956,0.230850814118,0.230833711163,0.230816612091,0.230799516899,0.230782425586,0.230765338151,0.230748254592,0.230731174908,0.230714099097,0.230697027158,0.23067995909,0.230662894891,0.230645834559,0.230628778094,0.230611725493,0.230594676755,0.230577631879,0.230560590864,0.230543553707,0.230526520408,0.230509490965,0.230492465376,0.23047544364,0.230458425756,0.230441411723,0.230424401538,0.2304073952,0.230390392709,0.230373394062,0.230356399258,0.230339408295,0.230322421173,0.230305437889,0.230288458443,0.230271482833,0.230254511057,0.230237543114,0.230220579003,0.230203618721,0.230186662269,0.230169709644,0.230152760844,0.230135815869,0.230118874717,0.230101937387,0.230085003877,0.230068074185,0.230051148311,0.230034226253,0.230017308009,0.230000393578,0.229983482959,0.22996657615,0.229949673149,0.229932773956,0.229915878569,0.229898986987,0.229882099208,0.22986521523,0.229848335052,0.229831458674,0.229814586093,0.229797717307,0.229780852317,0.22976399112,0.229747133714,0.229730280099,0.229713430273,0.229696584234,0.229679741981,0.229662903514,0.229646068829,0.229629237927,0.229612410805,0.229595587462,0.229578767897,0.229561952108,0.229545140094,0.229528331853,0.229511527385,0.229494726687,0.229477929759,0.229461136598,0.229444347204,0.229427561575,0.22941077971,0.229394001607,0.229377227265,0.229360456683,0.229343689859,0.229326926791,0.229310167479,0.22929341192,0.229276660115,0.22925991206,0.229243167755,0.229226427199,0.229209690389,0.229192957325,0.229176228006,0.229159502429,0.229142780593,0.229126062498,0.229109348141,0.229092637522,0.229075930638,0.229059227489,0.229042528074,0.229025832389,0.229009140436,0.228992452211,0.228975767714,0.228959086943,0.228942409897,0.228925736574,0.228909066974,0.228892401094,0.228875738934,0.228859080491,0.228842425765,0.228825774755,0.228809127458,0.228792483874,0.228775844,0.228759207837,0.228742575382,0.228725946634,0.228709321591,0.228692700253,0.228676082618,0.228659468684,0.228642858451,0.228626251916,0.228609649079,0.228593049937,0.228576454491,0.228559862738,0.228543274676,0.228526690306,0.228510109624,0.228493532631,0.228476959324,0.228460389702,0.228443823764,0.228427261509,0.228410702934,0.22839414804,0.228377596824,0.228361049284,0.228344505421,0.228327965232,0.228311428716,0.228294895871,0.228278366697,0.228261841192,0.228245319354,0.228228801182,0.228212286676,0.228195775833,0.228179268652,0.228162765132,0.228146265271,0.228129769069,0.228113276523,0.228096787633,0.228080302397,0.228063820814,0.228047342882,0.228030868601,0.228014397968,0.227997930982,0.227981467643,0.227965007948,0.227948551897,0.227932099487,0.227915650719,0.22789920559,0.227882764098,0.227866326244,0.227849892025,0.22783346144,0.227817034488,0.227800611167,0.227784191476,0.227767775414,0.227751362979,0.227734954171,0.227718548987,0.227702147426,0.227685749488,0.22766935517,0.227652964472,0.227636577392,0.227620193928,0.227603814081,0.227587437847,0.227571065226,0.227554696216,0.227538330817,0.227521969026,0.227505610843,0.227489256267,0.227472905295,0.227456557926,0.22744021416,0.227423873995,0.22740753743,0.227391204462,0.227374875092,0.227358549317,0.227342227137,0.22732590855,0.227309593554,0.227293282149,0.227276974333,0.227260670105,0.227244369463,0.227228072407,0.227211778934,0.227195489044,0.227179202735,0.227162920006,0.227146640856,0.227130365283,0.227114093286,0.227097824864,0.227081560015,0.227065298738,0.227049041033,0.227032786896,0.227016536328,0.227000289327,0.226984045891,0.22696780602,0.226951569712,0.226935336965,0.226919107779,0.226902882152,0.226886660082,0.22687044157,0.226854226612,0.226838015209,0.226821807358,0.226805603058,0.226789402309,0.226773205108,0.226757011455,0.226740821348,0.226724634786,0.226708451767,0.226692272291,0.226676096356,0.22665992396,0.226643755103,0.226627589784,0.226611428,0.22659526975,0.226579115034,0.22656296385,0.226546816197,0.226530672074,0.226514531478,0.226498394409,0.226482260866,0.226466130848,0.226450004352,0.226433881378,0.226417761925,0.226401645991,0.226385533575,0.226369424676,0.226353319292,0.226337217422,0.226321119065,0.22630502422,0.226288932885,0.226272845059,0.22625676074,0.226240679929,0.226224602622,0.22620852882,0.22619245852,0.226176391721,0.226160328423,0.226144268624,0.226128212322,0.226112159517,0.226096110206,0.22608006439,0.226064022066,0.226047983233,0.226031947891,0.226015916037,0.225999887671,0.225983862791,0.225967841396,0.225951823484,0.225935809055,0.225919798108,0.22590379064,0.225887786651,0.22587178614,0.225855789104,0.225839795544,0.225823805457,0.225807818843,0.2257918357,0.225775856027,0.225759879822,0.225743907085,0.225727937814,0.225711972008,0.225696009665,0.225680050785,0.225664095366,0.225648143407,0.225632194907,0.225616249864,0.225600308278,0.225584370146,0.225568435468,0.225552504242,0.225536576468,0.225520652143,0.225504731267,0.225488813839,0.225472899857,0.22545698932,0.225441082226,0.225425178575,0.225409278366,0.225393381596,0.225377488265,0.225361598372,0.225345711915,0.225329828893,0.225313949305,0.225298073149,0.225282200425,0.225266331131,0.225250465266,0.225234602829,0.225218743818,0.225202888232,0.225187036071,0.225171187332,0.225155342014,0.225139500117,0.225123661639,0.225107826578,0.225091994935,0.225076166706,0.225060341892,0.225044520491,0.225028702501,0.225012887922,0.224997076752,0.224981268991,0.224965464636,0.224949663686,0.224933866141,0.224918071999,0.224902281259,0.22488649392,0.22487070998,0.224854929439,0.224839152294,0.224823378546,0.224807608192,0.224791841231,0.224776077662,0.224760317485,0.224744560697,0.224728807298,0.224713057286,0.224697310661,0.22468156742,0.224665827563,0.224650091088,0.224634357995,0.224618628281,0.224602901947,0.22458717899,0.22457145941,0.224555743205,0.224540030373,0.224524320915,0.224508614828,0.224492912112,0.224477212765,0.224461516786,0.224445824174,0.224430134927,0.224414449045,0.224398766526,0.224383087369,0.224367411572,0.224351739135,0.224336070057,0.224320404336,0.224304741971,0.224289082961,0.224273427304,0.224257774999,0.224242126046,0.224226480443,0.224210838189,0.224195199282,0.224179563722,0.224163931507,0.224148302636,0.224132677108,0.224117054921,0.224101436075,0.224085820569,0.2240702084,0.224054599568,0.224038994072,0.224023391911,0.224007793083,0.223992197587,0.223976605422,0.223961016587,0.22394543108,0.223929848901,0.223914270048,0.22389869452,0.223883122316,0.223867553435,0.223851987875,0.223836425636,0.223820866715,0.223805311113,0.223789758827,0.223774209857,0.223758664202,0.223743121859,0.223727582829,0.22371204711,0.2236965147,0.223680985599,0.223665459805,0.223649937318,0.223634418135,0.223618902257,0.223603389681,0.223587880406,0.223572374432,0.223556871757,0.22354137238,0.2235258763,0.223510383516,0.223494894026,0.223479407829,0.223463924925,0.223448445311,0.223432968988,0.223417495953,0.223402026205,0.223386559744,0.223371096568,0.223355636676,0.223340180067,0.223324726739,0.223309276692,0.223293829924,0.223278386435,0.223262946222,0.223247509286,0.223232075624,0.223216645236,0.22320121812,0.223185794275,0.223170373701,0.223154956395,0.223139542357,0.223124131586,0.22310872408,0.223093319839,0.22307791886,0.223062521144,0.223047126688,0.223031735493,0.223016347555,0.223000962876,0.222985581452,0.222970203284,0.222954828369,0.222939456707,0.222924088297,0.222908723138,0.222893361227,0.222878002565,0.22286264715,0.222847294981,0.222831946057,0.222816600376,0.222801257938,0.222785918742,0.222770582785,0.222755250067,0.222739920588,0.222724594345,0.222709271338,0.222693951565,0.222678635026,0.222663321719,0.222648011643,0.222632704797,0.222617401179,0.22260210079,0.222586803627,0.222571509689,0.222556218976,0.222540931485,0.222525647217,0.22251036617,0.222495088342,0.222479813733,0.222464542341,0.222449274166,0.222434009206,0.22241874746,0.222403488927,0.222388233606,0.222372981496,0.222357732595,0.222342486902,0.222327244417,0.222312005138,0.222296769064,0.222281536194,0.222266306527,0.222251080061,0.222235856796,0.222220636731,0.222205419864,0.222190206193,0.22217499572,0.222159788441,0.222144584355,0.222129383463,0.222114185762,0.222098991252,0.222083799931,0.222068611798,0.222053426852,0.222038245093,0.222023066518,0.222007891127,0.221992718918,0.221977549892,0.221962384045,0.221947221378,0.221932061889,0.221916905578,0.221901752442,0.221886602481,0.221871455694,0.221856312079,0.221841171636,0.221826034364,0.22181090026,0.221795769325,0.221780641557,0.221765516956,0.221750395518,0.221735277245,0.221720162135,0.221705050186,0.221689941397,0.221674835768,0.221659733297,0.221644633983,0.221629537826,0.221614444823,0.221599354974,0.221584268278,0.221569184733,0.221554104339,0.221539027095,0.221523952999,0.22150888205,0.221493814247,0.221478749589,0.221463688076,0.221448629705,0.221433574476,0.221418522388,0.221403473439,0.221388427628,0.221373384955,0.221358345419,0.221343309017,0.22132827575,0.221313245616,0.221298218614,0.221283194742,0.221268174,0.221253156387,0.221238141902,0.221223130543,0.221208122309,0.2211931172,0.221178115214,0.22116311635,0.221148120607,0.221133127984,0.22111813848,0.221103152094,0.221088168824,0.221073188671,0.221058211631,0.221043237705,0.221028266892,0.22101329919,0.220998334598,0.220983373115,0.22096841474,0.220953459473,0.220938507311,0.220923558254,0.2209086123,0.22089366945,0.220878729701,0.220863793052,0.220848859503,0.220833929053,0.220819001699,0.220804077442,0.22078915628,0.220774238212,0.220759323237,0.220744411354,0.220729502562,0.22071459686,0.220699694246,0.22068479472,0.220669898281,0.220655004927,0.220640114657,0.220625227471,0.220610343368,0.220595462345,0.220580584403,0.220565709539,0.220550837754,0.220535969046,0.220521103413,0.220506240856,0.220491381372,0.220476524961,0.220461671622,0.220446821353,0.220431974154,0.220417130023,0.22040228896,0.220387450963,0.220372616031,0.220357784163,0.220342955359,0.220328129617,0.220313306936,0.220298487315,0.220283670752,0.220268857248,0.220254046801,0.220239239409,0.220224435072,0.220209633789,0.220194835558,0.220180040379,0.22016524825,0.220150459171,0.22013567314,0.220120890157,0.22010611022,0.220091333328,0.22007655948,0.220061788676,0.220047020913,0.220032256192,0.220017494511,0.220002735868,0.219987980264,0.219973227696,0.219958478164,0.219943731668,0.219928988204,0.219914247774,0.219899510375,0.219884776007,0.219870044668,0.219855316358,0.219840591075,0.219825868818,0.219811149588,0.219796433381,0.219781720198,0.219767010037,0.219752302897,0.219737598777,0.219722897677,0.219708199595,0.21969350453,0.21967881248,0.219664123446,0.219649437426,0.219634754419,0.219620074424,0.219605397439,0.219590723465,0.219576052499,0.219561384541,0.21954671959,0.219532057644,0.219517398703,0.219502742766,0.219488089831,0.219473439898,0.219458792966,0.219444149033,0.219429508099,0.219414870162,0.219400235221,0.219385603276,0.219370974325,0.219356348368,0.219341725403,0.219327105429,0.219312488446,0.219297874452,0.219283263446,0.219268655427,0.219254050395,0.219239448348,0.219224849284,0.219210253204,0.219195660107,0.21918106999,0.219166482853,0.219151898696,0.219137317516,0.219122739313,0.219108164087,0.219093591835,0.219079022558,0.219064456253,0.21904989292,0.219035332558,0.219020775166,0.219006220743,0.218991669288,0.2189771208,0.218962575278,0.21894803272,0.218933493126,0.218918956495,0.218904422826,0.218889892118,0.218875364369,0.218860839579,0.218846317747,0.218831798872,0.218817282952,0.218802769988,0.218788259976,0.218773752918,0.218759248811,0.218744747655,0.218730249449,0.218715754192,0.218701261882,0.218686772519,0.218672286101,0.218657802628,0.218643322099,0.218628844513,0.218614369868,0.218599898164,0.2185854294,0.218570963574,0.218556500686,0.218542040735,0.218527583719,0.218513129638,0.21849867849,0.218484230276,0.218469784993,0.218455342641,0.218440903218,0.218426466724,0.218412033158,0.218397602518,0.218383174805,0.218368750015,0.21835432815,0.218339909208,0.218325493187,0.218311080087,0.218296669906,0.218282262645,0.218267858301,0.218253456874,0.218239058362,0.218224662766,0.218210270083,0.218195880313,0.218181493455,0.218167109508,0.218152728471,0.218138350343,0.218123975122,0.218109602808,0.218095233401,0.218080866898,0.218066503299,0.218052142603,0.21803778481,0.218023429917,0.218009077924,0.21799472883,0.217980382634,0.217966039336,0.217951698933,0.217937361426,0.217923026812,0.217908695092,0.217894366264,0.217880040327,0.217865717281,0.217851397123,0.217837079854,0.217822765472,0.217808453977,0.217794145367,0.217779839641,0.217765536799,0.217751236839,0.21773693976,0.217722645562,0.217708354243,0.217694065803,0.217679780241,0.217665497555,0.217651217744,0.217636940808,0.217622666746,0.217608395557,0.217594127239,0.217579861792,0.217565599215,0.217551339507,0.217537082666,0.217522828693,0.217508577585,0.217494329342,0.217480083963,0.217465841448,0.217451601794,0.217437365001,0.217423131069,0.217408899996,0.217394671781,0.217380446423,0.217366223921,0.217352004275,0.217337787483,0.217323573544,0.217309362458,0.217295154223,0.217280948839,0.217266746304,0.217252546618,0.21723834978,0.217224155788,0.217209964642,0.217195776341,0.217181590883,0.217167408269,0.217153228496,0.217139051564,0.217124877472,0.21711070622,0.217096537805,0.217082372227,0.217068209486,0.21705404958,0.217039892508,0.217025738269,0.217011586863,0.216997438288,0.216983292543,0.216969149628,0.216955009542,0.216940872283,0.216926737851,0.216912606244,0.216898477463,0.216884351505,0.21687022837,0.216856108057,0.216841990565,0.216827875893,0.21681376404,0.216799655005,0.216785548787,0.216771445386,0.2167573448,0.216743247028,0.21672915207,0.216715059924,0.216700970589,0.216686884066,0.216672800352,0.216658719446,0.216644641349,0.216630566058,0.216616493573,0.216602423893,0.216588357017,0.216574292945,0.216560231674,0.216546173205,0.216532117535,0.216518064666,0.216504014594,0.21648996732,0.216475922843,0.216461881161,0.216447842274,0.21643380618,0.21641977288,0.216405742371,0.216391714653,0.216377689725,0.216363667586,0.216349648235,0.216335631672,0.216321617895,0.216307606903,0.216293598695,0.216279593271,0.21626559063,0.21625159077,0.216237593691,0.216223599391,0.216209607871,0.216195619128,0.216181633163,0.216167649973,0.216153669559,0.216139691919,0.216125717052,0.216111744957,0.216097775634,0.216083809082,0.216069845299,0.216055884284,0.216041926038,0.216027970558,0.216014017844,0.216000067896,0.215986120711,0.215972176289,0.21595823463,0.215944295732,0.215930359595,0.215916426216,0.215902495597,0.215888567735,0.21587464263,0.21586072028,0.215846800686,0.215832883845,0.215818969758,0.215805058423,0.215791149838,0.215777244005,0.21576334092,0.215749440585,0.215735542996,0.215721648155,0.215707756059,0.215693866708,0.215679980101,0.215666096237,0.215652215115,0.215638336734,0.215624461094,0.215610588193,0.21559671803,0.215582850605,0.215568985917,0.215555123964,0.215541264747,0.215527408263,0.215513554512,0.215499703494,0.215485855206,0.215472009649,0.215458166822,0.215444326722,0.215430489351,0.215416654706,0.215402822787,0.215388993593,0.215375167123,0.215361343376,0.215347522351,0.215333704048,0.215319888465,0.215306075601,0.215292265456,0.215278458029,0.215264653318,0.215250851324,0.215237052044,0.215223255479,0.215209461626,0.215195670486,0.215181882058,0.21516809634,0.215154313331,0.215140533032,0.21512675544,0.215112980555,0.215099208376,0.215085438902,0.215071672133,0.215057908067,0.215044146703,0.215030388041,0.21501663208,0.215002878819,0.214989128256,0.214975380391,0.214961635224,0.214947892753,0.214934152977,0.214920415896,0.214906681508,0.214892949813,0.214879220809,0.214865494497,0.214851770875,0.214838049941,0.214824331697,0.214810616139,0.214796903268,0.214783193082,0.214769485582,0.214755780765,0.214742078631,0.214728379179,0.214714682408,0.214700988318,0.214687296907,0.214673608174,0.21465992212,0.214646238742,0.214632558039,0.214618880012,0.214605204659,0.21459153198,0.214577861973,0.214564194637,0.214550529972,0.214536867976,0.21452320865,0.214509551991,0.214495898,0.214482246675,0.214468598015,0.21445495202,0.214441308688,0.214427668019,0.214414030012,0.214400394666,0.21438676198,0.214373131953,0.214359504585,0.214345879874,0.21433225782,0.214318638422,0.214305021678,0.214291407589,0.214277796153,0.214264187369,0.214250581236,0.214236977754,0.214223376922,0.214209778738,0.214196183202,0.214182590314,0.214169000072,0.214155412475,0.214141827522,0.214128245213,0.214114665547,0.214101088523,0.21408751414,0.214073942397,0.214060373293,0.214046806828,0.214033243,0.214019681809,0.214006123254,0.213992567334,0.213979014047,0.213965463395,0.213951915374,0.213938369985,0.213924827227,0.213911287099,0.213897749599,0.213884214728,0.213870682484,0.213857152866,0.213843625874,0.213830101506,0.213816579763,0.213803060642,0.213789544143,0.213776030266,0.213762519009,0.213749010371,0.213735504353,0.213722000952,0.213708500168,0.213695002,0.213681506447,0.213668013509,0.213654523184,0.213641035472,0.213627550372,0.213614067883,0.213600588004,0.213587110735,0.213573636074,0.21356016402,0.213546694574,0.213533227733,0.213519763497,0.213506301865,0.213492842837,0.213479386412,0.213465932588,0.213452481365,0.213439032741,0.213425586717,0.213412143292,0.213398702463,0.213385264232,0.213371828596,0.213358395555,0.213344965108,0.213331537254,0.213318111993,0.213304689323,0.213291269244,0.213277851755,0.213264436855,0.213251024543,0.213237614818,0.21322420768,0.213210803127,0.213197401159,0.213184001776,0.213170604975,0.213157210757,0.21314381912,0.213130430064,0.213117043588,0.21310365969,0.213090278371,0.213076899629,0.213063523463,0.213050149873,0.213036778858,0.213023410417,0.213010044549,0.212996681253,0.212983320529,0.212969962375,0.212956606791,0.212943253776,0.212929903329,0.212916555449,0.212903210136,0.212889867388,0.212876527205,0.212863189587,0.212849854531,0.212836522037,0.212823192106,0.212809864734,0.212796539923,0.21278321767,0.212769897976,0.212756580839,0.212743266258,0.212729954234,0.212716644764,0.212703337848,0.212690033485,0.212676731675,0.212663432416,0.212650135708,0.21263684155,0.212623549941,0.21261026088,0.212596974367,0.2125836904,0.212570408979,0.212557130103,0.212543853771,0.212530579983,0.212517308737,0.212504040033,0.212490773869,0.212477510246,0.212464249162,0.212450990616,0.212437734608,0.212424481137,0.212411230202,0.212397981802,0.212384735936,0.212371492603,0.212358251804,0.212345013536,0.212331777799,0.212318544593,0.212305313916,0.212292085767,0.212278860147,0.212265637053,0.212252416486,0.212239198444,0.212225982926,0.212212769933,0.212199559462,0.212186351513,0.212173146086,0.212159943179,0.212146742792,0.212133544924,0.212120349573,0.212107156741,0.212093966424,0.212080778623,0.212067593337,0.212054410565,0.212041230306,0.21202805256,0.212014877325,0.212001704601,0.211988534387,0.211975366683,0.211962201486,0.211949038798,0.211935878616,0.211922720939,0.211909565769,0.211896413102,0.211883262939,0.211870115279,0.211856970121,0.211843827464,0.211830687307,0.21181754965,0.211804414491,0.211791281831,0.211778151667,0.211765024,0.211751898829,0.211738776152,0.21172565597,0.21171253828,0.211699423083,0.211686310377,0.211673200162,0.211660092437,0.211646987202,0.211633884454,0.211620784194,0.211607686421,0.211594591134,0.211581498332,0.211568408015,0.211555320181,0.21154223483,0.211529151961,0.211516071573,0.211502993665,0.211489918238,0.211476845289,0.211463774818,0.211450706824,0.211437641307,0.211424578265,0.211411517699,0.211398459606,0.211385403987,0.21137235084,0.211359300165,0.211346251961,0.211333206227,0.211320162963,0.211307122167,0.211294083839,0.211281047977,0.211268014582,0.211254983653,0.211241955188,0.211228929187,0.211215905649,0.211202884573,0.211189865959,0.211176849806,0.211163836112,0.211150824878,0.211137816102,0.211124809783,0.211111805922,0.211098804516,0.211085805565,0.21107280907,0.211059815027,0.211046823438,0.211033834301,0.211020847615,0.211007863379,0.210994881594,0.210981902257,0.210968925369,0.210955950928,0.210942978934,0.210930009385,0.210917042282,0.210904077623,0.210891115407,0.210878155635,0.210865198304,0.210852243415,0.210839290966,0.210826340956,0.210813393386,0.210800448254,0.210787505559,0.2107745653,0.210761627478,0.21074869209,0.210735759137,0.210722828617,0.21070990053,0.210696974875,0.210684051651,0.210671130857,0.210658212493,0.210645296558,0.210632383051,0.210619471972,0.210606563319,0.210593657091,0.210580753289,0.210567851911,0.210554952956,0.210542056424,0.210529162314,0.210516270626,0.210503381357,0.210490494509,0.210477610079,0.210464728067,0.210451848473,0.210438971295,0.210426096533,0.210413224186,0.210400354253,0.210387486734,0.210374621628,0.210361758933,0.21034889865,0.210336040778,0.210323185315,0.210310332261,0.210297481615,0.210284633376,0.210271787545,0.210258944119,0.210246103098,0.210233264482,0.210220428269,0.21020759446,0.210194763052,0.210181934046,0.21016910744,0.210156283234,0.210143461427,0.210130642019,0.210117825008,0.210105010394,0.210092198176,0.210079388353,0.210066580925,0.210053775891,0.210040973249,0.210028173,0.210015375143,0.210002579676,0.209989786599,0.209976995912,0.209964207613,0.209951421702,0.209938638177,0.20992585704,0.209913078287,0.209900301919,0.209887527936,0.209874756335,0.209861987117,0.209849220281,0.209836455826,0.209823693751,0.209810934056,0.209798176739,0.209785421801,0.20977266924,0.209759919055,0.209747171246,0.209734425812,0.209721682752,0.209708942066,0.209696203752,0.209683467811,0.209670734241,0.209658003041,0.209645274212,0.209632547751,0.209619823659,0.209607101934,0.209594382576,0.209581665584,0.209568950958,0.209556238696,0.209543528798,0.209530821263,0.20951811609,0.20950541328,0.20949271283,0.20948001474,0.209467319009,0.209454625638,0.209441934624,0.209429245967,0.209416559667,0.209403875723,0.209391194133,0.209378514898,0.209365838016,0.209353163487,0.209340491311,0.209327821485,0.20931515401,0.209302488885,0.209289826109,0.209277165681,0.209264507601,0.209251851868,0.209239198481,0.209226547439,0.209213898742,0.209201252389,0.209188608379,0.209175966712,0.209163327387,0.209150690402,0.209138055758,0.209125423453,0.209112793487,0.20910016586,0.209087540569,0.209074917616,0.209062296998,0.209049678715,0.209037062767,0.209024449153,0.209011837871,0.208999228922,0.208986622304,0.208974018017,0.208961416061,0.208948816433,0.208936219134,0.208923624163,0.20891103152,0.208898441202,0.208885853211,0.208873267544,0.208860684202,0.208848103183,0.208835524487,0.208822948114,0.208810374061,0.208797802329,0.208785232918,0.208772665825,0.208760101051,0.208747538594,0.208734978455,0.208722420632,0.208709865125,0.208697311932,0.208684761054,0.208672212489,0.208659666237,0.208647122297,0.208634580668,0.208622041349,0.208609504341,0.208596969641,0.208584437251,0.208571907167,0.208559379391,0.208546853921,0.208534330756,0.208521809897,0.208509291341,0.208496775089,0.20848426114,0.208471749492,0.208459240146,0.2084467331,0.208434228354,0.208421725907,0.208409225759,0.208396727908,0.208384232354,0.208371739097,0.208359248135,0.208346759468,0.208334273095,0.208321789015,0.208309307228,0.208296827733,0.208284350529,0.208271875616,0.208259402993,0.208246932659,0.208234464613,0.208221998855,0.208209535384,0.208197074199,0.2081846153,0.208172158686,0.208159704356,0.20814725231,0.208134802546,0.208122355064,0.208109909864,0.208097466944,0.208085026304,0.208072587944,0.208060151862,0.208047718058,0.208035286531,0.20802285728,0.208010430305,0.207998005606,0.20798558318,0.207973163028,0.207960745149,0.207948329542,0.207935916207,0.207923505142,0.207911096348,0.207898689823,0.207886285567,0.207873883578,0.207861483857,0.207849086403,0.207836691214,0.207824298291,0.207811907632,0.207799519237,0.207787133105,0.207774749236,0.207762367628,0.207749988281,0.207737611194,0.207725236367,0.207712863799,0.20770049349,0.207688125437,0.207675759642,0.207663396103,0.207651034819,0.20763867579,0.207626319015,0.207613964493,0.207601612224,0.207589262207,0.207576914441,0.207564568926,0.20755222566,0.207539884644,0.207527545876,0.207515209357,0.207502875084,0.207490543058,0.207478213277,0.207465885741,0.20745356045,0.207441237403,0.207428916598,0.207416598036,0.207404281715,0.207391967636,0.207379655796,0.207367346196,0.207355038835,0.207342733712,0.207330430826,0.207318130178,0.207305831765,0.207293535588,0.207281241645,0.207268949937,0.207256660461,0.207244373219,0.207232088208,0.207219805429,0.20720752488,0.207195246562,0.207182970472,0.207170696611,0.207158424978,0.207146155573,0.207133888393,0.20712162344,0.207109360711,0.207097100207,0.207084841927,0.20707258587,0.207060332036,0.207048080423,0.207035831031,0.207023583859,0.207011338908,0.206999096175,0.20698685566,0.206974617364,0.206962381284,0.20695014742,0.206937915773,0.20692568634,0.206913459121,0.206901234116,0.206889011324,0.206876790744,0.206864572375,0.206852356218,0.206840142271,0.206827930533,0.206815721004,0.206803513683,0.20679130857,0.206779105664,0.206766904964,0.206754706469,0.206742510179,0.206730316093,0.206718124211,0.206705934531,0.206693747054,0.206681561778,0.206669378703,0.206657197827,0.206645019152,0.206632842675,0.206620668396,0.206608496315,0.20659632643,0.206584158741,0.206571993248,0.20655982995,0.206547668845,0.206535509934,0.206523353216,0.20651119869,0.206499046355,0.206486896211,0.206474748257,0.206462602493,0.206450458917,0.206438317529,0.206426178329,0.206414041315,0.206401906487,0.206389773845,0.206377643388,0.206365515114,0.206353389024,0.206341265117,0.206329143392,0.206317023848,0.206304906485,0.206292791302,0.206280678298,0.206268567474,0.206256458827,0.206244352358,0.206232248065,0.206220145949,0.206208046008,0.206195948242,0.20618385265,0.206171759231,0.206159667986,0.206147578912,0.20613549201,0.206123407279,0.206111324718,0.206099244326,0.206087166103,0.206075090049,0.206063016162,0.206050944442,0.206038874888,0.2060268075,0.206014742276,0.206002679217,0.205990618322,0.205978559589,0.205966503019,0.20595444861,0.205942396363,0.205930346275,0.205918298347,0.205906252579,0.205894208968,0.205882167516,0.20587012822,0.205858091081,0.205846056098,0.205834023269,0.205821992595,0.205809964075,0.205797937708,0.205785913494,0.205773891431,0.20576187152,0.205749853759,0.205737838148,0.205725824686,0.205713813373,0.205701804207,0.205689797189,0.205677792318,0.205665789592,0.205653789012,0.205641790577,0.205629794286,0.205617800138,0.205605808132,0.205593818269,0.205581830548,0.205569844967,0.205557861526,0.205545880225,0.205533901063,0.205521924038,0.205509949152,0.205497976402,0.205486005789,0.205474037311,0.205462070969,0.20545010676,0.205438144686,0.205426184744,0.205414226935,0.205402271258,0.205390317712,0.205378366296,0.20536641701,0.205354469854,0.205342524826,0.205330581925,0.205318641153,0.205306702506,0.205294765986,0.205282831591,0.205270899321,0.205258969175,0.205247041153,0.205235115253,0.205223191476,0.205211269819,0.205199350284,0.205187432869,0.205175517574,0.205163604398,0.20515169334,0.2051397844,0.205127877577,0.20511597287,0.205104070279,0.205092169803,0.205080271442,0.205068375195,0.205056481061,0.205044589039,0.20503269913,0.205020811332,0.205008925644,0.204997042067,0.204985160599,0.20497328124,0.204961403989,0.204949528846,0.20493765581,0.20492578488,0.204913916055,0.204902049336,0.204890184721,0.204878322209,0.204866461801,0.204854603496,0.204842747292,0.20483089319,0.204819041188,0.204807191286,0.204795343483,0.204783497779,0.204771654174,0.204759812665,0.204747973254,0.204736135939,0.204724300719,0.204712467595,0.204700636564,0.204688807628,0.204676980784,0.204665156033,0.204653333374,0.204641512806,0.204629694328,0.204617877941,0.204606063642,0.204594251433,0.204582441311,0.204570633277,0.20455882733,0.204547023469,0.204535221694,0.204523422003,0.204511624397,0.204499828875,0.204488035435,0.204476244078,0.204464454803,0.204452667609,0.204440882496,0.204429099463,0.204417318509,0.204405539634,0.204393762837,0.204381988117,0.204370215475,0.204358444908,0.204346676418,0.204334910002,0.20432314566,0.204311383393,0.204299623198,0.204287865077,0.204276109027,0.204264355048,0.20425260314,0.204240853302,0.204229105534,0.204217359834,0.204205616203,0.204193874639,0.204182135142,0.204170397712,0.204158662347,0.204146929048,0.204135197813,0.204123468642,0.204111741534,0.204100016489,0.204088293506,0.204076572585,0.204064853724,0.204053136924,0.204041422183,0.204029709501,0.204017998878,0.204006290312,0.203994583804,0.203982879352,0.203971176956,0.203959476616,0.20394777833,0.203936082098,0.20392438792,0.203912695795,0.203901005722,0.203889317701,0.20387763173,0.203865947811,0.203854265941,0.20384258612,0.203830908348,0.203819232624,0.203807558947,0.203795887318,0.203784217734,0.203772550196,0.203760884703,0.203749221255,0.20373755985,0.203725900488,0.203714243169,0.203702587892,0.203690934656,0.203679283461,0.203667634307,0.203655987191,0.203644342115,0.203632699077,0.203621058077,0.203609419114,0.203597782187,0.203586147297,0.203574514441,0.203562883621,0.203551254834,0.203539628081,0.203528003361,0.203516380673,0.203504760017,0.203493141392,0.203481524798,0.203469910234,0.203458297698,0.203446687192,0.203435078714,0.203423472263,0.203411867839,0.203400265441,0.203388665069,0.203377066722,0.2033654704,0.203353876102,0.203342283827,0.203330693574,0.203319105344,0.203307519136,0.203295934948,0.20328435278,0.203272772633,0.203261194504,0.203249618394,0.203238044302,0.203226472228,0.20321490217,0.203203334128,0.203191768102,0.203180204091,0.203168642094,0.203157082111,0.203145524141,0.203133968184,0.203122414239,0.203110862305,0.203099312382,0.20308776447,0.203076218567,0.203064674673,0.203053132787,0.20304159291,0.203030055039,0.203018519176,0.203006985318,0.202995453466,0.202983923619,0.202972395776,0.202960869937,0.202949346101,0.202937824268,0.202926304436,0.202914786606,0.202903270777,0.202891756948,0.202880245118,0.202868735288,0.202857227456,0.202845721622,0.202834217785,0.202822715945,0.202811216101,0.202799718252,0.202788222399,0.202776728539,0.202765236674,0.202753746801,0.202742258921,0.202730773034,0.202719289137,0.202707807232,0.202696327316,0.202684849391,0.202673373454,0.202661899506,0.202650427546,0.202638957573,0.202627489587,0.202616023587,0.202604559572,0.202593097543,0.202581637498,0.202570179437,0.202558723359,0.202547269263,0.20253581715,0.202524367018,0.202512918868,0.202501472697,0.202490028507,0.202478586295,0.202467146063,0.202455707808,0.20244427153,0.20243283723,0.202421404906,0.202409974557,0.202398546184,0.202387119785,0.20237569536,0.202364272909,0.202352852431,0.202341433924,0.20233001739,0.202318602826,0.202307190233,0.20229577961,0.202284370957,0.202272964272,0.202261559555,0.202250156806,0.202238756024,0.202227357209,0.202215960359,0.202204565475,0.202193172556,0.2021817816,0.202170392609,0.20215900558,0.202147620514,0.20213623741,0.202124856268,0.202113477086,0.202102099864,0.202090724602,0.202079351299,0.202067979954,0.202056610567,0.202045243138,0.202033877665,0.202022514149,0.202011152588,0.201999792982,0.201988435331,0.201977079634,0.20196572589,0.201954374099,0.20194302426,0.201931676372,0.201920330436,0.20190898645,0.201897644415,0.201886304328,0.201874966191,0.201863630001,0.20185229576,0.201840963465,0.201829633117,0.201818304715,0.201806978258,0.201795653746,0.201784331179,0.201773010555,0.201761691874,0.201750375136,0.20173906034,0.201727747486,0.201716436572,0.201705127599,0.201693820565,0.20168251547,0.201671212315,0.201659911097,0.201648611817,0.201637314474,0.201626019067,0.201614725596,0.20160343406,0.201592144459,0.201580856792,0.201569571059,0.201558287258,0.20154700539,0.201535725454,0.20152444745,0.201513171376,0.201501897232,0.201490625018,0.201479354733,0.201468086377,0.201456819949,0.201445555448,0.201434292874,0.201423032226,0.201411773504,0.201400516707,0.201389261835,0.201378008887,0.201366757863,0.201355508761,0.201344261582,0.201333016324,0.201321772988,0.201310531573,0.201299292078,0.201288054503,0.201276818846,0.201265585108,0.201254353288,0.201243123386,0.2012318954,0.201220669331,0.201209445177,0.201198222939,0.201187002615,0.201175784205,0.201164567709,0.201153353126,0.201142140455,0.201130929696,0.201119720849,0.201108513912,0.201097308886,0.201086105769,0.201074904561,0.201063705262,0.201052507871,0.201041312387,0.20103011881,0.20101892714,0.201007737375,0.200996549516,0.200985363562,0.200974179511,0.200962997364,0.200951817121,0.200940638779,0.20092946234,0.200918287803,0.200907115166,0.200895944429,0.200884775592,0.200873608655,0.200862443616,0.200851280475,0.200840119232,0.200828959886,0.200817802436,0.200806646883,0.200795493225,0.200784341461,0.200773191592,0.200762043617,0.200750897535,0.200739753346,0.200728611049,0.200717470643,0.200706332129,0.200695195505,0.200684060771,0.200672927927,0.200661796971,0.200650667904,0.200639540724,0.200628415432,0.200617292026,0.200606170507,0.200595050873,0.200583933125,0.200572817261,0.20056170328,0.200550591184,0.20053948097,0.200528372639,0.200517266189,0.200506161621,0.200495058934,0.200483958127,0.200472859199,0.200461762151,0.200450666981,0.20043957369,0.200428482276,0.200417392739,0.200406305078,0.200395219293,0.200384135384,0.200373053349,0.200361973189,0.200350894903,0.20033981849,0.200328743949,0.200317671281,0.200306600484,0.200295531559,0.200284464504,0.200273399319,0.200262336003,0.200251274557,0.200240214979,0.200229157268,0.200218101425,0.200207047449,0.200195995339,0.200184945095,0.200173896716,0.200162850202,0.200151805552,0.200140762766,0.200129721842,0.200118682782,0.200107645583,0.200096610245,0.200085576769,0.200074545153,0.200063515397,0.2000524875,0.200041461462,0.200030437283,0.200019414961,0.200008394497,0.199997375889,0.199986359137,0.199975344241,0.1999643312,0.199953320014,0.199942310681,0.199931303203,0.199920297577,0.199909293804,0.199898291882,0.199887291812,0.199876293593,0.199865297225,0.199854302706,0.199843310036,0.199832319215,0.199821330243,0.199810343118,0.199799357841,0.19978837441,0.199777392825,0.199766413086,0.199755435192,0.199744459142,0.199733484937,0.199722512575,0.199711542056,0.19970057338,0.199689606545,0.199678641552,0.1996676784,0.199656717089,0.199645757617,0.199634799984,0.19962384419,0.199612890235,0.199601938117,0.199590987837,0.199580039393,0.199569092785,0.199558148014,0.199547205077,0.199536263975,0.199525324707,0.199514387272,0.199503451671,0.199492517902,0.199481585966,0.199470655861,0.199459727586,0.199448801143,0.199437876529,0.199426953744,0.199416032789,0.199405113662,0.199394196363,0.199383280891,0.199372367246,0.199361455427,0.199350545435,0.199339637267,0.199328730924,0.199317826406,0.199306923711,0.199296022839,0.199285123791,0.199274226564,0.199263331159,0.199252437575,0.199241545812,0.199230655869,0.199219767745,0.199208881441,0.199197996955,0.199187114287,0.199176233437,0.199165354404,0.199154477187,0.199143601787,0.199132728202,0.199121856431,0.199110986476,0.199100118334,0.199089252006,0.19907838749,0.199067524787,0.199056663896,0.199045804817,0.199034947548,0.19902409209,0.199013238441,0.199002386602,0.198991536571,0.198980688349,0.198969841935,0.198958997328,0.198948154528,0.198937313534,0.198926474346,0.198915636963,0.198904801385,0.198893967612,0.198883135641,0.198872305475,0.19886147711,0.198850650549,0.198839825788,0.198829002829,0.198818181671,0.198807362313,0.198796544754,0.198785728995,0.198774915034,0.198764102872,0.198753292507,0.198742483939,0.198731677168,0.198720872193,0.198710069014,0.19869926763,0.19868846804,0.198677670245,0.198666874243,0.198656080034,0.198645287618,0.198634496994,0.198623708161,0.19861292112,0.198602135869,0.198591352408,0.198580570737,0.198569790854,0.198559012761,0.198548236455,0.198537461937,0.198526689206,0.198515918262,0.198505149103,0.198494381731,0.198483616143,0.198472852339,0.19846209032,0.198451330085,0.198440571632,0.198429814962,0.198419060074,0.198408306967,0.198397555642,0.198386806097,0.198376058332,0.198365312347,0.198354568141,0.198343825713,0.198333085063,0.198322346191,0.198311609096,0.198300873777,0.198290140235,0.198279408468,0.198268678476,0.198257950259,0.198247223815,0.198236499146,0.198225776249,0.198215055125,0.198204335773,0.198193618193,0.198182902383,0.198172188345,0.198161476076,0.198150765577,0.198140056847,0.198129349885,0.198118644692,0.198107941267,0.198097239608,0.198086539716,0.19807584159,0.19806514523,0.198054450635,0.198043757805,0.198033066739,0.198022377436,0.198011689897,0.19800100412,0.197990320106,0.197979637853,0.197968957361,0.197958278631,0.19794760166,0.197936926449,0.197926252997,0.197915581304,0.19790491137,0.197894243193,0.197883576773,0.19787291211,0.197862249203,0.197851588053,0.197840928657,0.197830271017,0.19781961513,0.197808960998,0.197798308618,0.197787657992,0.197777009118,0.197766361996,0.197755716626,0.197745073006,0.197734431137,0.197723791017,0.197713152647,0.197702516026,0.197691881154,0.197681248029,0.197670616652,0.197659987022,0.197649359139,0.197638733001,0.197628108609,0.197617485963,0.19760686506,0.197596245902,0.197585628488,0.197575012816,0.197564398887,0.197553786701,0.197543176256,0.197532567552,0.197521960589,0.197511355366,0.197500751883,0.197490150139,0.197479550134,0.197468951867,0.197458355338,0.197447760547,0.197437167492,0.197426576173,0.197415986591,0.197405398743,0.197394812631,0.197384228253,0.197373645609,0.197363064699,0.197352485521,0.197341908076,0.197331332363,0.197320758381,0.197310186131,0.197299615611,0.197289046821,0.197278479761,0.19726791443,0.197257350828,0.197246788954,0.197236228807,0.197225670388,0.197215113696,0.197204558729,0.197194005489,0.197183453974,0.197172904183,0.197162356118,0.197151809775,0.197141265157,0.197130722261,0.197120181088,0.197109641636,0.197099103906,0.197088567898,0.197078033609,0.197067501041,0.197056970192,0.197046441063,0.197035913652,0.197025387959,0.197014863984,0.197004341726,0.196993821185,0.19698330236,0.196972785251,0.196962269857,0.196951756178,0.196941244213,0.196930733963,0.196920225425,0.196909718601,0.196899213489,0.196888710089,0.196878208401,0.196867708423,0.196857210156,0.1968467136,0.196836218753,0.196825725615,0.196815234186,0.196804744465,0.196794256451,0.196783770145,0.196773285546,0.196762802653,0.196752321466,0.196741841985,0.196731364208,0.196720888136,0.196710413768,0.196699941103,0.196689470141,0.196679000882,0.196668533325,0.19665806747,0.196647603316,0.196637140862,0.196626680109,0.196616221055,0.196605763701,0.196595308046,0.196584854089,0.196574401829,0.196563951268,0.196553502403,0.196543055235,0.196532609762,0.196522165986,0.196511723904,0.196501283517,0.196490844824,0.196480407825,0.196469972519,0.196459538906,0.196449106985,0.196438676756,0.196428248218,0.196417821371,0.196407396215,0.196396972748,0.196386550972,0.196376130884,0.196365712484,0.196355295773,0.19634488075,0.196334467413,0.196324055764,0.1963136458,0.196303237523,0.19629283093,0.196282426023,0.1962720228,0.196261621261,0.196251221406,0.196240823233,0.196230426743,0.196220031935,0.196209638809,0.196199247364,0.1961888576,0.196178469516,0.196168083111,0.196157698386,0.19614731534,0.196136933973,0.196126554283,0.196116176271,0.196105799936,0.196095425277,0.196085052295,0.196074680988,0.196064311356,0.196053943399,0.196043577117,0.196033212508,0.196022849573,0.19601248831,0.19600212872,0.195991770802,0.195981414556,0.19597105998,0.195960707075,0.195950355841,0.195940006276,0.19592965838,0.195919312153,0.195908967595,0.195898624704,0.195888283481,0.195877943924,0.195867606035,0.195857269811,0.195846935253,0.19583660236,0.195826271132,0.195815941568,0.195805613668,0.195795287431,0.195784962857,0.195774639945,0.195764318696,0.195753999107,0.19574368118,0.195733364914,0.195723050308,0.195712737361,0.195702426074,0.195692116446,0.195681808475,0.195671502163,0.195661197508,0.195650894511,0.195640593169,0.195630293484,0.195619995455,0.19560969908,0.195599404361,0.195589111295,0.195578819884,0.195568530126,0.195558242021,0.195547955568,0.195537670767,0.195527387618,0.19551710612,0.195506826273,0.195496548076,0.195486271529,0.195475996631,0.195465723382,0.195455451781,0.195445181829,0.195434913524,0.195424646866,0.195414381854,0.195404118489,0.19539385677,0.195383596696,0.195373338267,0.195363081482,0.195352826341,0.195342572844,0.19533232099,0.195322070779,0.195311822209,0.195301575282,0.195291329995,0.19528108635,0.195270844345,0.195260603979,0.195250365254,0.195240128167,0.195229892719,0.195219658909,0.195209426737,0.195199196203,0.195188967305,0.195178740043,0.195168514418,0.195158290428,0.195148068073,0.195137847352,0.195127628266,0.195117410814,0.195107194994,0.195096980808,0.195086768254,0.195076557332,0.195066348042,0.195056140382,0.195045934354,0.195035729955,0.195025527187,0.195015326047,0.195005126537,0.194994928654,0.1949847324,0.194974537774,0.194964344774,0.194954153402,0.194943963655,0.194933775534,0.194923589039,0.194913404169,0.194903220923,0.194893039301,0.194882859302,0.194872680927,0.194862504175,0.194852329044,0.194842155536,0.194831983649,0.194821813383,0.194811644738,0.194801477712,0.194791312307,0.19478114852,0.194770986352,0.194760825803,0.194750666872,0.194740509558,0.194730353861,0.19472019978,0.194710047316,0.194699896468,0.194689747234,0.194679599616,0.194669453612,0.194659309222,0.194649166446,0.194639025282,0.194628885732,0.194618747794,0.194608611467,0.194598476752,0.194588343648,0.194578212154,0.19456808227,0.194557953996,0.194547827331,0.194537702275,0.194527578827,0.194517456988,0.194507336755,0.19449721813,0.194487101111,0.194476985699,0.194466871892,0.19445675969,0.194446649094,0.194436540102,0.194426432714,0.194416326929,0.194406222748,0.194396120169,0.194386019192,0.194375919818,0.194365822045,0.194355725873,0.194345631301,0.19433553833,0.194325446958,0.194315357186,0.194305269012,0.194295182437,0.19428509746,0.19427501408,0.194264932297,0.194254852112,0.194244773522,0.194234696529,0.19422462113,0.194214547327,0.194204475118,0.194194404504,0.194184335483,0.194174268055,0.194164202221,0.194154137978,0.194144075328,0.194134014269,0.194123954802,0.194113896925,0.194103840638,0.194093785941,0.194083732834,0.194073681316,0.194063631386,0.194053583044,0.194043536291,0.194033491124,0.194023447544,0.194013405551,0.194003365144,0.193993326323,0.193983289086,0.193973253435,0.193963219368,0.193953186884,0.193943155984,0.193933126668,0.193923098933,0.193913072782,0.193903048211,0.193893025223,0.193883003815,0.193872983988,0.19386296574,0.193852949073,0.193842933985,0.193832920475,0.193822908544,0.193812898191,0.193802889416,0.193792882218,0.193782876596,0.193772872551,0.193762870082,0.193752869188,0.193742869869,0.193732872125,0.193722875955,0.193712881359,0.193702888336,0.193692896886,0.193682907009,0.193672918703,0.19366293197,0.193652946808,0.193642963216,0.193632981195,0.193623000744,0.193613021863,0.19360304455,0.193593068807,0.193583094632,0.193573122025,0.193563150985,0.193553181512,0.193543213606,0.193533247267,0.193523282493,0.193513319284,0.193503357641,0.193493397562,0.193483439047,0.193473482096,0.193463526708,0.193453572884,0.193443620621,0.193433669921,0.193423720783,0.193413773206,0.193403827189,0.193393882733,0.193383939837,0.193373998501,0.193364058724,0.193354120505,0.193344183845,0.193334248743,0.193324315199,0.193314383211,0.19330445278,0.193294523906,0.193284596587,0.193274670824,0.193264746616,0.193254823962,0.193244902863,0.193234983317,0.193225065325,0.193215148886,0.193205233999,0.193195320665,0.193185408882,0.193175498651,0.19316558997,0.19315568284,0.193145777261,0.19313587323,0.193125970749,0.193116069817,0.193106170434,0.193096272598,0.19308637631,0.19307648157,0.193066588376,0.193056696728,0.193046806627,0.193036918071,0.19302703106,0.193017145594,0.193007261673,0.192997379295,0.192987498461,0.19297761917,0.192967741422,0.192957865216,0.192947990551,0.192938117429,0.192928245847,0.192918375807,0.192908507306,0.192898640345,0.192888774924,0.192878911042,0.192869048698,0.192859187893,0.192849328626,0.192839470896,0.192829614702,0.192819760046,0.192809906926,0.192800055341,0.192790205292,0.192780356778,0.192770509799,0.192760664354,0.192750820442,0.192740978064,0.192731137219,0.192721297906,0.192711460125,0.192701623877,0.192691789159,0.192681955973,0.192672124317,0.192662294191,0.192652465595,0.192642638528,0.19263281299,0.192622988981,0.1926131665,0.192603345546,0.19259352612,0.192583708221,0.192573891848,0.192564077001,0.19255426368,0.192544451885,0.192534641614,0.192524832867,0.192515025645,0.192505219947,0.192495415771,0.192485613119,0.192475811989,0.192466012381,0.192456214295,0.19244641773,0.192436622687,0.192426829163,0.19241703716,0.192407246676,0.192397457712,0.192387670266,0.192377884339,0.19236809993,0.192358317039,0.192348535665,0.192338755808,0.192328977467,0.192319200643,0.192309425334,0.19229965154,0.192289879262,0.192280108497,0.192270339247,0.192260571511,0.192250805288,0.192241040578,0.19223127738,0.192221515695,0.192211755521,0.192201996859,0.192192239707,0.192182484066,0.192172729935,0.192162977314,0.192153226202,0.192143476599,0.192133728505,0.192123981919,0.19211423684,0.192104493269,0.192094751205,0.192085010647,0.192075271596,0.19206553405,0.19205579801,0.192046063475,0.192036330444,0.192026598918,0.192016868895,0.192007140376,0.19199741336,0.191987687846,0.191977963835,0.191968241326,0.191958520317,0.19194880081,0.191939082804,0.191929366298,0.191919651292,0.191909937785,0.191900225777,0.191890515269,0.191880806258,0.191871098745,0.19186139273,0.191851688212,0.191841985191,0.191832283666,0.191822583637,0.191812885103,0.191803188065,0.191793492521,0.191783798472,0.191774105917,0.191764414856,0.191754725287,0.191745037212,0.191735350629,0.191725665538,0.191715981939,0.191706299831,0.191696619214,0.191686940087,0.19167726245,0.191667586304,0.191657911646,0.191648238477,0.191638566798,0.191628896606,0.191619227902,0.191609560685,0.191599894955,0.191590230712,0.191580567955,0.191570906684,0.191561246899,0.191551588598,0.191541931783,0.191532276451,0.191522622603,0.191512970239,0.191503319358,0.19149366996,0.191484022044,0.19147437561,0.191464730658,0.191455087187,0.191445445196,0.191435804687,0.191426165657,0.191416528107,0.191406892036,0.191397257444,0.19138762433,0.191377992695,0.191368362537,0.191358733857,0.191349106653,0.191339480926,0.191329856676,0.191320233901,0.191310612602,0.191300992777,0.191291374428,0.191281757552,0.191272142151,0.191262528223,0.191252915768,0.191243304786,0.191233695276,0.191224087239,0.191214480673,0.191204875578,0.191195271954,0.1911856698,0.191176069116,0.191166469903,0.191156872158,0.191147275882,0.191137681075,0.191128087737,0.191118495865,0.191108905462,0.191099316525,0.191089729055,0.191080143051,0.191070558513,0.191060975441,0.191051393834,0.191041813691,0.191032235013,0.191022657799,0.191013082048,0.191003507761,0.190993934937,0.190984363575,0.190974793675,0.190965225237,0.19095565826,0.190946092744,0.190936528689,0.190926966093,0.190917404958,0.190907845282,0.190898287066,0.190888730307,0.190879175008,0.190869621166,0.190860068781,0.190850517854,0.190840968384,0.19083142037,0.190821873812,0.19081232871,0.190802785063,0.190793242871,0.190783702134,0.190774162851,0.190764625021,0.190755088645,0.190745553722,0.190736020252,0.190726488234,0.190716957668,0.190707428554,0.19069790089,0.190688374678,0.190678849916,0.190669326603,0.190659804741,0.190650284328,0.190640765364,0.190631247848,0.19062173178,0.190612217161,0.190602703989,0.190593192263,0.190583681985,0.190574173153,0.190564665766,0.190555159826,0.19054565533,0.190536152279,0.190526650673,0.190517150511,0.190507651793,0.190498154517,0.190488658685,0.190479164296,0.190469671348,0.190460179843,0.190450689779,0.190441201156,0.190431713974,0.190422228232,0.19041274393,0.190403261068,0.190393779645,0.19038429966,0.190374821115,0.190365344007,0.190355868338,0.190346394105,0.19033692131,0.190327449951,0.190317980029,0.190308511543,0.190299044492,0.190289578876,0.190280114695,0.190270651949,0.190261190637,0.190251730758,0.190242272313,0.190232815301,0.190223359721,0.190213905573,0.190204452858,0.190195001574,0.190185551721,0.190176103299,0.190166656307,0.190157210745,0.190147766613,0.19013832391,0.190128882636,0.190119442791,0.190110004374,0.190100567385,0.190091131823,0.190081697688,0.19007226498,0.190062833699,0.190053403843,0.190043975413,0.190034548409,0.190025122829,0.190015698674,0.190006275943,0.189996854636,0.189987434752,0.189978016291,0.189968599253,0.189959183638,0.189949769444,0.189940356672,0.189930945321,0.189921535392,0.189912126882,0.189902719793,0.189893314124,0.189883909874,0.189874507043,0.189865105631,0.189855705638,0.189846307062,0.189836909904,0.189827514163,0.189818119839,0.189808726932,0.189799335441,0.189789945366,0.189780556706,0.189771169461,0.189761783631,0.189752399215,0.189743016214,0.189733634626,0.189724254451,0.189714875689,0.18970549834,0.189696122403,0.189686747878,0.189677374764,0.189668003062,0.18965863277,0.189649263889,0.189639896418,0.189630530356,0.189621165704,0.189611802461,0.189602440626,0.189593080199,0.189583721181,0.18957436357,0.189565007366,0.189555652569,0.189546299179,0.189536947194,0.189527596616,0.189518247442,0.189508899674,0.18949955331,0.189490208351,0.189480864796,0.189471522644,0.189462181895,0.18945284255,0.189443504607,0.189434168066,0.189424832927,0.189415499189,0.189406166852,0.189396835917,0.189387506381,0.189378178246,0.18936885151,0.189359526173,0.189350202236,0.189340879697,0.189331558556,0.189322238814,0.189312920468,0.18930360352,0.189294287969,0.189284973815,0.189275661056,0.189266349693,0.189257039726,0.189247731153,0.189238423976,0.189229118192,0.189219813803,0.189210510807,0.189201209205,0.189191908995,0.189182610178,0.189173312754,0.189164016721,0.18915472208,0.18914542883,0.18913613697,0.189126846501,0.189117557423,0.189108269734,0.189098983434,0.189089698523,0.189080415002,0.189071132868,0.189061852123,0.189052572765,0.189043294794,0.18903401821,0.189024743013,0.189015469203,0.189006196778,0.188996925738,0.188987656084,0.188978387815,0.18896912093,0.188959855429,0.188950591312,0.188941328578,0.188932067228,0.18892280726,0.188913548675,0.188904291471,0.188895035649,0.188885781209,0.18887652815,0.188867276471,0.188858026173,0.188848777254,0.188839529715,0.188830283555,0.188821038775,0.188811795372,0.188802553348,0.188793312702,0.188784073433,0.188774835542,0.188765599027,0.188756363889,0.188747130126,0.18873789774,0.188728666729,0.188719437093,0.188710208831,0.188700981944,0.188691756432,0.188682532292,0.188673309526,0.188664088133,0.188654868113,0.188645649465,0.188636432189,0.188627216285,0.188618001751,0.188608788589,0.188599576797,0.188590366376,0.188581157324,0.188571949642,0.188562743329,0.188553538385,0.188544334809,0.188535132601,0.188525931762,0.188516732289,0.188507534184,0.188498337446,0.188489142074,0.188479948068,0.188470755428,0.188461564153,0.188452374243,0.188443185698,0.188433998518,0.188424812701,0.188415628248,0.188406445158,0.188397263432,0.188388083068,0.188378904066,0.188369726426,0.188360550148,0.188351375231,0.188342201675,0.18833302948,0.188323858644,0.188314689169,0.188305521054,0.188296354297,0.188287188899,0.18827802486,0.18826886218,0.188259700857,0.188250540891,0.188241382283,0.188232225031,0.188223069136,0.188213914598,0.188204761415,0.188195609587,0.188186459115,0.188177309997,0.188168162234,0.188159015825,0.18814987077,0.188140727068,0.188131584719,0.188122443723,0.188113304079,0.188104165788,0.188095028848,0.18808589326,0.188076759022,0.188067626136,0.1880584946,0.188049364413,0.188040235577,0.18803110809,0.188021981952,0.188012857162,0.188003733721,0.187994611628,0.187985490883,0.187976371485,0.187967253434,0.18795813673,0.187949021372,0.18793990736,0.187930794693,0.187921683372,0.187912573396,0.187903464765,0.187894357478,0.187885251535,0.187876146935,0.187867043679,0.187857941766,0.187848841195,0.187839741967,0.187830644081,0.187821547536,0.187812452333,0.18780335847,0.187794265948,0.187785174767,0.187776084925,0.187766996423,0.18775790926,0.187748823437,0.187739738951,0.187730655804,0.187721573995,0.187712493524,0.18770341439,0.187694336593,0.187685260132,0.187676185008,0.18766711122,0.187658038767,0.18764896765,0.187639897867,0.187630829419,0.187621762306,0.187612696526,0.18760363208,0.187594568967,0.187585507187,0.18757644674,0.187567387625,0.187558329842,0.187549273391,0.187540218271,0.187531164482,0.187522112023,0.187513060895,0.187504011097,0.187494962628,0.187485915489,0.187476869678,0.187467825197,0.187458782043,0.187449740218,0.18744069972,0.187431660549,0.187422622706,0.187413586189,0.187404550999,0.187395517134,0.187386484595,0.187377453382,0.187368423493,0.187359394929,0.18735036769,0.187341341774,0.187332317182,0.187323293914,0.187314271969,0.187305251346,0.187296232045,0.187287214067,0.187278197411,0.187269182075,0.187260168061,0.187251155368,0.187242143995,0.187233133942,0.187224125209,0.187215117795,0.1872061117,0.187197106924,0.187188103467,0.187179101328,0.187170100506,0.187161101002,0.187152102815,0.187143105945,0.187134110391,0.187125116154,0.187116123232,0.187107131626,0.187098141335,0.187089152359,0.187080164697,0.187071178349,0.187062193316,0.187053209596,0.187044227189,0.187035246095,0.187026266314,0.187017287845,0.187008310687,0.186999334842,0.186990360307,0.186981387084,0.186972415171,0.186963444569,0.186954475276,0.186945507293,0.18693654062,0.186927575255,0.186918611199,0.186909648452,0.186900687012,0.186891726881,0.186882768056,0.186873810539,0.186864854328,0.186855899424,0.186846945826,0.186837993534,0.186829042547,0.186820092865,0.186811144488,0.186802197416,0.186793251647,0.186784307183,0.186775364022,0.186766422164,0.186757481609,0.186748542357,0.186739604407,0.186730667759,0.186721732412,0.186712798367,0.186703865622,0.186694934178,0.186686004035,0.186677075191,0.186668147647,0.186659221403,0.186650296457,0.18664137281,0.186632450462,0.186623529411,0.186614609659,0.186605691203,0.186596774045,0.186587858184,0.186578943619,0.18657003035,0.186561118377,0.1865522077,0.186543298317,0.18653439023,0.186525483437,0.186516577938,0.186507673733,0.186498770822,0.186489869204,0.186480968879,0.186472069847,0.186463172107,0.186454275659,0.186445380503,0.186436486638,0.186427594064,0.18641870278,0.186409812788,0.186400924085,0.186392036672,0.186383150548,0.186374265714,0.186365382168,0.186356499911,0.186347618943,0.186338739262,0.186329860868,0.186320983762,0.186312107943,0.18630323341,0.186294360164,0.186285488203,0.186276617529,0.186267748139,0.186258880035,0.186250013215,0.18624114768,0.186232283429,0.186223420461,0.186214558777,0.186205698377,0.186196839259,0.186187981423,0.18617912487,0.186170269599,0.186161415609,0.1861525629,0.186143711473,0.186134861326,0.186126012459,0.186117164872,0.186108318565,0.186099473537,0.186090629789,0.186081787319,0.186072946127,0.186064106214,0.186055267579,0.186046430221,0.18603759414,0.186028759336,0.186019925808,0.186011093557,0.186002262582,0.185993432882,0.185984604458,0.185975777308,0.185966951434,0.185958126833,0.185949303507,0.185940481455,0.185931660675,0.18592284117,0.185914022936,0.185905205976,0.185896390287,0.185887575871,0.185878762725,0.185869950852,0.185861140249,0.185852330917,0.185843522855,0.185834716063,0.185825910541,0.185817106288,0.185808303304,0.185799501589,0.185790701143,0.185781901964,0.185773104054,0.185764307411,0.185755512035,0.185746717926,0.185737925084,0.185729133508,0.185720343198,0.185711554154,0.185702766375,0.185693979861,0.185685194612,0.185676410628,0.185667627907,0.185658846451,0.185650066258,0.185641287328,0.185632509661,0.185623733256,0.185614958114,0.185606184234,0.185597411615,0.185588640258,0.185579870162,0.185571101327,0.185562333752,0.185553567437,0.185544802383,0.185536038587,0.185527276051,0.185518514774,0.185509754755,0.185500995995,0.185492238493,0.185483482248,0.185474727261,0.185465973531,0.185457221057,0.18544846984,0.185439719879,0.185430971174,0.185422223725,0.185413477531,0.185404732592,0.185395988907,0.185387246477,0.185378505301,0.185369765378,0.185361026709,0.185352289293,0.18534355313,0.185334818219,0.185326084561,0.185317352154,0.185308620999,0.185299891095,0.185291162443,0.185282435041,0.185273708889,0.185264983987,0.185256260335,0.185247537933,0.18523881678,0.185230096875,0.18522137822,0.185212660812,0.185203944652,0.18519522974,0.185186516076,0.185177803658,0.185169092487,0.185160382563,0.185151673885,0.185142966452,0.185134260265,0.185125555324,0.185116851627,0.185108149175,0.185099447967,0.185090748003,0.185082049283,0.185073351806,0.185064655573,0.185055960582,0.185047266834,0.185038574328,0.185029883064,0.185021193042,0.18501250426,0.18500381672,0.184995130421,0.184986445362,0.184977761543,0.184969078964,0.184960397624,0.184951717524,0.184943038663,0.18493436104,0.184925684656,0.184917009509,0.1849083356,0.184899662929,0.184890991495,0.184882321297,0.184873652337,0.184864984612,0.184856318123,0.18484765287,0.184838988852,0.18483032607,0.184821664522,0.184813004208,0.184804345129,0.184795687283,0.184787030671,0.184778375292,0.184769721147,0.184761068233,0.184752416553,0.184743766104,0.184735116887,0.184726468901,0.184717822147,0.184709176623,0.18470053233,0.184691889268,0.184683247435,0.184674606832,0.184665967458,0.184657329314,0.184648692398,0.18464005671,0.184631422251,0.18462278902,0.184614157017,0.18460552624,0.184596896691,0.184588268368,0.184579641272,0.184571015402,0.184562390758,0.18455376734,0.184545145146,0.184536524178,0.184527904434,0.184519285915,0.184510668619,0.184502052548,0.1844934377,0.184484824075,0.184476211673,0.184467600493,0.184458990536,0.184450381801,0.184441774288,0.184433167996,0.184424562925,0.184415959075,0.184407356445,0.184398755036,0.184390154847,0.184381555877,0.184372958127,0.184364361596,0.184355766283,0.184347172189,0.184338579314,0.184329987656,0.184321397216,0.184312807993,0.184304219987,0.184295633198,0.184287047625,0.184278463268,0.184269880128,0.184261298203,0.184252717493,0.184244137998,0.184235559718,0.184226982652,0.184218406801,0.184209832163,0.184201258738,0.184192686527,0.184184115529,0.184175545744,0.184166977171,0.18415840981,0.184149843661,0.184141278723,0.184132714996,0.184124152481,0.184115591176,0.184107031081,0.184098472197,0.184089914522,0.184081358056,0.1840728028,0.184064248753,0.184055695914,0.184047144284,0.184038593861,0.184030044647,0.18402149664,0.184012949839,0.184004404246,0.18399585986,0.183987316679,0.183978774705,0.183970233936,0.183961694373,0.183953156015,0.183944618862,0.183936082913,0.183927548169,0.183919014629,0.183910482292,0.183901951159,0.183893421228,0.183884892501,0.183876364976,0.183867838654,0.183859313533,0.183850789614,0.183842266897,0.183833745381,0.183825225065,0.18381670595,0.183808188035,0.183799671321,0.183791155806,0.18378264149,0.183774128373,0.183765616455,0.183757105736,0.183748596215,0.183740087892,0.183731580767,0.183723074839,0.183714570108,0.183706066574,0.183697564237,0.183689063095,0.18368056315,0.183672064401,0.183663566846,0.183655070487,0.183646575323,0.183638081353,0.183629588578,0.183621096996,0.183612606609,0.183604117414,0.183595629413,0.183587142605,0.183578656989,0.183570172565,0.183561689334,0.183553207294,0.183544726445,0.183536246788,0.183527768322,0.183519291046,0.18351081496,0.183502340064,0.183493866359,0.183485393842,0.183476922515,0.183468452376,0.183459983427,0.183451515665,0.183443049092,0.183434583706,0.183426119507,0.183417656496,0.183409194672,0.183400734035,0.183392274583,0.183383816318,0.183375359239,0.183366903345,0.183358448636,0.183349995112,0.183341542773,0.183333091618,0.183324641648,0.183316192861,0.183307745257,0.183299298837,0.1832908536,0.183282409545,0.183273966673,0.183265524983,0.183257084475,0.183248645149,0.183240207003,0.183231770039,0.183223334255,0.183214899652,0.183206466229,0.183198033986,0.183189602923,0.183181173039,0.183172744333,0.183164316807,0.183155890459,0.18314746529,0.183139041298,0.183130618484,0.183122196847,0.183113776387,0.183105357104,0.183096938998,0.183088522068,0.183080106314,0.183071691736,0.183063278333,0.183054866105,0.183046455052,0.183038045174,0.18302963647,0.18302122894,0.183012822583,0.1830044174,0.182996013391,0.182987610554,0.18297920889,0.182970808399,0.182962409079,0.182954010931,0.182945613955,0.18293721815,0.182928823516,0.182920430053,0.18291203776,0.182903646637,0.182895256685,0.182886867901,0.182878480287,0.182870093843,0.182861708567,0.182853324459,0.18284494152,0.182836559748,0.182828179145,0.182819799708,0.182811421439,0.182803044337,0.182794668401,0.182786293632,0.182777920028,0.18276954759,0.182761176318,0.182752806211,0.182744437269,0.182736069492,0.182727702879,0.18271933743,0.182710973144,0.182702610023,0.182694248064,0.182685887269,0.182677527636,0.182669169166,0.182660811858,0.182652455712,0.182644100728,0.182635746905,0.182627394243,0.182619042742,0.182610692401,0.182602343221,0.1825939952,0.18258564834,0.182577302639,0.182568958097,0.182560614713,0.182552272489,0.182543931423,0.182535591515,0.182527252765,0.182518915172,0.182510578737,0.182502243458,0.182493909337,0.182485576371,0.182477244562,0.182468913909,0.182460584412,0.18245225607,0.182443928883,0.182435602851,0.182427277973,0.18241895425,0.18241063168,0.182402310265,0.182393990003,0.182385670894,0.182377352938,0.182369036135,0.182360720484,0.182352405985,0.182344092638,0.182335780443,0.182327469398,0.182319159505,0.182310850763,0.182302543172,0.18229423673,0.182285931439,0.182277627297,0.182269324304,0.182261022461,0.182252721767,0.182244422221,0.182236123823,0.182227826574,0.182219530473,0.182211235519,0.182202941712,0.182194649052,0.182186357539,0.182178067173,0.182169777952,0.182161489878,0.182153202949,0.182144917166,0.182136632528,0.182128349034,0.182120066685,0.182111785481,0.182103505421,0.182095226504,0.182086948731,0.182078672101,0.182070396614,0.18206212227,0.182053849068,0.182045577009,0.182037306091,0.182029036315,0.182020767681,0.182012500187,0.182004233835,0.181995968623,0.181987704551,0.181979441619,0.181971179828,0.181962919175,0.181954659662,0.181946401288,0.181938144053,0.181929887956,0.181921632997,0.181913379176,0.181905126493,0.181896874947,0.181888624539,0.181880375267,0.181872127132,0.181863880133,0.18185563427,0.181847389543,0.181839145952,0.181830903496,0.181822662174,0.181814421988,0.181806182936,0.181797945018,0.181789708235,0.181781472584,0.181773238068,0.181765004684,0.181756772434,0.181748541316,0.18174031133,0.181732082477,0.181723854755,0.181715628165,0.181707402706,0.181699178378,0.181690955181,0.181682733115,0.181674512179,0.181666292373,0.181658073696,0.181649856149,0.181641639731,0.181633424443,0.181625210282,0.181616997251,0.181608785347,0.181600574571,0.181592364923,0.181584156403,0.181575949009,0.181567742743,0.181559537602,0.181551333589,0.181543130701,0.181534928939,0.181526728303,0.181518528792,0.181510330406,0.181502133144,0.181493937007,0.181485741995,0.181477548106,0.181469355341,0.1814611637,0.181452973181,0.181444783786,0.181436595513,0.181428408363,0.181420222335,0.181412037429,0.181403853644,0.181395670981,0.181387489439,0.181379309018,0.181371129717,0.181362951537,0.181354774476,0.181346598536,0.181338423715,0.181330250013,0.181322077431,0.181313905967,0.181305735621,0.181297566394,0.181289398285,0.181281231294,0.18127306542,0.181264900663,0.181256737024,0.181248574501,0.181240413094,0.181232252804,0.181224093629,0.18121593557,0.181207778627,0.181199622799,0.181191468085,0.181183314487,0.181175162002,0.181167010632,0.181158860376,0.181150711233,0.181142563204,0.181134416287,0.181126270484,0.181118125793,0.181109982214,0.181101839748,0.181093698393,0.18108555815,0.181077419018,0.181069280997,0.181061144087,0.181053008288,0.181044873598,0.181036740019,0.18102860755,0.181020476189,0.181012345939,0.181004216797,0.180996088764,0.180987961839,0.180979836023,0.180971711314,0.180963587713,0.18095546522,0.180947343834,0.180939223555,0.180931104382,0.180922986316,0.180914869356,0.180906753502,0.180898638754,0.180890525111,0.180882412573,0.180874301141,0.180866190812,0.180858081589,0.180849973469,0.180841866453,0.180833760541,0.180825655732,0.180817552026,0.180809449423,0.180801347923,0.180793247525,0.18078514823,0.180777050036,0.180768952943,0.180760856952,0.180752762062,0.180744668273,0.180736575585,0.180728483996,0.180720393508,0.18071230412,0.180704215831,0.180696128642,0.180688042552,0.18067995756,0.180671873667,0.180663790872,0.180655709176,0.180647628577,0.180639549076,0.180631470672,0.180623393365,0.180615317154,0.180607242041,0.180599168024,0.180591095102,0.180583023277,0.180574952547,0.180566882912,0.180558814372,0.180550746927,0.180542680577,0.180534615321,0.180526551159,0.18051848809,0.180510426116,0.180502365234,0.180494305446,0.18048624675,0.180478189147,0.180470132636,0.180462077217,0.18045402289,0.180445969654,0.18043791751,0.180429866456,0.180421816494,0.180413767621,0.18040571984,0.180397673148,0.180389627546,0.180381583033,0.18037353961,0.180365497276,0.18035745603,0.180349415873,0.180341376804,0.180333338824,0.180325301931,0.180317266125,0.180309231407,0.180301197776,0.180293165232,0.180285133774,0.180277103403,0.180269074117,0.180261045917,0.180253018803,0.180244992774,0.18023696783,0.180228943971,0.180220921197,0.180212899506,0.1802048789,0.180196859378,0.180188840939,0.180180823583,0.180172807311,0.180164792121,0.180156778014,0.180148764989,0.180140753046,0.180132742185,0.180124732406,0.180116723708,0.180108716091,0.180100709555,0.180092704099,0.180084699724,0.180076696429,0.180068694213,0.180060693078,0.180052693021,0.180044694044,0.180036696145,0.180028699326,0.180020703584,0.180012708921,0.180004715335,0.179996722827,0.179988731397,0.179980741044,0.179972751767,0.179964763567,0.179956776444,0.179948790397,0.179940805425,0.179932821529,0.179924838709,0.179916856964,0.179908876294,0.179900896698,0.179892918177,0.17988494073,0.179876964357,0.179868989058,0.179861014832,0.17985304168,0.1798450696,0.179837098593,0.179829128659,0.179821159797,0.179813192007,0.179805225288,0.179797259641,0.179789295066,0.179781331561,0.179773369127,0.179765407764,0.179757447471,0.179749488248,0.179741530095,0.179733573011,0.179725616997,0.179717662052,0.179709708175,0.179701755367,0.179693803628,0.179685852956,0.179677903353,0.179669954817,0.179662007348,0.179654060947,0.179646115612,0.179638171344,0.179630228143,0.179622286008,0.179614344938,0.179606404934,0.179598465996,0.179590528123,0.179582591315,0.179574655571,0.179566720892,0.179558787278,0.179550854727,0.17954292324,0.179534992816,0.179527063456,0.179519135159,0.179511207924,0.179503281752,0.179495356643,0.179487432595,0.179479509609,0.179471587685,0.179463666822,0.17945574702,0.179447828279,0.179439910599,0.179431993979,0.179424078419,0.179416163919,0.179408250479,0.179400338098,0.179392426777,0.179384516514,0.17937660731,0.179368699164,0.179360792077,0.179352886047,0.179344981076,0.179337077162,0.179329174305,0.179321272505,0.179313371762,0.179305472075,0.179297573445,0.17928967587,0.179281779352,0.179273883889,0.179265989482,0.179258096129,0.179250203832,0.179242312589,0.1792344224,0.179226533266,0.179218645185,0.179210758159,0.179202872185,0.179194987265,0.179187103398,0.179179220584,0.179171338822,0.179163458112,0.179155578454,0.179147699848,0.179139822294,0.179131945791,0.179124070339,0.179116195938,0.179108322587,0.179100450287,0.179092579036,0.179084708836,0.179076839686,0.179068971584,0.179061104532,0.179053238529,0.179045373575,0.179037509669,0.179029646811,0.179021785001,0.179013924239,0.179006064524,0.178998205857,0.178990348237,0.178982491664,0.178974636137,0.178966781656,0.178958928222,0.178951075833,0.17894322449,0.178935374193,0.17892752494,0.178919676733,0.17891182957,0.178903983452,0.178896138377,0.178888294347,0.178880451361,0.178872609418,0.178864768518,0.178856928662,0.178849089848,0.178841252077,0.178833415348,0.178825579661,0.178817745016,0.178809911413,0.178802078851,0.17879424733,0.17878641685,0.178778587411,0.178770759013,0.178762931654,0.178755105336,0.178747280057,0.178739455818,0.178731632618,0.178723810457,0.178715989335,0.178708169252,0.178700350207,0.1786925322,0.178684715231,0.1786768993,0.178669084406,0.178661270549,0.17865345773,0.178645645947,0.1786378352,0.17863002549,0.178622216816,0.178614409177,0.178606602574,0.178598797007,0.178590992474,0.178583188977,0.178575386514,0.178567585086,0.178559784691,0.178551985331,0.178544187004,0.178536389711,0.178528593451,0.178520798224,0.17851300403,0.178505210868,0.178497418739,0.178489627642,0.178481837577,0.178474048543,0.178466260541,0.17845847357,0.178450687629,0.17844290272,0.178435118841,0.178427335992,0.178419554173,0.178411773384,0.178403993625,0.178396214895,0.178388437194,0.178380660522,0.178372884878,0.178365110263,0.178357336676,0.178349564117,0.178341792586,0.178334022082,0.178326252606,0.178318484156,0.178310716733,0.178302950337,0.178295184967,0.178287420624,0.178279657306,0.178271895014,0.178264133747,0.178256373505,0.178248614289,0.178240856097,0.178233098929,0.178225342786,0.178217587667,0.178209833572,0.1782020805,0.178194328452,0.178186577426,0.178178827424,0.178171078445,0.178163330487,0.178155583552,0.178147837639,0.178140092748,0.178132348879,0.17812460603,0.178116864203,0.178109123396,0.178101383611,0.178093644845,0.1780859071,0.178078170375,0.178070434669,0.178062699983,0.178054966316,0.178047233668,0.178039502039,0.178031771429,0.178024041837,0.178016313263,0.178008585706,0.178000859168,0.177993133647,0.177985409143,0.177977685657,0.177969963187,0.177962241733,0.177954521296,0.177946801875,0.17793908347,0.17793136608,0.177923649706,0.177915934347,0.177908220003,0.177900506674,0.177892794359,0.177885083058,0.177877372772,0.177869663499,0.17786195524,0.177854247995,0.177846541762,0.177838836543,0.177831132336,0.177823429141,0.177815726959,0.177808025789,0.177800325631,0.177792626484,0.177784928349,0.177777231225,0.177769535111,0.177761840009,0.177754145916,0.177746452834,0.177738760762,0.1777310697,0.177723379648,0.177715690604,0.17770800257,0.177700315545,0.177692629528,0.17768494452,0.17767726052,0.177669577528,0.177661895544,0.177654214567,0.177646534598,0.177638855636,0.17763117768,0.177623500732,0.177615824789,0.177608149853,0.177600475923,0.177592802999,0.17758513108,0.177577460166,0.177569790258,0.177562121354,0.177554453455,0.177546786561,0.177539120671,0.177531455784,0.177523791902,0.177516129023,0.177508467147,0.177500806275,0.177493146405,0.177485487538,0.177477829673,0.177470172811,0.17746251695,0.177454862092,0.177447208234,0.177439555378,0.177431903524,0.17742425267,0.177416602817,0.177408953964,0.177401306111,0.177393659259,0.177386013406,0.177378368552,0.177370724698,0.177363081844,0.177355439988,0.17734779913,0.177340159271,0.177332520411,0.177324882548,0.177317245683,0.177309609816,0.177301974946,0.177294341073,0.177286708197,0.177279076318,0.177271445435,0.177263815548,0.177256186658,0.177248558763,0.177240931864,0.17723330596,0.177225681052,0.177218057138,0.177210434219,0.177202812294,0.177195191364,0.177187571428,0.177179952485,0.177172334536,0.177164717581,0.177157101619,0.177149486649,0.177141872673,0.177134259689,0.177126647697,0.177119036697,0.17711142669,0.177103817673,0.177096209649,0.177088602615,0.177080996572,0.177073391521,0.177065787459,0.177058184388,0.177050582308,0.177042981217,0.177035381115,0.177027782004,0.177020183881,0.177012586747,0.177004990603,0.176997395447,0.176989801279,0.176982208099,0.176974615907,0.176967024703,0.176959434487,0.176951845258,0.176944257016,0.17693666976,0.176929083492,0.176921498209,0.176913913913,0.176906330603,0.176898748279,0.17689116694,0.176883586587,0.176876007219,0.176868428835,0.176860851437,0.176853275022,0.176845699592,0.176838125146,0.176830551684,0.176822979206,0.176815407711,0.176807837199,0.17680026767,0.176792699123,0.17678513156,0.176777564978,0.176769999379,0.176762434761,0.176754871125,0.176747308471,0.176739746798,0.176732186106,0.176724626394,0.176717067664,0.176709509913,0.176701953143,0.176694397353,0.176686842542,0.176679288711,0.17667173586,0.176664183987,0.176656633093,0.176649083178,0.176641534241,0.176633986283,0.176626439302,0.1766188933,0.176611348275,0.176603804227,0.176596261157,0.176588719063,0.176581177946,0.176573637806,0.176566098642,0.176558560454,0.176551023242,0.176543487005,0.176535951744,0.176528417458,0.176520884148,0.176513351812,0.17650582045,0.176498290063,0.176490760651,0.176483232212,0.176475704747,0.176468178255,0.176460652737,0.176453128192,0.176445604619,0.17643808202,0.176430560393,0.176423039738,0.176415520055,0.176408001344,0.176400483605,0.176392966837,0.17638545104,0.176377936214,0.176370422359,0.176362909474,0.17635539756,0.176347886616,0.176340376642,0.176332867638,0.176325359603,0.176317852537,0.176310346441,0.176302841313,0.176295337154,0.176287833963,0.176280331741,0.176272830487,0.1762653302,0.176257830881,0.17625033253,0.176242835145,0.176235338728,0.176227843277,0.176220348793,0.176212855275,0.176205362724,0.176197871138,0.176190380518,0.176182890863,0.176175402174,0.17616791445,0.17616042769,0.176152941896,0.176145457065,0.176137973199,0.176130490297,0.176123008359,0.176115527384,0.176108047373,0.176100568325,0.17609309024,0.176085613118,0.176078136958,0.176070661761,0.176063187525,0.176055714252,0.17604824194,0.17604077059,0.176033300201,0.176025830773,0.176018362307,0.1760108948,0.176003428255,0.175995962669,0.175988498044,0.175981034378,0.175973571672,0.175966109925,0.175958649138,0.17595118931,0.17594373044,0.175936272529,0.175928815576,0.175921359582,0.175913904545,0.175906450467,0.175898997345,0.175891545182,0.175884093975,0.175876643725,0.175869194432,0.175861746095,0.175854298715,0.175846852291,0.175839406822,0.17583196231,0.175824518753,0.175817076151,0.175809634504,0.175802193812,0.175794754074,0.175787315291,0.175779877463,0.175772440588,0.175765004667,0.1757575697,0.175750135686,0.175742702625,0.175735270518,0.175727839363,0.17572040916,0.17571297991,0.175705551613,0.175698124267,0.175690697873,0.17568327243,0.175675847939,0.175668424399,0.175661001809,0.175653580171,0.175646159483,0.175638739745,0.175631320958,0.17562390312,0.175616486232,0.175609070294,0.175601655305,0.175594241265,0.175586828173,0.175579416031,0.175572004837,0.175564594591,0.175557185293,0.175549776943,0.175542369541,0.175534963086,0.175527557578,0.175520153018,0.175512749404,0.175505346737,0.175497945016,0.175490544241,0.175483144413,0.17547574553,0.175468347593,0.175460950601,0.175453554555,0.175446159453,0.175438765296,0.175431372084,0.175423979816,0.175416588492,0.175409198113,0.175401808677,0.175394420184,0.175387032635,0.175379646029,0.175372260366,0.175364875646,0.175357491868,0.175350109033,0.17534272714,0.175335346188,0.175327966179,0.175320587111,0.175313208984,0.175305831798,0.175298455554,0.17529108025,0.175283705886,0.175276332463,0.17526895998,0.175261588437,0.175254217834,0.17524684817,0.175239479445,0.175232111659,0.175224744813,0.175217378905,0.175210013935,0.175202649904,0.175195286811,0.175187924656,0.175180563438,0.175173203158,0.175165843815,0.17515848541,0.175151127941,0.175143771409,0.175136415813,0.175129061153,0.17512170743,0.175114354642,0.175107002791,0.175099651874,0.175092301893,0.175084952847,0.175077604736,0.175070257559,0.175062911317,0.175055566009,0.175048221636,0.175040878196,0.17503353569,0.175026194117,0.175018853477,0.175011513771,0.175004174997,0.174996837156,0.174989500248,0.174982164271,0.174974829227,0.174967495115,0.174960161934,0.174952829685,0.174945498367,0.17493816798,0.174930838524,0.174923509998,0.174916182403,0.174908855739,0.174901530004,0.174894205199,0.174886881324,0.174879558379,0.174872236362,0.174864915275,0.174857595116,0.174850275886,0.174842957585,0.174835640212,0.174828323767,0.17482100825,0.17481369366,0.174806379998,0.174799067263,0.174791755456,0.174784444575,0.174777134621,0.174769825593,0.174762517492,0.174755210316,0.174747904067,0.174740598743,0.174733294345,0.174725990872,0.174718688324,0.174711386701,0.174704086003,0.174696786229,0.17468948738,0.174682189454,0.174674892453,0.174667596375,0.174660301221,0.17465300699,0.174645713682,0.174638421297,0.174631129835,0.174623839295,0.174616549678,0.174609260983,0.174601973209,0.174594686358,0.174587400428,0.174580115419,0.174572831331,0.174565548165,0.174558265919,0.174550984594,0.174543704189,0.174536424704,0.174529146139,0.174521868494,0.174514591769,0.174507315962,0.174500041075,0.174492767108,0.174485494058,0.174478221928,0.174470950716,0.174463680422,0.174456411046,0.174449142588,0.174441875047,0.174434608424,0.174427342718,0.174420077929,0.174412814057,0.174405551102,0.174398289063,0.17439102794,0.174383767733,0.174376508442,0.174369250067,0.174361992608,0.174354736063,0.174347480434,0.174340225719,0.174332971919,0.174325719034,0.174318467063,0.174311216006,0.174303965863,0.174296716634,0.174289468318,0.174282220916,0.174274974426,0.17426772885,0.174260484186,0.174253240435,0.174245997596,0.17423875567,0.174231514655,0.174224274552,0.174217035361,0.174209797081,0.174202559712,0.174195323254,0.174188087707,0.174180853071,0.174173619345,0.174166386529,0.174159154624,0.174151923628,0.174144693542,0.174137464365,0.174130236097,0.174123008739,0.174115782289,0.174108556748,0.174101332116,0.174094108392,0.174086885576,0.174079663668,0.174072442667,0.174065222574,0.174058003389,0.17405078511,0.174043567739,0.174036351274,0.174029135716,0.174021921064,0.174014707318,0.174007494478,0.174000282545,0.173993071516,0.173985861393,0.173978652176,0.173971443863,0.173964236455,0.173957029952,0.173949824353,0.173942619658,0.173935415868,0.173928212981,0.173921010998,0.173913809919,0.173906609743,0.17389941047,0.1738922121,0.173885014632,0.173877818067,0.173870622405,0.173863427644,0.173856233786,0.173849040829,0.173841848774,0.17383465762,0.173827467368,0.173820278016,0.173813089566,0.173805902015,0.173798715366,0.173791529616,0.173784344767,0.173777160817,0.173769977768,0.173762795617,0.173755614366,0.173748434014,0.173741254561,0.173734076007,0.173726898351,0.173719721593,0.173712545734,0.173705370772,0.173698196709,0.173691023543,0.173683851274,0.173676679903,0.173669509428,0.173662339851,0.17365517117,0.173648003385,0.173640836497,0.173633670505,0.173626505408,0.173619341208,0.173612177902,0.173605015492,0.173597853978,0.173590693358,0.173583533633,0.173576374802,0.173569216866,0.173562059824,0.173554903676,0.173547748422,0.173540594061,0.173533440594,0.17352628802,0.173519136339,0.173511985551,0.173504835656,0.173497686653,0.173490538543,0.173483391324,0.173476244997,0.173469099563,0.173461955019,0.173454811367,0.173447668607,0.173440526737,0.173433385758,0.173426245669,0.173419106471,0.173411968164,0.173404830746,0.173397694218,0.17339055858,0.173383423831,0.173376289972,0.173369157001,0.17336202492,0.173354893727,0.173347763423,0.173340634007,0.173333505479,0.17332637784,0.173319251088,0.173312125224,0.173305000247,0.173297876157,0.173290752955,0.173283630639,0.17327650921,0.173269388668,0.173262269011,0.173255150241,0.173248032357,0.173240915359,0.173233799246,0.173226684018,0.173219569676,0.173212456219,0.173205343646,0.173198231958,0.173191121155,0.173184011236,0.173176902201,0.17316979405,0.173162686782,0.173155580398,0.173148474897,0.17314137028,0.173134266545,0.173127163694,0.173120061724,0.173112960638,0.173105860433,0.173098761111,0.17309166267,0.173084565111,0.173077468433,0.173070372637,0.173063277722,0.173056183688,0.173049090534,0.173041998262,0.173034906869,0.173027816357,0.173020726724,0.173013637972,0.173006550099,0.172999463105,0.172992376991,0.172985291756,0.1729782074,0.172971123923,0.172964041324,0.172956959603,0.172949878761,0.172942798796,0.172935719709,0.1729286415,0.172921564169,0.172914487714,0.172907412137,0.172900337437,0.172893263613,0.172886190666,0.172879118595,0.1728720474,0.172864977081,0.172857907638,0.172850839071,0.172843771379,0.172836704562,0.17282963862,0.172822573553,0.172815509361,0.172808446043,0.1728013836,0.172794322031,0.172787261335,0.172780201514,0.172773142566,0.172766084491,0.17275902729,0.172751970961,0.172744915506,0.172737860923,0.172730807213,0.172723754375,0.172716702409,0.172709651315,0.172702601093,0.172695551742,0.172688503263,0.172681455655,0.172674408918,0.172667363052,0.172660318056,0.172653273932,0.172646230677,0.172639188292,0.172632146778,0.172625106133,0.172618066358,0.172611027452,0.172603989415,0.172596952248,0.172589915949,0.172582880519,0.172575845958,0.172568812264,0.172561779439,0.172554747482,0.172547716393,0.172540686171,0.172533656817,0.172526628329,0.172519600709,0.172512573956,0.17250554807,0.17249852305,0.172491498896,0.172484475608,0.172477453187,0.172470431631,0.172463410941,0.172456391116,0.172449372157,0.172442354062,0.172435336833,0.172428320468,0.172421304967,0.172414290331,0.17240727656,0.172400263652,0.172393251608,0.172386240427,0.17237923011,0.172372220657,0.172365212066,0.172358204339,0.172351197474,0.172344191471,0.172337186331,0.172330182053,0.172323178638,0.172316176084,0.172309174391,0.172302173561,0.172295173591,0.172288174483,0.172281176235,0.172274178848,0.172267182322,0.172260186657,0.172253191851,0.172246197906,0.17223920482,0.172232212594,0.172225221228,0.172218230721,0.172211241073,0.172204252284,0.172197264354,0.172190277283,0.172183291069,0.172176305715,0.172169321218,0.172162337579,0.172155354798,0.172148372875,0.172141391809,0.1721344116,0.172127432248,0.172120453753,0.172113476115,0.172106499333,0.172099523407,0.172092548338,0.172085574124,0.172078600767,0.172071628264,0.172064656618,0.172057685826,0.17205071589,0.172043746809,0.172036778582,0.17202981121,0.172022844692,0.172015879028,0.172008914219,0.172001950263,0.171994987161,0.171988024912,0.171981063517,0.171974102974,0.171967143285,0.171960184449,0.171953226465,0.171946269333,0.171939313054,0.171932357627,0.171925403052,0.171918449328,0.171911496456,0.171904544436,0.171897593266,0.171890642948,0.17188369348,0.171876744863,0.171869797097,0.171862850181,0.171855904115,0.171848958899,0.171842014532,0.171835071016,0.171828128349,0.171821186531,0.171814245562,0.171807305442,0.171800366171,0.171793427748,0.171786490174,0.171779553448,0.17177261757,0.171765682539,0.171758748357,0.171751815022,0.171744882534,0.171737950894,0.1717310201,0.171724090154,0.171717161053,0.1717102328,0.171703305393,0.171696378831,0.171689453116,0.171682528247,0.171675604223,0.171668681044,0.171661758711,0.171654837223,0.171647916579,0.171640996781,0.171634077827,0.171627159717,0.171620242452,0.17161332603,0.171606410453,0.171599495719,0.171592581828,0.171585668781,0.171578756577,0.171571845216,0.171564934698,0.171558025023,0.171551116189,0.171544208199,0.17153730105,0.171530394743,0.171523489278,0.171516584655,0.171509680873,0.171502777932,0.171495875833,0.171488974574,0.171482074156,0.171475174579,0.171468275842,0.171461377945,0.171454480888,0.171447584671,0.171440689294,0.171433794756,0.171426901058,0.171420008199,0.171413116179,0.171406224997,0.171399334655,0.17139244515,0.171385556485,0.171378668657,0.171371781667,0.171364895515,0.171358010201,0.171351125724,0.171344242084,0.171337359282,0.171330477316,0.171323596188,0.171316715895,0.17130983644,0.17130295782,0.171296080037,0.171289203089,0.171282326978,0.171275451701,0.171268577261,0.171261703655,0.171254830885,0.171247958949,0.171241087848,0.171234217582,0.17122734815,0.171220479552,0.171213611788,0.171206744858,0.171199878762,0.171193013499,0.17118614907,0.171179285474,0.171172422711,0.17116556078,0.171158699683,0.171151839417,0.171144979985,0.171138121384,0.171131263615,0.171124406678,0.171117550573,0.171110695299,0.171103840857,0.171096987245,0.171090134465,0.171083282515,0.171076431396,0.171069581108,0.171062731649,0.171055883021,0.171049035223,0.171042188255,0.171035342116,0.171028496806,0.171021652326,0.171014808675,0.171007965853,0.17100112386,0.170994282695,0.170987442359,0.17098060285,0.17097376417,0.170966926318,0.170960089294,0.170953253097,0.170946417728,0.170939583186,0.17093274947,0.170925916582,0.170919084521,0.170912253286,0.170905422877,0.170898593295,0.170891764539,0.170884936608,0.170878109504,0.170871283225,0.170864457771,0.170857633142,0.170850809339,0.17084398636,0.170837164206,0.170830342877,0.170823522372,0.170816702691,0.170809883835,0.170803065802,0.170796248593,0.170789432207,0.170782616645,0.170775801906,0.170768987989,0.170762174896,0.170755362626,0.170748551178,0.170741740552,0.170734930749,0.170728121767,0.170721313608,0.17071450627,0.170707699753,0.170700894058,0.170694089185,0.170687285132,0.1706804819,0.170673679489,0.170666877898,0.170660077127,0.170653277177,0.170646478047,0.170639679737,0.170632882246,0.170626085575,0.170619289723,0.170612494691,0.170605700477,0.170598907082,0.170592114506,0.170585322749,0.170578531809,0.170571741688,0.170564952385,0.1705581639,0.170551376233,0.170544589383,0.17053780335,0.170531018135,0.170524233737,0.170517450155,0.17051066739,0.170503885442,0.17049710431,0.170490323994,0.170483544495,0.170476765811,0.170469987943,0.17046321089,0.170456434653,0.170449659231,0.170442884624,0.170436110831,0.170429337854,0.170422565691,0.170415794343,0.170409023808,0.170402254088,0.170395485182,0.170388717089,0.17038194981,0.170375183344,0.170368417691,0.170361652852,0.170354888825,0.170348125611,0.17034136321,0.170334601621,0.170327840845,0.17032108088,0.170314321727,0.170307563386,0.170300805857,0.170294049139,0.170287293232,0.170280538136,0.170273783852,0.170267030378,0.170260277714,0.170253525861,0.170246774818,0.170240024586,0.170233275163,0.17022652655,0.170219778747,0.170213031753,0.170206285568,0.170199540193,0.170192795626,0.170186051868,0.170179308919,0.170172566778,0.170165825446,0.170159084922,0.170152345205,0.170145606297,0.170138868196,0.170132130903,0.170125394416,0.170118658738,0.170111923866,0.170105189801,0.170098456542,0.17009172409,0.170084992445,0.170078261605,0.170071531572,0.170064802344,0.170058073922,0.170051346306,0.170044619495,0.170037893489,0.170031168288,0.170024443892,0.170017720301,0.170010997514,0.170004275532,0.169997554354,0.16999083398,0.16998411441,0.169977395644,0.169970677681,0.169963960522,0.169957244166,0.169950528613,0.169943813862,0.169937099915,0.16993038677,0.169923674428,0.169916962888,0.16991025215,0.169903542214,0.16989683308,0.169890124748,0.169883417217,0.169876710487,0.169870004558,0.169863299431,0.169856595104,0.169849891578,0.169843188852,0.169836486927,0.169829785802,0.169823085477,0.169816385952,0.169809687226,0.169802989301,0.169796292174,0.169789595847,0.169782900318,0.169776205589,0.169769511658,0.169762818526,0.169756126192,0.169749434657,0.16974274392,0.16973605398,0.169729364839,0.169722676495,0.169715988948,0.169709302199,0.169702616247,0.169695931091,0.169689246733,0.169682563171,0.169675880406,0.169669198437,0.169662517264,0.169655836888,0.169649157307,0.169642478522,0.169635800532,0.169629123338,0.169622446938,0.169615771334,0.169609096525,0.169602422511,0.169595749291,0.169589076865,0.169582405234,0.169575734397,0.169569064354,0.169562395105,0.169555726649,0.169549058986,0.169542392117,0.169535726042,0.169529060759,0.169522396269,0.169515732571,0.169509069666,0.169502407554,0.169495746233,0.169489085705,0.169482425968,0.169475767023,0.16946910887,0.169462451508,0.169455794938,0.169449139158,0.169442484169,0.169435829972,0.169429176564,0.169422523947,0.169415872121,0.169409221084,0.169402570838,0.169395921381,0.169389272714,0.169382624837,0.169375977748,0.169369331449,0.169362685939,0.169356041218,0.169349397286,0.169342754142,0.169336111786,0.169329470219,0.16932282944,0.169316189448,0.169309550245,0.169302911829,0.1692962742,0.169289637359,0.169283001305,0.169276366038,0.169269731557,0.169263097864,0.169256464956,0.169249832835,0.169243201501,0.169236570952,0.169229941189,0.169223312212,0.169216684021,0.169210056614,0.169203429994,0.169196804158,0.169190179107,0.169183554841,0.169176931359,0.169170308662,0.169163686749,0.169157065621,0.169150445276,0.169143825715,0.169137206938,0.169130588944,0.169123971734,0.169117355307,0.169110739663,0.169104124802,0.169097510724,0.169090897428,0.169084284914,0.169077673183,0.169071062234,0.169064452067,0.169057842682,0.169051234078,0.169044626256,0.169038019215,0.169031412956,0.169024807477,0.169018202779,0.169011598862,0.169004995726,0.16899839337,0.168991791794,0.168985190998,0.168978590982,0.168971991746,0.16896539329,0.168958795613,0.168952198715,0.168945602597,0.168939007257,0.168932412697,0.168925818915,0.168919225911,0.168912633686,0.168906042239,0.16889945157,0.16889286168,0.168886272567,0.168879684231,0.168873096673,0.168866509892,0.168859923889,0.168853338662,0.168846754212,0.168840170539,0.168833587643,0.168827005523,0.168820424179,0.168813843611,0.168807263819,0.168800684802,0.168794106562,0.168787529097,0.168780952407,0.168774376492,0.168767801352,0.168761226987,0.168754653397,0.168748080581,0.16874150854,0.168734937272,0.168728366779,0.16872179706,0.168715228115,0.168708659943,0.168702092544,0.168695525919,0.168688960067,0.168682394988,0.168675830682,0.168669267148,0.168662704387,0.168656142399,0.168649581182,0.168643020738,0.168636461066,0.168629902165,0.168623344036,0.168616786678,0.168610230092,0.168603674277,0.168597119233,0.168590564959,0.168584011457,0.168577458725,0.168570906763,0.168564355571,0.16855780515,0.168551255498,0.168544706617,0.168538158505,0.168531611162,0.168525064589,0.168518518785,0.168511973749,0.168505429483,0.168498885986,0.168492343257,0.168485801296,0.168479260104,0.168472719679,0.168466180023,0.168459641135,0.168453103014,0.16844656566,0.168440029074,0.168433493255,0.168426958204,0.168420423919,0.1684138904,0.168407357649,0.168400825664,0.168394294445,0.168387763992,0.168381234305,0.168374705384,0.168368177229,0.168361649839,0.168355123214,0.168348597355,0.168342072261,0.168335547932,0.168329024367,0.168322501567,0.168315979532,0.168309458261,0.168302937754,0.168296418011,0.168289899032,0.168283380817,0.168276863365,0.168270346676,0.168263830751,0.168257315589,0.16825080119,0.168244287554,0.16823777468,0.168231262569,0.16822475122,0.168218240634,0.168211730809,0.168205221747,0.168198713446,0.168192205906,0.168185699129,0.168179193112,0.168172687857,0.168166183362,0.168159679629,0.168153176656,0.168146674444,0.168140172992,0.1681336723,0.168127172369,0.168120673197,0.168114174785,0.168107677133,0.168101180241,0.168094684107,0.168088188733,0.168081694118,0.168075200262,0.168068707165,0.168062214826,0.168055723246,0.168049232424,0.16804274236,0.168036253054,0.168029764506,0.168023276716,0.168016789683,0.168010303408,0.16800381789,0.167997333129,0.167990849125,0.167984365878,0.167977883388,0.167971401654,0.167964920676,0.167958440455,0.16795196099,0.16794548228,0.167939004327,0.167932527129,0.167926050686,0.167919574999,0.167913100067,0.16790662589,0.167900152468,0.167893679801,0.167887207888,0.16788073673,0.167874266326,0.167867796676,0.16786132778,0.167854859638,0.16784839225,0.167841925615,0.167835459734,0.167828994605,0.16782253023,0.167816066608,0.167809603739,0.167803141623,0.167796680259,0.167790219647,0.167783759787,0.16777730068,0.167770842325,0.167764384721,0.167757927869,0.167751471768,0.167745016419,0.167738561821,0.167732107974,0.167725654878,0.167719202533,0.167712750938,0.167706300094,0.16769985,0.167693400656,0.167686952062,0.167680504218,0.167674057124,0.167667610779,0.167661165184,0.167654720338,0.167648276241,0.167641832893,0.167635390294,0.167628948444,0.167622507342,0.167616066989,0.167609627383,0.167603188526,0.167596750417,0.167590313056,0.167583876442,0.167577440576,0.167571005458,0.167564571086,0.167558137462,0.167551704584,0.167545272454,0.16753884107,0.167532410432,0.167525980541,0.167519551396,0.167513122997,0.167506695344,0.167500268437,0.167493842275,0.167487416859,0.167480992188,0.167474568262,0.167468145082,0.167461722646,0.167455300955,0.167448880009,0.167442459807,0.167436040349,0.167429621636,0.167423203666,0.167416786441,0.167410369959,0.167403954221,0.167397539226,0.167391124975,0.167384711466,0.167378298701,0.167371886678,0.167365475399,0.167359064862,0.167352655067,0.167346246014,0.167339837704,0.167333430136,0.167327023309,0.167320617224,0.167314211881,0.167307807279,0.167301403419,0.167295000299,0.167288597921,0.167282196283,0.167275795386,0.16726939523,0.167262995814,0.167256597138,0.167250199202,0.167243802006,0.167237405551,0.167231009834,0.167224614858,0.16721822062,0.167211827122,0.167205434364,0.167199042344,0.167192651062,0.16718626052,0.167179870716,0.167173481651,0.167167093323,0.167160705734,0.167154318883,0.167147932769,0.167141547394,0.167135162755,0.167128778854,0.167122395691,0.167116013264,0.167109631575,0.167103250622,0.167096870406,0.167090490926,0.167084112183,0.167077734176,0.167071356905,0.16706498037,0.167058604571,0.167052229507,0.167045855179,0.167039481586,0.167033108729,0.167026736607,0.167020365219,0.167013994567,0.167007624649,0.167001255466,0.166994887017,0.166988519302,0.166982152321,0.166975786074,0.166969420561,0.166963055782,0.166956691736,0.166950328424,0.166943965845,0.166937603999,0.166931242886,0.166924882505,0.166918522858,0.166912163943,0.16690580576,0.166899448309,0.166893091591,0.166886735604,0.16688038035,0.166874025827,0.166867672035,0.166861318975,0.166854966647,0.166848615049,0.166842264182,0.166835914046,0.166829564641,0.166823215966,0.166816868022,0.166810520808,0.166804174324,0.16679782857,0.166791483546,0.166785139252,0.166778795687,0.166772452851,0.166766110745,0.166759769368,0.16675342872,0.1667470888,0.16674074961,0.166734411148,0.166728073414,0.166721736409,0.166715400132,0.166709064583,0.166702729761,0.166696395668,0.166690062302,0.166683729663,0.166677397752,0.166671066567,0.16666473611,0.16665840638,0.166652077376,0.166645749099,0.166639421549,0.166633094725,0.166626768627,0.166620443255,0.166614118608,0.166607794688,0.166601471493,0.166595149024,0.16658882728,0.166582506261,0.166576185967,0.166569866398,0.166563547554,0.166557229435,0.16655091204,0.166544595369,0.166538279423,0.1665319642,0.166525649702,0.166519335927,0.166513022876,0.166506710549,0.166500398944,0.166494088064,0.166487777906,0.166481468471,0.166475159759,0.166468851769,0.166462544502,0.166456237958,0.166449932136,0.166443627036,0.166437322657,0.166431019001,0.166424716066,0.166418413853,0.166412112362,0.166405811591,0.166399511542,0.166393212214,0.166386913606,0.16638061572,0.166374318554,0.166368022108,0.166361726383,0.166355431377,0.166349137092,0.166342843527,0.166336550682,0.166330258556,0.166323967149,0.166317676462,0.166311386494,0.166305097245,0.166298808715,0.166292520904,0.166286233812,0.166279947438,0.166273661782,0.166267376845,0.166261092625,0.166254809124,0.166248526341,0.166242244275,0.166235962926,0.166229682295,0.166223402382,0.166217123185,0.166210844705,0.166204566943,0.166198289897,0.166192013567,0.166185737954,0.166179463058,0.166173188877,0.166166915412,0.166160642664,0.166154370631,0.166148099313,0.166141828712,0.166135558825,0.166129289654,0.166123021198,0.166116753456,0.16611048643,0.166104220118,0.16609795452,0.166091689637,0.166085425469,0.166079162014,0.166072899273,0.166066637246,0.166060375933,0.166054115333,0.166047855447,0.166041596274,0.166035337814,0.166029080067,0.166022823033,0.166016566712,0.166010311103,0.166004056207,0.165997802023,0.165991548552,0.165985295792,0.165979043744,0.165972792408,0.165966541784,0.165960291871,0.165954042669,0.165947794179,0.1659415464,0.165935299332,0.165929052974,0.165922807328,0.165916562391,0.165910318166,0.16590407465,0.165897831845,0.165891589749,0.165885348364,0.165879107688,0.165872867722,0.165866628465,0.165860389918,0.16585415208,0.165847914951,0.16584167853,0.165835442819,0.165829207816,0.165822973522,0.165816739936,0.165810507058,0.165804274889,0.165798043427,0.165791812673,0.165785582627,0.165779353288,0.165773124657,0.165766896734,0.165760669517,0.165754443007,0.165748217204,0.165741992108,0.165735767719,0.165729544036,0.165723321059,0.165717098789,0.165710877225,0.165704656366,0.165698436214,0.165692216767,0.165685998026,0.16567977999,0.165673562659,0.165667346033,0.165661130113,0.165654914897,0.165648700386,0.16564248658,0.165636273478,0.16563006108,0.165623849387,0.165617638397,0.165611428112,0.16560521853,0.165599009652,0.165592801478,0.165586594006,0.165580387239,0.165574181174,0.165567975812,0.165561771153,0.165555567197,0.165549363943,0.165543161392,0.165536959543,0.165530758396,0.165524557952,0.165518358209,0.165512159168,0.165505960829,0.165499763191,0.165493566254,0.165487370019,0.165481174485,0.165474979652,0.165468785519,0.165462592088,0.165456399357,0.165450207326,0.165444015996,0.165437825366,0.165431635436,0.165425446206,0.165419257675,0.165413069844,0.165406882713,0.165400696281,0.165394510549,0.165388325515,0.16538214118,0.165375957545,0.165369774608,0.165363592369,0.165357410829,0.165351229988,0.165345049844,0.165338870399,0.165332691651,0.165326513601,0.165320336249,0.165314159595,0.165307983637,0.165301808377,0.165295633815,0.165289459949,0.16528328678,0.165277114307,0.165270942532,0.165264771452,0.165258601069,0.165252431383,0.165246262392,0.165240094097,0.165233926498,0.165227759595,0.165221593387,0.165215427875,0.165209263057,0.165203098935,0.165196935508,0.165190772776,0.165184610739,0.165178449396,0.165172288748,0.165166128794,0.165159969534,0.165153810968,0.165147653096,0.165141495919,0.165135339434,0.165129183644,0.165123028546,0.165116874142,0.165110720431,0.165104567414,0.165098415089,0.165092263457,0.165086112517,0.16507996227,0.165073812715,0.165067663853,0.165061515683,0.165055368204,0.165049221418,0.165043075323,0.16503692992,0.165030785208,0.165024641188,0.165018497858,0.16501235522,0.165006213273,0.165000072016,0.164993931451,0.164987791575,0.164981652391,0.164975513896,0.164969376092,0.164963238977,0.164957102553,0.164950966818,0.164944831773,0.164938697417,0.164932563751,0.164926430774,0.164920298486,0.164914166887,0.164908035976,0.164901905755,0.164895776222,0.164889647378,0.164883519221,0.164877391753,0.164871264974,0.164865138882,0.164859013477,0.164852888761,0.164846764732,0.164840641391,0.164834518736,0.164828396769,0.164822275489,0.164816154896,0.164810034989,0.16480391577,0.164797797236,0.164791679389,0.164785562229,0.164779445754,0.164773329965,0.164767214863,0.164761100446,0.164754986714,0.164748873668,0.164742761307,0.164736649632,0.164730538642,0.164724428336,0.164718318716,0.16471220978,0.164706101528,0.164699993961,0.164693887079,0.16468778088,0.164681675366,0.164675570535,0.164669466389,0.164663362926,0.164657260146,0.16465115805,0.164645056637,0.164638955907,0.164632855861,0.164626756497,0.164620657816,0.164614559817,0.164608462501,0.164602365868,0.164596269916,0.164590174647,0.16458408006,0.164577986155,0.164571892931,0.164565800389,0.164559708528,0.164553617349,0.164547526851,0.164541437034,0.164535347898,0.164529259443,0.164523171668,0.164517084574,0.164510998161,0.164504912428,0.164498827375,0.164492743002,0.164486659309,0.164480576296,0.164474493962,0.164468412308,0.164462331334,0.164456251039,0.164450171423,0.164444092486,0.164438014228,0.164431936648,0.164425859748,0.164419783526,0.164413707982,0.164407633116,0.164401558929,0.16439548542,0.164389412588,0.164383340435,0.164377268959,0.16437119816,0.164365128039,0.164359058595,0.164352989829,0.164346921739,0.164340854326,0.16433478759,0.16432872153,0.164322656147,0.164316591441,0.16431052741,0.164304464056,0.164298401378,0.164292339375,0.164286278049,0.164280217397,0.164274157422,0.164268098122,0.164262039497,0.164255981547,0.164249924272,0.164243867671,0.164237811746,0.164231756495,0.164225701919,0.164219648017,0.164213594789,0.164207542235,0.164201490355,0.16419543915,0.164189388617,0.164183338759,0.164177289573,0.164171241062,0.164165193223,0.164159146057,0.164153099565,0.164147053745,0.164141008598,0.164134964123,0.164128920321,0.164122877191,0.164116834734,0.164110792948,0.164104751835,0.164098711393,0.164092671623,0.164086632524,0.164080594097,0.164074556342,0.164068519257,0.164062482844,0.164056447101,0.164050412029,0.164044377628,0.164038343898,0.164032310838,0.164026278448,0.164020246728,0.164014215679,0.164008185299,0.164002155589,0.163996126549,0.163990098179,0.163984070478,0.163978043446,0.163972017083,0.16396599139,0.163959966365,0.16395394201,0.163947918322,0.163941895304,0.163935872954,0.163929851272,0.163923830258,0.163917809913,0.163911790235,0.163905771225,0.163899752883,0.163893735209,0.163887718202,0.163881701862,0.163875686189,0.163869671184,0.163863656845,0.163857643173,0.163851630168,0.16384561783,0.163839606158,0.163833595152,0.163827584812,0.163821575139,0.163815566131,0.163809557789,0.163803550113,0.163797543103,0.163791536758,0.163785531078,0.163779526063,0.163773521714,0.163767518029,0.16376151501,0.163755512655,0.163749510964,0.163743509939,0.163737509577,0.16373150988,0.163725510846,0.163719512477,0.163713514771,0.16370751773,0.163701521351,0.163695525637,0.163689530585,0.163683536197,0.163677542472,0.16367154941,0.163665557011,0.163659565274,0.1636535742,0.163647583789,0.16364159404,0.163635604953,0.163629616528,0.163623628766,0.163617641665,0.163611655226,0.163605669448,0.163599684332,0.163593699877,0.163587716084,0.163581732952,0.16357575048,0.16356976867,0.16356378752,0.163557807031,0.163551827203,0.163545848035,0.163539869527,0.163533891679,0.163527914492,0.163521937964,0.163515962096,0.163509986888,0.163504012339,0.163498038449,0.163492065219,0.163486092648,0.163480120736,0.163474149483,0.163468178889,0.163462208954,0.163456239677,0.163450271058,0.163444303098,0.163438335796,0.163432369152,0.163426403166,0.163420437838,0.163414473167,0.163408509154,0.163402545799,0.163396583101,0.16339062106,0.163384659676,0.163378698949,0.163372738879,0.163366779466,0.163360820709,0.163354862609,0.163348905165,0.163342948377,0.163336992246,0.16333103677,0.16332508195,0.163319127786,0.163313174278,0.163307221425,0.163301269228,0.163295317686,0.163289366799,0.163283416567,0.16327746699,0.163271518068,0.1632655698,0.163259622187,0.163253675228,0.163247728924,0.163241783273,0.163235838277,0.163229893935,0.163223950247,0.163218007212,0.163212064831,0.163206123103,0.163200182029,0.163194241607,0.163188301839,0.163182362724,0.163176424262,0.163170486452,0.163164549295,0.163158612791,0.163152676939,0.163146741739,0.163140807191,0.163134873295,0.163128940051,0.163123007459,0.163117075519,0.16311114423,0.163105213592,0.163099283606,0.163093354271,0.163087425587,0.163081497553,0.163075570171,0.163069643439,0.163063717358,0.163057791927,0.163051867147,0.163045943017,0.163040019536,0.163034096706,0.163028174526,0.163022252995,0.163016332114,0.163010411882,0.1630044923,0.162998573367,0.162992655083,0.162986737448,0.162980820462,0.162974904125,0.162968988437,0.162963073396,0.162957159005,0.162951245261,0.162945332166,0.162939419719,0.162933507919,0.162927596768,0.162921686264,0.162915776408,0.162909867199,0.162903958638,0.162898050723,0.162892143456,0.162886236836,0.162880330862,0.162874425536,0.162868520856,0.162862616822,0.162856713435,0.162850810694,0.162844908599,0.16283900715,0.162833106347,0.16282720619,0.162821306679,0.162815407813,0.162809509592,0.162803612017,0.162797715087,0.162791818802,0.162785923162,0.162780028166,0.162774133816,0.16276824011,0.162762347048,0.162756454631,0.162750562858,0.16274467173,0.162738781245,0.162732891404,0.162727002207,0.162721113653,0.162715225743,0.162709338477,0.162703451853,0.162697565873,0.162691680536,0.162685795842,0.162679911791,0.162674028382,0.162668145616,0.162662263492,0.162656382011,0.162650501172,0.162644620975,0.16263874142,0.162632862507,0.162626984236,0.162621106606,0.162615229618,0.162609353271,0.162603477565,0.162597602501,0.162591728078,0.162585854295,0.162579981154,0.162574108653,0.162568236792,0.162562365573,0.162556494993,0.162550625054,0.162544755755,0.162538887095,0.162533019076,0.162527151696,0.162521284956,0.162515418856,0.162509553395,0.162503688573,0.162497824391,0.162491960847,0.162486097942,0.162480235677,0.162474374049,0.162468513061,0.162462652711,0.162456792999,0.162450933926,0.16244507549,0.162439217693,0.162433360533,0.162427504012,0.162421648128,0.162415792881,0.162409938272,0.1624040843,0.162398230965,0.162392378268,0.162386526207,0.162380674783,0.162374823996,0.162368973845,0.162363124331,0.162357275454,0.162351427212,0.162345579607,0.162339732637,0.162333886304,0.162328040606,0.162322195544,0.162316351118,0.162310507327,0.162304664171,0.162298821651,0.162292979765,0.162287138515,0.162281297899,0.162275457919,0.162269618572,0.162263779861,0.162257941783,0.16225210434,0.162246267531,0.162240431356,0.162234595815,0.162228760908,0.162222926635,0.162217092995,0.162211259989,0.162205427615,0.162199595876,0.162193764769,0.162187934295,0.162182104454,0.162176275246,0.162170446671,0.162164618728,0.162158791417,0.162152964739,0.162147138693,0.162141313279,0.162135488497,0.162129664347,0.162123840829,0.162118017942,0.162112195687,0.162106374063,0.16210055307,0.162094732709,0.162088912979,0.162083093879,0.16207727541,0.162071457572,0.162065640365,0.162059823788,0.162054007841,0.162048192525,0.162042377839,0.162036563783,0.162030750356,0.16202493756,0.162019125393,0.162013313855,0.162007502947,0.162001692669,0.161995883019,0.161990073999,0.161984265608,0.161978457845,0.161972650711,0.161966844206,0.161961038329,0.161955233081,0.161949428461,0.161943624469,0.161937821106,0.16193201837,0.161926216262,0.161920414782,0.161914613929,0.161908813704,0.161903014106,0.161897215135,0.161891416792,0.161885619076,0.161879821986,0.161874025524,0.161868229688,0.161862434478,0.161856639895,0.161850845939,0.161845052608,0.161839259904,0.161833467826,0.161827676374,0.161821885547,0.161816095346,0.161810305771,0.161804516821,0.161798728496,0.161792940797,0.161787153723,0.161781367273,0.161775581449,0.161769796249,0.161764011674,0.161758227723,0.161752444397,0.161746661696,0.161740879618,0.161735098164,0.161729317335,0.161723537129,0.161717757547,0.161711978589,0.161706200254,0.161700422542,0.161694645454,0.161688868989,0.161683093147,0.161677317928,0.161671543332,0.161665769358,0.161659996007,0.161654223279,0.161648451173,0.161642679689,0.161636908828,0.161631138588,0.16162536897,0.161619599975,0.1616138316,0.161608063848,0.161602296717,0.161596530207,0.161590764319,0.161584999051,0.161579234405,0.16157347038,0.161567706975,0.161561944191,0.161556182028,0.161550420485,0.161544659563,0.161538899261,0.161533139578,0.161527380516,0.161521622074,0.161515864252,0.161510107049,0.161504350466,0.161498594502,0.161492839158,0.161487084433,0.161481330327,0.16147557684,0.161469823972,0.161464071722,0.161458320092,0.161452569079,0.161446818686,0.16144106891,0.161435319753,0.161429571214,0.161423823293,0.16141807599,0.161412329305,0.161406583237,0.161400837787,0.161395092955,0.161389348739,0.161383605141,0.16137786216,0.161372119796,0.161366378049,0.161360636919,0.161354896405,0.161349156508,0.161343417227,0.161337678563,0.161331940515,0.161326203083,0.161320466266,0.161314730066,0.161308994482,0.161303259513,0.16129752516,0.161291791422,0.1612860583,0.161280325793,0.1612745939,0.161268862623,0.161263131961,0.161257401914,0.161251672481,0.161245943663,0.161240215459,0.161234487869,0.161228760894,0.161223034533,0.161217308786,0.161211583652,0.161205859133,0.161200135227,0.161194411935,0.161188689256,0.16118296719,0.161177245738,0.161171524899,0.161165804673,0.161160085059,0.161154366059,0.161148647671,0.161142929896,0.161137212733,0.161131496183,0.161125780244,0.161120064918,0.161114350204,0.161108636102,0.161102922611,0.161097209733,0.161091497465,0.16108578581,0.161080074765,0.161074364332,0.16106865451,0.161062945299,0.161057236699,0.16105152871,0.161045821331,0.161040114563,0.161034408406,0.161028702858,0.161022997921,0.161017293595,0.161011589878,0.161005886771,0.161000184274,0.160994482387,0.160988781109,0.160983080441,0.160977380382,0.160971680932,0.160965982092,0.16096028386,0.160954586238,0.160948889224,0.160943192819,0.160937497023,0.160931801835,0.160926107255,0.160920413284,0.160914719921,0.160909027166,0.160903335019,0.16089764348,0.160891952549,0.160886262225,0.160880572508,0.1608748834,0.160869194898,0.160863507004,0.160857819716,0.160852133036,0.160846446962,0.160840761496,0.160835076636,0.160829392382,0.160823708735,0.160818025694,0.160812343259,0.160806661431,0.160800980208,0.160795299592,0.160789619581,0.160783940176,0.160778261376,0.160772583182,0.160766905593,0.160761228609,0.160755552231,0.160749876457,0.160744201289,0.160738526725,0.160732852766,0.160727179411,0.160721506661,0.160715834516,0.160710162974,0.160704492037,0.160698821704,0.160693151975,0.160687482849,0.160681814327,0.160676146409,0.160670479094,0.160664812383,0.160659146275,0.16065348077,0.160647815869,0.16064215157,0.160636487874,0.160630824781,0.16062516229,0.160619500402,0.160613839117,0.160608178433,0.160602518352,0.160596858873,0.160591199996,0.160585541721,0.160579884048,0.160574226976,0.160568570506,0.160562914638,0.16055725937,0.160551604704,0.160545950639,0.160540297175,0.160534644312,0.16052899205,0.160523340389,0.160517689328,0.160512038867,0.160506389007,0.160500739747,0.160495091088,0.160489443028,0.160483795569,0.160478148709,0.160472502449,0.160466856789,0.160461211728,0.160455567266,0.160449923404,0.160444280141,0.160438637477,0.160432995412,0.160427353946,0.160421713079,0.160416072811,0.160410433141,0.160404794069,0.160399155596,0.160393517721,0.160387880444,0.160382243765,0.160376607685,0.160370972201,0.160365337316,0.160359703028,0.160354069338,0.160348436245,0.16034280375,0.160337171851,0.16033154055,0.160325909845,0.160320279738,0.160314650227,0.160309021313,0.160303392995,0.160297765274,0.160292138149,0.160286511621,0.160280885688,0.160275260352,0.160269635611,0.160264011466,0.160258387917,0.160252764963,0.160247142605,0.160241520843,0.160235899675,0.160230279103,0.160224659126,0.160219039743,0.160213420956,0.160207802763,0.160202185165,0.160196568161,0.160190951752,0.160185335937,0.160179720717,0.16017410609,0.160168492058,0.160162878619,0.160157265774,0.160151653523,0.160146041865,0.160140430801,0.16013482033,0.160129210453,0.160123601168,0.160117992477,0.160112384378,0.160106776873,0.16010116996,0.16009556364,0.160089957912,0.160084352777,0.160078748234,0.160073144283,0.160067540924,0.160061938157,0.160056335982,0.160050734399,0.160045133408,0.160039533008,0.1600339332,0.160028333983,0.160022735357,0.160017137323,0.160011539879,0.160005943026,0.160000346765,0.159994751094,0.159989156013,0.159983561523,0.159977967624,0.159972374315,0.159966781596,0.159961189467,0.159955597928,0.159950006979,0.15994441662,0.159938826851,0.159933237671,0.15992764908,0.159922061079,0.159916473668,0.159910886845,0.159905300611,0.159899714967,0.159894129911,0.159888545444,0.159882961566,0.159877378276,0.159871795574,0.159866213461,0.159860631937,0.159855051,0.159849470651,0.15984389089,0.159838311717,0.159832733132,0.159827155134,0.159821577724,0.159816000901,0.159810424666,0.159804849018,0.159799273956,0.159793699482,0.159788125595,0.159782552294,0.15977697958,0.159771407453,0.159765835912,0.159760264957,0.159754694589,0.159749124807,0.159743555611,0.159737987,0.159732418976,0.159726851537,0.159721284684,0.159715718417,0.159710152735,0.159704587638,0.159699023126,0.1596934592,0.159687895859,0.159682333102,0.15967677093,0.159671209343,0.159665648341,0.159660087923,0.15965452809,0.15964896884,0.159643410175,0.159637852094,0.159632294597,0.159626737684,0.159621181355,0.159615625609,0.159610070447,0.159604515868,0.159598961873,0.159593408461,0.159587855632,0.159582303386,0.159576751723,0.159571200643,0.159565650145,0.159560100231,0.159554550899,0.159549002149,0.159543453981,0.159537906396,0.159532359393,0.159526812972,0.159521267133,0.159515721876,0.1595101772,0.159504633106,0.159499089593,0.159493546662,0.159488004313,0.159482462544,0.159476921357,0.15947138075,0.159465840724,0.15946030128,0.159454762416,0.159449224132,0.159443686429,0.159438149306,0.159432612764,0.159427076802,0.15942154142,0.159416006617,0.159410472395,0.159404938753,0.15939940569,0.159393873207,0.159388341303,0.159382809978,0.159377279233,0.159371749067,0.15936621948,0.159360690472,0.159355162043,0.159349634192,0.159344106921,0.159338580227,0.159333054113,0.159327528576,0.159322003618,0.159316479238,0.159310955436,0.159305432212,0.159299909566,0.159294387497,0.159288866006,0.159283345093,0.159277824757,0.159272304998,0.159266785817,0.159261267213,0.159255749185,0.159250231735,0.159244714862,0.159239198565,0.159233682845,0.159228167701,0.159222653134,0.159217139143,0.159211625728,0.15920611289,0.159200600627,0.15919508894,0.15918957783,0.159184067294,0.159178557335,0.159173047951,0.159167539142,0.159162030909,0.159156523251,0.159151016168,0.15914550966,0.159140003727,0.159134498368,0.159128993585,0.159123489376,0.159117985741,0.159112482681,0.159106980195,0.159101478283,0.159095976946,0.159090476182,0.159084975993,0.159079476377,0.159073977335,0.159068478866,0.159062980971,0.159057483649,0.159051986901,0.159046490725,0.159040995123,0.159035500094,0.159030005638,0.159024511754,0.159019018444,0.159013525705,0.15900803354,0.159002541946,0.158997050925,0.158991560476,0.1589860706,0.158980581295,0.158975092562,0.158969604401,0.158964116811,0.158958629794,0.158953143347,0.158947657472,0.158942172169,0.158936687436,0.158931203275,0.158925719684,0.158920236665,0.158914754216,0.158909272338,0.15890379103,0.158898310293,0.158892830127,0.15888735053,0.158881871504,0.158876393048,0.158870915162,0.158865437846,0.158859961099,0.158854484923,0.158849009316,0.158843534278,0.15883805981,0.158832585911,0.158827112581,0.15882163982,0.158816167629,0.158810696006,0.158805224952,0.158799754467,0.15879428455,0.158788815202,0.158783346422,0.158777878211,0.158772410567,0.158766943492,0.158761476985,0.158756011046,0.158750545674,0.15874508087,0.158739616634,0.158734152965,0.158728689864,0.15872322733,0.158717765363,0.158712303964,0.158706843131,0.158701382865,0.158695923166,0.158690464034,0.158685005469,0.15867954747,0.158674090037,0.158668633171,0.15866317687,0.158657721136,0.158652265969,0.158646811366,0.15864135733,0.15863590386,0.158630450955,0.158624998616,0.158619546842,0.158614095633,0.15860864499,0.158603194912,0.158597745398,0.15859229645,0.158586848067,0.158581400248,0.158575952994,0.158570506305,0.15856506018,0.158559614619,0.158554169623,0.15854872519,0.158543281322,0.158537838018,0.158532395277,0.158526953101,0.158521511488,0.158516070438,0.158510629952,0.15850519003,0.158499750671,0.158494311874,0.158488873641,0.158483435971,0.158477998864,0.15847256232,0.158467126338,0.158461690919,0.158456256062,0.158450821768,0.158445388036,0.158439954866,0.158434522258,0.158429090213,0.158423658729,0.158418227807,0.158412797447,0.158407367648,0.158401938411,0.158396509735,0.158391081621,0.158385654067,0.158380227075,0.158374800644,0.158369374774,0.158363949465,0.158358524716,0.158353100528,0.158347676901,0.158342253834,0.158336831327,0.158331409381,0.158325987994,0.158320567168,0.158315146902,0.158309727195,0.158304308049,0.158298889462,0.158293471434,0.158288053966,0.158282637058,0.158277220709,0.158271804918,0.158266389687,0.158260975015,0.158255560902,0.158250147348,0.158244734352,0.158239321915,0.158233910036,0.158228498716,0.158223087954,0.15821767775,0.158212268105,0.158206859017,0.158201450488,0.158196042516,0.158190635102,0.158185228245,0.158179821946,0.158174416205,0.158169011021,0.158163606394,0.158158202324,0.158152798811,0.158147395855,0.158141993456,0.158136591614,0.158131190329,0.1581257896,0.158120389427,0.158114989811,0.158109590751,0.158104192248,0.1580987943,0.158093396909,0.158088000073,0.158082603793,0.158077208069,0.1580718129,0.158066418287,0.158061024229,0.158055630727,0.15805023778,0.158044845388,0.158039453551,0.158034062269,0.158028671542,0.158023281369,0.158017891752,0.158012502688,0.158007114179,0.158001726225,0.157996338825,0.157990951979,0.157985565687,0.157980179949,0.157974794765,0.157969410134,0.157964026058,0.157958642535,0.157953259565,0.157947877149,0.157942495286,0.157937113977,0.15793173322,0.157926353017,0.157920973366,0.157915594268,0.157910215723,0.157904837731,0.157899460291,0.157894083404,0.157888707069,0.157883331286,0.157877956055,0.157872581377,0.15786720725,0.157861833675,0.157856460652,0.157851088181,0.157845716261,0.157840344893,0.157834974076,0.157829603811,0.157824234097,0.157818864934,0.157813496322,0.15780812826,0.15780276075,0.15779739379,0.157792027381,0.157786661523,0.157781296215,0.157775931457,0.15777056725,0.157765203593,0.157759840485,0.157754477928,0.157749115921,0.157743754463,0.157738393556,0.157733033197,0.157727673389,0.157722314129,0.157716955419,0.157711597259,0.157706239647,0.157700882584,0.157695526071,0.157690170106,0.15768481469,0.157679459822,0.157674105503,0.157668751733,0.157663398511,0.157658045837,0.157652693711,0.157647342134,0.157641991104,0.157636640623,0.157631290689,0.157625941303,0.157620592464,0.157615244173,0.15760989643,0.157604549233,0.157599202584,0.157593856483,0.157588510928,0.15758316592,0.157577821459,0.157572477545,0.157567134177,0.157561791356,0.157556449082,0.157551107354,0.157545766172,0.157540425536,0.157535085447,0.157529745904,0.157524406906,0.157519068454,0.157513730548,0.157508393188,0.157503056373,0.157497720104,0.15749238438,0.157487049201,0.157481714568,0.15747638048,0.157471046936,0.157465713938,0.157460381484,0.157455049575,0.15744971821,0.157444387391,0.157439057115,0.157433727384,0.157428398197,0.157423069554,0.157417741456,0.157412413901,0.15740708689,0.157401760423,0.1573964345,0.15739110912,0.157385784283,0.15738045999,0.157375136241,0.157369813034,0.157364490371,0.157359168251,0.157353846674,0.157348525639,0.157343205147,0.157337885198,0.157332565792,0.157327246928,0.157321928606,0.157316610827,0.15731129359,0.157305976895,0.157300660742,0.15729534513,0.157290030061,0.157284715533,0.157279401547,0.157274088103,0.1572687752,0.157263462838,0.157258151018,0.157252839739,0.157247529001,0.157242218804,0.157236909148,0.157231600032,0.157226291457,0.157220983423,0.15721567593,0.157210368977,0.157205062564,0.157199756691,0.157194451359,0.157189146567,0.157183842314,0.157178538602,0.157173235429,0.157167932796,0.157162630703,0.157157329149,0.157152028135,0.157146727659,0.157141427724,0.157136128327,0.157130829469,0.15712553115,0.15712023337,0.157114936129,0.157109639427,0.157104343263,0.157099047638,0.157093752551,0.157088458002,0.157083163992,0.15707787052,0.157072577585,0.157067285189,0.157061993331,0.15705670201,0.157051411227,0.157046120982,0.157040831274,0.157035542103,0.15703025347,0.157024965374,0.157019677815,0.157014390793,0.157009104308,0.15700381836,0.156998532949,0.156993248074,0.156987963736,0.156982679934,0.156977396669,0.15697211394,0.156966831747,0.156961550091,0.15695626897,0.156950988386,0.156945708337,0.156940428824,0.156935149846,0.156929871405,0.156924593498,0.156919316127,0.156914039292,0.156908762991,0.156903487226,0.156898211995,0.1568929373,0.156887663139,0.156882389513,0.156877116422,0.156871843866,0.156866571844,0.156861300356,0.156856029402,0.156850758983,0.156845489098,0.156840219747,0.156834950929,0.156829682646,0.156824414896,0.156819147681,0.156813880998,0.156808614849,0.156803349234,0.156798084151,0.156792819602,0.156787555586,0.156782292103,0.156777029153,0.156771766736,0.156766504852,0.1567612435,0.156755982681,0.156750722394,0.15674546264,0.156740203418,0.156734944728,0.15672968657,0.156724428945,0.156719171851,0.156713915289,0.156708659259,0.15670340376,0.156698148794,0.156692894358,0.156687640454,0.156682387081,0.15667713424,0.15667188193,0.15666663015,0.156661378902,0.156656128184,0.156650877997,0.156645628341,0.156640379216,0.156635130621,0.156629882556,0.156624635022,0.156619388018,0.156614141544,0.1566088956,0.156603650186,0.156598405302,0.156593160947,0.156587917123,0.156582673828,0.156577431062,0.156572188826,0.156566947119,0.156561705941,0.156556465293,0.156551225174,0.156545985583,0.156540746522,0.156535507989,0.156530269985,0.156525032509,0.156519795562,0.156514559144,0.156509323254,0.156504087892,0.156498853058,0.156493618752,0.156488384974,0.156483151724,0.156477919002,0.156472686808,0.156467455141,0.156462224002,0.156456993391,0.156451763306,0.156446533749,0.156441304719,0.156436076216,0.15643084824,0.156425620791,0.156420393869,0.156415167474,0.156409941605,0.156404716263,0.156399491447,0.156394267158,0.156389043395,0.156383820158,0.156378597447,0.156373375262,0.156368153604,0.156362932471,0.156357711863,0.156352491782,0.156347272226,0.156342053195,0.15633683469,0.15633161671,0.156326399256,0.156321182326,0.156315965922,0.156310750043,0.156305534688,0.156300319858,0.156295105553,0.156289891772,0.156284678516,0.156279465785,0.156274253578,0.156269041895,0.156263830736,0.156258620101,0.15625340999,0.156248200403,0.15624299134,0.156237782801,0.156232574785,0.156227367292,0.156222160324,0.156216953878,0.156211747956,0.156206542557,0.156201337681,0.156196133328,0.156190929498,0.156185726191,0.156180523407,0.156175321145,0.156170119406,0.156164918189,0.156159717495,0.156154517323,0.156149317673,0.156144118545,0.15613891994,0.156133721856,0.156128524294,0.156123327254,0.156118130736,0.156112934739,0.156107739264,0.15610254431,0.156097349877,0.156092155966,0.156086962576,0.156081769707,0.156076577359,0.156071385532,0.156066194225,0.15606100344,0.156055813175,0.15605062343,0.156045434206,0.156040245503,0.156035057319,0.156029869656,0.156024682513,0.15601949589,0.156014309787,0.156009124204,0.15600393914,0.155998754596,0.155993570572,0.155988387067,0.155983204082,0.155978021616,0.155972839669,0.155967658242,0.155962477333,0.155957296943,0.155952117073,0.155946937721,0.155941758887,0.155936580573,0.155931402777,0.155926225499,0.15592104874,0.155915872499,0.155910696776,0.155905521571,0.155900346885,0.155895172716,0.155889999065,0.155884825932,0.155879653316,0.155874481218,0.155869309638,0.155864138575,0.155858968029,0.155853798001,0.155848628489,0.155843459495,0.155838291018,0.155833123057,0.155827955614,0.155822788687,0.155817622276,0.155812456383,0.155807291005,0.155802126144,0.1557969618,0.155791797971,0.155786634659,0.155781471862,0.155776309582,0.155771147817,0.155765986568,0.155760825835,0.155755665618,0.155750505915,0.155745346729,0.155740188057,0.155735029901,0.15572987226,0.155724715134,0.155719558523,0.155714402427,0.155709246846,0.155704091779,0.155698937228,0.15569378319,0.155688629667,0.155683476659,0.155678324165,0.155673172185,0.155668020719,0.155662869767,0.155657719329,0.155652569405,0.155647419994,0.155642271098,0.155637122715,0.155631974845,0.155626827489,0.155621680647,0.155616534317,0.155611388501,0.155606243198,0.155601098407,0.15559595413,0.155590810366,0.155585667114,0.155580524375,0.155575382148,0.155570240434,0.155565099233,0.155559958543,0.155554818366,0.155549678701,0.155544539548,0.155539400907,0.155534262778,0.155529125161,0.155523988055,0.155518851462,0.155513715379,0.155508579808,0.155503444749,0.155498310201,0.155493176163,0.155488042638,0.155482909623,0.155477777119,0.155472645126,0.155467513643,0.155462382672,0.155457252211,0.15545212226,0.15544699282,0.15544186389,0.155436735471,0.155431607561,0.155426480162,0.155421353273,0.155416226894,0.155411101024,0.155405975665,0.155400850815,0.155395726474,0.155390602643,0.155385479322,0.155380356509,0.155375234206,0.155370112413,0.155364991128,0.155359870352,0.155354750085,0.155349630327,0.155344511078,0.155339392337,0.155334274105,0.155329156381,0.155324039166,0.155318922459,0.155313806261,0.15530869057,0.155303575388,0.155298460713,0.155293346546,0.155288232887,0.155283119736,0.155278007093,0.155272894957,0.155267783328,0.155262672207,0.155257561593,0.155252451487,0.155247341887,0.155242232795,0.155237124209,0.155232016131,0.155226908559,0.155221801494,0.155216694935,0.155211588883,0.155206483337,0.155201378298,0.155196273765,0.155191169739,0.155186066218,0.155180963203,0.155175860695,0.155170758692,0.155165657195,0.155160556204,0.155155455718,0.155150355738,0.155145256263,0.155140157294,0.155135058829,0.155129960871,0.155124863417,0.155119766468,0.155114670024,0.155109574085,0.155104478651,0.155099383721,0.155094289296,0.155089195376,0.15508410196,0.155079009048,0.15507391664,0.155068824737,0.155063733338,0.155058642443,0.155053552052,0.155048462164,0.155043372781,0.155038283901,0.155033195524,0.155028107652,0.155023020282,0.155017933416,0.155012847053,0.155007761194,0.155002675837,0.154997590984,0.154992506633,0.154987422785,0.15498233944,0.154977256598,0.154972174258,0.154967092421,0.154962011086,0.154956930254,0.154951849923,0.154946770095,0.154941690769,0.154936611945,0.154931533623,0.154926455803,0.154921378485,0.154916301668,0.154911225353,0.154906149539,0.154901074227,0.154895999416,0.154890925107,0.154885851298,0.154880777991,0.154875705184,0.154870632879,0.154865561075,0.154860489771,0.154855418968,0.154850348665,0.154845278863,0.154840209561,0.15483514076,0.154830072459,0.154825004658,0.154819937358,0.154814870557,0.154809804256,0.154804738455,0.154799673154,0.154794608353,0.154789544051,0.154784480249,0.154779416946,0.154774354142,0.154769291838,0.154764230033,0.154759168727,0.15475410792,0.154749047612,0.154743987803,0.154738928493,0.154733869681,0.154728811368,0.154723753553,0.154718696237,0.154713639419,0.1547085831,0.154703527279,0.154698471956,0.154693417131,0.154688362803,0.154683308974,0.154678255643,0.154673202809,0.154668150473,0.154663098634,0.154658047293,0.154652996449,0.154647946103,0.154642896253,0.154637846901,0.154632798046,0.154627749688,0.154622701827,0.154617654462,0.154612607594,0.154607561223,0.154602515349,0.154597469971,0.154592425089,0.154587380704,0.154582336814,0.154577293421,0.154572250525,0.154567208124,0.154562166219,0.154557124809,0.154552083896,0.154547043478,0.154542003556,0.154536964129,0.154531925197,0.154526886761,0.154521848821,0.154516811375,0.154511774424,0.154506737969,0.154501702008,0.154496666542,0.154491631571,0.154486597095,0.154481563113,0.154476529626,0.154471496633,0.154466464135,0.15446143213,0.15445640062,0.154451369605,0.154446339083,0.154441309055,0.154436279521,0.15443125048,0.154426221934,0.154421193881,0.154416166321,0.154411139255,0.154406112682,0.154401086603,0.154396061017,0.154391035924,0.154386011324,0.154380987217,0.154375963603,0.154370940481,0.154365917853,0.154360895717,0.154355874073,0.154350852923,0.154345832264,0.154340812098,0.154335792424,0.154330773242,0.154325754552,0.154320736355,0.154315718649,0.154310701435,0.154305684713,0.154300668482,0.154295652743,0.154290637496,0.154285622739,0.154280608475,0.154275594701,0.154270581419,0.154265568628,0.154260556328,0.154255544519,0.1542505332,0.154245522373,0.154240512036,0.15423550219,0.154230492834,0.154225483969,0.154220475594,0.154215467709,0.154210460315,0.154205453411,0.154200446997,0.154195441073,0.154190435638,0.154185430694,0.154180426239,0.154175422274,0.154170418799,0.154165415813,0.154160413316,0.154155411309,0.154150409791,0.154145408762,0.154140408223,0.154135408172,0.15413040861,0.154125409537,0.154120410953,0.154115412858,0.154110415251,0.154105418133,0.154100421503,0.154095425361,0.154090429708,0.154085434543,0.154080439866,0.154075445677,0.154070451977,0.154065458764,0.154060466038,0.154055473801,0.154050482051,0.154045490789,0.154040500015,0.154035509727,0.154030519927,0.154025530615,0.154020541789,0.154015553451,0.154010565599,0.154005578235,0.154000591357,0.153995604966,0.153990619062,0.153985633645,0.153980648714,0.153975664269,0.153970680311,0.153965696839,0.153960713853,0.153955731354,0.15395074934,0.153945767813,0.153940786771,0.153935806215,0.153930826145,0.153925846561,0.153920867462,0.153915888849,0.153910910721,0.153905933079,0.153900955921,0.153895979249,0.153891003062,0.15388602736,0.153881052143,0.153876077411,0.153871103164,0.153866129401,0.153861156123,0.153856183329,0.15385121102,0.153846239196,0.153841267855,0.153836296999,0.153831326627,0.15382635674,0.153821387336,0.153816418416,0.15381144998,0.153806482027,0.153801514559,0.153796547574,0.153791581072,0.153786615054,0.153781649519,0.153776684468,0.153771719899,0.153766755814,0.153761792212,0.153756829093,0.153751866457,0.153746904303,0.153741942633,0.153736981445,0.153732020739,0.153727060516,0.153722100776,0.153717141517,0.153712182741,0.153707224448,0.153702266636,0.153697309306,0.153692352459,0.153687396093,0.153682440209,0.153677484806,0.153672529886,0.153667575447,0.153662621489,0.153657668013,0.153652715018,0.153647762504,0.153642810471,0.15363785892,0.153632907849,0.15362795726,0.153623007151,0.153618057523,0.153613108376,0.153608159709,0.153603211523,0.153598263817,0.153593316592,0.153588369847,0.153583423582,0.153578477798,0.153573532493,0.153568587668,0.153563643324,0.153558699459,0.153553756074,0.153548813168,0.153543870743,0.153538928796,0.153533987329,0.153529046342,0.153524105834,0.153519165805,0.153514226255,0.153509287184,0.153504348592,0.153499410479,0.153494472845,0.15348953569,0.153484599013,0.153479662815,0.153474727095,0.153469791854,0.153464857091,0.153459922806,0.153454989,0.153450055671,0.153445122821,0.153440190449,0.153435258554,0.153430327138,0.153425396199,0.153420465737,0.153415535754,0.153410606248,0.153405677219,0.153400748667,0.153395820593,0.153390892996,0.153385965876,0.153381039233,0.153376113067,0.153371187378,0.153366262166,0.153361337431,0.153356413172,0.153351489389,0.153346566084,0.153341643254,0.153336720901,0.153331799024,0.153326877624,0.153321956699,0.153317036251,0.153312116278,0.153307196781,0.15330227776,0.153297359215,0.153292441146,0.153287523551,0.153282606433,0.15327768979,0.153272773622,0.153267857929,0.153262942712,0.15325802797,0.153253113703,0.15324819991,0.153243286593,0.15323837375,0.153233461382,0.153228549489,0.15322363807,0.153218727126,0.153213816656,0.153208906661,0.153203997139,0.153199088092,0.153194179519,0.15318927142,0.153184363795,0.153179456644,0.153174549967,0.153169643763,0.153164738033,0.153159832776,0.153154927993,0.153150023684,0.153145119848,0.153140216485,0.153135313595,0.153130411178,0.153125509234,0.153120607764,0.153115706766,0.153110806241,0.153105906188,0.153101006609,0.153096107501,0.153091208867,0.153086310705,0.153081413015,0.153076515797,0.153071619051,0.153066722778,0.153061826977,0.153056931647,0.15305203679,0.153047142404,0.15304224849,0.153037355048,0.153032462077,0.153027569578,0.15302267755,0.153017785994,0.153012894908,0.153008004294,0.153003114151,0.15299822448,0.152993335279,0.152988446549,0.152983558289,0.152978670501,0.152973783183,0.152968896336,0.152964009959,0.152959124053,0.152954238617,0.152949353651,0.152944469156,0.15293958513,0.152934701575,0.15292981849,0.152924935875,0.152920053729,0.152915172053,0.152910290847,0.15290541011,0.152900529843,0.152895650046,0.152890770718,0.152885891859,0.152881013469,0.152876135548,0.152871258097,0.152866381114,0.152861504601,0.152856628556,0.15285175298,0.152846877873,0.152842003234,0.152837129064,0.152832255362,0.152827382129,0.152822509364,0.152817637067,0.152812765239,0.152807893878,0.152803022986,0.152798152561,0.152793282604,0.152788413116,0.152783544094,0.152778675541,0.152773807455,0.152768939836,0.152764072685,0.152759206001,0.152754339785,0.152749474035,0.152744608753,0.152739743938,0.15273487959,0.152730015708,0.152725152294,0.152720289346,0.152715426865,0.15271056485,0.152705703302,0.15270084222,0.152695981605,0.152691121456,0.152686261773,0.152681402556,0.152676543806,0.152671685521,0.152666827702,0.152661970349,0.152657113462,0.152652257041,0.152647401085,0.152642545594,0.15263769057,0.15263283601,0.152627981916,0.152623128287,0.152618275123,0.152613422424,0.15260857019,0.152603718422,0.152598867118,0.152594016279,0.152589165904,0.152584315994,0.152579466549,0.152574617568,0.152569769052,0.152564921,0.152560073412,0.152555226288,0.152550379629,0.152545533433,0.152540687702,0.152535842434,0.152530997631,0.152526153291,0.152521309414,0.152516466001,0.152511623052,0.152506780566,0.152501938544,0.152497096985,0.152492255889,0.152487415256,0.152482575086,0.152477735379,0.152472896136,0.152468057355,0.152463219036,0.152458381181,0.152453543788,0.152448706858,0.15244387039,0.152439034384,0.152434198841,0.15242936376,0.152424529142,0.152419694985,0.152414861291,0.152410028058,0.152405195287,0.152400362978,0.152395531131,0.152390699746,0.152385868822,0.152381038359,0.152376208358,0.152371378819,0.15236654974,0.152361721123,0.152356892967,0.152352065273,0.152347238039,0.152342411266,0.152337584954,0.152332759102,0.152327933712,0.152323108782,0.152318284312,0.152313460303,0.152308636755,0.152303813666,0.152298991038,0.152294168871,0.152289347163,0.152284525915,0.152279705128,0.1522748848,0.152270064932,0.152265245524,0.152260426575,0.152255608086,0.152250790057,0.152245972487,0.152241155376,0.152236338725,0.152231522533,0.1522267068,0.152221891526,0.152217076711,0.152212262355,0.152207448458,0.15220263502,0.152197822041,0.15219300952,0.152188197457,0.152183385854,0.152178574708,0.152173764021,0.152168953792,0.152164144022,0.152159334709,0.152154525855,0.152149717459,0.15214490952,0.152140102039,0.152135295017,0.152130488451,0.152125682344,0.152120876694,0.152116071501,0.152111266766,0.152106462488,0.152101658668,0.152096855304,0.152092052398,0.152087249949,0.152082447957,0.152077646421,0.152072845343,0.152068044721,0.152063244556,0.152058444847,0.152053645595,0.152048846799,0.15204404846,0.152039250577,0.15203445315,0.15202965618,0.152024859666,0.152020063607,0.152015268005,0.152010472858,0.152005678167,0.152000883932,0.151996090153,0.151991296829,0.15198650396,0.151981711548,0.15197691959,0.151972128088,0.151967337041,0.151962546449,0.151957756312,0.15195296663,0.151948177403,0.151943388631,0.151938600314,0.151933812451,0.151929025043,0.15192423809,0.151919451591,0.151914665547,0.151909879957,0.151905094821,0.151900310139,0.151895525912,0.151890742138,0.151885958819,0.151881175953,0.151876393542,0.151871611584,0.151866830079,0.151862049029,0.151857268432,0.151852488288,0.151847708598,0.151842929361,0.151838150578,0.151833372247,0.15182859437,0.151823816946,0.151819039975,0.151814263456,0.151809487391,0.151804711778,0.151799936618,0.151795161911,0.151790387656,0.151785613853,0.151780840503,0.151776067606,0.15177129516,0.151766523167,0.151761751626,0.151756980537,0.1517522099,0.151747439714,0.151742669981,0.151737900699,0.151733131869,0.151728363491,0.151723595564,0.151718828089,0.151714061065,0.151709294492,0.151704528371,0.151699762701,0.151694997482,0.151690232714,0.151685468397,0.15168070453,0.151675941115,0.15167117815,0.151666415637,0.151661653573,0.15165689196,0.151652130798,0.151647370086,0.151642609825,0.151637850013,0.151633090652,0.151628331741,0.15162357328,0.151618815269,0.151614057708,0.151609300597,0.151604543935,0.151599787723,0.151595031961,0.151590276648,0.151585521785,0.151580767371,0.151576013406,0.151571259891,0.151566506825,0.151561754208,0.15155700204,0.151552250321,0.151547499051,0.151542748229,0.151537997857,0.151533247933,0.151528498458,0.151523749431,0.151519000853,0.151514252723,0.151509505041,0.151504757808,0.151500011023,0.151495264686,0.151490518797,0.151485773356,0.151481028363,0.151476283818,0.151471539721,0.151466796071,0.151462052869,0.151457310114,0.151452567807,0.151447825948,0.151443084535,0.15143834357,0.151433603052,0.151428862982,0.151424123358,0.151419384181,0.151414645452,0.151409907169,0.151405169332,0.151400431943,0.151395695,0.151390958504,0.151386222454,0.151381486851,0.151376751694,0.151372016983,0.151367282718,0.1513625489,0.151357815527,0.151353082601,0.151348350121,0.151343618086,0.151338886497,0.151334155354,0.151329424657,0.151324694405,0.151319964599,0.151315235238,0.151310506322,0.151305777852,0.151301049827,0.151296322247,0.151291595112,0.151286868422,0.151282142177,0.151277416377,0.151272691022,0.151267966112,0.151263241646,0.151258517625,0.151253794048,0.151249070916,0.151244348228,0.151239625984,0.151234904185,0.15123018283,0.151225461919,0.151220741452,0.151216021429,0.15121130185,0.151206582714,0.151201864023,0.151197145775,0.151192427971,0.15118771061,0.151182993692,0.151178277219,0.151173561188,0.151168845601,0.151164130457,0.151159415756,0.151154701498,0.151149987683,0.151145274311,0.151140561381,0.151135848895,0.151131136851,0.15112642525,0.151121714091,0.151117003375,0.151112293102,0.15110758327,0.151102873881,0.151098164935,0.15109345643,0.151088748367,0.151084040747,0.151079333568,0.151074626832,0.151069920537,0.151065214683,0.151060509272,0.151055804302,0.151051099773,0.151046395686,0.151041692041,0.151036988836,0.151032286073,0.151027583751,0.15102288187,0.151018180431,0.151013479432,0.151008778874,0.151004078757,0.15099937908,0.150994679845,0.15098998105,0.150985282695,0.150980584781,0.150975887308,0.150971190274,0.150966493681,0.150961797529,0.150957101816,0.150952406543,0.150947711711,0.150943017318,0.150938323365,0.150933629852,0.150928936779,0.150924244145,0.150919551951,0.150914860196,0.150910168881,0.150905478006,0.150900787569,0.150896097572,0.150891408014,0.150886718895,0.150882030215,0.150877341974,0.150872654172,0.150867966809,0.150863279884,0.150858593398,0.150853907351,0.150849221742,0.150844536572,0.15083985184,0.150835167547,0.150830483692,0.150825800275,0.150821117296,0.150816434755,0.150811752652,0.150807070988,0.150802389761,0.150797708971,0.15079302862,0.150788348706,0.15078366923,0.150778990191,0.15077431159,0.150769633426,0.150764955699,0.15076027841,0.150755601558,0.150750925143,0.150746249165,0.150741573624,0.15073689852,0.150732223852,0.150727549622,0.150722875828,0.150718202471,0.15071352955,0.150708857066,0.150704185018,0.150699513406,0.150694842231,0.150690171492,0.15068550119,0.150680831323,0.150676161892,0.150671492897,0.150666824339,0.150662156216,0.150657488528,0.150652821277,0.15064815446,0.15064348808,0.150638822135,0.150634156625,0.150629491551,0.150624826912,0.150620162708,0.150615498939,0.150610835606,0.150606172707,0.150601510243,0.150596848214,0.15059218662,0.150587525461,0.150582864736,0.150578204446,0.15057354459,0.150568885169,0.150564226182,0.150559567629,0.150554909511,0.150550251827,0.150545594577,0.150540937761,0.150536281379,0.150531625431,0.150526969916,0.150522314836,0.150517660189,0.150513005976,0.150508352196,0.15050369885,0.150499045937,0.150494393458,0.150489741412,0.150485089799,0.15048043862,0.150475787873,0.150471137559,0.150466487679,0.150461838231,0.150457189216,0.150452540634,0.150447892485,0.150443244768,0.150438597484,0.150433950632,0.150429304213,0.150424658225,0.150420012671,0.150415367548,0.150410722858,0.1504060786,0.150401434773,0.150396791379,0.150392148416,0.150387505886,0.150382863787,0.15037822212,0.150373580884,0.15036894008,0.150364299707,0.150359659766,0.150355020256,0.150350381177,0.15034574253,0.150341104314,0.150336466529,0.150331829174,0.150327192251,0.150322555759,0.150317919697,0.150313284066,0.150308648866,0.150304014097,0.150299379757,0.150294745849,0.150290112371,0.150285479323,0.150280846705,0.150276214518,0.150271582761,0.150266951433,0.150262320536,0.150257690069,0.150253060031,0.150248430424,0.150243801246,0.150239172498,0.150234544179,0.15022991629,0.15022528883,0.1502206618,0.150216035199,0.150211409027,0.150206783285,0.150202157972,0.150197533087,0.150192908632,0.150188284606,0.150183661008,0.15017903784,0.1501744151,0.150169792788,0.150165170906,0.150160549451,0.150155928426,0.150151307828,0.150146687659,0.150142067919,0.150137448606,0.150132829722,0.150128211265,0.150123593237,0.150118975637,0.150114358464,0.150109741719,0.150105125402,0.150100509513,0.150095894051,0.150091279017,0.15008666441,0.150082050231,0.150077436479,0.150072823154,0.150068210256,0.150063597786,0.150058985743,0.150054374126,0.150049762937,0.150045152174,0.150040541838,0.150035931929,0.150031322447,0.150026713391,0.150022104762,0.150017496559,0.150012888783,0.150008281433,0.150003674509,0.149999068012,0.14999446194,0.149989856295,0.149985251076,0.149980646282,0.149976041915,0.149971437973,0.149966834457,0.149962231367,0.149957628702,0.149953026463,0.14994842465,0.149943823262,0.149939222299,0.149934621761,0.149930021649,0.149925421962,0.149920822699,0.149916223862,0.14991162545,0.149907027463,0.149902429901,0.149897832763,0.14989323605,0.149888639762,0.149884043898,0.149879448458,0.149874853444,0.149870258853,0.149865664687,0.149861070945,0.149856477627,0.149851884733,0.149847292264,0.149842700218,0.149838108596,0.149833517398,0.149828926624,0.149824336273,0.149819746347,0.149815156843,0.149810567763,0.149805979107,0.149801390874,0.149796803065,0.149792215678,0.149787628715,0.149783042175,0.149778456058,0.149773870364,0.149769285093,0.149764700244,0.149760115819,0.149755531816,0.149750948236,0.149746365078,0.149741782343,0.149737200031,0.149732618141,0.149728036673,0.149723455627,0.149718875004,0.149714294803,0.149709715024,0.149705135667,0.149700556732,0.149695978218,0.149691400127,0.149686822457,0.149682245209,0.149677668383,0.149673091978,0.149668515994,0.149663940432,0.149659365292,0.149654790572,0.149650216274,0.149645642397,0.149641068941,0.149636495907,0.149631923293,0.1496273511,0.149622779328,0.149618207976,0.149613637046,0.149609066536,0.149604496446,0.149599926777,0.149595357529,0.149590788701,0.149586220293,0.149581652306,0.149577084738,0.149572517591,0.149567950864,0.149563384557,0.14955881867,0.149554253202,0.149549688155,0.149545123527,0.149540559319,0.14953599553,0.149531432161,0.149526869212,0.149522306681,0.149517744571,0.149513182879,0.149508621607,0.149504060754,0.14949950032,0.149494940305,0.149490380709,0.149485821532,0.149481262773,0.149476704434,0.149472146513,0.149467589011,0.149463031927,0.149458475262,0.149453919015,0.149449363187,0.149444807777,0.149440252785,0.149435698212,0.149431144056,0.149426590319,0.149422036999,0.149417484098,0.149412931614,0.149408379548,0.1494038279,0.14939927667,0.149394725857,0.149390175462,0.149385625484,0.149381075924,0.149376526781,0.149371978055,0.149367429747,0.149362881855,0.149358334381,0.149353787324,0.149349240684,0.149344694461,0.149340148654,0.149335603264,0.149331058291,0.149326513735,0.149321969595,0.149317425872,0.149312882565,0.149308339675,0.149303797201,0.149299255143,0.149294713501,0.149290172276,0.149285631466,0.149281091073,0.149276551095,0.149272011534,0.149267472388,0.149262933658,0.149258395344,0.149253857445,0.149249319962,0.149244782894,0.149240246242,0.149235710005,0.149231174184,0.149226638777,0.149222103786,0.14921756921,0.149213035049,0.149208501303,0.149203967972,0.149199435056,0.149194902555,0.149190370468,0.149185838796,0.149181307539,0.149176776696,0.149172246267,0.149167716254,0.149163186654,0.149158657469,0.149154128698,0.149149600341,0.149145072398,0.149140544869,0.149136017754,0.149131491053,0.149126964766,0.149122438893,0.149117913433,0.149113388387,0.149108863755,0.149104339536,0.149099815731,0.149095292339,0.14909076936,0.149086246795,0.149081724643,0.149077202904,0.149072681578,0.149068160665,0.149063640165,0.149059120078,0.149054600404,0.149050081143,0.149045562294,0.149041043858,0.149036525834,0.149032008223,0.149027491025,0.149022974239,0.149018457865,0.149013941903,0.149009426354,0.149004911217,0.149000396492,0.148995882179,0.148991368277,0.148986854788,0.148982341711,0.148977829045,0.148973316791,0.148968804949,0.148964293518,0.148959782498,0.148955271891,0.148950761694,0.148946251909,0.148941742535,0.148937233572,0.148932725021,0.14892821688,0.148923709151,0.148919201832,0.148914694924,0.148910188428,0.148905682341,0.148901176666,0.148896671401,0.148892166547,0.148887662103,0.14888315807,0.148878654447,0.148874151234,0.148869648432,0.14886514604,0.148860644058,0.148856142486,0.148851641324,0.148847140572,0.14884264023,0.148838140297,0.148833640775,0.148829141662,0.148824642959,0.148820144665,0.148815646781,0.148811149306,0.148806652241,0.148802155585,0.148797659338,0.1487931635,0.148788668072,0.148784173052,0.148779678442,0.14877518424,0.148770690448,0.148766197064,0.148761704089,0.148757211522,0.148752719365,0.148748227615,0.148743736275,0.148739245343,0.148734754819,0.148730264703,0.148725774996,0.148721285697,0.148716796806,0.148712308323,0.148707820248,0.148703332581,0.148698845322,0.148694358471,0.148689872028,0.148685385992,0.148680900364,0.148676415143,0.14867193033,0.148667445924,0.148662961926,0.148658478335,0.148653995152,0.148649512375,0.148645030006,0.148640548044,0.148636066488,0.14863158534,0.148627104599,0.148622624264,0.148618144337,0.148613664815,0.148609185701,0.148604706993,0.148600228692,0.148595750797,0.148591273309,0.148586796226,0.148582319551,0.148577843281,0.148573367417,0.14856889196,0.148564416909,0.148559942263,0.148555468024,0.14855099419,0.148546520762,0.14854204774,0.148537575123,0.148533102912,0.148528631107,0.148524159707,0.148519688712,0.148515218123,0.148510747939,0.148506278161,0.148501808787,0.148497339819,0.148492871255,0.148488403097,0.148483935343,0.148479467995,0.148475001051,0.148470534512,0.148466068377,0.148461602647,0.148457137322,0.148452672401,0.148448207885,0.148443743773,0.148439280065,0.148434816762,0.148430353863,0.148425891367,0.148421429276,0.148416967589,0.148412506306,0.148408045427,0.148403584951,0.148399124879,0.148394665211,0.148390205947,0.148385747086,0.148381288629,0.148376830575,0.148372372924,0.148367915677,0.148363458833,0.148359002392,0.148354546355,0.14835009072,0.148345635489,0.14834118066,0.148336726234,0.148332272212,0.148327818591,0.148323365374,0.148318912559,0.148314460147,0.148310008138,0.148305556531,0.148301105326,0.148296654524,0.148292204123,0.148287754126,0.14828330453,0.148278855336,0.148274406545,0.148269958155,0.148265510168,0.148261062582,0.148256615398,0.148252168616,0.148247722235,0.148243276256,0.148238830679,0.148234385503,0.148229940728,0.148225496355,0.148221052383,0.148216608813,0.148212165643,0.148207722875,0.148203280508,0.148198838542,0.148194396977,0.148189955812,0.148185515049,0.148181074686,0.148176634724,0.148172195163,0.148167756002,0.148163317242,0.148158878882,0.148154440922,0.148150003363,0.148145566205,0.148141129446,0.148136693088,0.148132257129,0.148127821571,0.148123386413,0.148118951654,0.148114517296,0.148110083337,0.148105649778,0.148101216619,0.148096783859,0.148092351499,0.148087919539,0.148083487977,0.148079056816,0.148074626053,0.14807019569,0.148065765726,0.148061336161,0.148056906995,0.148052478228,0.14804804986,0.148043621891,0.148039194321,0.14803476715,0.148030340377,0.148025914003,0.148021488028,0.148017062451,0.148012637273,0.148008212493,0.148003788111,0.147999364128,0.147994940542,0.147990517355,0.147986094567,0.147981672176,0.147977250183,0.147972828588,0.147968407391,0.147963986592,0.147959566191,0.147955146187,0.147950726581,0.147946307372,0.147941888561,0.147937470148,0.147933052132,0.147928634513,0.147924217291,0.147919800467,0.14791538404,0.14791096801,0.147906552377,0.147902137141,0.147897722301,0.147893307859,0.147888893814,0.147884480165,0.147880066913,0.147875654057,0.147871241598,0.147866829536,0.14786241787,0.1478580066,0.147853595727,0.14784918525,0.147844775169,0.147840365484,0.147835956196,0.147831547303,0.147827138806,0.147822730706,0.147818323001,0.147813915691,0.147809508778,0.14780510226,0.147800696138,0.147796290412,0.14779188508,0.147787480145,0.147783075604,0.147778671459,0.14777426771,0.147769864355,0.147765461395,0.147761058831,0.147756656662,0.147752254887,0.147747853508,0.147743452523,0.147739051933,0.147734651738,0.147730251937,0.147725852531,0.14772145352,0.147717054903,0.14771265668,0.147708258852,0.147703861418,0.147699464378,0.147695067733,0.147690671481,0.147686275624,0.147681880161,0.147677485091,0.147673090416,0.147668696134,0.147664302246,0.147659908752,0.147655515651,0.147651122944,0.147646730631,0.147642338711,0.147637947184,0.147633556051,0.147629165311,0.147624774964,0.147620385011,0.14761599545,0.147611606283,0.147607217509,0.147602829127,0.147598441139,0.147594053543,0.14758966634,0.14758527953,0.147580893112,0.147576507087,0.147572121455,0.147567736215,0.147563351367,0.147558966912,0.147554582849,0.147550199178,0.147545815899,0.147541433013,0.147537050518,0.147532668416,0.147528286706,0.147523905387,0.14751952446,0.147515143925,0.147510763782,0.14750638403,0.14750200467,0.147497625701,0.147493247124,0.147488868939,0.147484491144,0.147480113741,0.147475736729,0.147471360109,0.147466983879,0.147462608041,0.147458232593,0.147453857537,0.147449482871,0.147445108596,0.147440734712,0.147436361219,0.147431988116,0.147427615404,0.147423243083,0.147418871152,0.147414499611,0.147410128461,0.147405757701,0.147401387331,0.147397017351,0.147392647762,0.147388278562,0.147383909753,0.147379541333,0.147375173304,0.147370805664,0.147366438414,0.147362071554,0.147357705083,0.147353339002,0.147348973311,0.147344608009,0.147340243096,0.147335878573,0.147331514439,0.147327150695,0.147322787339,0.147318424373,0.147314061796,0.147309699608,0.147305337808,0.147300976398,0.147296615377,0.147292254744,0.1472878945,0.147283534645,0.147279175178,0.1472748161,0.14727045741,0.147266099109,0.147261741196,0.147257383672,0.147253026535,0.147248669787,0.147244313427,0.147239957455,0.147235601872,0.147231246676,0.147226891868,0.147222537448,0.147218183416,0.147213829771,0.147209476514,0.147205123645,0.147200771164,0.14719641907,0.147192067363,0.147187716044,0.147183365112,0.147179014567,0.14717466441,0.147170314639,0.147165965256,0.14716161626,0.147157267651,0.147152919429,0.147148571594,0.147144224145,0.147139877083,0.147135530408,0.14713118412,0.147126838218,0.147122492703,0.147118147574,0.147113802832,0.147109458476,0.147105114506,0.147100770923,0.147096427725,0.147092084914,0.147087742489,0.14708340045,0.147079058797,0.14707471753,0.147070376648,0.147066036153,0.147061696043,0.147057356319,0.14705301698,0.147048678027,0.14704433946,0.147040001278,0.147035663481,0.147031326069,0.147026989043,0.147022652402,0.147018316147,0.147013980276,0.14700964479,0.14700530969,0.147000974974,0.146996640643,0.146992306697,0.146987973136,0.146983639959,0.146979307167,0.14697497476,0.146970642737,0.146966311098,0.146961979844,0.146957648975,0.146953318489,0.146948988388,0.146944658671,0.146940329338,0.14693600039,0.146931671825,0.146927343644,0.146923015847,0.146918688434,0.146914361405,0.146910034759,0.146905708497,0.146901382619,0.146897057124,0.146892732013,0.146888407285,0.146884082941,0.14687975898,0.146875435402,0.146871112207,0.146866789396,0.146862466968,0.146858144922,0.14685382326,0.14684950198,0.146845181084,0.14684086057,0.146836540439,0.146832220691,0.146827901325,0.146823582342,0.146819263742,0.146814945524,0.146810627688,0.146806310235,0.146801993164,0.146797676475,0.146793360169,0.146789044244,0.146784728702,0.146780413542,0.146776098763,0.146771784367,0.146767470352,0.146763156719,0.146758843468,0.146754530599,0.146750218111,0.146745906005,0.14674159428,0.146737282937,0.146732971975,0.146728661395,0.146724351195,0.146720041377,0.146715731941,0.146711422885,0.14670711421,0.146702805917,0.146698498004,0.146694190472,0.146689883321,0.146685576551,0.146681270161,0.146676964153,0.146672658524,0.146668353277,0.14666404841,0.146659743923,0.146655439816,0.14665113609,0.146646832745,0.146642529779,0.146638227194,0.146633924988,0.146629623163,0.146625321718,0.146621020653,0.146616719967,0.146612419662,0.146608119736,0.14660382019,0.146599521023,0.146595222236,0.146590923829,0.146586625801,0.146582328152,0.146578030883,0.146573733993,0.146569437483,0.146565141352,0.146560845599,0.146556550226,0.146552255232,0.146547960617,0.146543666381,0.146539372524,0.146535079045,0.146530785945,0.146526493224,0.146522200882,0.146517908918,0.146513617333,0.146509326126,0.146505035298,0.146500744848,0.146496454776,0.146492165083,0.146487875767,0.14648358683,0.146479298271,0.14647501009,0.146470722287,0.146466434862,0.146462147815,0.146457861145,0.146453574853,0.146449288939,0.146445003403,0.146440718244,0.146436433463,0.146432149059,0.146427865033,0.146423581384,0.146419298112,0.146415015217,0.1464107327,0.14640645056,0.146402168797,0.146397887411,0.146393606402,0.14638932577,0.146385045514,0.146380765636,0.146376486134,0.146372207009,0.146367928261,0.146363649889,0.146359371893,0.146355094274,0.146350817032,0.146346540166,0.146342263676,0.146337987563,0.146333711825,0.146329436464,0.146325161479,0.14632088687,0.146316612637,0.146312338779,0.146308065298,0.146303792192,0.146299519463,0.146295247109,0.14629097513,0.146286703527,0.1462824323,0.146278161448,0.146273890971,0.14626962087,0.146265351145,0.146261081794,0.146256812819,0.146252544218,0.146248275993,0.146244008143,0.146239740668,0.146235473568,0.146231206842,0.146226940492,0.146222674516,0.146218408915,0.146214143689,0.146209878837,0.14620561436,0.146201350257,0.146197086528,0.146192823174,0.146188560195,0.146184297589,0.146180035358,0.146175773501,0.146171512018,0.146167250909,0.146162990174,0.146158729813,0.146154469825,0.146150210212,0.146145950973,0.146141692107,0.146137433614,0.146133175496,0.146128917751,0.146124660379,0.146120403381,0.146116146756,0.146111890505,0.146107634626,0.146103379122,0.14609912399,0.146094869231,0.146090614845,0.146086360833,0.146082107193,0.146077853926,0.146073601032,0.146069348511,0.146065096362,0.146060844587,0.146056593183,0.146052342153,0.146048091495,0.146043841209,0.146039591296,0.146035341755,0.146031092586,0.14602684379,0.146022595365,0.146018347313,0.146014099633,0.146009852325,0.146005605389,0.146001358825,0.145997112632,0.145992866812,0.145988621363,0.145984376286,0.14598013158,0.145975887246,0.145971643284,0.145967399693,0.145963156474,0.145958913625,0.145954671148,0.145950429043,0.145946187308,0.145941945945,0.145937704953,0.145933464332,0.145929224082,0.145924984202,0.145920744694,0.145916505557,0.14591226679,0.145908028394,0.145903790368,0.145899552713,0.145895315429,0.145891078515,0.145886841972,0.145882605799,0.145878369996,0.145874134564,0.145869899502,0.14586566481,0.145861430488,0.145857196536,0.145852962954,0.145848729742,0.1458444969,0.145840264428,0.145836032326,0.145831800593,0.14582756923,0.145823338237,0.145819107613,0.145814877359,0.145810647474,0.145806417958,0.145802188812,0.145797960036,0.145793731628,0.14578950359,0.145785275921,0.145781048621,0.14577682169,0.145772595128,0.145768368934,0.14576414311,0.145759917655,0.145755692568,0.14575146785,0.145747243501,0.14574301952,0.145738795908,0.145734572664,0.145730349789,0.145726127282,0.145721905143,0.145717683373,0.145713461971,0.145709240937,0.145705020271,0.145700799973,0.145696580044,0.145692360482,0.145688141288,0.145683922462,0.145679704004,0.145675485913,0.14567126819,0.145667050835,0.145662833848,0.145658617227,0.145654400975,0.14565018509,0.145645969572,0.145641754421,0.145637539638,0.145633325222,0.145629111173,0.145624897491,0.145620684176,0.145616471229,0.145612258648,0.145608046434,0.145603834587,0.145599623106,0.145595411993,0.145591201246,0.145586990865,0.145582780851,0.145578571204,0.145574361923,0.145570153009,0.145565944461,0.145561736279,0.145557528463,0.145553321014,0.14554911393,0.145544907213,0.145540700862,0.145536494877,0.145532289258,0.145528084004,0.145523879117,0.145519674595,0.145515470439,0.145511266648,0.145507063223,0.145502860164,0.14549865747,0.145494455142,0.145490253179,0.145486051581,0.145481850349,0.145477649482,0.14547344898,0.145469248843,0.145465049071,0.145460849664,0.145456650622,0.145452451945,0.145448253633,0.145444055686,0.145439858103,0.145435660886,0.145431464032,0.145427267544,0.14542307142,0.14541887566,0.145414680265,0.145410485234,0.145406290568,0.145402096266,0.145397902328,0.145393708754,0.145389515544,0.145385322699,0.145381130217,0.1453769381,0.145372746346,0.145368554956,0.14536436393,0.145360173268,0.145355982969,0.145351793034,0.145347603462,0.145343414255,0.14533922541,0.145335036929,0.145330848811,0.145326661057,0.145322473666,0.145318286638,0.145314099974,0.145309913672,0.145305727734,0.145301542158,0.145297356946,0.145293172096,0.145288987609,0.145284803485,0.145280619724,0.145276436325,0.145272253289,0.145268070616,0.145263888305,0.145259706357,0.145255524771,0.145251343547,0.145247162686,0.145242982187,0.14523880205,0.145234622276,0.145230442863,0.145226263813,0.145222085125,0.145217906798,0.145213728834,0.145209551231,0.14520537399,0.145201197111,0.145197020593,0.145192844437,0.145188668643,0.14518449321,0.145180318139,0.145176143429,0.145171969081,0.145167795094,0.145163621468,0.145159448203,0.1451552753,0.145151102757,0.145146930576,0.145142758756,0.145138587297,0.145134416198,0.145130245461,0.145126075084,0.145121905068,0.145117735412,0.145113566118,0.145109397184,0.14510522861,0.145101060397,0.145096892544,0.145092725052,0.14508855792,0.145084391149,0.145080224737,0.145076058686,0.145071892995,0.145067727664,0.145063562693,0.145059398082,0.14505523383,0.145051069939,0.145046906408,0.145042743236,0.145038580424,0.145034417972,0.145030255879,0.145026094146,0.145021932772,0.145017771758,0.145013611103,0.145009450807,0.145005290871,0.145001131294,0.144996972076,0.144992813217,0.144988654718,0.144984496577,0.144980338795,0.144976181373,0.144972024309,0.144967867604,0.144963711258,0.14495955527,0.144955399641,0.144951244371,0.144947089459,0.144942934906,0.144938780712,0.144934626875,0.144930473397,0.144926320278,0.144922167516,0.144918015113,0.144913863068,0.144909711381,0.144905560052,0.144901409082,0.144897258469,0.144893108214,0.144888958316,0.144884808777,0.144880659595,0.144876510771,0.144872362305,0.144868214196,0.144864066445,0.144859919051,0.144855772015,0.144851625336,0.144847479014,0.14484333305,0.144839187442,0.144835042192,0.144830897299,0.144826752764,0.144822608585,0.144818464763,0.144814321298,0.14481017819,0.144806035439,0.144801893044,0.144797751006,0.144793609325,0.144789468001,0.144785327033,0.144781186421,0.144777046166,0.144772906267,0.144768766725,0.144764627539,0.144760488709,0.144756350236,0.144752212118,0.144748074357,0.144743936951,0.144739799902,0.144735663209,0.144731526871,0.144727390889,0.144723255264,0.144719119993,0.144714985079,0.14471085052,0.144706716317,0.144702582469,0.144698448977,0.14469431584,0.144690183058,0.144686050632,0.144681918561,0.144677786845,0.144673655485,0.144669524479,0.144665393829,0.144661263534,0.144657133593,0.144653004008,0.144648874777,0.144644745902,0.144640617381,0.144636489214,0.144632361402,0.144628233945,0.144624106843,0.144619980095,0.144615853701,0.144611727662,0.144607601977,0.144603476647,0.144599351671,0.144595227049,0.144591102781,0.144586978867,0.144582855307,0.144578732101,0.144574609249,0.144570486751,0.144566364607,0.144562242817,0.14455812138,0.144554000298,0.144549879568,0.144545759193,0.144541639171,0.144537519502,0.144533400187,0.144529281225,0.144525162616,0.144521044361,0.144516926459,0.14451280891,0.144508691715,0.144504574872,0.144500458383,0.144496342246,0.144492226462,0.144488111032,0.144483995954,0.144479881229,0.144475766856,0.144471652836,0.144467539169,0.144463425855,0.144459312893,0.144455200283,0.144451088026,0.144446976121,0.144442864569,0.144438753369,0.144434642521,0.144430532025,0.144426421881,0.14442231209,0.14441820265,0.144414093562,0.144409984827,0.144405876443,0.144401768411,0.144397660731,0.144393553402,0.144389446425,0.1443853398,0.144381233526,0.144377127604,0.144373022033,0.144368916814,0.144364811946,0.144360707429,0.144356603264,0.14435249945,0.144348395987,0.144344292875,0.144340190114,0.144336087704,0.144331985645,0.144327883937,0.14432378258,0.144319681574,0.144315580919,0.144311480614,0.14430738066,0.144303281056,0.144299181803,0.144295082901,0.144290984349,0.144286886147,0.144282788296,0.144278690795,0.144274593644,0.144270496844,0.144266400393,0.144262304293,0.144258208543,0.144254113143,0.144250018093,0.144245923392,0.144241829042,0.144237735041,0.14423364139,0.144229548089,0.144225455138,0.144221362536,0.144217270283,0.144213178381,0.144209086827,0.144204995623,0.144200904769,0.144196814263,0.144192724107,0.1441886343,0.144184544843,0.144180455734,0.144176366975,0.144172278564,0.144168190503,0.14416410279,0.144160015426,0.144155928411,0.144151841745,0.144147755427,0.144143669459,0.144139583838,0.144135498567,0.144131413644,0.144127329069,0.144123244843,0.144119160965,0.144115077435,0.144110994254,0.14410691142,0.144102828935,0.144098746798,0.14409466501,0.144090583569,0.144086502476,0.144082421731,0.144078341334,0.144074261285,0.144070181583,0.144066102229,0.144062023223,0.144057944565,0.144053866254,0.14404978829,0.144045710674,0.144041633406,0.144037556485,0.144033479911,0.144029403684,0.144025327805,0.144021252273,0.144017177088,0.14401310225,0.144009027759,0.144004953615,0.144000879818,0.143996806367,0.143992733264,0.143988660507,0.143984588097,0.143980516034,0.143976444318,0.143972372948,0.143968301924,0.143964231247,0.143960160917,0.143956090932,0.143952021294,0.143947952003,0.143943883057,0.143939814458,0.143935746205,0.143931678298,0.143927610737,0.143923543522,0.143919476653,0.14391541013,0.143911343953,0.143907278121,0.143903212636,0.143899147496,0.143895082701,0.143891018252,0.143886954149,0.143882890391,0.143878826979,0.143874763912,0.14387070119,0.143866638814,0.143862576783,0.143858515097,0.143854453756,0.14385039276,0.14384633211,0.143842271804,0.143838211844,0.143834152228,0.143830092957,0.143826034031,0.143821975449,0.143817917213,0.143813859321,0.143809801774,0.143805744571,0.143801687712,0.143797631199,0.143793575029,0.143789519204,0.143785463723,0.143781408587,0.143777353795,0.143773299347,0.143769245243,0.143765191483,0.143761138067,0.143757084995,0.143753032267,0.143748979883,0.143744927843,0.143740876146,0.143736824793,0.143732773784,0.143728723119,0.143724672797,0.143720622819,0.143716573184,0.143712523892,0.143708474944,0.14370442634,0.143700378078,0.14369633016,0.143692282585,0.143688235353,0.143684188465,0.143680141919,0.143676095716,0.143672049857,0.14366800434,0.143663959166,0.143659914335,0.143655869846,0.143651825701,0.143647781897,0.143643738437,0.143639695319,0.143635652544,0.143631610111,0.14362756802,0.143623526272,0.143619484866,0.143615443803,0.143611403082,0.143607362702,0.143603322665,0.14359928297,0.143595243617,0.143591204607,0.143587165938,0.14358312761,0.143579089625,0.143575051982,0.14357101468,0.14356697772,0.143562941101,0.143558904824,0.143554868889,0.143550833295,0.143546798043,0.143542763132,0.143538728562,0.143534694334,0.143530660447,0.143526626901,0.143522593696,0.143518560832,0.14351452831,0.143510496128,0.143506464288,0.143502432788,0.143498401629,0.143494370811,0.143490340334,0.143486310198,0.143482280402,0.143478250947,0.143474221832,0.143470193058,0.143466164624,0.143462136531,0.143458108778,0.143454081366,0.143450054294,0.143446027562,0.14344200117,0.143437975118,0.143433949407,0.143429924035,0.143425899004,0.143421874312,0.143417849961,0.143413825949,0.143409802277,0.143405778945,0.143401755952,0.1433977333,0.143393710986,0.143389689013,0.143385667379,0.143381646084,0.143377625129,0.143373604513,0.143369584237,0.143365564299,0.143361544701,0.143357525442,0.143353506523,0.143349487942,0.143345469701,0.143341451798,0.143337434234,0.14333341701,0.143329400124,0.143325383577,0.143321367368,0.143317351499,0.143313335968,0.143309320776,0.143305305922,0.143301291407,0.14329727723,0.143293263391,0.143289249891,0.14328523673,0.143281223906,0.143277211421,0.143273199274,0.143269187465,0.143265175994,0.143261164862,0.143257154067,0.14325314361,0.143249133491,0.14324512371,0.143241114267,0.143237105161,0.143233096393,0.143229087963,0.143225079871,0.143221072116,0.143217064698,0.143213057618,0.143209050875,0.14320504447,0.143201038402,0.143197032672,0.143193027278,0.143189022222,0.143185017503,0.143181013121,0.143177009076,0.143173005368,0.143169001997,0.143164998963,0.143160996265,0.143156993905,0.143152991881,0.143148990194,0.143144988844,0.14314098783,0.143136987153,0.143132986812,0.143128986808,0.14312498714,0.143120987809,0.143116988813,0.143112990155,0.143108991832,0.143104993846,0.143100996195,0.143096998881,0.143093001903,0.143089005261,0.143085008955,0.143081012984,0.14307701735,0.143073022051,0.143069027088,0.143065032461,0.14306103817,0.143057044214,0.143053050594,0.143049057309,0.143045064359,0.143041071746,0.143037079467,0.143033087524,0.143029095916,0.143025104643,0.143021113706,0.143017123104,0.143013132836,0.143009142904,0.143005153307,0.143001164045,0.142997175118,0.142993186525,0.142989198268,0.142985210345,0.142981222757,0.142977235503,0.142973248584,0.142969262,0.142965275751,0.142961289835,0.142957304255,0.142953319008,0.142949334096,0.142945349518,0.142941365275,0.142937381366,0.142933397791,0.14292941455,0.142925431643,0.14292144907,0.142917466831,0.142913484926,0.142909503355,0.142905522117,0.142901541214,0.142897560644,0.142893580408,0.142889600506,0.142885620937,0.142881641701,0.1428776628,0.142873684231,0.142869705996,0.142865728095,0.142861750527,0.142857773292,0.14285379639,0.142849819821,0.142845843586,0.142841867683,0.142837892114,0.142833916878,0.142829941974,0.142825967404,0.142821993166,0.142818019261,0.142814045689,0.14281007245,0.142806099543,0.142802126969,0.142798154728,0.142794182819,0.142790211242,0.142786239998,0.142782269086,0.142778298507,0.14277432826,0.142770358345,0.142766388762,0.142762419512,0.142758450593,0.142754482007,0.142750513753,0.14274654583,0.14274257824,0.142738610981,0.142734644054,0.142730677459,0.142726711196,0.142722745265,0.142718779665,0.142714814396,0.14271084946,0.142706884854,0.14270292058,0.142698956638,0.142694993027,0.142691029747,0.142687066799,0.142683104182,0.142679141895,0.142675179941,0.142671218317,0.142667257024,0.142663296062,0.142659335431,0.142655375131,0.142651415162,0.142647455524,0.142643496216,0.142639537239,0.142635578593,0.142631620277,0.142627662292,0.142623704638,0.142619747314,0.14261579032,0.142611833657,0.142607877324,0.142603921322,0.14259996565,0.142596010308,0.142592055296,0.142588100614,0.142584146262,0.14258019224,0.142576238549,0.142572285187,0.142568332155,0.142564379453,0.142560427081,0.142556475038,0.142552523325,0.142548571942,0.142544620888,0.142540670164,0.14253671977,0.142532769705,0.142528819969,0.142524870563,0.142520921486,0.142516972738,0.14251302432,0.142509076231,0.142505128471,0.14250118104,0.142497233938,0.142493287165,0.142489340721,0.142485394606,0.14248144882,0.142477503363,0.142473558234,0.142469613434,0.142465668963,0.142461724821,0.142457781007,0.142453837522,0.142449894365,0.142445951536,0.142442009036,0.142438066865,0.142434125022,0.142430183507,0.14242624232,0.142422301461,0.142418360931,0.142414420728,0.142410480854,0.142406541308,0.14240260209,0.142398663199,0.142394724637,0.142390786402,0.142386848495,0.142382910916,0.142378973664,0.14237503674,0.142371100144,0.142367163875,0.142363227934,0.14235929232,0.142355357034,0.142351422074,0.142347487443,0.142343553138,0.142339619161,0.142335685511,0.142331752188,0.142327819192,0.142323886523,0.142319954182,0.142316022167,0.142312090479,0.142308159118,0.142304228083,0.142300297376,0.142296366995,0.142292436941,0.142288507214,0.142284577813,0.142280648739,0.142276719991,0.142272791569,0.142268863474,0.142264935706,0.142261008264,0.142257081148,0.142253154358,0.142249227894,0.142245301757,0.142241375945,0.14223745046,0.142233525301,0.142229600467,0.14222567596,0.142221751778,0.142217827922,0.142213904392,0.142209981188,0.14220605831,0.142202135757,0.142198213529,0.142194291627,0.142190370051,0.1421864488,0.142182527875,0.142178607275,0.142174687,0.14217076705,0.142166847426,0.142162928127,0.142159009153,0.142155090504,0.142151172181,0.142147254182,0.142143336508,0.142139419159,0.142135502135,0.142131585436,0.142127669061,0.142123753012,0.142119837287,0.142115921886,0.142112006811,0.142108092059,0.142104177633,0.14210026353,0.142096349753,0.142092436299,0.14208852317,0.142084610365,0.142080697885,0.142076785728,0.142072873896,0.142068962388,0.142065051204,0.142061140344,0.142057229808,0.142053319595,0.142049409707,0.142045500143,0.142041590902,0.142037681985,0.142033773392,0.142029865122,0.142025957176,0.142022049554,0.142018142255,0.14201423528,0.142010328628,0.142006422299,0.142002516294,0.141998610612,0.141994705253,0.141990800218,0.141986895506,0.141982991116,0.14197908705,0.141975183307,0.141971279887,0.14196737679,0.141963474016,0.141959571564,0.141955669436,0.14195176763,0.141947866147,0.141943964986,0.141940064148,0.141936163633,0.14193226344,0.14192836357,0.141924464023,0.141920564797,0.141916665894,0.141912767313,0.141908869055,0.141904971119,0.141901073505,0.141897176213,0.141893279243,0.141889382595,0.14188548627,0.141881590266,0.141877694584,0.141873799224,0.141869904186,0.141866009469,0.141862115075,0.141858221002,0.14185432725,0.141850433821,0.141846540713,0.141842647926,0.141838755461,0.141834863317,0.141830971495,0.141827079993,0.141823188814,0.141819297955,0.141815407418,0.141811517202,0.141807627307,0.141803737733,0.14179984848,0.141795959548,0.141792070937,0.141788182647,0.141784294677,0.141780407029,0.141776519701,0.141772632694,0.141768746008,0.141764859642,0.141760973597,0.141757087873,0.141753202468,0.141749317385,0.141745432622,0.141741548179,0.141737664056,0.141733780254,0.141729896772,0.14172601361,0.141722130768,0.141718248247,0.141714366045,0.141710484163,0.141706602602,0.14170272136,0.141698840438,0.141694959836,0.141691079554,0.141687199592,0.141683319949,0.141679440626,0.141675561622,0.141671682938,0.141667804574,0.141663926529,0.141660048804,0.141656171397,0.141652294311,0.141648417543,0.141644541095,0.141640664966,0.141636789156,0.141632913666,0.141629038494,0.141625163641,0.141621289108,0.141617414893,0.141613540998,0.141609667421,0.141605794163,0.141601921223,0.141598048603,0.141594176301,0.141590304318,0.141586432653,0.141582561307,0.14157869028,0.141574819571,0.14157094918,0.141567079108,0.141563209354,0.141559339918,0.141555470801,0.141551602002,0.141547733521,0.141543865358,0.141539997513,0.141536129986,0.141532262778,0.141528395887,0.141524529314,0.141520663059,0.141516797121,0.141512931502,0.1415090662,0.141505201216,0.14150133655,0.141497472201,0.141493608169,0.141489744456,0.141485881059,0.14148201798,0.141478155219,0.141474292774,0.141470430648,0.141466568838,0.141462707345,0.14145884617,0.141454985312,0.141451124771,0.141447264546,0.141443404639,0.141439545049,0.141435685776,0.141431826819,0.14142796818,0.141424109857,0.141420251851,0.141416394161,0.141412536788,0.141408679732,0.141404822992,0.141400966569,0.141397110462,0.141393254672,0.141389399198,0.14138554404,0.141381689199,0.141377834674,0.141373980465,0.141370126572,0.141366272996,0.141362419735,0.141358566791,0.141354714162,0.141350861849,0.141347009853,0.141343158172,0.141339306807,0.141335455757,0.141331605024,0.141327754606,0.141323904504,0.141320054717,0.141316205246,0.14131235609,0.14130850725,0.141304658726,0.141300810516,0.141296962622,0.141293115044,0.14128926778,0.141285420832,0.141281574199,0.141277727881,0.141273881878,0.14127003619,0.141266190817,0.141262345759,0.141258501016,0.141254656588,0.141250812475,0.141246968676,0.141243125192,0.141239282023,0.141235439168,0.141231596628,0.141227754403,0.141223912492,0.141220070896,0.141216229614,0.141212388646,0.141208547993,0.141204707654,0.141200867629,0.141197027919,0.141193188523,0.141189349441,0.141185510672,0.141181672218,0.141177834078,0.141173996252,0.14117015874,0.141166321542,0.141162484657,0.141158648087,0.14115481183,0.141150975886,0.141147140257,0.141143304941,0.141139469938,0.141135635249,0.141131800874,0.141127966812,0.141124133063,0.141120299628,0.141116466506,0.141112633698,0.141108801202,0.14110496902,0.141101137151,0.141097305595,0.141093474352,0.141089643422,0.141085812805,0.141081982501,0.14107815251,0.141074322831,0.141070493466,0.141066664413,0.141062835673,0.141059007246,0.141055179131,0.141051351329,0.141047523839,0.141043696662,0.141039869797,0.141036043245,0.141032217005,0.141028391077,0.141024565462,0.141020740159,0.141016915168,0.14101309049,0.141009266123,0.141005442068,0.141001618326,0.140997794895,0.140993971777,0.14099014897,0.140986326476,0.140982504293,0.140978682421,0.140974860862,0.140971039614,0.140967218678,0.140963398054,0.140959577741,0.140955757739,0.140951938049,0.140948118671,0.140944299604,0.140940480848,0.140936662403,0.14093284427,0.140929026448,0.140925208937,0.140921391738,0.140917574849,0.140913758271,0.140909942005,0.140906126049,0.140902310405,0.140898495071,0.140894680048,0.140890865336,0.140887050935,0.140883236844,0.140879423064,0.140875609595,0.140871796436,0.140867983588,0.14086417105,0.140860358823,0.140856546906,0.1408527353,0.140848924003,0.140845113018,0.140841302342,0.140837491977,0.140833681921,0.140829872176,0.140826062741,0.140822253616,0.140818444801,0.140814636296,0.140810828101,0.140807020216,0.14080321264,0.140799405375,0.140795598419,0.140791791773,0.140787985436,0.140784179409,0.140780373692,0.140776568284,0.140772763186,0.140768958397,0.140765153917,0.140761349747,0.140757545886,0.140753742335,0.140749939093,0.140746136159,0.140742333535,0.140738531221,0.140734729215,0.140730927518,0.14072712613,0.140723325052,0.140719524282,0.140715723821,0.140711923668,0.140708123825,0.14070432429,0.140700525064,0.140696726147,0.140692927538,0.140689129238,0.140685331246,0.140681533563,0.140677736188,0.140673939122,0.140670142363,0.140666345914,0.140662549772,0.140658753939,0.140654958414,0.140651163197,0.140647368288,0.140643573688,0.140639779395,0.14063598541,0.140632191733,0.140628398364,0.140624605303,0.14062081255,0.140617020105,0.140613227967,0.140609436137,0.140605644615,0.1406018534,0.140598062492,0.140594271893,0.1405904816,0.140586691616,0.140582901938,0.140579112568,0.140575323505,0.14057153475,0.140567746302,0.140563958161,0.140560170327,0.1405563828,0.14055259558,0.140548808667,0.140545022061,0.140541235763,0.140537449771,0.140533664086,0.140529878707,0.140526093636,0.140522308871,0.140518524413,0.140514740261,0.140510956416,0.140507172878,0.140503389646,0.140499606721,0.140495824102,0.14049204179,0.140488259784,0.140484478084,0.14048069669,0.140476915603,0.140473134822,0.140469354347,0.140465574178,0.140461794315,0.140458014758,0.140454235507,0.140450456563,0.140446677924,0.14044289959,0.140439121563,0.140435343842,0.140431566426,0.140427789316,0.140424012511,0.140420236012,0.140416459819,0.140412683931,0.140408908349,0.140405133072,0.140401358101,0.140397583435,0.140393809074,0.140390035018,0.140386261268,0.140382487823,0.140378714683,0.140374941848,0.140371169318,0.140367397094,0.140363625174,0.140359853559,0.140356082249,0.140352311244,0.140348540544,0.140344770149,0.140341000058,0.140337230272,0.140333460791,0.140329691614,0.140325922742,0.140322154175,0.140318385911,0.140314617953,0.140310850299,0.140307082949,0.140303315903,0.140299549162,0.140295782725,0.140292016592,0.140288250764,0.140284485239,0.140280720019,0.140276955102,0.14027319049,0.140269426182,0.140265662177,0.140261898476,0.14025813508,0.140254371987,0.140250609197,0.140246846712,0.14024308453,0.140239322652,0.140235561077,0.140231799806,0.140228038839,0.140224278174,0.140220517814,0.140216757756,0.140212998003,0.140209238552,0.140205479404,0.14020172056,0.140197962019,0.140194203781,0.140190445847,0.140186688215,0.140182930886,0.140179173861,0.140175417138,0.140171660718,0.140167904601,0.140164148787,0.140160393275,0.140156638067,0.140152883161,0.140149128557,0.140145374257,0.140141620258,0.140137866563,0.14013411317,0.140130360079,0.14012660729,0.140122854804,0.140119102621,0.140115350739,0.14011159916,0.140107847883,0.140104096909,0.140100346236,0.140096595865,0.140092845797,0.14008909603,0.140085346566,0.140081597403,0.140077848542,0.140074099984,0.140070351726,0.140066603771,0.140062856117,0.140059108765,0.140055361715,0.140051614966,0.140047868519,0.140044122373,0.140040376529,0.140036630986,0.140032885745,0.140029140804,0.140025396166,0.140021651828,0.140017907792,0.140014164057,0.140010420623,0.14000667749,0.140002934658,0.139999192127,0.139995449898,0.139991707969,0.139987966341,0.139984225014,0.139980483987,0.139976743262,0.139973002837,0.139969262713,0.13996552289,0.139961783367,0.139958044145,0.139954305223,0.139950566602,0.139946828281,0.139943090261,0.139939352541,0.139935615121,0.139931878002,0.139928141183,0.139924404664,0.139920668446,0.139916932527,0.139913196909,0.13990946159,0.139905726572,0.139901991854,0.139898257435,0.139894523317,0.139890789498,0.139887055979,0.13988332276,0.139879589841,0.139875857221,0.139872124901,0.13986839288,0.13986466116,0.139860929738,0.139857198616,0.139853467794,0.139849737271,0.139846007047,0.139842277123,0.139838547498,0.139834818172,0.139831089146,0.139827360419,0.13982363199,0.139819903861,0.139816176031,0.1398124485,0.139808721268,0.139804994335,0.1398012677,0.139797541365,0.139793815328,0.13979008959,0.139786364151,0.139782639011,0.139778914169,0.139775189626,0.139771465381,0.139767741435,0.139764017787,0.139760294438,0.139756571387,0.139752848635,0.139749126181,0.139745404025,0.139741682167,0.139737960608,0.139734239347,0.139730518384,0.139726797719,0.139723077352,0.139719357283,0.139715637512,0.139711918039,0.139708198864,0.139704479986,0.139700761407,0.139697043125,0.139693325141,0.139689607455,0.139685890066,0.139682172975,0.139678456182,0.139674739686,0.139671023487,0.139667307586,0.139663591983,0.139659876676,0.139656161667,0.139652446956,0.139648732541,0.139645018424,0.139641304604,0.139637591082,0.139633877856,0.139630164927,0.139626452295,0.139622739961,0.139619027923,0.139615316182,0.139611604738,0.139607893591,0.13960418274,0.139600472186,0.139596761929,0.139593051969,0.139589342305,0.139585632938,0.139581923867,0.139578215093,0.139574506615,0.139570798434,0.139567090549,0.139563382961,0.139559675668,0.139555968672,0.139552261973,0.139548555569,0.139544849461,0.13954114365,0.139537438135,0.139533732915,0.139530027992,0.139526323365,0.139522619033,0.139518914998,0.139515211258,0.139511507814,0.139507804665,0.139504101813,0.139500399256,0.139496696995,0.139492995029,0.139489293359,0.139485591984,0.139481890905,0.139478190121,0.139474489633,0.13947078944,0.139467089542,0.13946338994,0.139459690633,0.139455991621,0.139452292904,0.139448594482,0.139444896355,0.139441198524,0.139437500987,0.139433803745,0.139430106798,0.139426410147,0.139422713789,0.139419017727,0.13941532196,0.139411626487,0.139407931309,0.139404236425,0.139400541836,0.139396847542,0.139393153542,0.139389459837,0.139385766426,0.13938207331,0.139378380488,0.13937468796,0.139370995726,0.139367303787,0.139363612142,0.139359920791,0.139356229735,0.139352538972,0.139348848503,0.139345158329,0.139341468448,0.139337778862,0.139334089569,0.13933040057,0.139326711865,0.139323023454,0.139319335336,0.139315647512,0.139311959982,0.139308272745,0.139304585802,0.139300899153,0.139297212797,0.139293526734,0.139289840965,0.13928615549,0.139282470307,0.139278785418,0.139275100822,0.13927141652,0.139267732511,0.139264048794,0.139260365371,0.139256682241,0.139252999404,0.13924931686,0.139245634609,0.139241952651,0.139238270986,0.139234589614,0.139230908534,0.139227227747,0.139223547253,0.139219867052,0.139216187143,0.139212507527,0.139208828203,0.139205149172,0.139201470434,0.139197791988,0.139194113834,0.139190435973,0.139186758404,0.139183081127,0.139179404143,0.13917572745,0.13917205105,0.139168374942,0.139164699127,0.139161023603,0.139157348371,0.139153673431,0.139149998784,0.139146324428,0.139142650364,0.139138976592,0.139135303111,0.139131629923,0.139127957026,0.139124284421,0.139120612107,0.139116940085,0.139113268354,0.139109596916,0.139105925768,0.139102254912,0.139098584348,0.139094914074,0.139091244092,0.139087574402,0.139083905002,0.139080235894,0.139076567077,0.139072898551,0.139069230317,0.139065562373,0.13906189472,0.139058227359,0.139054560288,0.139050893508,0.139047227019,0.139043560821,0.139039894913,0.139036229297,0.139032563971,0.139028898936,0.139025234191,0.139021569737,0.139017905573,0.1390142417,0.139010578118,0.139006914826,0.139003251824,0.138999589113,0.138995926692,0.138992264561,0.138988602721,0.138984941171,0.138981279911,0.138977618941,0.138973958261,0.138970297871,0.138966637771,0.138962977962,0.138959318442,0.138955659212,0.138952000272,0.138948341622,0.138944683261,0.13894102519,0.138937367409,0.138933709918,0.138930052716,0.138926395804,0.138922739182,0.138919082849,0.138915426805,0.138911771051,0.138908115586,0.138904460411,0.138900805525,0.138897150928,0.138893496621,0.138889842602,0.138886188873,0.138882535433,0.138878882282,0.13887522942,0.138871576848,0.138867924564,0.138864272569,0.138860620863,0.138856969446,0.138853318318,0.138849667478,0.138846016927,0.138842366665,0.138838716692,0.138835067007,0.138831417611,0.138827768504,0.138824119685,0.138820471154,0.138816822912,0.138813174958,0.138809527293,0.138805879916,0.138802232827,0.138798586027,0.138794939514,0.13879129329,0.138787647355,0.138784001707,0.138780356347,0.138776711275,0.138773066492,0.138769421996,0.138765777788,0.138762133868,0.138758490236,0.138754846891,0.138751203835,0.138747561066,0.138743918585,0.138740276391,0.138736634485,0.138732992867,0.138729351536,0.138725710493,0.138722069737,0.138718429269,0.138714789088,0.138711149194,0.138707509587,0.138703870268,0.138700231236,0.138696592492,0.138692954034,0.138689315864,0.13868567798,0.138682040384,0.138678403075,0.138674766052,0.138671129317,0.138667492868,0.138663856706,0.138660220832,0.138656585243,0.138652949942,0.138649314927,0.138645680199,0.138642045758,0.138638411603,0.138634777735,0.138631144153,0.138627510858,0.138623877849,0.138620245127,0.13861661269,0.138612980541,0.138609348677,0.1386057171,0.138602085809,0.138598454804,0.138594824085,0.138591193653,0.138587563506,0.138583933646,0.138580304071,0.138576674782,0.13857304578,0.138569417063,0.138565788632,0.138562160487,0.138558532627,0.138554905053,0.138551277765,0.138547650763,0.138544024046,0.138540397615,0.138536771469,0.138533145609,0.138529520034,0.138525894745,0.138522269741,0.138518645022,0.138515020589,0.138511396441,0.138507772578,0.138504149,0.138500525708,0.1384969027,0.138493279978,0.138489657541,0.138486035389,0.138482413521,0.138478791939,0.138475170642,0.138471549629,0.138467928901,0.138464308458,0.1384606883,0.138457068427,0.138453448838,0.138449829533,0.138446210514,0.138442591779,0.138438973328,0.138435355162,0.13843173728,0.138428119683,0.13842450237,0.138420885342,0.138417268597,0.138413652137,0.138410035962,0.13840642007,0.138402804463,0.138399189139,0.1383955741,0.138391959345,0.138388344873,0.138384730686,0.138381116783,0.138377503163,0.138373889828,0.138370276776,0.138366664008,0.138363051523,0.138359439323,0.138355827406,0.138352215772,0.138348604423,0.138344993356,0.138341382574,0.138337772074,0.138334161859,0.138330551926,0.138326942277,0.138323332912,0.138319723829,0.13831611503,0.138312506514,0.138308898281,0.138305290332,0.138301682665,0.138298075281,0.138294468181,0.138290861364,0.138287254829,0.138283648577,0.138280042609,0.138276436923,0.13827283152,0.138269226399,0.138265621562,0.138262017007,0.138258412735,0.138254808745,0.138251205038,0.138247601614,0.138243998472,0.138240395612,0.138236793035,0.13823319074,0.138229588728,0.138225986998,0.13822238555,0.138218784385,0.138215183502,0.138211582901,0.138207982582,0.138204382545,0.13820078279,0.138197183317,0.138193584127,0.138189985218,0.138186386591,0.138182788246,0.138179190183,0.138175592401,0.138171994902,0.138168397684,0.138164800748,0.138161204093,0.13815760772,0.138154011629,0.138150415819,0.138146820291,0.138143225044,0.138139630079,0.138136035395,0.138132440993,0.138128846871,0.138125253031,0.138121659473,0.138118066195,0.138114473199,0.138110880484,0.13810728805,0.138103695897,0.138100104025,0.138096512434,0.138092921124,0.138089330095,0.138085739347,0.138082148879,0.138078558693,0.138074968787,0.138071379162,0.138067789818,0.138064200754,0.138060611971,0.138057023468,0.138053435247,0.138049847305,0.138046259644,0.138042672264,0.138039085164,0.138035498344,0.138031911805,0.138028325546,0.138024739567,0.138021153869,0.13801756845,0.138013983312,0.138010398454,0.138006813876,0.138003229578,0.13799964556,0.137996061822,0.137992478364,0.137988895186,0.137985312287,0.137981729669,0.13797814733,0.137974565271,0.137970983492,0.137967401992,0.137963820773,0.137960239832,0.137956659171,0.13795307879,0.137949498689,0.137945918866,0.137942339323,0.13793876006,0.137935181076,0.137931602371,0.137928023945,0.137924445799,0.137920867932,0.137917290344,0.137913713035,0.137910136006,0.137906559255,0.137902982783,0.137899406591,0.137895830677,0.137892255042,0.137888679686,0.137885104609,0.137881529811,0.137877955291,0.137874381051,0.137870807089,0.137867233405,0.13786366,0.137860086874,0.137856514026,0.137852941457,0.137849369167,0.137845797154,0.13784222542,0.137838653965,0.137835082788,0.137831511889,0.137827941268,0.137824370926,0.137820800862,0.137817231076,0.137813661568,0.137810092338,0.137806523386,0.137802954712,0.137799386316,0.137795818198,0.137792250358,0.137788682796,0.137785115512,0.137781548505,0.137777981776,0.137774415325,0.137770849152,0.137767283256,0.137763717638,0.137760152297,0.137756587234,0.137753022449,0.137749457941,0.13774589371,0.137742329757,0.137738766081,0.137735202682,0.137731639561,0.137728076717,0.13772451415,0.13772095186,0.137717389847,0.137713828112,0.137710266653,0.137706705472,0.137703144568,0.13769958394,0.13769602359,0.137692463516,0.137688903719,0.137685344199,0.137681784956,0.137678225989,0.1376746673,0.137671108887,0.13766755075,0.13766399289,0.137660435307,0.137656878,0.13765332097,0.137649764216,0.137646207739,0.137642651537,0.137639095613,0.137635539964,0.137631984592,0.137628429496,0.137624874677,0.137621320133,0.137617765866,0.137614211874,0.137610658159,0.13760710472,0.137603551557,0.137599998669,0.137596446058,0.137592893723,0.137589341663,0.137585789879,0.137582238371,0.137578687139,0.137575136182,0.137571585501,0.137568035096,0.137564484966,0.137560935112,0.137557385533,0.13755383623,0.137550287202,0.13754673845,0.137543189973,0.137539641771,0.137536093845,0.137532546194,0.137528998818,0.137525451718,0.137521904892,0.137518358342,0.137514812067,0.137511266066,0.137507720341,0.137504174891,0.137500629716,0.137497084816,0.13749354019,0.13748999584,0.137486451764,0.137482907963,0.137479364437,0.137475821185,0.137472278208,0.137468735506,0.137465193078,0.137461650925,0.137458109047,0.137454567442,0.137451026113,0.137447485058,0.137443944277,0.13744040377,0.137436863538,0.13743332358,0.137429783896,0.137426244487,0.137422705351,0.13741916649,0.137415627903,0.13741208959,0.137408551551,0.137405013786,0.137401476295,0.137397939078,0.137394402135,0.137390865465,0.137387329069,0.137383792948,0.1373802571,0.137376721525,0.137373186224,0.137369651197,0.137366116444,0.137362581964,0.137359047757,0.137355513825,0.137351980165,0.137348446779,0.137344913666,0.137341380827,0.137337848261,0.137334315968,0.137330783949,0.137327252203,0.13732372073,0.13732018953,0.137316658603,0.137313127949,0.137309597568,0.137306067461,0.137302537626,0.137299008064,0.137295478775,0.137291949759,0.137288421016,0.137284892545,0.137281364348,0.137277836423,0.13727430877,0.137270781391,0.137267254284,0.137263727449,0.137260200887,0.137256674598,0.137253148581,0.137249622836,0.137246097364,0.137242572164,0.137239047237,0.137235522582,0.137231998199,0.137228474088,0.137224950249,0.137221426683,0.137217903389,0.137214380367,0.137210857617,0.137207335139,0.137203812933,0.137200290998,0.137196769336,0.137193247946,0.137189726827,0.13718620598,0.137182685405,0.137179165102,0.137175645071,0.137172125311,0.137168605822,0.137165086606,0.137161567661,0.137158048987,0.137154530585,0.137151012454,0.137147494595,0.137143977007,0.137140459691,0.137136942645,0.137133425872,0.137129909369,0.137126393137,0.137122877177,0.137119361488,0.13711584607,0.137112330923,0.137108816047,0.137105301442,0.137101787108,0.137098273045,0.137094759253,0.137091245731,0.137087732481,0.137084219501,0.137080706792,0.137077194354,0.137073682186,0.137070170289,0.137066658663,0.137063147307,0.137059636222,0.137056125407,0.137052614863,0.137049104589,0.137045594586,0.137042084853,0.13703857539,0.137035066198,0.137031557276,0.137028048624,0.137024540242,0.137021032131,0.137017524289,0.137014016718,0.137010509417,0.137007002385,0.137003495624,0.136999989133,0.136996482912,0.13699297696,0.136989471278,0.136985965867,0.136982460724,0.136978955852,0.13697545125,0.136971946917,0.136968442853,0.13696493906,0.136961435535,0.136957932281,0.136954429296,0.13695092658,0.136947424134,0.136943921957,0.13694042005,0.136936918412,0.136933417043,0.136929915943,0.136926415113,0.136922914552,0.13691941426,0.136915914237,0.136912414483,0.136908914998,0.136905415783,0.136901916836,0.136898418158,0.136894919749,0.136891421609,0.136887923738,0.136884426136,0.136880928802,0.136877431737,0.136873934941,0.136870438414,0.136866942155,0.136863446165,0.136859950443,0.13685645499,0.136852959805,0.136849464889,0.136845970241,0.136842475862,0.136838981751,0.136835487908,0.136831994334,0.136828501028,0.13682500799,0.13682151522,0.136818022718,0.136814530485,0.13681103852,0.136807546822,0.136804055393,0.136800564232,0.136797073338,0.136793582713,0.136790092355,0.136786602266,0.136783112444,0.13677962289,0.136776133603,0.136772644584,0.136769155833,0.13676566735,0.136762179134,0.136758691186,0.136755203505,0.136751716092,0.136748228946,0.136744742068,0.136741255457,0.136737769114,0.136734283038,0.136730797229,0.136727311687,0.136723826413,0.136720341405,0.136716856665,0.136713372192,0.136709887986,0.136706404048,0.136702920376,0.136699436971,0.136695953833,0.136692470962,0.136688988358,0.136685506021,0.136682023951,0.136678542147,0.13667506061,0.13667157934,0.136668098337,0.1366646176,0.13666113713,0.136657656926,0.136654176989,0.136650697318,0.136647217914,0.136643738777,0.136640259905,0.1366367813,0.136633302962,0.13662982489,0.136626347084,0.136622869544,0.13661939227,0.136615915263,0.136612438522,0.136608962047,0.136605485838,0.136602009894,0.136598534217,0.136595058806,0.136591583661,0.136588108782,0.136584634169,0.136581159821,0.136577685739,0.136574211923,0.136570738373,0.136567265088,0.136563792069,0.136560319316,0.136556846828,0.136553374606,0.136549902649,0.136546430958,0.136542959532,0.136539488372,0.136536017477,0.136532546848,0.136529076483,0.136525606384,0.136522136551,0.136518666982,0.136515197679,0.136511728641,0.136508259868,0.13650479136,0.136501323117,0.136497855139,0.136494387426,0.136490919978,0.136487452795,0.136483985877,0.136480519223,0.136477052835,0.136473586711,0.136470120852,0.136466655257,0.136463189928,0.136459724863,0.136456260062,0.136452795527,0.136449331255,0.136445867248,0.136442403506,0.136438940028,0.136435476815,0.136432013866,0.136428551181,0.13642508876,0.136421626604,0.136418164712,0.136414703084,0.136411241721,0.136407780621,0.136404319786,0.136400859215,0.136397398907,0.136393938864,0.136390479085,0.13638701957,0.136383560318,0.13638010133,0.136376642607,0.136373184147,0.13636972595,0.136366268018,0.136362810349,0.136359352944,0.136355895803,0.136352438925,0.13634898231,0.13634552596,0.136342069872,0.136338614048,0.136335158488,0.136331703191,0.136328248157,0.136324793387,0.13632133888,0.136317884636,0.136314430656,0.136310976938,0.136307523484,0.136304070293,0.136300617365,0.1362971647,0.136293712298,0.136290260159,0.136286808283,0.13628335667,0.13627990532,0.136276454233,0.136273003408,0.136269552847,0.136266102548,0.136262652512,0.136259202738,0.136255753227,0.136252303979,0.136248854993,0.13624540627,0.13624195781,0.136238509612,0.136235061676,0.136231614003,0.136228166592,0.136224719444,0.136221272558,0.136217825934,0.136214379572,0.136210933473,0.136207487636,0.136204042061,0.136200596748,0.136197151697,0.136193706909,0.136190262382,0.136186818117,0.136183374114,0.136179930374,0.136176486895,0.136173043678,0.136169600722,0.136166158029,0.136162715597,0.136159273427,0.136155831519,0.136152389872,0.136148948487,0.136145507364,0.136142066502,0.136138625902,0.136135185563,0.136131745486,0.13612830567,0.136124866115,0.136121426822,0.13611798779,0.136114549019,0.13611111051,0.136107672262,0.136104234275,0.136100796549,0.136097359085,0.136093921881,0.136090484938,0.136087048257,0.136083611837,0.136080175677,0.136076739778,0.136073304141,0.136069868764,0.136066433648,0.136062998793,0.136059564198,0.136056129865,0.136052695792,0.136049261979,0.136045828428,0.136042395136,0.136038962106,0.136035529336,0.136032096826,0.136028664577,0.136025232589,0.13602180086,0.136018369392,0.136014938185,0.136011507238,0.136008076551,0.136004646124,0.136001215957,0.135997786051,0.135994356405,0.135990927018,0.135987497892,0.135984069026,0.13598064042,0.135977212074,0.135973783988,0.135970356162,0.135966928595,0.135963501289,0.135960074242,0.135956647455,0.135953220928,0.13594979466,0.135946368653,0.135942942904,0.135939517416,0.135936092187,0.135932667217,0.135929242507,0.135925818057,0.135922393866,0.135918969934,0.135915546262,0.135912122849,0.135908699695,0.135905276801,0.135901854166,0.13589843179,0.135895009673,0.135891587816,0.135888166217,0.135884744878,0.135881323798,0.135877902976,0.135874482414,0.135871062111,0.135867642066,0.135864222281,0.135860802754,0.135857383486,0.135853964477,0.135850545727,0.135847127235,0.135843709002,0.135840291028,0.135836873312,0.135833455855,0.135830038656,0.135826621716,0.135823205035,0.135819788612,0.135816372447,0.135812956541,0.135809540893,0.135806125504,0.135802710372,0.135799295499,0.135795880885,0.135792466528,0.13578905243,0.135785638589,0.135782225007,0.135778811683,0.135775398617,0.135771985809,0.135768573259,0.135765160967,0.135761748933,0.135758337156,0.135754925638,0.135751514377,0.135748103374,0.135744692629,0.135741282141,0.135737871912,0.135734461939,0.135731052225,0.135727642768,0.135724233568,0.135720824626,0.135717415942,0.135714007515,0.135710599345,0.135707191433,0.135703783778,0.135700376381,0.135696969241,0.135693562358,0.135690155732,0.135686749363,0.135683343252,0.135679937398,0.1356765318,0.13567312646,0.135669721377,0.135666316551,0.135662911982,0.13565950767,0.135656103614,0.135652699816,0.135649296274,0.135645892989,0.135642489961,0.13563908719,0.135635684675,0.135632282417,0.135628880416,0.135625478671,0.135622077183,0.135618675952,0.135615274976,0.135611874258,0.135608473796,0.13560507359,0.13560167364,0.135598273947,0.135594874511,0.13559147533,0.135588076406,0.135584677738,0.135581279326,0.135577881171,0.135574483271,0.135571085628,0.13556768824,0.135564291109,0.135560894234,0.135557497614,0.135554101251,0.135550705143,0.135547309291,0.135543913696,0.135540518355,0.135537123271,0.135533728443,0.13553033387,0.135526939552,0.135523545491,0.135520151685,0.135516758135,0.13551336484,0.1355099718,0.135506579016,0.135503186488,0.135499794215,0.135496402197,0.135493010435,0.135489618928,0.135486227676,0.13548283668,0.135479445939,0.135476055453,0.135472665222,0.135469275246,0.135465885525,0.13546249606,0.135459106849,0.135455717894,0.135452329193,0.135448940747,0.135445552557,0.135442164621,0.13543877694,0.135435389513,0.135432002342,0.135428615425,0.135425228763,0.135421842356,0.135418456203,0.135415070305,0.135411684661,0.135408299272,0.135404914138,0.135401529258,0.135398144632,0.135394760261,0.135391376144,0.135387992282,0.135384608674,0.13538122532,0.13537784222,0.135374459375,0.135371076784,0.135367694447,0.135364312364,0.135360930535,0.135357548961,0.13535416764,0.135350786573,0.135347405761,0.135344025202,0.135340644897,0.135337264846,0.135333885049,0.135330505506,0.135327126216,0.135323747181,0.135320368399,0.13531698987,0.135313611596,0.135310233575,0.135306855807,0.135303478293,0.135300101033,0.135296724026,0.135293347272,0.135289970772,0.135286594526,0.135283218533,0.135279842793,0.135276467306,0.135273092073,0.135269717093,0.135266342366,0.135262967892,0.135259593672,0.135256219704,0.13525284599,0.135249472528,0.13524609932,0.135242726365,0.135239353662,0.135235981213,0.135232609016,0.135229237073,0.135225865382,0.135222493944,0.135219122758,0.135215751826,0.135212381146,0.135209010719,0.135205640544,0.135202270622,0.135198900953,0.135195531536,0.135192162372,0.13518879346,0.1351854248,0.135182056393,0.135178688239,0.135175320337,0.135171952687,0.135168585289,0.135165218144,0.13516185125,0.135158484609,0.135155118221,0.135151752084,0.135148386199,0.135145020567,0.135141655186,0.135138290058,0.135134925181,0.135131560557,0.135128196184,0.135124832063,0.135121468194,0.135118104577,0.135114741212,0.135111378098,0.135108015237,0.135104652626,0.135101290268,0.135097928161,0.135094566306,0.135091204702,0.13508784335,0.135084482249,0.1350811214,0.135077760802,0.135074400456,0.135071040361,0.135067680517,0.135064320925,0.135060961584,0.135057602494,0.135054243656,0.135050885068,0.135047526732,0.135044168647,0.135040810813,0.13503745323,0.135034095898,0.135030738817,0.135027381987,0.135024025408,0.13502066908,0.135017313003,0.135013957176,0.135010601601,0.135007246276,0.135003891202,0.135000536378,0.134997181806,0.134993827484,0.134990473412,0.134987119591,0.134983766021,0.134980412701,0.134977059632,0.134973706813,0.134970354245,0.134967001927,0.13496364986,0.134960298042,0.134956946476,0.134953595159,0.134950244093,0.134946893276,0.13494354271,0.134940192395,0.134936842329,0.134933492513,0.134930142948,0.134926793632,0.134923444567,0.134920095751,0.134916747186,0.13491339887,0.134910050804,0.134906702988,0.134903355422,0.134900008106,0.134896661039,0.134893314222,0.134889967655,0.134886621337,0.134883275269,0.134879929451,0.134876583882,0.134873238563,0.134869893493,0.134866548672,0.134863204102,0.13485985978,0.134856515708,0.134853171885,0.134849828312,0.134846484988,0.134843141913,0.134839799087,0.134836456511,0.134833114183,0.134829772105,0.134826430276,0.134823088696,0.134819747365,0.134816406283,0.13481306545,0.134809724866,0.13480638453,0.134803044444,0.134799704606,0.134796365018,0.134793025678,0.134789686587,0.134786347744,0.134783009151,0.134779670806,0.134776332709,0.134772994861,0.134769657262,0.134766319911,0.134762982809,0.134759645955,0.13475630935,0.134752972993,0.134749636885,0.134746301025,0.134742965413,0.134739630049,0.134736294934,0.134732960067,0.134729625448,0.134726291078,0.134722956955,0.134719623081,0.134716289454,0.134712956076,0.134709622946,0.134706290063,0.134702957429,0.134699625042,0.134696292904,0.134692961013,0.13468962937,0.134686297975,0.134682966828,0.134679635928,0.134676305276,0.134672974872,0.134669644716,0.134666314807,0.134662985145,0.134659655732,0.134656326565,0.134652997646,0.134649668975,0.134646340551,0.134643012375,0.134639684445,0.134636356764,0.134633029329,0.134629702142,0.134626375202,0.134623048509,0.134619722063,0.134616395865,0.134613069914,0.134609744209,0.134606418752,0.134603093542,0.134599768579,0.134596443863,0.134593119393,0.134589795171,0.134586471196,0.134583147467,0.134579823985,0.13457650075,0.134573177762,0.13456985502,0.134566532525,0.134563210277,0.134559888276,0.134556566521,0.134553245012,0.13454992375,0.134546602735,0.134543281966,0.134539961444,0.134536641168,0.134533321138,0.134530001355,0.134526681818,0.134523362527,0.134520043483,0.134516724684,0.134513406132,0.134510087827,0.134506769767,0.134503451953,0.134500134386,0.134496817065,0.134493499989,0.13449018316,0.134486866576,0.134483550239,0.134480234147,0.134476918301,0.134473602701,0.134470287347,0.134466972239,0.134463657376,0.134460342759,0.134457028388,0.134453714263,0.134450400383,0.134447086748,0.13444377336,0.134440460216,0.134437147319,0.134433834666,0.13443052226,0.134427210098,0.134423898182,0.134420586511,0.134417275086,0.134413963906,0.134410652971,0.134407342282,0.134404031837,0.134400721638,0.134397411684,0.134394101975,0.134390792511,0.134387483292,0.134384174318,0.134380865589,0.134377557105,0.134374248866,0.134370940872,0.134367633123,0.134364325619,0.134361018359,0.134357711344,0.134354404574,0.134351098049,0.134347791768,0.134344485732,0.134341179941,0.134337874394,0.134334569092,0.134331264034,0.134327959221,0.134324654652,0.134321350328,0.134318046248,0.134314742412,0.134311438821,0.134308135474,0.134304832372,0.134301529513,0.134298226899,0.134294924529,0.134291622404,0.134288320522,0.134285018885,0.134281717491,0.134278416342,0.134275115437,0.134271814775,0.134268514358,0.134265214184,0.134261914255,0.134258614569,0.134255315127,0.134252015929,0.134248716975,0.134245418264,0.134242119798,0.134238821575,0.134235523595,0.134232225859,0.134228928367,0.134225631118,0.134222334113,0.134219037352,0.134215740833,0.134212444559,0.134209148527,0.13420585274,0.134202557195,0.134199261894,0.134195966836,0.134192672021,0.13418937745,0.134186083122,0.134182789037,0.134179495195,0.134176201596,0.13417290824,0.134169615127,0.134166322258,0.134163029631,0.134159737248,0.134156445107,0.134153153209,0.134149861554,0.134146570142,0.134143278973,0.134139988046,0.134136697363,0.134133406922,0.134130116723,0.134126826768,0.134123537055,0.134120247584,0.134116958356,0.134113669371,0.134110380628,0.134107092128,0.13410380387,0.134100515855,0.134097228082,0.134093940551,0.134090653263,0.134087366217,0.134084079413,0.134080792852,0.134077506533,0.134074220456,0.134070934621,0.134067649028,0.134064363677,0.134061078569,0.134057793702,0.134054509078,0.134051224695,0.134047940554,0.134044656656,0.134041372999,0.134038089584,0.134034806411,0.134031523479,0.13402824079,0.134024958342,0.134021676136,0.134018394172,0.134015112449,0.134011830968,0.134008549728,0.13400526873,0.134001987974,0.133998707459,0.133995427185,0.133992147153,0.133988867363,0.133985587814,0.133982308506,0.133979029439,0.133975750614,0.13397247203,0.133969193687,0.133965915586,0.133962637725,0.133959360106,0.133956082728,0.133952805591,0.133949528695,0.13394625204,0.133942975626,0.133939699453,0.133936423521,0.13393314783,0.13392987238,0.133926597171,0.133923322202,0.133920047474,0.133916772987,0.133913498741,0.133910224736,0.133906950971,0.133903677446,0.133900404163,0.13389713112,0.133893858317,0.133890585755,0.133887313434,0.133884041353,0.133880769512,0.133877497912,0.133874226552,0.133870955433,0.133867684554,0.133864413915,0.133861143517,0.133857873358,0.13385460344,0.133851333762,0.133848064324,0.133844795127,0.133841526169,0.133838257452,0.133834988974,0.133831720736,0.133828452739,0.133825184981,0.133821917464,0.133818650186,0.133815383148,0.13381211635,0.133808849791,0.133805583473,0.133802317394,0.133799051554,0.133795785955,0.133792520595,0.133789255475,0.133785990594,0.133782725953,0.133779461551,0.133776197389,0.133772933466,0.133769669783,0.133766406339,0.133763143135,0.13375988017,0.133756617444,0.133753354958,0.13375009271,0.133746830703,0.133743568934,0.133740307404,0.133737046114,0.133733785062,0.13373052425,0.133727263677,0.133724003343,0.133720743247,0.133717483391,0.133714223774,0.133710964396,0.133707705256,0.133704446355,0.133701187694,0.13369792927,0.133694671086,0.133691413141,0.133688155434,0.133684897966,0.133681640736,0.133678383745,0.133675126993,0.133671870479,0.133668614204,0.133665358167,0.133662102369,0.133658846809,0.133655591488,0.133652336405,0.13364908156,0.133645826954,0.133642572586,0.133639318456,0.133636064564,0.133632810911,0.133629557496,0.133626304319,0.13362305138,0.133619798679,0.133616546216,0.133613293992,0.133610042005,0.133606790256,0.133603538745,0.133600287473,0.133597036438,0.133593785641,0.133590535081,0.13358728476,0.133584034676,0.13358078483,0.133577535222,0.133574285851,0.133571036718,0.133567787823,0.133564539165,0.133561290745,0.133558042563,0.133554794618,0.13355154691,0.13354829944,0.133545052207,0.133541805212,0.133538558454,0.133535311933,0.13353206565,0.133528819604,0.133525573795,0.133522328223,0.133519082889,0.133515837792,0.133512592932,0.133509348309,0.133506103923,0.133502859774,0.133499615862,0.133496372187,0.133493128749,0.133489885548,0.133486642584,0.133483399857,0.133480157367,0.133476915113,0.133473673096,0.133470431316,0.133467189773,0.133463948466,0.133460707397,0.133457466563,0.133454225967,0.133450985607,0.133447745483,0.133444505596,0.133441265946,0.133438026532,0.133434787354,0.133431548413,0.133428309708,0.13342507124,0.133421833007,0.133418595012,0.133415357252,0.133412119729,0.133408882442,0.133405645391,0.133402408576,0.133399171997,0.133395935655,0.133392699548,0.133389463678,0.133386228043,0.133382992645,0.133379757482,0.133376522556,0.133373287865,0.13337005341,0.133366819191,0.133363585208,0.133360351461,0.133357117949,0.133353884673,0.133350651633,0.133347418828,0.133344186259,0.133340953926,0.133337721828,0.133334489966,0.133331258339,0.133328026948,0.133324795792,0.133321564872,0.133318334187,0.133315103737,0.133311873523,0.133308643544,0.1333054138,0.133302184292,0.133298955019,0.133295725981,0.133292497178,0.133289268611,0.133286040278,0.133282812181,0.133279584318,0.133276356691,0.133273129299,0.133269902142,0.133266675219,0.133263448532,0.133260222079,0.133256995861,0.133253769879,0.133250544131,0.133247318617,0.133244093339,0.133240868295,0.133237643486,0.133234418911,0.133231194571,0.133227970466,0.133224746596,0.133221522959,0.133218299558,0.133215076391,0.133211853458,0.13320863076,0.133205408296,0.133202186067,0.133198964071,0.133195742311,0.133192520784,0.133189299492,0.133186078434,0.13318285761,0.133179637021,0.133176416665,0.133173196544,0.133169976657,0.133166757003,0.133163537584,0.133160318399,0.133157099448,0.133153880731,0.133150662247,0.133147443998,0.133144225982,0.133141008201,0.133137790653,0.133134573339,0.133131356258,0.133128139412,0.133124922799,0.133121706419,0.133118490274,0.133115274362,0.133112058683,0.133108843238,0.133105628027,0.133102413049,0.133099198305,0.133095983794,0.133092769516,0.133089555472,0.133086341661,0.133083128083,0.133079914739,0.133076701628,0.13307348875,0.133070276106,0.133067063695,0.133063851516,0.133060639571,0.133057427859,0.13305421638,0.133051005135,0.133047794122,0.133044583342,0.133041372795,0.133038162481,0.1330349524,0.133031742552,0.133028532937,0.133025323554,0.133022114404,0.133018905487,0.133015696803,0.133012488352,0.133009280133,0.133006072146,0.133002864393,0.132999656872,0.132996449583,0.132993242527,0.132990035704,0.132986829113,0.132983622755,0.132980416629,0.132977210735,0.132974005074,0.132970799645,0.132967594448,0.132964389484,0.132961184751,0.132957980252,0.132954775984,0.132951571948,0.132948368145,0.132945164574,0.132941961234,0.132938758127,0.132935555252,0.132932352609,0.132929150198,0.132925948019,0.132922746071,0.132919544356,0.132916342872,0.132913141621,0.132909940601,0.132906739813,0.132903539256,0.132900338932,0.132897138839,0.132893938977,0.132890739348,0.13288753995,0.132884340783,0.132881141848,0.132877943145,0.132874744673,0.132871546432,0.132868348423,0.132865150646,0.1328619531,0.132858755785,0.132855558701,0.132852361849,0.132849165228,0.132845968838,0.13284277268,0.132839576752,0.132836381056,0.132833185591,0.132829990357,0.132826795354,0.132823600583,0.132820406042,0.132817211732,0.132814017653,0.132810823805,0.132807630188,0.132804436802,0.132801243647,0.132798050722,0.132794858029,0.132791665566,0.132788473334,0.132785281332,0.132782089562,0.132778898021,0.132775706712,0.132772515633,0.132769324785,0.132766134167,0.13276294378,0.132759753623,0.132756563697,0.132753374001,0.132750184536,0.132746995301,0.132743806296,0.132740617522,0.132737428978,0.132734240664,0.13273105258,0.132727864727,0.132724677104,0.132721489711,0.132718302548,0.132715115615,0.132711928912,0.13270874244,0.132705556197,0.132702370184,0.132699184402,0.132695998849,0.132692813526,0.132689628433,0.13268644357,0.132683258937,0.132680074533,0.132676890359,0.132673706415,0.132670522701,0.132667339217,0.132664155962,0.132660972936,0.132657790141,0.132654607574,0.132651425238,0.132648243131,0.132645061253,0.132641879605,0.132638698186,0.132635516997,0.132632336037,0.132629155307,0.132625974806,0.132622794534,0.132619614491,0.132616434678,0.132613255093,0.132610075738,0.132606896613,0.132603717716,0.132600539048,0.13259736061,0.1325941824,0.13259100442,0.132587826668,0.132584649146,0.132581471852,0.132578294788,0.132575117952,0.132571941345,0.132568764967,0.132565588818,0.132562412897,0.132559237205,0.132556061742,0.132552886508,0.132549711502,0.132546536725,0.132543362177,0.132540187857,0.132537013766,0.132533839903,0.132530666269,0.132527492863,0.132524319685,0.132521146736,0.132517974016,0.132514801524,0.13251162926,0.132508457224,0.132505285417,0.132502113838,0.132498942487,0.132495771365,0.13249260047,0.132489429804,0.132486259366,0.132483089156,0.132479919174,0.13247674942,0.132473579894,0.132470410596,0.132467241525,0.132464072683,0.132460904069,0.132457735683,0.132454567524,0.132451399593,0.13244823189,0.132445064415,0.132441897168,0.132438730148,0.132435563356,0.132432396792,0.132429230455,0.132426064346,0.132422898464,0.13241973281,0.132416567383,0.132413402184,0.132410237212,0.132407072468,0.132403907951,0.132400743662,0.1323975796,0.132394415765,0.132391252158,0.132388088777,0.132384925624,0.132381762699,0.1323786,0.132375437529,0.132372275284,0.132369113267,0.132365951477,0.132362789914,0.132359628578,0.132356467469,0.132353306587,0.132350145931,0.132346985503,0.132343825302,0.132340665327,0.13233750558,0.132334346059,0.132331186765,0.132328027697,0.132324868857,0.132321710243,0.132318551856,0.132315393695,0.132312235761,0.132309078054,0.132305920573,0.132302763318,0.132299606291,0.132296449489,0.132293292914,0.132290136566,0.132286980444,0.132283824548,0.132280668879,0.132277513436,0.132274358219,0.132271203228,0.132268048464,0.132264893926,0.132261739614,0.132258585528,0.132255431669,0.132252278035,0.132249124628,0.132245971446,0.132242818491,0.132239665762,0.132236513258,0.132233360981,0.132230208929,0.132227057103,0.132223905503,0.132220754129,0.132217602981,0.132214452059,0.132211301362,0.132208150891,0.132205000646,0.132201850626,0.132198700832,0.132195551264,0.132192401921,0.132189252804,0.132186103912,0.132182955246,0.132179806805,0.13217665859,0.1321735106,0.132170362835,0.132167215296,0.132164067982,0.132160920894,0.132157774031,0.132154627393,0.13215148098,0.132148334793,0.13214518883,0.132142043093,0.132138897581,0.132135752294,0.132132607232,0.132129462396,0.132126317784,0.132123173397,0.132120029235,0.132116885298,0.132113741586,0.132110598099,0.132107454837,0.1321043118,0.132101168987,0.132098026399,0.132094884036,0.132091741898,0.132088599984,0.132085458295,0.132082316831,0.132079175591,0.132076034576,0.132072893785,0.132069753219,0.132066612878,0.13206347276,0.132060332868,0.1320571932,0.132054053756,0.132050914536,0.132047775541,0.132044636771,0.132041498224,0.132038359902,0.132035221804,0.13203208393,0.132028946281,0.132025808855,0.132022671654,0.132019534677,0.132016397924,0.132013261395,0.13201012509,0.132006989009,0.132003853152,0.132000717519,0.131997582109,0.131994446924,0.131991311963,0.131988177225,0.131985042711,0.131981908421,0.131978774355,0.131975640513,0.131972506894,0.131969373499,0.131966240327,0.131963107379,0.131959974655,0.131956842155,0.131953709877,0.131950577824,0.131947445994,0.131944314387,0.131941183004,0.131938051844,0.131934920908,0.131931790195,0.131928659705,0.131925529438,0.131922399395,0.131919269575,0.131916139979,0.131913010605,0.131909881455,0.131906752528,0.131903623824,0.131900495343,0.131897367085,0.13189423905,0.131891111238,0.13188798365,0.131884856284,0.131881729141,0.131878602221,0.131875475524,0.131872349049,0.131869222798,0.131866096769,0.131862970963,0.13185984538,0.13185672002,0.131853594882,0.131850469967,0.131847345274,0.131844220804,0.131841096557,0.131837972533,0.13183484873,0.131831725151,0.131828601794,0.131825478659,0.131822355747,0.131819233057,0.131816110589,0.131812988344,0.131809866321,0.131806744521,0.131803622943,0.131800501587,0.131797380453,0.131794259541,0.131791138852,0.131788018385,0.131784898139,0.131781778116,0.131778658315,0.131775538736,0.131772419379,0.131769300244,0.131766181331,0.13176306264,0.131759944171,0.131756825924,0.131753707898,0.131750590094,0.131747472512,0.131744355152,0.131741238014,0.131738121097,0.131735004402,0.131731887929,0.131728771677,0.131725655647,0.131722539839,0.131719424252,0.131716308886,0.131713193743,0.13171007882,0.131706964119,0.13170384964,0.131700735382,0.131697621345,0.131694507529,0.131691393935,0.131688280563,0.131685167411,0.131682054481,0.131678941772,0.131675829284,0.131672717017,0.131669604972,0.131666493147,0.131663381544,0.131660270161,0.131657159,0.13165404806,0.13165093734,0.131647826842,0.131644716565,0.131641606508,0.131638496673,0.131635387058,0.131632277664,0.131629168491,0.131626059538,0.131622950807,0.131619842296,0.131616734005,0.131613625936,0.131610518087,0.131607410458,0.131604303051,0.131601195864,0.131598088897,0.131594982151,0.131591875625,0.13158876932,0.131585663235,0.131582557371,0.131579451727,0.131576346303,0.1315732411,0.131570136117,0.131567031354,0.131563926812,0.131560822489,0.131557718387,0.131554614505,0.131551510844,0.131548407402,0.13154530418,0.131542201179,0.131539098397,0.131535995836,0.131532893494,0.131529791373,0.131526689471,0.13152358779,0.131520486328,0.131517385086,0.131514284064,0.131511183262,0.131508082679,0.131504982316,0.131501882173,0.13149878225,0.131495682546,0.131492583062,0.131489483798,0.131486384753,0.131483285928,0.131480187322,0.131477088936,0.131473990769,0.131470892822,0.131467795095,0.131464697586,0.131461600297,0.131458503228,0.131455406377,0.131452309746,0.131449213335,0.131446117142,0.131443021169,0.131439925415,0.13143682988,0.131433734565,0.131430639468,0.131427544591,0.131424449933,0.131421355493,0.131418261273,0.131415167272,0.13141207349,0.131408979926,0.131405886582,0.131402793456,0.13139970055,0.131396607862,0.131393515393,0.131390423143,0.131387331111,0.131384239299,0.131381147705,0.131378056329,0.131374965173,0.131371874235,0.131368783515,0.131365693015,0.131362602732,0.131359512669,0.131356422823,0.131353333197,0.131350243789,0.131347154599,0.131344065627,0.131340976874,0.13133788834,0.131334800023,0.131331711925,0.131328624045,0.131325536384,0.131322448941,0.131319361716,0.131316274709,0.13131318792,0.131310101349,0.131307014997,0.131303928862,0.131300842946,0.131297757247,0.131294671767,0.131291586504,0.13128850146,0.131285416633,0.131282332024,0.131279247634,0.13127616346,0.131273079505,0.131269995768,0.131266912248,0.131263828946,0.131260745862,0.131257662995,0.131254580347,0.131251497915,0.131248415702,0.131245333706,0.131242251927,0.131239170366,0.131236089023,0.131233007897,0.131229926988,0.131226846297,0.131223765823,0.131220685567,0.131217605528,0.131214525706,0.131211446102,0.131208366715,0.131205287545,0.131202208592,0.131199129857,0.131196051339,0.131192973038,0.131189894954,0.131186817087,0.131183739437,0.131180662004,0.131177584788,0.13117450779,0.131171431008,0.131168354443,0.131165278095,0.131162201964,0.13115912605,0.131156050352,0.131152974872,0.131149899608,0.131146824561,0.131143749731,0.131140675117,0.131137600721,0.13113452654,0.131131452577,0.13112837883,0.131125305299,0.131122231986,0.131119158888,0.131116086008,0.131113013343,0.131109940895,0.131106868664,0.131103796649,0.13110072485,0.131097653268,0.131094581902,0.131091510753,0.131088439819,0.131085369102,0.131082298601,0.131079228317,0.131076158248,0.131073088396,0.13107001876,0.131066949339,0.131063880135,0.131060811147,0.131057742375,0.131054673819,0.131051605479,0.131048537355,0.131045469447,0.131042401755,0.131039334279,0.131036267018,0.131033199973,0.131030133144,0.131027066531,0.131024000134,0.131020933952,0.131017867986,0.131014802236,0.131011736701,0.131008671382,0.131005606279,0.131002541391,0.130999476719,0.130996412262,0.13099334802,0.130990283994,0.130987220184,0.130984156589,0.130981093209,0.130978030045,0.130974967096,0.130971904363,0.130968841844,0.130965779541,0.130962717453,0.130959655581,0.130956593923,0.130953532481,0.130950471254,0.130947410242,0.130944349445,0.130941288863,0.130938228496,0.130935168345,0.130932108408,0.130929048686,0.130925989179,0.130922929887,0.13091987081,0.130916811947,0.1309137533,0.130910694867,0.130907636649,0.130904578646,0.130901520858,0.130898463284,0.130895405925,0.130892348781,0.130889291851,0.130886235136,0.130883178636,0.13088012235,0.130877066278,0.130874010422,0.130870954779,0.130867899351,0.130864844138,0.130861789138,0.130858734354,0.130855679783,0.130852625427,0.130849571286,0.130846517358,0.130843463645,0.130840410146,0.130837356861,0.13083430379,0.130831250934,0.130828198292,0.130825145864,0.130822093649,0.130819041649,0.130815989863,0.130812938291,0.130809886933,0.130806835789,0.130803784859,0.130800734143,0.13079768364,0.130794633352,0.130791583277,0.130788533416,0.130785483769,0.130782434336,0.130779385116,0.13077633611,0.130773287318,0.130770238739,0.130767190374,0.130764142223,0.130761094285,0.130758046561,0.13075499905,0.130751951753,0.130748904669,0.130745857799,0.130742811142,0.130739764698,0.130736718468,0.130733672452,0.130730626648,0.130727581058,0.130724535681,0.130721490518,0.130718445568,0.13071540083,0.130712356306,0.130709311996,0.130706267898,0.130703224013,0.130700180342,0.130697136884,0.130694093638,0.130691050606,0.130688007786,0.13068496518,0.130681922787,0.130678880606,0.130675838638,0.130672796883,0.130669755341,0.130666714012,0.130663672896,0.130660631992,0.130657591301,0.130654550823,0.130651510557,0.130648470504,0.130645430664,0.130642391036,0.130639351621,0.130636312419,0.130633273428,0.130630234651,0.130627196086,0.130624157733,0.130621119593,0.130618081666,0.13061504395,0.130612006447,0.130608969157,0.130605932078,0.130602895212,0.130599858558,0.130596822117,0.130593785887,0.13059074987,0.130587714065,0.130584678472,0.130581643092,0.130578607923,0.130575572966,0.130572538222,0.130569503689,0.130566469369,0.13056343526,0.130560401363,0.130557367679,0.130554334206,0.130551300945,0.130548267896,0.130545235058,0.130542202433,0.130539170019,0.130536137817,0.130533105826,0.130530074048,0.130527042481,0.130524011125,0.130520979982,0.130517949049,0.130514918329,0.13051188782,0.130508857522,0.130505827436,0.130502797562,0.130499767899,0.130496738447,0.130493709207,0.130490680178,0.13048765136,0.130484622754,0.130481594359,0.130478566175,0.130475538203,0.130472510442,0.130469482891,0.130466455553,0.130463428425,0.130460401508,0.130457374803,0.130454348308,0.130451322025,0.130448295953,0.130445270091,0.130442244441,0.130439219002,0.130436193773,0.130433168756,0.130430143949,0.130427119353,0.130424094968,0.130421070794,0.13041804683,0.130415023078,0.130411999536,0.130408976204,0.130405953084,0.130402930174,0.130399907475,0.130396884986,0.130393862708,0.13039084064,0.130387818783,0.130384797137,0.130381775701,0.130378754475,0.13037573346,0.130372712656,0.130369692061,0.130366671677,0.130363651504,0.13036063154,0.130357611787,0.130354592245,0.130351572912,0.13034855379,0.130345534878,0.130342516176,0.130339497684,0.130336479402,0.130333461331,0.130330443469,0.130327425818,0.130324408377,0.130321391145,0.130318374124,0.130315357312,0.130312340711,0.130309324319,0.130306308137,0.130303292165,0.130300276403,0.130297260851,0.130294245508,0.130291230375,0.130288215452,0.130285200739,0.130282186235,0.130279171941,0.130276157857,0.130273143982,0.130270130317,0.130267116861,0.130264103615,0.130261090578,0.130258077751,0.130255065133,0.130252052725,0.130249040526,0.130246028537,0.130243016757,0.130240005186,0.130236993824,0.130233982672,0.130230971729,0.130227960996,0.130224950471,0.130221940156,0.13021893005,0.130215920153,0.130212910465,0.130209900987,0.130206891717,0.130203882656,0.130200873805,0.130197865162,0.130194856729,0.130191848504,0.130188840488,0.130185832681,0.130182825084,0.130179817694,0.130176810514,0.130173803543,0.13017079678,0.130167790226,0.130164783881,0.130161777745,0.130158771817,0.130155766098,0.130152760587,0.130149755285,0.130146750192,0.130143745307,0.130140740631,0.130137736163,0.130134731904,0.130131727853,0.130128724011,0.130125720377,0.130122716952,0.130119713735,0.130116710726,0.130113707925,0.130110705333,0.130107702949,0.130104700774,0.130101698806,0.130098697047,0.130095695496,0.130092694153,0.130089693019,0.130086692092,0.130083691373,0.130080690863,0.13007769056,0.130074690466,0.130071690579,0.130068690901,0.13006569143,0.130062692168,0.130059693113,0.130056694266,0.130053695627,0.130050697196,0.130047698972,0.130044700956,0.130041703148,0.130038705548,0.130035708156,0.130032710971,0.130029713994,0.130026717224,0.130023720662,0.130020724307,0.130017728161,0.130014732221,0.130011736489,0.130008740965,0.130005745648,0.130002750538,0.129999755636,0.129996760942,0.129993766454,0.129990772174,0.129987778102,0.129984784236,0.129981790578,0.129978797127,0.129975803883,0.129972810847,0.129969818017,0.129966825395,0.12996383298,0.129960840772,0.129957848771,0.129954856977,0.12995186539,0.129948874011,0.129945882838,0.129942891872,0.129939901113,0.129936910561,0.129933920215,0.129930930077,0.129927940146,0.129924950421,0.129921960903,0.129918971592,0.129915982487,0.12991299359,0.129910004898,0.129907016414,0.129904028136,0.129901040065,0.129898052201,0.129895064543,0.129892077091,0.129889089846,0.129886102808,0.129883115976,0.129880129351,0.129877142932,0.129874156719,0.129871170713,0.129868184913,0.129865199319,0.129862213932,0.129859228751,0.129856243776,0.129853259008,0.129850274445,0.129847290089,0.129844305939,0.129841321995,0.129838338258,0.129835354726,0.129832371401,0.129829388281,0.129826405368,0.12982342266,0.129820440159,0.129817457863,0.129814475774,0.12981149389,0.129808512212,0.12980553074,0.129802549474,0.129799568414,0.129796587559,0.12979360691,0.129790626467,0.12978764623,0.129784666198,0.129781686372,0.129778706752,0.129775727337,0.129772748128,0.129769769124,0.129766790326,0.129763811734,0.129760833347,0.129757855165,0.129754877189,0.129751899419,0.129748921853,0.129745944494,0.129742967339,0.12973999039,0.129737013646,0.129734037107,0.129731060774,0.129728084646,0.129725108723,0.129722133006,0.129719157493,0.129716182186,0.129713207083,0.129710232186,0.129707257494,0.129704283007,0.129701308725,0.129698334648,0.129695360776,0.129692387109,0.129689413647,0.12968644039,0.129683467338,0.12968049449,0.129677521848,0.12967454941,0.129671577177,0.129668605148,0.129665633325,0.129662661706,0.129659690292,0.129656719083,0.129653748078,0.129650777278,0.129647806682,0.129644836291,0.129641866105,0.129638896123,0.129635926346,0.129632956773,0.129629987404,0.12962701824,0.129624049281,0.129621080525,0.129618111975,0.129615143628,0.129612175486,0.129609207548,0.129606239815,0.129603272285,0.12960030496,0.129597337839,0.129594370922,0.12959140421,0.129588437701,0.129585471397,0.129582505297,0.129579539401,0.129576573708,0.12957360822,0.129570642936,0.129567677856,0.12956471298,0.129561748307,0.129558783839,0.129555819575,0.129552855514,0.129549891657,0.129546928004,0.129543964555,0.129541001309,0.129538038267,0.129535075429,0.129532112795,0.129529150364,0.129526188137,0.129523226114,0.129520264294,0.129517302677,0.129514341265,0.129511380055,0.12950841905,0.129505458247,0.129502497648,0.129499537253,0.129496577061,0.129493617073,0.129490657287,0.129487697705,0.129484738327,0.129481779152,0.129478820179,0.129475861411,0.129472902845,0.129469944483,0.129466986324,0.129464028367,0.129461070615,0.129458113065,0.129455155718,0.129452198574,0.129449241634,0.129446284896,0.129443328361,0.12944037203,0.129437415901,0.129434459975,0.129431504252,0.129428548732,0.129425593415,0.1294226383,0.129419683389,0.12941672868,0.129413774174,0.129410819871,0.12940786577,0.129404911872,0.129401958177,0.129399004684,0.129396051394,0.129393098307,0.129390145422,0.129387192739,0.12938424026,0.129381287982,0.129378335908,0.129375384035,0.129372432365,0.129369480898,0.129366529633,0.12936357857,0.129360627709,0.129357677051,0.129354726595,0.129351776342,0.12934882629,0.129345876441,0.129342926794,0.129339977349,0.129337028107,0.129334079066,0.129331130228,0.129328181592,0.129325233157,0.129322284925,0.129319336895,0.129316389067,0.129313441441,0.129310494016,0.129307546794,0.129304599773,0.129301652955,0.129298706338,0.129295759923,0.12929281371,0.129289867699,0.129286921889,0.129283976281,0.129281030875,0.129278085671,0.129275140668,0.129272195867,0.129269251267,0.129266306869,0.129263362673,0.129260418678,0.129257474885,0.129254531293,0.129251587903,0.129248644714,0.129245701727,0.129242758941,0.129239816356,0.129236873973,0.129233931791,0.12923098981,0.129228048031,0.129225106453,0.129222165076,0.129219223901,0.129216282927,0.129213342154,0.129210401582,0.129207461211,0.129204521041,0.129201581072,0.129198641305,0.129195701738,0.129192762373,0.129189823208,0.129186884245,0.129183945482,0.129181006921,0.12917806856,0.1291751304,0.129172192441,0.129169254683,0.129166317126,0.129163379769,0.129160442613,0.129157505658,0.129154568904,0.12915163235,0.129148695997,0.129145759845,0.129142823894,0.129139888143,0.129136952592,0.129134017242,0.129131082093,0.129128147144,0.129125212396,0.129122277848,0.1291193435,0.129116409353,0.129113475407,0.129110541661,0.129107608115,0.129104674769,0.129101741624,0.129098808679,0.129095875935,0.12909294339,0.129090011046,0.129087078902,0.129084146958,0.129081215215,0.129078283671,0.129075352328,0.129072421184,0.129069490241,0.129066559498,0.129063628954,0.129060698611,0.129057768468,0.129054838525,0.129051908781,0.129048979238,0.129046049894,0.12904312075,0.129040191806,0.129037263062,0.129034334518,0.129031406173,0.129028478029,0.129025550083,0.129022622338,0.129019694792,0.129016767446,0.1290138403,0.129010913353,0.129007986606,0.129005060058,0.12900213371,0.128999207561,0.128996281612,0.128993355862,0.128990430312,0.128987504961,0.128984579809,0.128981654857,0.128978730105,0.128975805551,0.128972881197,0.128969957042,0.128967033087,0.128964109331,0.128961185774,0.128958262416,0.128955339257,0.128952416297,0.128949493537,0.128946570976,0.128943648613,0.12894072645,0.128937804486,0.128934882721,0.128931961155,0.128929039788,0.12892611862,0.12892319765,0.12892027688,0.128917356308,0.128914435936,0.128911515762,0.128908595787,0.128905676011,0.128902756433,0.128899837055,0.128896917875,0.128893998893,0.128891080111,0.128888161527,0.128885243141,0.128882324955,0.128879406967,0.128876489177,0.128873571586,0.128870654193,0.128867736999,0.128864820004,0.128861903207,0.128858986608,0.128856070208,0.128853154006,0.128850238002,0.128847322197,0.12884440659,0.128841491182,0.128838575971,0.128835660959,0.128832746146,0.12882983153,0.128826917112,0.128824002893,0.128821088872,0.128818175049,0.128815261424,0.128812347997,0.128809434768,0.128806521737,0.128803608904,0.12880069627,0.128797783833,0.128794871594,0.128791959553,0.12878904771,0.128786136064,0.128783224617,0.128780313367,0.128777402315,0.128774491461,0.128771580805,0.128768670346,0.128765760085,0.128762850022,0.128759940157,0.128757030489,0.128754121018,0.128751211746,0.128748302671,0.128745393793,0.128742485113,0.128739576631,0.128736668345,0.128733760258,0.128730852368,0.128727944675,0.12872503718,0.128722129882,0.128719222781,0.128716315878,0.128713409172,0.128710502663,0.128707596352,0.128704690237,0.12870178432,0.128698878601,0.128695973078,0.128693067753,0.128690162624,0.128687257693,0.128684352959,0.128681448422,0.128678544082,0.128675639939,0.128672735992,0.128669832243,0.128666928691,0.128664025336,0.128661122178,0.128658219216,0.128655316452,0.128652413884,0.128649511513,0.128646609339,0.128643707361,0.128640805581,0.128637903997,0.12863500261,0.128632101419,0.128629200425,0.128626299628,0.128623399027,0.128620498623,0.128617598416,0.128614698405,0.12861179859,0.128608898973,0.128605999551,0.128603100326,0.128600201298,0.128597302465,0.12859440383,0.12859150539,0.128588607147,0.128585709101,0.12858281125,0.128579913596,0.128577016138,0.128574118877,0.128571221811,0.128568324942,0.128565428269,0.128562531792,0.128559635511,0.128556739427,0.128553843538,0.128550947845,0.128548052349,0.128545157048,0.128542261944,0.128539367035,0.128536472323,0.128533577806,0.128530683485,0.128527789361,0.128524895431,0.128522001698,0.128519108161,0.128516214819,0.128513321673,0.128510428723,0.128507535969,0.12850464341,0.128501751047,0.12849885888,0.128495966908,0.128493075132,0.128490183552,0.128487292167,0.128484400978,0.128481509984,0.128478619186,0.128475728583,0.128472838175,0.128469947963,0.128467057947,0.128464168126,0.1284612785,0.12845838907,0.128455499835,0.128452610795,0.12844972195,0.128446833301,0.128443944847,0.128441056588,0.128438168525,0.128435280656,0.128432392983,0.128429505505,0.128426618222,0.128423731134,0.128420844241,0.128417957543,0.128415071041,0.128412184733,0.12840929862,0.128406412702,0.128403526979,0.128400641451,0.128397756118,0.12839487098,0.128391986036,0.128389101287,0.128386216734,0.128383332375,0.12838044821,0.128377564241,0.128374680466,0.128371796886,0.1283689135,0.128366030309,0.128363147313,0.128360264511,0.128357381904,0.128354499492,0.128351617274,0.128348735251,0.128345853422,0.128342971787,0.128340090347,0.128337209101,0.12833432805,0.128331447193,0.128328566531,0.128325686063,0.128322805789,0.12831992571,0.128317045824,0.128314166133,0.128311286637,0.128308407334,0.128305528226,0.128302649311,0.128299770591,0.128296892065,0.128294013734,0.128291135596,0.128288257652,0.128285379902,0.128282502347,0.128279624985,0.128276747817,0.128273870844,0.128270994064,0.128268117478,0.128265241086,0.128262364887,0.128259488883,0.128256613073,0.128253737456,0.128250862033,0.128247986803,0.128245111768,0.128242236926,0.128239362278,0.128236487824,0.128233613563,0.128230739495,0.128227865622,0.128224991942,0.128222118455,0.128219245162,0.128216372063,0.128213499157,0.128210626444,0.128207753925,0.1282048816,0.128202009467,0.128199137529,0.128196265783,0.128193394231,0.128190522872,0.128187651706,0.128184780734,0.128181909955,0.128179039369,0.128176168977,0.128173298777,0.128170428771,0.128167558958,0.128164689338,0.128161819911,0.128158950677,0.128156081636,0.128153212788,0.128150344134,0.128147475672,0.128144607403,0.128141739327,0.128138871444,0.128136003754,0.128133136257,0.128130268953,0.128127401841,0.128124534923,0.128121668197,0.128118801664,0.128115935323,0.128113069176,0.128110203221,0.128107337458,0.128104471889,0.128101606512,0.128098741328,0.128095876336,0.128093011537,0.12809014693,0.128087282516,0.128084418295,0.128081554266,0.128078690429,0.128075826785,0.128072963333,0.128070100074,0.128067237007,0.128064374132,0.12806151145,0.12805864896,0.128055786663,0.128052924557,0.128050062644,0.128047200923,0.128044339395,0.128041478058,0.128038616914,0.128035755962,0.128032895202,0.128030034634,0.128027174258,0.128024314074,0.128021454082,0.128018594283,0.128015734675,0.128012875259,0.128010016035,0.128007157003,0.128004298163,0.128001439515,0.127998581059,0.127995722795,0.127992864722,0.127990006841,0.127987149152,0.127984291655,0.12798143435,0.127978577236,0.127975720314,0.127972863583,0.127970007044,0.127967150697,0.127964294542,0.127961438578,0.127958582805,0.127955727224,0.127952871835,0.127950016637,0.127947161631,0.127944306816,0.127941452193,0.12793859776,0.12793574352,0.127932889471,0.127930035613,0.127927181946,0.127924328471,0.127921475187,0.127918622094,0.127915769192,0.127912916482,0.127910063963,0.127907211635,0.127904359498,0.127901507553,0.127898655798,0.127895804235,0.127892952862,0.127890101681,0.127887250691,0.127884399891,0.127881549283,0.127878698866,0.127875848639,0.127872998604,0.127870148759,0.127867299106,0.127864449643,0.127861600371,0.12785875129,0.127855902399,0.1278530537,0.127850205191,0.127847356873,0.127844508745,0.127841660808,0.127838813062,0.127835965507,0.127833118142,0.127830270968,0.127827423984,0.127824577191,0.127821730589,0.127818884177,0.127816037955,0.127813191924,0.127810346083,0.127807500433,0.127804654974,0.127801809704,0.127798964625,0.127796119736,0.127793275038,0.12779043053,0.127787586212,0.127784742085,0.127781898148,0.127779054401,0.127776210844,0.127773367477,0.127770524301,0.127767681314,0.127764838518,0.127761995912,0.127759153496,0.12775631127,0.127753469234,0.127750627388,0.127747785732,0.127744944265,0.127742102989,0.127739261903,0.127736421007,0.1277335803,0.127730739784,0.127727899457,0.12772505932,0.127722219373,0.127719379616,0.127716540048,0.12771370067,0.127710861482,0.127708022483,0.127705183675,0.127702345055,0.127699506626,0.127696668386,0.127693830336,0.127690992475,0.127688154804,0.127685317322,0.12768248003,0.127679642927,0.127676806013,0.12767396929,0.127671132755,0.12766829641,0.127665460254,0.127662624288,0.127659788511,0.127656952923,0.127654117525,0.127651282316,0.127648447296,0.127645612465,0.127642777823,0.127639943371,0.127637109108,0.127634275034,0.127631441149,0.127628607453,0.127625773946,0.127622940629,0.1276201075,0.12761727456,0.12761444181,0.127611609248,0.127608776875,0.127605944692,0.127603112697,0.127600280891,0.127597449274,0.127594617845,0.127591786606,0.127588955555,0.127586124693,0.12758329402,0.127580463536,0.12757763324,0.127574803133,0.127571973215,0.127569143485,0.127566313944,0.127563484592,0.127560655428,0.127557826453,0.127554997666,0.127552169068,0.127549340658,0.127546512437,0.127543684404,0.12754085656,0.127538028904,0.127535201437,0.127532374158,0.127529547067,0.127526720165,0.127523893451,0.127521066925,0.127518240587,0.127515414438,0.127512588477,0.127509762704,0.12750693712,0.127504111724,0.127501286515,0.127498461495,0.127495636663,0.127492812019,0.127489987563,0.127487163296,0.127484339216,0.127481515324,0.12747869162,0.127475868104,0.127473044777,0.127470221637,0.127467398685,0.12746457592,0.127461753344,0.127458930956,0.127456108755,0.127453286742,0.127450464917,0.12744764328,0.12744482183,0.127442000568,0.127439179494,0.127436358607,0.127433537909,0.127430717397,0.127427897074,0.127425076938,0.127422256989,0.127419437228,0.127416617655,0.127413798269,0.12741097907,0.127408160059,0.127405341236,0.1274025226,0.127399704151,0.12739688589,0.127394067816,0.127391249929,0.12738843223,0.127385614718,0.127382797393,0.127379980256,0.127377163305,0.127374346542,0.127371529967,0.127368713578,0.127365897376,0.127363081362,0.127360265535,0.127357449894,0.127354634441,0.127351819175,0.127349004096,0.127346189204,0.127343374499,0.127340559981,0.12733774565,0.127334931506,0.127332117548,0.127329303778,0.127326490194,0.127323676797,0.127320863587,0.127318050564,0.127315237728,0.127312425078,0.127309612615,0.127306800339,0.12730398825,0.127301176347,0.127298364631,0.127295553101,0.127292741758,0.127289930602,0.127287119632,0.127284308849,0.127281498253,0.127278687842,0.127275877619,0.127273067582,0.127270257731,0.127267448067,0.127264638589,0.127261829297,0.127259020192,0.127256211273,0.127253402541,0.127250593994,0.127247785634,0.127244977461,0.127242169473,0.127239361672,0.127236554057,0.127233746628,0.127230939386,0.127228132329,0.127225325459,0.127222518774,0.127219712276,0.127216905964,0.127214099838,0.127211293898,0.127208488144,0.127205682575,0.127202877193,0.127200071997,0.127197266986,0.127194462162,0.127191657523,0.127188853071,0.127186048804,0.127183244722,0.127180440827,0.127177637118,0.127174833594,0.127172030256,0.127169227103,0.127166424136,0.127163621355,0.12716081876,0.12715801635,0.127155214126,0.127152412087,0.127149610234,0.127146808567,0.127144007085,0.127141205788,0.127138404677,0.127135603752,0.127132803011,0.127130002457,0.127127202087,0.127124401903,0.127121601905,0.127118802092,0.127116002464,0.127113203021,0.127110403764,0.127107604692,0.127104805805,0.127102007103,0.127099208587,0.127096410255,0.127093612109,0.127090814148,0.127088016372,0.127085218781,0.127082421376,0.127079624155,0.127076827119,0.127074030269,0.127071233603,0.127068437122,0.127065640826,0.127062844716,0.12706004879,0.127057253049,0.127054457492,0.127051662121,0.127048866934,0.127046071933,0.127043277116,0.127040482484,0.127037688036,0.127034893774,0.127032099696,0.127029305802,0.127026512094,0.12702371857,0.12702092523,0.127018132075,0.127015339105,0.127012546319,0.127009753718,0.127006961302,0.12700416907,0.127001377022,0.126998585159,0.12699579348,0.126993001986,0.126990210676,0.12698741955,0.126984628609,0.126981837852,0.12697904728,0.126976256891,0.126973466687,0.126970676668,0.126967886832,0.126965097181,0.126962307714,0.126959518431,0.126956729332,0.126953940417,0.126951151687,0.126948363141,0.126945574778,0.1269427866,0.126939998606,0.126937210795,0.126934423169,0.126931635727,0.126928848468,0.126926061394,0.126923274503,0.126920487797,0.126917701274,0.126914914935,0.12691212878,0.126909342809,0.126906557021,0.126903771418,0.126900985998,0.126898200761,0.126895415709,0.12689263084,0.126889846155,0.126887061653,0.126884277335,0.126881493201,0.12687870925,0.126875925483,0.126873141899,0.126870358499,0.126867575283,0.12686479225,0.1268620094,0.126859226734,0.126856444251,0.126853661951,0.126850879835,0.126848097903,0.126845316154,0.126842534588,0.126839753205,0.126836972006,0.126834190989,0.126831410156,0.126828629507,0.12682584904,0.126823068757,0.126820288657,0.12681750874,0.126814729006,0.126811949455,0.126809170088,0.126806390903,0.126803611901,0.126800833083,0.126798054447,0.126795275995,0.126792497725,0.126789719638,0.126786941735,0.126784164014,0.126781386476,0.126778609121,0.126775831948,0.126773054959,0.126770278152,0.126767501528,0.126764725087,0.126761948829,0.126759172753,0.12675639686,0.12675362115,0.126750845622,0.126748070277,0.126745295115,0.126742520135,0.126739745338,0.126736970723,0.126734196291,0.126731422042,0.126728647974,0.12672587409,0.126723100388,0.126720326868,0.126717553531,0.126714780376,0.126712007403,0.126709234613,0.126706462005,0.12670368958,0.126700917336,0.126698145275,0.126695373397,0.1266926017,0.126689830186,0.126687058854,0.126684287704,0.126681516736,0.126678745951,0.126675975347,0.126673204926,0.126670434687,0.12666766463,0.126664894754,0.126662125061,0.12665935555,0.126656586221,0.126653817074,0.126651048108,0.126648279325,0.126645510724,0.126642742304,0.126639974066,0.12663720601,0.126634438136,0.126631670444,0.126628902933,0.126626135605,0.126623368458,0.126620601492,0.126617834709,0.126615068107,0.126612301686,0.126609535448,0.126606769391,0.126604003515,0.126601237822,0.126598472309,0.126595706979,0.126592941829,0.126590176862,0.126587412075,0.126584647471,0.126581883047,0.126579118805,0.126576354745,0.126573590866,0.126570827168,0.126568063651,0.126565300316,0.126562537163,0.12655977419,0.126557011399,0.126554248789,0.12655148636,0.126548724112,0.126545962046,0.12654320016,0.126540438456,0.126537676933,0.126534915591,0.126532154431,0.126529393451,0.126526632652,0.126523872034,0.126521111598,0.126518351342,0.126515591267,0.126512831373,0.12651007166,0.126507312128,0.126504552777,0.126501793607,0.126499034618,0.126496275809,0.126493517181,0.126490758734,0.126488000468,0.126485242383,0.126482484478,0.126479726754,0.126476969211,0.126474211848,0.126471454666,0.126468697664,0.126465940844,0.126463184203,0.126460427744,0.126457671465,0.126454915366,0.126452159448,0.12644940371,0.126446648153,0.126443892777,0.12644113758,0.126438382564,0.126435627729,0.126432873074,0.126430118599,0.126427364305,0.126424610191,0.126421856257,0.126419102503,0.12641634893,0.126413595537,0.126410842324,0.126408089292,0.126405336439,0.126402583767,0.126399831275,0.126397078963,0.126394326831,0.126391574879,0.126388823107,0.126386071515,0.126383320103,0.126380568872,0.12637781782,0.126375066948,0.126372316256,0.126369565744,0.126366815412,0.12636406526,0.126361315287,0.126358565495,0.126355815882,0.126353066449,0.126350317196,0.126347568123,0.126344819229,0.126342070515,0.126339321981,0.126336573626,0.126333825451,0.126331077456,0.126328329641,0.126325582005,0.126322834548,0.126320087271,0.126317340174,0.126314593256,0.126311846518,0.126309099959,0.12630635358,0.12630360738,0.126300861359,0.126298115518,0.126295369857,0.126292624374,0.126289879071,0.126287133948,0.126284389003,0.126281644238,0.126278899653,0.126276155246,0.126273411019,0.126270666971,0.126267923102,0.126265179412,0.126262435902,0.12625969257,0.126256949418,0.126254206445,0.126251463651,0.126248721036,0.1262459786,0.126243236343,0.126240494265,0.126237752365,0.126235010645,0.126232269104,0.126229527742,0.126226786559,0.126224045554,0.126221304729,0.126218564082,0.126215823614,0.126213083325,0.126210343214,0.126207603283,0.12620486353,0.126202123956,0.12619938456,0.126196645344,0.126193906306,0.126191167446,0.126188428765,0.126185690263,0.12618295194,0.126180213795,0.126177475828,0.12617473804,0.126172000431,0.126169263,0.126166525747,0.126163788673,0.126161051778,0.12615831506,0.126155578522,0.126152842161,0.126150105979,0.126147369975,0.12614463415,0.126141898503,0.126139163034,0.126136427743,0.126133692631,0.126130957697,0.126128222941,0.126125488363,0.126122753964,0.126120019742,0.126117285699,0.126114551834,0.126111818147,0.126109084637,0.126106351306,0.126103618153,0.126100885178,0.126098152381,0.126095419762,0.126092687321,0.126089955058,0.126087222973,0.126084491065,0.126081759336,0.126079027784,0.12607629641,0.126073565214,0.126070834196,0.126068103356,0.126065372693,0.126062642208,0.126059911901,0.126057181771,0.126054451819,0.126051722045,0.126048992449,0.12604626303,0.126043533788,0.126040804724,0.126038075838,0.12603534713,0.126032618598,0.126029890245,0.126027162069,0.12602443407,0.126021706249,0.126018978605,0.126016251138,0.126013523849,0.126010796738,0.126008069803,0.126005343046,0.126002616467,0.125999890064,0.125997163839,0.125994437791,0.125991711921,0.125988986227,0.125986260711,0.125983535372,0.12598081021,0.125978085225,0.125975360418,0.125972635787,0.125969911334,0.125967187057,0.125964462958,0.125961739035,0.12595901529,0.125956291722,0.12595356833,0.125950845116,0.125948122078,0.125945399218,0.125942676534,0.125939954027,0.125937231697,0.125934509544,0.125931787568,0.125929065768,0.125926344145,0.125923622699,0.12592090143,0.125918180337,0.125915459421,0.125912738682,0.12591001812,0.125907297734,0.125904577524,0.125901857492,0.125899137635,0.125896417956,0.125893698453,0.125890979126,0.125888259976,0.125885541003,0.125882822206,0.125880103585,0.125877385141,0.125874666873,0.125871948782,0.125869230867,0.125866513128,0.125863795566,0.12586107818,0.12585836097,0.125855643937,0.125852927079,0.125850210398,0.125847493894,0.125844777565,0.125842061413,0.125839345437,0.125836629637,0.125833914013,0.125831198565,0.125828483293,0.125825768197,0.125823053278,0.125820338534,0.125817623966,0.125814909575,0.125812195359,0.125809481319,0.125806767456,0.125804053768,0.125801340256,0.12579862692,0.12579591376,0.125793200775,0.125790487967,0.125787775334,0.125785062877,0.125782350596,0.12577963849,0.12577692656,0.125774214806,0.125771503228,0.125768791825,0.125766080598,0.125763369547,0.125760658671,0.12575794797,0.125755237446,0.125752527097,0.125749816923,0.125747106925,0.125744397102,0.125741687455,0.125738977984,0.125736268687,0.125733559567,0.125730850621,0.125728141851,0.125725433256,0.125722724837,0.125720016593,0.125717308524,0.125714600631,0.125711892913,0.12570918537,0.125706478002,0.12570377081,0.125701063793,0.125698356951,0.125695650284,0.125692943792,0.125690237475,0.125687531333,0.125684825367,0.125682119575,0.125679413959,0.125676708518,0.125674003251,0.12567129816,0.125668593243,0.125665888502,0.125663183935,0.125660479544,0.125657775327,0.125655071285,0.125652367418,0.125649663726,0.125646960208,0.125644256866,0.125641553698,0.125638850705,0.125636147886,0.125633445243,0.125630742774,0.125628040479,0.12562533836,0.125622636415,0.125619934645,0.125617233049,0.125614531628,0.125611830381,0.125609129309,0.125606428412,0.125603727689,0.12560102714,0.125598326766,0.125595626567,0.125592926541,0.125590226691,0.125587527014,0.125584827513,0.125582128185,0.125579429032,0.125576730053,0.125574031248,0.125571332618,0.125568634162,0.12556593588,0.125563237772,0.125560539839,0.12555784208,0.125555144495,0.125552447084,0.125549749847,0.125547052785,0.125544355896,0.125541659182,0.125538962642,0.125536266275,0.125533570083,0.125530874065,0.12552817822,0.12552548255,0.125522787054,0.125520091731,0.125517396583,0.125514701608,0.125512006807,0.12550931218,0.125506617727,0.125503923448,0.125501229342,0.125498535411,0.125495841653,0.125493148069,0.125490454658,0.125487761421,0.125485068358,0.125482375469,0.125479682753,0.125476990211,0.125474297842,0.125471605647,0.125468913626,0.125466221778,0.125463530104,0.125460838603,0.125458147276,0.125455456122,0.125452765142,0.125450074335,0.125447383701,0.125444693241,0.125442002955,0.125439312841,0.125436622902,0.125433933135,0.125431243542,0.125428554122,0.125425864875,0.125423175801,0.125420486901,0.125417798174,0.12541510962,0.12541242124,0.125409733032,0.125407044998,0.125404357137,0.125401669448,0.125398981933,0.125396294592,0.125393607423,0.125390920427,0.125388233604,0.125385546954,0.125382860477,0.125380174173,0.125377488042,0.125374802084,0.125372116299,0.125369430687,0.125366745247,0.125364059981,0.125361374887,0.125358689966,0.125356005218,0.125353320643,0.12535063624,0.12534795201,0.125345267953,0.125342584069,0.125339900357,0.125337216818,0.125334533451,0.125331850257,0.125329167236,0.125326484387,0.125323801711,0.125321119208,0.125318436877,0.125315754718,0.125313072732,0.125310390919,0.125307709278,0.125305027809,0.125302346513,0.125299665389,0.125296984438,0.125294303659,0.125291623052,0.125288942617,0.125286262355,0.125283582266,0.125280902348,0.125278222603,0.12527554303,0.125272863629,0.1252701844,0.125267505344,0.12526482646,0.125262147748,0.125259469208,0.12525679084,0.125254112644,0.12525143462,0.125248756768,0.125246079089,0.125243401581,0.125240724245,0.125238047082,0.12523537009,0.12523269327,0.125230016622,0.125227340146,0.125224663842,0.12522198771,0.12521931175,0.125216635961,0.125213960345,0.1252112849,0.125208609626,0.125205934525,0.125203259595,0.125200584837,0.125197910251,0.125195235837,0.125192561594,0.125189887522,0.125187213623,0.125184539895,0.125181866338,0.125179192954,0.12517651974,0.125173846698,0.125171173828,0.125168501129,0.125165828602,0.125163156246,0.125160484062,0.125157812049,0.125155140207,0.125152468537,0.125149797039,0.125147125711,0.125144454555,0.12514178357,0.125139112757,0.125136442114,0.125133771644,0.125131101344,0.125128431215,0.125125761258,0.125123091472,0.125120421857,0.125117752413,0.125115083141,0.125112414039,0.125109745109,0.125107076349,0.125104407761,0.125101739344,0.125099071097,0.125096403022,0.125093735118,0.125091067384,0.125088399822,0.125085732431,0.12508306521,0.125080398161,0.125077731282,0.125075064574,0.125072398037,0.125069731671,0.125067065475,0.125064399451,0.125061733597,0.125059067914,0.125056402401,0.125053737059,0.125051071888,0.125048406888,0.125045742058,0.125043077399,0.125040412911,0.125037748593,0.125035084446,0.125032420469,0.125029756663,0.125027093027,0.125024429562,0.125021766268,0.125019103144,0.12501644019,0.125013777407,0.125011114794,0.125008452352,0.12500579008,0.125003127978,0.125000466047,0.124997804286,0.124995142695,0.124992481274,0.124989820024,0.124987158944,0.124984498035,0.124981837295,0.124979176726,0.124976516327,0.124973856098,0.12497119604,0.124968536151,0.124965876432,0.124963216884,0.124960557506,0.124957898297,0.124955239259,0.124952580391,0.124949921693,0.124947263165,0.124944604806,0.124941946618,0.1249392886,0.124936630751,0.124933973073,0.124931315564,0.124928658225,0.124926001056,0.124923344057,0.124920687227,0.124918030568,0.124915374078,0.124912717758,0.124910061607,0.124907405627,0.124904749816,0.124902094174,0.124899438703,0.124896783401,0.124894128268,0.124891473306,0.124888818512,0.124886163889,0.124883509435,0.12488085515,0.124878201035,0.124875547089,0.124872893313,0.124870239707,0.124867586269,0.124864933002,0.124862279903,0.124859626974,0.124856974215,0.124854321624,0.124851669204,0.124849016952,0.12484636487,0.124843712957,0.124841061213,0.124838409638,0.124835758233,0.124833106997,0.12483045593,0.124827805032,0.124825154303,0.124822503744,0.124819853353,0.124817203132,0.12481455308,0.124811903197,0.124809253483,0.124806603938,0.124803954562,0.124801305355,0.124798656316,0.124796007447,0.124793358747,0.124790710216,0.124788061853,0.12478541366,0.124782765635,0.124780117779,0.124777470092,0.124774822574,0.124772175225,0.124769528044,0.124766881032,0.124764234189,0.124761587515,0.124758941009,0.124756294672,0.124753648503,0.124751002504,0.124748356673,0.12474571101,0.124743065516,0.124740420191,0.124737775034,0.124735130046,0.124732485226,0.124729840575,0.124727196092,0.124724551778,0.124721907632,0.124719263655,0.124716619846,0.124713976205,0.124711332733,0.124708689429,0.124706046294,0.124703403327,0.124700760528,0.124698117897,0.124695475435,0.124692833141,0.124690191015,0.124687549058,0.124684907268,0.124682265647,0.124679624194,0.124676982909,0.124674341792,0.124671700844,0.124669060063,0.124666419451,0.124663779006,0.12466113873,0.124658498621,0.124655858681,0.124653218909,0.124650579304,0.124647939868,0.124645300599,0.124642661499,0.124640022566,0.124637383801,0.124634745204,0.124632106775,0.124629468513,0.12462683042,0.124624192494,0.124621554736,0.124618917146,0.124616279723,0.124613642469,0.124611005382,0.124608368462,0.12460573171,0.124603095126,0.12460045871,0.124597822461,0.12459518638,0.124592550466,0.12458991472,0.124587279141,0.12458464373,0.124582008487,0.124579373411,0.124576738502,0.124574103761,0.124571469187,0.124568834781,0.124566200542,0.124563566471,0.124560932566,0.12455829883,0.12455566526,0.124553031858,0.124550398623,0.124547765555,0.124545132655,0.124542499922,0.124539867356,0.124537234957,0.124534602726,0.124531970661,0.124529338764,0.124526707034,0.124524075471,0.124521444075,0.124518812846,0.124516181785,0.12451355089,0.124510920162,0.124508289602,0.124505659208,0.124503028981,0.124500398921,0.124497769029,0.124495139303,0.124492509744,0.124489880352,0.124487251126,0.124484622068,0.124481993176,0.124479364452,0.124476735894,0.124474107502,0.124471479278,0.12446885122,0.124466223329,0.124463595605,0.124460968048,0.124458340657,0.124455713432,0.124453086375,0.124450459484,0.124447832759,0.124445206202,0.12444257981,0.124439953586,0.124437327527,0.124434701636,0.124432075911,0.124429450352,0.12442682496,0.124424199734,0.124421574675,0.124418949782,0.124416325055,0.124413700495,0.124411076101,0.124408451873,0.124405827812,0.124403203917,0.124400580189,0.124397956626,0.12439533323,0.12439271,0.124390086937,0.124387464039,0.124384841308,0.124382218743,0.124379596344,0.124376974111,0.124374352044,0.124371730144,0.124369108409,0.124366486841,0.124363865438,0.124361244202,0.124358623131,0.124356002227,0.124353381488,0.124350760916,0.124348140509,0.124345520268,0.124342900193,0.124340280285,0.124337660541,0.124335040964,0.124332421553,0.124329802307,0.124327183227,0.124324564313,0.124321945565,0.124319326983,0.124316708566,0.124314090315,0.124311472229,0.124308854309,0.124306236555,0.124303618967,0.124301001544,0.124298384287,0.124295767195,0.124293150269,0.124290533508,0.124287916913,0.124285300484,0.12428268422,0.124280068121,0.124277452188,0.12427483642,0.124272220818,0.124269605381,0.12426699011,0.124264375004,0.124261760063,0.124259145288,0.124256530678,0.124253916233,0.124251301953,0.124248687839,0.12424607389,0.124243460106,0.124240846488,0.124238233035,0.124235619747,0.124233006624,0.124230393666,0.124227780873,0.124225168245,0.124222555783,0.124219943485,0.124217331353,0.124214719386,0.124212107583,0.124209495946,0.124206884474,0.124204273167,0.124201662024,0.124199051047,0.124196440234,0.124193829587,0.124191219104,0.124188608786,0.124185998633,0.124183388645,0.124180778821,0.124178169163,0.124175559669,0.12417295034,0.124170341176,0.124167732176,0.124165123341,0.124162514671,0.124159906165,0.124157297825,0.124154689648,0.124152081637,0.12414947379,0.124146866107,0.12414425859,0.124141651236,0.124139044048,0.124136437024,0.124133830164,0.124131223469,0.124128616938,0.124126010572,0.12412340437,0.124120798332,0.124118192459,0.124115586751,0.124112981206,0.124110375826,0.124107770611,0.124105165559,0.124102560672,0.124099955949,0.124097351391,0.124094746997,0.124092142767,0.124089538701,0.124086934799,0.124084331062,0.124081727489,0.124079124079,0.124076520834,0.124073917753,0.124071314837,0.124068712084,0.124066109495,0.12406350707,0.12406090481,0.124058302713,0.12405570078,0.124053099012,0.124050497407,0.124047895966,0.124045294689,0.124042693576,0.124040092627,0.124037491841,0.12403489122,0.124032290762,0.124029690469,0.124027090339,0.124024490372,0.12402189057,0.124019290931,0.124016691456,0.124014092145,0.124011492997,0.124008894013,0.124006295193,0.124003696536,0.124001098043,0.123998499714,0.123995901548,0.123993303546,0.123990705707,0.123988108032,0.12398551052,0.123982913172,0.123980315987,0.123977718966,0.123975122108,0.123972525414,0.123969928883,0.123967332516,0.123964736311,0.123962140271,0.123959544393,0.123956948679,0.123954353128,0.123951757741,0.123949162517,0.123946567456,0.123943972558,0.123941377823,0.123938783252,0.123936188844,0.123933594599,0.123931000517,0.123928406599,0.123925812843,0.123923219251,0.123920625822,0.123918032555,0.123915439452,0.123912846512,0.123910253735,0.123907661121,0.12390506867,0.123902476382,0.123899884256,0.123897292294,0.123894700495,0.123892108858,0.123889517385,0.123886926074,0.123884334926,0.123881743941,0.123879153119,0.12387656246,0.123873971963,0.123871381629,0.123868791458,0.12386620145,0.123863611604,0.123861021921,0.123858432401,0.123855843043,0.123853253848,0.123850664816,0.123848075946,0.123845487239,0.123842898694,0.123840310312,0.123837722093,0.123835134036,0.123832546141,0.123829958409,0.12382737084,0.123824783433,0.123822196188,0.123819609106,0.123817022186,0.123814435429,0.123811848834,0.123809262402,0.123806676131,0.123804090023,0.123801504078,0.123798918294,0.123796332673,0.123793747214,0.123791161918,0.123788576783,0.123785991811,0.123783407001,0.123780822353,0.123778237867,0.123775653544,0.123773069382,0.123770485383,0.123767901546,0.123765317871,0.123762734358,0.123760151006,0.123757567817,0.12375498479,0.123752401925,0.123749819222,0.123747236681,0.123744654301,0.123742072084,0.123739490029,0.123736908135,0.123734326403,0.123731744833,0.123729163425,0.123726582179,0.123724001095,0.123721420172,0.123718839411,0.123716258812,0.123713678375,0.123711098099,0.123708517985,0.123705938033,0.123703358242,0.123700778613,0.123698199145,0.12369561984,0.123693040695,0.123690461713,0.123687882892,0.123685304232,0.123682725734,0.123680147398,0.123677569222,0.123674991209,0.123672413357,0.123669835666,0.123667258137,0.123664680769,0.123662103563,0.123659526518,0.123656949634,0.123654372912,0.123651796351,0.123649219951,0.123646643712,0.123644067635,0.123641491719,0.123638915965,0.123636340371,0.123633764939,0.123631189668,0.123628614558,0.123626039609,0.123623464821,0.123620890195,0.12361831573,0.123615741425,0.123613167282,0.1236105933,0.123608019479,0.123605445819,0.123602872319,0.123600298981,0.123597725804,0.123595152788,0.123592579933,0.123590007238,0.123587434705,0.123584862332,0.12358229012,0.123579718069,0.123577146179,0.12357457445,0.123572002882,0.123569431474,0.123566860227,0.123564289141,0.123561718216,0.123559147451,0.123556576847,0.123554006404,0.123551436121,0.123548865999,0.123546296038,0.123543726237,0.123541156597,0.123538587118,0.123536017799,0.12353344864,0.123530879642,0.123528310805,0.123525742128,0.123523173612,0.123520605256,0.12351803706,0.123515469025,0.12351290115,0.123510333436,0.123507765882,0.123505198489,0.123502631256,0.123500064183,0.12349749727,0.123494930518,0.123492363926,0.123489797495,0.123487231223,0.123484665112,0.123482099161,0.12347953337,0.12347696774,0.123474402269,0.123471836959,0.123469271809,0.123466706819,0.123464141989,0.123461577319,0.123459012809,0.123456448459,0.12345388427,0.12345132024,0.12344875637,0.12344619266,0.123443629111,0.123441065721,0.123438502491,0.123435939421,0.123433376511,0.123430813761,0.12342825117,0.12342568874,0.123423126469,0.123420564358,0.123418002407,0.123415440616,0.123412878985,0.123410317513,0.123407756201,0.123405195049,0.123402634056,0.123400073223,0.12339751255,0.123394952036,0.123392391682,0.123389831488,0.123387271453,0.123384711578,0.123382151862,0.123379592306,0.12337703291,0.123374473673,0.123371914595,0.123369355677,0.123366796919,0.123364238319,0.12336167988,0.123359121599,0.123356563479,0.123354005517,0.123351447715,0.123348890072,0.123346332589,0.123343775264,0.1233412181,0.123338661094,0.123336104248,0.123333547561,0.123330991033,0.123328434664,0.123325878455,0.123323322405,0.123320766514,0.123318210782,0.123315655209,0.123313099795,0.123310544541,0.123307989445,0.123305434509,0.123302879732,0.123300325113,0.123297770654,0.123295216354,0.123292662212,0.12329010823,0.123287554407,0.123285000742,0.123282447237,0.12327989389,0.123277340702,0.123274787674,0.123272234804,0.123269682092,0.12326712954,0.123264577146,0.123262024912,0.123259472836,0.123256920918,0.12325436916,0.12325181756,0.123249266119,0.123246714837,0.123244163713,0.123241612748,0.123239061941,0.123236511294,0.123233960804,0.123231410474,0.123228860302,0.123226310288,0.123223760433,0.123221210737,0.123218661199,0.123216111819,0.123213562599,0.123211013536,0.123208464632,0.123205915886,0.123203367299,0.12320081887,0.1231982706,0.123195722488,0.123193174534,0.123190626739,0.123188079101,0.123185531623,0.123182984302,0.12318043714,0.123177890136,0.12317534329,0.123172796602,0.123170250073,0.123167703702,0.123165157489,0.123162611434,0.123160065537,0.123157519798,0.123154974218,0.123152428795,0.123149883531,0.123147338424,0.123144793476,0.123142248686,0.123139704053,0.123137159579,0.123134615262,0.123132071104,0.123129527103,0.123126983261,0.123124439576,0.123121896049,0.12311935268,0.123116809469,0.123114266416,0.12311172352,0.123109180783,0.123106638203,0.123104095781,0.123101553517,0.12309901141,0.123096469461,0.12309392767,0.123091386036,0.12308884456,0.123086303242,0.123083762082,0.123081221079,0.123078680234,0.123076139546,0.123073599016,0.123071058643,0.123068518428,0.12306597837,0.12306343847,0.123060898728,0.123058359143,0.123055819715,0.123053280445,0.123050741332,0.123048202377,0.123045663579,0.123043124939,0.123040586455,0.12303804813,0.123035509961,0.12303297195,0.123030434096,0.123027896399,0.12302535886,0.123022821478,0.123020284253,0.123017747185,0.123015210275,0.123012673522,0.123010136925,0.123007600486,0.123005064205,0.12300252808,0.122999992112,0.122997456302,0.122994920648,0.122992385152,0.122989849812,0.12298731463,0.122984779605,0.122982244736,0.122979710025,0.12297717547,0.122974641073,0.122972106832,0.122969572749,0.122967038822,0.122964505052,0.122961971439,0.122959437983,0.122956904683,0.122954371541,0.122951838555,0.122949305726,0.122946773054,0.122944240538,0.122941708179,0.122939175977,0.122936643932,0.122934112043,0.122931580311,0.122929048736,0.122926517317,0.122923986055,0.122921454949,0.122918924,0.122916393208,0.122913862572,0.122911332093,0.12290880177,0.122906271604,0.122903741594,0.12290121174,0.122898682044,0.122896152503,0.122893623119,0.122891093891,0.12288856482,0.122886035905,0.122883507147,0.122880978545,0.122878450099,0.122875921809,0.122873393676,0.122870865699,0.122868337878,0.122865810214,0.122863282705,0.122860755353,0.122858228158,0.122855701118,0.122853174234,0.122850647507,0.122848120936,0.122845594521,0.122843068262,0.122840542159,0.122838016212,0.122835490421,0.122832964786,0.122830439307,0.122827913985,0.122825388818,0.122822863807,0.122820338952,0.122817814253,0.12281528971,0.122812765323,0.122810241092,0.122807717017,0.122805193097,0.122802669334,0.122800145726,0.122797622274,0.122795098977,0.122792575837,0.122790052852,0.122787530023,0.12278500735,0.122782484833,0.122779962471,0.122777440265,0.122774918214,0.12277239632,0.12276987458,0.122767352997,0.122764831569,0.122762310296,0.12275978918,0.122757268218,0.122754747413,0.122752226762,0.122749706268,0.122747185929,0.122744665745,0.122742145716,0.122739625844,0.122737106126,0.122734586564,0.122732067158,0.122729547906,0.12272702881,0.12272450987,0.122721991085,0.122719472455,0.12271695398,0.122714435661,0.122711917497,0.122709399488,0.122706881634,0.122704363936,0.122701846393,0.122699329005,0.122696811772,0.122694294694,0.122691777772,0.122689261004,0.122686744392,0.122684227935,0.122681711633,0.122679195486,0.122676679494,0.122674163657,0.122671647975,0.122669132447,0.122666617075,0.122664101858,0.122661586796,0.122659071889,0.122656557137,0.122654042539,0.122651528097,0.122649013809,0.122646499676,0.122643985698,0.122641471875,0.122638958207,0.122636444693,0.122633931334,0.12263141813,0.122628905081,0.122626392186,0.122623879446,0.122621366861,0.122618854431,0.122616342155,0.122613830034,0.122611318067,0.122608806255,0.122606294598,0.122603783095,0.122601271747,0.122598760553,0.122596249514,0.122593738629,0.122591227899,0.122588717323,0.122586206902,0.122583696635,0.122581186523,0.122578676565,0.122576166761,0.122573657112,0.122571147618,0.122568638277,0.122566129091,0.122563620059,0.122561111182,0.122558602459,0.12255609389,0.122553585475,0.122551077215,0.122548569109,0.122546061157,0.122543553359,0.122541045716,0.122538538227,0.122536030891,0.12253352371,0.122531016683,0.122528509811,0.122526003092,0.122523496527,0.122520990117,0.12251848386,0.122515977757,0.122513471809,0.122510966014,0.122508460374,0.122505954887,0.122503449554,0.122500944376,0.122498439351,0.12249593448,0.122493429763,0.1224909252,0.12248842079,0.122485916535,0.122483412433,0.122480908485,0.122478404691,0.122475901051,0.122473397564,0.122470894231,0.122468391052,0.122465888027,0.122463385155,0.122460882437,0.122458379873,0.122455877462,0.122453375205,0.122450873101,0.122448371152,0.122445869355,0.122443367713,0.122440866223,0.122438364888,0.122435863706,0.122433362677,0.122430861802,0.12242836108,0.122425860512,0.122423360097,0.122420859836,0.122418359728,0.122415859773,0.122413359972,0.122410860325,0.12240836083,0.122405861489,0.122403362301,0.122400863267,0.122398364385,0.122395865657,0.122393367083,0.122390868661,0.122388370393,0.122385872278,0.122383374316,0.122380876508,0.122378378852,0.12237588135,0.122373384,0.122370886804,0.122368389761,0.122365892871,0.122363396134,0.122360899551,0.12235840312,0.122355906842,0.122353410717,0.122350914745,0.122348418926,0.122345923261,0.122343427748,0.122340932388,0.122338437181,0.122335942126,0.122333447225,0.122330952476,0.122328457881,0.122325963438,0.122323469148,0.122320975011,0.122318481026,0.122315987195,0.122313493516,0.12231099999,0.122308506616,0.122306013396,0.122303520327,0.122301027412,0.122298534649,0.122296042039,0.122293549582,0.122291057277,0.122288565125,0.122286073125,0.122283581278,0.122281089583,0.122278598041,0.122276106652,0.122273615415,0.12227112433,0.122268633398,0.122266142619,0.122263651992,0.122261161517,0.122258671195,0.122256181025,0.122253691007,0.122251201142,0.122248711429,0.122246221869,0.12224373246,0.122241243205,0.122238754101,0.12223626515,0.122233776351,0.122231287704,0.122228799209,0.122226310867,0.122223822676,0.122221334638,0.122218846752,0.122216359019,0.122213871437,0.122211384008,0.12220889673,0.122206409605,0.122203922632,0.12220143581,0.122198949141,0.122196462624,0.122193976259,0.122191490046,0.122189003984,0.122186518075,0.122184032318,0.122181546713,0.122179061259,0.122176575957,0.122174090808,0.12217160581,0.122169120964,0.12216663627,0.122164151728,0.122161667337,0.122159183098,0.122156699011,0.122154215076,0.122151731293,0.122149247661,0.122146764181,0.122144280853,0.122141797676,0.122139314651,0.122136831778,0.122134349056,0.122131866486,0.122129384067,0.1221269018,0.122124419685,0.122121937721,0.122119455909,0.122116974248,0.122114492739,0.122112011381,0.122109530175,0.12210704912,0.122104568217,0.122102087465,0.122099606865,0.122097126416,0.122094646118,0.122092165972,0.122089685977,0.122087206133,0.122084726441,0.1220822469,0.12207976751,0.122077288272,0.122074809184,0.122072330249,0.122069851464,0.12206737283,0.122064894348,0.122062416017,0.122059937837,0.122057459808,0.122054981931,0.122052504204,0.122050026629,0.122047549205,0.122045071931,0.122042594809,0.122040117838,0.122037641018,0.122035164349,0.122032687831,0.122030211464,0.122027735248,0.122025259183,0.122022783268,0.122020307505,0.122017831893,0.122015356431,0.122012881121,0.122010405961,0.122007930952,0.122005456094,0.122002981387,0.12200050683,0.121998032424,0.12199555817,0.121993084065,0.121990610112,0.121988136309,0.121985662657,0.121983189156,0.121980715805,0.121978242606,0.121975769556,0.121973296658,0.12197082391,0.121968351312,0.121965878865,0.121963406569,0.121960934423,0.121958462428,0.121955990583,0.121953518889,0.121951047346,0.121948575952,0.12194610471,0.121943633617,0.121941162676,0.121938691884,0.121936221243,0.121933750752,0.121931280412,0.121928810222,0.121926340183,0.121923870294,0.121921400555,0.121918930966,0.121916461528,0.12191399224,0.121911523102,0.121909054114,0.121906585277,0.12190411659,0.121901648053,0.121899179666,0.121896711429,0.121894243343,0.121891775406,0.12188930762,0.121886839984,0.121884372498,0.121881905162,0.121879437976,0.12187697094,0.121874504054,0.121872037318,0.121869570732,0.121867104296,0.12186463801,0.121862171874,0.121859705888,0.121857240052,0.121854774366,0.121852308829,0.121849843443,0.121847378206,0.121844913119,0.121842448182,0.121839983395,0.121837518758,0.12183505427,0.121832589932,0.121830125744,0.121827661705,0.121825197817,0.121822734078,0.121820270489,0.121817807049,0.121815343759,0.121812880619,0.121810417628,0.121807954787,0.121805492095,0.121803029553,0.121800567161,0.121798104918,0.121795642825,0.121793180881,0.121790719087,0.121788257442,0.121785795947,0.121783334601,0.121780873405,0.121778412358,0.12177595146,0.121773490712,0.121771030114,0.121768569664,0.121766109364,0.121763649213,0.121761189212,0.12175872936,0.121756269657,0.121753810104,0.1217513507,0.121748891445,0.121746432339,0.121743973382,0.121741514575,0.121739055917,0.121736597408,0.121734139048,0.121731680837,0.121729222776,0.121726764864,0.1217243071,0.121721849486,0.121719392021,0.121716934705,0.121714477538,0.121712020519,0.12170956365,0.12170710693,0.121704650359,0.121702193937,0.121699737664,0.12169728154,0.121694825564,0.121692369738,0.12168991406,0.121687458532,0.121685003152,0.121682547921,0.121680092838,0.121677637905,0.121675183121,0.121672728485,0.121670273998,0.121667819659,0.12166536547,0.121662911429,0.121660457537,0.121658003794,0.121655550199,0.121653096753,0.121650643455,0.121648190306,0.121645737306,0.121643284455,0.121640831752,0.121638379197,0.121635926791,0.121633474534,0.121631022425,0.121628570465,0.121626118653,0.121623666989,0.121621215474,0.121618764108,0.12161631289,0.12161386182,0.121611410899,0.121608960126,0.121606509502,0.121604059026,0.121601608698,0.121599158519,0.121596708488,0.121594258605,0.12159180887,0.121589359284,0.121586909846,0.121584460556,0.121582011415,0.121579562422,0.121577113577,0.12157466488,0.121572216331,0.12156976793,0.121567319678,0.121564871573,0.121562423617,0.121559975809,0.121557528149,0.121555080637,0.121552633273,0.121550186057,0.121547738989,0.121545292069,0.121542845297,0.121540398673,0.121537952197,0.121535505869,0.121533059689,0.121530613656,0.121528167772,0.121525722036,0.121523276447,0.121520831006,0.121518385713,0.121515940568,0.121513495571,0.121511050721,0.12150860602,0.121506161466,0.121503717059,0.121501272801,0.12149882869,0.121496384727,0.121493940911,0.121491497244,0.121489053724,0.121486610351,0.121484167126,0.121481724049,0.12147928112,0.121476838338,0.121474395703,0.121471953216,0.121469510877,0.121467068685,0.121464626641,0.121462184744,0.121459742995,0.121457301393,0.121454859938,0.121452418631,0.121449977472,0.121447536459,0.121445095595,0.121442654877,0.121440214307,0.121437773884,0.121435333609,0.121432893481,0.1214304535,0.121428013667,0.12142557398,0.121423134441,0.12142069505,0.121418255805,0.121415816708,0.121413377758,0.121410938955,0.121408500299,0.12140606179,0.121403623429,0.121401185215,0.121398747147,0.121396309227,0.121393871454,0.121391433828,0.121388996349,0.121386559017,0.121384121832,0.121381684795,0.121379247904,0.12137681116,0.121374374563,0.121371938113,0.12136950181,0.121367065653,0.121364629644,0.121362193782,0.121359758066,0.121357322497,0.121354887076,0.121352451801,0.121350016672,0.121347581691,0.121345146856,0.121342712168,0.121340277627,0.121337843233,0.121335408985,0.121332974884,0.12133054093,0.121328107122,0.121325673462,0.121323239947,0.12132080658,0.121318373359,0.121315940284,0.121313507356,0.121311074575,0.12130864194,0.121306209452,0.121303777111,0.121301344916,0.121298912867,0.121296480965,0.121294049209,0.1212916176,0.121289186137,0.121286754821,0.121284323651,0.121281892628,0.121279461751,0.12127703102,0.121274600436,0.121272169997,0.121269739706,0.12126730956,0.121264879561,0.121262449708,0.121260020002,0.121257590441,0.121255161027,0.12125273176,0.121250302638,0.121247873662,0.121245444833,0.12124301615,0.121240587613,0.121238159222,0.121235730977,0.121233302879,0.121230874926,0.12122844712,0.121226019459,0.121223591945,0.121221164577,0.121218737354,0.121216310278,0.121213883348,0.121211456563,0.121209029925,0.121206603433,0.121204177086,0.121201750886,0.121199324831,0.121196898922,0.121194473159,0.121192047542,0.121189622071,0.121187196745,0.121184771566,0.121182346532,0.121179921644,0.121177496902,0.121175072305,0.121172647854,0.121170223549,0.12116779939,0.121165375376,0.121162951508,0.121160527786,0.121158104209,0.121155680778,0.121153257493,0.121150834353,0.121148411359,0.12114598851,0.121143565807,0.12114114325,0.121138720838,0.121136298571,0.12113387645,0.121131454475,0.121129032645,0.12112661096,0.121124189421,0.121121768027,0.121119346779,0.121116925676,0.121114504719,0.121112083907,0.12110966324,0.121107242719,0.121104822343,0.121102402112,0.121099982026,0.121097562086,0.121095142291,0.121092722642,0.121090303137,0.121087883778,0.121085464564,0.121083045496,0.121080626572,0.121078207794,0.12107578916,0.121073370672,0.121070952329,0.121068534132,0.121066116079,0.121063698171,0.121061280408,0.121058862791,0.121056445318,0.121054027991,0.121051610808,0.121049193771,0.121046776878,0.121044360131,0.121041943528,0.121039527071,0.121037110758,0.12103469459,0.121032278567,0.121029862689,0.121027446956,0.121025031367,0.121022615924,0.121020200625,0.121017785471,0.121015370462,0.121012955598,0.121010540878,0.121008126304,0.121005711874,0.121003297588,0.121000883448,0.120998469452,0.1209960556,0.120993641894,0.120991228332,0.120988814914,0.120986401642,0.120983988514,0.12098157553,0.120979162691,0.120976749997,0.120974337447,0.120971925042,0.120969512781,0.120967100664,0.120964688693,0.120962276865,0.120959865182,0.120957453644,0.12095504225,0.120952631,0.120950219895,0.120947808934,0.120945398118,0.120942987446,0.120940576918,0.120938166535,0.120935756296,0.120933346201,0.12093093625,0.120928526444,0.120926116782,0.120923707264,0.120921297891,0.120918888662,0.120916479577,0.120914070636,0.120911661839,0.120909253186,0.120906844678,0.120904436314,0.120902028093,0.120899620017,0.120897212085,0.120894804297,0.120892396654,0.120889989154,0.120887581798,0.120885174586,0.120882767518,0.120880360594,0.120877953815,0.120875547179,0.120873140687,0.120870734339,0.120868328135,0.120865922074,0.120863516158,0.120861110386,0.120858704757,0.120856299272,0.120853893931,0.120851488734,0.120849083681,0.120846678771,0.120844274005,0.120841869383,0.120839464905,0.12083706057,0.12083465638,0.120832252332,0.120829848429,0.120827444669,0.120825041053,0.12082263758,0.120820234251,0.120817831066,0.120815428024,0.120813025126,0.120810622372,0.120808219761,0.120805817293,0.120803414969,0.120801012789,0.120798610752,0.120796208859,0.120793807109,0.120791405502,0.120789004039,0.120786602719,0.120784201543,0.12078180051,0.120779399621,0.120776998875,0.120774598272,0.120772197812,0.120769797496,0.120767397323,0.120764997294,0.120762597408,0.120760197665,0.120757798065,0.120755398609,0.120752999295,0.120750600125,0.120748201099,0.120745802215,0.120743403474,0.120741004877,0.120738606423,0.120736208112,0.120733809944,0.120731411919,0.120729014037,0.120726616299,0.120724218703,0.12072182125,0.120719423941,0.120717026774,0.120714629751,0.12071223287,0.120709836132,0.120707439538,0.120705043086,0.120702646777,0.120700250611,0.120697854588,0.120695458708,0.120693062971,0.120690667376,0.120688271925,0.120685876616,0.12068348145,0.120681086427,0.120678691547,0.120676296809,0.120673902214,0.120671507762,0.120669113453,0.120666719286,0.120664325262,0.120661931381,0.120659537642,0.120657144046,0.120654750593,0.120652357282,0.120649964114,0.120647571089,0.120645178206,0.120642785466,0.120640392868,0.120638000413,0.1206356081,0.12063321593,0.120630823902,0.120628432017,0.120626040274,0.120623648674,0.120621257216,0.120618865901,0.120616474728,0.120614083697,0.120611692809,0.120609302063,0.12060691146,0.120604520999,0.12060213068,0.120599740503,0.120597350469,0.120594960577,0.120592570828,0.12059018122,0.120587791755,0.120585402432,0.120583013252,0.120580624213,0.120578235317,0.120575846563,0.120573457951,0.120571069481,0.120568681153,0.120566292968,0.120563904924,0.120561517023,0.120559129264,0.120556741646,0.120554354171,0.120551966838,0.120549579647,0.120547192598,0.120544805691,0.120542418926,0.120540032302,0.120537645821,0.120535259482,0.120532873284,0.120530487229,0.120528101315,0.120525715544,0.120523329914,0.120520944426,0.120518559079,0.120516173875,0.120513788813,0.120511403892,0.120509019113,0.120506634476,0.12050424998,0.120501865626,0.120499481414,0.120497097344,0.120494713416,0.120492329629,0.120489945983,0.12048756248,0.120485179118,0.120482795897,0.120480412819,0.120478029882,0.120475647086,0.120473264432,0.12047088192,0.120468499549,0.120466117319,0.120463735232,0.120461353285,0.12045897148,0.120456589817,0.120454208295,0.120451826914,0.120449445675,0.120447064578,0.120444683621,0.120442302807,0.120439922133,0.120437541601,0.12043516121,0.120432780961,0.120430400852,0.120428020886,0.12042564106,0.120423261376,0.120420881833,0.120418502431,0.12041612317,0.120413744051,0.120411365073,0.120408986236,0.12040660754,0.120404228986,0.120401850572,0.1203994723,0.120397094169,0.120394716178,0.120392338329,0.120389960621,0.120387583055,0.120385205629,0.120382828344,0.1203804512,0.120378074197,0.120375697336,0.120373320615,0.120370944035,0.120368567596,0.120366191298,0.120363815141,0.120361439125,0.12035906325,0.120356687516,0.120354311922,0.12035193647,0.120349561158,0.120347185987,0.120344810957,0.120342436068,0.120340061319,0.120337686711,0.120335312244,0.120332937918,0.120330563732,0.120328189688,0.120325815784,0.12032344202,0.120321068397,0.120318694915,0.120316321574,0.120313948373,0.120311575313,0.120309202393,0.120306829614,0.120304456976,0.120302084478,0.120299712121,0.120297339904,0.120294967828,0.120292595892,0.120290224097,0.120287852442,0.120285480927,0.120283109554,0.12028073832,0.120278367227,0.120275996275,0.120273625462,0.120271254791,0.120268884259,0.120266513868,0.120264143617,0.120261773507,0.120259403537,0.120257033707,0.120254664017,0.120252294468,0.120249925059,0.12024755579,0.120245186662,0.120242817673,0.120240448825,0.120238080117,0.12023571155,0.120233343122,0.120230974834,0.120228606687,0.12022623868,0.120223870813,0.120221503086,0.120219135499,0.120216768052,0.120214400745,0.120212033578,0.120209666552,0.120207299665,0.120204932918,0.120202566311,0.120200199844,0.120197833518,0.120195467331,0.120193101284,0.120190735377,0.120188369609,0.120186003982,0.120183638495,0.120181273147,0.120178907939,0.120176542871,0.120174177943,0.120171813155,0.120169448506,0.120167083998,0.120164719629,0.120162355399,0.12015999131,0.12015762736,0.12015526355,0.12015289988,0.120150536349,0.120148172958,0.120145809707,0.120143446595,0.120141083623,0.12013872079,0.120136358097,0.120133995544,0.12013163313,0.120129270856,0.120126908721,0.120124546726,0.12012218487,0.120119823154,0.120117461577,0.12011510014,0.120112738842,0.120110377684,0.120108016665,0.120105655785,0.120103295045,0.120100934445,0.120098573983,0.120096213661,0.120093853479,0.120091493436,0.120089133532,0.120086773767,0.120084414142,0.120082054655,0.120079695309,0.120077336101,0.120074977033,0.120072618104,0.120070259314,0.120067900663,0.120065542151,0.120063183779,0.120060825546,0.120058467452,0.120056109497,0.120053751681,0.120051394004,0.120049036467,0.120046679068,0.120044321809,0.120041964688,0.120039607707,0.120037250864,0.120034894161,0.120032537596,0.120030181171,0.120027824884,0.120025468737,0.120023112728,0.120020756859,0.120018401128,0.120016045536,0.120013690083,0.120011334769,0.120008979593,0.120006624557,0.120004269659,0.120001914901,0.119999560281,0.119997205799,0.119994851457,0.119992497253,0.119990143188,0.119987789262,0.119985435475,0.119983081826,0.119980728316,0.119978374944,0.119976021711,0.119973668617,0.119971315662,0.119968962845,0.119966610167,0.119964257627,0.119961905226,0.119959552963,0.119957200839,0.119954848854,0.119952497007,0.119950145299,0.119947793729,0.119945442297,0.119943091005,0.11994073985,0.119938388834,0.119936037956,0.119933687217,0.119931336616,0.119928986154,0.11992663583,0.119924285644,0.119921935597,0.119919585688,0.119917235917,0.119914886285,0.119912536791,0.119910187435,0.119907838218,0.119905489139,0.119903140198,0.119900791395,0.11989844273,0.119896094204,0.119893745816,0.119891397566,0.119889049454,0.11988670148,0.119884353644,0.119882005947,0.119879658388,0.119877310966,0.119874963683,0.119872616538,0.119870269531,0.119867922662,0.119865575931,0.119863229338,0.119860882883,0.119858536566,0.119856190387,0.119853844346,0.119851498443,0.119849152677,0.11984680705,0.119844461561,0.119842116209,0.119839770996,0.11983742592,0.119835080982,0.119832736182,0.119830391519,0.119828046995,0.119825702608,0.119823358359,0.119821014248,0.119818670275,0.119816326439,0.119813982741,0.119811639181,0.119809295759,0.119806952474,0.119804609327,0.119802266317,0.119799923445,0.119797580711,0.119795238114,0.119792895655,0.119790553334,0.11978821115,0.119785869104,0.119783527195,0.119781185424,0.11977884379,0.119776502294,0.119774160935,0.119771819714,0.119769478631,0.119767137684,0.119764796875,0.119762456204,0.11976011567,0.119757775274,0.119755435014,0.119753094893,0.119750754908,0.119748415061,0.119746075351,0.119743735779,0.119741396344,0.119739057046,0.119736717885,0.119734378862,0.119732039976,0.119729701227,0.119727362616,0.119725024141,0.119722685804,0.119720347604,0.119718009541,0.119715671616,0.119713333827,0.119710996176,0.119708658662,0.119706321285,0.119703984044,0.119701646942,0.119699309976,0.119696973147,0.119694636455,0.1196922999,0.119689963482,0.119687627202,0.119685291058,0.119682955051,0.119680619181,0.119678283448,0.119675947852,0.119673612393,0.119671277071,0.119668941886,0.119666606838,0.119664271926,0.119661937151,0.119659602514,0.119657268013,0.119654933648,0.119652599421,0.11965026533,0.119647931377,0.119645597559,0.119643263879,0.119640930335,0.119638596929,0.119636263658,0.119633930525,0.119631597528,0.119629264668,0.119626931944,0.119624599358,0.119622266907,0.119619934594,0.119617602417,0.119615270376,0.119612938472,0.119610606705,0.119608275074,0.11960594358,0.119603612222,0.119601281001,0.119598949916,0.119596618968,0.119594288156,0.119591957481,0.119589626942,0.11958729654,0.119584966273,0.119582636144,0.119580306151,0.119577976294,0.119575646573,0.119573316989,0.119570987541,0.119568658229,0.119566329054,0.119564000015,0.119561671112,0.119559342346,0.119557013716,0.119554685222,0.119552356864,0.119550028643,0.119547700557,0.119545372608,0.119543044795,0.119540717118,0.119538389578,0.119536062173,0.119533734905,0.119531407772,0.119529080776,0.119526753916,0.119524427192,0.119522100604,0.119519774152,0.119517447836,0.119515121656,0.119512795612,0.119510469704,0.119508143932,0.119505818296,0.119503492796,0.119501167432,0.119498842204,0.119496517111,0.119494192155,0.119491867334,0.11948954265,0.119487218101,0.119484893688,0.119482569411,0.119480245269,0.119477921264,0.119475597394,0.11947327366,0.119470950062,0.119468626599,0.119466303272,0.119463980081,0.119461657026,0.119459334106,0.119457011322,0.119454688674,0.119452366161,0.119450043784,0.119447721543,0.119445399437,0.119443077467,0.119440755632,0.119438433933,0.11943611237,0.119433790942,0.11943146965,0.119429148493,0.119426827472,0.119424506586,0.119422185835,0.11941986522,0.119417544741,0.119415224397,0.119412904188,0.119410584115,0.119408264178,0.119405944375,0.119403624708,0.119401305177,0.11939898578,0.119396666519,0.119394347394,0.119392028404,0.119389709549,0.119387390829,0.119385072244,0.119382753795,0.119380435481,0.119378117303,0.119375799259,0.119373481351,0.119371163578,0.11936884594,0.119366528437,0.119364211069,0.119361893837,0.11935957674,0.119357259777,0.11935494295,0.119352626258,0.119350309701,0.119347993279,0.119345676993,0.119343360841,0.119341044824,0.119338728942,0.119336413195,0.119334097584,0.119331782107,0.119329466765,0.119327151558,0.119324836486,0.119322521549,0.119320206747,0.119317892079,0.119315577547,0.119313263149,0.119310948887,0.119308634759,0.119306320766,0.119304006907,0.119301693184,0.119299379595,0.119297066141,0.119294752822,0.119292439638,0.119290126588,0.119287813673,0.119285500893,0.119283188248,0.119280875737,0.119278563361,0.119276251119,0.119273939012,0.11927162704,0.119269315202,0.119267003499,0.119264691931,0.119262380497,0.119260069197,0.119257758033,0.119255447002,0.119253136107,0.119250825345,0.119248514719,0.119246204227,0.119243893869,0.119241583645,0.119239273557,0.119236963602,0.119234653782,0.119232344097,0.119230034545,0.119227725129,0.119225415846,0.119223106698,0.119220797684,0.119218488805,0.11921618006,0.119213871449,0.119211562972,0.11920925463,0.119206946422,0.119204638348,0.119202330409,0.119200022604,0.119197714932,0.119195407396,0.119193099993,0.119190792724,0.11918848559,0.11918617859,0.119183871724,0.119181564992,0.119179258394,0.11917695193,0.1191746456,0.119172339405,0.119170033343,0.119167727416,0.119165421622,0.119163115963,0.119160810437,0.119158505046,0.119156199788,0.119153894664,0.119151589675,0.119149284819,0.119146980097,0.119144675509,0.119142371056,0.119140066735,0.119137762549,0.119135458497,0.119133154578,0.119130850794,0.119128547143,0.119126243626,0.119123940243,0.119121636993,0.119119333878,0.119117030896,0.119114728047,0.119112425333,0.119110122752,0.119107820305,0.119105517992,0.119103215812,0.119100913766,0.119098611854,0.119096310075,0.11909400843,0.119091706918,0.119089405541,0.119087104296,0.119084803186,0.119082502208,0.119080201365,0.119077900655,0.119075600078,0.119073299635,0.119070999326,0.119068699149,0.119066399107,0.119064099198,0.119061799422,0.11905949978,0.119057200271,0.119054900895,0.119052601653,0.119050302545,0.119048003569,0.119045704727,0.119043406019,0.119041107444,0.119038809002,0.119036510693,0.119034212518,0.119031914475,0.119029616567,0.119027318791,0.119025021149,0.11902272364,0.119020426264,0.119018129021,0.119015831911,0.119013534935,0.119011238092,0.119008941382,0.119006644805,0.119004348361,0.11900205205,0.118999755873,0.118997459828,0.118995163917,0.118992868138,0.118990572493,0.118988276981,0.118985981602,0.118983686355,0.118981391242,0.118979096262,0.118976801414,0.1189745067,0.118972212119,0.11896991767,0.118967623354,0.118965329172,0.118963035122,0.118960741205,0.118958447421,0.11895615377,0.118953860251,0.118951566866,0.118949273613,0.118946980493,0.118944687506,0.118942394652,0.11894010193,0.118937809341,0.118935516885,0.118933224562,0.118930932371,0.118928640313,0.118926348388,0.118924056595,0.118921764935,0.118919473408,0.118917182013,0.118914890751,0.118912599622,0.118910308625,0.11890801776,0.118905727029,0.11890343643,0.118901145963,0.118898855629,0.118896565427,0.118894275358,0.118891985422,0.118889695618,0.118887405946,0.118885116407,0.118882827,0.118880537726,0.118878248584,0.118875959574,0.118873670697,0.118871381953,0.11886909334,0.11886680486,0.118864516513,0.118862228297,0.118859940214,0.118857652263,0.118855364445,0.118853076759,0.118850789205,0.118848501783,0.118846214494,0.118843927337,0.118841640312,0.118839353419,0.118837066658,0.11883478003,0.118832493534,0.11883020717,0.118827920938,0.118825634838,0.11882334887,0.118821063035,0.118818777331,0.11881649176,0.11881420632,0.118811921013,0.118809635837,0.118807350794,0.118805065883,0.118802781103,0.118800496456,0.118798211941,0.118795927557,0.118793643306,0.118791359186,0.118789075199,0.118786791343,0.118784507619,0.118782224027,0.118779940567,0.118777657239,0.118775374043,0.118773090978,0.118770808045,0.118768525244,0.118766242575,0.118763960038,0.118761677632,0.118759395359,0.118757113216,0.118754831206,0.118752549327,0.11875026758,0.118747985965,0.118745704482,0.11874342313,0.118741141909,0.118738860821,0.118736579864,0.118734299038,0.118732018345,0.118729737783,0.118727457352,0.118725177053,0.118722896885,0.118720616849,0.118718336945,0.118716057172,0.118713777531,0.118711498021,0.118709218642,0.118706939395,0.11870466028,0.118702381296,0.118700102443,0.118697823722,0.118695545132,0.118693266673,0.118690988346,0.11868871015,0.118686432086,0.118684154153,0.118681876351,0.118679598681,0.118677321141,0.118675043734,0.118672766457,0.118670489312,0.118668212298,0.118665935415,0.118663658663,0.118661382043,0.118659105554,0.118656829195,0.118654552969,0.118652276873,0.118650000908,0.118647725075,0.118645449373,0.118643173801,0.118640898361,0.118638623052,0.118636347874,0.118634072827,0.118631797912,0.118629523127,0.118627248473,0.11862497395,0.118622699558,0.118620425298,0.118618151168,0.118615877169,0.118613603301,0.118611329564,0.118609055958,0.118606782483,0.118604509138,0.118602235925,0.118599962843,0.118597689891,0.11859541707,0.11859314438,0.118590871821,0.118588599392,0.118586327095,0.118584054928,0.118581782892,0.118579510987,0.118577239212,0.118574967568,0.118572696055,0.118570424673,0.118568153421,0.1185658823,0.11856361131,0.11856134045,0.118559069721,0.118556799123,0.118554528655,0.118552258318,0.118549988111,0.118547718035,0.11854544809,0.118543178275,0.118540908591,0.118538639037,0.118536369613,0.118534100321,0.118531831158,0.118529562126,0.118527293225,0.118525024454,0.118522755814,0.118520487304,0.118518218924,0.118515950675,0.118513682556,0.118511414568,0.11850914671,0.118506878982,0.118504611385,0.118502343918,0.118500076581,0.118497809374,0.118495542298,0.118493275352,0.118491008537,0.118488741852,0.118486475296,0.118484208872,0.118481942577,0.118479676413,0.118477410378,0.118475144474,0.1184728787,0.118470613057,0.118468347543,0.11846608216,0.118463816906,0.118461551783,0.11845928679,0.118457021927,0.118454757194,0.118452492591,0.118450228118,0.118447963775,0.118445699562,0.118443435479,0.118441171527,0.118438907704,0.118436644011,0.118434380448,0.118432117015,0.118429853712,0.118427590539,0.118425327495,0.118423064582,0.118420801798,0.118418539145,0.118416276621,0.118414014227,0.118411751963,0.118409489829,0.118407227824,0.11840496595,0.118402704205,0.11840044259,0.118398181104,0.118395919749,0.118393658523,0.118391397427,0.11838913646,0.118386875624,0.118384614917,0.118382354339,0.118380093891,0.118377833573,0.118375573385,0.118373313326,0.118371053397,0.118368793597,0.118366533927,0.118364274386,0.118362014976,0.118359755694,0.118357496542,0.11835523752,0.118352978627,0.118350719864,0.11834846123,0.118346202726,0.118343944351,0.118341686105,0.118339427989,0.118337170003,0.118334912145,0.118332654418,0.118330396819,0.11832813935,0.11832588201,0.1183236248,0.118321367719,0.118319110767,0.118316853945,0.118314597252,0.118312340688,0.118310084254,0.118307827948,0.118305571772,0.118303315725,0.118301059808,0.11829880402,0.11829654836,0.11829429283,0.11829203743,0.118289782158,0.118287527016,0.118285272002,0.118283017118,0.118280762363,0.118278507737,0.11827625324,0.118273998872,0.118271744633,0.118269490524,0.118267236543,0.118264982691,0.118262728969,0.118260475375,0.11825822191,0.118255968575,0.118253715368,0.11825146229,0.118249209342,0.118246956522,0.118244703831,0.118242451269,0.118240198835,0.118237946531,0.118235694356,0.118233442309,0.118231190391,0.118228938602,0.118226686942,0.118224435411,0.118222184009,0.118219932735,0.11821768159,0.118215430574,0.118213179686,0.118210928928,0.118208678298,0.118206427796,0.118204177424,0.11820192718,0.118199677065,0.118197427078,0.11819517722,0.118192927491,0.11819067789,0.118188428418,0.118186179074,0.11818392986,0.118181680773,0.118179431815,0.118177182986,0.118174934285,0.118172685713,0.11817043727,0.118168188954,0.118165940768,0.11816369271,0.11816144478,0.118159196979,0.118156949306,0.118154701761,0.118152454345,0.118150207058,0.118147959898,0.118145712868,0.118143465965,0.118141219191,0.118138972545,0.118136726028,0.118134479638,0.118132233378,0.118129987245,0.118127741241,0.118125495365,0.118123249617,0.118121003997,0.118118758506,0.118116513143,0.118114267908,0.118112022801,0.118109777823,0.118107532972,0.11810528825,0.118103043656,0.11810079919,0.118098554853,0.118096310643,0.118094066561,0.118091822608,0.118089578782,0.118087335085,0.118085091515,0.118082848074,0.118080604761,0.118078361576,0.118076118518,0.118073875589,0.118071632788,0.118069390114,0.118067147569,0.118064905151,0.118062662862,0.1180604207,0.118058178666,0.118055936761,0.118053694983,0.118051453332,0.11804921181,0.118046970416,0.118044729149,0.11804248801,0.118040246999,0.118038006116,0.118035765361,0.118033524733,0.118031284233,0.118029043861,0.118026803616,0.1180245635,0.118022323511,0.118020083649,0.118017843916,0.11801560431,0.118013364831,0.118011125481,0.118008886258,0.118006647162,0.118004408194,0.118002169354,0.117999930641,0.117997692056,0.117995453599,0.117993215269,0.117990977066,0.117988738992,0.117986501044,0.117984263224,0.117982025532,0.117979787967,0.117977550529,0.117975313219,0.117973076037,0.117970838982,0.117968602054,0.117966365253,0.11796412858,0.117961892035,0.117959655617,0.117957419326,0.117955183162,0.117952947126,0.117950711217,0.117948475436,0.117946239781,0.117944004254,0.117941768855,0.117939533582,0.117937298437,0.117935063419,0.117932828528,0.117930593764,0.117928359128,0.117926124619,0.117923890237,0.117921655982,0.117919421854,0.117917187854,0.11791495398,0.117912720234,0.117910486614,0.117908253122,0.117906019757,0.117903786519,0.117901553408,0.117899320424,0.117897087567,0.117894854837,0.117892622234,0.117890389759,0.11788815741,0.117885925188,0.117883693093,0.117881461125,0.117879229283,0.117876997569,0.117874765982,0.117872534521,0.117870303188,0.117868071981,0.117865840901,0.117863609948,0.117861379122,0.117859148423,0.117856917851,0.117854687405,0.117852457086,0.117850226894,0.117847996828,0.11784576689,0.117843537078,0.117841307393,0.117839077834,0.117836848402,0.117834619097,0.117832389919,0.117830160867,0.117827931942,0.117825703144,0.117823474472,0.117821245927,0.117819017508,0.117816789216,0.117814561051,0.117812333012,0.1178101051,0.117807877314,0.117805649655,0.117803422122,0.117801194716,0.117798967436,0.117796740283,0.117794513256,0.117792286356,0.117790059582,0.117787832935,0.117785606414,0.117783380019,0.117781153751,0.11777892761,0.117776701594,0.117774475705,0.117772249943,0.117770024306,0.117767798796,0.117765573413,0.117763348155,0.117761123024,0.11775889802,0.117756673141,0.117754448389,0.117752223763,0.117749999263,0.11774777489,0.117745550642,0.117743326521,0.117741102526,0.117738878657,0.117736654915,0.117734431298,0.117732207808,0.117729984444,0.117727761206,0.117725538094,0.117723315108,0.117721092248,0.117718869514,0.117716646907,0.117714424425,0.117712202069,0.11770997984,0.117707757736,0.117705535759,0.117703313907,0.117701092181,0.117698870582,0.117696649108,0.11769442776,0.117692206538,0.117689985442,0.117687764472,0.117685543628,0.11768332291,0.117681102318,0.117678881851,0.11767666151,0.117674441295,0.117672221206,0.117670001243,0.117667781406,0.117665561694,0.117663342108,0.117661122648,0.117658903313,0.117656684105,0.117654465022,0.117652246064,0.117650027233,0.117647808527,0.117645589947,0.117643371492,0.117641153163,0.11763893496,0.117636716882,0.11763449893,0.117632281104,0.117630063403,0.117627845828,0.117625628378,0.117623411054,0.117621193855,0.117618976782,0.117616759835,0.117614543013,0.117612326316,0.117610109745,0.1176078933,0.117605676979,0.117603460785,0.117601244716,0.117599028772,0.117596812953,0.11759459726,0.117592381693,0.11759016625,0.117587950934,0.117585735742,0.117583520676,0.117581305735,0.11757909092,0.117576876229,0.117574661664,0.117572447225,0.11757023291,0.117568018721,0.117565804658,0.117563590719,0.117561376906,0.117559163217,0.117556949654,0.117554736217,0.117552522904,0.117550309716,0.117548096654,0.117545883717,0.117543670905,0.117541458218,0.117539245656,0.11753703322,0.117534820908,0.117532608721,0.11753039666,0.117528184723,0.117525972912,0.117523761226,0.117521549664,0.117519338228,0.117517126916,0.11751491573,0.117512704668,0.117510493732,0.11750828292,0.117506072234,0.117503861672,0.117501651235,0.117499440923,0.117497230736,0.117495020674,0.117492810737,0.117490600924,0.117488391237,0.117486181674,0.117483972236,0.117481762922,0.117479553734,0.11747734467,0.117475135732,0.117472926917,0.117470718228,0.117468509663,0.117466301223,0.117464092908,0.117461884718,0.117459676652,0.117457468711,0.117455260894,0.117453053202,0.117450845635,0.117448638192,0.117446430874,0.117444223681,0.117442016612,0.117439809668,0.117437602848,0.117435396153,0.117433189582,0.117430983136,0.117428776815,0.117426570617,0.117424364545,0.117422158597,0.117419952773,0.117417747074,0.117415541499,0.117413336049,0.117411130723,0.117408925522,0.117406720445,0.117404515492,0.117402310664,0.11740010596,0.11739790138,0.117395696925,0.117393492594,0.117391288388,0.117389084305,0.117386880348,0.117384676514,0.117382472804,0.117380269219,0.117378065758,0.117375862422,0.117373659209,0.117371456121,0.117369253157,0.117367050317,0.117364847602,0.11736264501,0.117360442543,0.117358240199,0.11735603798,0.117353835885,0.117351633915,0.117349432068,0.117347230345,0.117345028747,0.117342827272,0.117340625922,0.117338424695,0.117336223593,0.117334022614,0.11733182176,0.11732962103,0.117327420423,0.117325219941,0.117323019582,0.117320819348,0.117318619237,0.11731641925,0.117314219387,0.117312019649,0.117309820034,0.117307620542,0.117305421175,0.117303221932,0.117301022812,0.117298823816,0.117296624944,0.117294426196,0.117292227572,0.117290029071,0.117287830695,0.117285632442,0.117283434312,0.117281236307,0.117279038425,0.117276840667,0.117274643032,0.117272445522,0.117270248135,0.117268050871,0.117265853731,0.117263656715,0.117261459823,0.117259263054,0.117257066409,0.117254869887,0.117252673489,0.117250477215,0.117248281064,0.117246085036,0.117243889132,0.117241693352,0.117239497695,0.117237302162,0.117235106752,0.117232911466,0.117230716303,0.117228521264,0.117226326348,0.117224131555,0.117221936886,0.11721974234,0.117217547918,0.117215353619,0.117213159444,0.117210965391,0.117208771463,0.117206577657,0.117204383975,0.117202190416,0.117199996981,0.117197803668,0.117195610479,0.117193417414,0.117191224471,0.117189031652,0.117186838956,0.117184646383,0.117182453934,0.117180261608,0.117178069404,0.117175877325,0.117173685368,0.117171493534,0.117169301824,0.117167110236,0.117164918772,0.117162727431,0.117160536213,0.117158345118,0.117156154146,0.117153963298,0.117151772572,0.117149581969,0.117147391489,0.117145201133,0.117143010899,0.117140820789,0.117138630801,0.117136440936,0.117134251194,0.117132061576,0.11712987208,0.117127682707,0.117125493457,0.11712330433,0.117121115326,0.117118926444,0.117116737686,0.11711454905,0.117112360537,0.117110172147,0.11710798388,0.117105795736,0.117103607715,0.117101419816,0.11709923204,0.117097044387,0.117094856856,0.117092669449,0.117090482164,0.117088295001,0.117086107962,0.117083921045,0.117081734251,0.117079547579,0.11707736103,0.117075174604,0.117072988301,0.11707080212,0.117068616062,0.117066430126,0.117064244313,0.117062058622,0.117059873054,0.117057687609,0.117055502286,0.117053317086,0.117051132008,0.117048947053,0.11704676222,0.117044577509,0.117042392922,0.117040208456,0.117038024113,0.117035839893,0.117033655795,0.117031471819,0.117029287966,0.117027104235,0.117024920627,0.117022737141,0.117020553777,0.117018370536,0.117016187417,0.11701400442,0.117011821546,0.117009638794,0.117007456164,0.117005273656,0.117003091271,0.117000909008,0.116998726868,0.116996544849,0.116994362953,0.116992181179,0.116989999527,0.116987817997,0.11698563659,0.116983455305,0.116981274142,0.116979093101,0.116976912182,0.116974731385,0.11697255071,0.116970370158,0.116968189727,0.116966009419,0.116963829233,0.116961649169,0.116959469226,0.116957289406,0.116955109708,0.116952930132,0.116950750678,0.116948571346,0.116946392136,0.116944213047,0.116942034081,0.116939855237,0.116937676514,0.116935497914,0.116933319435,0.116931141079,0.116928962844,0.116926784731,0.11692460674,0.116922428871,0.116920251123,0.116918073498,0.116915895994,0.116913718612,0.116911541352,0.116909364214,0.116907187197,0.116905010302,0.116902833529,0.116900656878,0.116898480348,0.11689630394,0.116894127654,0.116891951489,0.116889775447,0.116887599525,0.116885423726,0.116883248048,0.116881072492,0.116878897057,0.116876721744,0.116874546553,0.116872371483,0.116870196535,0.116868021708,0.116865847003,0.116863672419,0.116861497957,0.116859323617,0.116857149398,0.1168549753,0.116852801324,0.116850627469,0.116848453736,0.116846280125,0.116844106634,0.116841933266,0.116839760018,0.116837586892,0.116835413888,0.116833241005,0.116831068243,0.116828895602,0.116826723083,0.116824550685,0.116822378409,0.116820206254,0.11681803422,0.116815862307,0.116813690516,0.116811518846,0.116809347297,0.11680717587,0.116805004564,0.116802833379,0.116800662315,0.116798491372,0.116796320551,0.116794149851,0.116791979272,0.116789808814,0.116787638477,0.116785468261,0.116783298167,0.116781128194,0.116778958341,0.11677678861,0.116774619,0.116772449511,0.116770280143,0.116768110896,0.11676594177,0.116763772765,0.116761603881,0.116759435118,0.116757266477,0.116755097956,0.116752929556,0.116750761277,0.116748593119,0.116746425082,0.116744257165,0.11674208937,0.116739921696,0.116737754142,0.11673558671,0.116733419398,0.116731252207,0.116729085137,0.116726918188,0.116724751359,0.116722584652,0.116720418065,0.116718251599,0.116716085253,0.116713919029,0.116711752925,0.116709586942,0.11670742108,0.116705255338,0.116703089718,0.116700924217,0.116698758838,0.116696593579,0.116694428441,0.116692263424,0.116690098527,0.116687933751,0.116685769095,0.11668360456,0.116681440146,0.116679275852,0.116677111679,0.116674947626,0.116672783694,0.116670619882,0.116668456191,0.116666292621,0.116664129171,0.116661965841,0.116659802632,0.116657639543,0.116655476575,0.116653313728,0.116651151,0.116648988394,0.116646825907,0.116644663541,0.116642501296,0.11664033917,0.116638177166,0.116636015281,0.116633853517,0.116631691873,0.11662953035,0.116627368947,0.116625207664,0.116623046501,0.116620885459,0.116618724537,0.116616563735,0.116614403054,0.116612242492,0.116610082051,0.116607921731,0.11660576153,0.11660360145,0.116601441489,0.116599281649,0.116597121929,0.11659496233,0.11659280285,0.11659064349,0.116588484251,0.116586325132,0.116584166133,0.116582007254,0.116579848494,0.116577689855,0.116575531337,0.116573372938,0.116571214659,0.1165690565,0.116566898461,0.116564740542,0.116562582743,0.116560425064,0.116558267505,0.116556110066,0.116553952747,0.116551795548,0.116549638469,0.11654748151,0.11654532467,0.116543167951,0.116541011351,0.116538854871,0.116536698511,0.116534542271,0.116532386151,0.11653023015,0.116528074269,0.116525918508,0.116523762867,0.116521607346,0.116519451944,0.116517296662,0.1165151415,0.116512986458,0.116510831535,0.116508676732,0.116506522049,0.116504367485,0.116502213041,0.116500058717,0.116497904512,0.116495750427,0.116493596462,0.116491442616,0.11648928889,0.116487135284,0.116484981797,0.116482828429,0.116480675181,0.116478522053,0.116476369044,0.116474216155,0.116472063385,0.116469910735,0.116467758204,0.116465605793,0.116463453501,0.116461301329,0.116459149276,0.116456997342,0.116454845528,0.116452693834,0.116450542259,0.116448390803,0.116446239466,0.116444088249,0.116441937151,0.116439786173,0.116437635314,0.116435484574,0.116433333954,0.116431183453,0.116429033071,0.116426882809,0.116424732665,0.116422582641,0.116420432737,0.116418282951,0.116416133285,0.116413983738,0.11641183431,0.116409685001,0.116407535812,0.116405386742,0.116403237791,0.116401088959,0.116398940246,0.116396791652,0.116394643177,0.116392494822,0.116390346586,0.116388198468,0.11638605047,0.116383902591,0.116381754831,0.11637960719,0.116377459668,0.116375312265,0.11637316498,0.116371017815,0.116368870769,0.116366723842,0.116364577034,0.116362430345,0.116360283775,0.116358137323,0.116355990991,0.116353844778,0.116351698683,0.116349552707,0.11634740685,0.116345261113,0.116343115493,0.116340969993,0.116338824612,0.116336679349,0.116334534205,0.11633238918,0.116330244274,0.116328099487,0.116325954818,0.116323810268,0.116321665837,0.116319521524,0.116317377331,0.116315233256,0.116313089299,0.116310945462,0.116308801743,0.116306658142,0.116304514661,0.116302371298,0.116300228053,0.116298084928,0.116295941921,0.116293799032,0.116291656262,0.116289513611,0.116287371078,0.116285228664,0.116283086368,0.116280944191,0.116278802132,0.116276660192,0.11627451837,0.116272376667,0.116270235083,0.116268093616,0.116265952269,0.116263811039,0.116261669929,0.116259528936,0.116257388062,0.116255247307,0.116253106669,0.116250966151,0.11624882575,0.116246685468,0.116244545304,0.116242405259,0.116240265332,0.116238125523,0.116235985832,0.11623384626,0.116231706806,0.116229567471,0.116227428253,0.116225289154,0.116223150173,0.116221011311,0.116218872566,0.11621673394,0.116214595432,0.116212457042,0.116210318771,0.116208180617,0.116206042582,0.116203904665,0.116201766866,0.116199629185,0.116197491622,0.116195354177,0.11619321685,0.116191079642,0.116188942551,0.116186805579,0.116184668724,0.116182531988,0.11618039537,0.116178258869,0.116176122487,0.116173986223,0.116171850076,0.116169714048,0.116167578137,0.116165442345,0.11616330667,0.116161171113,0.116159035675,0.116156900354,0.116154765151,0.116152630066,0.116150495099,0.116148360249,0.116146225518,0.116144090904,0.116141956408,0.11613982203,0.11613768777,0.116135553628,0.116133419603,0.116131285696,0.116129151907,0.116127018236,0.116124884682,0.116122751246,0.116120617928,0.116118484727,0.116116351645,0.11611421868,0.116112085832,0.116109953102,0.11610782049,0.116105687996,0.116103555619,0.11610142336,0.116099291218,0.116097159194,0.116095027288,0.116092895499,0.116090763828,0.116088632274,0.116086500838,0.116084369519,0.116082238318,0.116080107234,0.116077976268,0.11607584542,0.116073714688,0.116071584075,0.116069453579,0.1160673232,0.116065192939,0.116063062795,0.116060932768,0.116058802859,0.116056673067,0.116054543393,0.116052413836,0.116050284397,0.116048155074,0.116046025869,0.116043896782,0.116041767812,0.116039638959,0.116037510223,0.116035381605,0.116033253104,0.11603112472,0.116028996453,0.116026868304,0.116024740272,0.116022612357,0.116020484559,0.116018356879,0.116016229315,0.116014101869,0.11601197454,0.116009847329,0.116007720234,0.116005593257,0.116003466396,0.116001339653,0.115999213027,0.115997086518,0.115994960126,0.115992833851,0.115990707693,0.115988581652,0.115986455728,0.115984329922,0.115982204232,0.115980078659,0.115977953203,0.115975827865,0.115973702643,0.115971577538,0.11596945255,0.115967327679,0.115965202925,0.115963078288,0.115960953768,0.115958829365,0.115956705078,0.115954580909,0.115952456856,0.11595033292,0.115948209101,0.115946085399,0.115943961814,0.115941838346,0.115939714994,0.115937591759,0.115935468641,0.11593334564,0.115931222755,0.115929099987,0.115926977336,0.115924854802,0.115922732384,0.115920610083,0.115918487899,0.115916365831,0.11591424388,0.115912122046,0.115910000328,0.115907878728,0.115905757243,0.115903635876,0.115901514624,0.11589939349,0.115897272472,0.115895151571,0.115893030786,0.115890910118,0.115888789566,0.115886669131,0.115884548812,0.11588242861,0.115880308525,0.115878188556,0.115876068703,0.115873948967,0.115871829347,0.115869709844,0.115867590457,0.115865471187,0.115863352033,0.115861232995,0.115859114074,0.115856995269,0.115854876581,0.115852758009,0.115850639553,0.115848521214,0.115846402991,0.115844284884,0.115842166893,0.115840049019,0.115837931262,0.11583581362,0.115833696095,0.115831578686,0.115829461393,0.115827344216,0.115825227156,0.115823110212,0.115820993384,0.115818876672,0.115816760077,0.115814643598,0.115812527234,0.115810410987,0.115808294857,0.115806178842,0.115804062943,0.115801947161,0.115799831494,0.115797715944,0.115795600509,0.115793485191,0.115791369989,0.115789254903,0.115787139933,0.115785025079,0.115782910341,0.115780795719,0.115778681213,0.115776566823,0.115774452548,0.11577233839,0.115770224348,0.115768110422,0.115765996611,0.115763882917,0.115761769338,0.115759655876,0.115757542529,0.115755429298,0.115753316183,0.115751203184,0.115749090301,0.115746977533,0.115744864882,0.115742752346,0.115740639926,0.115738527621,0.115736415433,0.11573430336,0.115732191403,0.115730079562,0.115727967836,0.115725856227,0.115723744733,0.115721633354,0.115719522092,0.115717410945,0.115715299913,0.115713188998,0.115711078198,0.115708967513,0.115706856945,0.115704746492,0.115702636154,0.115700525932,0.115698415826,0.115696305835,0.11569419596,0.115692086201,0.115689976556,0.115687867028,0.115685757615,0.115683648317,0.115681539135,0.115679430069,0.115677321118,0.115675212282,0.115673103562,0.115670994958,0.115668886469,0.115666778095,0.115664669837,0.115662561694,0.115660453666,0.115658345754,0.115656237957,0.115654130276,0.11565202271,0.115649915259,0.115647807924,0.115645700704,0.115643593599,0.11564148661,0.115639379735,0.115637272977,0.115635166333,0.115633059805,0.115630953392,0.115628847094,0.115626740911,0.115624634844,0.115622528892,0.115620423055,0.115618317333,0.115616211726,0.115614106235,0.115612000859,0.115609895597,0.115607790451,0.115605685421,0.115603580505,0.115601475704,0.115599371019,0.115597266448,0.115595161993,0.115593057652,0.115590953427,0.115588849317,0.115586745321,0.115584641441,0.115582537676,0.115580434026,0.115578330491,0.11557622707,0.115574123765,0.115572020575,0.115569917499,0.115567814539,0.115565711693,0.115563608963,0.115561506347,0.115559403846,0.11555730146,0.115555199189,0.115553097033,0.115550994992,0.115548893065,0.115546791254,0.115544689557,0.115542587975,0.115540486508,0.115538385155,0.115536283917,0.115534182794,0.115532081786,0.115529980893,0.115527880114,0.11552577945,0.115523678901,0.115521578467,0.115519478147,0.115517377942,0.115515277851,0.115513177875,0.115511078014,0.115508978268,0.115506878636,0.115504779119,0.115502679716,0.115500580428,0.115498481254,0.115496382196,0.115494283251,0.115492184422,0.115490085706,0.115487987106,0.11548588862,0.115483790248,0.115481691991,0.115479593848,0.11547749582,0.115475397906,0.115473300107,0.115471202422,0.115469104852,0.115467007396,0.115464910055,0.115462812828,0.115460715715,0.115458618717,0.115456521833,0.115454425064,0.115452328409,0.115450231868,0.115448135441,0.115446039129,0.115443942932,0.115441846848,0.115439750879,0.115437655024,0.115435559283,0.115433463657,0.115431368145,0.115429272747,0.115427177464,0.115425082294,0.115422987239,0.115420892298,0.115418797471,0.115416702759,0.11541460816,0.115412513676,0.115410419306,0.11540832505,0.115406230908,0.115404136881,0.115402042967,0.115399949168,0.115397855482,0.115395761911,0.115393668454,0.11539157511,0.115389481881,0.115387388766,0.115385295765,0.115383202878,0.115381110105,0.115379017446,0.115376924901,0.11537483247,0.115372740153,0.11537064795,0.11536855586,0.115366463885,0.115364372024,0.115362280276,0.115360188643,0.115358097123,0.115356005717,0.115353914426,0.115351823248,0.115349732183,0.115347641233,0.115345550397,0.115343459674,0.115341369065,0.11533927857,0.115337188189,0.115335097921,0.115333007768,0.115330917728,0.115328827801,0.115326737989,0.11532464829,0.115322558705,0.115320469234,0.115318379876,0.115316290632,0.115314201502,0.115312112485,0.115310023582,0.115307934793,0.115305846117,0.115303757555,0.115301669107,0.115299580772,0.115297492551,0.115295404443,0.115293316449,0.115291228569,0.115289140802,0.115287053148,0.115284965609,0.115282878182,0.115280790869,0.11527870367,0.115276616584,0.115274529612,0.115272442753,0.115270356008,0.115268269376,0.115266182857,0.115264096452,0.11526201016,0.115259923982,0.115257837917,0.115255751966,0.115253666127,0.115251580403,0.115249494791,0.115247409293,0.115245323908,0.115243238637,0.115241153479,0.115239068434,0.115236983503,0.115234898685,0.11523281398,0.115230729388,0.11522864491,0.115226560544,0.115224476293,0.115222392154,0.115220308128,0.115218224216,0.115216140417,0.115214056731,0.115211973158,0.115209889699,0.115207806352,0.115205723119,0.115203639999,0.115201556992,0.115199474098,0.115197391317,0.11519530865,0.115193226095,0.115191143653,0.115189061325,0.115186979109,0.115184897007,0.115182815018,0.115180733141,0.115178651378,0.115176569727,0.11517448819,0.115172406766,0.115170325454,0.115168244256,0.11516616317,0.115164082198,0.115162001338,0.115159920591,0.115157839957,0.115155759436,0.115153679028,0.115151598733,0.115149518551,0.115147438481,0.115145358525,0.115143278681,0.11514119895,0.115139119332,0.115137039827,0.115134960434,0.115132881154,0.115130801987,0.115128722933,0.115126643992,0.115124565163,0.115122486447,0.115120407844,0.115118329353,0.115116250975,0.11511417271,0.115112094557,0.115110016518,0.11510793859,0.115105860776,0.115103783074,0.115101705485,0.115099628008,0.115097550644,0.115095473393,0.115093396254,0.115091319228,0.115089242314,0.115087165513,0.115085088824,0.115083012248,0.115080935785,0.115078859434,0.115076783195,0.115074707069,0.115072631056,0.115070555155,0.115068479366,0.11506640369,0.115064328127,0.115062252676,0.115060177337,0.11505810211,0.115056026996,0.115053951995,0.115051877106,0.115049802329,0.115047727665,0.115045653113,0.115043578673,0.115041504345,0.11503943013,0.115037356028,0.115035282037,0.115033208159,0.115031134393,0.11502906074,0.115026987198,0.115024913769,0.115022840452,0.115020767248,0.115018694155,0.115016621175,0.115014548307,0.115012475552,0.115010402908,0.115008330377,0.115006257957,0.11500418565,0.115002113455,0.115000041373,0.114997969402,0.114995897543,0.114993825797,0.114991754163,0.11498968264,0.11498761123,0.114985539932,0.114983468746,0.114981397672,0.11497932671,0.11497725586,0.114975185122,0.114973114496,0.114971043982,0.11496897358,0.11496690329,0.114964833112,0.114962763045,0.114960693091,0.114958623249,0.114956553519,0.1149544839,0.114952414394,0.114950344999,0.114948275716,0.114946206545,0.114944137486,0.114942068539,0.114939999704,0.11493793098,0.114935862369,0.114933793869,0.114931725481,0.114929657204,0.11492758904,0.114925520987,0.114923453046,0.114921385217,0.114919317499,0.114917249894,0.114915182399,0.114913115017,0.114911047746,0.114908980587,0.11490691354,0.114904846605,0.114902779781,0.114900713068,0.114898646468,0.114896579979,0.114894513601,0.114892447335,0.114890381181,0.114888315139,0.114886249207,0.114884183388,0.11488211768,0.114880052084,0.114877986599,0.114875921226,0.114873855964,0.114871790813,0.114869725775,0.114867660847,0.114865596032,0.114863531327,0.114861466734,0.114859402253,0.114857337883,0.114855273624,0.114853209477,0.114851145441,0.114849081517,0.114847017704,0.114844954002,0.114842890412,0.114840826933,0.114838763566,0.114836700309,0.114834637165,0.114832574131,0.114830511209,0.114828448398,0.114826385698,0.11482432311,0.114822260633,0.114820198267,0.114818136012,0.114816073869,0.114814011837,0.114811949916,0.114809888106,0.114807826407,0.11480576482,0.114803703344,0.114801641979,0.114799580725,0.114797519582,0.114795458551,0.11479339763,0.114791336821,0.114789276123,0.114787215536,0.11478515506,0.114783094695,0.114781034441,0.114778974298,0.114776914266,0.114774854345,0.114772794536,0.114770734837,0.114768675249,0.114766615773,0.114764556407,0.114762497152,0.114760438008,0.114758378976,0.114756320054,0.114754261243,0.114752202543,0.114750143954,0.114748085475,0.114746027108,0.114743968852,0.114741910706,0.114739852672,0.114737794748,0.114735736935,0.114733679233,0.114731621641,0.114729564161,0.114727506791,0.114725449532,0.114723392384,0.114721335347,0.114719278421,0.114717221605,0.1147151649,0.114713108306,0.114711051822,0.114708995449,0.114706939187,0.114704883036,0.114702826995,0.114700771065,0.114698715245,0.114696659537,0.114694603938,0.114692548451,0.114690493074,0.114688437808,0.114686382652,0.114684327607,0.114682272673,0.114680217849,0.114678163136,0.114676108533,0.114674054041,0.114671999659,0.114669945388,0.114667891228,0.114665837178,0.114663783238,0.114661729409,0.11465967569,0.114657622082,0.114655568585,0.114653515197,0.114651461921,0.114649408754,0.114647355698,0.114645302753,0.114643249918,0.114641197193,0.114639144579,0.114637092075,0.114635039681,0.114632987398,0.114630935225,0.114628883162,0.11462683121,0.114624779368,0.114622727636,0.114620676015,0.114618624504,0.114616573103,0.114614521812,0.114612470632,0.114610419562,0.114608368602,0.114606317753,0.114604267013,0.114602216384,0.114600165865,0.114598115456,0.114596065157,0.114594014969,0.114591964891,0.114589914922,0.114587865064,0.114585815316,0.114583765679,0.114581716151,0.114579666733,0.114577617426,0.114575568228,0.114573519141,0.114571470164,0.114569421296,0.114567372539,0.114565323892,0.114563275354,0.114561226927,0.11455917861,0.114557130403,0.114555082305,0.114553034318,0.114550986441,0.114548938673,0.114546891016,0.114544843468,0.114542796031,0.114540748703,0.114538701485,0.114536654377,0.114534607379,0.114532560491,0.114530513713,0.114528467044,0.114526420485,0.114524374037,0.114522327698,0.114520281468,0.114518235349,0.114516189339,0.11451414344,0.11451209765,0.114510051969,0.114508006399,0.114505960938,0.114503915587,0.114501870345,0.114499825214,0.114497780192,0.114495735279,0.114493690477,0.114491645784,0.114489601201,0.114487556727,0.114485512363,0.114483468109,0.114481423964,0.114479379929,0.114477336003,0.114475292187,0.114473248481,0.114471204884,0.114469161397,0.11446711802,0.114465074751,0.114463031593,0.114460988544,0.114458945604,0.114456902774,0.114454860054,0.114452817443,0.114450774941,0.114448732549,0.114446690266,0.114444648093,0.114442606029,0.114440564075,0.11443852223,0.114436480494,0.114434438868,0.114432397352,0.114430355944,0.114428314646,0.114426273458,0.114424232378,0.114422191408,0.114420150548,0.114418109796,0.114416069154,0.114414028622,0.114411988198,0.114409947884,0.114407907679,0.114405867584,0.114403827597,0.11440178772,0.114399747952,0.114397708293,0.114395668744,0.114393629304,0.114391589973,0.114389550751,0.114387511638,0.114385472634,0.11438343374,0.114381394955,0.114379356279,0.114377317712,0.114375279254,0.114373240905,0.114371202665,0.114369164535,0.114367126513,0.1143650886,0.114363050797,0.114361013103,0.114358975517,0.114356938041,0.114354900674,0.114352863415,0.114350826266,0.114348789226,0.114346752294,0.114344715472,0.114342678759,0.114340642154,0.114338605659,0.114336569272,0.114334532994,0.114332496825,0.114330460766,0.114328424815,0.114326388972,0.114324353239,0.114322317615,0.114320282099,0.114318246693,0.114316211395,0.114314176206,0.114312141125,0.114310106154,0.114308071291,0.114306036538,0.114304001892,0.114301967356,0.114299932929,0.11429789861,0.1142958644,0.114293830298,0.114291796306,0.114289762422,0.114287728646,0.11428569498,0.114283661422,0.114281627973,0.114279594632,0.1142775614,0.114275528277,0.114273495262,0.114271462356,0.114269429559,0.11426739687,0.11426536429,0.114263331818,0.114261299455,0.1142592672,0.114257235054,0.114255203017,0.114253171088,0.114251139268,0.114249107556,0.114247075952,0.114245044457,0.114243013071,0.114240981793,0.114238950624,0.114236919562,0.11423488861,0.114232857766,0.11423082703,0.114228796403,0.114226765884,0.114224735473,0.114222705171,0.114220674977,0.114218644892,0.114216614915,0.114214585046,0.114212555286,0.114210525634,0.11420849609,0.114206466655,0.114204437328,0.114202408109,0.114200378998,0.114198349996,0.114196321102,0.114194292316,0.114192263639,0.11419023507,0.114188206608,0.114186178256,0.114184150011,0.114182121875,0.114180093846,0.114178065926,0.114176038114,0.11417401041,0.114171982815,0.114169955327,0.114167927948,0.114165900677,0.114163873514,0.114161846459,0.114159819512,0.114157792673,0.114155765942,0.114153739319,0.114151712804,0.114149686398,0.114147660099,0.114145633908,0.114143607826,0.114141581851,0.114139555984,0.114137530226,0.114135504575,0.114133479032,0.114131453598,0.114129428271,0.114127403052,0.114125377941,0.114123352938,0.114121328043,0.114119303256,0.114117278576,0.114115254005,0.114113229541,0.114111205185,0.114109180937,0.114107156797,0.114105132765,0.114103108841,0.114101085024,0.114099061315,0.114097037714,0.114095014221,0.114092990835,0.114090967558,0.114088944388,0.114086921325,0.114084898371,0.114082875524,0.114080852785,0.114078830154,0.11407680763,0.114074785214,0.114072762905,0.114070740705,0.114068718612,0.114066696626,0.114064674749,0.114062652979,0.114060631316,0.114058609761,0.114056588314,0.114054566974,0.114052545742,0.114050524618,0.114048503601,0.114046482691,0.114044461889,0.114042441195,0.114040420608,0.114038400129,0.114036379757,0.114034359493,0.114032339336,0.114030319286,0.114028299345,0.11402627951,0.114024259783,0.114022240163,0.114020220651,0.114018201247,0.114016181949,0.114014162759,0.114012143677,0.114010124702,0.114008105834,0.114006087073,0.11400406842,0.114002049875,0.114000031436,0.113998013105,0.113995994881,0.113993976765,0.113991958756,0.113989940854,0.113987923059,0.113985905372,0.113983887792,0.113981870319,0.113979852953,0.113977835695,0.113975818543,0.113973801499,0.113971784563,0.113969767733,0.113967751011,0.113965734395,0.113963717887,0.113961701486,0.113959685192,0.113957669006,0.113955652926,0.113953636954,0.113951621088,0.11394960533,0.113947589679,0.113945574135,0.113943558698,0.113941543368,0.113939528145,0.113937513029,0.11393549802,0.113933483118,0.113931468324,0.113929453636,0.113927439055,0.113925424581,0.113923410214,0.113921395954,0.113919381801,0.113917367755,0.113915353816,0.113913339984,0.113911326259,0.11390931264,0.113907299129,0.113905285724,0.113903272426,0.113901259235,0.113899246151,0.113897233174,0.113895220304,0.11389320754,0.113891194884,0.113889182334,0.113887169891,0.113885157555,0.113883145325,0.113881133202,0.113879121186,0.113877109277,0.113875097475,0.113873085779,0.11387107419,0.113869062708,0.113867051332,0.113865040063,0.113863028901,0.113861017845,0.113859006896,0.113856996054,0.113854985319,0.11385297469,0.113850964167,0.113848953752,0.113846943443,0.11384493324,0.113842923144,0.113840913155,0.113838903272,0.113836893496,0.113834883827,0.113832874263,0.113830864807,0.113828855457,0.113826846213,0.113824837076,0.113822828046,0.113820819122,0.113818810304,0.113816801593,0.113814792989,0.113812784491,0.113810776099,0.113808767814,0.113806759635,0.113804751562,0.113802743596,0.113800735737,0.113798727983,0.113796720337,0.113794712796,0.113792705362,0.113790698034,0.113788690813,0.113786683697,0.113784676689,0.113782669786,0.11378066299,0.1137786563,0.113776649716,0.113774643239,0.113772636868,0.113770630603,0.113768624444,0.113766618392,0.113764612446,0.113762606606,0.113760600872,0.113758595245,0.113756589723,0.113754584308,0.113752578999,0.113750573796,0.113748568699,0.113746563709,0.113744558824,0.113742554046,0.113740549374,0.113738544808,0.113736540348,0.113734535994,0.113732531746,0.113730527604,0.113728523568,0.113726519639,0.113724515815,0.113722512098,0.113720508486,0.11371850498,0.113716501581,0.113714498287,0.1137124951,0.113710492018,0.113708489042,0.113706486172,0.113704483409,0.113702480751,0.113700478199,0.113698475753,0.113696473413,0.113694471179,0.11369246905,0.113690467028,0.113688465111,0.1136864633,0.113684461596,0.113682459997,0.113680458503,0.113678457116,0.113676455834,0.113674454659,0.113672453589,0.113670452624,0.113668451766,0.113666451013,0.113664450367,0.113662449825,0.11366044939,0.11365844906,0.113656448836,0.113654448718,0.113652448706,0.113650448799,0.113648448998,0.113646449302,0.113644449712,0.113642450228,0.113640450849,0.113638451577,0.113636452409,0.113634453348,0.113632454392,0.113630455541,0.113628456796,0.113626458157,0.113624459623,0.113622461195,0.113620462873,0.113618464655,0.113616466544,0.113614468538,0.113612470637,0.113610472842,0.113608475153,0.113606477569,0.11360448009,0.113602482717,0.11360048545,0.113598488288,0.113596491231,0.11359449428,0.113592497434,0.113590500693,0.113588504058,0.113586507529,0.113584511104,0.113582514786,0.113580518572,0.113578522464,0.113576526461,0.113574530564,0.113572534771,0.113570539085,0.113568543503,0.113566548027,0.113564552656,0.11356255739,0.11356056223,0.113558567175,0.113556572225,0.11355457738,0.113552582641,0.113550588007,0.113548593478,0.113546599054,0.113544604736,0.113542610522,0.113540616414,0.113538622411,0.113536628513,0.113534634721,0.113532641033,0.113530647451,0.113528653974,0.113526660601,0.113524667334,0.113522674172,0.113520681116,0.113518688164,0.113516695317,0.113514702575,0.113512709939,0.113510717407,0.113508724981,0.113506732659,0.113504740443,0.113502748331,0.113500756325,0.113498764423,0.113496772627,0.113494780935,0.113492789349,0.113490797867,0.11348880649,0.113486815218,0.113484824052,0.11348283299,0.113480842033,0.113478851181,0.113476860433,0.113474869791,0.113472879253,0.113470888821,0.113468898493,0.11346690827,0.113464918152,0.113462928138,0.11346093823,0.113458948426,0.113456958727,0.113454969133,0.113452979644,0.113450990259,0.113449000979,0.113447011804,0.113445022734,0.113443033768,0.113441044907,0.113439056151,0.1134370675,0.113435078953,0.113433090511,0.113431102173,0.113429113941,0.113427125812,0.113425137789,0.11342314987,0.113421162056,0.113419174346,0.113417186741,0.113415199241,0.113413211845,0.113411224554,0.113409237367,0.113407250285,0.113405263308,0.113403276435,0.113401289666,0.113399303002,0.113397316443,0.113395329988,0.113393343637,0.113391357391,0.11338937125,0.113387385213,0.113385399281,0.113383413452,0.113381427729,0.11337944211,0.113377456595,0.113375471185,0.113373485879,0.113371500677,0.11336951558,0.113367530587,0.113365545699,0.113363560915,0.113361576235,0.11335959166,0.113357607189,0.113355622822,0.11335363856,0.113351654402,0.113349670348,0.113347686398,0.113345702553,0.113343718812,0.113341735176,0.113339751643,0.113337768215,0.113335784891,0.113333801671,0.113331818556,0.113329835545,0.113327852638,0.113325869835,0.113323887136,0.113321904541,0.113319922051,0.113317939665,0.113315957383,0.113313975205,0.113311993131,0.113310011161,0.113308029295,0.113306047534,0.113304065876,0.113302084323,0.113300102874,0.113298121528,0.113296140287,0.11329415915,0.113292178117,0.113290197188,0.113288216363,0.113286235642,0.113284255025,0.113282274512,0.113280294102,0.113278313797,0.113276333596,0.113274353499,0.113272373506,0.113270393616,0.113268413831,0.113266434149,0.113264454572,0.113262475098,0.113260495728,0.113258516462,0.1132565373,0.113254558242,0.113252579287,0.113250600437,0.11324862169,0.113246643047,0.113244664508,0.113242686073,0.113240707741,0.113238729514,0.11323675139,0.11323477337,0.113232795453,0.113230817641,0.113228839932,0.113226862326,0.113224884825,0.113222907427,0.113220930133,0.113218952943,0.113216975856,0.113214998873,0.113213021994,0.113211045218,0.113209068546,0.113207091978,0.113205115513,0.113203139152,0.113201162895,0.113199186741,0.113197210691,0.113195234744,0.113193258901,0.113191283161,0.113189307525,0.113187331993,0.113185356564,0.113183381239,0.113181406017,0.113179430898,0.113177455884,0.113175480972,0.113173506165,0.11317153146,0.113169556859,0.113167582362,0.113165607968,0.113163633677,0.11316165949,0.113159685407,0.113157711426,0.113155737549,0.113153763776,0.113151790106,0.113149816539,0.113147843076,0.113145869716,0.113143896459,0.113141923306,0.113139950256,0.11313797731,0.113136004466,0.113134031726,0.11313205909,0.113130086556,0.113128114126,0.113126141799,0.113124169576,0.113122197455,0.113120225438,0.113118253524,0.113116281714,0.113114310006,0.113112338402,0.113110366901,0.113108395503,0.113106424209,0.113104453017,0.113102481929,0.113100510944,0.113098540062,0.113096569283,0.113094598607,0.113092628034,0.113090657565,0.113088687199,0.113086716935,0.113084746775,0.113082776718,0.113080806764,0.113078836913,0.113076867165,0.11307489752,0.113072927978,0.113070958539,0.113068989203,0.11306701997,0.11306505084,0.113063081813,0.113061112889,0.113059144068,0.11305717535,0.113055206735,0.113053238223,0.113051269813,0.113049301507,0.113047333304,0.113045365203,0.113043397205,0.113041429311,0.113039461519,0.11303749383,0.113035526244,0.11303355876,0.11303159138,0.113029624102,0.113027656928,0.113025689856,0.113023722886,0.11302175602,0.113019789256,0.113017822595,0.113015856037,0.113013889582,0.11301192323,0.11300995698,0.113007990833,0.113006024788,0.113004058847,0.113002093008,0.113000127271,0.112998161638,0.112996196107,0.112994230679,0.112992265353,0.11299030013,0.11298833501,0.112986369992,0.112984405077,0.112982440265,0.112980475555,0.112978510948,0.112976546443,0.112974582041,0.112972617742,0.112970653545,0.11296868945,0.112966725459,0.112964761569,0.112962797783,0.112960834098,0.112958870517,0.112956907037,0.112954943661,0.112952980387,0.112951017215,0.112949054145,0.112947091179,0.112945128314,0.112943165552,0.112941202893,0.112939240335,0.112937277881,0.112935315528,0.112933353278,0.112931391131,0.112929429086,0.112927467143,0.112925505302,0.112923543564,0.112921581929,0.112919620395,0.112917658964,0.112915697635,0.112913736409,0.112911775284,0.112909814262,0.112907853343,0.112905892525,0.11290393181,0.112901971197,0.112900010687,0.112898050278,0.112896089972,0.112894129768,0.112892169666,0.112890209667,0.112888249769,0.112886289974,0.112884330281,0.11288237069,0.112880411202,0.112878451815,0.112876492531,0.112874533348,0.112872574268,0.11287061529,0.112868656414,0.11286669764,0.112864738968,0.112862780398,0.112860821931,0.112858863565,0.112856905301,0.11285494714,0.11285298908,0.112851031123,0.112849073267,0.112847115514,0.112845157862,0.112843200313,0.112841242865,0.112839285519,0.112837328276,0.112835371134,0.112833414094,0.112831457156,0.11282950032,0.112827543586,0.112825586954,0.112823630424,0.112821673996,0.112819717669,0.112817761445,0.112815805322,0.112813849301,0.112811893382,0.112809937565,0.11280798185,0.112806026236,0.112804070724,0.112802115314,0.112800160006,0.1127982048,0.112796249695,0.112794294692,0.112792339791,0.112790384992,0.112788430294,0.112786475698,0.112784521204,0.112782566812,0.112780612521,0.112778658332,0.112776704244,0.112774750259,0.112772796375,0.112770842592,0.112768888911,0.112766935332,0.112764981855,0.112763028479,0.112761075205,0.112759122032,0.112757168961,0.112755215991,0.112753263124,0.112751310357,0.112749357692,0.112747405129,0.112745452667,0.112743500307,0.112741548049,0.112739595892,0.112737643836,0.112735691882,0.112733740029,0.112731788278,0.112729836628,0.11272788508,0.112725933633,0.112723982288,0.112722031044,0.112720079902,0.112718128861,0.112716177921,0.112714227083,0.112712276346,0.11271032571,0.112708375176,0.112706424744,0.112704474412,0.112702524182,0.112700574053,0.112698624026,0.1126966741,0.112694724275,0.112692774552,0.11269082493,0.112688875409,0.112686925989,0.112684976671,0.112683027454,0.112681078338,0.112679129324,0.11267718041,0.112675231598,0.112673282887,0.112671334278,0.112669385769,0.112667437362,0.112665489056,0.112663540851,0.112661592747,0.112659644745,0.112657696843,0.112655749043,0.112653801344,0.112651853746,0.112649906249,0.112647958853,0.112646011558,0.112644064364,0.112642117272,0.11264017028,0.11263822339,0.1126362766,0.112634329912,0.112632383324,0.112630436838,0.112628490453,0.112626544168,0.112624597985,0.112622651903,0.112620705921,0.112618760041,0.112616814261,0.112614868583,0.112612923005,0.112610977529,0.112609032153,0.112607086878,0.112605141705,0.112603196632,0.11260125166,0.112599306788,0.112597362018,0.112595417349,0.11259347278,0.112591528312,0.112589583945,0.112587639679,0.112585695514,0.11258375145,0.112581807486,0.112579863623,0.112577919861,0.1125759762,0.112574032639,0.11257208918,0.112570145821,0.112568202563,0.112566259405,0.112564316348,0.112562373392,0.112560430537,0.112558487782,0.112556545128,0.112554602575,0.112552660122,0.11255071777,0.112548775519,0.112546833368,0.112544891318,0.112542949369,0.11254100752,0.112539065772,0.112537124125,0.112535182578,0.112533241131,0.112531299786,0.11252935854,0.112527417396,0.112525476352,0.112523535408,0.112521594565,0.112519653823,0.112517713181,0.112515772639,0.112513832198,0.112511891858,0.112509951618,0.112508011478,0.112506071439,0.112504131501,0.112502191663,0.112500251925,0.112498312288,0.112496372751,0.112494433314,0.112492493978,0.112490554743,0.112488615607,0.112486676573,0.112484737638,0.112482798804,0.11248086007,0.112478921437,0.112476982904,0.112475044471,0.112473106138,0.112471167906,0.112469229774,0.112467291743,0.112465353812,0.11246341598,0.11246147825,0.112459540619,0.112457603089,0.112455665659,0.112453728329,0.1124517911,0.11244985397,0.112447916941,0.112445980012,0.112444043184,0.112442106455,0.112440169827,0.112438233298,0.11243629687,0.112434360542,0.112432424315,0.112430488187,0.112428552159,0.112426616232,0.112424680405,0.112422744677,0.11242080905,0.112418873523,0.112416938096,0.112415002769,0.112413067542,0.112411132415,0.112409197389,0.112407262462,0.112405327635,0.112403392908,0.112401458281,0.112399523755,0.112397589328,0.112395655001,0.112393720774,0.112391786647,0.11238985262,0.112387918693,0.112385984866,0.112384051139,0.112382117511,0.112380183984,0.112378250557,0.112376317229,0.112374384001,0.112372450873,0.112370517845,0.112368584917,0.112366652089,0.11236471936,0.112362786732,0.112360854203,0.112358921774,0.112356989445,0.112355057215,0.112353125086,0.112351193056,0.112349261126,0.112347329295,0.112345397565,0.112343465934,0.112341534403,0.112339602971,0.11233767164,0.112335740408,0.112333809275,0.112331878243,0.11232994731,0.112328016476,0.112326085743,0.112324155109,0.112322224575,0.11232029414,0.112318363805,0.11231643357,0.112314503434,0.112312573398,0.112310643461,0.112308713624,0.112306783886,0.112304854249,0.11230292471,0.112300995271,0.112299065932,0.112297136693,0.112295207552,0.112293278512,0.112291349571,0.112289420729,0.112287491987,0.112285563344,0.112283634801,0.112281706357,0.112279778013,0.112277849768,0.112275921623,0.112273993577,0.11227206563,0.112270137783,0.112268210035,0.112266282387,0.112264354838,0.112262427389,0.112260500038,0.112258572788,0.112256645636,0.112254718584,0.112252791631,0.112250864778,0.112248938024,0.112247011369,0.112245084813,0.112243158357,0.112241232,0.112239305743,0.112237379584,0.112235453525,0.112233527565,0.112231601705,0.112229675943,0.112227750281,0.112225824718,0.112223899254,0.11222197389,0.112220048624,0.112218123458,0.112216198391,0.112214273423,0.112212348555,0.112210423785,0.112208499115,0.112206574543,0.112204650071,0.112202725698,0.112200801424,0.11219887725,0.112196953174,0.112195029197,0.11219310532,0.112191181541,0.112189257862,0.112187334281,0.1121854108,0.112183487418,0.112181564134,0.11217964095,0.112177717865,0.112175794878,0.112173871991,0.112171949203,0.112170026513,0.112168103923,0.112166181432,0.112164259039,0.112162336745,0.112160414551,0.112158492455,0.112156570458,0.11215464856,0.112152726761,0.112150805061,0.11214888346,0.112146961958,0.112145040554,0.112143119249,0.112141198044,0.112139276937,0.112137355928,0.112135435019,0.112133514209,0.112131593497,0.112129672884,0.11212775237,0.112125831954,0.112123911638,0.11212199142,0.112120071301,0.11211815128,0.112116231359,0.112114311536,0.112112391812,0.112110472186,0.112108552659,0.112106633231,0.112104713902,0.112102794671,0.112100875539,0.112098956506,0.112097037571,0.112095118735,0.112093199997,0.112091281358,0.112089362818,0.112087444376,0.112085526033,0.112083607789,0.112081689643,0.112079771596,0.112077853647,0.112075935797,0.112074018045,0.112072100392,0.112070182837,0.112068265381,0.112066348024,0.112064430765,0.112062513604,0.112060596542,0.112058679578,0.112056762713,0.112054845947,0.112052929278,0.112051012709,0.112049096237,0.112047179864,0.11204526359,0.112043347414,0.112041431336,0.112039515357,0.112037599476,0.112035683694,0.11203376801,0.112031852424,0.112029936936,0.112028021547,0.112026106257,0.112024191064,0.11202227597,0.112020360974,0.112018446077,0.112016531278,0.112014616577,0.112012701974,0.11201078747,0.112008873064,0.112006958756,0.112005044546,0.112003130435,0.112001216422,0.111999302507,0.11199738869,0.111995474972,0.111993561351,0.111991647829,0.111989734405,0.111987821079,0.111985907852,0.111983994722,0.111982081691,0.111980168758,0.111978255923,0.111976343186,0.111974430547,0.111972518006,0.111970605564,0.111968693219,0.111966780972,0.111964868824,0.111962956774,0.111961044821,0.111959132967,0.111957221211,0.111955309553,0.111953397992,0.11195148653,0.111949575166,0.1119476639,0.111945752732,0.111943841661,0.111941930689,0.111940019815,0.111938109038,0.11193619836,0.111934287779,0.111932377297,0.111930466912,0.111928556625,0.111926646437,0.111924736346,0.111922826353,0.111920916457,0.11191900666,0.11191709696,0.111915187359,0.111913277855,0.111911368449,0.111909459141,0.111907549931,0.111905640818,0.111903731803,0.111901822886,0.111899914067,0.111898005346,0.111896096722,0.111894188196,0.111892279768,0.111890371438,0.111888463205,0.11188655507,0.111884647033,0.111882739094,0.111880831252,0.111878923508,0.111877015862,0.111875108313,0.111873200862,0.111871293508,0.111869386253,0.111867479094,0.111865572034,0.111863665071,0.111861758206,0.111859851438,0.111857944768,0.111856038196,0.111854131721,0.111852225343,0.111850319064,0.111848412881,0.111846506797,0.11184460081,0.11184269492,0.111840789128,0.111838883434,0.111836977837,0.111835072337,0.111833166935,0.11183126163,0.111829356423,0.111827451314,0.111825546302,0.111823641387,0.11182173657,0.11181983185,0.111817927227,0.111816022702,0.111814118275,0.111812213945,0.111810309712,0.111808405576,0.111806501538,0.111804597598,0.111802693754,0.111800790008,0.11179888636,0.111796982808,0.111795079354,0.111793175998,0.111791272738,0.111789369576,0.111787466512,0.111785563544,0.111783660674,0.111781757901,0.111779855225,0.111777952647,0.111776050166,0.111774147782,0.111772245495,0.111770343306,0.111768441213,0.111766539218,0.11176463732,0.111762735519,0.111760833816};

      // Linear interpolation
      output(indGrid) = linInterp(x(indGrid), xGrid, yGrid, equalSpaces);

    } else if (nu == 1) {

      // Obtained from R
      arma::vec xGrid = arma::linspace<arma::vec>(0, 13, 30001);
      arma::vec yGrid = {0,0.000216572803201,0.000432957981121,0.000649155686146,0.000865166070536,0.00108098928643,0.00129662548584,0.00151207482067,0.00172733744267,0.0019424135035,0.00215730315468,0.00237200654761,0.00258652383358,0.00280085516373,0.00301500068911,0.00322896056062,0.00344273492906,0.0036563239451,0.00386972775927,0.00408294652201,0.00429598038363,0.00450882949429,0.00472149400406,0.00493397406289,0.00514626982059,0.00535838142685,0.00557030903126,0.00578205278326,0.00599361283219,0.00620498932726,0.00641618241758,0.0066271922521,0.00683801897968,0.00704866274905,0.00725912370883,0.0074694020075,0.00767949779343,0.00788941121488,0.00809914241998,0.00830869155673,0.00851805877304,0.00872724421666,0.00893624803526,0.00914507037637,0.00935371138739,0.00956217121564,0.00977045000827,0.00997854791236,0.0101864650748,0.0103942016425,0.0106017577621,0.0108091335802,0.0110163292432,0.0112233448975,0.0114301806893,0.0116368367648,0.0118433132699,0.0120496103505,0.0122557281523,0.012461666821,0.0126674265022,0.0128730073411,0.0130784094831,0.0132836330734,0.0134886782569,0.0136935451787,0.0138982339836,0.0141027448162,0.0143070778211,0.0145112331428,0.0147152109256,0.0149190113139,0.0151226344516,0.0153260804827,0.0155293495512,0.0157324418009,0.0159353573753,0.016138096418,0.0163406590725,0.0165430454819,0.0167452557896,0.0169472901385,0.0171491486716,0.0173508315318,0.0175523388618,0.0177536708042,0.0179548275014,0.0181558090959,0.0183566157299,0.0185572475456,0.018757704685,0.01895798729,0.0191580955023,0.0193580294638,0.019557789316,0.0197573752002,0.019956787258,0.0201560256304,0.0203550904587,0.0205539818837,0.0207527000465,0.0209512450878,0.0211496171481,0.0213478163682,0.0215458428884,0.021743696849,0.0219413783902,0.0221388876522,0.0223362247748,0.0225333898981,0.0227303831616,0.0229272047052,0.0231238546682,0.0233203331901,0.0235166404102,0.0237127764678,0.0239087415017,0.0241045356512,0.0243001590549,0.0244956118516,0.02469089418,0.0248860061785,0.0250809479857,0.0252757197396,0.0254703215787,0.0256647536408,0.025859016064,0.0260531089861,0.0262470325448,0.0264407868778,0.0266343721226,0.0268277884165,0.0270210358969,0.027214114701,0.0274070249658,0.0275997668282,0.0277923404251,0.0279847458933,0.0281769833693,0.0283690529897,0.0285609548909,0.0287526892092,0.0289442560807,0.0291356556416,0.0293268880277,0.029517953375,0.0297088518192,0.0298995834959,0.0300901485407,0.030280547089,0.030470779276,0.0306608452369,0.0308507451069,0.031040479021,0.0312300471139,0.0314194495205,0.0316086863754,0.0317977578132,0.0319866639682,0.0321754049748,0.0323639809673,0.0325523920797,0.032740638446,0.0329287202001,0.0331166374758,0.0333043904068,0.0334919791266,0.0336794037688,0.0338666644666,0.0340537613532,0.0342406945619,0.0344274642256,0.0346140704774,0.0348005134498,0.0349867932758,0.0351729100879,0.0353588640186,0.0355446552002,0.035730283765,0.0359157498453,0.036101053573,0.0362861950802,0.0364711744986,0.03665599196,0.0368406475962,0.0370251415384,0.0372094739184,0.0373936448672,0.0375776545162,0.0377615029964,0.0379451904389,0.0381287169745,0.038312082734,0.0384952878482,0.0386783324475,0.0388612166625,0.0390439406235,0.0392265044608,0.0394089083046,0.0395911522848,0.0397732365315,0.0399551611744,0.0401369263434,0.040318532168,0.0404999787777,0.0406812663021,0.0408623948704,0.0410433646117,0.0412241756553,0.0414048281302,0.0415853221651,0.041765657889,0.0419458354305,0.0421258549182,0.0423057164805,0.042485420246,0.0426649663427,0.042844354899,0.0430235860429,0.0432026599022,0.043381576605,0.043560336279,0.0437389390518,0.0439173850509,0.0440956744038,0.0442738072379,0.0444517836805,0.0446296038585,0.0448072678992,0.0449847759293,0.0451621280758,0.0453393244654,0.0455163652247,0.0456932504802,0.0458699803583,0.0460465549855,0.0462229744878,0.0463992389914,0.0465753486224,0.0467513035065,0.0469271037698,0.0471027495378,0.0472782409361,0.0474535780903,0.0476287611258,0.0478037901679,0.0479786653417,0.0481533867725,0.0483279545851,0.0485023689045,0.0486766298554,0.0488507375627,0.0490246921508,0.0491984937443,0.0493721424676,0.0495456384449,0.0497189818005,0.0498921726584,0.0500652111426,0.0502380973772,0.0504108314857,0.050583413592,0.0507558438195,0.0509281222919,0.0511002491325,0.0512722244646,0.0514440484114,0.051615721096,0.0517872426414,0.0519586131705,0.052129832806,0.0523009016708,0.0524718198873,0.0526425875781,0.0528132048656,0.052983671872,0.0531539887197,0.0533241555306,0.0534941724268,0.0536640395303,0.0538337569627,0.0540033248459,0.0541727433014,0.0543420124508,0.0545111324155,0.0546801033167,0.0548489252757,0.0550175984137,0.0551861228516,0.0553544987104,0.0555227261109,0.0556908051738,0.0558587360198,0.0560265187695,0.0561941535431,0.0563616404612,0.0565289796439,0.0566961712113,0.0568632152836,0.0570301119807,0.0571968614224,0.0573634637285,0.0575299190187,0.0576962274125,0.0578623890294,0.0580284039888,0.0581942724099,0.058359994412,0.058525570114,0.0586909996351,0.058856283094,0.0590214206097,0.0591864123007,0.0593512582857,0.0595159586832,0.0596805136117,0.0598449231893,0.0600091875344,0.0601733067651,0.0603372809993,0.0605011103552,0.0606647949504,0.0608283349027,0.0609917303298,0.0611549813492,0.0613180880784,0.0614810506347,0.0616438691355,0.0618065436979,0.0619690744389,0.0621314614756,0.0622937049249,0.0624558049035,0.0626177615282,0.0627795749155,0.062941245182,0.0631027724441,0.0632641568181,0.0634253984203,0.0635864973668,0.0637474537736,0.0639082677567,0.0640689394319,0.064229468915,0.0643898563218,0.0645501017676,0.0647102053681,0.0648701672387,0.0650299874946,0.065189666251,0.065349203623,0.0655085997257,0.065667854674,0.0658269685826,0.0659859415665,0.0661447737401,0.0663034652181,0.0664620161149,0.0666204265449,0.0667786966224,0.0669368264615,0.0670948161763,0.0672526658809,0.0674103756892,0.0675679457148,0.0677253760717,0.0678826668734,0.0680398182335,0.0681968302653,0.0683537030824,0.0685104367978,0.0686670315249,0.0688234873766,0.0689798044661,0.0691359829061,0.0692920228095,0.069447924289,0.0696036874572,0.0697593124267,0.06991479931,0.0700701482193,0.0702253592669,0.070380432565,0.0705353682257,0.070690166361,0.0708448270828,0.0709993505029,0.071153736733,0.0713079858847,0.0714620980697,0.0716160733992,0.0717699119848,0.0719236139376,0.0720771793689,0.0722306083897,0.072383901111,0.0725370576438,0.0726900780989,0.0728429625869,0.0729957112187,0.0731483241046,0.0733008013552,0.0734531430808,0.0736053493917,0.0737574203982,0.0739093562103,0.0740611569381,0.0742128226914,0.0743643535802,0.0745157497142,0.0746670112029,0.0748181381561,0.0749691306833,0.0751199888937,0.0752707128967,0.0754213028015,0.0755717587174,0.0757220807532,0.075872269018,0.0760223236207,0.07617224467,0.0763220322746,0.0764716865431,0.0766212075841,0.0767705955059,0.076919850417,0.0770689724256,0.0772179616398,0.0773668181677,0.0775155421173,0.0776641335966,0.0778125927133,0.0779609195752,0.0781091142899,0.078257176965,0.078405107708,0.0785529066262,0.0787005738269,0.0788481094174,0.0789955135048,0.0791427861961,0.0792899275982,0.0794369378181,0.0795838169625,0.0797305651381,0.0798771824515,0.0800236690093,0.0801700249178,0.0803162502835,0.0804623452125,0.0806083098111,0.0807541441854,0.0808998484413,0.0810454226848,0.0811908670218,0.0813361815579,0.0814813663988,0.0816264216501,0.0817713474174,0.0819161438059,0.0820608109211,0.0822053488682,0.0823497577523,0.0824940376785,0.0826381887518,0.0827822110771,0.0829261047591,0.0830698699027,0.0832135066125,0.083357014993,0.0835003951487,0.0836436471841,0.0837867712033,0.0839297673106,0.0840726356103,0.0842153762063,0.0843579892025,0.084500474703,0.0846428328114,0.0847850636316,0.0849271672671,0.0850691438215,0.0852109933983,0.0853527161009,0.0854943120325,0.0856357812963,0.0857771239956,0.0859183402334,0.0860594301125,0.086200393736,0.0863412312067,0.0864819426271,0.0866225281001,0.086762987728,0.0869033216135,0.0870435298589,0.0871836125664,0.0873235698384,0.0874634017769,0.0876031084841,0.0877426900618,0.0878821466121,0.0880214782366,0.0881606850371,0.0882997671154,0.0884387245729,0.088577557511,0.0887162660314,0.0888548502351,0.0889933102236,0.0891316460978,0.089269857959,0.0894079459081,0.089545910046,0.0896837504736,0.0898214672915,0.0899590606005,0.0900965305011,0.0902338770939,0.0903711004792,0.0905082007575,0.0906451780289,0.0907820323937,0.0909187639519,0.0910553728037,0.0911918590488,0.0913282227872,0.0914644641186,0.0916005831427,0.0917365799593,0.0918724546677,0.0920082073674,0.0921438381579,0.0922793471384,0.0924147344081,0.0925500000661,0.0926851442116,0.0928201669435,0.0929550683607,0.0930898485619,0.093224507646,0.0933590457116,0.0934934628572,0.0936277591814,0.0937619347826,0.0938959897591,0.0940299242091,0.0941637382309,0.0942974319226,0.0944310053821,0.0945644587074,0.0946977919964,0.0948310053468,0.0949640988564,0.0950970726228,0.0952299267435,0.0953626613159,0.0954952764376,0.0956277722058,0.0957601487177,0.0958924060704,0.0960245443611,0.0961565636868,0.0962884641443,0.0964202458306,0.0965519088423,0.0966834532761,0.0968148792287,0.0969461867966,0.0970773760762,0.0972084471639,0.0973394001561,0.0974702351488,0.0976009522383,0.0977315515206,0.0978620330917,0.0979923970476,0.0981226434839,0.0982527724965,0.0983827841811,0.0985126786333,0.0986424559485,0.0987721162223,0.0989016595499,0.0990310860267,0.0991603957479,0.0992895888086,0.0994186653039,0.0995476253287,0.099676468978,0.0998051963466,0.0999338075293,0.100062302621,0.100190681715,0.100318944908,0.100447092293,0.100575123964,0.100703040017,0.100830840544,0.100958525641,0.101086095401,0.101213549919,0.101340889288,0.101468113602,0.101595222955,0.10172221744,0.101849097152,0.101975862183,0.102102512628,0.102229048579,0.10235547013,0.102481777374,0.102607970405,0.102734049314,0.102860014196,0.102985865144,0.103111602249,0.103237225606,0.103362735306,0.103488131442,0.103613414107,0.103738583394,0.103863639394,0.1039885822,0.104113411905,0.1042381286,0.104362732378,0.10448722333,0.104611601549,0.104735867126,0.104860020154,0.104984060724,0.105107988928,0.105231804857,0.105355508603,0.105479100257,0.105602579911,0.105725947655,0.105849203582,0.105972347783,0.106095380347,0.106218301367,0.106341110934,0.106463809138,0.106586396069,0.10670887182,0.106831236479,0.106953490139,0.107075632889,0.107197664819,0.107319586021,0.107441396585,0.1075630966,0.107684686156,0.107806165345,0.107927534255,0.108048792977,0.108169941601,0.108290980215,0.108411908911,0.108532727778,0.108653436904,0.10877403638,0.108894526295,0.109014906739,0.1091351778,0.109255339568,0.109375392132,0.10949533558,0.109615170003,0.109734895489,0.109854512126,0.109974020003,0.11009341921,0.110212709834,0.110331891965,0.11045096569,0.110569931098,0.110688788278,0.110807537317,0.110926178305,0.111044711328,0.111163136475,0.111281453834,0.111399663494,0.11151776554,0.111635760063,0.111753647148,0.111871426884,0.111989099359,0.112106664659,0.112224122873,0.112341474087,0.112458718389,0.112575855866,0.112692886605,0.112809810693,0.112926628217,0.113043339264,0.113159943921,0.113276442275,0.113392834413,0.11350912042,0.113625300383,0.11374137439,0.113857342527,0.113973204879,0.114088961533,0.114204612575,0.114320158092,0.11443559817,0.114550932894,0.114666162351,0.114781286625,0.114896305805,0.115011219973,0.115126029218,0.115240733623,0.115355333276,0.11546982826,0.115584218661,0.115698504566,0.115812686058,0.115926763224,0.116040736148,0.116154604916,0.116268369611,0.11638203032,0.116495587127,0.116609040117,0.116722389374,0.116835634983,0.11694877703,0.117061815597,0.11717475077,0.117287582632,0.11740031127,0.117512936765,0.117625459203,0.117737878668,0.117850195244,0.117962409014,0.118074520063,0.118186528474,0.118298434331,0.118410237718,0.118521938719,0.118633537416,0.118745033894,0.118856428235,0.118967720524,0.119078910843,0.119189999276,0.119300985905,0.119411870815,0.119522654087,0.119633335805,0.119743916051,0.119854394909,0.119964772462,0.120075048791,0.120185223979,0.12029529811,0.120405271265,0.120515143527,0.120624914978,0.120734585701,0.120844155777,0.120953625289,0.121062994319,0.121172262949,0.121281431261,0.121390499336,0.121499467257,0.121608335105,0.121717102962,0.12182577091,0.12193433903,0.122042807403,0.122151176112,0.122259445236,0.122367614859,0.12247568506,0.122583655921,0.122691527523,0.122799299947,0.122906973274,0.123014547585,0.123122022961,0.123229399483,0.12333667723,0.123443856285,0.123550936727,0.123657918636,0.123764802094,0.123871587181,0.123978273977,0.124084862562,0.124191353016,0.12429774542,0.124404039853,0.124510236396,0.124616335128,0.12472233613,0.12482823948,0.12493404526,0.125039753548,0.125145364424,0.125250877968,0.125356294258,0.125461613375,0.125566835398,0.125671960406,0.125776988478,0.125881919694,0.125986754132,0.126091491872,0.126196132992,0.126300677571,0.126405125689,0.126509477424,0.126613732854,0.126717892059,0.126821955116,0.126925922105,0.127029793104,0.12713356819,0.127237247444,0.127340830942,0.127444318763,0.127547710985,0.127651007686,0.127754208945,0.127857314838,0.127960325445,0.128063240842,0.128166061108,0.128268786321,0.128371416557,0.128473951894,0.128576392411,0.128678738184,0.12878098929,0.128883145808,0.128985207814,0.129087175385,0.1291890486,0.129290827533,0.129392512264,0.129494102868,0.129595599423,0.129697002005,0.129798310691,0.129899525559,0.130000646683,0.130101674142,0.130202608011,0.130303448367,0.130404195286,0.130504848845,0.13060540912,0.130705876187,0.130806250122,0.130906531002,0.131006718902,0.131106813898,0.131206816067,0.131306725484,0.131406542224,0.131506266364,0.131605897979,0.131705437145,0.131804883937,0.131904238431,0.132003500703,0.132102670826,0.132201748878,0.132300734932,0.132399629065,0.132498431351,0.132597141865,0.132695760683,0.132794287879,0.132892723527,0.132991067704,0.133089320484,0.13318748194,0.133285552149,0.133383531184,0.13348141912,0.133579216032,0.133676921993,0.133774537078,0.133872061362,0.133969494919,0.134066837822,0.134164090146,0.134261251964,0.134358323352,0.134455304382,0.134552195128,0.134648995665,0.134745706066,0.134842326405,0.134938856755,0.13503529719,0.135131647783,0.135227908607,0.135324079737,0.135420161245,0.135516153205,0.13561205569,0.135707868773,0.135803592527,0.135899227024,0.135994772339,0.136090228544,0.136185595711,0.136280873914,0.136376063225,0.136471163717,0.136566175463,0.136661098534,0.136755933004,0.136850678945,0.13694533643,0.13703990553,0.137134386317,0.137228778865,0.137323083245,0.137417299529,0.13751142779,0.137605468098,0.137699420527,0.137793285148,0.137887062033,0.137980751253,0.13807435288,0.138167866986,0.138261293642,0.138354632921,0.138447884892,0.138541049628,0.1386341272,0.138727117679,0.138820021137,0.138912837644,0.139005567272,0.139098210092,0.139190766174,0.13928323559,0.139375618411,0.139467914706,0.139560124548,0.139652248007,0.139744285152,0.139836236056,0.139928100789,0.14001987942,0.14011157202,0.140203178661,0.140294699411,0.140386134342,0.140477483522,0.140568747024,0.140659924916,0.140751017269,0.140842024152,0.140932945636,0.14102378179,0.141114532684,0.141205198388,0.141295778972,0.141386274504,0.141476685056,0.141567010695,0.141657251492,0.141747407517,0.141837478837,0.141927465524,0.142017367645,0.14210718527,0.142196918468,0.142286567309,0.142376131861,0.142465612193,0.142555008374,0.142644320473,0.142733548559,0.1428226927,0.142911752966,0.143000729424,0.143089622143,0.143178431192,0.143267156639,0.143355798553,0.143444357002,0.143532832054,0.143621223777,0.143709532241,0.143797757512,0.143885899658,0.143973958749,0.144061934851,0.144149828033,0.144237638363,0.144325365908,0.144413010736,0.144500572914,0.144588052511,0.144675449594,0.144762764231,0.144849996488,0.144937146434,0.145024214135,0.14511119966,0.145198103074,0.145284924446,0.145371663843,0.145458321331,0.145544896978,0.14563139085,0.145717803015,0.14580413354,0.14589038249,0.145976549934,0.146062635937,0.146148640567,0.14623456389,0.146320405972,0.14640616688,0.146491846681,0.14657744544,0.146662963224,0.1467484001,0.146833756134,0.146919031391,0.147004225939,0.147089339842,0.147174373167,0.14725932598,0.147344198347,0.147428990334,0.147513702007,0.14759833343,0.147682884671,0.147767355794,0.147851746865,0.14793605795,0.148020289114,0.148104440423,0.148188511941,0.148272503735,0.14835641587,0.14844024841,0.148524001421,0.148607674969,0.148691269117,0.148774783931,0.148858219477,0.148941575819,0.149024853021,0.149108051149,0.149191170267,0.149274210441,0.149357171734,0.149440054212,0.149522857938,0.149605582977,0.149688229395,0.149770797254,0.14985328662,0.149935697557,0.150018030128,0.150100284399,0.150182460433,0.150264558294,0.150346578046,0.150428519754,0.150510383481,0.150592169292,0.150673877249,0.150755507416,0.150837059858,0.150918534638,0.15099993182,0.151081251467,0.151162493642,0.15124365841,0.151324745833,0.151405755975,0.151486688899,0.151567544669,0.151648323347,0.151729024997,0.151809649682,0.151890197465,0.151970668409,0.152051062576,0.152131380031,0.152211620835,0.152291785051,0.152371872742,0.152451883972,0.152531818801,0.152611677294,0.152691459512,0.152771165518,0.152850795375,0.152930349145,0.15300982689,0.153089228672,0.153168554554,0.153247804598,0.153326978867,0.153406077421,0.153485100323,0.153564047636,0.153642919421,0.15372171574,0.153800436654,0.153879082226,0.153957652518,0.154036147591,0.154114567506,0.154192912325,0.154271182111,0.154349376923,0.154427496825,0.154505541876,0.154583512139,0.154661407675,0.154739228545,0.15481697481,0.154894646531,0.15497224377,0.155049766587,0.155127215043,0.1552045892,0.155281889119,0.155359114859,0.155436266482,0.155513344049,0.155590347621,0.155667277257,0.15574413302,0.155820914968,0.155897623163,0.155974257665,0.156050818535,0.156127305833,0.156203719619,0.156280059953,0.156356326896,0.156432520508,0.156508640849,0.15658468798,0.156660661959,0.156736562847,0.156812390704,0.15688814559,0.156963827565,0.157039436688,0.15711497302,0.157190436619,0.157265827546,0.157341145859,0.15741639162,0.157491564886,0.157566665719,0.157641694176,0.157716650317,0.157791534202,0.15786634589,0.157941085439,0.15801575291,0.158090348361,0.158164871852,0.158239323441,0.158313703187,0.15838801115,0.158462247388,0.158536411959,0.158610504924,0.158684526339,0.158758476265,0.15883235476,0.158906161882,0.15897989769,0.159053562242,0.159127155597,0.159200677813,0.159274128949,0.159347509062,0.159420818212,0.159494056456,0.159567223853,0.15964032046,0.159713346336,0.159786301538,0.159859186125,0.159932000155,0.160004743685,0.160077416773,0.160150019478,0.160222551856,0.160295013966,0.160367405865,0.160439727611,0.160511979261,0.160584160873,0.160656272505,0.160728314213,0.160800286055,0.160872188088,0.16094402037,0.161015782958,0.16108747591,0.161159099281,0.16123065313,0.161302137513,0.161373552487,0.16144489811,0.161516174438,0.161587381529,0.161658519438,0.161729588222,0.16180058794,0.161871518646,0.161942380398,0.162013173253,0.162083897266,0.162154552494,0.162225138995,0.162295656824,0.162366106037,0.162436486691,0.162506798842,0.162577042547,0.162647217862,0.162717324842,0.162787363543,0.162857334023,0.162927236337,0.16299707054,0.163066836689,0.16313653484,0.163206165048,0.163275727369,0.163345221859,0.163414648574,0.163484007568,0.163553298899,0.163622522621,0.16369167879,0.163760767461,0.16382978869,0.163898742532,0.163967629043,0.164036448277,0.16410520029,0.164173885138,0.164242502874,0.164311053555,0.164379537236,0.164447953971,0.164516303815,0.164584586824,0.164652803052,0.164720952554,0.164789035385,0.1648570516,0.164925001253,0.164992884399,0.165060701093,0.165128451388,0.165196135341,0.165263753005,0.165331304434,0.165398789683,0.165466208807,0.16553356186,0.165600848895,0.165668069968,0.165735225132,0.165802314441,0.165869337951,0.165936295713,0.166003187784,0.166070014216,0.166136775064,0.166203470381,0.166270100221,0.166336664638,0.166403163686,0.166469597419,0.166535965889,0.166602269152,0.166668507259,0.166734680266,0.166800788225,0.166866831189,0.166932809213,0.166998722349,0.167064570651,0.167130354172,0.167196072966,0.167261727085,0.167327316583,0.167392841512,0.167458301926,0.167523697878,0.167589029421,0.167654296608,0.167719499491,0.167784638123,0.167849712558,0.167914722848,0.167979669045,0.168044551203,0.168109369374,0.16817412361,0.168238813965,0.16830344049,0.168368003238,0.168432502262,0.168496937614,0.168561309346,0.16862561751,0.16868986216,0.168754043346,0.168818161121,0.168882215538,0.168946206648,0.169010134503,0.169073999156,0.169137800658,0.169201539061,0.169265214418,0.169328826779,0.169392376197,0.169455862723,0.16951928641,0.169582647309,0.169645945471,0.169709180948,0.169772353792,0.169835464053,0.169898511785,0.169961497037,0.170024419861,0.17008728031,0.170150078433,0.170212814282,0.170275487909,0.170338099364,0.170400648699,0.170463135965,0.170525561212,0.170587924493,0.170650225857,0.170712465356,0.17077464304,0.170836758961,0.170898813169,0.170960805715,0.17102273665,0.171084606023,0.171146413887,0.171208160292,0.171269845287,0.171331468925,0.171393031254,0.171454532326,0.171515972191,0.171577350899,0.1716386685,0.171699925046,0.171761120585,0.171822255169,0.171883328847,0.171944341669,0.172005293687,0.172066184948,0.172127015505,0.172187785405,0.1722484947,0.17230914344,0.172369731673,0.172430259451,0.172490726822,0.172551133836,0.172611480544,0.172671766994,0.172731993236,0.17279215932,0.172852265295,0.172912311211,0.172972297116,0.173032223062,0.173092089096,0.173151895269,0.173211641628,0.173271328225,0.173330955107,0.173390522324,0.173450029926,0.17350947796,0.173568866477,0.173628195524,0.173687465152,0.173746675409,0.173805826344,0.173864918005,0.173923950442,0.173982923703,0.174041837837,0.174100692893,0.174159488919,0.174218225964,0.174276904076,0.174335523304,0.174394083697,0.174452585302,0.174511028169,0.174569412345,0.174627737879,0.174686004819,0.174744213214,0.174802363111,0.174860454559,0.174918487606,0.1749764623,0.17503437869,0.175092236822,0.175150036745,0.175207778507,0.175265462156,0.17532308774,0.175380655306,0.175438164903,0.175495616578,0.175553010378,0.175610346352,0.175667624547,0.17572484501,0.17578200779,0.175839112934,0.175896160488,0.175953150501,0.17601008302,0.176066958093,0.176123775766,0.176180536087,0.176237239103,0.176293884862,0.17635047341,0.176407004795,0.176463479064,0.176519896263,0.17657625644,0.176632559642,0.176688805916,0.176744995309,0.176801127867,0.176857203637,0.176913222666,0.176969185002,0.17702509069,0.177080939777,0.17713673231,0.177192468336,0.1772481479,0.17730377105,0.177359337833,0.177414848293,0.177470302479,0.177525700436,0.177581042211,0.177636327849,0.177691557398,0.177746730904,0.177801848411,0.177856909968,0.17791191562,0.177966865412,0.178021759391,0.178076597604,0.178131380095,0.178186106912,0.178240778099,0.178295393703,0.178349953769,0.178404458343,0.178458907472,0.1785133012,0.178567639573,0.178621922638,0.178676150439,0.178730323022,0.178784440433,0.178838502717,0.178892509919,0.178946462086,0.179000359262,0.179054201493,0.179107988824,0.1791617213,0.179215398967,0.179269021869,0.179322590052,0.179376103561,0.179429562442,0.179482966738,0.179536316496,0.17958961176,0.179642852575,0.179696038985,0.179749171037,0.179802248774,0.179855272242,0.179908241485,0.179961156548,0.180014017476,0.180066824313,0.180119577104,0.180172275893,0.180224920725,0.180277511645,0.180330048697,0.180382531925,0.180434961374,0.180487337089,0.180539659113,0.18059192749,0.180644142266,0.180696303484,0.180748411189,0.180800465424,0.180852466234,0.180904413663,0.180956307754,0.181008148553,0.181059936102,0.181111670447,0.18116335163,0.181214979695,0.181266554687,0.181318076649,0.181369545625,0.181420961659,0.181472324794,0.181523635074,0.181574892543,0.181626097244,0.18167724922,0.181728348516,0.181779395175,0.18183038924,0.181881330755,0.181932219762,0.181983056306,0.18203384043,0.182084572176,0.182135251589,0.182185878711,0.182236453585,0.182286976256,0.182337446765,0.182387865155,0.182438231471,0.182488545754,0.182538808048,0.182589018396,0.18263917684,0.182689283424,0.18273933819,0.182789341181,0.18283929244,0.18288919201,0.182939039932,0.182988836251,0.183038581008,0.183088274246,0.183137916007,0.183187506335,0.183237045271,0.183286532858,0.183335969139,0.183385354155,0.183434687949,0.183483970564,0.183533202042,0.183582382424,0.183631511754,0.183680590073,0.183729617423,0.183778593847,0.183827519386,0.183876394083,0.18392521798,0.183973991118,0.18402271354,0.184071385287,0.184120006402,0.184168576925,0.1842170969,0.184265566367,0.184313985369,0.184362353946,0.184410672142,0.184458939997,0.184507157553,0.184555324851,0.184603441934,0.184651508842,0.184699525617,0.1847474923,0.184795408934,0.184843275558,0.184891092215,0.184938858946,0.184986575793,0.185034242795,0.185081859995,0.185129427433,0.185176945152,0.185224413191,0.185271831592,0.185319200397,0.185366519645,0.185413789378,0.185461009637,0.185508180463,0.185555301896,0.185602373978,0.18564939675,0.185696370251,0.185743294523,0.185790169606,0.185836995542,0.18588377237,0.185930500132,0.185977178868,0.186023808618,0.186070389423,0.186116921324,0.186163404361,0.186209838573,0.186256224003,0.186302560689,0.186348848673,0.186395087994,0.186441278694,0.186487420811,0.186533514386,0.18657955946,0.186625556073,0.186671504264,0.186717404073,0.186763255541,0.186809058708,0.186854813613,0.186900520297,0.186946178799,0.18699178916,0.187037351418,0.187082865615,0.187128331789,0.18717374998,0.187219120228,0.187264442573,0.187309717055,0.187354943712,0.187400122585,0.187445253712,0.187490337135,0.187535372891,0.187580361021,0.187625301563,0.187670194558,0.187715040045,0.187759838062,0.18780458865,0.187849291847,0.187893947692,0.187938556226,0.187983117486,0.188027631513,0.188072098344,0.18811651802,0.18816089058,0.188205216061,0.188249494504,0.188293725947,0.188337910429,0.188382047989,0.188426138666,0.188470182499,0.188514179526,0.188558129786,0.188602033318,0.188645890161,0.188689700352,0.188733463932,0.188777180938,0.188820851409,0.188864475383,0.1889080529,0.188951583997,0.188995068713,0.189038507086,0.189081899155,0.189125244958,0.189168544534,0.18921179792,0.189255005155,0.189298166277,0.189341281324,0.189384350335,0.189427373347,0.189470350399,0.189513281529,0.189556166775,0.189599006174,0.189641799765,0.189684547586,0.189727249675,0.189769906069,0.189812516806,0.189855081925,0.189897601462,0.189940075456,0.189982503945,0.190024886966,0.190067224557,0.190109516755,0.190151763599,0.190193965125,0.190236121371,0.190278232375,0.190320298174,0.190362318806,0.190404294309,0.190446224718,0.190488110073,0.190529950409,0.190571745766,0.190613496179,0.190655201686,0.190696862324,0.190738478131,0.190780049144,0.190821575399,0.190863056934,0.190904493786,0.190945885992,0.190987233589,0.191028536614,0.191069795103,0.191111009095,0.191152178625,0.191193303731,0.191234384449,0.191275420817,0.19131641287,0.191357360646,0.191398264182,0.191439123513,0.191479938678,0.191520709711,0.191561436651,0.191602119533,0.191642758395,0.191683353271,0.1917239042,0.191764411218,0.19180487436,0.191845293664,0.191885669165,0.1919260009,0.191966288906,0.192006533218,0.192046733873,0.192086890907,0.192127004356,0.192167074256,0.192207100644,0.192247083556,0.192287023027,0.192326919093,0.192366771792,0.192406581158,0.192446347228,0.192486070037,0.192525749622,0.192565386018,0.192604979261,0.192644529387,0.192684036432,0.192723500431,0.19276292142,0.192802299436,0.192841634512,0.192880926686,0.192920175993,0.192959382468,0.192998546147,0.193037667066,0.193076745259,0.193115780763,0.193154773612,0.193193723843,0.19323263149,0.193271496589,0.193310319175,0.193349099284,0.193387836951,0.193426532211,0.193465185099,0.19350379565,0.1935423639,0.193580889884,0.193619373636,0.193657815193,0.193696214588,0.193734571857,0.193772887035,0.193811160157,0.193849391257,0.193887580371,0.193925727534,0.19396383278,0.194001896144,0.194039917661,0.194077897365,0.194115835292,0.194153731477,0.194191585953,0.194229398755,0.194267169918,0.194304899477,0.194342587467,0.194380233921,0.194417838874,0.194455402361,0.194492924416,0.194530405073,0.194567844368,0.194605242334,0.194642599005,0.194679914417,0.194717188602,0.194754421597,0.194791613434,0.194828764147,0.194865873772,0.194902942343,0.194939969892,0.194976956455,0.195013902066,0.195050806758,0.195087670566,0.195124493523,0.195161275663,0.195198017022,0.195234717631,0.195271377526,0.195307996739,0.195344575306,0.195381113259,0.195417610632,0.19545406746,0.195490483775,0.195526859612,0.195563195004,0.195599489985,0.195635744588,0.195671958847,0.195708132796,0.195744266467,0.195780359895,0.195816413113,0.195852426155,0.195888399053,0.195924331842,0.195960224554,0.195996077223,0.196031889882,0.196067662564,0.196103395304,0.196139088133,0.196174741085,0.196210354194,0.196245927492,0.196281461013,0.196316954789,0.196352408854,0.19638782324,0.196423197982,0.196458533111,0.196493828661,0.196529084664,0.196564301154,0.196599478163,0.196634615725,0.196669713871,0.196704772635,0.19673979205,0.196774772149,0.196809712963,0.196844614526,0.196879476871,0.196914300029,0.196949084035,0.196983828919,0.197018534716,0.197053201457,0.197087829174,0.197122417901,0.19715696767,0.197191478513,0.197225950463,0.197260383551,0.197294777811,0.197329133275,0.197363449975,0.197397727942,0.197431967211,0.197466167812,0.197500329778,0.197534453141,0.197568537933,0.197602584187,0.197636591934,0.197670561206,0.197704492036,0.197738384456,0.197772238497,0.197806054192,0.197839831571,0.197873570669,0.197907271515,0.197940934143,0.197974558583,0.198008144869,0.19804169303,0.1980752031,0.19810867511,0.198142109092,0.198175505077,0.198208863097,0.198242183184,0.198275465369,0.198308709683,0.198341916159,0.198375084828,0.198408215722,0.198441308871,0.198474364308,0.198507382063,0.198540362169,0.198573304656,0.198606209556,0.1986390769,0.19867190672,0.198704699046,0.198737453911,0.198770171344,0.198802851378,0.198835494044,0.198868099373,0.198900667395,0.198933198143,0.198965691646,0.198998147937,0.199030567046,0.199062949003,0.199095293841,0.19912760159,0.199159872281,0.199192105945,0.199224302613,0.199256462315,0.199288585083,0.199320670946,0.199352719937,0.199384732086,0.199416707423,0.199448645979,0.199480547785,0.199512412871,0.199544241269,0.199576033008,0.199607788119,0.199639506634,0.199671188582,0.199702833993,0.199734442899,0.19976601533,0.199797551316,0.199829050887,0.199860514075,0.199891940909,0.199923331419,0.199954685637,0.199986003592,0.200017285314,0.200048530835,0.200079740183,0.200110913389,0.200142050484,0.200173151497,0.200204216459,0.200235245399,0.200266238348,0.200297195336,0.200328116392,0.200359001548,0.200389850832,0.200420664275,0.200451441906,0.200482183756,0.200512889854,0.20054356023,0.200574194915,0.200604793937,0.200635357327,0.200665885114,0.200696377328,0.200726833999,0.200757255157,0.20078764083,0.200817991049,0.200848305844,0.200878585243,0.200908829277,0.200939037975,0.200969211367,0.200999349481,0.201029452348,0.201059519997,0.201089552458,0.201119549759,0.20114951193,0.201179439,0.201209330999,0.201239187957,0.201269009901,0.201298796863,0.20132854887,0.201358265952,0.201387948138,0.201417595457,0.201447207939,0.201476785613,0.201506328507,0.201535836651,0.201565310074,0.201594748805,0.201624152872,0.201653522305,0.201682857133,0.201712157384,0.201741423088,0.201770654274,0.201799850969,0.201829013204,0.201858141007,0.201887234406,0.201916293431,0.201945318109,0.201974308471,0.202003264544,0.202032186358,0.20206107394,0.20208992732,0.202118746526,0.202147531587,0.202176282531,0.202204999386,0.202233682182,0.202262330947,0.202290945709,0.202319526496,0.202348073338,0.202376586262,0.202405065297,0.202433510471,0.202461921812,0.202490299349,0.202518643111,0.202546953124,0.202575229418,0.202603472021,0.202631680961,0.202659856266,0.202687997964,0.202716106084,0.202744180653,0.202772221699,0.202800229251,0.202828203337,0.202856143984,0.202884051221,0.202911925076,0.202939765576,0.202967572749,0.202995346624,0.203023087228,0.203050794589,0.203078468735,0.203106109693,0.203133717492,0.203161292159,0.203188833722,0.203216342208,0.203243817646,0.203271260063,0.203298669487,0.203326045945,0.203353389464,0.203380700073,0.203407977799,0.20343522267,0.203462434713,0.203489613955,0.203516760424,0.203543874148,0.203570955154,0.203598003469,0.20362501912,0.203652002136,0.203678952543,0.203705870368,0.20373275564,0.203759608385,0.20378642863,0.203813216403,0.203839971731,0.203866694641,0.203893385161,0.203920043316,0.203946669136,0.203973262645,0.203999823873,0.204026352845,0.204052849589,0.204079314132,0.204105746501,0.204132146722,0.204158514823,0.20418485083,0.204211154771,0.204237426673,0.204263666561,0.204289874464,0.204316050408,0.204342194419,0.204368306524,0.204394386751,0.204420435126,0.204446451675,0.204472436426,0.204498389404,0.204524310637,0.204550200152,0.204576057974,0.204601884131,0.204627678648,0.204653441553,0.204679172872,0.204704872631,0.204730540858,0.204756177577,0.204781782817,0.204807356603,0.204832898961,0.204858409918,0.204883889501,0.204909337735,0.204934754648,0.204960140264,0.204985494611,0.205010817715,0.205036109601,0.205061370297,0.205086599828,0.20511179822,0.2051369655,0.205162101694,0.205187206827,0.205212280926,0.205237324017,0.205262336126,0.205287317279,0.205312267502,0.20533718682,0.205362075261,0.205386932849,0.20541175961,0.205436555571,0.205461320758,0.205486055196,0.20551075891,0.205535431928,0.205560074274,0.205584685974,0.205609267055,0.205633817541,0.205658337458,0.205682826833,0.205707285691,0.205731714057,0.205756111957,0.205780479416,0.205804816461,0.205829123117,0.205853399408,0.205877645362,0.205901861003,0.205926046356,0.205950201448,0.205974326303,0.205998420947,0.206022485405,0.206046519703,0.206070523867,0.20609449792,0.20611844189,0.2061423558,0.206166239677,0.206190093545,0.206213917429,0.206237711356,0.206261475349,0.206285209435,0.206308913638,0.206332587984,0.206356232497,0.206379847203,0.206403432127,0.206426987293,0.206450512728,0.206474008454,0.206497474499,0.206520910886,0.206544317641,0.206567694789,0.206591042354,0.206614360361,0.206637648836,0.206660907802,0.206684137286,0.206707337311,0.206730507902,0.206753649085,0.206776760883,0.206799843322,0.206822896427,0.206845920221,0.20686891473,0.206891879979,0.206914815991,0.206937722792,0.206960600406,0.206983448857,0.207006268171,0.207029058371,0.207051819483,0.20707455153,0.207097254537,0.207119928529,0.20714257353,0.207165189564,0.207187776656,0.207210334831,0.207232864111,0.207255364523,0.207277836089,0.207300278835,0.207322692785,0.207345077962,0.207367434392,0.207389762098,0.207412061104,0.207434331435,0.207456573115,0.207478786168,0.207500970617,0.207523126488,0.207545253804,0.20756735259,0.207589422869,0.207611464664,0.207633478002,0.207655462904,0.207677419396,0.207699347501,0.207721247242,0.207743118645,0.207764961733,0.207786776529,0.207808563058,0.207830321343,0.207852051408,0.207873753277,0.207895426974,0.207917072522,0.207938689945,0.207960279267,0.207981840512,0.208003373702,0.208024878863,0.208046356017,0.208067805188,0.208089226399,0.208110619675,0.208131985039,0.208153322514,0.208174632123,0.208195913891,0.208217167841,0.208238393996,0.20825959238,0.208280763015,0.208301905926,0.208323021136,0.208344108669,0.208365168546,0.208386200793,0.208407205432,0.208428182486,0.208449131979,0.208470053934,0.208490948374,0.208511815323,0.208532654803,0.208553466838,0.208574251451,0.208595008665,0.208615738503,0.208636440988,0.208657116144,0.208677763994,0.208698384559,0.208718977865,0.208739543932,0.208760082785,0.208780594447,0.20880107894,0.208821536288,0.208841966512,0.208862369637,0.208882745685,0.208903094679,0.208923416641,0.208943711595,0.208963979564,0.208984220569,0.209004434635,0.209024621783,0.209044782036,0.209064915418,0.20908502195,0.209105101656,0.209125154558,0.209145180678,0.20916518004,0.209185152666,0.209205098578,0.2092250178,0.209244910353,0.209264776261,0.209284615545,0.209304428228,0.209324214333,0.209343973882,0.209363706898,0.209383413402,0.209403093418,0.209422746968,0.209442374074,0.209461974759,0.209481549044,0.209501096952,0.209520618506,0.209540113727,0.209559582639,0.209579025262,0.20959844162,0.209617831735,0.209637195628,0.209656533322,0.20967584484,0.209695130202,0.209714389433,0.209733622552,0.209752829583,0.209772010548,0.209791165468,0.209810294366,0.209829397264,0.209848474183,0.209867525146,0.209886550175,0.209905549291,0.209924522516,0.209943469873,0.209962391383,0.209981287069,0.210000156951,0.210019001052,0.210037819393,0.210056611997,0.210075378885,0.210094120079,0.210112835601,0.210131525472,0.210150189715,0.21016882835,0.210187441399,0.210206028885,0.210224590828,0.210243127251,0.210261638175,0.210280123621,0.210298583612,0.210317018168,0.210335427312,0.210353811064,0.210372169447,0.210390502481,0.210408810188,0.21042709259,0.210445349708,0.210463581564,0.210481788179,0.210499969573,0.21051812577,0.210536256789,0.210554362653,0.210572443382,0.210590498998,0.210608529522,0.210626534976,0.210644515381,0.210662470757,0.210680401126,0.21069830651,0.210716186929,0.210734042405,0.210751872959,0.210769678611,0.210787459383,0.210805215297,0.210822946372,0.210840652631,0.210858334094,0.210875990783,0.210893622717,0.210911229919,0.210928812409,0.210946370208,0.210963903337,0.210981411817,0.210998895669,0.211016354914,0.211033789573,0.211051199666,0.211068585214,0.211085946239,0.211103282761,0.2111205948,0.211137882378,0.211155145515,0.211172384232,0.21118959855,0.211206788489,0.21122395407,0.211241095314,0.211258212242,0.211275304873,0.211292373229,0.211309417331,0.211326437198,0.211343432852,0.211360404312,0.2113773516,0.211394274736,0.211411173741,0.211428048635,0.211444899438,0.211461726171,0.211478528854,0.211495307508,0.211512062154,0.211528792811,0.211545499499,0.211562182241,0.211578841054,0.211595475961,0.211612086981,0.211628674134,0.211645237441,0.211661776922,0.211678292598,0.211694784488,0.211711252612,0.211727696992,0.211744117647,0.211760514596,0.211776887862,0.211793237462,0.211809563419,0.211825865751,0.211842144478,0.211858399622,0.211874631201,0.211890839237,0.211907023748,0.211923184755,0.211939322278,0.211955436337,0.211971526952,0.211987594142,0.212003637928,0.212019658329,0.212035655366,0.212051629058,0.212067579426,0.212083506488,0.212099410265,0.212115290776,0.212131148042,0.212146982082,0.212162792916,0.212178580564,0.212194345045,0.212210086379,0.212225804585,0.212241499684,0.212257171695,0.212272820638,0.212288446532,0.212304049398,0.212319629253,0.212335186119,0.212350720015,0.212366230959,0.212381718973,0.212397184075,0.212412626284,0.212428045621,0.212443442105,0.212458815755,0.21247416659,0.21248949463,0.212504799895,0.212520082404,0.212535342176,0.212550579231,0.212565793587,0.212580985265,0.212596154283,0.212611300662,0.212626424419,0.212641525575,0.212656604148,0.212671660159,0.212686693625,0.212701704567,0.212716693003,0.212731658953,0.212746602436,0.212761523471,0.212776422077,0.212791298273,0.212806152078,0.212820983512,0.212835792594,0.212850579341,0.212865343775,0.212880085913,0.212894805774,0.212909503378,0.212924178744,0.21293883189,0.212953462836,0.2129680716,0.212982658201,0.212997222658,0.213011764991,0.213026285217,0.213040783356,0.213055259427,0.213069713448,0.213084145438,0.213098555416,0.213112943401,0.213127309412,0.213141653467,0.213155975585,0.213170275785,0.213184554085,0.213198810504,0.213213045062,0.213227257775,0.213241448664,0.213255617746,0.213269765041,0.213283890566,0.213297994342,0.213312076385,0.213326136714,0.213340175349,0.213354192308,0.213368187609,0.21338216127,0.213396113311,0.213410043749,0.213423952603,0.213437839892,0.213451705633,0.213465549846,0.213479372548,0.213493173759,0.213506953496,0.213520711777,0.213534448622,0.213548164048,0.213561858074,0.213575530717,0.213589181997,0.213602811932,0.213616420539,0.213630007837,0.213643573844,0.213657118578,0.213670642058,0.213684144302,0.213697625327,0.213711085153,0.213724523796,0.213737941276,0.21375133761,0.213764712816,0.213778066913,0.213791399918,0.21380471185,0.213818002726,0.213831272565,0.213844521384,0.213857749202,0.213870956036,0.213884141904,0.213897306825,0.213910450816,0.213923573895,0.213936676081,0.21394975739,0.213962817841,0.213975857451,0.213988876239,0.214001874222,0.214014851418,0.214027807845,0.214040743521,0.214053658463,0.214066552689,0.214079426217,0.214092279065,0.214105111249,0.214117922789,0.214130713701,0.214143484004,0.214156233714,0.214168962851,0.21418167143,0.21419435947,0.214207026988,0.214219674002,0.21423230053,0.214244906589,0.214257492196,0.21427005737,0.214282602127,0.214295126485,0.214307630461,0.214320114074,0.21433257734,0.214345020277,0.214357442902,0.214369845233,0.214382227287,0.214394589081,0.214406930634,0.214419251961,0.214431553081,0.214443834011,0.214456094768,0.214468335369,0.214480555832,0.214492756174,0.214504936413,0.214517096564,0.214529236647,0.214541356677,0.214553456672,0.21456553665,0.214577596627,0.214589636621,0.214601656648,0.214613656726,0.214625636872,0.214637597103,0.214649537437,0.214661457889,0.214673358478,0.21468523922,0.214697100132,0.214708941232,0.214720762536,0.214732564061,0.214744345825,0.214756107844,0.214767850135,0.214779572716,0.214791275602,0.214802958811,0.214814622361,0.214826266267,0.214837890546,0.214849495216,0.214861080294,0.214872645795,0.214884191737,0.214895718137,0.214907225012,0.214918712378,0.214930180252,0.21494162865,0.214953057591,0.214964467089,0.214975857162,0.214987227827,0.2149985791,0.215009910998,0.215021223538,0.215032516736,0.215043790609,0.215055045173,0.215066280445,0.215077496442,0.21508869318,0.215099870676,0.215111028946,0.215122168007,0.215133287876,0.215144388568,0.2151554701,0.21516653249,0.215177575752,0.215188599904,0.215199604963,0.215210590944,0.215221557864,0.215232505739,0.215243434586,0.215254344421,0.21526523526,0.215276107121,0.215286960018,0.215297793969,0.21530860899,0.215319405097,0.215330182306,0.215340940634,0.215351680096,0.21536240071,0.215373102491,0.215383785455,0.215394449619,0.215405094999,0.215415721611,0.215426329471,0.215436918596,0.215447489001,0.215458040703,0.215468573717,0.21547908806,0.215489583749,0.215500060798,0.215510519224,0.215520959043,0.215531380271,0.215541782924,0.215552167019,0.21556253257,0.215572879595,0.215583208108,0.215593518127,0.215603809666,0.215614082743,0.215624337372,0.215634573569,0.215644791352,0.215654990734,0.215665171733,0.215675334364,0.215685478643,0.215695604586,0.215705712208,0.215715801526,0.215725872555,0.215735925311,0.215745959809,0.215755976066,0.215765974098,0.215775953919,0.215785915545,0.215795858993,0.215805784278,0.215815691416,0.215825580422,0.215835451312,0.215845304101,0.215855138806,0.215864955441,0.215874754023,0.215884534567,0.215894297088,0.215904041603,0.215913768126,0.215923476673,0.21593316726,0.215942839903,0.215952494616,0.215962131415,0.215971750316,0.215981351334,0.215990934484,0.216000499783,0.216010047245,0.216019576885,0.21602908872,0.216038582765,0.216048059034,0.216057517544,0.216066958309,0.216076381346,0.216085786668,0.216095174293,0.216104544234,0.216113896507,0.216123231128,0.216132548112,0.216141847474,0.216151129228,0.216160393392,0.216169639978,0.216178869004,0.216188080484,0.216197274433,0.216206450866,0.216215609798,0.216224751246,0.216233875223,0.216242981745,0.216252070827,0.216261142483,0.21627019673,0.216279233583,0.216288253055,0.216297255163,0.216306239921,0.216315207344,0.216324157447,0.216333090246,0.216342005755,0.21635090399,0.216359784964,0.216368648694,0.216377495193,0.216386324478,0.216395136563,0.216403931462,0.216412709191,0.216421469764,0.216430213197,0.216438939504,0.2164476487,0.2164563408,0.216465015819,0.216473673771,0.216482314672,0.216490938535,0.216499545376,0.21650813521,0.216516708051,0.216525263914,0.216533802814,0.216542324765,0.216550829782,0.21655931788,0.216567789074,0.216576243378,0.216584680806,0.216593101374,0.216601505095,0.216609891986,0.216618262059,0.21662661533,0.216634951814,0.216643271525,0.216651574477,0.216659860685,0.216668130163,0.216676382927,0.21668461899,0.216692838367,0.216701041072,0.216709227121,0.216717396527,0.216725549305,0.216733685469,0.216741805034,0.216749908014,0.216757994423,0.216766064276,0.216774117588,0.216782154372,0.216790174643,0.216798178416,0.216806165704,0.216814136522,0.216822090884,0.216830028805,0.216837950299,0.21684585538,0.216853744062,0.216861616359,0.216869472287,0.216877311859,0.216885135088,0.216892941991,0.21690073258,0.21690850687,0.216916264874,0.216924006608,0.216931732085,0.21693944132,0.216947134326,0.216954811118,0.21696247171,0.216970116115,0.216977744348,0.216985356423,0.216992952355,0.217000532156,0.217008095841,0.217015643425,0.21702317492,0.217030690342,0.217038189704,0.217045673019,0.217053140303,0.217060591569,0.21706802683,0.217075446102,0.217082849397,0.21709023673,0.217097608114,0.217104963564,0.217112303093,0.217119626715,0.217126934444,0.217134226294,0.217141502279,0.217148762412,0.217156006707,0.217163235179,0.21717044784,0.217177644705,0.217184825787,0.217191991101,0.217199140659,0.217206274476,0.217213392565,0.21722049494,0.217227581615,0.217234652603,0.217241707918,0.217248747574,0.217255771584,0.217262779962,0.217269772722,0.217276749877,0.217283711441,0.217290657428,0.21729758785,0.217304502722,0.217311402057,0.217318285868,0.21732515417,0.217332006976,0.217338844299,0.217345666153,0.217352472551,0.217359263506,0.217366039033,0.217372799145,0.217379543855,0.217386273176,0.217392987122,0.217399685707,0.217406368943,0.217413036845,0.217419689425,0.217426326698,0.217432948675,0.217439555372,0.2174461468,0.217452722974,0.217459283906,0.217465829611,0.217472360101,0.217478875389,0.217485375489,0.217491860415,0.217498330178,0.217504784794,0.217511224274,0.217517648632,0.217524057882,0.217530452036,0.217536831108,0.217543195111,0.217549544058,0.217555877962,0.217562196836,0.217568500694,0.217574789549,0.217581063413,0.2175873223,0.217593566223,0.217599795195,0.21760600923,0.217612208339,0.217618392537,0.217624561836,0.217630716249,0.217636855789,0.21764298047,0.217649090305,0.217655185305,0.217661265485,0.217667330857,0.217673381435,0.217679417231,0.217685438257,0.217691444528,0.217697436056,0.217703412854,0.217709374934,0.217715322311,0.217721254995,0.217727173001,0.217733076342,0.217738965029,0.217744839077,0.217750698497,0.217756543303,0.217762373507,0.217768189123,0.217773990162,0.217779776638,0.217785548564,0.217791305952,0.217797048815,0.217802777165,0.217808491016,0.21781419038,0.21781987527,0.217825545699,0.217831201679,0.217836843222,0.217842470343,0.217848083052,0.217853681363,0.217859265289,0.217864834842,0.217870390034,0.217875930879,0.217881457389,0.217886969576,0.217892467453,0.217897951033,0.217903420328,0.21790887535,0.217914316113,0.217919742629,0.217925154909,0.217930552968,0.217935936816,0.217941306467,0.217946661934,0.217952003228,0.217957330362,0.217962643348,0.217967942199,0.217973226928,0.217978497547,0.217983754067,0.217988996502,0.217994224864,0.217999439165,0.218004639418,0.218009825635,0.218014997828,0.218020156009,0.218025300192,0.218030430388,0.218035546609,0.218040648868,0.218045737177,0.218050811549,0.218055871995,0.218060918528,0.21806595116,0.218070969903,0.21807597477,0.218080965773,0.218085942924,0.218090906234,0.218095855717,0.218100791385,0.218105713249,0.218110621322,0.218115515616,0.218120396143,0.218125262915,0.218130115944,0.218134955242,0.218139780822,0.218144592696,0.218149390875,0.218154175372,0.218158946199,0.218163703367,0.218168446889,0.218173176777,0.218177893043,0.218182595698,0.218187284755,0.218191960227,0.218196622123,0.218201270458,0.218205905242,0.218210526488,0.218215134207,0.218219728412,0.218224309114,0.218228876326,0.218233430059,0.218237970324,0.218242497135,0.218247010502,0.218251510439,0.218255996955,0.218260470064,0.218264929777,0.218269376106,0.218273809063,0.21827822866,0.218282634908,0.218287027819,0.218291407405,0.218295773678,0.218300126649,0.21830446633,0.218308792733,0.21831310587,0.218317405753,0.218321692392,0.2183259658,0.218330225988,0.218334472969,0.218338706753,0.218342927353,0.21834713478,0.218351329046,0.218355510162,0.21835967814,0.218363832991,0.218367974728,0.218372103361,0.218376218903,0.218380321365,0.218384410758,0.218388487094,0.218392550385,0.218396600642,0.218400637876,0.2184046621,0.218408673324,0.21841267156,0.21841665682,0.218420629116,0.218424588457,0.218428534857,0.218432468327,0.218436388877,0.21844029652,0.218444191266,0.218448073128,0.218451942116,0.218455798242,0.218459641518,0.218463471954,0.218467289563,0.218471094355,0.218474886341,0.218478665534,0.218482431945,0.218486185584,0.218489926464,0.218493654595,0.218497369989,0.218501072656,0.218504762609,0.218508439859,0.218512104416,0.218515756293,0.2185193955,0.218523022048,0.218526635949,0.218530237214,0.218533825854,0.218537401881,0.218540965305,0.218544516138,0.218548054391,0.218551580075,0.218555093201,0.21855859378,0.218562081824,0.218565557344,0.21856902035,0.218572470854,0.218575908867,0.2185793344,0.218582747464,0.21858614807,0.218589536229,0.218592911953,0.218596275251,0.218599626137,0.218602964619,0.21860629071,0.21860960442,0.218612905761,0.218616194743,0.218619471377,0.218622735674,0.218625987646,0.218629227303,0.218632454656,0.218635669716,0.218638872494,0.218642063001,0.218645241248,0.218648407246,0.218651561005,0.218654702537,0.218657831852,0.218660948961,0.218664053876,0.218667146606,0.218670227163,0.218673295558,0.218676351801,0.218679395903,0.218682427875,0.218685447729,0.218688455473,0.21869145112,0.218694434681,0.218697406165,0.218700365584,0.218703312948,0.218706248268,0.218709171556,0.21871208282,0.218714982073,0.218717869325,0.218720744587,0.218723607869,0.218726459182,0.218729298537,0.218732125944,0.218734941414,0.218737744957,0.218740536585,0.218743316308,0.218746084136,0.21874884008,0.218751584151,0.218754316359,0.218757036715,0.218759745229,0.218762441912,0.218765126775,0.218767799828,0.218770461081,0.218773110546,0.218775748232,0.21877837415,0.218780988311,0.218783590725,0.218786181402,0.218788760354,0.21879132759,0.218793883121,0.218796426958,0.21879895911,0.218801479589,0.218803988404,0.218806485567,0.218808971087,0.218811444976,0.218813907242,0.218816357897,0.218818796952,0.218821224416,0.218823640299,0.218826044613,0.218828437368,0.218830818573,0.218833188239,0.218835546377,0.218837892997,0.218840228108,0.218842551723,0.218844863849,0.218847164499,0.218849453682,0.218851731408,0.218853997688,0.218856252532,0.21885849595,0.218860727953,0.21886294855,0.218865157752,0.218867355568,0.218869542011,0.218871717088,0.218873880811,0.21887603319,0.218878174234,0.218880303955,0.218882422362,0.218884529465,0.218886625274,0.2188887098,0.218890783053,0.218892845042,0.218894895778,0.218896935271,0.218898963531,0.218900980568,0.218902986392,0.218904981014,0.218906964442,0.218908936688,0.218910897761,0.218912847672,0.218914786429,0.218916714044,0.218918630526,0.218920535886,0.218922430132,0.218924313276,0.218926185327,0.218928046295,0.21892989619,0.218931735023,0.218933562802,0.218935379538,0.218937185241,0.21893897992,0.218940763586,0.218942536249,0.218944297917,0.218946048602,0.218947788313,0.21894951706,0.218951234853,0.218952941701,0.218954637615,0.218956322604,0.218957996679,0.218959659848,0.218961312121,0.21896295351,0.218964584022,0.218966203669,0.218967812459,0.218969410403,0.218970997511,0.218972573791,0.218974139254,0.21897569391,0.218977237768,0.218978770838,0.21898029313,0.218981804653,0.218983305417,0.218984795432,0.218986274707,0.218987743252,0.218989201078,0.218990648192,0.218992084606,0.218993510328,0.218994925369,0.218996329737,0.218997723443,0.218999106497,0.219000478906,0.219001840683,0.219003191835,0.219004532373,0.219005862305,0.219007181643,0.219008490394,0.219009788569,0.219011076178,0.219012353229,0.219013619732,0.219014875697,0.219016121134,0.219017356051,0.219018580458,0.219019794366,0.219020997782,0.219022190717,0.219023373181,0.219024545182,0.219025706729,0.219026857834,0.219027998504,0.21902912875,0.21903024858,0.219031358004,0.219032457032,0.219033545673,0.219034623936,0.219035691831,0.219036749367,0.219037796553,0.219038833399,0.219039859914,0.219040876107,0.219041881989,0.219042877567,0.219043862851,0.219044837852,0.219045802577,0.219046757036,0.219047701239,0.219048635195,0.219049558913,0.219050472402,0.219051375672,0.219052268732,0.21905315159,0.219054024257,0.219054886742,0.219055739053,0.219056581201,0.219057413193,0.21905823504,0.21905904675,0.219059848333,0.219060639798,0.219061421154,0.21906219241,0.219062953575,0.219063704659,0.21906444567,0.219065176619,0.219065897513,0.219066608362,0.219067309175,0.219067999961,0.219068680729,0.219069351489,0.219070012249,0.219070663019,0.219071303807,0.219071934623,0.219072555475,0.219073166373,0.219073767326,0.219074358342,0.219074939431,0.219075510602,0.219076071864,0.219076623225,0.219077164695,0.219077696283,0.219078217997,0.219078729847,0.219079231842,0.21907972399,0.2190802063,0.219080678782,0.219081141444,0.219081594296,0.219082037345,0.219082470602,0.219082894075,0.219083307772,0.219083711703,0.219084105877,0.219084490302,0.219084864988,0.219085229943,0.219085585176,0.219085930696,0.219086266511,0.219086592631,0.219086909064,0.21908721582,0.219087512907,0.219087800333,0.219088078108,0.21908834624,0.219088604738,0.219088853612,0.219089092869,0.219089322518,0.219089542569,0.219089753029,0.219089953908,0.219090145214,0.219090326957,0.219090499144,0.219090661785,0.219090814888,0.219090958462,0.219091092515,0.219091217057,0.219091332096,0.21909143764,0.219091533698,0.219091620279,0.219091697392,0.219091765045,0.219091823247,0.219091872006,0.219091911331,0.219091941231,0.219091961714,0.219091972789,0.219091974464,0.219091966749,0.219091949651,0.219091923179,0.219091887341,0.219091842147,0.219091787605,0.219091723723,0.219091650509,0.219091567974,0.219091476124,0.219091374968,0.219091264515,0.219091144774,0.219091015753,0.21909087746,0.219090729903,0.219090573092,0.219090407035,0.21909023174,0.219090047216,0.219089853471,0.219089650513,0.219089438352,0.219089216994,0.21908898645,0.219088746726,0.219088497833,0.219088239777,0.219087972568,0.219087696213,0.219087410722,0.219087116102,0.219086812362,0.21908649951,0.219086177555,0.219085846504,0.219085506367,0.219085157151,0.219084798865,0.219084431518,0.219084055117,0.21908366967,0.219083275187,0.219082871675,0.219082459143,0.219082037598,0.21908160705,0.219081167506,0.219080718975,0.219080261465,0.219079794984,0.21907931954,0.219078835142,0.219078341797,0.219077839515,0.219077328303,0.219076808169,0.219076279122,0.21907574117,0.219075194321,0.219074638583,0.219074073964,0.219073500473,0.219072918117,0.219072326906,0.219071726846,0.219071117946,0.219070500214,0.219069873658,0.219069238287,0.219068594109,0.219067941131,0.219067279361,0.219066608809,0.219065929481,0.219065241386,0.219064544532,0.219063838928,0.21906312458,0.219062401497,0.219061669688,0.21906092916,0.219060179921,0.219059421979,0.219058655342,0.219057880019,0.219057096017,0.219056303345,0.219055502009,0.219054692019,0.219053873382,0.219053046106,0.219052210199,0.21905136567,0.219050512525,0.219049650773,0.219048780423,0.219047901481,0.219047013955,0.219046117855,0.219045213187,0.21904429996,0.21904337818,0.219042447858,0.219041508999,0.219040561612,0.219039605705,0.219038641286,0.219037668363,0.219036686943,0.219035697034,0.219034698645,0.219033691782,0.219032676455,0.21903165267,0.219030620435,0.219029579759,0.219028530648,0.219027473112,0.219026407157,0.219025332791,0.219024250023,0.21902315886,0.219022059309,0.219020951379,0.219019835077,0.219018710411,0.219017577389,0.219016436018,0.219015286306,0.219014128261,0.219012961891,0.219011787203,0.219010604205,0.219009412904,0.219008213309,0.219007005427,0.219005789266,0.219004564833,0.219003332137,0.219002091183,0.219000841982,0.218999584539,0.218998318863,0.218997044961,0.218995762841,0.218994472511,0.218993173977,0.218991867249,0.218990552332,0.218989229236,0.218987897967,0.218986558533,0.218985210942,0.218983855201,0.218982491318,0.2189811193,0.218979739155,0.218978350891,0.218976954514,0.218975550033,0.218974137456,0.218972716788,0.218971288039,0.218969851215,0.218968406324,0.218966953374,0.218965492372,0.218964023326,0.218962546242,0.218961061129,0.218959567994,0.218958066845,0.218956557688,0.218955040532,0.218953515383,0.218951982249,0.218950441139,0.218948892058,0.218947335014,0.218945770016,0.218944197069,0.218942616183,0.218941027363,0.218939430618,0.218937825955,0.218936213381,0.218934592903,0.21893296453,0.218931328267,0.218929684124,0.218928032106,0.218926372222,0.218924704478,0.218923028882,0.218921345442,0.218919654164,0.218917955056,0.218916248125,0.218914533379,0.218912810824,0.218911080469,0.21890934232,0.218907596385,0.21890584267,0.218904081184,0.218902311933,0.218900534925,0.218898750166,0.218896957665,0.218895157429,0.218893349463,0.218891533777,0.218889710377,0.21888787927,0.218886040463,0.218884193964,0.21888233978,0.218880477918,0.218878608385,0.218876731188,0.218874846335,0.218872953833,0.218871053688,0.218869145909,0.218867230502,0.218865307474,0.218863376832,0.218861438584,0.218859492736,0.218857539297,0.218855578272,0.218853609669,0.218851633495,0.218849649758,0.218847658463,0.218845659619,0.218843653233,0.21884163931,0.21883961786,0.218837588888,0.218835552402,0.218833508408,0.218831456914,0.218829397927,0.218827331453,0.218825257501,0.218823176076,0.218821087186,0.218818990838,0.218816887039,0.218814775795,0.218812657115,0.218810531004,0.21880839747,0.21880625652,0.21880410816,0.218801952399,0.218799789241,0.218797618696,0.218795440769,0.218793255467,0.218791062798,0.218788862768,0.218786655384,0.218784440653,0.218782218583,0.218779989179,0.21877775245,0.218775508401,0.218773257039,0.218770998373,0.218768732407,0.21876645915,0.218764178608,0.218761890788,0.218759595697,0.218757293341,0.218754983728,0.218752666864,0.218750342757,0.218748011412,0.218745672838,0.21874332704,0.218740974025,0.218738613801,0.218736246373,0.21873387175,0.218731489937,0.218729100941,0.21872670477,0.21872430143,0.218721890927,0.218719473269,0.218717048462,0.218714616513,0.218712177429,0.218709731216,0.218707277881,0.218704817432,0.218702349874,0.218699875214,0.218697393459,0.218694904616,0.218692408691,0.218689905692,0.218687395624,0.218684878495,0.218682354311,0.218679823079,0.218677284806,0.218674739497,0.21867218716,0.218669627802,0.218667061429,0.218664488048,0.218661907665,0.218659320287,0.218656725921,0.218654124573,0.21865151625,0.218648900958,0.218646278705,0.218643649496,0.218641013338,0.218638370238,0.218635720203,0.218633063238,0.218630399351,0.218627728548,0.218625050836,0.218622366221,0.21861967471,0.218616976309,0.218614271025,0.218611558865,0.218608839834,0.218606113939,0.218603381188,0.218600641586,0.21859789514,0.218595141857,0.218592381742,0.218589614803,0.218586841045,0.218584060476,0.218581273102,0.21857847893,0.218575677965,0.218572870214,0.218570055684,0.218567234381,0.218564406311,0.218561571482,0.218558729899,0.218555881569,0.218553026498,0.218550164693,0.21854729616,0.218544420906,0.218541538936,0.218538650258,0.218535754877,0.218532852801,0.218529944035,0.218527028585,0.218524106459,0.218521177663,0.218518242202,0.218515300084,0.218512351314,0.218509395899,0.218506433846,0.21850346516,0.218500489848,0.218497507916,0.218494519371,0.218491524218,0.218488522465,0.218485514117,0.218482499181,0.218479477664,0.21847644957,0.218473414907,0.218470373681,0.218467325898,0.218464271565,0.218461210687,0.218458143272,0.218455069324,0.218451988851,0.218448901858,0.218445808353,0.218442708341,0.218439601828,0.21843648882,0.218433369324,0.218430243347,0.218427110893,0.21842397197,0.218420826584,0.21841767474,0.218414516445,0.218411351706,0.218408180527,0.218405002916,0.218401818879,0.218398628422,0.21839543155,0.218392228271,0.21838901859,0.218385802513,0.218382580047,0.218379351197,0.21837611597,0.218372874372,0.218369626409,0.218366372087,0.218363111412,0.218359844391,0.218356571029,0.218353291332,0.218350005307,0.218346712959,0.218343414296,0.218340109322,0.218336798043,0.218333480467,0.218330156599,0.218326826444,0.21832349001,0.218320147302,0.218316798325,0.218313443087,0.218310081593,0.21830671385,0.218303339862,0.218299959636,0.218296573179,0.218293180496,0.218289781593,0.218286376477,0.218282965152,0.218279547625,0.218276123903,0.218272693991,0.218269257895,0.21826581562,0.218262367174,0.218258912561,0.218255451789,0.218251984862,0.218248511787,0.218245032569,0.218241547216,0.218238055731,0.218234558122,0.218231054394,0.218227544554,0.218224028607,0.218220506558,0.218216978415,0.218213444182,0.218209903866,0.218206357473,0.218202805008,0.218199246477,0.218195681886,0.218192111242,0.218188534549,0.218184951814,0.218181363042,0.21817776824,0.218174167413,0.218170560568,0.218166947709,0.218163328842,0.218159703975,0.218156073111,0.218152436258,0.21814879342,0.218145144605,0.218141489817,0.218137829062,0.218134162346,0.218130489675,0.218126811055,0.218123126491,0.218119435989,0.218115739555,0.218112037195,0.218108328914,0.218104614718,0.218100894614,0.218097168605,0.2180934367,0.218089698902,0.218085955218,0.218082205654,0.218078450214,0.218074688906,0.218070921735,0.218067148705,0.218063369824,0.218059585097,0.218055794528,0.218051998125,0.218048195893,0.218044387837,0.218040573963,0.218036754277,0.218032928784,0.21802909749,0.218025260401,0.218021417523,0.21801756886,0.218013714419,0.218009854205,0.218005988224,0.218002116481,0.217998238983,0.217994355734,0.21799046674,0.217986572007,0.217982671541,0.217978765347,0.21797485343,0.217970935797,0.217967012453,0.217963083403,0.217959148652,0.217955208208,0.217951262074,0.217947310257,0.217943352763,0.217939389596,0.217935420762,0.217931446267,0.217927466116,0.217923480315,0.21791948887,0.217915491785,0.217911489067,0.217907480721,0.217903466752,0.217899447166,0.217895421969,0.217891391165,0.217887354761,0.217883312762,0.217879265173,0.217875212,0.217871153248,0.217867088923,0.21786301903,0.217858943574,0.217854862562,0.217850775999,0.217846683889,0.217842586239,0.217838483053,0.217834374338,0.217830260099,0.217826140341,0.217822015069,0.21781788429,0.217813748008,0.217809606228,0.217805458957,0.217801306199,0.21779714796,0.217792984245,0.217788815061,0.217784640411,0.217780460302,0.217776274738,0.217772083726,0.21776788727,0.217763685376,0.21775947805,0.217755265295,0.217751047119,0.217746823526,0.217742594522,0.217738360112,0.2177341203,0.217729875094,0.217725624497,0.217721368515,0.217717107154,0.217712840419,0.217708568315,0.217704290847,0.217700008021,0.217695719841,0.217691426314,0.217687127444,0.217682823238,0.217678513699,0.217674198833,0.217669878646,0.217665553143,0.217661222329,0.217656886209,0.217652544788,0.217648198072,0.217643846067,0.217639488776,0.217635126206,0.217630758362,0.217626385248,0.217622006871,0.217617623235,0.217613234345,0.217608840207,0.217604440826,0.217600036207,0.217595626355,0.217591211276,0.217586790974,0.217582365455,0.217577934724,0.217573498786,0.217569057646,0.21756461131,0.217560159782,0.217555703068,0.217551241173,0.217546774102,0.217542301859,0.217537824452,0.217533341883,0.217528854159,0.217524361285,0.217519863265,0.217515360105,0.21751085181,0.217506338386,0.217501819836,0.217497296166,0.217492767382,0.217488233488,0.21748369449,0.217479150392,0.2174746012,0.217470046919,0.217465487553,0.217460923108,0.217456353589,0.217451779001,0.217447199348,0.217442614637,0.217438024872,0.217433430058,0.2174288302,0.217424225303,0.217419615372,0.217415000412,0.217410380429,0.217405755426,0.21740112541,0.217396490385,0.217391850356,0.217387205328,0.217382555307,0.217377900296,0.217373240302,0.217368575329,0.217363905382,0.217359230465,0.217354550585,0.217349865746,0.217345175953,0.21734048121,0.217335781524,0.217331076898,0.217326367338,0.217321652848,0.217316933434,0.2173122091,0.217307479852,0.217302745694,0.217298006631,0.217293262669,0.217288513811,0.217283760064,0.217279001431,0.217274237918,0.21726946953,0.217264696271,0.217259918147,0.217255135161,0.21725034732,0.217245554628,0.21724075709,0.217235954711,0.217231147495,0.217226335448,0.217221518574,0.217216696878,0.217211870365,0.21720703904,0.217202202908,0.217197361973,0.217192516241,0.217187665715,0.217182810402,0.217177950305,0.21717308543,0.217168215782,0.217163341364,0.217158462183,0.217153578242,0.217148689547,0.217143796103,0.217138897913,0.217133994984,0.217129087319,0.217124174924,0.217119257802,0.21711433596,0.217109409402,0.217104478132,0.217099542156,0.217094601477,0.217089656101,0.217084706033,0.217079751277,0.217074791838,0.217069827721,0.21706485893,0.21705988547,0.217054907346,0.217049924563,0.217044937125,0.217039945037,0.217034948303,0.217029946929,0.217024940919,0.217019930278,0.217014915011,0.217009895121,0.217004870614,0.216999841495,0.216994807768,0.216989769438,0.216984726509,0.216979678986,0.216974626874,0.216969570178,0.216964508902,0.21695944305,0.216954372628,0.21694929764,0.21694421809,0.216939133984,0.216934045326,0.21692895212,0.216923854371,0.216918752084,0.216913645264,0.216908533914,0.21690341804,0.216898297646,0.216893172737,0.216888043317,0.216882909391,0.216877770964,0.216872628039,0.216867480623,0.216862328718,0.216857172331,0.216852011465,0.216846846124,0.216841676314,0.21683650204,0.216831323305,0.216826140113,0.216820952471,0.216815760382,0.21681056385,0.216805362881,0.216800157479,0.216794947648,0.216789733393,0.216784514718,0.216779291628,0.216774064127,0.21676883222,0.216763595912,0.216758355206,0.216753110108,0.216747860622,0.216742606752,0.216737348503,0.216732085879,0.216726818885,0.216721547525,0.216716271804,0.216710991726,0.216705707296,0.216700418517,0.216695125396,0.216689827935,0.21668452614,0.216679220014,0.216673909563,0.216668594791,0.216663275701,0.2166579523,0.21665262459,0.216647292577,0.216641956265,0.216636615658,0.216631270761,0.216625921577,0.216620568113,0.216615210371,0.216609848356,0.216604482073,0.216599111526,0.21659373672,0.216588357658,0.216582974346,0.216577586787,0.216572194986,0.216566798947,0.216561398675,0.216555994174,0.216550585449,0.216545172503,0.216539755341,0.216534333967,0.216528908387,0.216523478603,0.216518044621,0.216512606444,0.216507164078,0.216501717526,0.216496266793,0.216490811883,0.2164853528,0.216479889548,0.216474422133,0.216468950558,0.216463474827,0.216457994946,0.216452510917,0.216447022746,0.216441530437,0.216436033993,0.21643053342,0.216425028721,0.216419519901,0.216414006965,0.216408489915,0.216402968757,0.216397443495,0.216391914133,0.216386380675,0.216380843126,0.21637530149,0.216369755771,0.216364205973,0.2163586521,0.216353094157,0.216347532149,0.216341966078,0.21633639595,0.216330821769,0.216325243538,0.216319661262,0.216314074946,0.216308484593,0.216302890207,0.216297291794,0.216291689356,0.216286082899,0.216280472426,0.216274857941,0.21626923945,0.216263616955,0.216257990461,0.216252359972,0.216246725493,0.216241087028,0.21623544458,0.216229798154,0.216224147754,0.216218493384,0.216212835048,0.216207172751,0.216201506496,0.216195836288,0.216190162131,0.216184484029,0.216178801986,0.216173116006,0.216167426093,0.216161732251,0.216156034486,0.216150332799,0.216144627197,0.216138917682,0.216133204259,0.216127486932,0.216121765705,0.216116040583,0.216110311568,0.216104578666,0.21609884188,0.216093101215,0.216087356674,0.216081608261,0.216075855982,0.216070099839,0.216064339836,0.216058575979,0.21605280827,0.216047036714,0.216041261316,0.216035482078,0.216029699005,0.216023912101,0.21601812137,0.216012326816,0.216006528443,0.216000726255,0.215994920257,0.215989110451,0.215983296843,0.215977479436,0.215971658233,0.21596583324,0.21596000446,0.215954171897,0.215948335556,0.215942495439,0.215936651551,0.215930803897,0.215924952479,0.215919097302,0.21591323837,0.215907375687,0.215901509257,0.215895639084,0.215889765171,0.215883887524,0.215878006144,0.215872121038,0.215866232208,0.215860339658,0.215854443393,0.215848543416,0.215842639731,0.215836732343,0.215830821255,0.21582490647,0.215818987994,0.215813065829,0.215807139981,0.215801210451,0.215795277246,0.215789340368,0.215783399821,0.215777455609,0.215771507737,0.215765556208,0.215759601025,0.215753642193,0.215747679716,0.215741713598,0.215735743842,0.215729770452,0.215723793432,0.215717812786,0.215711828518,0.215705840632,0.215699849131,0.215693854019,0.215687855301,0.21568185298,0.21567584706,0.215669837544,0.215663824437,0.215657807743,0.215651787464,0.215645763606,0.215639736171,0.215633705165,0.215627670589,0.215621632449,0.215615590748,0.21560954549,0.215603496679,0.215597444318,0.215591388411,0.215585328963,0.215579265976,0.215573199455,0.215567129403,0.215561055825,0.215554978723,0.215548898103,0.215542813966,0.215536726318,0.215530635163,0.215524540502,0.215518442342,0.215512340685,0.215506235534,0.215500126895,0.21549401477,0.215487899163,0.215481780079,0.21547565752,0.21546953149,0.215463401994,0.215457269035,0.215451132616,0.215444992742,0.215438849415,0.215432702641,0.215426552422,0.215420398762,0.215414241666,0.215408081135,0.215401917175,0.215395749789,0.215389578981,0.215383404754,0.215377227112,0.215371046059,0.215364861599,0.215358673734,0.215352482469,0.215346287808,0.215340089754,0.21533388831,0.215327683481,0.21532147527,0.215315263681,0.215309048717,0.215302830382,0.215296608679,0.215290383613,0.215284155187,0.215277923405,0.21527168827,0.215265449785,0.215259207956,0.215252962784,0.215246714274,0.215240462429,0.215234207253,0.21522794875,0.215221686923,0.215215421776,0.215209153312,0.215202881535,0.215196606449,0.215190328056,0.215184046362,0.215177761369,0.215171473081,0.215165181501,0.215158886634,0.215152588482,0.215146287049,0.215139982339,0.215133674356,0.215127363102,0.215121048582,0.215114730799,0.215108409757,0.215102085459,0.215095757908,0.215089427109,0.215083093064,0.215076755778,0.215070415254,0.215064071495,0.215057724505,0.215051374288,0.215045020847,0.215038664185,0.215032304306,0.215025941214,0.215019574911,0.215013205403,0.215006832691,0.21500045678,0.214994077673,0.214987695374,0.214981309886,0.214974921212,0.214968529357,0.214962134323,0.214955736114,0.214949334733,0.214942930185,0.214936522472,0.214930111598,0.214923697567,0.214917280381,0.214910860045,0.214904436561,0.214898009934,0.214891580167,0.214885147262,0.214878711225,0.214872272057,0.214865829763,0.214859384346,0.214852935809,0.214846484157,0.214840029391,0.214833571516,0.214827110536,0.214820646452,0.21481417927,0.214807708993,0.214801235623,0.214794759164,0.21478827962,0.214781796994,0.214775311289,0.21476882251,0.214762330658,0.214755835738,0.214749337753,0.214742836707,0.214736332603,0.214729825443,0.214723315232,0.214716801974,0.21471028567,0.214703766325,0.214697243943,0.214690718525,0.214684190077,0.214677658601,0.2146711241,0.214664586578,0.214658046039,0.214651502485,0.21464495592,0.214638406347,0.21463185377,0.214625298192,0.214618739616,0.214612178046,0.214605613485,0.214599045936,0.214592475403,0.214585901888,0.214579325396,0.21457274593,0.214566163492,0.214559578087,0.214552989717,0.214546398386,0.214539804097,0.214533206854,0.21452660666,0.214520003517,0.21451339743,0.214506788401,0.214500176434,0.214493561533,0.2144869437,0.214480322938,0.214473699252,0.214467072644,0.214460443118,0.214453810676,0.214447175323,0.214440537061,0.214433895893,0.214427251824,0.214420604855,0.214413954991,0.214407302235,0.214400646589,0.214393988058,0.214387326644,0.21438066235,0.214373995181,0.214367325138,0.214360652226,0.214353976448,0.214347297806,0.214340616304,0.214333931945,0.214327244733,0.214320554671,0.214313861761,0.214307166007,0.214300467413,0.214293765981,0.214287061715,0.214280354617,0.214273644692,0.214266931942,0.21426021637,0.21425349798,0.214246776775,0.214240052758,0.214233325932,0.2142265963,0.214219863866,0.214213128633,0.214206390603,0.214199649781,0.214192906168,0.214186159769,0.214179410587,0.214172658624,0.214165903884,0.214159146371,0.214152386086,0.214145623033,0.214138857216,0.214132088638,0.214125317301,0.214118543209,0.214111766365,0.214104986772,0.214098204433,0.214091419352,0.214084631531,0.214077840974,0.214071047683,0.214064251662,0.214057452914,0.214050651443,0.21404384725,0.214037040339,0.214030230714,0.214023418378,0.214016603333,0.214009785582,0.214002965129,0.213996141977,0.213989316129,0.213982487588,0.213975656357,0.213968822439,0.213961985838,0.213955146555,0.213948304595,0.213941459961,0.213934612655,0.21392776268,0.21392091004,0.213914054738,0.213907196776,0.213900336158,0.213893472888,0.213886606966,0.213879738398,0.213872867186,0.213865993333,0.213859116842,0.213852237716,0.213845355958,0.213838471571,0.213831584558,0.213824694922,0.213817802667,0.213810907795,0.213804010309,0.213797110213,0.213790207508,0.213783302199,0.213776394288,0.213769483779,0.213762570674,0.213755654976,0.213748736688,0.213741815814,0.213734892355,0.213727966317,0.2137210377,0.213714106509,0.213707172745,0.213700236413,0.213693297515,0.213686356055,0.213679412034,0.213672465457,0.213665516325,0.213658564643,0.213651610412,0.213644653637,0.213637694319,0.213630732462,0.213623768069,0.213616801143,0.213609831686,0.213602859702,0.213595885194,0.213588908164,0.213581928615,0.213574946551,0.213567961975,0.213560974888,0.213553985295,0.213546993197,0.213539998599,0.213533001503,0.213526001912,0.213518999828,0.213511995255,0.213504988196,0.213497978653,0.213490966629,0.213483952128,0.213476935152,0.213469915704,0.213462893788,0.213455869405,0.213448842559,0.213441813252,0.213434781488,0.21342774727,0.2134207106,0.213413671481,0.213406629916,0.213399585908,0.21339253946,0.213385490574,0.213378439255,0.213371385503,0.213364329323,0.213357270717,0.213350209688,0.213343146239,0.213336080373,0.213329012092,0.2133219414,0.213314868298,0.213307792791,0.213300714881,0.213293634571,0.213286551863,0.213279466761,0.213272379267,0.213265289384,0.213258197115,0.213251102462,0.21324400543,0.213236906019,0.213229804234,0.213222700077,0.213215593551,0.213208484658,0.213201373402,0.213194259785,0.21318714381,0.21318002548,0.213172904798,0.213165781766,0.213158656387,0.213151528664,0.213144398601,0.213137266198,0.21313013146,0.21312299439,0.213115854989,0.213108713261,0.213101569208,0.213094422834,0.213087274141,0.213080123132,0.213072969809,0.213065814175,0.213058656234,0.213051495987,0.213044333438,0.21303716859,0.213030001444,0.213022832005,0.213015660274,0.213008486254,0.213001309949,0.21299413136,0.212986950491,0.212979767344,0.212972581922,0.212965394228,0.212958204265,0.212951012034,0.21294381754,0.212936620784,0.21292942177,0.2129222205,0.212915016976,0.212907811202,0.212900603181,0.212893392914,0.212886180405,0.212878965656,0.212871748671,0.212864529451,0.212857307999,0.212850084319,0.212842858413,0.212835630283,0.212828399932,0.212821167363,0.212813932579,0.212806695582,0.212799456375,0.21279221496,0.212784971341,0.21277772552,0.2127704775,0.212763227282,0.212755974871,0.212748720268,0.212741463477,0.212734204499,0.212726943339,0.212719679997,0.212712414477,0.212705146782,0.212697876914,0.212690604876,0.21268333067,0.2126760543,0.212668775767,0.212661495074,0.212654212225,0.212646927221,0.212639640066,0.212632350761,0.21262505931,0.212617765716,0.21261046998,0.212603172105,0.212595872094,0.21258856995,0.212581265676,0.212573959273,0.212566650744,0.212559340093,0.212552027321,0.212544712432,0.212537395427,0.21253007631,0.212522755083,0.212515431748,0.212508106309,0.212500778768,0.212493449127,0.212486117389,0.212478783556,0.212471447632,0.212464109619,0.212456769518,0.212449427334,0.212442083068,0.212434736723,0.212427388302,0.212420037806,0.21241268524,0.212405330605,0.212397973903,0.212390615138,0.212383254312,0.212375891427,0.212368526487,0.212361159492,0.212353790447,0.212346419354,0.212339046215,0.212331671032,0.212324293809,0.212316914548,0.212309533251,0.212302149921,0.21229476456,0.212287377171,0.212279987756,0.212272596319,0.212265202861,0.212257807385,0.212250409893,0.212243010389,0.212235608874,0.212228205351,0.212220799822,0.212213392291,0.212205982759,0.212198571229,0.212191157704,0.212183742186,0.212176324677,0.21216890518,0.212161483698,0.212154060233,0.212146634787,0.212139207363,0.212131777964,0.212124346591,0.212116913248,0.212109477937,0.212102040661,0.212094601421,0.21208716022,0.212079717061,0.212072271946,0.212064824878,0.212057375859,0.212049924892,0.212042471978,0.212035017122,0.212027560324,0.212020101587,0.212012640915,0.212005178308,0.211997713771,0.211990247304,0.211982778911,0.211975308595,0.211967836356,0.211960362199,0.211952886125,0.211945408137,0.211937928237,0.211930446427,0.211922962711,0.21191547709,0.211907989567,0.211900500144,0.211893008824,0.211885515609,0.211878020502,0.211870523505,0.21186302462,0.211855523849,0.211848021196,0.211840516663,0.211833010251,0.211825501964,0.211817991804,0.211810479772,0.211802965873,0.211795450107,0.211787932477,0.211780412986,0.211772891637,0.21176536843,0.21175784337,0.211750316458,0.211742787696,0.211735257087,0.211727724634,0.211720190338,0.211712654202,0.211705116229,0.211697576421,0.211690034779,0.211682491307,0.211674946007,0.211667398882,0.211659849932,0.211652299162,0.211644746573,0.211637192168,0.211629635948,0.211622077917,0.211614518076,0.211606956429,0.211599392976,0.211591827722,0.211584260667,0.211576691815,0.211569121167,0.211561548726,0.211553974495,0.211546398475,0.211538820669,0.211531241079,0.211523659708,0.211516076557,0.21150849163,0.211500904929,0.211493316455,0.211485726211,0.2114781342,0.211470540423,0.211462944883,0.211455347583,0.211447748524,0.211440147709,0.211432545141,0.211424940821,0.211417334751,0.211409726935,0.211402117374,0.211394506071,0.211386893028,0.211379278246,0.21137166173,0.21136404348,0.211356423499,0.211348801789,0.211341178353,0.211333553193,0.211325926311,0.211318297709,0.21131066739,0.211303035355,0.211295401608,0.211287766151,0.211280128984,0.211272490112,0.211264849536,0.211257207259,0.211249563282,0.211241917607,0.211234270238,0.211226621177,0.211218970425,0.211211317985,0.211203663859,0.211196008049,0.211188350558,0.211180691387,0.21117303054,0.211165368018,0.211157703823,0.211150037958,0.211142370425,0.211134701226,0.211127030363,0.211119357839,0.211111683656,0.211104007816,0.211096330321,0.211088651173,0.211080970375,0.211073287928,0.211065603836,0.2110579181,0.211050230722,0.211042541705,0.21103485105,0.211027158761,0.211019464838,0.211011769286,0.211004072104,0.210996373297,0.210988672865,0.210980970811,0.210973267138,0.210965561847,0.210957854941,0.210950146421,0.210942436291,0.210934724551,0.210927011205,0.210919296255,0.210911579702,0.210903861548,0.210896141797,0.21088842045,0.210880697509,0.210872972977,0.210865246855,0.210857519146,0.210849789852,0.210842058975,0.210834326517,0.21082659248,0.210818856867,0.210811119679,0.210803380919,0.210795640589,0.210787898691,0.210780155228,0.2107724102,0.210764663611,0.210756915462,0.210749165757,0.210741414495,0.210733661681,0.210725907316,0.210718151402,0.210710393941,0.210702634936,0.210694874388,0.2106871123,0.210679348673,0.21067158351,0.210663816814,0.210656048585,0.210648278826,0.21064050754,0.210632734728,0.210624960392,0.210617184535,0.210609407159,0.210601628265,0.210593847856,0.210586065934,0.210578282501,0.210570497559,0.21056271111,0.210554923156,0.2105471337,0.210539342743,0.210531550287,0.210523756335,0.210515960889,0.21050816395,0.210500365521,0.210492565604,0.210484764201,0.210476961314,0.210469156945,0.210461351095,0.210453543768,0.210445734965,0.210437924689,0.21043011294,0.210422299722,0.210414485037,0.210406668885,0.210398851271,0.210391032194,0.210383211659,0.210375389666,0.210367566217,0.210359741316,0.210351914963,0.21034408716,0.210336257911,0.210328427216,0.210320595078,0.2103127615,0.210304926481,0.210297090026,0.210289252136,0.210281412813,0.210273572059,0.210265729876,0.210257886265,0.21025004123,0.210242194772,0.210234346893,0.210226497595,0.21021864688,0.21021079475,0.210202941207,0.210195086253,0.210187229891,0.210179372121,0.210171512946,0.210163652369,0.21015579039,0.210147927013,0.210140062239,0.210132196069,0.210124328507,0.210116459553,0.210108589211,0.210100717481,0.210092844367,0.210084969869,0.21007709399,0.210069216733,0.210061338097,0.210053458087,0.210045576704,0.210037693949,0.210029809825,0.210021924333,0.210014037476,0.210006149256,0.209998259674,0.209990368733,0.209982476435,0.20997458278,0.209966687772,0.209958791413,0.209950893703,0.209942994646,0.209935094243,0.209927192496,0.209919289407,0.209911384978,0.209903479211,0.209895572108,0.209887663671,0.209879753901,0.209871842801,0.209863930373,0.209856016618,0.209848101539,0.209840185137,0.209832267414,0.209824348372,0.209816428014,0.209808506341,0.209800583354,0.209792659057,0.20978473345,0.209776806536,0.209768878317,0.209760948794,0.20975301797,0.209745085846,0.209737152424,0.209729217706,0.209721281695,0.209713344391,0.209705405798,0.209697465916,0.209689524747,0.209681582294,0.209673638559,0.209665693543,0.209657747248,0.209649799676,0.209641850829,0.20963390071,0.209625949318,0.209617996658,0.20961004273,0.209602087536,0.209594131079,0.209586173359,0.20957821438,0.209570254142,0.209562292649,0.209554329901,0.2095463659,0.209538400649,0.209530434148,0.209522466401,0.209514497409,0.209506527174,0.209498555697,0.209490582981,0.209482609027,0.209474633837,0.209466657413,0.209458679757,0.209450700871,0.209442720757,0.209434739415,0.20942675685,0.209418773061,0.209410788051,0.209402801822,0.209394814375,0.209386825713,0.209378835837,0.20937084475,0.209362852452,0.209354858946,0.209346864234,0.209338868317,0.209330871197,0.209322872876,0.209314873356,0.209306872639,0.209298870727,0.20929086762,0.209282863322,0.209274857834,0.209266851158,0.209258843295,0.209250834247,0.209242824017,0.209234812606,0.209226800015,0.209218786247,0.209210771303,0.209202755186,0.209194737897,0.209186719437,0.209178699809,0.209170679014,0.209162657055,0.209154633932,0.209146609648,0.209138584205,0.209130557603,0.209122529846,0.209114500935,0.209106470872,0.209098439658,0.209090407295,0.209082373785,0.20907433913,0.209066303331,0.20905826639,0.20905022831,0.209042189091,0.209034148736,0.209026107247,0.209018064624,0.209010020871,0.209001975988,0.208993929977,0.20898588284,0.20897783458,0.208969785197,0.208961734693,0.208953683071,0.208945630331,0.208937576477,0.208929521508,0.208921465428,0.208913408237,0.208905349939,0.208897290533,0.208889230023,0.208881168409,0.208873105695,0.20886504188,0.208856976967,0.208848910959,0.208840843855,0.208832775659,0.208824706372,0.208816635996,0.208808564532,0.208800491982,0.208792418348,0.208784343631,0.208776267834,0.208768190958,0.208760113005,0.208752033976,0.208743953873,0.208735872698,0.208727790452,0.208719707138,0.208711622757,0.20870353731,0.2086954508,0.208687363228,0.208679274596,0.208671184905,0.208663094157,0.208655002354,0.208646909498,0.20863881559,0.208630720632,0.208622624626,0.208614527573,0.208606429475,0.208598330333,0.208590230151,0.208582128928,0.208574026666,0.208565923369,0.208557819036,0.20854971367,0.208541607273,0.208533499846,0.20852539139,0.208517281908,0.208509171401,0.208501059871,0.20849294732,0.208484833748,0.208476719159,0.208468603553,0.208460486932,0.208452369297,0.208444250652,0.208436130996,0.208428010332,0.208419888661,0.208411765986,0.208403642307,0.208395517627,0.208387391946,0.208379265268,0.208371137592,0.208363008922,0.208354879258,0.208346748602,0.208338616956,0.208330484322,0.208322350701,0.208314216094,0.208306080504,0.208297943932,0.20828980638,0.208281667849,0.208273528341,0.208265387857,0.2082572464,0.208249103971,0.20824096057,0.208232816201,0.208224670865,0.208216524563,0.208208377297,0.208200229068,0.208192079878,0.20818392973,0.208175778623,0.208167626561,0.208159473544,0.208151319574,0.208143164653,0.208135008782,0.208126851964,0.208118694199,0.208110535489,0.208102375836,0.208094215241,0.208086053706,0.208077891233,0.208069727823,0.208061563478,0.2080533982,0.208045231989,0.208037064847,0.208028896777,0.20802072778,0.208012557856,0.208004387009,0.207996215239,0.207988042548,0.207979868938,0.20797169441,0.207963518965,0.207955342606,0.207947165334,0.207938987151,0.207930808057,0.207922628055,0.207914447146,0.207906265332,0.207898082614,0.207889898994,0.207881714474,0.207873529054,0.207865342737,0.207857155525,0.207848967417,0.207840778418,0.207832588526,0.207824397746,0.207816206077,0.207808013521,0.20779982008,0.207791625756,0.20778343055,0.207775234464,0.207767037499,0.207758839656,0.207750640938,0.207742441345,0.20773424088,0.207726039544,0.207717837338,0.207709634264,0.207701430323,0.207693225517,0.207685019848,0.207676813317,0.207668605925,0.207660397675,0.207652188567,0.207643978603,0.207635767785,0.207627556114,0.207619343592,0.20761113022,0.207602916,0.207594700933,0.20758648502,0.207578268264,0.207570050666,0.207561832227,0.207553612949,0.207545392833,0.207537171881,0.207528950094,0.207520727474,0.207512504022,0.20750427974,0.20749605463,0.207487828692,0.207479601929,0.207471374341,0.207463145931,0.207454916699,0.207446686648,0.207438455778,0.207430224092,0.207421991591,0.207413758275,0.207405524148,0.20739728921,0.207389053462,0.207380816906,0.207372579545,0.207364341378,0.207356102408,0.207347862636,0.207339622063,0.207331380692,0.207323138523,0.207314895558,0.207306651798,0.207298407245,0.207290161901,0.207281915766,0.207273668843,0.207265421133,0.207257172636,0.207248923356,0.207240673292,0.207232422448,0.207224170823,0.20721591842,0.207207665239,0.207199411283,0.207191156553,0.207182901051,0.207174644777,0.207166387733,0.207158129921,0.207149871342,0.207141611997,0.207133351889,0.207125091018,0.207116829386,0.207108566994,0.207100303844,0.207092039937,0.207083775275,0.207075509859,0.20706724369,0.20705897677,0.207050709101,0.207042440683,0.207034171518,0.207025901609,0.207017630955,0.207009359558,0.207001087421,0.206992814544,0.206984540928,0.206976266576,0.206967991489,0.206959715667,0.206951439113,0.206943161827,0.206934883812,0.206926605069,0.206918325598,0.206910045402,0.206901764482,0.206893482839,0.206885200475,0.206876917391,0.206868633588,0.206860349068,0.206852063833,0.206843777883,0.206835491221,0.206827203846,0.206818915762,0.206810626969,0.206802337469,0.206794047262,0.206785756352,0.206777464738,0.206769172422,0.206760879406,0.206752585691,0.206744291278,0.206735996169,0.206727700365,0.206719403868,0.206711106678,0.206702808798,0.206694510229,0.206686210971,0.206677911027,0.206669610398,0.206661309085,0.206653007089,0.206644704413,0.206636401056,0.206628097022,0.20661979231,0.206611486922,0.206603180861,0.206594874126,0.20658656672,0.206578258644,0.206569949899,0.206561640486,0.206553330408,0.206545019664,0.206536708257,0.206528396189,0.206520083459,0.20651177007,0.206503456023,0.20649514132,0.206486825961,0.206478509948,0.206470193283,0.206461875966,0.206453557999,0.206445239384,0.206436920122,0.206428600213,0.206420279661,0.206411958464,0.206403636626,0.206395314148,0.20638699103,0.206378667274,0.206370342882,0.206362017855,0.206353692193,0.206345365899,0.206337038974,0.206328711419,0.206320383235,0.206312054424,0.206303724987,0.206295394926,0.206287064241,0.206278732934,0.206270401006,0.206262068459,0.206253735294,0.206245401512,0.206237067115,0.206228732104,0.20622039648,0.206212060244,0.206203723398,0.206195385943,0.206187047881,0.206178709212,0.206170369939,0.206162030062,0.206153689582,0.206145348501,0.206137006821,0.206128664542,0.206120321666,0.206111978194,0.206103634127,0.206095289467,0.206086944215,0.206078598373,0.206070251941,0.20606190492,0.206053557313,0.206045209121,0.206036860344,0.206028510984,0.206020161042,0.20601181052,0.206003459419,0.20599510774,0.205986755484,0.205978402653,0.205970049248,0.20596169527,0.205953340721,0.205944985601,0.205936629912,0.205928273656,0.205919916834,0.205911559446,0.205903201494,0.20589484298,0.205886483904,0.205878124268,0.205869764074,0.205861403321,0.205853042013,0.205844680149,0.205836317732,0.205827954762,0.205819591241,0.20581122717,0.20580286255,0.205794497383,0.205786131669,0.20577776541,0.205769398608,0.205761031262,0.205752663376,0.20574429495,0.205735925985,0.205727556482,0.205719186443,0.205710815869,0.205702444761,0.205694073121,0.205685700949,0.205677328247,0.205668955016,0.205660581258,0.205652206973,0.205643832163,0.205635456829,0.205627080972,0.205618704594,0.205610327696,0.205601950278,0.205593572343,0.205585193891,0.205576814924,0.205568435443,0.205560055449,0.205551674943,0.205543293926,0.205534912401,0.205526530367,0.205518147827,0.205509764781,0.20550138123,0.205492997177,0.205484612621,0.205476227565,0.205467842009,0.205459455955,0.205451069403,0.205442682356,0.205434294814,0.205425906778,0.205417518251,0.205409129232,0.205400739723,0.205392349725,0.20538395924,0.205375568269,0.205367176812,0.205358784872,0.205350392449,0.205341999544,0.205333606159,0.205325212295,0.205316817953,0.205308423134,0.20530002784,0.205291632071,0.205283235829,0.205274839115,0.20526644193,0.205258044276,0.205249646153,0.205241247562,0.205232848506,0.205224448985,0.205216048999,0.205207648552,0.205199247643,0.205190846273,0.205182444445,0.205174042159,0.205165639416,0.205157236217,0.205148832565,0.205140428459,0.205132023901,0.205123618892,0.205115213433,0.205106807526,0.205098401171,0.20508999437,0.205081587125,0.205073179435,0.205064771303,0.205056362728,0.205047953714,0.20503954426,0.205031134369,0.20502272404,0.205014313275,0.205005902076,0.204997490443,0.204989078378,0.204980665882,0.204972252956,0.204963839601,0.204955425818,0.204947011608,0.204938596973,0.204930181914,0.204921766431,0.204913350526,0.204904934201,0.204896517456,0.204888100292,0.20487968271,0.204871264713,0.2048628463,0.204854427473,0.204846008233,0.204837588581,0.204829168518,0.204820748046,0.204812327165,0.204803905878,0.204795484184,0.204787062084,0.204778639581,0.204770216676,0.204761793368,0.20475336966,0.204744945552,0.204736521046,0.204728096143,0.204719670844,0.20471124515,0.204702819062,0.204694392581,0.204685965709,0.204677538446,0.204669110793,0.204660682753,0.204652254325,0.204643825511,0.204635396312,0.204626966729,0.204618536764,0.204610106417,0.204601675689,0.204593244582,0.204584813096,0.204576381234,0.204567948995,0.204559516381,0.204551083393,0.204542650032,0.2045342163,0.204525782196,0.204517347724,0.204508912882,0.204500477674,0.204492042099,0.204483606159,0.204475169854,0.204466733187,0.204458296158,0.204449858768,0.204441421018,0.204432982909,0.204424544443,0.20441610562,0.204407666442,0.20439922691,0.204390787024,0.204382346786,0.204373906197,0.204365465257,0.204357023969,0.204348582333,0.20434014035,0.204331698021,0.204323255348,0.204314812331,0.204306368972,0.204297925271,0.204289481229,0.204281036848,0.204272592129,0.204264147073,0.204255701681,0.204247255953,0.204238809891,0.204230363497,0.204221916771,0.204213469713,0.204205022326,0.20419657461,0.204188126567,0.204179678197,0.204171229501,0.204162780481,0.204154331137,0.204145881471,0.204137431484,0.204128981176,0.204120530549,0.204112079604,0.204103628341,0.204095176763,0.204086724869,0.204078272662,0.204069820141,0.204061367308,0.204052914165,0.204044460711,0.204036006949,0.204027552879,0.204019098503,0.20401064382,0.204002188833,0.203993733542,0.203985277949,0.203976822054,0.203968365858,0.203959909363,0.203951452569,0.203942995478,0.203934538091,0.203926080408,0.203917622431,0.20390916416,0.203900705597,0.203892246743,0.203883787598,0.203875328164,0.203866868442,0.203858408433,0.203849948137,0.203841487556,0.203833026692,0.203824565543,0.203816104113,0.203807642402,0.20379918041,0.20379071814,0.203782255591,0.203773792765,0.203765329664,0.203756866287,0.203748402636,0.203739938712,0.203731474516,0.203723010049,0.203714545312,0.203706080305,0.203697615031,0.20368914949,0.203680683683,0.203672217611,0.203663751274,0.203655284675,0.203646817814,0.203638350691,0.203629883309,0.203621415667,0.203612947768,0.203604479611,0.203596011198,0.20358754253,0.203579073609,0.203570604434,0.203562135006,0.203553665328,0.2035451954,0.203536725222,0.203528254796,0.203519784123,0.203511313204,0.20350284204,0.203494370631,0.203485898979,0.203477427085,0.20346895495,0.203460482574,0.203452009959,0.203443537105,0.203435064014,0.203426590687,0.203418117124,0.203409643326,0.203401169296,0.203392695032,0.203384220537,0.203375745811,0.203367270856,0.203358795672,0.20335032026,0.203341844622,0.203333368758,0.203324892669,0.203316416356,0.20330793982,0.203299463062,0.203290986084,0.203282508885,0.203274031467,0.203265553831,0.203257075978,0.203248597909,0.203240119624,0.203231641125,0.203223162413,0.203214683489,0.203206204353,0.203197725006,0.203189245451,0.203180765686,0.203172285714,0.203163805535,0.20315532515,0.203146844561,0.203138363768,0.203129882772,0.203121401573,0.203112920174,0.203104438575,0.203095956777,0.20308747478,0.203078992586,0.203070510196,0.20306202761,0.20305354483,0.203045061856,0.20303657869,0.203028095332,0.203019611784,0.203011128045,0.203002644118,0.202994160003,0.2029856757,0.202977191212,0.202968706539,0.202960221681,0.20295173664,0.202943251416,0.202934766012,0.202926280426,0.202917794661,0.202909308717,0.202900822596,0.202892336298,0.202883849823,0.202875363174,0.202866876351,0.202858389354,0.202849902185,0.202841414845,0.202832927334,0.202824439654,0.202815951806,0.202807463789,0.202798975606,0.202790487257,0.202781998742,0.202773510064,0.202765021223,0.202756532219,0.202748043054,0.202739553729,0.202731064244,0.2027225746,0.202714084799,0.20270559484,0.202697104726,0.202688614457,0.202680124034,0.202671633457,0.202663142728,0.202654651848,0.202646160817,0.202637669636,0.202629178307,0.20262068683,0.202612195205,0.202603703435,0.20259521152,0.20258671946,0.202578227257,0.202569734911,0.202561242423,0.202552749795,0.202544257027,0.20253576412,0.202527271075,0.202518777892,0.202510284574,0.202501791119,0.202493297531,0.202484803808,0.202476309953,0.202467815965,0.202459321847,0.202450827599,0.202442333221,0.202433838715,0.202425344081,0.20241684932,0.202408354434,0.202399859422,0.202391364287,0.202382869029,0.202374373648,0.202365878145,0.202357382522,0.202348886779,0.202340390918,0.202331894938,0.202323398842,0.202314902629,0.2023064063,0.202297909857,0.2022894133,0.202280916631,0.20227241985,0.202263922957,0.202255425954,0.202246928842,0.202238431621,0.202229934293,0.202221436858,0.202212939316,0.20220444167,0.202195943919,0.202187446065,0.202178948109,0.20217045005,0.202161951891,0.202153453632,0.202144955274,0.202136456817,0.202127958263,0.202119459612,0.202110960866,0.202102462024,0.202093963089,0.20208546406,0.202076964939,0.202068465726,0.202059966422,0.202051467029,0.202042967547,0.202034467976,0.202025968318,0.202017468573,0.202008968743,0.202000468828,0.201991968829,0.201983468746,0.201974968582,0.201966468335,0.201957968008,0.201949467601,0.201940967116,0.201932466551,0.20192396591,0.201915465192,0.201906964398,0.201898463529,0.201889962586,0.20188146157,0.201872960482,0.201864459322,0.201855958091,0.20184745679,0.20183895542,0.201830453982,0.201821952476,0.201813450903,0.201804949265,0.201796447561,0.201787945793,0.201779443962,0.201770942068,0.201762440113,0.201753938096,0.201745436019,0.201736933883,0.201728431688,0.201719929436,0.201711427126,0.201702924761,0.20169442234,0.201685919865,0.201677417335,0.201668914753,0.201660412119,0.201651909434,0.201643406698,0.201634903912,0.201626401078,0.201617898195,0.201609395266,0.201600892289,0.201592389267,0.2015838862,0.201575383089,0.201566879935,0.201558376738,0.2015498735,0.20154137022,0.2015328669,0.201524363542,0.201515860144,0.201507356709,0.201498853237,0.201490349728,0.201481846185,0.201473342606,0.201464838994,0.201456335349,0.201447831672,0.201439327963,0.201430824224,0.201422320454,0.201413816656,0.201405312829,0.201396808975,0.201388305094,0.201379801187,0.201371297255,0.201362793298,0.201354289318,0.201345785315,0.20133728129,0.201328777243,0.201320273176,0.201311769089,0.201303264984,0.20129476086,0.201286256718,0.20127775256,0.201269248387,0.201260744197,0.201252239994,0.201243735777,0.201235231547,0.201226727306,0.201218223052,0.201209718789,0.201201214515,0.201192710233,0.201184205942,0.201175701644,0.20116719734,0.201158693029,0.201150188713,0.201141684393,0.201133180069,0.201124675742,0.201116171413,0.201107667083,0.201099162752,0.201090658421,0.201082154091,0.201073649762,0.201065145436,0.201056641114,0.201048136795,0.20103963248,0.201031128171,0.201022623869,0.201014119573,0.201005615285,0.200997111005,0.200988606734,0.200980102474,0.200971598223,0.200963093985,0.200954589758,0.200946085544,0.200937581344,0.200929077159,0.200920572988,0.200912068833,0.200903564695,0.200895060574,0.200886556471,0.200878052387,0.200869548322,0.200861044278,0.200852540255,0.200844036253,0.200835532274,0.200827028318,0.200818524386,0.200810020479,0.200801516597,0.200793012742,0.200784508913,0.200776005112,0.200767501339,0.200758997595,0.200750493881,0.200741990197,0.200733486545,0.200724982925,0.200716479337,0.200707975783,0.200699472263,0.200690968778,0.200682465328,0.200673961915,0.200665458539,0.2006569552,0.2006484519,0.200639948639,0.200631445418,0.200622942238,0.200614439099,0.200605936002,0.200597432948,0.200588929937,0.200580426971,0.200571924049,0.200563421173,0.200554918343,0.200546415561,0.200537912826,0.20052941014,0.200520907503,0.200512404916,0.200503902379,0.200495399894,0.20048689746,0.20047839508,0.200469892753,0.20046139048,0.200452888262,0.200444386099,0.200435883993,0.200427381943,0.200418879951,0.200410378018,0.200401876144,0.200393374329,0.200384872575,0.200376370882,0.200367869251,0.200359367682,0.200350866177,0.200342364735,0.200333863358,0.200325362047,0.200316860801,0.200308359623,0.200299858511,0.200291357468,0.200282856494,0.200274355589,0.200265854754,0.20025735399,0.200248853297,0.200240352677,0.20023185213,0.200223351656,0.200214851257,0.200206350932,0.200197850684,0.200189350511,0.200180850416,0.200172350398,0.200163850459,0.200155350599,0.200146850818,0.200138351118,0.2001298515,0.200121351963,0.200112852508,0.200104353137,0.200095853849,0.200087354646,0.200078855528,0.200070356496,0.20006185755,0.200053358692,0.200044859921,0.200036361239,0.200027862647,0.200019364144,0.200010865732,0.200002367411,0.199993869182,0.199985371045,0.199976873002,0.199968375052,0.199959877197,0.199951379438,0.199942881774,0.199934384207,0.199925886737,0.199917389365,0.199908892092,0.199900394918,0.199891897843,0.199883400869,0.199874903997,0.199866407226,0.199857910558,0.199849413993,0.199840917532,0.199832421175,0.199823924923,0.199815428778,0.199806932738,0.199798436806,0.199789940982,0.199781445266,0.199772949659,0.199764454162,0.199755958775,0.199747463499,0.199738968335,0.199730473284,0.199721978345,0.19971348352,0.199704988809,0.199696494213,0.199687999733,0.199679505369,0.199671011121,0.199662516992,0.19965402298,0.199645529087,0.199637035314,0.199628541661,0.199620048128,0.199611554717,0.199603061428,0.199594568261,0.199586075218,0.199577582299,0.199569089504,0.199560596835,0.199552104291,0.199543611874,0.199535119583,0.199526627421,0.199518135387,0.199509643482,0.199501151707,0.199492660062,0.199484168547,0.199475677165,0.199467185914,0.199458694797,0.199450203812,0.199441712962,0.199433222247,0.199424731666,0.199416241222,0.199407750915,0.199399260744,0.199390770711,0.199382280817,0.199373791062,0.199365301447,0.199356811971,0.199348322637,0.199339833445,0.199331344394,0.199322855486,0.199314366722,0.199305878102,0.199297389626,0.199288901296,0.199280413112,0.199271925074,0.199263437183,0.19925494944,0.199246461845,0.1992379744,0.199229487104,0.199220999958,0.199212512963,0.199204026119,0.199195539428,0.199187052889,0.199178566503,0.199170080271,0.199161594194,0.199153108271,0.199144622505,0.199136136894,0.199127651441,0.199119166145,0.199110681008,0.199102196028,0.199093711209,0.199085226549,0.19907674205,0.199068257712,0.199059773535,0.199051289521,0.19904280567,0.199034321983,0.19902583846,0.199017355101,0.199008871908,0.19900038888,0.19899190602,0.198983423326,0.1989749408,0.198966458442,0.198957976254,0.198949494235,0.198941012386,0.198932530707,0.1989240492,0.198915567865,0.198907086702,0.198898605713,0.198890124897,0.198881644255,0.198873163789,0.198864683497,0.198856203382,0.198847723443,0.198839243681,0.198830764098,0.198822284692,0.198813805466,0.198805326419,0.198796847552,0.198788368866,0.198779890361,0.198771412039,0.198762933898,0.198754455941,0.198745978167,0.198737500577,0.198729023173,0.198720545953,0.19871206892,0.198703592073,0.198695115413,0.198686638941,0.198678162657,0.198669686561,0.198661210656,0.19865273494,0.198644259415,0.198635784081,0.198627308938,0.198618833988,0.198610359231,0.198601884667,0.198593410297,0.198584936122,0.198576462141,0.198567988357,0.198559514769,0.198551041377,0.198542568183,0.198534095187,0.198525622389,0.198517149791,0.198508677392,0.198500205194,0.198491733196,0.198483261399,0.198474789805,0.198466318413,0.198457847224,0.198449376238,0.198440905457,0.198432434881,0.19842396451,0.198415494344,0.198407024385,0.198398554633,0.198390085089,0.198381615752,0.198373146625,0.198364677706,0.198356208997,0.198347740499,0.198339272211,0.198330804135,0.19832233627,0.198313868618,0.19830540118,0.198296933955,0.198288466944,0.198280000147,0.198271533566,0.198263067201,0.198254601052,0.198246135121,0.198237669406,0.19822920391,0.198220738632,0.198212273573,0.198203808734,0.198195344115,0.198186879717,0.19817841554,0.198169951585,0.198161487853,0.198153024343,0.198144561057,0.198136097994,0.198127635157,0.198119172544,0.198110710157,0.198102247996,0.198093786062,0.198085324355,0.198076862876,0.198068401625,0.198059940603,0.19805147981,0.198043019248,0.198034558915,0.198026098814,0.198017638944,0.198009179307,0.198000719902,0.19799226073,0.197983801791,0.197975343087,0.197966884618,0.197958426384,0.197949968385,0.197941510623,0.197933053098,0.19792459581,0.19791613876,0.197907681948,0.197899225376,0.197890769043,0.19788231295,0.197873857097,0.197865401485,0.197856946115,0.197848490988,0.197840036102,0.19783158146,0.197823127062,0.197814672907,0.197806218998,0.197797765333,0.197789311915,0.197780858743,0.197772405817,0.197763953139,0.197755500708,0.197747048526,0.197738596593,0.197730144909,0.197721693475,0.197713242292,0.197704791359,0.197696340678,0.197687890249,0.197679440072,0.197670990148,0.197662540478,0.197654091061,0.1976456419,0.197637192993,0.197628744341,0.197620295946,0.197611847808,0.197603399926,0.197594952302,0.197586504936,0.197578057829,0.19756961098,0.197561164392,0.197552718063,0.197544271995,0.197535826188,0.197527380643,0.19751893536,0.197510490339,0.197502045582,0.197493601088,0.197485156858,0.197476712893,0.197468269194,0.197459825759,0.197451382591,0.19744293969,0.197434497055,0.197426054689,0.19741761259,0.19740917076,0.197400729199,0.197392287908,0.197383846887,0.197375406137,0.197366965657,0.197358525449,0.197350085514,0.197341645851,0.197333206461,0.197324767344,0.197316328502,0.197307889934,0.197299451641,0.197291013624,0.197282575883,0.197274138418,0.19726570123,0.19725726432,0.197248827688,0.197240391334,0.197231955259,0.197223519464,0.197215083948,0.197206648713,0.197198213759,0.197189779086,0.197181344695,0.197172910587,0.197164476761,0.197156043218,0.19714760996,0.197139176985,0.197130744296,0.197122311892,0.197113879773,0.19710544794,0.197097016395,0.197088585136,0.197080154165,0.197071723482,0.197063293088,0.197054862983,0.197046433167,0.197038003642,0.197029574407,0.197021145462,0.19701271681,0.197004288449,0.196995860381,0.196987432605,0.196979005123,0.196970577935,0.19696215104,0.196953724441,0.196945298137,0.196936872129,0.196928446416,0.196920021001,0.196911595882,0.196903171061,0.196894746538,0.196886322314,0.196877898388,0.196869474762,0.196861051436,0.196852628411,0.196844205686,0.196835783262,0.19682736114,0.196818939321,0.196810517804,0.19680209659,0.19679367568,0.196785255074,0.196776834772,0.196768414776,0.196759995085,0.196751575699,0.196743156621,0.196734737849,0.196726319384,0.196717901227,0.196709483378,0.196701065838,0.196692648608,0.196684231686,0.196675815075,0.196667398774,0.196658982784,0.196650567106,0.196642151739,0.196633736685,0.196625321943,0.196616907515,0.1966084934,0.1966000796,0.196591666114,0.196583252943,0.196574840087,0.196566427548,0.196558015325,0.196549603418,0.196541191829,0.196532780558,0.196524369604,0.19651595897,0.196507548654,0.196499138658,0.196490728982,0.196482319626,0.196473910591,0.196465501877,0.196457093485,0.196448685416,0.196440277669,0.196431870244,0.196423463144,0.196415056367,0.196406649915,0.196398243787,0.196389837985,0.196381432508,0.196373027358,0.196364622534,0.196356218037,0.196347813868,0.196339410026,0.196331006513,0.196322603329,0.196314200473,0.196305797948,0.196297395752,0.196288993887,0.196280592353,0.19627219115,0.196263790279,0.196255389741,0.196246989535,0.196238589662,0.196230190122,0.196221790917,0.196213392046,0.19620499351,0.196196595309,0.196188197443,0.196179799914,0.196171402722,0.196163005866,0.196154609348,0.196146213167,0.196137817325,0.196129421821,0.196121026657,0.196112631832,0.196104237347,0.196095843203,0.196087449399,0.196079055936,0.196070662816,0.196062270037,0.196053877601,0.196045485507,0.196037093757,0.196028702351,0.196020311289,0.196011920571,0.196003530199,0.195995140172,0.195986750491,0.195978361156,0.195969972167,0.195961583526,0.195953195233,0.195944807287,0.19593641969,0.195928032442,0.195919645542,0.195911258993,0.195902872793,0.195894486944,0.195886101445,0.195877716298,0.195869331503,0.19586094706,0.195852562969,0.195844179231,0.195835795846,0.195827412816,0.195819030139,0.195810647817,0.19580226585,0.195793884238,0.195785502982,0.195777122082,0.195768741539,0.195760361353,0.195751981524,0.195743602053,0.195735222941,0.195726844187,0.195718465792,0.195710087756,0.19570171008,0.195693332765,0.19568495581,0.195676579216,0.195668202984,0.195659827114,0.195651451605,0.19564307646,0.195634701677,0.195626327259,0.195617953204,0.195609579513,0.195601206187,0.195592833226,0.19558446063,0.1955760884,0.195567716537,0.19555934504,0.19555097391,0.195542603148,0.195534232754,0.195525862728,0.19551749307,0.195509123782,0.195500754862,0.195492386313,0.195484018134,0.195475650326,0.195467282889,0.195458915823,0.195450549128,0.195442182806,0.195433816857,0.19542545128,0.195417086077,0.195408721248,0.195400356793,0.195391992712,0.195383629006,0.195375265676,0.195366902721,0.195358540142,0.19535017794,0.195341816115,0.195333454667,0.195325093596,0.195316732903,0.195308372589,0.195300012654,0.195291653098,0.195283293921,0.195274935124,0.195266576708,0.195258218672,0.195249861018,0.195241503745,0.195233146853,0.195224790344,0.195216434218,0.195208078474,0.195199723114,0.195191368138,0.195183013545,0.195174659337,0.195166305515,0.195157952077,0.195149599025,0.195141246359,0.195132894079,0.195124542187,0.195116190681,0.195107839563,0.195099488832,0.19509113849,0.195082788537,0.195074438973,0.195066089798,0.195057741012,0.195049392617,0.195041044613,0.195032696999,0.195024349777,0.195016002946,0.195007656507,0.19499931046,0.194990964807,0.194982619546,0.194974274679,0.194965930206,0.194957586127,0.194949242442,0.194940899152,0.194932556258,0.19492421376,0.194915871657,0.194907529951,0.194899188641,0.194890847729,0.194882507215,0.194874167098,0.194865827379,0.194857488059,0.194849149138,0.194840810616,0.194832472494,0.194824134772,0.194815797451,0.19480746053,0.19479912401,0.194790787892,0.194782452176,0.194774116862,0.19476578195,0.194757447442,0.194749113337,0.194740779635,0.194732446337,0.194724113444,0.194715780956,0.194707448872,0.194699117194,0.194690785922,0.194682455056,0.194674124596,0.194665794544,0.194657464898,0.19464913566,0.19464080683,0.194632478409,0.194624150396,0.194615822792,0.194607495597,0.194599168812,0.194590842437,0.194582516472,0.194574190918,0.194565865775,0.194557541044,0.194549216724,0.194540892817,0.194532569322,0.19452424624,0.194515923571,0.194507601316,0.194499279474,0.194490958047,0.194482637034,0.194474316437,0.194465996254,0.194457676487,0.194449357136,0.194441038202,0.194432719684,0.194424401583,0.194416083899,0.194407766633,0.194399449785,0.194391133356,0.194382817345,0.194374501753,0.19436618658,0.194357871828,0.194349557495,0.194341243583,0.194332930091,0.194324617021,0.194316304372,0.194307992145,0.19429968034,0.194291368957,0.194283057998,0.194274747461,0.194266437348,0.194258127658,0.194249818393,0.194241509552,0.194233201136,0.194224893145,0.19421658558,0.19420827844,0.194199971727,0.19419166544,0.194183359579,0.194175054146,0.194166749141,0.194158444563,0.194150140413,0.194141836692,0.1941335334,0.194125230537,0.194116928103,0.194108626099,0.194100324525,0.194092023382,0.19408372267,0.194075422388,0.194067122538,0.19405882312,0.194050524134,0.19404222558,0.194033927459,0.194025629771,0.194017332517,0.194009035696,0.194000739309,0.193992443357,0.19398414784,0.193975852757,0.19396755811,0.193959263898,0.193950970123,0.193942676784,0.193934383881,0.193926091416,0.193917799387,0.193909507797,0.193901216644,0.193892925929,0.193884635654,0.193876345817,0.193868056419,0.193859767461,0.193851478942,0.193843190864,0.193834903226,0.19382661603,0.193818329274,0.19381004296,0.193801757087,0.193793471657,0.193785186669,0.193776902123,0.193768618021,0.193760334362,0.193752051147,0.193743768376,0.193735486049,0.193727204166,0.193718922729,0.193710641736,0.19370236119,0.193694081089,0.193685801434,0.193677522226,0.193669243465,0.19366096515,0.193652687284,0.193644409864,0.193636132893,0.193627856371,0.193619580297,0.193611304671,0.193603029496,0.193594754769,0.193586480493,0.193578206667,0.193569933291,0.193561660366,0.193553387893,0.19354511587,0.1935368443,0.193528573181,0.193520302515,0.193512032301,0.193503762541,0.193495493234,0.19348722438,0.19347895598,0.193470688034,0.193462420543,0.193454153507,0.193445886925,0.193437620799,0.193429355129,0.193421089915,0.193412825157,0.193404560855,0.193396297011,0.193388033623,0.193379770693,0.193371508221,0.193363246207,0.193354984652,0.193346723555,0.193338462917,0.193330202738,0.193321943019,0.19331368376,0.193305424961,0.193297166622,0.193288908745,0.193280651328,0.193272394373,0.193264137879,0.193255881847,0.193247626277,0.19323937117,0.193231116526,0.193222862345,0.193214608628,0.193206355374,0.193198102584,0.193189850258,0.193181598397,0.193173347001,0.19316509607,0.193156845604,0.193148595605,0.193140346071,0.193132097003,0.193123848403,0.193115600269,0.193107352602,0.193099105403,0.193090858671,0.193082612408,0.193074366613,0.193066121286,0.193057876429,0.193049632041,0.193041388122,0.193033144673,0.193024901694,0.193016659185,0.193008417148,0.193000175581,0.192991934485,0.19298369386,0.192975453708,0.192967214027,0.192958974819,0.192950736084,0.192942497821,0.192934260032,0.192926022716,0.192917785874,0.192909549506,0.192901313612,0.192893078193,0.192884843248,0.192876608779,0.192868374785,0.192860141267,0.192851908225,0.19284367566,0.19283544357,0.192827211958,0.192818980823,0.192810750165,0.192802519985,0.192794290282,0.192786061058,0.192777832313,0.192769604046,0.192761376258,0.19275314895,0.192744922121,0.192736695772,0.192728469903,0.192720244515,0.192712019607,0.19270379518,0.192695571235,0.192687347771,0.192679124789,0.192670902288,0.192662680271,0.192654458735,0.192646237683,0.192638017114,0.192629797028,0.192621577426,0.192613358307,0.192605139673,0.192596921524,0.192588703859,0.19258048668,0.192572269985,0.192564053776,0.192555838053,0.192547622816,0.192539408066,0.192531193802,0.192522980025,0.192514766735,0.192506553933,0.192498341618,0.192490129791,0.192481918453,0.192473707603,0.192465497241,0.192457287369,0.192449077986,0.192440869093,0.192432660689,0.192424452776,0.192416245352,0.19240803842,0.192399831978,0.192391626027,0.192383420568,0.192375215601,0.192367011125,0.192358807142,0.19235060365,0.192342400652,0.192334198147,0.192325996134,0.192317794616,0.192309593591,0.19230139306,0.192293193023,0.19228499348,0.192276794433,0.19226859588,0.192260397823,0.192252200261,0.192244003195,0.192235806625,0.192227610552,0.192219414975,0.192211219894,0.192203025311,0.192194831225,0.192186637637,0.192178444547,0.192170251954,0.19216205986,0.192153868265,0.192145677168,0.19213748657,0.192129296472,0.192121106873,0.192112917775,0.192104729176,0.192096541077,0.19208835348,0.192080166383,0.192071979787,0.192063793693,0.1920556081,0.192047423009,0.19203923842,0.192031054333,0.192022870749,0.192014687668,0.19200650509,0.191998323015,0.191990141444,0.191981960377,0.191973779814,0.191965599755,0.1919574202,0.191949241151,0.191941062606,0.191932884567,0.191924707034,0.191916530006,0.191908353484,0.191900177469,0.19189200196,0.191883826957,0.191875652462,0.191867478474,0.191859304994,0.191851132021,0.191842959557,0.1918347876,0.191826616152,0.191818445213,0.191810274782,0.191802104861,0.19179393545,0.191785766548,0.191777598155,0.191769430274,0.191761262902,0.191753096041,0.191744929691,0.191736763852,0.191728598524,0.191720433708,0.191712269404,0.191704105612,0.191695942332,0.191687779565,0.19167961731,0.191671455568,0.19166329434,0.191655133625,0.191646973424,0.191638813737,0.191630654564,0.191622495905,0.191614337761,0.191606180132,0.191598023018,0.191589866419,0.191581710336,0.191573554769,0.191565399718,0.191557245183,0.191549091165,0.191540937663,0.191532784678,0.191524632211,0.191516480261,0.191508328828,0.191500177914,0.191492027517,0.191483877639,0.19147572828,0.191467579439,0.191459431118,0.191451283316,0.191443136033,0.19143498927,0.191426843027,0.191418697304,0.191410552101,0.191402407419,0.191394263258,0.191386119618,0.1913779765,0.191369833903,0.191361691828,0.191353550274,0.191345409243,0.191337268735,0.191329128749,0.191320989286,0.191312850346,0.191304711929,0.191296574036,0.191288436667,0.191280299822,0.191272163501,0.191264027704,0.191255892432,0.191247757685,0.191239623463,0.191231489767,0.191223356596,0.191215223951,0.191207091832,0.191198960239,0.191190829172,0.191182698632,0.19117456862,0.191166439134,0.191158310175,0.191150181744,0.191142053841,0.191133926466,0.191125799619,0.1911176733,0.19110954751,0.191101422249,0.191093297517,0.191085173314,0.191077049641,0.191068926497,0.191060803884,0.1910526818,0.191044560247,0.191036439225,0.191028318733,0.191020198772,0.191012079342,0.191003960444,0.190995842078,0.190987724243,0.190979606941,0.19097149017,0.190963373933,0.190955258228,0.190947143056,0.190939028417,0.190930914311,0.190922800739,0.1909146877,0.190906575196,0.190898463226,0.19089035179,0.190882240889,0.190874130523,0.190866020691,0.190857911395,0.190849802635,0.19084169441,0.190833586721,0.190825479568,0.190817372951,0.190809266871,0.190801161327,0.19079305632,0.190784951851,0.190776847919,0.190768744524,0.190760641667,0.190752539348,0.190744437567,0.190736336324,0.19072823562,0.190720135455,0.190712035829,0.190703936742,0.190695838194,0.190687740186,0.190679642717,0.190671545789,0.190663449401,0.190655353553,0.190647258245,0.190639163479,0.190631069253,0.190622975569,0.190614882426,0.190606789825,0.190598697765,0.190590606248,0.190582515272,0.19057442484,0.190566334949,0.190558245602,0.190550156797,0.190542068536,0.190533980818,0.190525893644,0.190517807013,0.190509720927,0.190501635384,0.190493550386,0.190485465933,0.190477382024,0.190469298661,0.190461215842,0.190453133569,0.190445051842,0.19043697066,0.190428890024,0.190420809935,0.190412730391,0.190404651395,0.190396572945,0.190388495042,0.190380417686,0.190372340877,0.190364264616,0.190356188903,0.190348113738,0.19034003912,0.190331965051,0.190323891531,0.190315818559,0.190307746136,0.190299674262,0.190291602937,0.190283532162,0.190275461937,0.190267392261,0.190259323135,0.19025125456,0.190243186535,0.19023511906,0.190227052136,0.190218985764,0.190210919942,0.190202854672,0.190194789953,0.190186725786,0.190178662171,0.190170599108,0.190162536597,0.190154474639,0.190146413234,0.190138352381,0.190130292081,0.190122232335,0.190114173142,0.190106114502,0.190098056417,0.190089998885,0.190081941908,0.190073885485,0.190065829616,0.190057774302,0.190049719543,0.190041665339,0.19003361169,0.190025558597,0.190017506059,0.190009454078,0.190001402652,0.189993351782,0.189985301469,0.189977251712,0.189969202512,0.189961153869,0.189953105783,0.189945058254,0.189937011283,0.18992896487,0.189920919014,0.189912873716,0.189904828976,0.189896784795,0.189888741172,0.189880698108,0.189872655603,0.189864613657,0.18985657227,0.189848531443,0.189840491175,0.189832451467,0.189824412319,0.189816373731,0.189808335703,0.189800298236,0.18979226133,0.189784224984,0.1897761892,0.189768153977,0.189760119315,0.189752085214,0.189744051676,0.189736018699,0.189727986284,0.189719954432,0.189711923142,0.189703892415,0.18969586225,0.189687832649,0.18967980361,0.189671775135,0.189663747223,0.189655719875,0.189647693091,0.189639666871,0.189631641215,0.189623616123,0.189615591596,0.189607567633,0.189599544235,0.189591521403,0.189583499135,0.189575477433,0.189567456297,0.189559435726,0.189551415721,0.189543396282,0.189535377409,0.189527359103,0.189519341363,0.18951132419,0.189503307584,0.189495291545,0.189487276073,0.189479261169,0.189471246832,0.189463233063,0.189455219862,0.189447207229,0.189439195164,0.189431183667,0.189423172739,0.18941516238,0.18940715259,0.189399143369,0.189391134717,0.189383126634,0.189375119121,0.189367112178,0.189359105804,0.189351100001,0.189343094768,0.189335090105,0.189327086013,0.189319082492,0.189311079541,0.189303077162,0.189295075353,0.189287074116,0.189279073451,0.189271073357,0.189263073836,0.189255074886,0.189247076508,0.189239078703,0.18923108147,0.18922308481,0.189215088723,0.189207093208,0.189199098267,0.189191103899,0.189183110105,0.189175116884,0.189167124237,0.189159132164,0.189151140666,0.189143149741,0.189135159391,0.189127169615,0.189119180414,0.189111191788,0.189103203737,0.189095216262,0.189087229361,0.189079243036,0.189071257287,0.189063272114,0.189055287517,0.189047303496,0.189039320051,0.189031337183,0.189023354891,0.189015373176,0.189007392038,0.188999411477,0.188991431493,0.188983452087,0.188975473258,0.188967495007,0.188959517334,0.188951540239,0.188943563722,0.188935587784,0.188927612424,0.188919637642,0.188911663439,0.188903689816,0.188895716771,0.188887744305,0.188879772419,0.188871801113,0.188863830386,0.188855860239,0.188847890672,0.188839921685,0.188831953279,0.188823985453,0.188816018207,0.188808051542,0.188800085459,0.188792119956,0.188784155034,0.188776190694,0.188768226935,0.188760263758,0.188752301163,0.18874433915,0.188736377718,0.188728416869,0.188720456603,0.188712496919,0.188704537817,0.188696579299,0.188688621363,0.188680664011,0.188672707241,0.188664751056,0.188656795453,0.188648840435,0.188640886,0.18863293215,0.188624978883,0.188617026201,0.188609074103,0.18860112259,0.188593171661,0.188585221318,0.188577271559,0.188569322386,0.188561373798,0.188553425795,0.188545478378,0.188537531547,0.188529585301,0.188521639642,0.188513694568,0.188505750081,0.188497806181,0.188489862867,0.18848192014,0.188473977999,0.188466036446,0.18845809548,0.188450155101,0.188442215309,0.188434276105,0.188426337489,0.188418399461,0.188410462021,0.188402525168,0.188394588904,0.188386653229,0.188378718142,0.188370783644,0.188362849734,0.188354916414,0.188346983682,0.18833905154,0.188331119988,0.188323189024,0.188315258651,0.188307328867,0.188299399673,0.188291471069,0.188283543056,0.188275615633,0.1882676888,0.188259762558,0.188251836906,0.188243911846,0.188235987377,0.188228063498,0.188220140211,0.188212217516,0.188204295412,0.188196373899,0.188188452979,0.18818053265,0.188172612914,0.18816469377,0.188156775218,0.188148857259,0.188140939892,0.188133023118,0.188125106937,0.188117191349,0.188109276354,0.188101361953,0.188093448145,0.18808553493,0.18807762231,0.188069710283,0.18806179885,0.188053888011,0.188045977766,0.188038068116,0.18803015906,0.188022250598,0.188014342732,0.18800643546,0.187998528784,0.187990622702,0.187982717216,0.187974812325,0.18796690803,0.18795900433,0.187951101226,0.187943198718,0.187935296806,0.18792739549,0.18791949477,0.187911594647,0.187903695121,0.187895796191,0.187887897858,0.187880000122,0.187872102982,0.18786420644,0.187856310496,0.187848415149,0.187840520399,0.187832626247,0.187824732693,0.187816839737,0.187808947378,0.187801055618,0.187793164457,0.187785273894,0.187777383929,0.187769494563,0.187761605796,0.187753717627,0.187745830058,0.187737943088,0.187730056717,0.187722170946,0.187714285774,0.187706401202,0.18769851723,0.187690633858,0.187682751085,0.187674868913,0.187666987341,0.18765910637,0.187651225999,0.187643346229,0.187635467059,0.18762758849,0.187619710523,0.187611833156,0.187603956391,0.187596080227,0.187588204665,0.187580329704,0.187572455345,0.187564581588,0.187556708433,0.18754883588,0.187540963929,0.18753309258,0.187525221834,0.187517351691,0.18750948215,0.187501613212,0.187493744876,0.187485877144,0.187478010015,0.18747014349,0.187462277567,0.187454412249,0.187446547533,0.187438683422,0.187430819915,0.187422957011,0.187415094711,0.187407233016,0.187399371925,0.187391511439,0.187383651557,0.187375792279,0.187367933607,0.187360075539,0.187352218077,0.187344361219,0.187336504967,0.18732864932,0.187320794279,0.187312939843,0.187305086013,0.187297232788,0.18728938017,0.187281528158,0.187273676751,0.187265825952,0.187257975758,0.187250126171,0.18724227719,0.187234428817,0.18722658105,0.18721873389,0.187210887337,0.187203041391,0.187195196053,0.187187351321,0.187179507198,0.187171663682,0.187163820773,0.187155978473,0.18714813678,0.187140295696,0.18713245522,0.187124615351,0.187116776092,0.187108937441,0.187101099398,0.187093261964,0.187085425139,0.187077588923,0.187069753315,0.187061918317,0.187054083929,0.187046250149,0.187038416979,0.187030584419,0.187022752468,0.187014921127,0.187007090396,0.186999260275,0.186991430764,0.186983601863,0.186975773573,0.186967945893,0.186960118823,0.186952292364,0.186944466516,0.186936641279,0.186928816652,0.186920992637,0.186913169233,0.18690534644,0.186897524259,0.186889702689,0.18688188173,0.186874061384,0.186866241649,0.186858422526,0.186850604014,0.186842786115,0.186834968829,0.186827152154,0.186819336092,0.186811520643,0.186803705806,0.186795891581,0.18678807797,0.186780264972,0.186772452586,0.186764640814,0.186756829655,0.186749019109,0.186741209177,0.186733399858,0.186725591153,0.186717783061,0.186709975584,0.18670216872,0.18669436247,0.186686556835,0.186678751813,0.186670947406,0.186663143614,0.186655340436,0.186647537873,0.186639735924,0.18663193459,0.186624133871,0.186616333768,0.186608534279,0.186600735405,0.186592937147,0.186585139504,0.186577342477,0.186569546066,0.18656175027,0.18655395509,0.186546160526,0.186538366577,0.186530573245,0.18652278053,0.18651498843,0.186507196947,0.18649940608,0.18649161583,0.186483826197,0.18647603718,0.18646824878,0.186460460998,0.186452673832,0.186444887283,0.186437101352,0.186429316038,0.186421531342,0.186413747263,0.186405963802,0.186398180958,0.186390398732,0.186382617124,0.186374836134,0.186367055763,0.186359276009,0.186351496874,0.186343718357,0.186335940459,0.186328163179,0.186320386518,0.186312610475,0.186304835052,0.186297060247,0.186289286061,0.186281512495,0.186273739548,0.18626596722,0.186258195511,0.186250424422,0.186242653953,0.186234884103,0.186227114873,0.186219346263,0.186211578272,0.186203810902,0.186196044152,0.186188278022,0.186180512512,0.186172747623,0.186164983354,0.186157219706,0.186149456679,0.186141694272,0.186133932486,0.186126171321,0.186118410777,0.186110650854,0.186102891553,0.186095132872,0.186087374813,0.186079617376,0.186071860559,0.186064104365,0.186056348792,0.186048593841,0.186040839512,0.186033085805,0.18602533272,0.186017580258,0.186009828417,0.186002077199,0.185994326603,0.18598657663,0.185978827279,0.185971078551,0.185963330445,0.185955582963,0.185947836103,0.185940089867,0.185932344253,0.185924599263,0.185916854896,0.185909111152,0.185901368032,0.185893625535,0.185885883662,0.185878142413,0.185870401787,0.185862661785,0.185854922407,0.185847183653,0.185839445524,0.185831708018,0.185823971137,0.18581623488,0.185808499247,0.185800764239,0.185793029856,0.185785296097,0.185777562963,0.185769830454,0.185762098569,0.18575436731,0.185746636676,0.185738906667,0.185731177283,0.185723448525,0.185715720392,0.185707992884,0.185700266002,0.185692539746,0.185684814116,0.185677089111,0.185669364732,0.185661640979,0.185653917852,0.185646195352,0.185638473477,0.185630752229,0.185623031607,0.185615311612,0.185607592243,0.1855998735,0.185592155385,0.185584437896,0.185576721034,0.185569004799,0.185561289191,0.18555357421,0.185545859856,0.185538146129,0.18553043303,0.185522720558,0.185515008713,0.185507297496,0.185499586907,0.185491876945,0.185484167611,0.185476458905,0.185468750827,0.185461043377,0.185453336555,0.185445630361,0.185437924795,0.185430219858,0.185422515549,0.185414811868,0.185407108816,0.185399406392,0.185391704597,0.185384003431,0.185376302894,0.185368602985,0.185360903706,0.185353205056,0.185345507034,0.185337809642,0.185330112879,0.185322416746,0.185314721242,0.185307026367,0.185299332122,0.185291638506,0.18528394552,0.185276253164,0.185268561438,0.185260870342,0.185253179876,0.18524549004,0.185237800833,0.185230112258,0.185222424312,0.185214736997,0.185207050312,0.185199364257,0.185191678834,0.18518399404,0.185176309878,0.185168626346,0.185160943445,0.185153261175,0.185145579536,0.185137898528,0.185130218152,0.185122538406,0.185114859292,0.185107180809,0.185099502957,0.185091825737,0.185084149148,0.185076473191,0.185068797866,0.185061123172,0.18505344911,0.18504577568,0.185038102882,0.185030430716,0.185022759182,0.185015088281,0.185007418011,0.184999748374,0.184992079369,0.184984410997,0.184976743257,0.184969076149,0.184961409674,0.184953743832,0.184946078623,0.184938414046,0.184930750103,0.184923086792,0.184915424114,0.18490776207,0.184900100658,0.18489243988,0.184884779735,0.184877120223,0.184869461345,0.184861803101,0.184854145489,0.184846488512,0.184838832168,0.184831176458,0.184823521382,0.184815866939,0.18480821313,0.184800559956,0.184792907415,0.184785255509,0.184777604237,0.184769953599,0.184762303595,0.184754654226,0.184747005491,0.184739357391,0.184731709925,0.184724063094,0.184716416898,0.184708771336,0.184701126409,0.184693482117,0.18468583846,0.184678195438,0.184670553051,0.184662911299,0.184655270182,0.184647629701,0.184639989855,0.184632350644,0.184624712069,0.184617074129,0.184609436824,0.184601800156,0.184594164123,0.184586528725,0.184578893964,0.184571259838,0.184563626348,0.184555993495,0.184548361277,0.184540729695,0.18453309875,0.184525468441,0.184517838768,0.184510209731,0.184502581331,0.184494953567,0.184487326439,0.184479699949,0.184472074094,0.184464448877,0.184456824296,0.184449200352,0.184441577045,0.184433954375,0.184426332342,0.184418710946,0.184411090186,0.184403470065,0.18439585058,0.184388231732,0.184380613522,0.184372995949,0.184365379014,0.184357762716,0.184350147056,0.184342532033,0.184334917648,0.1843273039,0.18431969079,0.184312078319,0.184304466485,0.184296855289,0.184289244731,0.184281634811,0.184274025529,0.184266416885,0.184258808879,0.184251201512,0.184243594783,0.184235988692,0.18422838324,0.184220778426,0.184213174251,0.184205570715,0.184197967817,0.184190365557,0.184182763937,0.184175162955,0.184167562612,0.184159962908,0.184152363843,0.184144765417,0.18413716763,0.184129570482,0.184121973974,0.184114378104,0.184106782874,0.184099188283,0.184091594332,0.18408400102,0.184076408347,0.184068816314,0.184061224921,0.184053634167,0.184046044053,0.184038454579,0.184030865744,0.18402327755,0.184015689995,0.18400810308,0.184000516805,0.18399293117,0.183985346175,0.183977761821,0.183970178107,0.183962595032,0.183955012599,0.183947430805,0.183939849652,0.183932269139,0.183924689267,0.183917110036,0.183909531445,0.183901953494,0.183894376185,0.183886799516,0.183879223488,0.1838716481,0.183864073354,0.183856499248,0.183848925784,0.18384135296,0.183833780778,0.183826209237,0.183818638337,0.183811068078,0.18380349846,0.183795929484,0.183788361149,0.183780793456,0.183773226404,0.183765659993,0.183758094224,0.183750529097,0.183742964611,0.183735400767,0.183727837565,0.183720275004,0.183712713086,0.183705151809,0.183697591174,0.183690031181,0.18368247183,0.183674913121,0.183667355055,0.18365979763,0.183652240848,0.183644684708,0.18363712921,0.183629574354,0.183622020141,0.183614466571,0.183606913642,0.183599361357,0.183591809714,0.183584258713,0.183576708355,0.18356915864,0.183561609567,0.183554061138,0.183546513351,0.183538966207,0.183531419705,0.183523873847,0.183516328632,0.18350878406,0.183501240131,0.183493696845,0.183486154202,0.183478612202,0.183471070846,0.183463530133,0.183455990063,0.183448450637,0.183440911854,0.183433373714,0.183425836219,0.183418299366,0.183410763157,0.183403227592,0.18339569267,0.183388158393,0.183380624758,0.183373091768,0.183365559422,0.183358027719,0.18335049666,0.183342966245,0.183335436475,0.183327907348,0.183320378865,0.183312851027,0.183305323832,0.183297797282,0.183290271376,0.183282746114,0.183275221497,0.183267697524,0.183260174195,0.183252651511,0.183245129471,0.183237608076,0.183230087325,0.183222567219,0.183215047758,0.183207528941,0.183200010769,0.183192493241,0.183184976359,0.183177460121,0.183169944528,0.18316242958,0.183154915277,0.183147401619,0.183139888606,0.183132376237,0.183124864514,0.183117353436,0.183109843004,0.183102333216,0.183094824074,0.183087315577,0.183079807725,0.183072300518,0.183064793957,0.183057288042,0.183049782771,0.183042278147,0.183034774167,0.183027270834,0.183019768146,0.183012266103,0.183004764706,0.182997263955,0.18298976385,0.18298226439,0.182974765577,0.182967267409,0.182959769887,0.18295227301,0.18294477678,0.182937281196,0.182929786258,0.182922291965,0.182914798319,0.182907305319,0.182899812965,0.182892321258,0.182884830196,0.182877339781,0.182869850012,0.182862360889,0.182854872413,0.182847384583,0.1828398974,0.182832410863,0.182824924972,0.182817439728,0.182809955131,0.18280247118,0.182794987875,0.182787505218,0.182780023207,0.182772541843,0.182765061125,0.182757581054,0.18275010163,0.182742622853,0.182735144723,0.18272766724,0.182720190403,0.182712714214,0.182705238671,0.182697763776,0.182690289528,0.182682815926,0.182675342972,0.182667870665,0.182660399006,0.182652927993,0.182645457628,0.18263798791,0.182630518839,0.182623050415,0.182615582639,0.182608115511,0.182600649029,0.182593183196,0.182585718009,0.182578253471,0.182570789579,0.182563326336,0.18255586374,0.182548401791,0.18254094049,0.182533479837,0.182526019832,0.182518560474,0.182511101764,0.182503643702,0.182496186288,0.182488729522,0.182481273403,0.182473817933,0.18246636311,0.182458908936,0.182451455409,0.18244400253,0.1824365503,0.182429098717,0.182421647783,0.182414197497,0.182406747859,0.182399298869,0.182391850527,0.182384402834,0.182376955789,0.182369509392,0.182362063643,0.182354618543,0.182347174092,0.182339730288,0.182332287133,0.182324844627,0.182317402769,0.18230996156,0.182302520999,0.182295081086,0.182287641823,0.182280203207,0.182272765241,0.182265327923,0.182257891254,0.182250455233,0.182243019862,0.182235585139,0.182228151065,0.182220717639,0.182213284863,0.182205852735,0.182198421257,0.182190990427,0.182183560246,0.182176130714,0.182168701831,0.182161273597,0.182153846012,0.182146419077,0.18213899279,0.182131567152,0.182124142164,0.182116717824,0.182109294134,0.182101871093,0.182094448701,0.182087026959,0.182079605866,0.182072185422,0.182064765627,0.182057346482,0.182049927986,0.182042510139,0.182035092942,0.182027676394,0.182020260496,0.182012845247,0.182005430648,0.181998016698,0.181990603398,0.181983190747,0.181975778746,0.181968367395,0.181960956693,0.181953546641,0.181946137238,0.181938728485,0.181931320382,0.181923912928,0.181916506125,0.181909099971,0.181901694467,0.181894289612,0.181886885408,0.181879481853,0.181872078948,0.181864676693,0.181857275088,0.181849874133,0.181842473828,0.181835074173,0.181827675168,0.181820276813,0.181812879108,0.181805482053,0.181798085648,0.181790689893,0.181783294788,0.181775900333,0.181768506529,0.181761113375,0.18175372087,0.181746329016,0.181738937813,0.181731547259,0.181724157356,0.181716768103,0.1817093795,0.181701991548,0.181694604246,0.181687217594,0.181679831593,0.181672446242,0.181665061542,0.181657677492,0.181650294092,0.181642911343,0.181635529244,0.181628147796,0.181620766998,0.181613386851,0.181606007354,0.181598628508,0.181591250313,0.181583872768,0.181576495873,0.18156911963,0.181561744037,0.181554369094,0.181546994802,0.181539621161,0.181532248171,0.181524875831,0.181517504143,0.181510133104,0.181502762717,0.181495392981,0.181488023895,0.18148065546,0.181473287676,0.181465920542,0.18145855406,0.181451188228,0.181443823048,0.181436458518,0.181429094639,0.181421731411,0.181414368834,0.181407006909,0.181399645634,0.18139228501,0.181384925037,0.181377565715,0.181370207044,0.181362849024,0.181355491655,0.181348134937,0.181340778871,0.181333423455,0.181326068691,0.181318714577,0.181311361115,0.181304008304,0.181296656145,0.181289304636,0.181281953779,0.181274603572,0.181267254017,0.181259905114,0.181252556861,0.18124520926,0.18123786231,0.181230516011,0.181223170364,0.181215825368,0.181208481023,0.18120113733,0.181193794288,0.181186451897,0.181179110158,0.18117176907,0.181164428633,0.181157088848,0.181149749715,0.181142411232,0.181135073402,0.181127736222,0.181120399694,0.181113063818,0.181105728593,0.18109839402,0.181091060098,0.181083726827,0.181076394208,0.181069062241,0.181061730925,0.181054400261,0.181047070248,0.181039740887,0.181032412178,0.18102508412,0.181017756714,0.181010429959,0.181003103856,0.180995778405,0.180988453605,0.180981129457,0.180973805961,0.180966483116,0.180959160923,0.180951839382,0.180944518492,0.180937198254,0.180929878668,0.180922559734,0.180915241451,0.18090792382,0.180900606841,0.180893290513,0.180885974838,0.180878659814,0.180871345442,0.180864031722,0.180856718653,0.180849406237,0.180842094472,0.180834783359,0.180827472898,0.180820163089,0.180812853931,0.180805545426,0.180798237572,0.18079093037,0.18078362382,0.180776317922,0.180769012676,0.180761708082,0.18075440414,0.18074710085,0.180739798211,0.180732496225,0.18072519489,0.180717894208,0.180710594177,0.180703294798,0.180695996072,0.180688697997,0.180681400574,0.180674103803,0.180666807685,0.180659512218,0.180652217403,0.18064492324,0.18063762973,0.180630336871,0.180623044664,0.18061575311,0.180608462207,0.180601171957,0.180593882358,0.180586593412,0.180579305117,0.180572017475,0.180564730485,0.180557444147,0.180550158461,0.180542873427,0.180535589045,0.180528305315,0.180521022237,0.180513739811,0.180506458038,0.180499176917,0.180491896447,0.18048461663,0.180477337465,0.180470058952,0.180462781091,0.180455503883,0.180448227326,0.180440951422,0.18043367617,0.180426401569,0.180419127622,0.180411854326,0.180404581682,0.180397309691,0.180390038351,0.180382767664,0.180375497629,0.180368228247,0.180360959516,0.180353691438,0.180346424011,0.180339157237,0.180331891115,0.180324625646,0.180317360828,0.180310096663,0.18030283315,0.180295570289,0.18028830808,0.180281046524,0.180273785619,0.180266525367,0.180259265767,0.18025200682,0.180244748524,0.180237490881,0.18023023389,0.180222977551,0.180215721864,0.18020846683,0.180201212448,0.180193958718,0.18018670564,0.180179453214,0.180172201441,0.18016495032,0.180157699851,0.180150450034,0.18014320087,0.180135952358,0.180128704498,0.18012145729,0.180114210734,0.180106964831,0.18009971958,0.180092474981,0.180085231034,0.18007798774,0.180070745098,0.180063503108,0.18005626177,0.180049021084,0.180041781051,0.18003454167,0.180027302941,0.180020064864,0.18001282744,0.180005590668,0.179998354548,0.17999111908,0.179983884264,0.179976650101,0.17996941659,0.179962183731,0.179954951524,0.179947719969,0.179940489067,0.179933258817,0.179926029219,0.179918800273,0.179911571979,0.179904344338,0.179897117349,0.179889891012,0.179882665327,0.179875440294,0.179868215914,0.179860992186,0.17985376911,0.179846546686,0.179839324914,0.179832103794,0.179824883327,0.179817663512,0.179810444349,0.179803225838,0.179796007979,0.179788790772,0.179781574218,0.179774358316,0.179767143065,0.179759928467,0.179752714521,0.179745501228,0.179738288586,0.179731076597,0.179723865259,0.179716654574,0.179709444541,0.17970223516,0.179695026431,0.179687818354,0.179680610929,0.179673404156,0.179666198036,0.179658992567,0.179651787751,0.179644583587,0.179637380074,0.179630177214,0.179622975006,0.17961577345,0.179608572546,0.179601372294,0.179594172694,0.179586973746,0.17957977545,0.179572577806,0.179565380814,0.179558184474,0.179550988786,0.179543793751,0.179536599367,0.179529405635,0.179522212555,0.179515020127,0.179507828351,0.179500637227,0.179493446755,0.179486256935,0.179479067767,0.179471879251,0.179464691387,0.179457504175,0.179450317614,0.179443131706,0.179435946449,0.179428761845,0.179421577892,0.179414394591,0.179407211942,0.179400029945,0.1793928486,0.179385667906,0.179378487865,0.179371308475,0.179364129738,0.179356951652,0.179349774217,0.179342597435,0.179335421305,0.179328245826,0.179321070999,0.179313896824,0.1793067233,0.179299550429,0.179292378209,0.179285206641,0.179278035725,0.17927086546,0.179263695847,0.179256526886,0.179249358577,0.179242190919,0.179235023913,0.179227857559,0.179220691856,0.179213526805,0.179206362406,0.179199198658,0.179192035562,0.179184873118,0.179177711325,0.179170550184,0.179163389694,0.179156229856,0.17914907067,0.179141912135,0.179134754252,0.179127597021,0.17912044044,0.179113284512,0.179106129235,0.179098974609,0.179091820636,0.179084667313,0.179077514642,0.179070362623,0.179063211255,0.179056060538,0.179048910473,0.179041761059,0.179034612297,0.179027464186,0.179020316727,0.179013169919,0.179006023762,0.178998878257,0.178991733403,0.178984589201,0.17897744565,0.17897030275,0.178963160502,0.178956018904,0.178948877959,0.178941737664,0.178934598021,0.178927459029,0.178920320688,0.178913182999,0.17890604596,0.178898909573,0.178891773837,0.178884638753,0.178877504319,0.178870370537,0.178863237406,0.178856104926,0.178848973097,0.17884184192,0.178834711393,0.178827581518,0.178820452293,0.17881332372,0.178806195798,0.178799068527,0.178791941906,0.178784815937,0.178777690619,0.178770565952,0.178763441936,0.178756318571,0.178749195857,0.178742073794,0.178734952382,0.17872783162,0.17872071151,0.17871359205,0.178706473242,0.178699355084,0.178692237577,0.178685120721,0.178678004516,0.178670888962,0.178663774058,0.178656659805,0.178649546203,0.178642433252,0.178635320952,0.178628209302,0.178621098303,0.178613987955,0.178606878257,0.17859976921,0.178592660814,0.178585553068,0.178578445973,0.178571339529,0.178564233735,0.178557128592,0.178550024099,0.178542920257,0.178535817066,0.178528714525,0.178521612635,0.178514511395,0.178507410805,0.178500310867,0.178493211578,0.17848611294,0.178479014953,0.178471917615,0.178464820929,0.178457724892,0.178450629506,0.178443534771,0.178436440685,0.17842934725,0.178422254466,0.178415162331,0.178408070847,0.178400980013,0.17839388983,0.178386800296,0.178379711413,0.17837262318,0.178365535597,0.178358448665,0.178351362382,0.17834427675,0.178337191768,0.178330107435,0.178323023753,0.178315940721,0.178308858339,0.178301776607,0.178294695526,0.178287615094,0.178280535312,0.17827345618,0.178266377698,0.178259299866,0.178252222683,0.178245146151,0.178238070269,0.178230995036,0.178223920454,0.178216846521,0.178209773238,0.178202700604,0.178195628621,0.178188557287,0.178181486603,0.178174416569,0.178167347184,0.178160278449,0.178153210364,0.178146142928,0.178139076142,0.178132010006,0.178124944519,0.178117879682,0.178110815495,0.178103751957,0.178096689068,0.178089626829,0.178082565239,0.178075504299,0.178068444009,0.178061384367,0.178054325376,0.178047267033,0.17804020934,0.178033152296,0.178026095902,0.178019040157,0.178011985061,0.178004930615,0.177997876817,0.177990823669,0.177983771171,0.177976719321,0.177969668121,0.17796261757,0.177955567667,0.177948518415,0.177941469811,0.177934421856,0.17792737455,0.177920327894,0.177913281886,0.177906236527,0.177899191818,0.177892147757,0.177885104345,0.177878061582,0.177871019468,0.177863978003,0.177856937187,0.17784989702,0.177842857501,0.177835818632,0.177828780411,0.177821742839,0.177814705915,0.17780766964,0.177800634014,0.177793599037,0.177786564708,0.177779531028,0.177772497997,0.177765465614,0.17775843388,0.177751402794,0.177744372357,0.177737342568,0.177730313428,0.177723284936,0.177716257093,0.177709229898,0.177702203352,0.177695177454,0.177688152204,0.177681127603,0.17767410365,0.177667080345,0.177660057688,0.17765303568,0.17764601432,0.177638993608,0.177631973545,0.177624954129,0.177617935362,0.177610917243,0.177603899772,0.177596882949,0.177589866774,0.177582851247,0.177575836368,0.177568822137,0.177561808554,0.177554795619,0.177547783332,0.177540771693,0.177533760701,0.177526750358,0.177519740662,0.177512731614,0.177505723214,0.177498715462,0.177491708357,0.1774847019,0.177477696091,0.177470690929,0.177463686415,0.177456682549,0.17744967933,0.177442676759,0.177435674836,0.17742867356,0.177421672931,0.17741467295,0.177407673616,0.17740067493,0.177393676891,0.1773866795,0.177379682756,0.177372686659,0.17736569121,0.177358696407,0.177351702253,0.177344708745,0.177337715884,0.177330723671,0.177323732105,0.177316741186,0.177309750914,0.17730276129,0.177295772312,0.177288783981,0.177281796298,0.177274809261,0.177267822871,0.177260837129,0.177253852033,0.177246867584,0.177239883782,0.177232900627,0.177225918118,0.177218936257,0.177211955042,0.177204974474,0.177197994552,0.177191015278,0.17718403665,0.177177058668,0.177170081334,0.177163104645,0.177156128604,0.177149153209,0.17714217846,0.177135204358,0.177128230903,0.177121258093,0.177114285931,0.177107314414,0.177100343544,0.177093373321,0.177086403743,0.177079434812,0.177072466527,0.177065498889,0.177058531896,0.17705156555,0.17704459985,0.177037634796,0.177030670388,0.177023706627,0.177016743511,0.177009781041,0.177002819217,0.17699585804,0.176988897508,0.176981937622,0.176974978382,0.176968019788,0.176961061839,0.176954104537,0.17694714788,0.176940191869,0.176933236504,0.176926281784,0.17691932771,0.176912374282,0.176905421499,0.176898469362,0.17689151787,0.176884567024,0.176877616824,0.176870667269,0.176863718359,0.176856770095,0.176849822476,0.176842875503,0.176835929175,0.176828983492,0.176822038455,0.176815094062,0.176808150315,0.176801207213,0.176794264757,0.176787322945,0.176780381779,0.176773441258,0.176766501381,0.17675956215,0.176752623564,0.176745685623,0.176738748326,0.176731811675,0.176724875668,0.176717940307,0.17671100559,0.176704071518,0.176697138091,0.176690205308,0.17668327317,0.176676341677,0.176669410829,0.176662480625,0.176655551066,0.176648622151,0.176641693881,0.176634766256,0.176627839274,0.176620912938,0.176613987245,0.176607062198,0.176600137794,0.176593214035,0.17658629092,0.17657936845,0.176572446623,0.176565525441,0.176558604903,0.176551685009,0.17654476576,0.176537847154,0.176530929193,0.176524011875,0.176517095202,0.176510179173,0.176503263787,0.176496349045,0.176489434948,0.176482521494,0.176475608684,0.176468696518,0.176461784995,0.176454874117,0.176447963882,0.17644105429,0.176434145343,0.176427237039,0.176420329378,0.176413422361,0.176406515988,0.176399610258,0.176392705172,0.176385800729,0.176378896929,0.176371993773,0.17636509126,0.17635818939,0.176351288164,0.176344387581,0.176337487641,0.176330588344,0.176323689691,0.176316791681,0.176309894313,0.176302997589,0.176296101508,0.17628920607,0.176282311274,0.176275417122,0.176268523613,0.176261630746,0.176254738522,0.176247846941,0.176240956003,0.176234065708,0.176227176055,0.176220287045,0.176213398678,0.176206510953,0.176199623871,0.176192737432,0.176185851635,0.17617896648,0.176172081968,0.176165198098,0.176158314871,0.176151432286,0.176144550343,0.176137669043,0.176130788385,0.176123908369,0.176117028995,0.176110150264,0.176103272175,0.176096394727,0.176089517922,0.176082641759,0.176075766238,0.176068891359,0.176062017122,0.176055143526,0.176048270573,0.176041398261,0.176034526591,0.176027655563,0.176020785177,0.176013915433,0.17600704633,0.176000177868,0.175993310049,0.175986442871,0.175979576334,0.175972710439,0.175965845185,0.175958980573,0.175952116603,0.175945253273,0.175938390585,0.175931528538,0.175924667133,0.175917806369,0.175910946246,0.175904086764,0.175897227923,0.175890369724,0.175883512165,0.175876655248,0.175869798971,0.175862943336,0.175856088341,0.175849233987,0.175842380274,0.175835527202,0.175828674771,0.175821822981,0.175814971831,0.175808121322,0.175801271454,0.175794422226,0.175787573639,0.175780725692,0.175773878386,0.175767031721,0.175760185695,0.175753340311,0.175746495566,0.175739651462,0.175732807999,0.175725965175,0.175719122992,0.175712281449,0.175705440546,0.175698600284,0.175691760661,0.175684921679,0.175678083336,0.175671245634,0.175664408571,0.175657572149,0.175650736366,0.175643901223,0.17563706672,0.175630232857,0.175623399634,0.17561656705,0.175609735106,0.175602903802,0.175596073137,0.175589243112,0.175582413726,0.17557558498,0.175568756873,0.175561929406,0.175555102578,0.175548276389,0.17554145084,0.17553462593,0.175527801659,0.175520978028,0.175514155036,0.175507332683,0.175500510969,0.175493689894,0.175486869458,0.175480049661,0.175473230503,0.175466411984,0.175459594104,0.175452776862,0.17544596026,0.175439144296,0.175432328971,0.175425514285,0.175418700237,0.175411886828,0.175405074058,0.175398261926,0.175391450433,0.175384639578,0.175377829362,0.175371019784,0.175364210844,0.175357402543,0.17535059488,0.175343787855,0.175336981469,0.175330175721,0.175323370611,0.175316566139,0.175309762305,0.175302959109,0.175296156551,0.175289354631,0.175282553349,0.175275752705,0.175268952699,0.175262153331,0.1752553546,0.175248556507,0.175241759052,0.175234962234,0.175228166055,0.175221370512,0.175214575608,0.17520778134,0.175200987711,0.175194194718,0.175187402363,0.175180610646,0.175173819566,0.175167029123,0.175160239317,0.175153450149,0.175146661617,0.175139873723,0.175133086466,0.175126299846,0.175119513863,0.175112728517,0.175105943808,0.175099159736,0.175092376301,0.175085593503,0.175078811341,0.175072029816,0.175065248928,0.175058468677,0.175051689062,0.175044910083,0.175038131742,0.175031354037,0.175024576968,0.175017800536,0.17501102474,0.17500424958,0.174997475057,0.17499070117,0.17498392792,0.174977155305,0.174970383327,0.174963611985,0.174956841279,0.174950071209,0.174943301775,0.174936532977,0.174929764815,0.174922997289,0.174916230399,0.174909464145,0.174902698526,0.174895933543,0.174889169196,0.174882405484,0.174875642408,0.174868879968,0.174862118163,0.174855356994,0.17484859646,0.174841836561,0.174835077298,0.174828318671,0.174821560678,0.174814803321,0.174808046599,0.174801290513,0.174794535061,0.174787780244,0.174781026063,0.174774272517,0.174767519605,0.174760767329,0.174754015687,0.174747264681,0.174740514309,0.174733764572,0.174727015469,0.174720267002,0.174713519169,0.17470677197,0.174700025406,0.174693279477,0.174686534182,0.174679789522,0.174673045496,0.174666302105,0.174659559348,0.174652817225,0.174646075736,0.174639334882,0.174632594662,0.174625855076,0.174619116124,0.174612377806,0.174605640122,0.174598903072,0.174592166656,0.174585430874,0.174578695725,0.174571961211,0.17456522733,0.174558494083,0.17455176147,0.17454502949,0.174538298144,0.174531567432,0.174524837353,0.174518107907,0.174511379095,0.174504650916,0.174497923371,0.174491196459,0.17448447018,0.174477744535,0.174471019522,0.174464295143,0.174457571397,0.174450848284,0.174444125804,0.174437403957,0.174430682743,0.174423962162,0.174417242213,0.174410522898,0.174403804215,0.174397086165,0.174390368748,0.174383651963,0.174376935811,0.174370220291,0.174363505404,0.174356791149,0.174350077527,0.174343364538,0.17433665218,0.174329940455,0.174323229362,0.174316518902,0.174309809073,0.174303099877,0.174296391313,0.174289683381,0.17428297608,0.174276269412,0.174269563376,0.174262857972,0.174256153199,0.174249449058,0.174242745549,0.174236042672,0.174229340426,0.174222638812,0.17421593783,0.174209237479,0.17420253776,0.174195838672,0.174189140215,0.17418244239,0.174175745196,0.174169048634,0.174162352703,0.174155657402,0.174148962733,0.174142268696,0.174135575289,0.174128882513,0.174122190368,0.174115498854,0.174108807972,0.174102117719,0.174095428098,0.174088739108,0.174082050748,0.174075363019,0.17406867592,0.174061989452,0.174055303615,0.174048618408,0.174041933832,0.174035249886,0.17402856657,0.174021883885,0.17401520183,0.174008520405,0.17400183961,0.173995159446,0.173988479912,0.173981801007,0.173975122733,0.173968445089,0.173961768075,0.17395509169,0.173948415936,0.173941740811,0.173935066316,0.17392839245,0.173921719215,0.173915046609,0.173908374632,0.173901703285,0.173895032568,0.17388836248,0.173881693021,0.173875024192,0.173868355992,0.173861688422,0.17385502148,0.173848355168,0.173841689485,0.173835024431,0.173828360006,0.17382169621,0.173815033043,0.173808370505,0.173801708596,0.173795047315,0.173788386664,0.173781726641,0.173775067247,0.173768408481,0.173761750344,0.173755092836,0.173748435956,0.173741779705,0.173735124082,0.173728469087,0.173721814721,0.173715160983,0.173708507873,0.173701855391,0.173695203538,0.173688552313,0.173681901715,0.173675251746,0.173668602405,0.173661953691,0.173655305606,0.173648658148,0.173642011319,0.173635365116,0.173628719542,0.173622074595,0.173615430276,0.173608786584,0.17360214352,0.173595501084,0.173588859275,0.173582218093,0.173575577538,0.173568937611,0.173562298311,0.173555659638,0.173549021593,0.173542384174,0.173535747383,0.173529111218,0.173522475681,0.17351584077,0.173509206486,0.17350257283,0.1734959398,0.173489307396,0.17348267562,0.17347604447,0.173469413946,0.173462784049,0.173456154779,0.173449526135,0.173442898117,0.173436270726,0.173429643962,0.173423017823,0.173416392311,0.173409767425,0.173403143165,0.173396519531,0.173389896523,0.173383274141,0.173376652385,0.173370031255,0.173363410751,0.173356790872,0.17335017162,0.173343552993,0.173336934992,0.173330317616,0.173323700866,0.173317084742,0.173310469243,0.173303854369,0.173297240121,0.173290626498,0.173284013501,0.173277401128,0.173270789381,0.173264178259,0.173257567762,0.173250957891,0.173244348644,0.173237740022,0.173231132025,0.173224524653,0.173217917906,0.173211311784,0.173204706286,0.173198101413,0.173191497165,0.173184893541,0.173178290542,0.173171688167,0.173165086417,0.173158485291,0.173151884789,0.173145284912,0.173138685659,0.17313208703,0.173125489026,0.173118891645,0.173112294889,0.173105698756,0.173099103248,0.173092508363,0.173085914103,0.173079320466,0.173072727453,0.173066135064,0.173059543298,0.173052952156,0.173046361638,0.173039771743,0.173033182472,0.173026593824,0.173020005799,0.173013418398,0.17300683162,0.173000245466,0.172993659934,0.172987075026,0.172980490741,0.172973907079,0.17296732404,0.172960741624,0.172954159831,0.17294757866,0.172940998113,0.172934418188,0.172927838886,0.172921260207,0.17291468215,0.172908104716,0.172901527904,0.172894951715,0.172888376149,0.172881801204,0.172875226883,0.172868653183,0.172862080106,0.17285550765,0.172848935817,0.172842364606,0.172835794017,0.17282922405,0.172822654705,0.172816085982,0.172809517881,0.172802950402,0.172796383544,0.172789817308,0.172783251694,0.172776686701,0.17277012233,0.17276355858,0.172756995452,0.172750432945,0.17274387106,0.172737309796,0.172730749153,0.172724189131,0.172717629731,0.172711070952,0.172704512793,0.172697955256,0.17269139834,0.172684842045,0.17267828637,0.172671731316,0.172665176884,0.172658623071,0.17265206988,0.172645517309,0.172638965359,0.172632414029,0.17262586332,0.172619313231,0.172612763762,0.172606214914,0.172599666686,0.172593119079,0.172586572091,0.172580025724,0.172573479977,0.172566934849,0.172560390342,0.172553846455,0.172547303187,0.17254076054,0.172534218512,0.172527677104,0.172521136316,0.172514596147,0.172508056598,0.172501517668,0.172494979358,0.172488441667,0.172481904596,0.172475368144,0.172468832311,0.172462297097,0.172455762503,0.172449228528,0.172442695172,0.172436162435,0.172429630317,0.172423098818,0.172416567937,0.172410037676,0.172403508033,0.172396979009,0.172390450604,0.172383922817,0.172377395649,0.1723708691,0.172364343169,0.172357817856,0.172351293162,0.172344769086,0.172338245629,0.172331722789,0.172325200568,0.172318678965,0.17231215798,0.172305637613,0.172299117864,0.172292598733,0.17228608022,0.172279562325,0.172273045047,0.172266528387,0.172260012345,0.172253496921,0.172246982114,0.172240467924,0.172233954352,0.172227441397,0.17222092906,0.17221441734,0.172207906238,0.172201395752,0.172194885884,0.172188376633,0.172181867999,0.172175359982,0.172168852582,0.172162345799,0.172155839632,0.172149334083,0.17214282915,0.172136324834,0.172129821134,0.172123318052,0.172116815585,0.172110313736,0.172103812502,0.172097311885,0.172090811885,0.172084312501,0.172077813733,0.172071315581,0.172064818045,0.172058321126,0.172051824822,0.172045329134,0.172038834063,0.172032339607,0.172025845767,0.172019352543,0.172012859935,0.172006367942,0.171999876565,0.171993385803,0.171986895657,0.171980406127,0.171973917212,0.171967428912,0.171960941227,0.171954454158,0.171947967704,0.171941481866,0.171934996642,0.171928512033,0.17192202804,0.171915544661,0.171909061897,0.171902579748,0.171896098214,0.171889617295,0.17188313699,0.1718766573,0.171870178225,0.171863699764,0.171857221918,0.171850744686,0.171844268068,0.171837792065,0.171831316676,0.171824841901,0.17181836774,0.171811894194,0.171805421262,0.171798948943,0.171792477239,0.171786006148,0.171779535672,0.171773065809,0.17176659656,0.171760127925,0.171753659903,0.171747192495,0.1717407257,0.171734259519,0.171727793952,0.171721328997,0.171714864656,0.171708400929,0.171701937815,0.171695475313,0.171689013425,0.17168255215,0.171676091488,0.171669631439,0.171663172003,0.17165671318,0.17165025497,0.171643797372,0.171637340387,0.171630884015,0.171624428255,0.171617973108,0.171611518573,0.171605064651,0.171598611341,0.171592158644,0.171585706559,0.171579255086,0.171572804225,0.171566353976,0.17155990434,0.171553455315,0.171547006903,0.171540559102,0.171534111913,0.171527665336,0.171521219371,0.171514774017,0.171508329276,0.171501885145,0.171495441626,0.171488998719,0.171482556423,0.171476114739,0.171469673666,0.171463233204,0.171456793353,0.171450354114,0.171443915485,0.171437477468,0.171431040062,0.171424603267,0.171418167082,0.171411731509,0.171405296546,0.171398862194,0.171392428453,0.171385995322,0.171379562802,0.171373130892,0.171366699593,0.171360268904,0.171353838826,0.171347409358,0.1713409805,0.171334552253,0.171328124615,0.171321697588,0.171315271171,0.171308845364,0.171302420166,0.171295995579,0.171289571601,0.171283148234,0.171276725475,0.171270303327,0.171263881788,0.171257460859,0.171251040539,0.171244620829,0.171238201728,0.171231783237,0.171225365354,0.171218948081,0.171212531418,0.171206115363,0.171199699917,0.171193285081,0.171186870853,0.171180457235,0.171174044225,0.171167631824,0.171161220032,0.171154808848,0.171148398273,0.171141988307,0.171135578949,0.1711291702,0.171122762059,0.171116354526,0.171109947602,0.171103541286,0.171097135579,0.171090730479,0.171084325988,0.171077922104,0.171071518829,0.171065116161,0.171058714102,0.17105231265,0.171045911806,0.17103951157,0.171033111941,0.17102671292,0.171020314507,0.171013916701,0.171007519503,0.171001122912,0.170994726928,0.170988331551,0.170981936782,0.17097554262,0.170969149065,0.170962756118,0.170956363777,0.170949972043,0.170943580916,0.170937190396,0.170930800483,0.170924411176,0.170918022476,0.170911634383,0.170905246896,0.170898860016,0.170892473742,0.170886088075,0.170879703014,0.17087331856,0.170866934711,0.170860551469,0.170854168833,0.170847786803,0.170841405379,0.170835024561,0.170828644349,0.170822264743,0.170815885743,0.170809507348,0.170803129559,0.170796752376,0.170790375798,0.170783999826,0.170777624459,0.170771249698,0.170764875542,0.170758501992,0.170752129046,0.170745756706,0.170739384971,0.170733013841,0.170726643316,0.170720273397,0.170713904082,0.170707535372,0.170701167266,0.170694799766,0.17068843287,0.170682066579,0.170675700892,0.17066933581,0.170662971333,0.17065660746,0.170650244191,0.170643881526,0.170637519466,0.17063115801,0.170624797158,0.170618436911,0.170612077267,0.170605718227,0.170599359792,0.17059300196,0.170586644732,0.170580288107,0.170573932087,0.17056757667,0.170561221856,0.170554867646,0.17054851404,0.170542161037,0.170535808638,0.170529456841,0.170523105648,0.170516755059,0.170510405072,0.170504055688,0.170497706908,0.17049135873,0.170485011156,0.170478664184,0.170472317815,0.170465972049,0.170459626885,0.170453282325,0.170446938366,0.170440595011,0.170434252257,0.170427910107,0.170421568558,0.170415227612,0.170408887268,0.170402547526,0.170396208387,0.170389869849,0.170383531914,0.17037719458,0.170370857849,0.170364521719,0.170358186191,0.170351851265,0.170345516941,0.170339183218,0.170332850097,0.170326517577,0.170320185659,0.170313854342,0.170307523626,0.170301193512,0.170294863999,0.170288535088,0.170282206777,0.170275879067,0.170269551959,0.170263225451,0.170256899545,0.170250574239,0.170244249534,0.170237925429,0.170231601926,0.170225279023,0.17021895672,0.170212635019,0.170206313917,0.170199993416,0.170193673515,0.170187354215,0.170181035515,0.170174717414,0.170168399915,0.170162083015,0.170155766715,0.170149451015,0.170143135915,0.170136821414,0.170130507514,0.170124194213,0.170117881512,0.170111569411,0.170105257909,0.170098947006,0.170092636703,0.170086327,0.170080017895,0.17007370939,0.170067401484,0.170061094178,0.17005478747,0.170048481361,0.170042175852,0.170035870941,0.170029566629,0.170023262916,0.170016959802,0.170010657287,0.17000435537,0.169998054051,0.169991753331,0.16998545321,0.169979153687,0.169972854763,0.169966556436,0.169960258708,0.169953961578,0.169947665046,0.169941369113,0.169935073777,0.169928779039,0.169922484899,0.169916191357,0.169909898413,0.169903606067,0.169897314318,0.169891023167,0.169884732613,0.169878442657,0.169872153298,0.169865864537,0.169859576373,0.169853288806,0.169847001836,0.169840715464,0.169834429689,0.16982814451,0.169821859929,0.169815575945,0.169809292557,0.169803009766,0.169796727572,0.169790445975,0.169784164974,0.16977788457,0.169771604763,0.169765325552,0.169759046937,0.169752768919,0.169746491497,0.169740214671,0.169733938441,0.169727662808,0.16972138777,0.169715113329,0.169708839483,0.169702566234,0.16969629358,0.169690021522,0.169683750059,0.169677479193,0.169671208922,0.169664939246,0.169658670166,0.169652401682,0.169646133792,0.169639866498,0.1696335998,0.169627333696,0.169621068188,0.169614803275,0.169608538956,0.169602275233,0.169596012105,0.169589749571,0.169583487633,0.169577226289,0.169570965539,0.169564705384,0.169558445824,0.169552186859,0.169545928487,0.169539670711,0.169533413528,0.16952715694,0.169520900946,0.169514645546,0.16950839074,0.169502136529,0.169495882911,0.169489629887,0.169483377457,0.169477125621,0.169470874378,0.16946462373,0.169458373675,0.169452124213,0.169445875345,0.169439627071,0.169433379389,0.169427132302,0.169420885807,0.169414639906,0.169408394598,0.169402149883,0.169395905761,0.169389662232,0.169383419296,0.169377176952,0.169370935202,0.169364694044,0.169358453479,0.169352213507,0.169345974128,0.16933973534,0.169333497146,0.169327259543,0.169321022534,0.169314786116,0.16930855029,0.169302315057,0.169296080416,0.169289846367,0.16928361291,0.169277380045,0.169271147772,0.16926491609,0.169258685001,0.169252454503,0.169246224596,0.169239995282,0.169233766559,0.169227538427,0.169221310887,0.169215083938,0.16920885758,0.169202631814,0.169196406639,0.169190182055,0.169183958062,0.16917773466,0.169171511849,0.169165289628,0.169159067999,0.169152846961,0.169146626513,0.169140406655,0.169134187389,0.169127968713,0.169121750627,0.169115533132,0.169109316227,0.169103099913,0.169096884189,0.169090669054,0.169084454511,0.169078240557,0.169072027193,0.169065814419,0.169059602235,0.169053390641,0.169047179636,0.169040969221,0.169034759396,0.169028550161,0.169022341515,0.169016133459,0.169009925992,0.169003719114,0.168997512826,0.168991307126,0.168985102016,0.168978897496,0.168972693564,0.168966490221,0.168960287467,0.168954085302,0.168947883726,0.168941682739,0.16893548234,0.168929282531,0.168923083309,0.168916884676,0.168910686632,0.168904489176,0.168898292309,0.168892096029,0.168885900338,0.168879705235,0.168873510721,0.168867316794,0.168861123455,0.168854930705,0.168848738542,0.168842546967,0.16883635598,0.16883016558,0.168823975768,0.168817786544,0.168811597907,0.168805409858,0.168799222396,0.168793035521,0.168786849234,0.168780663534,0.168774478421,0.168768293896,0.168762109957,0.168755926605,0.16874974384,0.168743561663,0.168737380072,0.168731199067,0.16872501865,0.168718838819,0.168712659574,0.168706480917,0.168700302845,0.16869412536,0.168687948462,0.168681772149,0.168675596423,0.168669421283,0.168663246729,0.168657072761,0.16865089938,0.168644726584,0.168638554374,0.16863238275,0.168626211711,0.168620041258,0.168613871391,0.168607702109,0.168601533413,0.168595365303,0.168589197778,0.168583030838,0.168576864483,0.168570698714,0.168564533529,0.16855836893,0.168552204916,0.168546041487,0.168539878643,0.168533716383,0.168527554709,0.168521393619,0.168515233113,0.168509073193,0.168502913857,0.168496755105,0.168490596938,0.168484439355,0.168478282357,0.168472125943,0.168465970113,0.168459814867,0.168453660205,0.168447506128,0.168441352634,0.168435199724,0.168429047398,0.168422895656,0.168416744497,0.168410593922,0.168404443931,0.168398294523,0.168392145699,0.168385997458,0.168379849801,0.168373702727,0.168367556236,0.168361410328,0.168355265003,0.168349120262,0.168342976103,0.168336832527,0.168330689535,0.168324547125,0.168318405298,0.168312264053,0.168306123391,0.168299983312,0.168293843815,0.168287704901,0.168281566569,0.16827542882,0.168269291652,0.168263155067,0.168257019064,0.168250883644,0.168244748805,0.168238614548,0.168232480873,0.16822634778,0.168220215269,0.16821408334,0.168207951992,0.168201821226,0.168195691041,0.168189561438,0.168183432416,0.168177303976,0.168171176117,0.168165048839,0.168158922143,0.168152796028,0.168146670493,0.16814054554,0.168134421168,0.168128297376,0.168122174166,0.168116051536,0.168109929487,0.168103808018,0.16809768713,0.168091566823,0.168085447096,0.16807932795,0.168073209384,0.168067091398,0.168060973993,0.168054857167,0.168048740922,0.168042625257,0.168036510171,0.168030395666,0.168024281741,0.168018168395,0.168012055629,0.168005943443,0.167999831837,0.16799372081,0.167987610362,0.167981500494,0.167975391206,0.167969282496,0.167963174366,0.167957066815,0.167950959844,0.167944853451,0.167938747638,0.167932642403,0.167926537747,0.16792043367,0.167914330172,0.167908227253,0.167902124912,0.16789602315,0.167889921967,0.167883821362,0.167877721335,0.167871621887,0.167865523017,0.167859424725,0.167853327011,0.167847229876,0.167841133319,0.167835037339,0.167828941938,0.167822847114,0.167816752868,0.1678106592,0.16780456611,0.167798473597,0.167792381662,0.167786290304,0.167780199524,0.167774109321,0.167768019696,0.167761930648,0.167755842177,0.167749754283,0.167743666966,0.167737580226,0.167731494064,0.167725408478,0.167719323469,0.167713239036,0.167707155181,0.167701071902,0.1676949892,0.167688907074,0.167682825525,0.167676744552,0.167670664155,0.167664584335,0.167658505091,0.167652426423,0.167646348332,0.167640270816,0.167634193876,0.167628117513,0.167622041725,0.167615966513,0.167609891877,0.167603817816,0.167597744331,0.167591671422,0.167585599088,0.167579527329,0.167573456146,0.167567385538,0.167561315506,0.167555246048,0.167549177166,0.167543108859,0.167537041127,0.16753097397,0.167524907388,0.16751884138,0.167512775948,0.16750671109,0.167500646807,0.167494583098,0.167488519964,0.167482457404,0.167476395419,0.167470334008,0.167464273171,0.167458212909,0.16745215322,0.167446094106,0.167440035566,0.1674339776,0.167427920208,0.167421863389,0.167415807145,0.167409751474,0.167403696377,0.167397641853,0.167391587903,0.167385534526,0.167379481723,0.167373429494,0.167367377837,0.167361326754,0.167355276244,0.167349226307,0.167343176943,0.167337128152,0.167331079935,0.167325032289,0.167318985217,0.167312938718,0.167306892791,0.167300847437,0.167294802655,0.167288758446,0.16728271481,0.167276671746,0.167270629254,0.167264587334,0.167258545987,0.167252505211,0.167246465008,0.167240425377,0.167234386318,0.16722834783,0.167222309915,0.167216272571,0.167210235799,0.167204199599,0.16719816397,0.167192128913,0.167186094427,0.167180060513,0.16717402717,0.167167994398,0.167161962198,0.167155930568,0.16714989951,0.167143869023,0.167137839107,0.167131809761,0.167125780987,0.167119752783,0.16711372515,0.167107698088,0.167101671596,0.167095645675,0.167089620324,0.167083595544,0.167077571334,0.167071547695,0.167065524625,0.167059502126,0.167053480197,0.167047458838,0.167041438049,0.16703541783,0.167029398181,0.167023379102,0.167017360592,0.167011342652,0.167005325282,0.166999308481,0.16699329225,0.166987276588,0.166981261496,0.166975246973,0.166969233019,0.166963219634,0.166957206819,0.166951194572,0.166945182895,0.166939171786,0.166933161247,0.166927151276,0.166921141874,0.16691513304,0.166909124776,0.16690311708,0.166897109952,0.166891103393,0.166885097402,0.16687909198,0.166873087126,0.16686708284,0.166861079122,0.166855075972,0.16684907339,0.166843071377,0.166837069931,0.166831069053,0.166825068743,0.166819069,0.166813069825,0.166807071218,0.166801073178,0.166795075706,0.166789078801,0.166783082464,0.166777086693,0.16677109149,0.166765096855,0.166759102786,0.166753109284,0.166747116349,0.166741123981,0.16673513218,0.166729140946,0.166723150279,0.166717160178,0.166711170644,0.166705181676,0.166699193275,0.16669320544,0.166687218172,0.16668123147,0.166675245334,0.166669259764,0.16666327476,0.166657290323,0.166651306451,0.166645323146,0.166639340406,0.166633358232,0.166627376624,0.166621395581,0.166615415104,0.166609435193,0.166603455847,0.166597477066,0.166591498851,0.166585521201,0.166579544117,0.166573567597,0.166567591643,0.166561616254,0.166555641429,0.16654966717,0.166543693476,0.166537720346,0.166531747781,0.166525775781,0.166519804345,0.166513833474,0.166507863168,0.166501893426,0.166495924248,0.166489955635,0.166483987585,0.1664780201,0.16647205318,0.166466086823,0.16646012103,0.166454155801,0.166448191136,0.166442227035,0.166436263498,0.166430300524,0.166424338114,0.166418376268,0.166412414985,0.166406454265,0.166400494109,0.166394534516,0.166388575487,0.16638261702,0.166376659117,0.166370701777,0.166364745,0.166358788786,0.166352833135,0.166346878046,0.16634092352,0.166334969557,0.166329016157,0.166323063319,0.166317111044,0.166311159331,0.166305208181,0.166299257593,0.166293307567,0.166287358103,0.166281409202,0.166275460862,0.166269513085,0.166263565869,0.166257619216,0.166251673124,0.166245727594,0.166239782626,0.166233838219,0.166227894374,0.166221951091,0.166216008369,0.166210066208,0.166204124609,0.166198183571,0.166192243094,0.166186303178,0.166180363824,0.16617442503,0.166168486798,0.166162549126,0.166156612015,0.166150675465,0.166144739475,0.166138804047,0.166132869179,0.166126934871,0.166121001124,0.166115067937,0.166109135311,0.166103203245,0.166097271739,0.166091340793,0.166085410407,0.166079480582,0.166073551316,0.16606762261,0.166061694464,0.166055766878,0.166049839852,0.166043913385,0.166037987478,0.16603206213,0.166026137342,0.166020213113,0.166014289443,0.166008366333,0.166002443782,0.165996521791,0.165990600358,0.165984679484,0.16597875917,0.165972839414,0.165966920217,0.165961001579,0.165955083499,0.165949165978,0.165943249016,0.165937332613,0.165931416767,0.165925501481,0.165919586752,0.165913672582,0.16590775897,0.165901845916,0.165895933421,0.165890021483,0.165884110103,0.165878199282,0.165872289018,0.165866379312,0.165860470163,0.165854561572,0.165848653539,0.165842746064,0.165836839146,0.165830932785,0.165825026981,0.165819121735,0.165813217047,0.165807312915,0.16580140934,0.165795506323,0.165789603862,0.165783701958,0.165777800612,0.165771899822,0.165765999588,0.165760099912,0.165754200792,0.165748302228,0.165742404221,0.16573650677,0.165730609876,0.165724713538,0.165718817756,0.165712922531,0.165707027861,0.165701133748,0.16569524019,0.165689347189,0.165683454743,0.165677562853,0.165671671519,0.16566578074,0.165659890517,0.16565400085,0.165648111738,0.165642223181,0.16563633518,0.165630447734,0.165624560844,0.165618674508,0.165612788728,0.165606903502,0.165601018832,0.165595134716,0.165589251156,0.16558336815,0.165577485699,0.165571603802,0.165565722461,0.165559841673,0.16555396144,0.165548081762,0.165542202638,0.165536324068,0.165530446053,0.165524568591,0.165518691684,0.165512815331,0.165506939531,0.165501064286,0.165495189595,0.165489315457,0.165483441873,0.165477568842,0.165471696366,0.165465824442,0.165459953073,0.165454082256,0.165448211993,0.165442342284,0.165436473127,0.165430604524,0.165424736474,0.165418868977,0.165413002033,0.165407135641,0.165401269803,0.165395404517,0.165389539784,0.165383675604,0.165377811976,0.165371948901,0.165366086379,0.165360224408,0.16535436299,0.165348502125,0.165342641812,0.16533678205,0.165330922841,0.165325064184,0.165319206079,0.165313348526,0.165307491524,0.165301635075,0.165295779177,0.165289923831,0.165284069036,0.165278214793,0.165272361102,0.165266507961,0.165260655373,0.165254803335,0.165248951849,0.165243100914,0.16523725053,0.165231400696,0.165225551414,0.165219702683,0.165213854503,0.165208006873,0.165202159795,0.165196313266,0.165190467289,0.165184621862,0.165178776985,0.165172932659,0.165167088883,0.165161245658,0.165155402983,0.165149560857,0.165143719282,0.165137878257,0.165132037782,0.165126197857,0.165120358482,0.165114519656,0.16510868138,0.165102843654,0.165097006478,0.165091169851,0.165085333773,0.165079498245,0.165073663266,0.165067828837,0.165061994956,0.165056161625,0.165050328843,0.16504449661,0.165038664926,0.165032833791,0.165027003204,0.165021173167,0.165015343678,0.165009514738,0.165003686346,0.164997858503,0.164992031208,0.164986204462,0.164980378264,0.164974552614,0.164968727513,0.16496290296,0.164957078955,0.164951255497,0.164945432588,0.164939610227,0.164933788413,0.164927967148,0.16492214643,0.164916326259,0.164910506637,0.164904687561,0.164898869034,0.164893051053,0.16488723362,0.164881416734,0.164875600396,0.164869784605,0.16486396936,0.164858154663,0.164852340513,0.164846526909,0.164840713852,0.164834901343,0.164829089379,0.164823277963,0.164817467093,0.16481165677,0.164805846993,0.164800037762,0.164794229078,0.16478842094,0.164782613348,0.164776806302,0.164770999803,0.164765193849,0.164759388442,0.16475358358,0.164747779264,0.164741975494,0.16473617227,0.164730369591,0.164724567458,0.16471876587,0.164712964828,0.164707164331,0.16470136438,0.164695564973,0.164689766112,0.164683967796,0.164678170025,0.1646723728,0.164666576119,0.164660779983,0.164654984392,0.164649189345,0.164643394843,0.164637600886,0.164631807474,0.164626014606,0.164620222282,0.164614430503,0.164608639268,0.164602848578,0.164597058431,0.164591268829,0.164585479771,0.164579691257,0.164573903286,0.16456811586,0.164562328977,0.164556542639,0.164550756843,0.164544971592,0.164539186884,0.16453340272,0.164527619099,0.164521836021,0.164516053487,0.164510271496,0.164504490048,0.164498709143,0.164492928781,0.164487148963,0.164481369687,0.164475590954,0.164469812764,0.164464035117,0.164458258012,0.16445248145,0.164446705431,0.164440929954,0.16443515502,0.164429380627,0.164423606778,0.16441783347,0.164412060705,0.164406288481,0.1644005168,0.164394745661,0.164388975064,0.164383205008,0.164377435495,0.164371666523,0.164365898093,0.164360130204,0.164354362857,0.164348596052,0.164342829788,0.164337064065,0.164331298884,0.164325534243,0.164319770144,0.164314006587,0.16430824357,0.164302481094,0.164296719159,0.164290957765,0.164285196912,0.164279436599,0.164273676827,0.164267917596,0.164262158906,0.164256400755,0.164250643146,0.164244886076,0.164239129547,0.164233373558,0.16422761811,0.164221863201,0.164216108833,0.164210355004,0.164204601715,0.164198848967,0.164193096758,0.164187345088,0.164181593959,0.164175843369,0.164170093318,0.164164343807,0.164158594836,0.164152846404,0.164147098511,0.164141351157,0.164135604343,0.164129858067,0.164124112331,0.164118367134,0.164112622475,0.164106878356,0.164101134775,0.164095391733,0.164089649229,0.164083907264,0.164078165838,0.16407242495,0.164066684601,0.16406094479,0.164055205517,0.164049466783,0.164043728586,0.164037990928,0.164032253808,0.164026517225,0.164020781181,0.164015045675,0.164009310706,0.164003576275,0.163997842381,0.163992109026,0.163986376207,0.163980643927,0.163974912183,0.163969180977,0.163963450309,0.163957720177,0.163951990583,0.163946261525,0.163940533005,0.163934805022,0.163929077575,0.163923350666,0.163917624293,0.163911898457,0.163906173158,0.163900448395,0.163894724169,0.163889000479,0.163883277325,0.163877554708,0.163871832628,0.163866111083,0.163860390074,0.163854669602,0.163848949666,0.163843230265,0.163837511401,0.163831793072,0.163826075279,0.163820358022,0.163814641301,0.163808925115,0.163803209464,0.163797494349,0.16379177977,0.163786065725,0.163780352217,0.163774639243,0.163768926804,0.163763214901,0.163757503532,0.163751792698,0.1637460824,0.163740372636,0.163734663407,0.163728954712,0.163723246553,0.163717538927,0.163711831837,0.163706125281,0.163700419259,0.163694713771,0.163689008818,0.163683304399,0.163677600514,0.163671897163,0.163666194346,0.163660492064,0.163654790315,0.163649089099,0.163643388418,0.16363768827,0.163631988656,0.163626289576,0.163620591029,0.163614893015,0.163609195535,0.163603498588,0.163597802174,0.163592106294,0.163586410947,0.163580716133,0.163575021851,0.163569328103,0.163563634888,0.163557942205,0.163552250055,0.163546558438,0.163540867354,0.163535176802,0.163529486782,0.163523797295,0.163518108341,0.163512419918,0.163506732028,0.16350104467,0.163495357845,0.163489671551,0.163483985789,0.16347830056,0.163472615862,0.163466931696,0.163461248061,0.163455564959,0.163449882388,0.163444200348,0.16343851884,0.163432837864,0.163427157419,0.163421477505,0.163415798122,0.163410119271,0.163404440951,0.163398763161,0.163393085903,0.163387409176,0.163381732979,0.163376057314,0.163370382179,0.163364707575,0.163359033501,0.163353359958,0.163347686945,0.163342014463,0.163336342511,0.16333067109,0.163325000199,0.163319329838,0.163313660007,0.163307990706,0.163302321935,0.163296653694,0.163290985983,0.163285318802,0.16327965215,0.163273986028,0.163268320436,0.163262655373,0.16325699084,0.163251326836,0.163245663362,0.163240000416,0.163234338001,0.163228676114,0.163223014756,0.163217353927,0.163211693628,0.163206033857,0.163200374615,0.163194715902,0.163189057718,0.163183400062,0.163177742935,0.163172086337,0.163166430267,0.163160774725,0.163155119712,0.163149465227,0.16314381127,0.163138157842,0.163132504941,0.163126852569,0.163121200724,0.163115549408,0.163109898619,0.163104248358,0.163098598625,0.163092949419,0.163087300742,0.163081652591,0.163076004968,0.163070357873,0.163064711305,0.163059065264,0.16305341975,0.163047774764,0.163042130305,0.163036486372,0.163030842967,0.163025200088,0.163019557737,0.163013915912,0.163008274614,0.163002633843,0.162996993598,0.16299135388,0.162985714688,0.162980076022,0.162974437883,0.16296880027,0.162963163184,0.162957526624,0.162951890589,0.162946255081,0.162940620099,0.162934985642,0.162929351712,0.162923718307,0.162918085428,0.162912453075,0.162906821247,0.162901189944,0.162895559168,0.162889928916,0.16288429919,0.162878669989,0.162873041314,0.162867413163,0.162861785538,0.162856158438,0.162850531863,0.162844905812,0.162839280287,0.162833655286,0.16282803081,0.162822406858,0.162816783431,0.162811160529,0.162805538151,0.162799916298,0.162794294969,0.162788674164,0.162783053883,0.162777434127,0.162771814894,0.162766196186,0.162760578001,0.162754960341,0.162749343204,0.162743726591,0.162738110502,0.162732494937,0.162726879895,0.162721265376,0.162715651381,0.16271003791,0.162704424961,0.162698812536,0.162693200635,0.162687589256,0.162681978401,0.162676368068,0.162670758258,0.162665148972,0.162659540208,0.162653931967,0.162648324249,0.162642717053,0.16263711038,0.162631504229,0.162625898601,0.162620293495,0.162614688912,0.162609084851,0.162603481312,0.162597878295,0.1625922758,0.162586673828,0.162581072377,0.162575471448,0.162569871041,0.162564271156,0.162558671792,0.16255307295,0.16254747463,0.162541876831,0.162536279554,0.162530682798,0.162525086563,0.16251949085,0.162513895658,0.162508300987,0.162502706837,0.162497113208,0.1624915201,0.162485927513,0.162480335446,0.162474743901,0.162469152876,0.162463562372,0.162457972388,0.162452382925,0.162446793983,0.162441205561,0.162435617659,0.162430030277,0.162424443416,0.162418857074,0.162413271253,0.162407685952,0.162402101171,0.162396516909,0.162390933168,0.162385349946,0.162379767244,0.162374185062,0.162368603399,0.162363022256,0.162357441632,0.162351861527,0.162346281942,0.162340702876,0.162335124329,0.162329546302,0.162323968793,0.162318391804,0.162312815333,0.162307239382,0.162301663949,0.162296089035,0.16229051464,0.162284940763,0.162279367405,0.162273794565,0.162268222244,0.162262650441,0.162257079157,0.16225150839,0.162245938142,0.162240368413,0.162234799201,0.162229230507,0.162223662331,0.162218094673,0.162212527533,0.162206960911,0.162201394806,0.162195829219,0.16219026415,0.162184699598,0.162179135563,0.162173572046,0.162168009046,0.162162446564,0.162156884599,0.16215132315,0.162145762219,0.162140201805,0.162134641908,0.162129082528,0.162123523664,0.162117965318,0.162112407488,0.162106850174,0.162101293378,0.162095737097,0.162090181334,0.162084626086,0.162079071355,0.16207351714,0.162067963442,0.162062410259,0.162056857593,0.162051305442,0.162045753808,0.162040202689,0.162034652087,0.162029102,0.162023552428,0.162018003373,0.162012454833,0.162006906808,0.162001359299,0.161995812305,0.161990265827,0.161984719864,0.161979174416,0.161973629483,0.161968085065,0.161962541163,0.161956997775,0.161951454902,0.161945912544,0.161940370701,0.161934829372,0.161929288558,0.161923748259,0.161918208474,0.161912669204,0.161907130448,0.161901592206,0.161896054478,0.161890517265,0.161884980566,0.161879444381,0.16187390871,0.161868373553,0.16186283891,0.161857304781,0.161851771165,0.161846238063,0.161840705475,0.1618351734,0.161829641839,0.161824110792,0.161818580257,0.161813050236,0.161807520729,0.161801991734,0.161796463253,0.161790935285,0.161785407829,0.161779880887,0.161774354458,0.161768828541,0.161763303137,0.161757778246,0.161752253868,0.161746730002,0.161741206649,0.161735683808,0.161730161479,0.161724639663,0.161719118359,0.161713597567,0.161708077288,0.16170255752,0.161697038265,0.161691519521,0.16168600129,0.16168048357,0.161674966362,0.161669449666,0.161663933481,0.161658417808,0.161652902646,0.161647387996,0.161641873858,0.16163636023,0.161630847114,0.161625334509,0.161619822416,0.161614310833,0.161608799761,0.161603289201,0.161597779151,0.161592269612,0.161586760584,0.161581252066,0.16157574406,0.161570236564,0.161564729578,0.161559223103,0.161553717138,0.161548211683,0.161542706739,0.161537202305,0.161531698381,0.161526194968,0.161520692064,0.16151518967,0.161509687786,0.161504186412,0.161498685548,0.161493185194,0.161487685349,0.161482186013,0.161476687188,0.161471188871,0.161465691064,0.161460193767,0.161454696979,0.1614492007,0.16144370493,0.161438209669,0.161432714917,0.161427220675,0.161421726941,0.161416233716,0.161410740999,0.161405248792,0.161399757093,0.161394265903,0.161388775221,0.161383285048,0.161377795383,0.161372306227,0.161366817578,0.161361329438,0.161355841807,0.161350354683,0.161344868067,0.161339381959,0.16133389636,0.161328411268,0.161322926684,0.161317442607,0.161311959038,0.161306475977,0.161300993424,0.161295511377,0.161290029839,0.161284548807,0.161279068283,0.161273588266,0.161268108757,0.161262629754,0.161257151259,0.16125167327,0.161246195789,0.161240718814,0.161235242346,0.161229766385,0.16122429093,0.161218815982,0.161213341541,0.161207867606,0.161202394177,0.161196921255,0.161191448839,0.16118597693,0.161180505526,0.161175034629,0.161169564238,0.161164094353,0.161158624973,0.1611531561,0.161147687732,0.16114221987,0.161136752514,0.161131285663,0.161125819318,0.161120353479,0.161114888145,0.161109423316,0.161103958992,0.161098495174,0.161093031861,0.161087569053,0.16108210675,0.161076644952,0.161071183659,0.161065722871,0.161060262588,0.16105480281,0.161049343536,0.161043884767,0.161038426502,0.161032968742,0.161027511486,0.161022054735,0.161016598488,0.161011142745,0.161005687507,0.161000232773,0.160994778542,0.160989324816,0.160983871594,0.160978418875,0.16097296666,0.16096751495,0.160962063742,0.160956613039,0.160951162839,0.160945713142,0.160940263949,0.16093481526,0.160929367073,0.16092391939,0.16091847221,0.160913025534,0.16090757936,0.160902133689,0.160896688522,0.160891243857,0.160885799695,0.160880356036,0.160874912879,0.160869470226,0.160864028074,0.160858586426,0.160853145279,0.160847704636,0.160842264494,0.160836824855,0.160831385718,0.160825947083,0.16082050895,0.160815071319,0.160809634191,0.160804197564,0.160798761439,0.160793325816,0.160787890694,0.160782456074,0.160777021956,0.160771588339,0.160766155224,0.16076072261,0.160755290497,0.160749858886,0.160744427776,0.160738997167,0.16073356706,0.160728137453,0.160722708347,0.160717279742,0.160711851638,0.160706424035,0.160700996933,0.160695570331,0.16069014423,0.160684718629,0.160679293529,0.160673868929,0.16066844483,0.160663021231,0.160657598132,0.160652175534,0.160646753435,0.160641331837,0.160635910738,0.16063049014,0.160625070041,0.160619650442,0.160614231343,0.160608812743,0.160603394644,0.160597977043,0.160592559943,0.160587143341,0.160581727239,0.160576311637,0.160570896533,0.160565481929,0.160560067824,0.160554654218,0.160549241111,0.160543828503,0.160538416394,0.160533004783,0.160527593672,0.160522183059,0.160516772944,0.160511363329,0.160505954212,0.160500545593,0.160495137472,0.16048972985,0.160484322727,0.160478916101,0.160473509974,0.160468104344,0.160462699213,0.16045729458,0.160451890444,0.160446486807,0.160441083667,0.160435681025,0.16043027888,0.160424877233,0.160419476084,0.160414075432,0.160408675277,0.16040327562,0.16039787646,0.160392477798,0.160387079632,0.160381681964,0.160376284792,0.160370888118,0.16036549194,0.16036009626,0.160354701076,0.160349306389,0.160343912198,0.160338518504,0.160333125307,0.160327732606,0.160322340402,0.160316948694,0.160311557482,0.160306166766,0.160300776547,0.160295386824,0.160289997597,0.160284608865,0.16027922063,0.160273832891,0.160268445647,0.160263058899,0.160257672647,0.160252286891,0.16024690163,0.160241516864,0.160236132594,0.160230748819,0.16022536554,0.160219982756,0.160214600467,0.160209218673,0.160203837374,0.160198456571,0.160193076262,0.160187696448,0.160182317129,0.160176938305,0.160171559975,0.16016618214,0.1601608048,0.160155427954,0.160150051603,0.160144675746,0.160139300383,0.160133925514,0.16012855114,0.16012317726,0.160117803874,0.160112430982,0.160107058584,0.16010168668,0.16009631527,0.160090944354,0.160085573931,0.160080204002,0.160074834566,0.160069465624,0.160064097176,0.160058729221,0.160053361759,0.160047994791,0.160042628316,0.160037262334,0.160031896845,0.160026531849,0.160021167346,0.160015803336,0.160010439819,0.160005076795,0.159999714263,0.159994352224,0.159988990678,0.159983629624,0.159978269063,0.159972908994,0.159967549418,0.159962190334,0.159956831742,0.159951473643,0.159946116035,0.15994075892,0.159935402296,0.159930046165,0.159924690525,0.159919335377,0.159913980721,0.159908626557,0.159903272884,0.159897919703,0.159892567014,0.159887214815,0.159881863109,0.159876511893,0.159871161169,0.159865810936,0.159860461195,0.159855111944,0.159849763184,0.159844414916,0.159839067138,0.159833719851,0.159828373055,0.159823026749,0.159817680934,0.15981233561,0.159806990777,0.159801646434,0.159796302581,0.159790959218,0.159785616346,0.159780273965,0.159774932073,0.159769590671,0.15976424976,0.159758909338,0.159753569407,0.159748229965,0.159742891013,0.159737552551,0.159732214578,0.159726877096,0.159721540102,0.159716203598,0.159710867584,0.159705532059,0.159700197023,0.159694862477,0.15968952842,0.159684194852,0.159678861773,0.159673529183,0.159668197082,0.15966286547,0.159657534346,0.159652203712,0.159646873566,0.159641543909,0.15963621474,0.15963088606,0.159625557869,0.159620230166,0.159614902951,0.159609576224,0.159604249986,0.159598924236,0.159593598974,0.1595882742,0.159582949913,0.159577626115,0.159572302805,0.159566979983,0.159561657648,0.159556335801,0.159551014441,0.159545693569,0.159540373185,0.159535053288,0.159529733878,0.159524414956,0.159519096521,0.159513778573,0.159508461113,0.159503144139,0.159497827652,0.159492511652,0.15948719614,0.159481881114,0.159476566574,0.159471252522,0.159465938956,0.159460625877,0.159455313284,0.159450001178,0.159444689558,0.159439378424,0.159434067777,0.159428757616,0.159423447941,0.159418138752,0.159412830049,0.159407521832,0.159402214101,0.159396906856,0.159391600097,0.159386293823,0.159380988035,0.159375682733,0.159370377916,0.159365073585,0.159359769739,0.159354466379,0.159349163503,0.159343861114,0.159338559209,0.159333257789,0.159327956855,0.159322656405,0.15931735644,0.159312056961,0.159306757966,0.159301459456,0.15929616143,0.159290863889,0.159285566833,0.159280270261,0.159274974174,0.159269678571,0.159264383453,0.159259088819,0.159253794669,0.159248501003,0.159243207821,0.159237915123,0.159232622909,0.159227331179,0.159222039933,0.159216749171,0.159211458893,0.159206169098,0.159200879787,0.159195590959,0.159190302615,0.159185014754,0.159179727377,0.159174440483,0.159169154072,0.159163868144,0.159158582699,0.159153297738,0.15914801326,0.159142729264,0.159137445751,0.159132162722,0.159126880174,0.15912159811,0.159116316528,0.159111035429,0.159105754813,0.159100474679,0.159095195027,0.159089915857,0.15908463717,0.159079358965,0.159074081242,0.159068804002,0.159063527243,0.159058250967,0.159052975172,0.159047699859,0.159042425028,0.159037150678,0.159031876811,0.159026603425,0.15902133052,0.159016058097,0.159010786156,0.159005514695,0.159000243716,0.158994973219,0.158989703202,0.158984433667,0.158979164613,0.158973896039,0.158968627947,0.158963360336,0.158958093205,0.158952826555,0.158947560386,0.158942294698,0.15893702949,0.158931764762,0.158926500516,0.158921236749,0.158915973463,0.158910710657,0.158905448332,0.158900186486,0.158894925121,0.158889664235,0.15888440383,0.158879143905,0.158873884459,0.158868625493,0.158863367008,0.158858109001,0.158852851475,0.158847594427,0.15884233786,0.158837081772,0.158831826163,0.158826571033,0.158821316383,0.158816062212,0.15881080852,0.158805555308,0.158800302574,0.158795050319,0.158789798543,0.158784547246,0.158779296428,0.158774046089,0.158768796228,0.158763546845,0.158758297942,0.158753049516,0.15874780157,0.158742554101,0.158737307111,0.158732060599,0.158726814565,0.15872156901,0.158716323932,0.158711079333,0.158705835211,0.158700591567,0.158695348401,0.158690105713,0.158684863503,0.15867962177,0.158674380515,0.158669139737,0.158663899437,0.158658659614,0.158653420268,0.1586481814,0.158642943009,0.158637705095,0.158632467658,0.158627230699,0.158621994216,0.15861675821,0.158611522681,0.158606287629,0.158601053053,0.158595818955,0.158590585333,0.158585352187,0.158580119518,0.158574887325,0.158569655609,0.158564424369,0.158559193605,0.158553963318,0.158548733507,0.158543504171,0.158538275312,0.158533046929,0.158527819022,0.15852259159,0.158517364634,0.158512138154,0.15850691215,0.158501686621,0.158496461568,0.15849123699,0.158486012888,0.158480789261,0.158475566109,0.158470343433,0.158465121232,0.158459899505,0.158454678254,0.158449457478,0.158444237177,0.158439017351,0.158433798,0.158428579123,0.158423360721,0.158418142794,0.158412925341,0.158407708363,0.158402491859,0.15839727583,0.158392060275,0.158386845194,0.158381630588,0.158376416456,0.158371202797,0.158365989613,0.158360776903,0.158355564667,0.158350352904,0.158345141616,0.158339930801,0.15833472046,0.158329510593,0.158324301199,0.158319092278,0.158313883831,0.158308675858,0.158303468357,0.15829826133,0.158293054777,0.158287848696,0.158282643089,0.158277437954,0.158272233293,0.158267029104,0.158261825388,0.158256622145,0.158251419375,0.158246217078,0.158241015253,0.1582358139,0.15823061302,0.158225412613,0.158220212678,0.158215013215,0.158209814225,0.158204615706,0.15819941766,0.158194220086,0.158189022984,0.158183826354,0.158178630196,0.158173434509,0.158168239295,0.158163044552,0.158157850281,0.158152656481,0.158147463153,0.158142270296,0.158137077911,0.158131885997,0.158126694555,0.158121503584,0.158116313084,0.158111123055,0.158105933497,0.15810074441,0.158095555794,0.158090367649,0.158085179975,0.158079992771,0.158074806039,0.158069619777,0.158064433985,0.158059248664,0.158054063814,0.158048879434,0.158043695524,0.158038512084,0.158033329115,0.158028146616,0.158022964587,0.158017783029,0.15801260194,0.158007421321,0.158002241172,0.157997061493,0.157991882283,0.157986703543,0.157981525273,0.157976347473,0.157971170142,0.15796599328,0.157960816888,0.157955640965,0.157950465512,0.157945290527,0.157940116012,0.157934941966,0.157929768389,0.157924595281,0.157919422642,0.157914250472,0.157909078771,0.157903907538,0.157898736774,0.157893566479,0.157888396652,0.157883227294,0.157878058404,0.157872889983,0.15786772203,0.157862554545,0.157857387529,0.15785222098,0.1578470549,0.157841889288,0.157836724143,0.157831559467,0.157826395259,0.157821231518,0.157816068245,0.15781090544,0.157805743102,0.157800581232,0.15779541983,0.157790258895,0.157785098427,0.157779938427,0.157774778893,0.157769619828,0.157764461229,0.157759303097,0.157754145433,0.157748988235,0.157743831504,0.15773867524,0.157733519443,0.157728364113,0.157723209249,0.157718054852,0.157712900922,0.157707747458,0.157702594461,0.15769744193,0.157692289865,0.157687138266,0.157681987134,0.157676836468,0.157671686268,0.157666536534,0.157661387266,0.157656238463,0.157651090127,0.157645942257,0.157640794852,0.157635647913,0.157630501439,0.157625355431,0.157620209889,0.157615064812,0.1576099202,0.157604776054,0.157599632373,0.157594489157,0.157589346407,0.157584204121,0.157579062301,0.157573920945,0.157568780054,0.157563639629,0.157558499668,0.157553360171,0.15754822114,0.157543082573,0.15753794447,0.157532806832,0.157527669659,0.15752253295,0.157517396705,0.157512260925,0.157507125608,0.157501990756,0.157496856368,0.157491722444,0.157486588984,0.157481455988,0.157476323455,0.157471191387,0.157466059782,0.157460928641,0.157455797963,0.15745066775,0.157445537999,0.157440408712,0.157435279888,0.157430151528,0.157425023631,0.157419896197,0.157414769227,0.157409642719,0.157404516674,0.157399391093,0.157394265974,0.157389141318,0.157384017125,0.157378893395,0.157373770127,0.157368647322,0.15736352498,0.1573584031,0.157353281683,0.157348160727,0.157343040235,0.157337920204,0.157332800636,0.15732768153,0.157322562886,0.157317444704,0.157312326984,0.157307209725,0.157302092929,0.157296976595,0.157291860722,0.157286745311,0.157281630361,0.157276515873,0.157271401847,0.157266288282,0.157261175179,0.157256062536,0.157250950355,0.157245838636,0.157240727377,0.15723561658,0.157230506243,0.157225396368,0.157220286953,0.157215177999,0.157210069506,0.157204961474,0.157199853903,0.157194746792,0.157189640142,0.157184533952,0.157179428223,0.157174322954,0.157169218145,0.157164113797,0.157159009909,0.157153906481,0.157148803513,0.157143701005,0.157138598957,0.15713349737,0.157128396242,0.157123295573,0.157118195365,0.157113095616,0.157107996327,0.157102897497,0.157097799127,0.157092701217,0.157087603766,0.157082506774,0.157077410241,0.157072314168,0.157067218554,0.157062123399,0.157057028703,0.157051934466,0.157046840688,0.157041747369,0.157036654508,0.157031562107,0.157026470164,0.157021378679,0.157016287654,0.157011197087,0.157006106978,0.157001017328,0.156995928136,0.156990839402,0.156985751127,0.156980663309,0.15697557595,0.156970489049,0.156965402606,0.156960316621,0.156955231094,0.156950146024,0.156945061413,0.156939977259,0.156934893563,0.156929810324,0.156924727543,0.156919645219,0.156914563353,0.156909481944,0.156904400992,0.156899320498,0.156894240461,0.156889160881,0.156884081758,0.156879003092,0.156873924883,0.156868847131,0.156863769836,0.156858692997,0.156853616616,0.15684854069,0.156843465222,0.15683839021,0.156833315655,0.156828241556,0.156823167913,0.156818094727,0.156813021997,0.156807949723,0.156802877905,0.156797806543,0.156792735638,0.156787665188,0.156782595195,0.156777525657,0.156772456575,0.156767387948,0.156762319778,0.156757252063,0.156752184803,0.156747117999,0.156742051651,0.156736985757,0.15673192032,0.156726855337,0.15672179081,0.156716726738,0.15671166312,0.156706599958,0.156701537251,0.156696474999,0.156691413202,0.15668635186,0.156681290972,0.156676230539,0.15667117056,0.156666111037,0.156661051967,0.156655993353,0.156650935192,0.156645877486,0.156640820234,0.156635763437,0.156630707093,0.156625651204,0.156620595769,0.156615540788,0.156610486261,0.156605432187,0.156600378568,0.156595325402,0.15659027269,0.156585220431,0.156580168627,0.156575117275,0.156570066377,0.156565015933,0.156559965942,0.156554916404,0.15654986732,0.156544818689,0.15653977051,0.156534722785,0.156529675513,0.156524628694,0.156519582328,0.156514536415,0.156509490954,0.156504445946,0.156499401391,0.156494357289,0.156489313639,0.156484270441,0.156479227696,0.156474185403,0.156469143563,0.156464102175,0.156459061239,0.156454020755,0.156448980724,0.156443941144,0.156438902017,0.156433863341,0.156428825117,0.156423787345,0.156418750025,0.156413713156,0.156408676739,0.156403640774,0.15639860526,0.156393570198,0.156388535587,0.156383501427,0.156378467719,0.156373434462,0.156368401656,0.156363369301,0.156358337397,0.156353305944,0.156348274943,0.156343244392,0.156338214292,0.156333184642,0.156328155444,0.156323126696,0.156318098398,0.156313070552,0.156308043155,0.156303016209,0.156297989714,0.156292963669,0.156287938074,0.156282912929,0.156277888234,0.15627286399,0.156267840195,0.156262816851,0.156257793956,0.156252771511,0.156247749516,0.156242727971,0.156237706875,0.156232686229,0.156227666033,0.156222646286,0.156217626989,0.156212608141,0.156207589742,0.156202571793,0.156197554293,0.156192537241,0.15618752064,0.156182504487,0.156177488783,0.156172473528,0.156167458722,0.156162444365,0.156157430456,0.156152416996,0.156147403985,0.156142391423,0.156137379309,0.156132367643,0.156127356426,0.156122345658,0.156117335337,0.156112325465,0.156107316041,0.156102307065,0.156097298537,0.156092290458,0.156087282826,0.156082275642,0.156077268906,0.156072262618,0.156067256777,0.156062251384,0.156057246439,0.156052241942,0.156047237892,0.156042234289,0.156037231134,0.156032228426,0.156027226165,0.156022224352,0.156017222985,0.156012222066,0.156007221594,0.156002221569,0.155997221991,0.155992222859,0.155987224175,0.155982225937,0.155977228146,0.155972230802,0.155967233904,0.155962237453,0.155957241448,0.15595224589,0.155947250778,0.155942256112,0.155937261893,0.155932268119,0.155927274792,0.155922281911,0.155917289476,0.155912297487,0.155907305944,0.155902314847,0.155897324196,0.15589233399,0.15588734423,0.155882354915,0.155877366047,0.155872377623,0.155867389645,0.155862402113,0.155857415026,0.155852428384,0.155847442187,0.155842456436,0.15583747113,0.155832486268,0.155827501852,0.155822517881,0.155817534354,0.155812551273,0.155807568636,0.155802586444,0.155797604696,0.155792623393,0.155787642535,0.155782662121,0.155777682152,0.155772702627,0.155767723546,0.155762744909,0.155757766717,0.155752788969,0.155747811665,0.155742834805,0.155737858389,0.155732882417,0.155727906889,0.155722931805,0.155717957164,0.155712982967,0.155708009214,0.155703035904,0.155698063038,0.155693090615,0.155688118636,0.1556831471,0.155678176007,0.155673205357,0.155668235151,0.155663265388,0.155658296068,0.155653327191,0.155648358757,0.155643390766,0.155638423217,0.155633456112,0.155628489449,0.155623523229,0.155618557451,0.155613592116,0.155608627224,0.155603662774,0.155598698766,0.155593735201,0.155588772078,0.155583809397,0.155578847159,0.155573885362,0.155568924008,0.155563963095,0.155559002625,0.155554042596,0.155549083009,0.155544123864,0.155539165161,0.155534206899,0.155529249079,0.155524291701,0.155519334764,0.155514378268,0.155509422214,0.155504466601,0.155499511429,0.155494556699,0.15548960241,0.155484648561,0.155479695154,0.155474742188,0.155469789663,0.155464837578,0.155459885935,0.155454934732,0.15544998397,0.155445033648,0.155440083768,0.155435134327,0.155430185327,0.155425236768,0.155420288649,0.15541534097,0.155410393731,0.155405446933,0.155400500575,0.155395554657,0.155390609179,0.15538566414,0.155380719542,0.155375775384,0.155370831665,0.155365888386,0.155360945547,0.155356003148,0.155351061188,0.155346119667,0.155341178586,0.155336237945,0.155331297742,0.15532635798,0.155321418656,0.155316479771,0.155311541326,0.15530660332,0.155301665752,0.155296728624,0.155291791934,0.155286855684,0.155281919872,0.155276984499,0.155272049564,0.155267115069,0.155262181011,0.155257247393,0.155252314212,0.15524738147,0.155242449167,0.155237517301,0.155232585874,0.155227654885,0.155222724335,0.155217794222,0.155212864547,0.15520793531,0.155203006512,0.155198078151,0.155193150227,0.155188222742,0.155183295694,0.155178369084,0.155173442911,0.155168517176,0.155163591878,0.155158667017,0.155153742594,0.155148818609,0.15514389506,0.155138971948,0.155134049274,0.155129127037,0.155124205237,0.155119283873,0.155114362947,0.155109442457,0.155104522404,0.155099602788,0.155094683609,0.155089764866,0.155084846559,0.15507992869,0.155075011256,0.155070094259,0.155065177698,0.155060261574,0.155055345886,0.155050430634,0.155045515818,0.155040601438,0.155035687494,0.155030773986,0.155025860914,0.155020948277,0.155016036077,0.155011124312,0.155006212983,0.155001302089,0.154996391631,0.154991481609,0.154986572021,0.15498166287,0.154976754153,0.154971845872,0.154966938026,0.154962030615,0.154957123639,0.154952217099,0.154947310993,0.154942405322,0.154937500086,0.154932595285,0.154927690919,0.154922786987,0.15491788349,0.154912980428,0.1549080778,0.154903175607,0.154898273848,0.154893372523,0.154888471633,0.154883571177,0.154878671155,0.154873771568,0.154868872414,0.154863973695,0.154859075409,0.154854177558,0.15484928014,0.154844383156,0.154839486606,0.154834590489,0.154829694807,0.154824799558,0.154819904742,0.15481501036,0.154810116411,0.154805222896,0.154800329814,0.154795437165,0.154790544949,0.154785653167,0.154780761817,0.154775870901,0.154770980418,0.154766090367,0.15476120075,0.154756311565,0.154751422813,0.154746534494,0.154741646608,0.154736759154,0.154731872132,0.154726985543,0.154722099387,0.154717213663,0.154712328371,0.154707443511,0.154702559084,0.154697675089,0.154692791525,0.154687908394,0.154683025695,0.154678143428,0.154673261593,0.154668380189,0.154663499218,0.154658618678,0.154653738569,0.154648858893,0.154643979647,0.154639100834,0.154634222452,0.154629344501,0.154624466981,0.154619589893,0.154614713236,0.15460983701,0.154604961215,0.154600085851,0.154595210918,0.154590336417,0.154585462346,0.154580588705,0.154575715496,0.154570842717,0.154565970369,0.154561098452,0.154556226965,0.154551355909,0.154546485283,0.154541615087,0.154536745322,0.154531875987,0.154527007083,0.154522138608,0.154517270564,0.154512402949,0.154507535765,0.15450266901,0.154497802686,0.154492936791,0.154488071326,0.154483206291,0.154478341685,0.154473477509,0.154468613763,0.154463750446,0.154458887559,0.154454025101,0.154449163072,0.154444301473,0.154439440302,0.154434579561,0.154429719249,0.154424859367,0.154419999913,0.154415140888,0.154410282292,0.154405424125,0.154400566386,0.154395709077,0.154390852196,0.154385995744,0.15438113972,0.154376284125,0.154371428958,0.154366574219,0.154361719909,0.154356866028,0.154352012574,0.154347159549,0.154342306952,0.154337454783,0.154332603041,0.154327751728,0.154322900843,0.154318050386,0.154313200356,0.154308350754,0.15430350158,0.154298652834,0.154293804515,0.154288956623,0.15428410916,0.154279262123,0.154274415514,0.154269569332,0.154264723577,0.15425987825,0.15425503335,0.154250188877,0.154245344831,0.154240501211,0.154235658019,0.154230815254,0.154225972915,0.154221131003,0.154216289518,0.15421144846,0.154206607828,0.154201767623,0.154196927844,0.154192088492,0.154187249566,0.154182411066,0.154177572992,0.154172735345,0.154167898124,0.154163061329,0.15415822496,0.154153389017,0.1541485535,0.154143718409,0.154138883744,0.154134049505,0.154129215691,0.154124382303,0.15411954934,0.154114716803,0.154109884692,0.154105053006,0.154100221745,0.15409539091,0.1540905605,0.154085730515,0.154080900956,0.154076071821,0.154071243112,0.154066414828,0.154061586968,0.154056759534,0.154051932524,0.154047105939,0.154042279779,0.154037454043,0.154032628733,0.154027803846,0.154022979385,0.154018155347,0.154013331735,0.154008508546,0.154003685782,0.153998863442,0.153994041526,0.153989220035,0.153984398967,0.153979578324,0.153974758104,0.153969938309,0.153965118937,0.153960299989,0.153955481465,0.153950663364,0.153945845688,0.153941028434,0.153936211605,0.153931395199,0.153926579216,0.153921763656,0.15391694852,0.153912133808,0.153907319518,0.153902505652,0.153897692208,0.153892879188,0.153888066591,0.153883254417,0.153878442665,0.153873631337,0.153868820431,0.153864009948,0.153859199888,0.15385439025,0.153849581035,0.153844772242,0.153839963872,0.153835155924,0.153830348399,0.153825541295,0.153820734614,0.153815928356,0.153811122519,0.153806317105,0.153801512112,0.153796707541,0.153791903393,0.153787099666,0.153782296361,0.153777493478,0.153772691016,0.153767888977,0.153763087358,0.153758286162,0.153753485386,0.153748685032,0.1537438851,0.153739085589,0.153734286499,0.15372948783,0.153724689583,0.153719891757,0.153715094351,0.153710297367,0.153705500803,0.153700704661,0.153695908939,0.153691113638,0.153686318758,0.153681524299,0.15367673026,0.153671936641,0.153667143444,0.153662350666,0.153657558309,0.153652766373,0.153647974856,0.15364318376,0.153638393085,0.153633602829,0.153628812993,0.153624023577,0.153619234582,0.153614446006,0.15360965785,0.153604870114,0.153600082798,0.153595295901,0.153590509424,0.153585723367,0.153580937729,0.153576152511,0.153571367712,0.153566583332,0.153561799372,0.153557015831,0.153552232709,0.153547450007,0.153542667723,0.153537885859,0.153533104413,0.153528323387,0.153523542779,0.153518762591,0.153513982821,0.15350920347,0.153504424537,0.153499646023,0.153494867928,0.153490090251,0.153485312993,0.153480536153,0.153475759731,0.153470983728,0.153466208143,0.153461432976,0.153456658228,0.153451883897,0.153447109984,0.15344233649,0.153437563413,0.153432790754,0.153428018514,0.15342324669,0.153418475285,0.153413704297,0.153408933727,0.153404163574,0.153399393839,0.153394624521,0.153389855621,0.153385087138,0.153380319073,0.153375551424,0.153370784193,0.153366017379,0.153361250982,0.153356485002,0.153351719439,0.153346954292,0.153342189563,0.153337425251,0.153332661355,0.153327897876,0.153323134813,0.153318372167,0.153313609938,0.153308848125,0.153304086729,0.153299325749,0.153294565185,0.153289805038,0.153285045307,0.153280285992,0.153275527093,0.15327076861,0.153266010543,0.153261252892,0.153256495657,0.153251738838,0.153246982435,0.153242226447,0.153237470875,0.153232715719,0.153227960978,0.153223206653,0.153218452743,0.153213699248,0.153208946169,0.153204193506,0.153199441257,0.153194689424,0.153189938006,0.153185187003,0.153180436415,0.153175686242,0.153170936484,0.153166187141,0.153161438213,0.153156689699,0.1531519416,0.153147193916,0.153142446647,0.153137699792,0.153132953351,0.153128207325,0.153123461714,0.153118716516,0.153113971734,0.153109227365,0.15310448341,0.15309973987,0.153094996744,0.153090254031,0.153085511733,0.153080769849,0.153076028378,0.153071287322,0.153066546679,0.15306180645,0.153057066634,0.153052327232,0.153047588244,0.153042849669,0.153038111508,0.15303337376,0.153028636425,0.153023899504,0.153019162995,0.153014426901,0.153009691219,0.15300495595,0.153000221094,0.152995486651,0.152990752622,0.152986019005,0.1529812858,0.152976553009,0.15297182063,0.152967088664,0.152962357111,0.15295762597,0.152952895242,0.152948164926,0.152943435022,0.152938705531,0.152933976452,0.152929247785,0.152924519531,0.152919791688,0.152915064258,0.152910337239,0.152905610633,0.152900884439,0.152896158656,0.152891433285,0.152886708326,0.152881983779,0.152877259643,0.152872535919,0.152867812606,0.152863089705,0.152858367216,0.152853645137,0.152848923471,0.152844202215,0.15283948137,0.152834760937,0.152830040915,0.152825321304,0.152820602104,0.152815883315,0.152811164937,0.152806446969,0.152801729413,0.152797012267,0.152792295532,0.152787579208,0.152782863294,0.152778147791,0.152773432698,0.152768718015,0.152764003743,0.152759289882,0.15275457643,0.152749863389,0.152745150758,0.152740438537,0.152735726727,0.152731015326,0.152726304335,0.152721593754,0.152716883583,0.152712173822,0.15270746447,0.152702755528,0.152698046996,0.152693338874,0.152688631161,0.152683923857,0.152679216963,0.152674510478,0.152669804403,0.152665098737,0.15266039348,0.152655688632,0.152650984194,0.152646280164,0.152641576543,0.152636873332,0.152632170529,0.152627468135,0.152622766151,0.152618064574,0.152613363407,0.152608662648,0.152603962298,0.152599262356,0.152594562823,0.152589863698,0.152585164981,0.152580466673,0.152575768773,0.152571071282,0.152566374198,0.152561677523,0.152556981256,0.152552285397,0.152547589945,0.152542894902,0.152538200267,0.152533506039,0.152528812219,0.152524118807,0.152519425802,0.152514733206,0.152510041016,0.152505349234,0.15250065786,0.152495966893,0.152491276333,0.152486586181,0.152481896435,0.152477207097,0.152472518166,0.152467829643,0.152463141526,0.152458453816,0.152453766513,0.152449079617,0.152444393128,0.152439707046,0.15243502137,0.152430336101,0.152425651238,0.152420966782,0.152416282733,0.15241159909,0.152406915854,0.152402233023,0.152397550599,0.152392868582,0.15238818697,0.152383505765,0.152378824966,0.152374144572,0.152369464585,0.152364785004,0.152360105829,0.152355427059,0.152350748695,0.152346070737,0.152341393185,0.152336716038,0.152332039296,0.152327362961,0.15232268703,0.152318011506,0.152313336386,0.152308661672,0.152303987363,0.152299313459,0.152294639961,0.152289966867,0.152285294179,0.152280621895,0.152275950017,0.152271278543,0.152266607474,0.15226193681,0.152257266551,0.152252596696,0.152247927246,0.152243258201,0.15223858956,0.152233921324,0.152229253492,0.152224586064,0.152219919041,0.152215252422,0.152210586207,0.152205920397,0.15220125499,0.152196589988,0.152191925389,0.152187261195,0.152182597404,0.152177934017,0.152173271034,0.152168608455,0.15216394628,0.152159284508,0.15215462314,0.152149962175,0.152145301614,0.152140641456,0.152135981701,0.15213132235,0.152126663403,0.152122004858,0.152117346717,0.152112688978,0.152108031643,0.152103374711,0.152098718182,0.152094062056,0.152089406332,0.152084751012,0.152080096094,0.152075441579,0.152070787466,0.152066133757,0.152061480449,0.152056827545,0.152052175042,0.152047522942,0.152042871245,0.15203821995,0.152033569057,0.152028918566,0.152024268477,0.152019618791,0.152014969506,0.152010320624,0.152005672143,0.152001024064,0.151996376388,0.151991729113,0.151987082239,0.151982435768,0.151977789698,0.151973144029,0.151968498762,0.151963853897,0.151959209433,0.15195456537,0.151949921709,0.151945278449,0.15194063559,0.151935993132,0.151931351076,0.15192670942,0.151922068166,0.151917427312,0.15191278686,0.151908146808,0.151903507157,0.151898867907,0.151894229057,0.151889590609,0.15188495256,0.151880314913,0.151875677665,0.151871040819,0.151866404372,0.151861768326,0.151857132681,0.151852497435,0.15184786259,0.151843228145,0.1518385941,0.151833960455,0.15182932721,0.151824694365,0.15182006192,0.151815429874,0.151810798229,0.151806166983,0.151801536137,0.15179690569,0.151792275643,0.151787645996,0.151783016748,0.151778387899,0.15177375945,0.1517691314,0.15176450375,0.151759876498,0.151755249646,0.151750623193,0.151745997139,0.151741371484,0.151736746228,0.151732121371,0.151727496912,0.151722872853,0.151718249192,0.15171362593,0.151709003067,0.151704380602,0.151699758536,0.151695136868,0.151690515599,0.151685894728,0.151681274255,0.151676654181,0.151672034505,0.151667415227,0.151662796348,0.151658177866,0.151653559783,0.151648942097,0.15164432481,0.15163970792,0.151635091428,0.151630475334,0.151625859638,0.151621244339,0.151616629438,0.151612014935,0.151607400829,0.15160278712,0.151598173809,0.151593560896,0.151588948379,0.15158433626,0.151579724539,0.151575113214,0.151570502286,0.151565891756,0.151561281623,0.151556671886,0.151552062547,0.151547453604,0.151542845058,0.151538236909,0.151533629157,0.151529021801,0.151524414842,0.15151980828,0.151515202114,0.151510596344,0.151505990971,0.151501385994,0.151496781414,0.15149217723,0.151487573442,0.15148297005,0.151478367054,0.151473764454,0.151469162251,0.151464560443,0.151459959031,0.151455358015,0.151450757395,0.151446157171,0.151441557342,0.151436957909,0.151432358872,0.15142776023,0.151423161983,0.151418564132,0.151413966677,0.151409369617,0.151404772952,0.151400176682,0.151395580808,0.151390985328,0.151386390244,0.151381795555,0.151377201261,0.151372607361,0.151368013857,0.151363420747,0.151358828033,0.151354235713,0.151349643787,0.151345052257,0.151340461121,0.151335870379,0.151331280032,0.151326690079,0.151322100521,0.151317511357,0.151312922587,0.151308334212,0.151303746231,0.151299158644,0.151294571451,0.151289984652,0.151285398247,0.151280812236,0.151276226618,0.151271641395,0.151267056565,0.15126247213,0.151257888087,0.151253304439,0.151248721184,0.151244138322,0.151239555854,0.15123497378,0.151230392099,0.151225810811,0.151221229916,0.151216649415,0.151212069307,0.151207489592,0.15120291027,0.151198331341,0.151193752805,0.151189174661,0.151184596911,0.151180019554,0.151175442589,0.151170866017,0.151166289838,0.151161714051,0.151157138657,0.151152563656,0.151147989047,0.15114341483,0.151138841006,0.151134267574,0.151129694534,0.151125121887,0.151120549631,0.151115977768,0.151111406297,0.151106835218,0.151102264531,0.151097694236,0.151093124332,0.151088554821,0.151083985701,0.151079416973,0.151074848636,0.151070280691,0.151065713138,0.151061145977,0.151056579206,0.151052012827,0.15104744684,0.151042881244,0.151038316039,0.151033751225,0.151029186803,0.151024622771,0.151020059131,0.151015495882,0.151010933023,0.151006370556,0.151001808479,0.150997246793,0.150992685498,0.150988124594,0.15098356408,0.150979003957,0.150974444225,0.150969884883,0.150965325931,0.15096076737,0.150956209199,0.150951651419,0.150947094029,0.150942537029,0.150937980419,0.150933424199,0.15092886837,0.15092431293,0.150919757881,0.150915203221,0.150910648951,0.150906095071,0.150901541581,0.15089698848,0.150892435769,0.150887883448,0.150883331516,0.150878779974,0.150874228821,0.150869678058,0.150865127684,0.150860577699,0.150856028104,0.150851478897,0.15084693008,0.150842381652,0.150837833614,0.150833285964,0.150828738703,0.150824191831,0.150819645348,0.150815099253,0.150810553548,0.150806008231,0.150801463303,0.150796918764,0.150792374613,0.15078783085,0.150783287476,0.150778744491,0.150774201893,0.150769659685,0.150765117864,0.150760576432,0.150756035387,0.150751494731,0.150746954463,0.150742414583,0.150737875091,0.150733335987,0.150728797271,0.150724258942,0.150719721002,0.150715183449,0.150710646284,0.150706109506,0.150701573116,0.150697037113,0.150692501498,0.150687966271,0.15068343143,0.150678896977,0.150674362912,0.150669829233,0.150665295942,0.150660763038,0.150656230521,0.150651698391,0.150647166648,0.150642635291,0.150638104322,0.15063357374,0.150629043544,0.150624513735,0.150619984313,0.150615455277,0.150610926628,0.150606398366,0.15060187049,0.150597343,0.150592815897,0.15058828918,0.15058376285,0.150579236905,0.150574711347,0.150570186175,0.150565661389,0.150561136989,0.150556612975,0.150552089347,0.150547566105,0.150543043249,0.150538520779,0.150533998694,0.150529476995,0.150524955682,0.150520434754,0.150515914212,0.150511394055,0.150506874284,0.150502354898,0.150497835897,0.150493317282,0.150488799052,0.150484281207,0.150479763747,0.150475246673,0.150470729983,0.150466213679,0.150461697759,0.150457182225,0.150452667075,0.15044815231,0.15044363793,0.150439123934,0.150434610323,0.150430097097,0.150425584255,0.150421071798,0.150416559725,0.150412048037,0.150407536733,0.150403025813,0.150398515278,0.150394005127,0.15038949536,0.150384985977,0.150380476978,0.150375968363,0.150371460133,0.150366952286,0.150362444823,0.150357937743,0.150353431048,0.150348924736,0.150344418808,0.150339913264,0.150335408103,0.150330903325,0.150326398932,0.150321894921,0.150317391294,0.15031288805,0.15030838519,0.150303882713,0.150299380619,0.150294878908,0.15029037758,0.150285876635,0.150281376074,0.150276875895,0.150272376099,0.150267876686,0.150263377655,0.150258879008,0.150254380743,0.150249882861,0.150245385361,0.150240888244,0.15023639151,0.150231895157,0.150227399188,0.1502229036,0.150218408395,0.150213913572,0.150209419132,0.150204925073,0.150200431397,0.150195938103,0.15019144519,0.15018695266,0.150182460512,0.150177968745,0.15017347736,0.150168986357,0.150164495736,0.150160005496,0.150155515639,0.150151026162,0.150146537067,0.150142048354,0.150137560022,0.150133072071,0.150128584502,0.150124097314,0.150119610507,0.150115124082,0.150110638037,0.150106152374,0.150101667091,0.15009718219,0.15009269767,0.15008821353,0.150083729771,0.150079246393,0.150074763396,0.15007028078,0.150065798544,0.150061316689,0.150056835214,0.15005235412,0.150047873406,0.150043393073,0.15003891312,0.150034433547,0.150029954355,0.150025475542,0.15002099711,0.150016519058,0.150012041386,0.150007564095,0.150003087183,0.149998610651,0.149994134498,0.149989658726,0.149985183334,0.149980708321,0.149976233688,0.149971759434,0.14996728556,0.149962812066,0.149958338951,0.149953866215,0.149949393859,0.149944921882,0.149940450285,0.149935979066,0.149931508227,0.149927037767,0.149922567687,0.149918097985,0.149913628662,0.149909159718,0.149904691153,0.149900222967,0.14989575516,0.149891287731,0.149886820681,0.14988235401,0.149877887717,0.149873421803,0.149868956268,0.14986449111,0.149860026332,0.149855561931,0.149851097909,0.149846634266,0.149842171,0.149837708113,0.149833245604,0.149828783473,0.149824321719,0.149819860344,0.149815399347,0.149810938728,0.149806478486,0.149802018623,0.149797559137,0.149793100028,0.149788641298,0.149784182945,0.149779724969,0.149775267371,0.149770810151,0.149766353307,0.149761896842,0.149757440753,0.149752985042,0.149748529708,0.149744074751,0.149739620171,0.149735165968,0.149730712143,0.149726258694,0.149721805622,0.149717352927,0.149712900609,0.149708448668,0.149703997103,0.149699545915,0.149695095104,0.149690644669,0.149686194611,0.149681744929,0.149677295624,0.149672846695,0.149668398143,0.149663949966,0.149659502166,0.149655054742,0.149650607695,0.149646161023,0.149641714728,0.149637268808,0.149632823265,0.149628378097,0.149623933305,0.149619488889,0.149615044849,0.149610601185,0.149606157896,0.149601714983,0.149597272445,0.149592830283,0.149588388496,0.149583947085,0.149579506049,0.149575065389,0.149570625103,0.149566185193,0.149561745659,0.149557306499,0.149552867715,0.149548429305,0.149543991271,0.149539553611,0.149535116326,0.149530679417,0.149526242882,0.149521806721,0.149517370936,0.149512935525,0.149508500489,0.149504065827,0.14949963154,0.149495197627,0.149490764089,0.149486330925,0.149481898136,0.14947746572,0.149473033679,0.149468602013,0.14946417072,0.149459739801,0.149455309257,0.149450879086,0.149446449289,0.149442019867,0.149437590818,0.149433162143,0.149428733841,0.149424305914,0.14941987836,0.149415451179,0.149411024373,0.149406597939,0.14940217188,0.149397746193,0.14939332088,0.149388895941,0.149384471374,0.149380047181,0.149375623361,0.149371199914,0.14936677684,0.14936235414,0.149357931812,0.149353509857,0.149349088275,0.149344667066,0.14934024623,0.149335825767,0.149331405676,0.149326985958,0.149322566612,0.14931814764,0.149313729039,0.149309310811,0.149304892956,0.149300475473,0.149296058362,0.149291641623,0.149287225257,0.149282809263,0.149278393641,0.149273978391,0.149269563513,0.149265149007,0.149260734874,0.149256321112,0.149251907721,0.149247494703,0.149243082057,0.149238669782,0.149234257879,0.149229846347,0.149225435187,0.149221024399,0.149216613982,0.149212203936,0.149207794262,0.149203384959,0.149198976027,0.149194567467,0.149190159278,0.14918575146,0.149181344013,0.149176936937,0.149172530232,0.149168123898,0.149163717935,0.149159312343,0.149154907122,0.149150502271,0.149146097791,0.149141693682,0.149137289944,0.149132886576,0.149128483578,0.149124080951,0.149119678695,0.149115276809,0.149110875293,0.149106474147,0.149102073372,0.149097672967,0.149093272932,0.149088873267,0.149084473973,0.149080075048,0.149075676493,0.149071278308,0.149066880493,0.149062483048,0.149058085973,0.149053689267,0.149049292931,0.149044896965,0.149040501368,0.14903610614,0.149031711283,0.149027316794,0.149022922676,0.149018528926,0.149014135546,0.149009742535,0.149005349893,0.14900095762,0.148996565717,0.148992174182,0.148987783017,0.14898339222,0.148979001793,0.148974611734,0.148970222044,0.148965832723,0.148961443771,0.148957055188,0.148952666973,0.148948279126,0.148943891648,0.148939504539,0.148935117798,0.148930731426,0.148926345422,0.148921959786,0.148917574519,0.148913189619,0.148908805088,0.148904420925,0.14890003713,0.148895653704,0.148891270645,0.148886887954,0.148882505631,0.148878123676,0.148873742088,0.148869360869,0.148864980017,0.148860599532,0.148856219416,0.148851839667,0.148847460285,0.148843081271,0.148838702625,0.148834324345,0.148829946433,0.148825568889,0.148821191711,0.148816814901,0.148812438458,0.148808062382,0.148803686673,0.148799311331,0.148794936356,0.148790561748,0.148786187507,0.148781813633,0.148777440125,0.148773066984,0.14876869421,0.148764321802,0.148759949761,0.148755578087,0.148751206779,0.148746835838,0.148742465263,0.148738095054,0.148733725211,0.148729355735,0.148724986625,0.148720617881,0.148716249504,0.148711881492,0.148707513847,0.148703146567,0.148698779653,0.148694413105,0.148690046924,0.148685681107,0.148681315657,0.148676950572,0.148672585853,0.1486682215,0.148663857512,0.148659493889,0.148655130633,0.148650767741,0.148646405215,0.148642043054,0.148637681259,0.148633319828,0.148628958763,0.148624598063,0.148620237729,0.148615877759,0.148611518154,0.148607158914,0.148602800039,0.148598441529,0.148594083384,0.148589725603,0.148585368188,0.148581011137,0.14857665445,0.148572298128,0.148567942171,0.148563586578,0.14855923135,0.148554876486,0.148550521986,0.148546167851,0.14854181408,0.148537460673,0.14853310763,0.148528754952,0.148524402637,0.148520050687,0.1485156991,0.148511347878,0.148506997019,0.148502646524,0.148498296393,0.148493946626,0.148489597222,0.148485248183,0.148480899506,0.148476551194,0.148472203244,0.148467855659,0.148463508436,0.148459161577,0.148454815082,0.14845046895,0.148446123181,0.148441777775,0.148437432732,0.148433088052,0.148428743736,0.148424399782,0.148420056192,0.148415712964,0.148411370099,0.148407027597,0.148402685458,0.148398343682,0.148394002268,0.148389661217,0.148385320528,0.148380980203,0.148376640239,0.148372300638,0.1483679614,0.148363622523,0.148359284009,0.148354945858,0.148350608068,0.148346270641,0.148341933576,0.148337596873,0.148333260532,0.148328924553,0.148324588936,0.148320253681,0.148315918788,0.148311584257,0.148307250087,0.148302916279,0.148298582833,0.148294249748,0.148289917025,0.148285584664,0.148281252664,0.148276921025,0.148272589748,0.148268258832,0.148263928278,0.148259598085,0.148255268253,0.148250938782,0.148246609672,0.148242280924,0.148237952536,0.148233624509,0.148229296844,0.148224969539,0.148220642595,0.148216316012,0.14821198979,0.148207663928,0.148203338427,0.148199013287,0.148194688507,0.148190364088,0.148186040029,0.148181716331,0.148177392993,0.148173070015,0.148168747398,0.148164425141,0.148160103244,0.148155781707,0.148151460531,0.148147139714,0.148142819258,0.148138499161,0.148134179425,0.148129860048,0.148125541031,0.148121222374,0.148116904077,0.14811258614,0.148108268562,0.148103951344,0.148099634485,0.148095317986,0.148091001846,0.148086686066,0.148082370645,0.148078055583,0.148073740881,0.148069426538,0.148065112554,0.14806079893,0.148056485664,0.148052172758,0.14804786021,0.148043548022,0.148039236192,0.148034924722,0.14803061361,0.148026302857,0.148021992463,0.148017682427,0.14801337275,0.148009063432,0.148004754472,0.148000445871,0.147996137628,0.147991829744,0.147987522218,0.147983215051,0.147978908241,0.14797460179,0.147970295698,0.147965989963,0.147961684586,0.147957379568,0.147953074907,0.147948770605,0.14794446666,0.147940163074,0.147935859845,0.147931556974,0.147927254461,0.147922952305,0.147918650507,0.147914349067,0.147910047984,0.147905747259,0.147901446891,0.147897146881,0.147892847228,0.147888547932,0.147884248994,0.147879950413,0.147875652189,0.147871354322,0.147867056813,0.14786275966,0.147858462865,0.147854166426,0.147849870345,0.14784557462,0.147841279252,0.147836984241,0.147832689587,0.147828395289,0.147824101348,0.147819807764,0.147815514536,0.147811221665,0.14780692915,0.147802636992,0.14779834519,0.147794053745,0.147789762655,0.147785471922,0.147781181545,0.147776891525,0.14777260186,0.147768312552,0.147764023599,0.147759735003,0.147755446763,0.147751158878,0.147746871349,0.147742584176,0.147738297359,0.147734010898,0.147729724792,0.147725439042,0.147721153647,0.147716868608,0.147712583924,0.147708299596,0.147704015623,0.147699732006,0.147695448744,0.147691165837,0.147686883285,0.147682601089,0.147678319248,0.147674037761,0.14766975663,0.147665475854,0.147661195433,0.147656915366,0.147652635655,0.147648356298,0.147644077296,0.147639798649,0.147635520357,0.147631242419,0.147626964835,0.147622687607,0.147618410732,0.147614134213,0.147609858047,0.147605582236,0.147601306779,0.147597031677,0.147592756929,0.147588482535,0.147584208495,0.147579934809,0.147575661477,0.147571388499,0.147567115876,0.147562843606,0.14755857169,0.147554300128,0.147550028919,0.147545758065,0.147541487564,0.147537217416,0.147532947622,0.147528678182,0.147524409096,0.147520140362,0.147515871983,0.147511603956,0.147507336283,0.147503068963,0.147498801997,0.147494535384,0.147490269123,0.147486003216,0.147481737662,0.147477472461,0.147473207613,0.147468943118,0.147464678976,0.147460415187,0.14745615175,0.147451888666,0.147447625935,0.147443363557,0.147439101531,0.147434839857,0.147430578537,0.147426317568,0.147422056953,0.147417796689,0.147413536778,0.147409277219,0.147405018013,0.147400759159,0.147396500656,0.147392242507,0.147387984709,0.147383727263,0.147379470169,0.147375213427,0.147370957037,0.147366700999,0.147362445312,0.147358189978,0.147353934995,0.147349680364,0.147345426084,0.147341172156,0.14733691858,0.147332665355,0.147328412481,0.147324159959,0.147319907789,0.147315655969,0.147311404501,0.147307153385,0.147302902619,0.147298652204,0.147294402141,0.147290152429,0.147285903067,0.147281654057,0.147277405398,0.147273157089,0.147268909131,0.147264661525,0.147260414268,0.147256167363,0.147251920808,0.147247674604,0.14724342875,0.147239183247,0.147234938095,0.147230693293,0.147226448841,0.147222204739,0.147217960988,0.147213717587,0.147209474537,0.147205231836,0.147200989486,0.147196747486,0.147192505836,0.147188264535,0.147184023585,0.147179782985,0.147175542734,0.147171302834,0.147167063283,0.147162824081,0.14715858523,0.147154346728,0.147150108576,0.147145870773,0.14714163332,0.147137396216,0.147133159462,0.147128923057,0.147124687001,0.147120451295,0.147116215938,0.14711198093,0.147107746271,0.147103511962,0.147099278001,0.14709504439,0.147090811127,0.147086578214,0.147082345649,0.147078113433,0.147073881566,0.147069650048,0.147065418878,0.147061188057,0.147056957585,0.147052727461,0.147048497686,0.147044268259,0.147040039181,0.147035810451,0.14703158207,0.147027354036,0.147023126352,0.147018899015,0.147014672026,0.147010445386,0.147006219094,0.14700199315,0.146997767554,0.146993542305,0.146989317405,0.146985092853,0.146980868648,0.146976644791,0.146972421282,0.146968198121,0.146963975307,0.146959752841,0.146955530723,0.146951308952,0.146947087528,0.146942866452,0.146938645724,0.146934425342,0.146930205308,0.146925985622,0.146921766282,0.14691754729,0.146913328645,0.146909110347,0.146904892396,0.146900674792,0.146896457534,0.146892240624,0.146888024061,0.146883807844,0.146879591975,0.146875376452,0.146871161276,0.146866946446,0.146862731963,0.146858517826,0.146854304036,0.146850090593,0.146845877496,0.146841664745,0.146837452341,0.146833240283,0.146829028571,0.146824817206,0.146820606186,0.146816395513,0.146812185186,0.146807975205,0.14680376557,0.146799556281,0.146795347337,0.14679113874,0.146786930488,0.146782722582,0.146778515022,0.146774307808,0.146770100939,0.146765894416,0.146761688239,0.146757482406,0.14675327692,0.146749071779,0.146744866983,0.146740662532,0.146736458427,0.146732254667,0.146728051253,0.146723848183,0.146719645459,0.146715443079,0.146711241045,0.146707039356,0.146702838012,0.146698637012,0.146694436358,0.146690236048,0.146686036083,0.146681836463,0.146677637187,0.146673438256,0.14666923967,0.146665041428,0.146660843531,0.146656645978,0.14665244877,0.146648251906,0.146644055386,0.146639859211,0.146635663379,0.146631467893,0.14662727275,0.146623077951,0.146618883497,0.146614689386,0.14661049562,0.146606302197,0.146602109119,0.146597916384,0.146593723993,0.146589531946,0.146585340242,0.146581148883,0.146576957867,0.146572767194,0.146568576865,0.14656438688,0.146560197238,0.146556007939,0.146551818984,0.146547630372,0.146543442104,0.146539254179,0.146535066597,0.146530879358,0.146526692462,0.146522505909,0.1465183197,0.146514133833,0.146509948309,0.146505763129,0.146501578291,0.146497393796,0.146493209643,0.146489025834,0.146484842367,0.146480659243,0.146476476461,0.146472294022,0.146468111926,0.146463930172,0.14645974876,0.146455567691,0.146451386964,0.14644720658,0.146443026537,0.146438846837,0.14643466748,0.146430488464,0.14642630979,0.146422131459,0.146417953469,0.146413775821,0.146409598516,0.146405421552,0.14640124493,0.14639706865,0.146392892711,0.146388717114,0.146384541859,0.146380366946,0.146376192374,0.146372018143,0.146367844254,0.146363670707,0.146359497501,0.146355324636,0.146351152113,0.146346979931,0.14634280809,0.14633863659,0.146334465432,0.146330294614,0.146326124138,0.146321954002,0.146317784208,0.146313614755,0.146309445642,0.14630527687,0.146301108439,0.146296940349,0.1462927726,0.146288605191,0.146284438123,0.146280271395,0.146276105008,0.146271938962,0.146267773256,0.14626360789,0.146259442865,0.14625527818,0.146251113835,0.14624694983,0.146242786166,0.146238622842,0.146234459858,0.146230297214,0.146226134911,0.146221972947,0.146217811323,0.146213650039,0.146209489095,0.14620532849,0.146201168226,0.146197008301,0.146192848716,0.14618868947,0.146184530564,0.146180371998,0.146176213771,0.146172055884,0.146167898336,0.146163741127,0.146159584258,0.146155427728,0.146151271538,0.146147115686,0.146142960174,0.146138805001,0.146134650167,0.146130495672,0.146126341516,0.146122187699,0.146118034221,0.146113881082,0.146109728281,0.14610557582,0.146101423697,0.146097271913,0.146093120467,0.14608896936,0.146084818592,0.146080668162,0.146076518071,0.146072368318,0.146068218904,0.146064069828,0.14605992109,0.146055772691,0.14605162463,0.146047476907,0.146043329522,0.146039182475,0.146035035767,0.146030889396,0.146026743364,0.146022597669,0.146018452312,0.146014307293,0.146010162612,0.146006018269,0.146001874263,0.145997730595,0.145993587265,0.145989444272,0.145985301617,0.1459811593,0.145977017319,0.145972875677,0.145968734371,0.145964593404,0.145960452773,0.145956312479,0.145952172523,0.145948032904,0.145943893622,0.145939754678,0.14593561607,0.145931477799,0.145927339866,0.145923202269,0.145919065009,0.145914928086,0.145910791499,0.14590665525,0.145902519337,0.145898383761,0.145894248521,0.145890113618,0.145885979052,0.145881844822,0.145877710929,0.145873577372,0.145869444151,0.145865311267,0.145861178719,0.145857046507,0.145852914631,0.145848783092,0.145844651888,0.145840521021,0.14583639049,0.145832260295,0.145828130436,0.145824000912,0.145819871725,0.145815742873,0.145811614357,0.145807486177,0.145803358333,0.145799230824,0.145795103651,0.145790976813,0.145786850311,0.145782724145,0.145778598314,0.145774472818,0.145770347658,0.145766222833,0.145762098343,0.145757974188,0.145753850369,0.145749726885,0.145745603736,0.145741480922,0.145737358443,0.145733236299,0.14572911449,0.145724993016,0.145720871876,0.145716751072,0.145712630602,0.145708510467,0.145704390667,0.145700271201,0.14569615207,0.145692033274,0.145687914812,0.145683796685,0.145679678892,0.145675561433,0.145671444309,0.145667327519,0.145663211063,0.145659094942,0.145654979154,0.145650863701,0.145646748582,0.145642633797,0.145638519346,0.145634405229,0.145630291446,0.145626177997,0.145622064882,0.1456179521,0.145613839653,0.145609727539,0.145605615758,0.145601504312,0.145597393199,0.145593282419,0.145589171973,0.145585061861,0.145580952082,0.145576842636,0.145572733524,0.145568624745,0.145564516299,0.145560408186,0.145556300407,0.145552192961,0.145548085848,0.145543979068,0.145539872621,0.145535766507,0.145531660725,0.145527555277,0.145523450162,0.145519345379,0.145515240929,0.145511136812,0.145507033028,0.145502929576,0.145498826457,0.14549472367,0.145490621216,0.145486519094,0.145482417305,0.145478315848,0.145474214724,0.145470113932,0.145466013472,0.145461913344,0.145457813549,0.145453714085,0.145449614954,0.145445516155,0.145441417688,0.145437319552,0.145433221749,0.145429124278,0.145425027138,0.14542093033,0.145416833854,0.14541273771,0.145408641898,0.145404546417,0.145400451267,0.14539635645,0.145392261963,0.145388167809,0.145384073985,0.145379980493,0.145375887333,0.145371794503,0.145367702005,0.145363609839,0.145359518003,0.145355426499,0.145351335325,0.145347244483,0.145343153972,0.145339063791,0.145334973942,0.145330884423,0.145326795236,0.145322706379,0.145318617853,0.145314529658,0.145310441793,0.145306354259,0.145302267056,0.145298180183,0.145294093641,0.145290007429,0.145285921547,0.145281835996,0.145277750776,0.145273665886,0.145269581325,0.145265497096,0.145261413196,0.145257329627,0.145253246387,0.145249163478,0.145245080899,0.14524099865,0.14523691673,0.145232835141,0.145228753881,0.145224672952,0.145220592352,0.145216512081,0.145212432141,0.14520835253,0.145204273249,0.145200194297,0.145196115675,0.145192037382,0.145187959419,0.145183881785,0.145179804481,0.145175727506,0.14517165086,0.145167574543,0.145163498556,0.145159422898,0.145155347569,0.145151272568,0.145147197897,0.145143123555,0.145139049542,0.145134975858,0.145130902503,0.145126829477,0.145122756779,0.14511868441,0.14511461237,0.145110540658,0.145106469276,0.145102398221,0.145098327496,0.145094257098,0.14509018703,0.145086117289,0.145082047877,0.145077978794,0.145073910038,0.145069841611,0.145065773512,0.145061705741,0.145057638299,0.145053571184,0.145049504398,0.14504543794,0.145041371809,0.145037306007,0.145033240532,0.145029175385,0.145025110566,0.145021046075,0.145016981912,0.145012918076,0.145008854568,0.145004791387,0.145000728534,0.144996666009,0.144992603811,0.14498854194,0.144984480397,0.144980419181,0.144976358292,0.144972297731,0.144968237497,0.14496417759,0.144960118011,0.144956058758,0.144951999833,0.144947941234,0.144943882963,0.144939825018,0.144935767401,0.14493171011,0.144927653146,0.144923596509,0.144919540199,0.144915484215,0.144911428558,0.144907373228,0.144903318224,0.144899263546,0.144895209196,0.144891155171,0.144887101473,0.144883048102,0.144878995057,0.144874942338,0.144870889945,0.144866837879,0.144862786138,0.144858734724,0.144854683636,0.144850632874,0.144846582438,0.144842532328,0.144838482544,0.144834433086,0.144830383954,0.144826335147,0.144822286666,0.144818238511,0.144814190682,0.144810143178,0.144806096,0.144802049147,0.14479800262,0.144793956419,0.144789910543,0.144785864992,0.144781819767,0.144777774867,0.144773730292,0.144769686042,0.144765642118,0.144761598519,0.144757555245,0.144753512296,0.144749469672,0.144745427373,0.144741385399,0.14473734375,0.144733302426,0.144729261426,0.144725220752,0.144721180402,0.144717140377,0.144713100676,0.1447090613,0.144705022249,0.144700983523,0.14469694512,0.144692907043,0.144688869289,0.14468483186,0.144680794756,0.144676757976,0.14467272152,0.144668685388,0.14466464958,0.144660614097,0.144656578938,0.144652544102,0.144648509591,0.144644475404,0.144640441541,0.144636408001,0.144632374786,0.144628341894,0.144624309326,0.144620277082,0.144616245161,0.144612213565,0.144608182291,0.144604151342,0.144600120716,0.144596090413,0.144592060434,0.144588030778,0.144584001446,0.144579972437,0.144575943751,0.144571915389,0.144567887349,0.144563859633,0.14455983224,0.144555805171,0.144551778424,0.144547752,0.144543725899,0.144539700121,0.144535674666,0.144531649534,0.144527624725,0.144523600239,0.144519576075,0.144515552234,0.144511528715,0.14450750552,0.144503482646,0.144499460096,0.144495437867,0.144491415962,0.144487394378,0.144483373117,0.144479352179,0.144475331562,0.144471311268,0.144467291296,0.144463271647,0.144459252319,0.144455233314,0.14445121463,0.144447196269,0.144443178229,0.144439160512,0.144435143116,0.144431126042,0.14442710929,0.14442309286,0.144419076752,0.144415060965,0.1444110455,0.144407030356,0.144403015534,0.144399001034,0.144394986855,0.144390972998,0.144386959462,0.144382946247,0.144378933354,0.144374920781,0.144370908531,0.144366896601,0.144362884993,0.144358873705,0.144354862739,0.144350852094,0.14434684177,0.144342831767,0.144338822084,0.144334812723,0.144330803683,0.144326794963,0.144322786564,0.144318778486,0.144314770728,0.144310763292,0.144306756175,0.14430274938,0.144298742905,0.14429473675,0.144290730916,0.144286725402,0.144282720209,0.144278715336,0.144274710783,0.14427070655,0.144266702638,0.144262699046,0.144258695774,0.144254692822,0.14425069019,0.144246687878,0.144242685887,0.144238684215,0.144234682863,0.14423068183,0.144226681118,0.144222680726,0.144218680653,0.1442146809,0.144210681466,0.144206682352,0.144202683558,0.144198685083,0.144194686928,0.144190689092,0.144186691576,0.144182694379,0.144178697501,0.144174700943,0.144170704704,0.144166708784,0.144162713183,0.144158717902,0.144154722939,0.144150728296,0.144146733972,0.144142739966,0.14413874628,0.144134752912,0.144130759864,0.144126767134,0.144122774723,0.14411878263,0.144114790857,0.144110799402,0.144106808266,0.144102817448,0.144098826949,0.144094836768,0.144090846906,0.144086857362,0.144082868137,0.14407887923,0.144074890641,0.14407090237,0.144066914418,0.144062926784,0.144058939468,0.144054952471,0.144050965791,0.144046979429,0.144042993386,0.14403900766,0.144035022252,0.144031037162,0.14402705239,0.144023067936,0.144019083799,0.14401509998,0.144011116479,0.144007133296,0.14400315043,0.143999167882,0.143995185651,0.143991203737,0.143987222142,0.143983240863,0.143979259902,0.143975279258,0.143971298932,0.143967318922,0.14396333923,0.143959359855,0.143955380798,0.143951402057,0.143947423633,0.143943445527,0.143939467737,0.143935490264,0.143931513108,0.143927536269,0.143923559747,0.143919583542,0.143915607653,0.143911632081,0.143907656826,0.143903681887,0.143899707265,0.14389573296,0.14389175897,0.143887785298,0.143883811942,0.143879838902,0.143875866178,0.143871893771,0.14386792168,0.143863949905,0.143859978447,0.143856007305,0.143852036478,0.143848065968,0.143844095774,0.143840125896,0.143836156334,0.143832187087,0.143828218157,0.143824249542,0.143820281243,0.14381631326,0.143812345593,0.143808378241,0.143804411205,0.143800444485,0.14379647808,0.14379251199,0.143788546217,0.143784580758,0.143780615615,0.143776650787,0.143772686275,0.143768722078,0.143764758196,0.143760794629,0.143756831378,0.143752868442,0.14374890582,0.143744943514,0.143740981523,0.143737019847,0.143733058485,0.143729097439,0.143725136708,0.143721176291,0.143717216189,0.143713256402,0.143709296929,0.143705337771,0.143701378928,0.143697420399,0.143693462185,0.143689504286,0.143685546701,0.14368158943,0.143677632474,0.143673675832,0.143669719504,0.143665763491,0.143661807791,0.143657852406,0.143653897336,0.143649942579,0.143645988136,0.143642034008,0.143638080193,0.143634126693,0.143630173506,0.143626220633,0.143622268074,0.143618315829,0.143614363897,0.14361041228,0.143606460976,0.143602509985,0.143598559309,0.143594608946,0.143590658896,0.14358670916,0.143582759737,0.143578810628,0.143574861832,0.143570913349,0.14356696518,0.143563017324,0.143559069781,0.143555122552,0.143551175635,0.143547229032,0.143543282742,0.143539336764,0.1435353911,0.143531445749,0.143527500711,0.143523555985,0.143519611572,0.143515667472,0.143511723685,0.143507780211,0.143503837049,0.1434998942,0.143495951664,0.14349200944,0.143488067528,0.143484125929,0.143480184643,0.143476243669,0.143472303007,0.143468362658,0.143464422621,0.143460482896,0.143456543483,0.143452604383,0.143448665595,0.143444727119,0.143440788954,0.143436851102,0.143432913562,0.143428976334,0.143425039418,0.143421102813,0.143417166521,0.14341323054,0.143409294871,0.143405359514,0.143401424468,0.143397489734,0.143393555312,0.143389621201,0.143385687402,0.143381753914,0.143377820738,0.143373887873,0.143369955319,0.143366023077,0.143362091146,0.143358159526,0.143354228218,0.14335029722,0.143346366534,0.143342436159,0.143338506095,0.143334576342,0.1433306469,0.143326717769,0.143322788949,0.14331886044,0.143314932241,0.143311004354,0.143307076777,0.14330314951,0.143299222555,0.14329529591,0.143291369576,0.143287443552,0.143283517839,0.143279592436,0.143275667344,0.143271742562,0.143267818091,0.143263893929,0.143259970078,0.143256046538,0.143252123307,0.143248200387,0.143244277777,0.143240355477,0.143236433487,0.143232511807,0.143228590438,0.143224669378,0.143220748628,0.143216828187,0.143212908057,0.143208988237,0.143205068726,0.143201149525,0.143197230633,0.143193312052,0.14318939378,0.143185475817,0.143181558164,0.14317764082,0.143173723786,0.143169807062,0.143165890646,0.14316197454,0.143158058744,0.143154143256,0.143150228078,0.143146313209,0.143142398649,0.143138484399,0.143134570457,0.143130656824,0.143126743501,0.143122830486,0.14311891778,0.143115005383,0.143111093295,0.143107181516,0.143103270046,0.143099358884,0.143095448031,0.143091537487,0.143087627251,0.143083717324,0.143079807705,0.143075898395,0.143071989393,0.1430680807,0.143064172315,0.143060264238,0.14305635647,0.14305244901,0.143048541858,0.143044635015,0.143040728479,0.143036822252,0.143032916333,0.143029010722,0.143025105419,0.143021200423,0.143017295736,0.143013391357,0.143009487285,0.143005583522,0.143001680066,0.142997776918,0.142993874077,0.142989971544,0.142986069319,0.142982167402,0.142978265791,0.142974364489,0.142970463494,0.142966562806,0.142962662426,0.142958762353,0.142954862588,0.142950963129,0.142947063978,0.142943165135,0.142939266598,0.142935368369,0.142931470446,0.142927572831,0.142923675523,0.142919778521,0.142915881827,0.142911985439,0.142908089359,0.142904193585,0.142900298118,0.142896402958,0.142892508104,0.142888613557,0.142884719317,0.142880825384,0.142876931757,0.142873038436,0.142869145422,0.142865252715,0.142861360314,0.142857468219,0.14285357643,0.142849684948,0.142845793772,0.142841902903,0.142838012339,0.142834122082,0.142830232131,0.142826342486,0.142822453147,0.142818564114,0.142814675387,0.142810786966,0.14280689885,0.142803011041,0.142799123537,0.14279523634,0.142791349448,0.142787462861,0.14278357658,0.142779690605,0.142775804936,0.142771919572,0.142768034513,0.14276414976,0.142760265313,0.142756381171,0.142752497334,0.142748613803,0.142744730576,0.142740847656,0.14273696504,0.142733082729,0.142729200724,0.142725319024,0.142721437628,0.142717556538,0.142713675753,0.142709795273,0.142705915097,0.142702035227,0.142698155661,0.1426942764,0.142690397444,0.142686518793,0.142682640446,0.142678762404,0.142674884667,0.142671007234,0.142667130106,0.142663253282,0.142659376763,0.142655500548,0.142651624637,0.142647749031,0.142643873729,0.142639998732,0.142636124038,0.142632249649,0.142628375564,0.142624501783,0.142620628307,0.142616755134,0.142612882265,0.1426090097,0.14260513744,0.142601265483,0.14259739383,0.142593522481,0.142589651435,0.142585780693,0.142581910256,0.142578040121,0.142574170291,0.142570300764,0.14256643154,0.14256256262,0.142558694004,0.142554825691,0.142550957681,0.142547089975,0.142543222572,0.142539355473,0.142535488676,0.142531622183,0.142527755994,0.142523890107,0.142520024523,0.142516159243,0.142512294265,0.142508429591,0.142504565219,0.142500701151,0.142496837385,0.142492973922,0.142489110763,0.142485247905,0.142481385351,0.142477523099,0.14247366115,0.142469799504,0.14246593816,0.142462077119,0.14245821638,0.142454355944,0.14245049581,0.142446635978,0.142442776449,0.142438917223,0.142435058298,0.142431199676,0.142427341356,0.142423483339,0.142419625623,0.14241576821,0.142411911098,0.142408054289,0.142404197782,0.142400341577,0.142396485673,0.142392630072,0.142388774772,0.142384919774,0.142381065078,0.142377210684,0.142373356591,0.1423695028,0.142365649311,0.142361796123,0.142357943237,0.142354090653,0.142350238369,0.142346386388,0.142342534708,0.142338683329,0.142334832251,0.142330981475,0.142327131,0.142323280826,0.142319430954,0.142315581382,0.142311732112,0.142307883143,0.142304034475,0.142300186107,0.142296338041,0.142292490276,0.142288642812,0.142284795648,0.142280948785,0.142277102224,0.142273255962,0.142269410002,0.142265564342,0.142261718983,0.142257873925,0.142254029167,0.142250184709,0.142246340552,0.142242496696,0.14223865314,0.142234809884,0.142230966929,0.142227124274,0.142223281919,0.142219439864,0.14221559811,0.142211756656,0.142207915502,0.142204074648,0.142200234094,0.14219639384,0.142192553886,0.142188714232,0.142184874878,0.142181035823,0.142177197069,0.142173358614,0.14216952046,0.142165682604,0.142161845049,0.142158007793,0.142154170837,0.14215033418,0.142146497823,0.142142661766,0.142138826007,0.142134990549,0.142131155389,0.142127320529,0.142123485969,0.142119651707,0.142115817745,0.142111984082,0.142108150718,0.142104317654,0.142100484888,0.142096652422,0.142092820254,0.142088988386,0.142085156816,0.142081325546,0.142077494574,0.142073663901,0.142069833527,0.142066003452,0.142062173675,0.142058344197,0.142054515018,0.142050686137,0.142046857555,0.142043029272,0.142039201287,0.1420353736,0.142031546212,0.142027719122,0.142023892331,0.142020065838,0.142016239643,0.142012413747,0.142008588149,0.142004762849,0.142000937847,0.141997113143,0.141993288737,0.141989464629,0.14198564082,0.141981817308,0.141977994094,0.141974171178,0.14197034856,0.14196652624,0.141962704217,0.141958882493,0.141955061066,0.141951239936,0.141947419105,0.141943598571,0.141939778334,0.141935958395,0.141932138754,0.141928319409,0.141924500363,0.141920681614,0.141916863162,0.141913045007,0.14190922715,0.141905409589,0.141901592327,0.141897775361,0.141893958692,0.141890142321,0.141886326246,0.141882510469,0.141878694988,0.141874879805,0.141871064918,0.141867250328,0.141863436035,0.141859622039,0.14185580834,0.141851994937,0.141848181831,0.141844369022,0.141840556509,0.141836744293,0.141832932374,0.141829120751,0.141825309424,0.141821498394,0.14181768766,0.141813877223,0.141810067082,0.141806257237,0.141802447689,0.141798638436,0.14179482948,0.14179102082,0.141787212457,0.141783404389,0.141779596617,0.141775789141,0.141771981962,0.141768175078,0.14176436849,0.141760562198,0.141756756202,0.141752950501,0.141749145097,0.141745339988,0.141741535174,0.141737730657,0.141733926435,0.141730122508,0.141726318877,0.141722515542,0.141718712502,0.141714909757,0.141711107308,0.141707305154,0.141703503296,0.141699701733,0.141695900465,0.141692099492,0.141688298814,0.141684498432,0.141680698345,0.141676898552,0.141673099055,0.141669299853,0.141665500946,0.141661702333,0.141657904016,0.141654105993,0.141650308265,0.141646510832,0.141642713694,0.14163891685,0.141635120301,0.141631324047,0.141627528087,0.141623732422,0.141619937052,0.141616141976,0.141612347194,0.141608552707,0.141604758514,0.141600964615,0.141597171011,0.141593377701,0.141589584686,0.141585791964,0.141581999537,0.141578207404,0.141574415564,0.141570624019,0.141566832768,0.141563041811,0.141559251148,0.141555460779,0.141551670704,0.141547880923,0.141544091435,0.141540302241,0.141536513341,0.141532724735,0.141528936422,0.141525148403,0.141521360677,0.141517573245,0.141513786107,0.141509999262,0.14150621271,0.141502426452,0.141498640488,0.141494854816,0.141491069438,0.141487284354,0.141483499562,0.141479715064,0.141475930859,0.141472146947,0.141468363328,0.141464580002,0.141460796969,0.141457014229,0.141453231782,0.141449449628,0.141445667767,0.141441886199,0.141438104923,0.141434323941,0.141430543251,0.141426762853,0.141422982749,0.141419202937,0.141415423417,0.141411644191,0.141407865256,0.141404086615,0.141400308265,0.141396530208,0.141392752444,0.141388974972,0.141385197792,0.141381420904,0.141377644309,0.141373868006,0.141370091995,0.141366316276,0.141362540849,0.141358765714,0.141354990872,0.141351216321,0.141347442062,0.141343668095,0.14133989442,0.141336121037,0.141332347946,0.141328575147,0.141324802639,0.141321030423,0.141317258498,0.141313486866,0.141309715525,0.141305944475,0.141302173717,0.14129840325,0.141294633075,0.141290863192,0.141287093599,0.141283324298,0.141279555289,0.14127578657,0.141272018143,0.141268250008,0.141264482163,0.141260714609,0.141256947347,0.141253180375,0.141249413695,0.141245647306,0.141241881207,0.1412381154,0.141234349883,0.141230584658,0.141226819723,0.141223055078,0.141219290725,0.141215526662,0.141211762891,0.141207999409,0.141204236218,0.141200473318,0.141196710709,0.14119294839,0.141189186361,0.141185424623,0.141181663175,0.141177902018,0.14117414115,0.141170380574,0.141166620287,0.141162860291,0.141159100585,0.141155341169,0.141151582043,0.141147823207,0.141144064661,0.141140306405,0.141136548439,0.141132790764,0.141129033378,0.141125276282,0.141121519475,0.141117762959,0.141114006732,0.141110250795,0.141106495148,0.14110273979,0.141098984722,0.141095229944,0.141091475455,0.141087721256,0.141083967346,0.141080213726,0.141076460395,0.141072707353,0.141068954601,0.141065202138,0.141061449964,0.14105769808,0.141053946484,0.141050195178,0.141046444161,0.141042693433,0.141038942995,0.141035192845,0.141031442984,0.141027693412,0.141023944129,0.141020195135,0.14101644643,0.141012698014,0.141008949886,0.141005202047,0.141001454497,0.140997707236,0.140993960263,0.140990213579,0.140986467183,0.140982721076,0.140978975257,0.140975229727,0.140971484486,0.140967739532,0.140963994867,0.140960250491,0.140956506402,0.140952762602,0.14094901909,0.140945275867,0.140941532931,0.140937790284,0.140934047924,0.140930305853,0.14092656407,0.140922822575,0.140919081367,0.140915340448,0.140911599816,0.140907859472,0.140904119416,0.140900379648,0.140896640168,0.140892900975,0.14088916207,0.140885423452,0.140881685123,0.14087794708,0.140874209326,0.140870471858,0.140866734678,0.140862997786,0.140859261181,0.140855524863,0.140851788833,0.14084805309,0.140844317634,0.140840582465,0.140836847584,0.140833112989,0.140829378682,0.140825644662,0.140821910929,0.140818177483,0.140814444323,0.140810711451,0.140806978866,0.140803246567,0.140799514556,0.140795782831,0.140792051393,0.140788320241,0.140784589376,0.140780858798,0.140777128507,0.140773398502,0.140769668784,0.140765939352,0.140762210207,0.140758481348,0.140754752775,0.140751024489,0.14074729649,0.140743568776,0.140739841349,0.140736114208,0.140732387353,0.140728660785,0.140724934502,0.140721208506,0.140717482796,0.140713757371,0.140710032233,0.140706307381,0.140702582815,0.140698858534,0.140695134539,0.140691410831,0.140687687408,0.14068396427,0.140680241419,0.140676518853,0.140672796573,0.140669074578,0.140665352869,0.140661631446,0.140657910308,0.140654189455,0.140650468888,0.140646748607,0.14064302861,0.140639308899,0.140635589474,0.140631870333,0.140628151478,0.140624432908,0.140620714624,0.140616996624,0.140613278909,0.14060956148,0.140605844336,0.140602127476,0.140598410902,0.140594694612,0.140590978607,0.140587262888,0.140583547453,0.140579832302,0.140576117437,0.140572402856,0.14056868856,0.140564974549,0.140561260822,0.14055754738,0.140553834222,0.140550121349,0.14054640876,0.140542696456,0.140538984436,0.1405352727,0.140531561249,0.140527850083,0.1405241392,0.140520428602,0.140516718288,0.140513008258,0.140509298512,0.14050558905,0.140501879872,0.140498170979,0.140494462369,0.140490754043,0.140487046002,0.140483338244,0.14047963077,0.14047592358,0.140472216673,0.140468510051,0.140464803712,0.140461097657,0.140457391885,0.140453686397,0.140449981193,0.140446276272,0.140442571635,0.140438867281,0.14043516321,0.140431459423,0.14042775592,0.1404240527,0.140420349763,0.140416647109,0.140412944739,0.140409242651,0.140405540847,0.140401839327,0.140398138089,0.140394437134,0.140390736462,0.140387036074,0.140383335968,0.140379636145,0.140375936605,0.140372237348,0.140368538374,0.140364839683,0.140361141274,0.140357443148,0.140353745305,0.140350047745,0.140346350467,0.140342653471,0.140338956759,0.140335260329,0.140331564181,0.140327868316,0.140324172733,0.140320477432,0.140316782414,0.140313087678,0.140309393225,0.140305699054,0.140302005165,0.140298311558,0.140294618233,0.140290925191,0.14028723243,0.140283539952,0.140279847755,0.140276155841,0.140272464208,0.140268772858,0.140265081789,0.140261391002,0.140257700497,0.140254010274,0.140250320332,0.140246630672,0.140242941294,0.140239252198,0.140235563383,0.14023187485,0.140228186598,0.140224498627,0.140220810939,0.140217123531,0.140213436405,0.140209749561,0.140206062997,0.140202376716,0.140198690715,0.140195004995,0.140191319557,0.1401876344,0.140183949524,0.140180264929,0.140176580615,0.140172896583,0.140169212831,0.14016552936,0.14016184617,0.140158163261,0.140154480633,0.140150798286,0.140147116219,0.140143434433,0.140139752928,0.140136071704,0.14013239076,0.140128710097,0.140125029715,0.140121349613,0.140117669792,0.140113990251,0.14011031099,0.14010663201,0.140102953311,0.140099274891,0.140095596753,0.140091918894,0.140088241315,0.140084564017,0.140080886999,0.140077210261,0.140073533804,0.140069857626,0.140066181728,0.140062506111,0.140058830773,0.140055155716,0.140051480938,0.14004780644,0.140044132222,0.140040458284,0.140036784626,0.140033111247,0.140029438148,0.140025765329,0.140022092789,0.140018420529,0.140014748549,0.140011076848,0.140007405426,0.140003734285,0.140000063422,0.139996392839,0.139992722535,0.139989052511,0.139985382766,0.1399817133,0.139978044114,0.139974375207,0.139970706579,0.13996703823,0.13996337016,0.139959702369,0.139956034857,0.139952367625,0.139948700671,0.139945033996,0.1399413676,0.139937701483,0.139934035645,0.139930370086,0.139926704805,0.139923039803,0.13991937508,0.139915710636,0.13991204647,0.139908382583,0.139904718974,0.139901055644,0.139897392592,0.139893729819,0.139890067325,0.139886405108,0.13988274317,0.139879081511,0.13987542013,0.139871759027,0.139868098202,0.139864437655,0.139860777387,0.139857117397,0.139853457685,0.13984979825,0.139846139094,0.139842480216,0.139838821616,0.139835163294,0.13983150525,0.139827847484,0.139824189995,0.139820532785,0.139816875852,0.139813219196,0.139809562819,0.139805906719,0.139802250897,0.139798595352,0.139794940085,0.139791285096,0.139787630384,0.139783975949,0.139780321792,0.139776667913,0.13977301431,0.139769360986,0.139765707938,0.139762055168,0.139758402674,0.139754750459,0.13975109852,0.139747446858,0.139743795474,0.139740144366,0.139736493536,0.139732842982,0.139729192706,0.139725542707,0.139721892984,0.139718243538,0.139714594369,0.139710945477,0.139707296862,0.139703648523,0.139700000462,0.139696352676,0.139692705168,0.139689057936,0.139685410981,0.139681764302,0.1396781179,0.139674471774,0.139670825924,0.139667180351,0.139663535055,0.139659890035,0.139656245291,0.139652600823,0.139648956631,0.139645312716,0.139641669077,0.139638025714,0.139634382627,0.139630739817,0.139627097282,0.139623455023,0.139619813041,0.139616171334,0.139612529903,0.139608888748,0.139605247869,0.139601607266,0.139597966938,0.139594326886,0.13959068711,0.13958704761,0.139583408385,0.139579769436,0.139576130762,0.139572492364,0.139568854242,0.139565216395,0.139561578823,0.139557941527,0.139554304506,0.139550667761,0.139547031291,0.139543395096,0.139539759177,0.139536123532,0.139532488163,0.139528853069,0.13952521825,0.139521583706,0.139517949438,0.139514315444,0.139510681725,0.139507048281,0.139503415112,0.139499782218,0.139496149599,0.139492517255,0.139488885186,0.139485253391,0.139481621871,0.139477990625,0.139474359655,0.139470728959,0.139467098537,0.13946346839,0.139459838518,0.13945620892,0.139452579596,0.139448950547,0.139445321773,0.139441693273,0.139438065047,0.139434437095,0.139430809418,0.139427182015,0.139423554886,0.139419928031,0.13941630145,0.139412675144,0.139409049111,0.139405423353,0.139401797868,0.139398172658,0.139394547721,0.139390923059,0.13938729867,0.139383674555,0.139380050714,0.139376427147,0.139372803853,0.139369180833,0.139365558087,0.139361935614,0.139358313415,0.13935469149,0.139351069838,0.13934744846,0.139343827355,0.139340206523,0.139336585965,0.139332965681,0.139329345669,0.139325725931,0.139322106467,0.139318487275,0.139314868357,0.139311249712,0.13930763134,0.139304013241,0.139300395416,0.139296777863,0.139293160583,0.139289543577,0.139285926843,0.139282310382,0.139278694194,0.139275078279,0.139271462637,0.139267847268,0.139264232171,0.139260617347,0.139257002796,0.139253388517,0.139249774512,0.139246160778,0.139242547317,0.139238934129,0.139235321213,0.13923170857,0.139228096199,0.139224484101,0.139220872275,0.139217260721,0.13921364944,0.13921003843,0.139206427694,0.139202817229,0.139199207036,0.139195597116,0.139191987468,0.139188378091,0.139184768987,0.139181160155,0.139177551595,0.139173943307,0.13917033529,0.139166727546,0.139163120073,0.139159512872,0.139155905943,0.139152299286,0.139148692901,0.139145086787,0.139141480944,0.139137875374,0.139134270075,0.139130665047,0.139127060292,0.139123455807,0.139119851594,0.139116247653,0.139112643983,0.139109040584,0.139105437456,0.1391018346,0.139098232015,0.139094629702,0.139091027659,0.139087425888,0.139083824388,0.139080223159,0.139076622201,0.139073021514,0.139069421098,0.139065820953,0.139062221079,0.139058621476,0.139055022144,0.139051423082,0.139047824292,0.139044225772,0.139040627523,0.139037029544,0.139033431837,0.1390298344,0.139026237233,0.139022640338,0.139019043712,0.139015447358,0.139011851273,0.13900825546,0.139004659916,0.139001064643,0.138997469641,0.138993874909,0.138990280447,0.138986686255,0.138983092333,0.138979498682,0.138975905301,0.13897231219,0.138968719349,0.138965126778,0.138961534478,0.138957942447,0.138954350686,0.138950759195,0.138947167974,0.138943577023,0.138939986342,0.138936395931,0.138932805789,0.138929215917,0.138925626315,0.138922036983,0.13891844792,0.138914859127,0.138911270603,0.138907682349,0.138904094365,0.13890050665,0.138896919204,0.138893332028,0.138889745121,0.138886158484,0.138882572116,0.138878986017,0.138875400188,0.138871814627,0.138868229336,0.138864644314,0.138861059562,0.138857475078,0.138853890864,0.138850306918,0.138846723242,0.138843139834,0.138839556695,0.138835973826,0.138832391225,0.138828808893,0.13882522683,0.138821645036,0.13881806351,0.138814482253,0.138810901265,0.138807320546,0.138803740095,0.138800159913,0.138796579999,0.138793000354,0.138789420977,0.138785841869,0.138782263029,0.138778684458,0.138775106155,0.13877152812,0.138767950354,0.138764372856,0.138760795626,0.138757218665,0.138753641971,0.138750065546,0.138746489389,0.1387429135,0.138739337879,0.138735762526,0.138732187441,0.138728612624,0.138725038074,0.138721463793,0.13871788978,0.138714316034,0.138710742557,0.138707169347,0.138703596404,0.13870002373,0.138696451323,0.138692879184,0.138689307312,0.138685735708,0.138682164372,0.138678593303,0.138675022501,0.138671451967,0.1386678817,0.138664311701,0.138660741969,0.138657172505,0.138653603307,0.138650034377,0.138646465715,0.138642897319,0.13863932919,0.138635761329,0.138632193735,0.138628626408,0.138625059348,0.138621492555,0.138617926028,0.138614359769,0.138610793777,0.138607228051,0.138603662593,0.138600097401,0.138596532476,0.138592967818,0.138589403426,0.138585839302,0.138582275443,0.138578711852,0.138575148527,0.138571585468,0.138568022677,0.138564460151,0.138560897892,0.1385573359,0.138553774174,0.138550212714,0.138546651521,0.138543090594,0.138539529933,0.138535969538,0.13853240941,0.138528849548,0.138525289952,0.138521730622,0.138518171558,0.13851461276,0.138511054229,0.138507495963,0.138503937963,0.138500380229,0.138496822761,0.138493265559,0.138489708623,0.138486151953,0.138482595548,0.138479039409,0.138475483536,0.138471927928,0.138468372586,0.13846481751,0.138461262699,0.138457708154,0.138454153875,0.13845059986,0.138447046112,0.138443492628,0.138439939411,0.138436386458,0.138432833771,0.138429281349,0.138425729192,0.138422177301,0.138418625675,0.138415074314,0.138411523218,0.138407972387,0.138404421821,0.138400871521,0.138397321485,0.138393771714,0.138390222209,0.138386672968,0.138383123992,0.138379575281,0.138376026835,0.138372478653,0.138368930737,0.138365383085,0.138361835698,0.138358288575,0.138354741717,0.138351195124,0.138347648795,0.138344102731,0.138340556931,0.138337011396,0.138333466125,0.138329921119,0.138326376377,0.1383228319,0.138319287686,0.138315743737,0.138312200053,0.138308656632,0.138305113476,0.138301570584,0.138298027956,0.138294485592,0.138290943492,0.138287401657,0.138283860085,0.138280318777,0.138276777733,0.138273236954,0.138269696438,0.138266156185,0.138262616197,0.138259076473,0.138255537012,0.138251997815,0.138248458882,0.138244920212,0.138241381806,0.138237843664,0.138234305785,0.138230768169,0.138227230818,0.138223693729,0.138220156905,0.138216620343,0.138213084045,0.13820954801,0.138206012239,0.138202476731,0.138198941486,0.138195406505,0.138191871786,0.138188337331,0.138184803139,0.13818126921,0.138177735544,0.138174202142,0.138170669002,0.138167136125,0.138163603511,0.13816007116,0.138156539072,0.138153007247,0.138149475684,0.138145944385,0.138142413348,0.138138882574,0.138135352063,0.138131821814,0.138128291828,0.138124762104,0.138121232644,0.138117703445,0.138114174509,0.138110645836,0.138107117425,0.138103589277,0.138100061391,0.138096533767,0.138093006406,0.138089479307,0.13808595247,0.138082425895,0.138078899583,0.138075373533,0.138071847745,0.138068322219,0.138064796955,0.138061271953,0.138057747214,0.138054222736,0.13805069852,0.138047174566,0.138043650874,0.138040127444,0.138036604276,0.138033081369,0.138029558725,0.138026036342,0.13802251422,0.138018992361,0.138015470763,0.138011949427,0.138008428352,0.138004907539,0.138001386987,0.137997866697,0.137994346668,0.137990826901,0.137987307395,0.137983788151,0.137980269168,0.137976750446,0.137973231986,0.137969713786,0.137966195848,0.137962678171,0.137959160756,0.137955643601,0.137952126708,0.137948610075,0.137945093704,0.137941577593,0.137938061744,0.137934546156,0.137931030828,0.137927515761,0.137924000955,0.13792048641,0.137916972126,0.137913458103,0.13790994434,0.137906430838,0.137902917596,0.137899404616,0.137895891896,0.137892379436,0.137888867237,0.137885355298,0.13788184362,0.137878332203,0.137874821045,0.137871310149,0.137867799512,0.137864289136,0.13786077902,0.137857269165,0.137853759569,0.137850250234,0.137846741159,0.137843232344,0.13783972379,0.137836215495,0.137832707461,0.137829199686,0.137825692171,0.137822184917,0.137818677922,0.137815171187,0.137811664712,0.137808158497,0.137804652542,0.137801146846,0.137797641411,0.137794136235,0.137790631318,0.137787126662,0.137783622264,0.137780118127,0.137776614249,0.137773110631,0.137769607272,0.137766104172,0.137762601332,0.137759098751,0.13775559643,0.137752094368,0.137748592566,0.137745091022,0.137741589738,0.137738088714,0.137734587948,0.137731087442,0.137727587194,0.137724087206,0.137720587477,0.137717088007,0.137713588796,0.137710089843,0.13770659115,0.137703092716,0.137699594541,0.137696096624,0.137692598966,0.137689101567,0.137685604427,0.137682107546,0.137678610923,0.137675114559,0.137671618454,0.137668122607,0.137664627019,0.137661131689,0.137657636618,0.137654141806,0.137650647251,0.137647152956,0.137643658918,0.137640165139,0.137636671619,0.137633178357,0.137629685352,0.137626192607,0.137622700119,0.13761920789,0.137615715919,0.137612224206,0.137608732751,0.137605241554,0.137601750615,0.137598259934,0.137594769511,0.137591279346,0.137587789439,0.13758429979,0.137580810398,0.137577321265,0.137573832389,0.137570343771,0.137566855411,0.137563367309,0.137559879464,0.137556391877,0.137552904547,0.137549417475,0.137545930661,0.137542444104,0.137538957804,0.137535471762,0.137531985978,0.137528500451,0.137525015181,0.137521530169,0.137518045413,0.137514560916,0.137511076675,0.137507592692,0.137504108965,0.137500625496,0.137497142285,0.13749365933,0.137490176632,0.137486694191,0.137483212008,0.137479730081,0.137476248411,0.137472766998,0.137469285842,0.137465804943,0.137462324301,0.137458843915,0.137455363787,0.137451883915,0.137448404299,0.137444924941,0.137441445839,0.137437966994,0.137434488405,0.137431010073,0.137427531997,0.137424054178,0.137420576615,0.137417099309,0.137413622259,0.137410145465,0.137406668928,0.137403192647,0.137399716623,0.137396240854,0.137392765342,0.137389290086,0.137385815087,0.137382340343,0.137378865855,0.137375391624,0.137371917649,0.137368443929,0.137364970466,0.137361497258,0.137358024307,0.137354551611,0.137351079171,0.137347606987,0.137344135059,0.137340663387,0.13733719197,0.137333720809,0.137330249904,0.137326779255,0.137323308861,0.137319838722,0.137316368839,0.137312899212,0.13730942984,0.137305960724,0.137302491863,0.137299023258,0.137295554908,0.137292086813,0.137288618974,0.13728515139,0.137281684061,0.137278216987,0.137274750169,0.137271283606,0.137267817298,0.137264351245,0.137260885447,0.137257419904,0.137253954616,0.137250489583,0.137247024805,0.137243560283,0.137240096015,0.137236632001,0.137233168243,0.13722970474,0.137226241491,0.137222778497,0.137219315758,0.137215853273,0.137212391043,0.137208929068,0.137205467348,0.137202005882,0.13719854467,0.137195083713,0.137191623011,0.137188162563,0.137184702369,0.13718124243,0.137177782745,0.137174323314,0.137170864138,0.137167405216,0.137163946549,0.137160488135,0.137157029976,0.137153572071,0.13715011442,0.137146657023,0.13714319988,0.137139742991,0.137136286357,0.137132829976,0.137129373849,0.137125917976,0.137122462357,0.137119006992,0.137115551881,0.137112097023,0.137108642419,0.137105188069,0.137101733973,0.137098280131,0.137094826542,0.137091373206,0.137087920124,0.137084467296,0.137081014722,0.1370775624,0.137074110333,0.137070658519,0.137067206958,0.13706375565,0.137060304596,0.137056853796,0.137053403248,0.137049952954,0.137046502913,0.137043053125,0.137039603591,0.137036154309,0.137032705281,0.137029256506,0.137025807984,0.137022359715,0.137018911699,0.137015463935,0.137012016425,0.137008569168,0.137005122164,0.137001675412,0.136998228913,0.136994782667,0.136991336674,0.136987890934,0.136984445446,0.136981000211,0.136977555229,0.136974110499,0.136970666022,0.136967221797,0.136963777825,0.136960334105,0.136956890638,0.136953447424,0.136950004461,0.136946561751,0.136943119294,0.136939677089,0.136936235136,0.136932793435,0.136929351987,0.13692591079,0.136922469846,0.136919029154,0.136915588714,0.136912148527,0.136908708591,0.136905268907,0.136901829476,0.136898390296,0.136894951368,0.136891512693,0.136888074269,0.136884636097,0.136881198176,0.136877760508,0.136874323091,0.136870885926,0.136867449013,0.136864012351,0.136860575941,0.136857139783,0.136853703876,0.136850268221,0.136846832817,0.136843397665,0.136839962764,0.136836528115,0.136833093717,0.136829659571,0.136826225675,0.136822792032,0.136819358639,0.136815925498,0.136812492608,0.136809059969,0.136805627581,0.136802195445,0.136798763559,0.136795331925,0.136791900542,0.13678846941,0.136785038529,0.136781607898,0.136778177519,0.136774747391,0.136771317513,0.136767887886,0.136764458511,0.136761029386,0.136757600511,0.136754171888,0.136750743515,0.136747315393,0.136743887521,0.1367404599,0.13673703253,0.13673360541,0.136730178541,0.136726751922,0.136723325554,0.136719899436,0.136716473569,0.136713047952,0.136709622585,0.136706197469,0.136702772603,0.136699347987,0.136695923622,0.136692499506,0.136689075641,0.136685652026,0.136682228662,0.136678805547,0.136675382682,0.136671960068,0.136668537703,0.136665115588,0.136661693723,0.136658272109,0.136654850744,0.136651429629,0.136648008764,0.136644588148,0.136641167782,0.136637747667,0.1366343278,0.136630908184,0.136627488817,0.1366240697,0.136620650832,0.136617232214,0.136613813846,0.136610395727,0.136606977857,0.136603560237,0.136600142866,0.136596725745,0.136593308873,0.136589892251,0.136586475878,0.136583059754,0.136579643879,0.136576228253,0.136572812877,0.13656939775,0.136565982872,0.136562568243,0.136559153863,0.136555739733,0.136552325851,0.136548912218,0.136545498834,0.136542085699,0.136538672813,0.136535260176,0.136531847788,0.136528435649,0.136525023758,0.136521612116,0.136518200723,0.136514789578,0.136511378683,0.136507968036,0.136504557637,0.136501147487,0.136497737586,0.136494327933,0.136490918528,0.136487509372,0.136484100465,0.136480691806,0.136477283395,0.136473875233,0.136470467319,0.136467059653,0.136463652236,0.136460245067,0.136456838146,0.136453431473,0.136450025048,0.136446618872,0.136443212943,0.136439807263,0.13643640183,0.136432996646,0.13642959171,0.136426187021,0.136422782581,0.136419378388,0.136415974443,0.136412570746,0.136409167297,0.136405764096,0.136402361142,0.136398958436,0.136395555978,0.136392153768,0.136388751805,0.136385350089,0.136381948622,0.136378547402,0.136375146429,0.136371745704,0.136368345226,0.136364944996,0.136361545013,0.136358145277,0.136354745789,0.136351346548,0.136347947555,0.136344548808,0.136341150309,0.136337752057,0.136334354053,0.136330956295,0.136327558785,0.136324161521,0.136320764505,0.136317367736,0.136313971213,0.136310574938,0.13630717891,0.136303783128,0.136300387594,0.136296992306,0.136293597265,0.136290202471,0.136286807924,0.136283413623,0.136280019569,0.136276625762,0.136273232202,0.136269838888,0.13626644582,0.136263053,0.136259660425,0.136256268098,0.136252876017,0.136249484182,0.136246092594,0.136242701252,0.136239310156,0.136235919307,0.136232528704,0.136229138347,0.136225748237,0.136222358373,0.136218968755,0.136215579383,0.136212190258,0.136208801378,0.136205412745,0.136202024357,0.136198636216,0.136195248321,0.136191860671,0.136188473268,0.13618508611,0.136181699199,0.136178312533,0.136174926113,0.136171539939,0.13616815401,0.136164768328,0.136161382891,0.1361579977,0.136154612754,0.136151228054,0.1361478436,0.136144459391,0.136141075428,0.13613769171,0.136134308238,0.136130925011,0.136127542029,0.136124159293,0.136120776803,0.136117394558,0.136114012558,0.136110630803,0.136107249294,0.136103868029,0.13610048701,0.136097106237,0.136093725708,0.136090345424,0.136086965386,0.136083585593,0.136080206044,0.136076826741,0.136073447683,0.136070068869,0.136066690301,0.136063311977,0.136059933898,0.136056556064,0.136053178475,0.136049801131,0.136046424031,0.136043047176,0.136039670566,0.1360362942,0.13603291808,0.136029542203,0.136026166572,0.136022791184,0.136019416042,0.136016041143,0.13601266649,0.136009292081,0.136005917916,0.136002543995,0.135999170319,0.135995796887,0.1359924237,0.135989050756,0.135985678057,0.135982305603,0.135978933392,0.135975561426,0.135972189703,0.135968818225,0.135965446991,0.135962076001,0.135958705254,0.135955334752,0.135951964494,0.13594859448,0.13594522471,0.135941855183,0.1359384859,0.135935116862,0.135931748066,0.135928379515,0.135925011208,0.135921643144,0.135918275324,0.135914907747,0.135911540414,0.135908173325,0.135904806479,0.135901439877,0.135898073518,0.135894707403,0.135891341531,0.135887975902,0.135884610517,0.135881245376,0.135877880477,0.135874515822,0.135871151411,0.135867787242,0.135864423317,0.135861059635,0.135857696196,0.135854333,0.135850970048,0.135847607338,0.135844244871,0.135840882648,0.135837520667,0.13583415893,0.135830797435,0.135827436184,0.135824075175,0.135820714409,0.135817353886,0.135813993606,0.135810633568,0.135807273773,0.135803914221,0.135800554912,0.135797195845,0.135793837021,0.13579047844,0.135787120101,0.135783762004,0.135780404151,0.135777046539,0.13577368917,0.135770332044,0.13576697516,0.135763618518,0.135760262119,0.135756905962,0.135753550047,0.135750194375,0.135746838945,0.135743483757,0.135740128811,0.135736774107,0.135733419646,0.135730065427,0.135726711449,0.135723357714,0.135720004221,0.13571665097,0.13571329796,0.135709945193,0.135706592668,0.135703240384,0.135699888342,0.135696536542,0.135693184984,0.135689833668,0.135686482593,0.135683131761,0.135679781169,0.13567643082,0.135673080712,0.135669730846,0.135666381221,0.135663031838,0.135659682696,0.135656333796,0.135652985137,0.13564963672,0.135646288544,0.135642940609,0.135639592916,0.135636245464,0.135632898254,0.135629551284,0.135626204556,0.135622858069,0.135619511824,0.135616165819,0.135612820056,0.135609474534,0.135606129252,0.135602784212,0.135599439413,0.135596094855,0.135592750538,0.135589406461,0.135586062626,0.135582719031,0.135579375678,0.135576032565,0.135572689693,0.135569347062,0.135566004671,0.135562662521,0.135559320612,0.135555978943,0.135552637516,0.135549296328,0.135545955382,0.135542614675,0.13553927421,0.135535933985,0.135532594,0.135529254256,0.135525914752,0.135522575488,0.135519236465,0.135515897682,0.13551255914,0.135509220838,0.135505882776,0.135502544954,0.135499207372,0.135495870031,0.135492532929,0.135489196068,0.135485859447,0.135482523066,0.135479186925,0.135475851024,0.135472515362,0.135469179941,0.13546584476,0.135462509818,0.135459175117,0.135455840655,0.135452506433,0.135449172451,0.135445838708,0.135442505206,0.135439171943,0.135435838919,0.135432506135,0.135429173591,0.135425841287,0.135422509222,0.135419177396,0.13541584581,0.135412514464,0.135409183356,0.135405852489,0.13540252186,0.135399191471,0.135395861322,0.135392531411,0.13538920174,0.135385872309,0.135382543116,0.135379214163,0.135375885448,0.135372556973,0.135369228737,0.13536590074,0.135362572982,0.135359245464,0.135355918184,0.135352591143,0.135349264341,0.135345937778,0.135342611454,0.135339285368,0.135335959522,0.135332633914,0.135329308545,0.135325983415,0.135322658524,0.135319333871,0.135316009457,0.135312685282,0.135309361345,0.135306037647,0.135302714187,0.135299390966,0.135296067984,0.135292745239,0.135289422734,0.135286100467,0.135282778438,0.135279456647,0.135276135095,0.135272813781,0.135269492706,0.135266171868,0.135262851269,0.135259530908,0.135256210786,0.135252890901,0.135249571255,0.135246251847,0.135242932676,0.135239613744,0.13523629505,0.135232976594,0.135229658375,0.135226340395,0.135223022653,0.135219705148,0.135216387881,0.135213070852,0.135209754061,0.135206437508,0.135203121192,0.135199805115,0.135196489274,0.135193173672,0.135189858307,0.13518654318,0.13518322829,0.135179913638,0.135176599223,0.135173285046,0.135169971106,0.135166657404,0.135163343939,0.135160030712,0.135156717721,0.135153404969,0.135150092453,0.135146780175,0.135143468134,0.13514015633,0.135136844764,0.135133533434,0.135130222342,0.135126911487,0.135123600869,0.135120290488,0.135116980344,0.135113670437,0.135110360767,0.135107051334,0.135103742138,0.135100433179,0.135097124456,0.135093815971,0.135090507722,0.13508719971,0.135083891935,0.135080584396,0.135077277095,0.13507397003,0.135070663201,0.135067356609,0.135064050254,0.135060744136,0.135057438254,0.135054132608,0.135050827199,0.135047522026,0.13504421709,0.13504091239,0.135037607927,0.1350343037,0.135030999709,0.135027695955,0.135024392437,0.135021089155,0.135017786109,0.1350144833,0.135011180726,0.135007878389,0.135004576288,0.135001274423,0.134997972794,0.134994671401,0.134991370244,0.134988069323,0.134984768638,0.134981468189,0.134978167976,0.134974867999,0.134971568257,0.134968268751,0.134964969482,0.134961670447,0.134958371649,0.134955073086,0.134951774759,0.134948476668,0.134945178812,0.134941881192,0.134938583807,0.134935286658,0.134931989745,0.134928693067,0.134925396624,0.134922100417,0.134918804445,0.134915508709,0.134912213208,0.134908917942,0.134905622912,0.134902328117,0.134899033557,0.134895739232,0.134892445143,0.134889151288,0.134885857669,0.134882564285,0.134879271136,0.134875978222,0.134872685543,0.134869393099,0.13486610089,0.134862808916,0.134859517177,0.134856225673,0.134852934404,0.13484964337,0.13484635257,0.134843062005,0.134839771675,0.13483648158,0.134833191719,0.134829902093,0.134826612702,0.134823323545,0.134820034623,0.134816745936,0.134813457483,0.134810169264,0.13480688128,0.134803593531,0.134800306016,0.134797018735,0.134793731689,0.134790444877,0.134787158299,0.134783871956,0.134780585847,0.134777299972,0.134774014332,0.134770728926,0.134767443754,0.134764158816,0.134760874112,0.134757589642,0.134754305406,0.134751021405,0.134747737637,0.134744454103,0.134741170804,0.134737887738,0.134734604906,0.134731322308,0.134728039944,0.134724757814,0.134721475917,0.134718194254,0.134714912825,0.13471163163,0.134708350668,0.13470506994,0.134701789446,0.134698509185,0.134695229158,0.134691949365,0.134688669805,0.134685390478,0.134682111385,0.134678832526,0.1346755539,0.134672275507,0.134668997347,0.134665719421,0.134662441729,0.134659164269,0.134655887043,0.13465261005,0.13464933329,0.134646056764,0.13464278047,0.13463950441,0.134636228583,0.134632952989,0.134629677628,0.1346264025,0.134623127605,0.134619852943,0.134616578514,0.134613304318,0.134610030354,0.134606756624,0.134603483126,0.134600209862,0.13459693683,0.13459366403,0.134590391464,0.13458711913,0.134583847029,0.134580575161,0.134577303525,0.134574032122,0.134570760951,0.134567490013,0.134564219307,0.134560948834,0.134557678594,0.134554408586,0.13455113881,0.134547869266,0.134544599955,0.134541330877,0.13453806203,0.134534793416,0.134531525035,0.134528256885,0.134524988968,0.134521721282,0.134518453829,0.134515186608,0.134511919619,0.134508652863,0.134505386338,0.134502120045,0.134498853985,0.134495588156,0.134492322559,0.134489057194,0.134485792061,0.13448252716,0.134479262491,0.134475998053,0.134472733847,0.134469469873,0.134466206131,0.13446294262,0.134459679342,0.134456416294,0.134453153479,0.134449890895,0.134446628542,0.134443366421,0.134440104532,0.134436842874,0.134433581448,0.134430320253,0.134427059289,0.134423798557,0.134420538056,0.134417277786,0.134414017748,0.134410757941,0.134407498366,0.134404239021,0.134400979908,0.134397721026,0.134394462375,0.134391203955,0.134387945766,0.134384687809,0.134381430082,0.134378172586,0.134374915322,0.134371658288,0.134368401485,0.134365144914,0.134361888573,0.134358632463,0.134355376584,0.134352120935,0.134348865517,0.134345610331,0.134342355374,0.134339100649,0.134335846154,0.13433259189,0.134329337857,0.134326084054,0.134322830481,0.13431957714,0.134316324028,0.134313071147,0.134309818497,0.134306566077,0.134303313888,0.134300061929,0.1342968102,0.134293558702,0.134290307434,0.134287056396,0.134283805588,0.134280555011,0.134277304664,0.134274054547,0.13427080466,0.134267555003,0.134264305577,0.13426105638,0.134257807414,0.134254558677,0.134251310171,0.134248061895,0.134244813848,0.134241566031,0.134238318445,0.134235071088,0.134231823961,0.134228577064,0.134225330396,0.134222083959,0.134218837751,0.134215591772,0.134212346024,0.134209100505,0.134205855216,0.134202610156,0.134199365326,0.134196120725,0.134192876354,0.134189632213,0.134186388301,0.134183144618,0.134179901165,0.134176657941,0.134173414947,0.134170172182,0.134166929646,0.13416368734,0.134160445263,0.134157203415,0.134153961796,0.134150720406,0.134147479246,0.134144238315,0.134140997612,0.134137757139,0.134134516895,0.13413127688,0.134128037094,0.134124797537,0.134121558209,0.13411831911,0.13411508024,0.134111841599,0.134108603186,0.134105365002,0.134102127048,0.134098889321,0.134095651824,0.134092414555,0.134089177515,0.134085940704,0.134082704122,0.134079467767,0.134076231642,0.134072995745,0.134069760077,0.134066524637,0.134063289426,0.134060054443,0.134056819688,0.134053585162,0.134050350864,0.134047116795,0.134043882954,0.134040649342,0.134037415957,0.134034182801,0.134030949873,0.134027717173,0.134024484702,0.134021252458,0.134018020443,0.134014788656,0.134011557097,0.134008325766,0.134005094663,0.134001863788,0.13399863314,0.133995402721,0.13399217253,0.133988942567,0.133985712831,0.133982483324,0.133979254044,0.133976024992,0.133972796167,0.133969567571,0.133966339202,0.133963111061,0.133959883147,0.133956655462,0.133953428003,0.133950200773,0.13394697377,0.133943746994,0.133940520446,0.133937294125,0.133934068032,0.133930842166,0.133927616528,0.133924391117,0.133921165934,0.133917940977,0.133914716248,0.133911491747,0.133908267472,0.133905043425,0.133901819605,0.133898596012,0.133895372646,0.133892149508,0.133888926596,0.133885703912,0.133882481454,0.133879259224,0.13387603722,0.133872815444,0.133869593894,0.133866372572,0.133863151476,0.133859930607,0.133856709965,0.13385348955,0.133850269362,0.1338470494,0.133843829665,0.133840610157,0.133837390875,0.13383417182,0.133830952992,0.13382773439,0.133824516015,0.133821297867,0.133818079945,0.133814862249,0.13381164478,0.133808427538,0.133805210521,0.133801993732,0.133798777168,0.133795560831,0.133792344721,0.133789128836,0.133785913178,0.133782697746,0.133779482541,0.133776267561,0.133773052808,0.133769838281,0.13376662398,0.133763409905,0.133760196056,0.133756982433,0.133753769036,0.133750555865,0.13374734292,0.133744130201,0.133740917708,0.133737705441,0.1337344934,0.133731281585,0.133728069995,0.133724858631,0.133721647493,0.133718436581,0.133715225894,0.133712015433,0.133708805198,0.133705595188,0.133702385404,0.133699175846,0.133695966513,0.133692757405,0.133689548524,0.133686339867,0.133683131436,0.133679923231,0.133676715251,0.133673507496,0.133670299967,0.133667092663,0.133663885584,0.13366067873,0.133657472102,0.133654265699,0.133651059521,0.133647853569,0.133644647841,0.133641442339,0.133638237062,0.13363503201,0.133631827183,0.13362862258,0.133625418203,0.133622214051,0.133619010124,0.133615806422,0.133612602945,0.133609399692,0.133606196664,0.133602993862,0.133599791284,0.13359658893,0.133593386802,0.133590184898,0.133586983219,0.133583781765,0.133580580535,0.13357737953,0.133574178749,0.133570978193,0.133567777862,0.133564577755,0.133561377872,0.133558178214,0.133554978781,0.133551779571,0.133548580587,0.133545381826,0.13354218329,0.133538984978,0.133535786891,0.133532589028,0.133529391389,0.133526193974,0.133522996784,0.133519799817,0.133516603075,0.133513406557,0.133510210263,0.133507014193,0.133503818347,0.133500622725,0.133497427327,0.133494232153,0.133491037203,0.133487842477,0.133484647975,0.133481453696,0.133478259642,0.133475065811,0.133471872204,0.133468678821,0.133465485661,0.133462292725,0.133459100013,0.133455907525,0.13345271526,0.133449523219,0.133446331401,0.133443139807,0.133439948436,0.133436757289,0.133433566366,0.133430375666,0.133427185189,0.133423994936,0.133420804906,0.133417615099,0.133414425516,0.133411236156,0.13340804702,0.133404858106,0.133401669416,0.133398480949,0.133395292706,0.133392104685,0.133388916888,0.133385729313,0.133382541962,0.133379354834,0.133376167929,0.133372981246,0.133369794787,0.133366608551,0.133363422538,0.133360236747,0.13335705118,0.133353865835,0.133350680713,0.133347495814,0.133344311138,0.133341126684,0.133337942454,0.133334758445,0.13333157466,0.133328391097,0.133325207757,0.13332202464,0.133318841745,0.133315659072,0.133312476622,0.133309294395,0.13330611239,0.133302930608,0.133299749048,0.13329656771,0.133293386595,0.133290205702,0.133287025032,0.133283844583,0.133280664357,0.133277484354,0.133274304572,0.133271125013,0.133267945676,0.133264766561,0.133261587668,0.133258408997,0.133255230549,0.133252052322,0.133248874317,0.133245696535,0.133242518974,0.133239341636,0.133236164519,0.133232987624,0.133229810951,0.1332266345,0.133223458271,0.133220282264,0.133217106478,0.133213930914,0.133210755572,0.133207580451,0.133204405553,0.133201230875,0.13319805642,0.133194882186,0.133191708174,0.133188534383,0.133185360814,0.133182187466,0.13317901434,0.133175841435,0.133172668752,0.13316949629,0.133166324049,0.13316315203,0.133159980232,0.133156808656,0.1331536373,0.133150466166,0.133147295253,0.133144124562,0.133140954091,0.133137783842,0.133134613814,0.133131444007,0.133128274421,0.133125105056,0.133121935912,0.133118766989,0.133115598288,0.133112429807,0.133109261547,0.133106093508,0.13310292569,0.133099758092,0.133096590716,0.13309342356,0.133090256625,0.133087089911,0.133083923418,0.133080757145,0.133077591093,0.133074425262,0.133071259651,0.133068094261,0.133064929092,0.133061764143,0.133058599414,0.133055434906,0.133052270619,0.133049106552,0.133045942706,0.13304277908,0.133039615674,0.133036452489,0.133033289524,0.133030126779,0.133026964255,0.133023801951,0.133020639867,0.133017478003,0.13301431636,0.133011154936,0.133007993733,0.13300483275,0.133001671987,0.132998511445,0.132995351122,0.132992191019,0.132989031136,0.132985871473,0.132982712031,0.132979552808,0.132976393805,0.132973235021,0.132970076458,0.132966918115,0.132963759991,0.132960602087,0.132957444403,0.132954286938,0.132951129694,0.132947972668,0.132944815863,0.132941659277,0.132938502911,0.132935346764,0.132932190837,0.13292903513,0.132925879642,0.132922724373,0.132919569324,0.132916414494,0.132913259884,0.132910105493,0.132906951322,0.132903797369,0.132900643637,0.132897490123,0.132894336829,0.132891183753,0.132888030897,0.132884878261,0.132881725843,0.132878573645,0.132875421665,0.132872269905,0.132869118364,0.132865967042,0.132862815939,0.132859665054,0.132856514389,0.132853363943,0.132850213716,0.132847063707,0.132843913918,0.132840764347,0.132837614995,0.132834465862,0.132831316948,0.132828168252,0.132825019775,0.132821871517,0.132818723477,0.132815575656,0.132812428054,0.13280928067,0.132806133505,0.132802986559,0.132799839831,0.132796693321,0.13279354703,0.132790400958,0.132787255103,0.132784109468,0.13278096405,0.132777818851,0.13277467387,0.132771529108,0.132768384564,0.132765240238,0.13276209613,0.132758952241,0.13275580857,0.132752665117,0.132749521882,0.132746378865,0.132743236066,0.132740093485,0.132736951123,0.132733808978,0.132730667051,0.132727525342,0.132724383852,0.132721242579,0.132718101524,0.132714960687,0.132711820067,0.132708679666,0.132705539482,0.132702399516,0.132699259768,0.132696120238,0.132692980925,0.13268984183,0.132686702952,0.132683564293,0.13268042585,0.132677287626,0.132674149619,0.132671011829,0.132667874257,0.132664736903,0.132661599765,0.132658462846,0.132655326143,0.132652189658,0.132649053391,0.132645917341,0.132642781508,0.132639645892,0.132636510494,0.132633375313,0.132630240349,0.132627105602,0.132623971072,0.13262083676,0.132617702664,0.132614568786,0.132611435125,0.132608301681,0.132605168454,0.132602035443,0.13259890265,0.132595770074,0.132592637715,0.132589505572,0.132586373647,0.132583241938,0.132580110446,0.132576979171,0.132573848113,0.132570717271,0.132567586646,0.132564456238,0.132561326047,0.132558196072,0.132555066314,0.132551936772,0.132548807447,0.132545678339,0.132542549447,0.132539420771,0.132536292313,0.13253316407,0.132530036044,0.132526908234,0.132523780641,0.132520653264,0.132517526104,0.13251439916,0.132511272432,0.13250814592,0.132505019625,0.132501893546,0.132498767683,0.132495642036,0.132492516605,0.132489391391,0.132486266392,0.13248314161,0.132480017043,0.132476892693,0.132473768559,0.132470644641,0.132467520938,0.132464397452,0.132461274181,0.132458151127,0.132455028288,0.132451905665,0.132448783258,0.132445661066,0.132442539091,0.132439417331,0.132436295787,0.132433174458,0.132430053346,0.132426932449,0.132423811767,0.132420691301,0.132417571051,0.132414451016,0.132411331197,0.132408211593,0.132405092205,0.132401973032,0.132398854074,0.132395735332,0.132392616806,0.132389498494,0.132386380398,0.132383262518,0.132380144852,0.132377027402,0.132373910167,0.132370793148,0.132367676343,0.132364559754,0.13236144338,0.13235832722,0.132355211276,0.132352095548,0.132348980034,0.132345864735,0.132342749651,0.132339634782,0.132336520128,0.132333405689,0.132330291465,0.132327177456,0.132324063661,0.132320950082,0.132317836717,0.132314723567,0.132311610632,0.132308497911,0.132305385406,0.132302273115,0.132299161038,0.132296049176,0.132292937529,0.132289826097,0.132286714879,0.132283603875,0.132280493086,0.132277382512,0.132274272152,0.132271162006,0.132268052075,0.132264942358,0.132261832856,0.132258723568,0.132255614494,0.132252505635,0.13224939699,0.132246288559,0.132243180342,0.13224007234,0.132236964552,0.132233856978,0.132230749618,0.132227642472,0.13222453554,0.132221428822,0.132218322319,0.132215216029,0.132212109954,0.132209004092,0.132205898444,0.132202793011,0.132199687791,0.132196582785,0.132193477993,0.132190373414,0.13218726905,0.132184164899,0.132181060962,0.132177957239,0.13217485373,0.132171750434,0.132168647352,0.132165544483,0.132162441828,0.132159339387,0.132156237159,0.132153135145,0.132150033344,0.132146931757,0.132143830383,0.132140729223,0.132137628276,0.132134527543,0.132131427023,0.132128326716,0.132125226623,0.132122126742,0.132119027076,0.132115927622,0.132112828382,0.132109729355,0.132106630541,0.13210353194,0.132100433553,0.132097335378,0.132094237417,0.132091139668,0.132088042133,0.132084944811,0.132081847702,0.132078750805,0.132075654122,0.132072557652,0.132069461394,0.13206636535,0.132063269518,0.132060173899,0.132057078493,0.1320539833,0.132050888319,0.132047793551,0.132044698996,0.132041604654,0.132038510524,0.132035416607,0.132032322903,0.132029229411,0.132026136132,0.132023043065,0.132019950211,0.132016857569,0.13201376514,0.132010672923,0.132007580919,0.132004489127,0.132001397547,0.13199830618,0.131995215025,0.131992124083,0.131989033353,0.131985942835,0.131982852529,0.131979762436,0.131976672554,0.131973582885,0.131970493428,0.131967404184,0.131964315151,0.13196122633,0.131958137722,0.131955049325,0.131951961141,0.131948873168,0.131945785408,0.131942697859,0.131939610522,0.131936523397,0.131933436484,0.131930349783,0.131927263294,0.131924177017,0.131921090951,0.131918005097,0.131914919455,0.131911834024,0.131908748805,0.131905663798,0.131902579003,0.131899494419,0.131896410047,0.131893325886,0.131890241937,0.131887158199,0.131884074673,0.131880991358,0.131877908255,0.131874825363,0.131871742682,0.131868660213,0.131865577956,0.131862495909,0.131859414074,0.13185633245,0.131853251038,0.131850169837,0.131847088846,0.131844008068,0.1318409275,0.131837847143,0.131834766998,0.131831687064,0.13182860734,0.131825527828,0.131822448527,0.131819369437,0.131816290557,0.131813211889,0.131810133432,0.131807055186,0.13180397715,0.131800899325,0.131797821712,0.131794744309,0.131791667116,0.131788590135,0.131785513364,0.131782436804,0.131779360455,0.131776284317,0.131773208389,0.131770132671,0.131767057165,0.131763981869,0.131760906783,0.131757831908,0.131754757244,0.13175168279,0.131748608546,0.131745534513,0.131742460691,0.131739387078,0.131736313677,0.131733240485,0.131730167504,0.131727094733,0.131724022173,0.131720949822,0.131717877682,0.131714805752,0.131711734033,0.131708662523,0.131705591224,0.131702520135,0.131699449256,0.131696378586,0.131693308127,0.131690237878,0.13168716784,0.131684098011,0.131681028392,0.131677958982,0.131674889783,0.131671820794,0.131668752015,0.131665683445,0.131662615085,0.131659546935,0.131656478995,0.131653411265,0.131650343744,0.131647276433,0.131644209332,0.13164114244,0.131638075758,0.131635009286,0.131631943023,0.13162887697,0.131625811126,0.131622745492,0.131619680067,0.131616614852,0.131613549846,0.13161048505,0.131607420463,0.131604356085,0.131601291917,0.131598227958,0.131595164209,0.131592100668,0.131589037337,0.131585974216,0.131582911303,0.1315798486,0.131576786106,0.131573723821,0.131570661745,0.131567599878,0.13156453822,0.131561476772,0.131558415532,0.131555354501,0.13155229368,0.131549233067,0.131546172663,0.131543112469,0.131540052483,0.131536992706,0.131533933138,0.131530873778,0.131527814628,0.131524755686,0.131521696953,0.131518638429,0.131515580113,0.131512522007,0.131509464109,0.131506406419,0.131503348938,0.131500291666,0.131497234602,0.131494177747,0.1314911211,0.131488064662,0.131485008433,0.131481952411,0.131478896599,0.131475840994,0.131472785598,0.131469730411,0.131466675432,0.131463620661,0.131460566098,0.131457511744,0.131454457598,0.13145140366,0.13144834993,0.131445296409,0.131442243096,0.13143918999,0.131436137093,0.131433084405,0.131430031924,0.131426979651,0.131423927586,0.131420875729,0.131417824081,0.13141477264,0.131411721407,0.131408670382,0.131405619565,0.131402568955,0.131399518554,0.13139646836,0.131393418375,0.131390368597,0.131387319026,0.131384269664,0.131381220509,0.131378171562,0.131375122823,0.131372074291,0.131369025966,0.13136597785,0.131362929941,0.131359882239,0.131356834745,0.131353787459,0.13135074038,0.131347693508,0.131344646844,0.131341600387,0.131338554138,0.131335508096,0.131332462261,0.131329416634,0.131326371214,0.131323326001,0.131320280996,0.131317236198,0.131314191607,0.131311147223,0.131308103046,0.131305059076,0.131302015314,0.131298971759,0.13129592841,0.131292885269,0.131289842335,0.131286799607,0.131283757087,0.131280714774,0.131277672667,0.131274630768,0.131271589075,0.13126854759,0.131265506311,0.131262465239,0.131259424373,0.131256383715,0.131253343263,0.131250303018,0.13124726298,0.131244223148,0.131241183523,0.131238144105,0.131235104893,0.131232065888,0.131229027089,0.131225988497,0.131222950112,0.131219911933,0.13121687396,0.131213836194,0.131210798635,0.131207761281,0.131204724135,0.131201687194,0.13119865046,0.131195613932,0.131192577611,0.131189541495,0.131186505586,0.131183469884,0.131180434387,0.131177399097,0.131174364012,0.131171329134,0.131168294462,0.131165259996,0.131162225737,0.131159191683,0.131156157835,0.131153124193,0.131150090758,0.131147057528,0.131144024504,0.131140991686,0.131137959074,0.131134926668,0.131131894468,0.131128862473,0.131125830684,0.131122799101,0.131119767724,0.131116736553,0.131113705587,0.131110674827,0.131107644272,0.131104613924,0.131101583781,0.131098553843,0.131095524111,0.131092494585,0.131089465264,0.131086436148,0.131083407238,0.131080378534,0.131077350035,0.131074321741,0.131071293653,0.13106826577,0.131065238093,0.131062210621,0.131059183354,0.131056156292,0.131053129436,0.131050102785,0.131047076339,0.131044050099,0.131041024063,0.131037998233,0.131034972608,0.131031947187,0.131028921972,0.131025896962,0.131022872157,0.131019847558,0.131016823163,0.131013798973,0.131010774988,0.131007751208,0.131004727632,0.131001704262,0.130998681096,0.130995658136,0.13099263538,0.130989612829,0.130986590483,0.130983568341,0.130980546404,0.130977524672,0.130974503145,0.130971481822,0.130968460704,0.13096543979,0.130962419081,0.130959398577,0.130956378277,0.130953358182,0.130950338291,0.130947318604,0.130944299123,0.130941279845,0.130938260772,0.130935241903,0.130932223239,0.130929204779,0.130926186523,0.130923168472,0.130920150625,0.130917132982,0.130914115543,0.130911098309,0.130908081279,0.130905064453,0.130902047831,0.130899031413,0.130896015199,0.130892999189,0.130889983384,0.130886967782,0.130883952385,0.130880937191,0.130877922201,0.130874907416,0.130871892834,0.130868878456,0.130865864282,0.130862850312,0.130859836546,0.130856822983,0.130853809624,0.130850796469,0.130847783518,0.130844770771,0.130841758227,0.130838745887,0.13083573375,0.130832721817,0.130829710088,0.130826698562,0.13082368724,0.130820676121,0.130817665206,0.130814654495,0.130811643987,0.130808633682,0.130805623581,0.130802613683,0.130799603988,0.130796594497,0.13079358521,0.130790576125,0.130787567244,0.130784558566,0.130781550091,0.13077854182,0.130775533752,0.130772525887,0.130769518225,0.130766510766,0.130763503511,0.130760496458,0.130757489609,0.130754482962,0.130751476519,0.130748470279,0.130745464241,0.130742458407,0.130739452775,0.130736447347,0.130733442121,0.130730437098,0.130727432278,0.130724427661,0.130721423247,0.130718419035,0.130715415027,0.130712411221,0.130709407617,0.130706404217,0.130703401019,0.130700398024,0.130697395231,0.130694392641,0.130691390253,0.130688388069,0.130685386086,0.130682384306,0.130679382729,0.130676381354,0.130673380182,0.130670379212,0.130667378444,0.130664377879,0.130661377516,0.130658377356,0.130655377398,0.130652377642,0.130649378088,0.130646378737,0.130643379588,0.130640380641,0.130637381896,0.130634383354,0.130631385013,0.130628386875,0.130625388939,0.130622391205,0.130619393673,0.130616396343,0.130613399215,0.130610402289,0.130607405565,0.130604409043,0.130601412722,0.130598416604,0.130595420688,0.130592424973,0.130589429461,0.13058643415,0.130583439041,0.130580444133,0.130577449428,0.130574454924,0.130571460622,0.130568466521,0.130565472623,0.130562478925,0.13055948543,0.130556492136,0.130553499044,0.130550506153,0.130547513463,0.130544520976,0.130541528689,0.130538536605,0.130535544721,0.130532553039,0.130529561559,0.130526570279,0.130523579202,0.130520588325,0.13051759765,0.130514607176,0.130511616903,0.130508626832,0.130505636962,0.130502647293,0.130499657825,0.130496668558,0.130493679493,0.130490690628,0.130487701965,0.130484713503,0.130481725242,0.130478737181,0.130475749322,0.130472761664,0.130469774207,0.130466786951,0.130463799895,0.130460813041,0.130457826387,0.130454839934,0.130451853682,0.130448867631,0.130445881781,0.130442896131,0.130439910683,0.130436925434,0.130433940387,0.13043095554,0.130427970894,0.130424986449,0.130422002204,0.13041901816,0.130416034316,0.130413050673,0.13041006723,0.130407083988,0.130404100946,0.130401118105,0.130398135464,0.130395153024,0.130392170784,0.130389188745,0.130386206905,0.130383225266,0.130380243828,0.13037726259,0.130374281551,0.130371300714,0.130368320076,0.130365339639,0.130362359401,0.130359379364,0.130356399527,0.130353419891,0.130350440454,0.130347461217,0.130344482181,0.130341503344,0.130338524707,0.130335546271,0.130332568034,0.130329589997,0.130326612161,0.130323634524,0.130320657087,0.130317679849,0.130314702812,0.130311725974,0.130308749337,0.130305772899,0.13030279666,0.130299820622,0.130296844783,0.130293869144,0.130290893704,0.130287918464,0.130284943424,0.130281968583,0.130278993942,0.130276019501,0.130273045258,0.130270071216,0.130267097373,0.130264123729,0.130261150285,0.13025817704,0.130255203995,0.130252231149,0.130249258502,0.130246286055,0.130243313807,0.130240341758,0.130237369909,0.130234398258,0.130231426807,0.130228455556,0.130225484503,0.13022251365,0.130219542995,0.13021657254,0.130213602284,0.130210632227,0.130207662369,0.13020469271,0.13020172325,0.130198753989,0.130195784927,0.130192816064,0.1301898474,0.130186878934,0.130183910668,0.1301809426,0.130177974732,0.130175007062,0.130172039591,0.130169072318,0.130166105245,0.13016313837,0.130160171694,0.130157205216,0.130154238937,0.130151272857,0.130148306976,0.130145341293,0.130142375808,0.130139410522,0.130136445435,0.130133480546,0.130130515856,0.130127551364,0.130124587071,0.130121622976,0.130118659079,0.130115695381,0.130112731881,0.13010976858,0.130106805477,0.130103842572,0.130100879865,0.130097917357,0.130094955047,0.130091992935,0.130089031021,0.130086069305,0.130083107788,0.130080146469,0.130077185348,0.130074224424,0.130071263699,0.130068303172,0.130065342843,0.130062382712,0.130059422779,0.130056463044,0.130053503507,0.130050544168,0.130047585027,0.130044626083,0.130041667338,0.13003870879,0.13003575044,0.130032792288,0.130029834333,0.130026876576,0.130023919017,0.130020961656,0.130018004492,0.130015047527,0.130012090758,0.130009134187,0.130006177814,0.130003221639,0.130000265661,0.12999730988,0.129994354297,0.129991398912,0.129988443724,0.129985488733,0.12998253394,0.129979579344,0.129976624945,0.129973670744,0.129970716741,0.129967762934,0.129964809325,0.129961855913,0.129958902699,0.129955949681,0.129952996861,0.129950044238,0.129947091812,0.129944139583,0.129941187552,0.129938235717,0.12993528408,0.12993233264,0.129929381396,0.12992643035,0.129923479501,0.129920528848,0.129917578393,0.129914628134,0.129911678073,0.129908728208,0.12990577854,0.129902829069,0.129899879795,0.129896930718,0.129893981837,0.129891033153,0.129888084666,0.129885136376,0.129882188282,0.129879240385,0.129876292685,0.129873345181,0.129870397874,0.129867450764,0.12986450385,0.129861557132,0.129858610611,0.129855664287,0.129852718159,0.129849772228,0.129846826493,0.129843880954,0.129840935612,0.129837990466,0.129835045517,0.129832100764,0.129829156207,0.129826211846,0.129823267682,0.129820323714,0.129817379942,0.129814436367,0.129811492987,0.129808549804,0.129805606817,0.129802664026,0.129799721431,0.129796779032,0.12979383683,0.129790894823,0.129787953012,0.129785011398,0.129782069979,0.129779128756,0.12977618773,0.129773246899,0.129770306264,0.129767365825,0.129764425581,0.129761485534,0.129758545682,0.129755606026,0.129752666566,0.129749727302,0.129746788233,0.12974384936,0.129740910683,0.129737972201,0.129735033915,0.129732095825,0.12972915793,0.129726220231,0.129723282728,0.129720345419,0.129717408307,0.12971447139,0.129711534668,0.129708598142,0.129705661811,0.129702725676,0.129699789736,0.129696853991,0.129693918442,0.129690983088,0.129688047929,0.129685112966,0.129682178198,0.129679243625,0.129676309247,0.129673375065,0.129670441078,0.129667507285,0.129664573688,0.129661640286,0.12965870708,0.129655774068,0.129652841251,0.12964990863,0.129646976203,0.129644043971,0.129641111934,0.129638180093,0.129635248446,0.129632316994,0.129629385737,0.129626454675,0.129623523807,0.129620593135,0.129617662657,0.129614732374,0.129611802286,0.129608872392,0.129605942693,0.129603013189,0.12960008388,0.129597154765,0.129594225845,0.129591297119,0.129588368588,0.129585440252,0.12958251211,0.129579584163,0.12957665641,0.129573728852,0.129570801488,0.129567874318,0.129564947343,0.129562020563,0.129559093976,0.129556167584,0.129553241387,0.129550315384,0.129547389575,0.12954446396,0.129541538539,0.129538613313,0.129535688281,0.129532763443,0.1295298388,0.12952691435,0.129523990095,0.129521066034,0.129518142167,0.129515218493,0.129512295014,0.129509371729,0.129506448638,0.129503525741,0.129500603038,0.129497680529,0.129494758214,0.129491836092,0.129488914165,0.129485992431,0.129483070891,0.129480149546,0.129477228393,0.129474307435,0.12947138667,0.129468466099,0.129465545722,0.129462625539,0.129459705549,0.129456785753,0.12945386615,0.129450946741,0.129448027526,0.129445108504,0.129442189675,0.129439271041,0.129436352599,0.129433434352,0.129430516297,0.129427598436,0.129424680769,0.129421763295,0.129418846014,0.129415928927,0.129413012033,0.129410095332,0.129407178825,0.129404262511,0.12940134639,0.129398430462,0.129395514728,0.129392599187,0.129389683838,0.129386768684,0.129383853722,0.129380938953,0.129378024378,0.129375109995,0.129372195806,0.129369281809,0.129366368006,0.129363454395,0.129360540978,0.129357627753,0.129354714721,0.129351801883,0.129348889237,0.129345976784,0.129343064524,0.129340152456,0.129337240582,0.1293343289,0.129331417411,0.129328506115,0.129325595011,0.1293226841,0.129319773382,0.129316862857,0.129313952524,0.129311042383,0.129308132436,0.12930522268,0.129302313118,0.129299403748,0.12929649457,0.129293585585,0.129290676793,0.129287768192,0.129284859785,0.129281951569,0.129279043546,0.129276135716,0.129273228078,0.129270320632,0.129267413378,0.129264506317,0.129261599448,0.129258692771,0.129255786286,0.129252879994,0.129249973894,0.129247067986,0.12924416227,0.129241256746,0.129238351414,0.129235446274,0.129232541327,0.129229636571,0.129226732008,0.129223827636,0.129220923457,0.129218019469,0.129215115673,0.129212212069,0.129209308658,0.129206405437,0.129203502409,0.129200599573,0.129197696928,0.129194794476,0.129191892215,0.129188990145,0.129186088268,0.129183186582,0.129180285088,0.129177383786,0.129174482675,0.129171581755,0.129168681028,0.129165780492,0.129162880147,0.129159979995,0.129157080033,0.129154180263,0.129151280685,0.129148381298,0.129145482102,0.129142583098,0.129139684286,0.129136785664,0.129133887234,0.129130988996,0.129128090948,0.129125193092,0.129122295428,0.129119397954,0.129116500672,0.129113603581,0.129110706681,0.129107809973,0.129104913455,0.129102017129,0.129099120993,0.129096225049,0.129093329296,0.129090433734,0.129087538363,0.129084643183,0.129081748194,0.129078853396,0.129075958789,0.129073064373,0.129070170147,0.129067276113,0.129064382269,0.129061488617,0.129058595155,0.129055701884,0.129052808804,0.129049915914,0.129047023215,0.129044130707,0.12904123839,0.129038346264,0.129035454328,0.129032562582,0.129029671028,0.129026779663,0.12902388849,0.129020997507,0.129018106715,0.129015216113,0.129012325701,0.12900943548,0.12900654545,0.12900365561,0.12900076596,0.128997876501,0.128994987232,0.128992098154,0.128989209266,0.128986320568,0.12898343206,0.128980543743,0.128977655616,0.128974767679,0.128971879933,0.128968992376,0.12896610501,0.128963217834,0.128960330848,0.128957444053,0.128954557447,0.128951671031,0.128948784806,0.12894589877,0.128943012925,0.128940127269,0.128937241804,0.128934356528,0.128931471443,0.128928586547,0.128925701841,0.128922817325,0.128919932999,0.128917048863,0.128914164916,0.12891128116,0.128908397593,0.128905514215,0.128902631028,0.12889974803,0.128896865222,0.128893982604,0.128891100175,0.128888217936,0.128885335887,0.128882454027,0.128879572356,0.128876690876,0.128873809584,0.128870928483,0.12886804757,0.128865166848,0.128862286314,0.12885940597,0.128856525816,0.128853645851,0.128850766075,0.128847886488,0.128845007091,0.128842127883,0.128839248865,0.128836370036,0.128833491396,0.128830612945,0.128827734683,0.128824856611,0.128821978728,0.128819101034,0.128816223529,0.128813346213,0.128810469086,0.128807592148,0.1288047154,0.12880183884,0.128798962469,0.128796086288,0.128793210295,0.128790334491,0.128787458876,0.128784583451,0.128781708214,0.128778833165,0.128775958306,0.128773083635,0.128770209154,0.128767334861,0.128764460757,0.128761586841,0.128758713114,0.128755839576,0.128752966227,0.128750093066,0.128747220094,0.128744347311,0.128741474716,0.12873860231,0.128735730092,0.128732858063,0.128729986223,0.12872711457,0.128724243107,0.128721371832,0.128718500745,0.128715629847,0.128712759137,0.128709888615,0.128707018282,0.128704148137,0.128701278181,0.128698408412,0.128695538833,0.128692669441,0.128689800237,0.128686931222,0.128684062395,0.128681193756,0.128678325306,0.128675457043,0.128672588969,0.128669721082,0.128666853384,0.128663985874,0.128661118552,0.128658251418,0.128655384472,0.128652517714,0.128649651143,0.128646784761,0.128643918567,0.12864105256,0.128638186742,0.128635321111,0.128632455668,0.128629590413,0.128626725346,0.128623860467,0.128620995775,0.128618131271,0.128615266955,0.128612402826,0.128609538886,0.128606675132,0.128603811567,0.128600948189,0.128598084999,0.128595221996,0.128592359181,0.128589496553,0.128586634113,0.12858377186,0.128580909795,0.128578047918,0.128575186227,0.128572324725,0.128569463409,0.128566602281,0.12856374134,0.128560880587,0.128558020021,0.128555159642,0.128552299451,0.128549439447,0.12854657963,0.12854372,0.128540860558,0.128538001302,0.128535142234,0.128532283353,0.128529424659,0.128526566153,0.128523707833,0.1285208497,0.128517991755,0.128515133996,0.128512276425,0.12850941904,0.128506561843,0.128503704832,0.128500848008,0.128497991371,0.128495134921,0.128492278658,0.128489422582,0.128486566693,0.12848371099,0.128480855474,0.128478000145,0.128475145003,0.128472290047,0.128469435278,0.128466580696,0.128463726301,0.128460872092,0.12845801807,0.128455164234,0.128452310585,0.128449457122,0.128446603847,0.128443750757,0.128440897854,0.128438045138,0.128435192608,0.128432340264,0.128429488107,0.128426636137,0.128423784352,0.128420932755,0.128418081343,0.128415230118,0.128412379079,0.128409528226,0.12840667756,0.12840382708,0.128400976786,0.128398126678,0.128395276757,0.128392427022,0.128389577473,0.12838672811,0.128383878933,0.128381029942,0.128378181137,0.128375332519,0.128372484086,0.128369635839,0.128366787779,0.128363939904,0.128361092216,0.128358244713,0.128355397396,0.128352550265,0.12834970332,0.128346856561,0.128344009987,0.1283411636,0.128338317398,0.128335471382,0.128332625552,0.128329779908,0.128326934449,0.128324089176,0.128321244089,0.128318399187,0.128315554471,0.12831270994,0.128309865596,0.128307021436,0.128304177463,0.128301333674,0.128298490072,0.128295646655,0.128292803423,0.128289960377,0.128287117516,0.128284274841,0.128281432351,0.128278590047,0.128275747927,0.128272905994,0.128270064245,0.128267222682,0.128264381304,0.128261540112,0.128258699104,0.128255858282,0.128253017645,0.128250177194,0.128247336927,0.128244496846,0.128241656949,0.128238817238,0.128235977712,0.128233138371,0.128230299215,0.128227460244,0.128224621458,0.128221782858,0.128218944442,0.128216106211,0.128213268165,0.128210430304,0.128207592627,0.128204755136,0.12820191783,0.128199080708,0.128196243771,0.128193407019,0.128190570452,0.128187734069,0.128184897872,0.128182061859,0.12817922603,0.128176390387,0.128173554928,0.128170719653,0.128167884564,0.128165049658,0.128162214938,0.128159380402,0.128156546051,0.128153711884,0.128150877901,0.128148044103,0.12814521049,0.128142377061,0.128139543816,0.128136710756,0.128133877881,0.128131045189,0.128128212682,0.128125380359,0.128122548221,0.128119716267,0.128116884497,0.128114052912,0.12811122151,0.128108390293,0.12810555926,0.128102728411,0.128099897747,0.128097067266,0.12809423697,0.128091406858,0.12808857693,0.128085747186,0.128082917625,0.128080088249,0.128077259057,0.128074430049,0.128071601225,0.128068772585,0.128065944129,0.128063115856,0.128060287768,0.128057459863,0.128054632143,0.128051804606,0.128048977253,0.128046150083,0.128043323098,0.128040496296,0.128037669678,0.128034843243,0.128032016993,0.128029190926,0.128026365042,0.128023539343,0.128020713827,0.128017888494,0.128015063345,0.12801223838,0.128009413598,0.128006589,0.128003764585,0.128000940353,0.127998116305,0.127995292441,0.12799246876,0.127989645262,0.127986821948,0.127983998817,0.127981175869,0.127978353105,0.127975530524,0.127972708127,0.127969885912,0.127967063881,0.127964242033,0.127961420368,0.127958598887,0.127955777589,0.127952956473,0.127950135541,0.127947314792,0.127944494226,0.127941673844,0.127938853644,0.127936033627,0.127933213793,0.127930394143,0.127927574675,0.12792475539,0.127921936288,0.127919117369,0.127916298633,0.12791348008,0.12791066171,0.127907843522,0.127905025518,0.127902207696,0.127899390057,0.127896572601,0.127893755327,0.127890938236,0.127888121328,0.127885304603,0.12788248806,0.1278796717,0.127876855522,0.127874039528,0.127871223715,0.127868408086,0.127865592638,0.127862777374,0.127859962292,0.127857147392,0.127854332675,0.12785151814,0.127848703788,0.127845889618,0.127843075631,0.127840261826,0.127837448203,0.127834634763,0.127831821505,0.127829008429,0.127826195535,0.127823382824,0.127820570295,0.127817757949,0.127814945784,0.127812133802,0.127809322002,0.127806510384,0.127803698948,0.127800887694,0.127798076622,0.127795265733,0.127792455025,0.1277896445,0.127786834156,0.127784023995,0.127781214015,0.127778404218,0.127775594602,0.127772785168,0.127769975916,0.127767166846,0.127764357958,0.127761549252,0.127758740728,0.127755932385,0.127753124224,0.127750316245,0.127747508448,0.127744700832,0.127741893399,0.127739086146,0.127736279076,0.127733472187,0.12773066548,0.127727858954,0.12772505261,0.127722246448,0.127719440467,0.127716634668,0.12771382905,0.127711023614,0.127708218359,0.127705413285,0.127702608393,0.127699803683,0.127696999154,0.127694194806,0.12769139064,0.127688586655,0.127685782851,0.127682979229,0.127680175788,0.127677372528,0.127674569449,0.127671766552,0.127668963836,0.127666161301,0.127663358947,0.127660556774,0.127657754783,0.127654952973,0.127652151343,0.127649349895,0.127646548628,0.127643747542,0.127640946637,0.127638145913,0.12763534537,0.127632545008,0.127629744827,0.127626944827,0.127624145007,0.127621345369,0.127618545912,0.127615746635,0.127612947539,0.127610148624,0.12760734989,0.127604551336,0.127601752964,0.127598954772,0.12759615676,0.12759335893,0.12759056128,0.127587763811,0.127584966522,0.127582169414,0.127579372487,0.12757657574,0.127573779174,0.127570982789,0.127568186584,0.127565390559,0.127562594715,0.127559799051,0.127557003568,0.127554208266,0.127551413143,0.127548618201,0.12754582344,0.127543028859,0.127540234458,0.127537440237,0.127534646197,0.127531852337,0.127529058658,0.127526265158,0.127523471839,0.1275206787,0.127517885741,0.127515092963,0.127512300364,0.127509507946,0.127506715708,0.12750392365,0.127501131772,0.127498340074,0.127495548556,0.127492757218,0.12748996606,0.127487175082,0.127484384284,0.127481593666,0.127478803228,0.12747601297,0.127473222892,0.127470432993,0.127467643275,0.127464853736,0.127462064377,0.127459275198,0.127456486199,0.127453697379,0.127450908739,0.127448120279,0.127445331999,0.127442543898,0.127439755977,0.127436968236,0.127434180674,0.127431393291,0.127428606089,0.127425819066,0.127423032222,0.127420245558,0.127417459074,0.127414672769,0.127411886643,0.127409100697,0.127406314931,0.127403529344,0.127400743936,0.127397958707,0.127395173658,0.127392388789,0.127389604098,0.127386819587,0.127384035255,0.127381251103,0.12737846713,0.127375683336,0.127372899721,0.127370116285,0.127367333029,0.127364549952,0.127361767053,0.127358984334,0.127356201794,0.127353419434,0.127350637252,0.127347855249,0.127345073425,0.127342291781,0.127339510315,0.127336729028,0.12733394792,0.127331166991,0.127328386241,0.12732560567,0.127322825278,0.127320045065,0.12731726503,0.127314485175,0.127311705498,0.127308926,0.12730614668,0.12730336754,0.127300588578,0.127297809795,0.12729503119,0.127292252764,0.127289474517,0.127286696449,0.127283918559,0.127281140847,0.127278363315,0.127275585961,0.127272808785,0.127270031788,0.127267254969,0.127264478329,0.127261701867,0.127258925584,0.127256149479,0.127253373553,0.127250597805,0.127247822235,0.127245046844,0.127242271631,0.127239496597,0.12723672174,0.127233947062,0.127231172562,0.127228398241,0.127225624098,0.127222850132,0.127220076345,0.127217302737,0.127214529306,0.127211756054,0.127208982979,0.127206210083,0.127203437365,0.127200664825,0.127197892462,0.127195120278,0.127192348272,0.127189576444,0.127186804794,0.127184033322,0.127181262027,0.127178490911,0.127175719973,0.127172949212,0.127170178629,0.127167408224,0.127164637997,0.127161867948,0.127159098076,0.127156328382,0.127153558866,0.127150789528,0.127148020367,0.127145251384,0.127142482579,0.127139713951,0.127136945501,0.127134177229,0.127131409134,0.127128641217,0.127125873477,0.127123105915,0.127120338531,0.127117571323,0.127114804294,0.127112037442,0.127109270767,0.12710650427,0.12710373795,0.127100971807,0.127098205842,0.127095440054,0.127092674444,0.127089909011,0.127087143755,0.127084378676,0.127081613775,0.127078849051,0.127076084504,0.127073320134,0.127070555942,0.127067791926,0.127065028088,0.127062264427,0.127059500943,0.127056737637,0.127053974507,0.127051211554,0.127048448778,0.12704568618,0.127042923758,0.127040161514,0.127037399446,0.127034637555,0.127031875841,0.127029114304,0.127026352944,0.127023591761,0.127020830755,0.127018069925,0.127015309273,0.127012548797,0.127009788498,0.127007028376,0.12700426843,0.127001508661,0.126998749069,0.126995989654,0.126993230415,0.126990471353,0.126987712467,0.126984953758,0.126982195226,0.12697943687,0.126976678691,0.126973920689,0.126971162862,0.126968405213,0.12696564774,0.126962890443,0.126960133323,0.126957376379,0.126954619612,0.126951863021,0.126949106606,0.126946350368,0.126943594306,0.126940838421,0.126938082712,0.126935327179,0.126932571822,0.126929816641,0.126927061637,0.126924306809,0.126921552157,0.126918797682,0.126916043382,0.126913289259,0.126910535312,0.12690778154,0.126905027945,0.126902274526,0.126899521284,0.126896768217,0.126894015326,0.126891262611,0.126888510072,0.126885757709,0.126883005522,0.126880253511,0.126877501675,0.126874750016,0.126871998533,0.126869247225,0.126866496093,0.126863745137,0.126860994357,0.126858243752,0.126855493324,0.126852743071,0.126849992993,0.126847243092,0.126844493366,0.126841743816,0.126838994441,0.126836245242,0.126833496219,0.126830747371,0.126827998699,0.126825250202,0.126822501881,0.126819753736,0.126817005766,0.126814257971,0.126811510352,0.126808762908,0.12680601564,0.126803268547,0.12680052163,0.126797774888,0.126795028321,0.12679228193,0.126789535713,0.126786789673,0.126784043807,0.126781298117,0.126778552602,0.126775807262,0.126773062097,0.126770317108,0.126767572294,0.126764827655,0.126762083191,0.126759338902,0.126756594788,0.12675385085,0.126751107086,0.126748363498,0.126745620084,0.126742876846,0.126740133782,0.126737390893,0.12673464818,0.126731905641,0.126729163277,0.126726421089,0.126723679075,0.126720937236,0.126718195571,0.126715454082,0.126712712767,0.126709971627,0.126707230662,0.126704489872,0.126701749256,0.126699008816,0.126696268549,0.126693528458,0.126690788541,0.126688048799,0.126685309231,0.126682569838,0.12667983062,0.126677091576,0.126674352707,0.126671614012,0.126668875492,0.126666137147,0.126663398975,0.126660660979,0.126657923156,0.126655185508,0.126652448035,0.126649710736,0.126646973611,0.126644236661,0.126641499885,0.126638763283,0.126636026856,0.126633290603,0.126630554524,0.126627818619,0.126625082889,0.126622347333,0.126619611951,0.126616876743,0.12661414171,0.12661140685,0.126608672165,0.126605937654,0.126603203316,0.126600469153,0.126597735164,0.126595001349,0.126592267708,0.126589534241,0.126586800948,0.126584067829,0.126581334884,0.126578602113,0.126575869515,0.126573137092,0.126570404843,0.126567672767,0.126564940865,0.126562209137,0.126559477583,0.126556746202,0.126554014995,0.126551283962,0.126548553103,0.126545822418,0.126543091906,0.126540361568,0.126537631403,0.126534901412,0.126532171595,0.126529441951,0.126526712481,0.126523983185,0.126521254062,0.126518525112,0.126515796336,0.126513067734,0.126510339305,0.126507611049,0.126504882967,0.126502155058,0.126499427323,0.126496699761,0.126493972373,0.126491245157,0.126488518116,0.126485791247,0.126483064552,0.12648033803,0.126477611681,0.126474885506,0.126472159503,0.126469433674,0.126466708018,0.126463982536,0.126461257226,0.12645853209,0.126455807126,0.126453082336,0.126450357719,0.126447633275,0.126444909004,0.126442184906,0.126439460981,0.126436737229,0.12643401365,0.126431290244,0.126428567011,0.126425843951,0.126423121063,0.126420398349,0.126417675807,0.126414953439,0.126412231243,0.12640950922,0.126406787369,0.126404065692,0.126401344187,0.126398622855,0.126395901696,0.126393180709,0.126390459895,0.126387739254,0.126385018786,0.12638229849,0.126379578366,0.126376858416,0.126374138637,0.126371419032,0.126368699599,0.126365980338,0.12636326125,0.126360542335,0.126357823592,0.126355105021,0.126352386623,0.126349668397,0.126346950344,0.126344232463,0.126341514754,0.126338797218,0.126336079854,0.126333362663,0.126330645643,0.126327928796,0.126325212121,0.126322495619,0.126319779289,0.126317063131,0.126314347145,0.126311631331,0.126308915689,0.12630620022,0.126303484923,0.126300769797,0.126298054844,0.126295340063,0.126292625454,0.126289911017,0.126287196752,0.126284482659,0.126281768738,0.126279054989,0.126276341412,0.126273628007,0.126270914773,0.126268201712,0.126265488822,0.126262776105,0.126260063559,0.126257351185,0.126254638982,0.126251926952,0.126249215093,0.126246503406,0.126243791891,0.126241080548,0.126238369376,0.126235658376,0.126232947547,0.12623023689,0.126227526405,0.126224816091,0.126222105949,0.126219395979,0.12621668618,0.126213976552,0.126211267096,0.126208557812,0.126205848699,0.126203139758,0.126200430988,0.126197722389,0.126195013962,0.126192305706,0.126189597622,0.126186889709,0.126184181967,0.126181474397,0.126178766998,0.12617605977,0.126173352713,0.126170645828,0.126167939114,0.126165232571,0.1261625262,0.126159819999,0.12615711397,0.126154408112,0.126151702425,0.126148996909,0.126146291564,0.12614358639,0.126140881387,0.126138176556,0.126135471895,0.126132767406,0.126130063087,0.126127358939,0.126124654963,0.126121951157,0.126119247522,0.126116544058,0.126113840765,0.126111137643,0.126108434691,0.126105731911,0.126103029301,0.126100326862,0.126097624594,0.126094922497,0.12609222057,0.126089518814,0.126086817229,0.126084115814,0.12608141457,0.126078713497,0.126076012594,0.126073311862,0.126070611301,0.12606791091,0.12606521069,0.12606251064,0.126059810761,0.126057111053,0.126054411514,0.126051712147,0.12604901295,0.126046313923,0.126043615066,0.126040916381,0.126038217865,0.12603551952,0.126032821345,0.126030123341,0.126027425506,0.126024727843,0.126022030349,0.126019333026,0.126016635873,0.12601393889,0.126011242077,0.126008545435,0.126005848963,0.12600315266,0.126000456529,0.125997760567,0.125995064775,0.125992369153,0.125989673702,0.12598697842,0.125984283309,0.125981588368,0.125978893596,0.125976198995,0.125973504563,0.125970810302,0.12596811621,0.125965422289,0.125962728537,0.125960034955,0.125957341543,0.125954648301,0.125951955229,0.125949262326,0.125946569594,0.125943877031,0.125941184638,0.125938492414,0.12593580036,0.125933108476,0.125930416762,0.125927725218,0.125925033843,0.125922342637,0.125919651602,0.125916960736,0.125914270039,0.125911579512,0.125908889155,0.125906198967,0.125903508949,0.1259008191,0.12589812942,0.125895439911,0.12589275057,0.125890061399,0.125887372398,0.125884683565,0.125881994903,0.125879306409,0.125876618085,0.12587392993,0.125871241945,0.125868554129,0.125865866482,0.125863179004,0.125860491696,0.125857804556,0.125855117586,0.125852430786,0.125849744154,0.125847057692,0.125844371398,0.125841685274,0.125838999319,0.125836313533,0.125833627916,0.125830942468,0.125828257189,0.125825572079,0.125822887138,0.125820202367,0.125817517764,0.12581483333,0.125812149065,0.125809464968,0.125806781041,0.125804097283,0.125801413693,0.125798730273,0.125796047021,0.125793363938,0.125790681023,0.125787998278,0.125785315701,0.125782633293,0.125779951054,0.125777268983,0.125774587081,0.125771905348,0.125769223784,0.125766542388,0.12576386116,0.125761180102,0.125758499212,0.12575581849,0.125753137937,0.125750457552,0.125747777336,0.125745097289,0.12574241741,0.125739737699,0.125737058157,0.125734378784,0.125731699578,0.125729020542,0.125726341673,0.125723662973,0.125720984441,0.125718306078,0.125715627882,0.125712949856,0.125710271997,0.125707594307,0.125704916785,0.125702239431,0.125699562245,0.125696885227,0.125694208378,0.125691531697,0.125688855184,0.125686178839,0.125683502662,0.125680826653,0.125678150813,0.12567547514,0.125672799635,0.125670124299,0.12566744913,0.12566477413,0.125662099297,0.125659424632,0.125656750136,0.125654075807,0.125651401646,0.125648727653,0.125646053827,0.12564338017,0.12564070668,0.125638033359,0.125635360205,0.125632687218,0.1256300144,0.125627341749,0.125624669266,0.125621996951,0.125619324803,0.125616652824,0.125613981011,0.125611309367,0.12560863789,0.12560596658,0.125603295438,0.125600624464,0.125597953657,0.125595283018,0.125592612547,0.125589942242,0.125587272106,0.125584602137,0.125581932335,0.125579262701,0.125576593234,0.125573923934,0.125571254802,0.125568585837,0.12556591704,0.12556324841,0.125560579947,0.125557911651,0.125555243523,0.125552575562,0.125549907769,0.125547240142,0.125544572683,0.125541905391,0.125539238266,0.125536571308,0.125533904517,0.125531237894,0.125528571438,0.125525905148,0.125523239026,0.125520573071,0.125517907283,0.125515241662,0.125512576208,0.125509910921,0.1255072458,0.125504580847,0.125501916061,0.125499251442,0.125496586989,0.125493922704,0.125491258585,0.125488594633,0.125485930848,0.12548326723,0.125480603779,0.125477940494,0.125475277376,0.125472614425,0.125469951641,0.125467289023,0.125464626572,0.125461964288,0.12545930217,0.12545664022,0.125453978435,0.125451316818,0.125448655366,0.125445994082,0.125443332964,0.125440672013,0.125438011228,0.125435350609,0.125432690158,0.125430029872,0.125427369753,0.125424709801,0.125422050015,0.125419390395,0.125416730942,0.125414071655,0.125411412534,0.12540875358,0.125406094792,0.125403436171,0.125400777715,0.125398119427,0.125395461304,0.125392803347,0.125390145557,0.125387487933,0.125384830475,0.125382173183,0.125379516058,0.125376859098,0.125374202305,0.125371545678,0.125368889217,0.125366232922,0.125363576793,0.12536092083,0.125358265033,0.125355609402,0.125352953937,0.125350298638,0.125347643505,0.125344988538,0.125342333737,0.125339679102,0.125337024633,0.125334370329,0.125331716191,0.12532906222,0.125326408414,0.125323754774,0.125321101299,0.12531844799,0.125315794848,0.12531314187,0.125310489059,0.125307836413,0.125305183933,0.125302531619,0.12529987947,0.125297227487,0.12529457567,0.125291924018,0.125289272531,0.125286621211,0.125283970055,0.125281319066,0.125278668242,0.125276017583,0.12527336709,0.125270716762,0.1252680666,0.125265416603,0.125262766772,0.125260117106,0.125257467605,0.12525481827,0.1252521691,0.125249520095,0.125246871256,0.125244222582,0.125241574073,0.12523892573,0.125236277551,0.125233629538,0.125230981691,0.125228334008,0.125225686491,0.125223039139,0.125220391951,0.125217744929,0.125215098073,0.125212451381,0.125209804854,0.125207158493,0.125204512296,0.125201866264,0.125199220398,0.125196574696,0.12519392916,0.125191283788,0.125188638582,0.12518599354,0.125183348663,0.125180703951,0.125178059404,0.125175415022,0.125172770804,0.125170126752,0.125167482864,0.125164839141,0.125162195583,0.12515955219,0.125156908961,0.125154265897,0.125151622998,0.125148980263,0.125146337694,0.125143695288,0.125141053048,0.125138410972,0.125135769061,0.125133127314,0.125130485732,0.125127844314,0.125125203061,0.125122561972,0.125119921048,0.125117280289,0.125114639694,0.125111999263,0.125109358997,0.125106718895,0.125104078958,0.125101439185,0.125098799577,0.125096160132,0.125093520853,0.125090881737,0.125088242786,0.125085603999,0.125082965376,0.125080326918,0.125077688624,0.125075050494,0.125072412528,0.125069774727,0.125067137089,0.125064499616,0.125061862307,0.125059225162,0.125056588181,0.125053951365,0.125051314712,0.125048678223,0.125046041899,0.125043405738,0.125040769742,0.125038133909,0.125035498241,0.125032862736,0.125030227395,0.125027592219,0.125024957206,0.125022322357,0.125019687672,0.125017053151,0.125014418793,0.1250117846,0.12500915057,0.125006516704,0.125003883002,0.125001249464,0.124998616089,0.124995982878,0.124993349831,0.124990716947,0.124988084228,0.124985451671,0.124982819279,0.12498018705,0.124977554985,0.124974923083,0.124972291345,0.12496965977,0.124967028359,0.124964397112,0.124961766028,0.124959135107,0.12495650435,0.124953873757,0.124951243327,0.12494861306,0.124945982957,0.124943353017,0.12494072324,0.124938093627,0.124935464177,0.124932834891,0.124930205768,0.124927576808,0.124924948011,0.124922319378,0.124919690908,0.124917062601,0.124914434458,0.124911806477,0.12490917866,0.124906551006,0.124903923515,0.124901296187,0.124898669022,0.124896042021,0.124893415182,0.124890788507,0.124888161994,0.124885535645,0.124882909459,0.124880283435,0.124877657575,0.124875031878,0.124872406343,0.124869780972,0.124867155763,0.124864530717,0.124861905835,0.124859281115,0.124856656558,0.124854032163,0.124851407932,0.124848783863,0.124846159957,0.124843536214,0.124840912634,0.124838289216,0.124835665961,0.124833042869,0.12483041994,0.124827797173,0.124825174569,0.124822552127,0.124819929848,0.124817307732,0.124814685778,0.124812063987,0.124809442358,0.124806820892,0.124804199589,0.124801578448,0.124798957469,0.124796336653,0.124793715999,0.124791095508,0.124788475179,0.124785855013,0.124783235009,0.124780615167,0.124777995488,0.124775375971,0.124772756617,0.124770137424,0.124767518394,0.124764899527,0.124762280821,0.124759662278,0.124757043897,0.124754425678,0.124751807622,0.124749189727,0.124746571995,0.124743954425,0.124741337017,0.124738719771,0.124736102688,0.124733485766,0.124730869006,0.124728252409,0.124725635973,0.1247230197,0.124720403588,0.124717787639,0.124715171851,0.124712556225,0.124709940762,0.12470732546,0.12470471032,0.124702095342,0.124699480526,0.124696865872,0.124694251379,0.124691637048,0.12468902288,0.124686408873,0.124683795027,0.124681181344,0.124678567822,0.124675954462,0.124673341263,0.124670728227,0.124668115352,0.124665502638,0.124662890087,0.124660277697,0.124657665468,0.124655053401,0.124652441496,0.124649829752,0.12464721817,0.124644606749,0.12464199549,0.124639384392,0.124636773456,0.124634162681,0.124631552068,0.124628941616,0.124626331325,0.124623721196,0.124621111229,0.124618501422,0.124615891777,0.124613282293,0.124610672971,0.12460806381,0.12460545481,0.124602845972,0.124600237294,0.124597628778,0.124595020423,0.12459241223,0.124589804197,0.124587196326,0.124584588616,0.124581981067,0.124579373679,0.124576766452,0.124574159386,0.124571552482,0.124568945738,0.124566339155,0.124563732734,0.124561126473,0.124558520374,0.124555914435,0.124553308658,0.124550703041,0.124548097585,0.12454549229,0.124542887156,0.124540282183,0.124537677371,0.12453507272,0.124532468229,0.124529863899,0.12452725973,0.124524655722,0.124522051875,0.124519448188,0.124516844662,0.124514241297,0.124511638092,0.124509035048,0.124506432165,0.124503829443,0.124501226881,0.124498624479,0.124496022239,0.124493420159,0.124490818239,0.12448821648,0.124485614881,0.124483013444,0.124480412166,0.124477811049,0.124475210093,0.124472609297,0.124470008661,0.124467408186,0.124464807871,0.124462207717,0.124459607723,0.124457007889,0.124454408216,0.124451808703,0.12444920935,0.124446610157,0.124444011125,0.124441412254,0.124438813542,0.124436214991,0.124433616599,0.124431018368,0.124428420298,0.124425822387,0.124423224636,0.124420627046,0.124418029616,0.124415432346,0.124412835236,0.124410238286,0.124407641496,0.124405044866,0.124402448396,0.124399852086,0.124397255936,0.124394659946,0.124392064116,0.124389468446,0.124386872936,0.124384277586,0.124381682396,0.124379087365,0.124376492494,0.124373897784,0.124371303233,0.124368708842,0.12436611461,0.124363520539,0.124360926627,0.124358332875,0.124355739283,0.12435314585,0.124350552577,0.124347959464,0.124345366511,0.124342773717,0.124340181082,0.124337588608,0.124334996293,0.124332404137,0.124329812141,0.124327220305,0.124324628628,0.124322037111,0.124319445753,0.124316854555,0.124314263516,0.124311672637,0.124309081917,0.124306491356,0.124303900955,0.124301310714,0.124298720631,0.124296130708,0.124293540945,0.124290951341,0.124288361896,0.12428577261,0.124283183484,0.124280594516,0.124278005709,0.12427541706,0.124272828571,0.12427024024,0.124267652069,0.124265064058,0.124262476205,0.124259888511,0.124257300977,0.124254713602,0.124252126385,0.124249539328,0.12424695243,0.124244365691,0.124241779111,0.12423919269,0.124236606428,0.124234020325,0.124231434381,0.124228848596,0.124226262969,0.124223677502,0.124221092194,0.124218507044,0.124215922053,0.124213337222,0.124210752549,0.124208168034,0.124205583679,0.124202999483,0.124200415445,0.124197831566,0.124195247845,0.124192664284,0.124190080881,0.124187497637,0.124184914551,0.124182331624,0.124179748856,0.124177166247,0.124174583796,0.124172001503,0.12416941937,0.124166837394,0.124164255578,0.12416167392,0.12415909242,0.124156511079,0.124153929896,0.124151348872,0.124148768006,0.124146187299,0.12414360675,0.12414102636,0.124138446128,0.124135866054,0.124133286139,0.124130706382,0.124128126783,0.124125547343,0.124122968061,0.124120388937,0.124117809971,0.124115231164,0.124112652515,0.124110074024,0.124107495692,0.124104917517,0.124102339501,0.124099761643,0.124097183943,0.124094606401,0.124092029017,0.124089451792,0.124086874724,0.124084297815,0.124081721063,0.12407914447,0.124076568034,0.124073991757,0.124071415637,0.124068839676,0.124066263872,0.124063688227,0.124061112739,0.124058537409,0.124055962237,0.124053387223,0.124050812367,0.124048237668,0.124045663128,0.124043088745,0.12404051452,0.124037940453,0.124035366543,0.124032792792,0.124030219198,0.124027645762,0.124025072483,0.124022499362,0.124019926399,0.124017353593,0.124014780945,0.124012208455,0.124009636122,0.124007063947,0.12400449193,0.124001920069,0.123999348367,0.123996776822,0.123994205435,0.123991634205,0.123989063132,0.123986492217,0.123983921459,0.123981350859,0.123978780416,0.123976210131,0.123973640003,0.123971070032,0.123968500219,0.123965930563,0.123963361065,0.123960791723,0.123958222539,0.123955653512,0.123953084643,0.123950515931,0.123947947376,0.123945378978,0.123942810737,0.123940242654,0.123937674727,0.123935106958,0.123932539346,0.123929971891,0.123927404593,0.123924837452,0.123922270469,0.123919703642,0.123917136972,0.12391457046,0.123912004104,0.123909437906,0.123906871864,0.123904305979,0.123901740252,0.123899174681,0.123896609267,0.12389404401,0.12389147891,0.123888913966,0.12388634918,0.12388378455,0.123881220078,0.123878655762,0.123876091602,0.1238735276,0.123870963754,0.123868400065,0.123865836533,0.123863273158,0.123860709939,0.123858146877,0.123855583971,0.123853021222,0.12385045863,0.123847896194,0.123845333915,0.123842771793,0.123840209827,0.123837648018,0.123835086365,0.123832524869,0.123829963529,0.123827402346,0.123824841319,0.123822280448,0.123819719734,0.123817159177,0.123814598776,0.123812038531,0.123809478443,0.123806918511,0.123804358735,0.123801799116,0.123799239653,0.123796680346,0.123794121196,0.123791562202,0.123789003364,0.123786444682,0.123783886157,0.123781327788,0.123778769575,0.123776211518,0.123773653617,0.123771095873,0.123768538284,0.123765980852,0.123763423576,0.123760866456,0.123758309492,0.123755752684,0.123753196032,0.123750639536,0.123748083196,0.123745527012,0.123742970984,0.123740415112,0.123737859396,0.123735303836,0.123732748432,0.123730193183,0.123727638091,0.123725083154,0.123722528373,0.123719973748,0.123717419279,0.123714864966,0.123712310809,0.123709756807,0.123707202961,0.123704649271,0.123702095736,0.123699542357,0.123696989134,0.123694436067,0.123691883155,0.123689330399,0.123686777798,0.123684225353,0.123681673064,0.123679120931,0.123676568952,0.12367401713,0.123671465463,0.123668913951,0.123666362595,0.123663811395,0.12366126035,0.12365870946,0.123656158726,0.123653608147,0.123651057724,0.123648507456,0.123645957344,0.123643407387,0.123640857585,0.123638307938,0.123635758447,0.123633209112,0.123630659931,0.123628110906,0.123625562036,0.123623013321,0.123620464762,0.123617916357,0.123615368108,0.123612820014,0.123610272076,0.123607724292,0.123605176664,0.12360262919,0.123600081872,0.123597534709,0.123594987701,0.123592440848,0.12358989415,0.123587347607,0.123584801219,0.123582254986,0.123579708908,0.123577162986,0.123574617218,0.123572071604,0.123569526146,0.123566980843,0.123564435695,0.123561890701,0.123559345863,0.123556801179,0.12355425665,0.123551712276,0.123549168057,0.123546623992,0.123544080082,0.123541536327,0.123538992727,0.123536449281,0.123533905991,0.123531362854,0.123528819873,0.123526277046,0.123523734374,0.123521191856,0.123518649494,0.123516107285,0.123513565231,0.123511023332,0.123508481588,0.123505939998,0.123503398562,0.123500857281,0.123498316154,0.123495775182,0.123493234365,0.123490693701,0.123488153193,0.123485612838,0.123483072638,0.123480532593,0.123477992702,0.123475452965,0.123472913382,0.123470373954,0.12346783468,0.123465295561,0.123462756596,0.123460217784,0.123457679128,0.123455140625,0.123452602277,0.123450064083,0.123447526043,0.123444988157,0.123442450425,0.123439912848,0.123437375424,0.123434838155,0.12343230104,0.123429764079,0.123427227272,0.123424690619,0.12342215412,0.123419617775,0.123417081584,0.123414545547,0.123412009664,0.123409473935,0.12340693836,0.123404402939,0.123401867672,0.123399332559,0.123396797599,0.123394262793,0.123391728142,0.123389193644,0.1233866593,0.123384125109,0.123381591073,0.12337905719,0.123376523461,0.123373989886,0.123371456465,0.123368923197,0.123366390083,0.123363857122,0.123361324316,0.123358791663,0.123356259163,0.123353726817,0.123351194625,0.123348662586,0.123346130701,0.12334359897,0.123341067392,0.123338535968,0.123336004697,0.123333473579,0.123330942615,0.123328411805,0.123325881148,0.123323350644,0.123320820294,0.123318290097,0.123315760054,0.123313230164,0.123310700428,0.123308170844,0.123305641414,0.123303112138,0.123300583015,0.123298054045,0.123295525228,0.123292996564,0.123290468054,0.123287939697,0.123285411493,0.123282883443,0.123280355545,0.123277827801,0.12327530021,0.123272772772,0.123270245487,0.123267718355,0.123265191377,0.123262664551,0.123260137879,0.123257611359,0.123255084993,0.123252558779,0.123250032719,0.123247506811,0.123244981057,0.123242455455,0.123239930007,0.123237404711,0.123234879568,0.123232354578,0.123229829741,0.123227305057,0.123224780526,0.123222256148,0.123219731922,0.123217207849,0.123214683929,0.123212160162,0.123209636547,0.123207113086,0.123204589777,0.12320206662,0.123199543617,0.123197020766,0.123194498067,0.123191975522,0.123189453129,0.123186930888,0.123184408801,0.123181886865,0.123179365083,0.123176843453,0.123174321975,0.12317180065,0.123169279478,0.123166758458,0.12316423759,0.123161716876,0.123159196313,0.123156675903,0.123154155645,0.12315163554,0.123149115587,0.123146595787,0.123144076139,0.123141556643,0.123139037299,0.123136518108,0.123133999069,0.123131480183,0.123128961449,0.123126442867,0.123123924437,0.123121406159,0.123118888034,0.123116370061,0.12311385224,0.123111334571,0.123108817055,0.12310629969,0.123103782478,0.123101265418,0.12309874851,0.123096231754,0.12309371515,0.123091198698,0.123088682398,0.12308616625,0.123083650254,0.12308113441,0.123078618719,0.123076103179,0.123073587791,0.123071072555,0.12306855747,0.123066042538,0.123063527758,0.123061013129,0.123058498653,0.123055984328,0.123053470155,0.123050956134,0.123048442264,0.123045928547,0.123043414981,0.123040901567,0.123038388305,0.123035875194,0.123033362235,0.123030849428,0.123028336772,0.123025824269,0.123023311916,0.123020799716,0.123018287667,0.123015775769,0.123013264024,0.123010752429,0.123008240987,0.123005729695,0.123003218556,0.123000707568,0.122998196731,0.122995686046,0.122993175512,0.12299066513,0.122988154899,0.12298564482,0.122983134892,0.122980625116,0.12297811549,0.122975606017,0.122973096694,0.122970587523,0.122968078503,0.122965569635,0.122963060918,0.122960552352,0.122958043937,0.122955535673,0.122953027561,0.1229505196,0.12294801179,0.122945504132,0.122942996624,0.122940489268,0.122937982063,0.122935475009,0.122932968106,0.122930461354,0.122927954753,0.122925448304,0.122922942005,0.122920435857,0.122917929861,0.122915424015,0.12291291832,0.122910412777,0.122907907384,0.122905402142,0.122902897052,0.122900392112,0.122897887323,0.122895382685,0.122892878198,0.122890373861,0.122887869676,0.122885365641,0.122882861757,0.122880358024,0.122877854442,0.12287535101,0.122872847729,0.122870344599,0.12286784162,0.122865338791,0.122862836113,0.122860333586,0.12285783121,0.122855328984,0.122852826908,0.122850324984,0.12284782321,0.122845321586,0.122842820113,0.122840318791,0.122837817619,0.122835316598,0.122832815727,0.122830315007,0.122827814437,0.122825314018,0.122822813749,0.12282031363,0.122817813662,0.122815313845,0.122812814178,0.122810314661,0.122807815294,0.122805316078,0.122802817013,0.122800318097,0.122797819332,0.122795320717,0.122792822253,0.122790323938,0.122787825774,0.122785327761,0.122782829897,0.122780332184,0.12277783462,0.122775337207,0.122772839945,0.122770342832,0.122767845869,0.122765349057,0.122762852395,0.122760355882,0.12275785952,0.122755363308,0.122752867246,0.122750371334,0.122747875572,0.12274537996,0.122742884498,0.122740389186,0.122737894023,0.122735399011,0.122732904149,0.122730409437,0.122727914874,0.122725420462,0.122722926199,0.122720432086,0.122717938123,0.12271544431,0.122712950646,0.122710457133,0.122707963769,0.122705470555,0.12270297749,0.122700484576,0.122697991811,0.122695499196,0.12269300673,0.122690514415,0.122688022248,0.122685530232,0.122683038365,0.122680546648,0.12267805508,0.122675563662,0.122673072394,0.122670581275,0.122668090305,0.122665599485,0.122663108815,0.122660618294,0.122658127923,0.122655637701,0.122653147629,0.122650657706,0.122648167932,0.122645678308,0.122643188833,0.122640699508,0.122638210332,0.122635721305,0.122633232428,0.1226307437,0.122628255121,0.122625766692,0.122623278411,0.122620790281,0.122618302299,0.122615814467,0.122613326783,0.122610839249,0.122608351865,0.122605864629,0.122603377543,0.122600890605,0.122598403817,0.122595917178,0.122593430688,0.122590944348,0.122588458156,0.122585972113,0.122583486219,0.122581000475,0.122578514879,0.122576029433,0.122573544135,0.122571058986,0.122568573987,0.122566089136,0.122563604434,0.122561119881,0.122558635477,0.122556151222,0.122553667116,0.122551183159,0.12254869935,0.12254621569,0.122543732179,0.122541248817,0.122538765604,0.122536282539,0.122533799624,0.122531316856,0.122528834238,0.122526351769,0.122523869448,0.122521387275,0.122518905252,0.122516423377,0.12251394165,0.122511460073,0.122508978644,0.122506497363,0.122504016231,0.122501535248,0.122499054413,0.122496573727,0.122494093189,0.1224916128,0.122489132559,0.122486652467,0.122484172523,0.122481692728,0.122479213081,0.122476733583,0.122474254232,0.122471775031,0.122469295977,0.122466817073,0.122464338316,0.122461859708,0.122459381248,0.122456902936,0.122454424773,0.122451946758,0.122449468891,0.122446991172,0.122444513602,0.12244203618,0.122439558906,0.12243708178,0.122434604802,0.122432127973,0.122429651292,0.122427174759,0.122424698373,0.122422222137,0.122419746048,0.122417270107,0.122414794314,0.12241231867,0.122409843173,0.122407367824,0.122404892624,0.122402417571,0.122399942666,0.12239746791,0.122394993301,0.12239251884,0.122390044527,0.122387570362,0.122385096345,0.122382622475,0.122380148754,0.12237767518,0.122375201755,0.122372728477,0.122370255347,0.122367782364,0.12236530953,0.122362836843,0.122360364304,0.122357891912,0.122355419668,0.122352947572,0.122350475624,0.122348003823,0.12234553217,0.122343060665,0.122340589307,0.122338118097,0.122335647035,0.12233317612,0.122330705352,0.122328234733,0.12232576426,0.122323293936,0.122320823758,0.122318353728,0.122315883846,0.122313414111,0.122310944524,0.122308475084,0.122306005792,0.122303536646,0.122301067649,0.122298598798,0.122296130095,0.12229366154,0.122291193132,0.122288724871,0.122286256757,0.122283788791,0.122281320971,0.1222788533,0.122276385775,0.122273918398,0.122271451168,0.122268984085,0.122266517149,0.12226405036,0.122261583719,0.122259117225,0.122256650878,0.122254184678,0.122251718625,0.122249252719,0.12224678696,0.122244321349,0.122241855884,0.122239390567,0.122236925396,0.122234460373,0.122231995496,0.122229530767,0.122227066184,0.122224601748,0.12222213746,0.122219673318,0.122217209323,0.122214745475,0.122212281774,0.12220981822,0.122207354812,0.122204891552,0.122202428438,0.122199965471,0.122197502651,0.122195039977,0.122192577451,0.122190115071,0.122187652838,0.122185190751,0.122182728812,0.122180267019,0.122177805372,0.122175343873,0.12217288252,0.122170421313,0.122167960253,0.12216549934,0.122163038574,0.122160577954,0.12215811748,0.122155657153,0.122153196973,0.122150736939,0.122148277052,0.122145817311,0.122143357717,0.122140898269,0.122138438967,0.122135979812,0.122133520804,0.122131061941,0.122128603226,0.122126144656,0.122123686233,0.122121227956,0.122118769826,0.122116311842,0.122113854004,0.122111396313,0.122108938768,0.122106481369,0.122104024116,0.122101567009,0.122099110049,0.122096653235,0.122094196567,0.122091740046,0.12208928367,0.122086827441,0.122084371358,0.122081915421,0.12207945963,0.122077003985,0.122074548486,0.122072093133,0.122069637927,0.122067182866,0.122064727951,0.122062273183,0.12205981856,0.122057364083,0.122054909753,0.122052455568,0.122050001529,0.122047547636,0.122045093889,0.122042640288,0.122040186833,0.122037733524,0.12203528036,0.122032827343,0.122030374471,0.122027921745,0.122025469165,0.12202301673,0.122020564442,0.122018112299,0.122015660301,0.12201320845,0.122010756744,0.122008305184,0.12200585377,0.122003402501,0.122000951378,0.121998500401,0.121996049569,0.121993598883,0.121991148342,0.121988697947,0.121986247698,0.121983797594,0.121981347635,0.121978897823,0.121976448155,0.121973998634,0.121971549257,0.121969100026,0.121966650941,0.121964202001,0.121961753207,0.121959304557,0.121956856054,0.121954407695,0.121951959482,0.121949511415,0.121947063493,0.121944615716,0.121942168084,0.121939720598,0.121937273257,0.121934826061,0.12193237901,0.121929932105,0.121927485345,0.12192503873,0.121922592261,0.121920145936,0.121917699757,0.121915253723,0.121912807834,0.12191036209,0.121907916491,0.121905471038,0.121903025729,0.121900580566,0.121898135548,0.121895690674,0.121893245946,0.121890801363,0.121888356924,0.121885912631,0.121883468483,0.121881024479,0.121878580621,0.121876136908,0.121873693339,0.121871249915,0.121868806637,0.121866363503,0.121863920514,0.12186147767,0.12185903497,0.121856592416,0.121854150006,0.121851707741,0.121849265621,0.121846823646,0.121844381815,0.121841940129,0.121839498588,0.121837057192,0.12183461594,0.121832174833,0.12182973387,0.121827293053,0.12182485238,0.121822411851,0.121819971467,0.121817531228,0.121815091133,0.121812651183,0.121810211378,0.121807771717,0.1218053322,0.121802892828,0.121800453601,0.121798014518,0.121795575579,0.121793136785,0.121790698135,0.12178825963,0.121785821269,0.121783383053,0.121780944981,0.121778507053,0.12177606927,0.121773631631,0.121771194137,0.121768756786,0.12176631958,0.121763882519,0.121761445601,0.121759008828,0.121756572199,0.121754135715,0.121751699374,0.121749263178,0.121746827126,0.121744391218,0.121741955454,0.121739519835,0.121737084359,0.121734649028,0.121732213841,0.121729778798,0.121727343899,0.121724909144,0.121722474533,0.121720040066,0.121717605743,0.121715171564,0.121712737529,0.121710303639,0.121707869892,0.121705436289,0.12170300283,0.121700569515,0.121698136344,0.121695703316,0.121693270433,0.121690837693,0.121688405098,0.121685972646,0.121683540338,0.121681108174,0.121678676154,0.121676244277,0.121673812544,0.121671380955,0.12166894951,0.121666518208,0.121664087051,0.121661656036,0.121659225166,0.121656794439,0.121654363856,0.121651933417,0.121649503121,0.121647072969,0.12164464296,0.121642213095,0.121639783374,0.121637353796,0.121634924361,0.121632495071,0.121630065923,0.12162763692,0.121625208059,0.121622779343,0.121620350769,0.12161792234,0.121615494053,0.12161306591,0.121610637911,0.121608210055,0.121605782342,0.121603354772,0.121600927346,0.121598500064,0.121596072924,0.121593645928,0.121591219076,0.121588792366,0.1215863658,0.121583939377,0.121581513097,0.121579086961,0.121576660968,0.121574235118,0.121571809411,0.121569383847,0.121566958427,0.121564533149,0.121562108015,0.121559683024,0.121557258176,0.121554833471,0.121552408909,0.121549984491,0.121547560215,0.121545136082,0.121542712093,0.121540288246,0.121537864542,0.121535440982,0.121533017564,0.121530594289,0.121528171158,0.121525748169,0.121523325323,0.12152090262,0.12151848006,0.121516057643,0.121513635368,0.121511213237,0.121508791248,0.121506369402,0.121503947699,0.121501526139,0.121499104721,0.121496683446,0.121494262314,0.121491841325,0.121489420479,0.121486999775,0.121484579214,0.121482158795,0.121479738519,0.121477318386,0.121474898396,0.121472478548,0.121470058843,0.12146763928,0.12146521986,0.121462800582,0.121460381447,0.121457962455,0.121455543605,0.121453124898,0.121450706333,0.121448287911,0.121445869631,0.121443451493,0.121441033498,0.121438615646,0.121436197936,0.121433780368,0.121431362943,0.12142894566,0.121426528519,0.121424111521,0.121421694665,0.121419277951,0.12141686138,0.121414444951,0.121412028664,0.12140961252,0.121407196518,0.121404780658,0.12140236494,0.121399949365,0.121397533931,0.12139511864,0.121392703491,0.121390288485,0.12138787362,0.121385458897,0.121383044317,0.121380629879,0.121378215583,0.121375801429,0.121373387417,0.121370973547,0.121368559819,0.121366146233,0.121363732789,0.121361319487,0.121358906327,0.121356493309,0.121354080433,0.121351667699,0.121349255107,0.121346842657,0.121344430348,0.121342018182,0.121339606157,0.121337194274,0.121334782534,0.121332370935,0.121329959477,0.121327548162,0.121325136988,0.121322725956,0.121320315066,0.121317904318,0.121315493711,0.121313083247,0.121310672923,0.121308262742,0.121305852702,0.121303442804,0.121301033047,0.121298623433,0.121296213959,0.121293804628,0.121291395438,0.121288986389,0.121286577483,0.121284168717,0.121281760094,0.121279351611,0.121276943271,0.121274535072,0.121272127014,0.121269719098,0.121267311323,0.12126490369,0.121262496198,0.121260088847,0.121257681638,0.121255274571,0.121252867644,0.12125046086,0.121248054216,0.121245647714,0.121243241353,0.121240835133,0.121238429055,0.121236023118,0.121233617323,0.121231211668,0.121228806155,0.121226400783,0.121223995552,0.121221590463,0.121219185514,0.121216780707,0.121214376041,0.121211971517,0.121209567133,0.12120716289,0.121204758789,0.121202354828,0.121199951009,0.121197547331,0.121195143794,0.121192740398,0.121190337143,0.121187934029,0.121185531056,0.121183128223,0.121180725532,0.121178322982,0.121175920573,0.121173518305,0.121171116177,0.121168714191,0.121166312345,0.121163910641,0.121161509077,0.121159107654,0.121156706372,0.121154305231,0.12115190423,0.121149503371,0.121147102652,0.121144702074,0.121142301636,0.12113990134,0.121137501184,0.121135101169,0.121132701294,0.121130301561,0.121127901968,0.121125502515,0.121123103203,0.121120704032,0.121118305002,0.121115906112,0.121113507363,0.121111108754,0.121108710286,0.121106311958,0.121103913771,0.121101515725,0.121099117819,0.121096720053,0.121094322428,0.121091924944,0.1210895276,0.121087130396,0.121084733333,0.12108233641,0.121079939628,0.121077542986,0.121075146484,0.121072750123,0.121070353902,0.121067957822,0.121065561882,0.121063166082,0.121060770422,0.121058374903,0.121055979524,0.121053584285,0.121051189187,0.121048794228,0.12104639941,0.121044004733,0.121041610195,0.121039215797,0.12103682154,0.121034427423,0.121032033446,0.121029639609,0.121027245912,0.121024852356,0.121022458939,0.121020065663,0.121017672526,0.12101527953,0.121012886674,0.121010493957,0.121008101381,0.121005708944,0.121003316648,0.121000924492,0.120998532475,0.120996140599,0.120993748862,0.120991357265,0.120988965809,0.120986574492,0.120984183315,0.120981792278,0.12097940138,0.120977010623,0.120974620005,0.120972229527,0.120969839189,0.120967448991,0.120965058932,0.120962669014,0.120960279234,0.120957889595,0.120955500096,0.120953110736,0.120950721515,0.120948332435,0.120945943494,0.120943554693,0.120941166031,0.120938777509,0.120936389127,0.120934000884,0.120931612781,0.120929224817,0.120926836993,0.120924449308,0.120922061763,0.120919674358,0.120917287091,0.120914899965,0.120912512978,0.12091012613,0.120907739422,0.120905352853,0.120902966424,0.120900580134,0.120898193983,0.120895807972,0.1208934221,0.120891036368,0.120888650775,0.120886265321,0.120883880006,0.120881494831,0.120879109795,0.120876724899,0.120874340141,0.120871955523,0.120869571044,0.120867186704,0.120864802504,0.120862418443,0.120860034521,0.120857650738,0.120855267094,0.120852883589,0.120850500224,0.120848116997,0.12084573391,0.120843350961,0.120840968152,0.120838585482,0.120836202951,0.120833820559,0.120831438306,0.120829056192,0.120826674217,0.120824292381,0.120821910684,0.120819529126,0.120817147706,0.120814766426,0.120812385285,0.120810004282,0.120807623419,0.120805242694,0.120802862108,0.120800481661,0.120798101353,0.120795721184,0.120793341153,0.120790961261,0.120788581508,0.120786201894,0.120783822419,0.120781443082,0.120779063884,0.120776684825,0.120774305904,0.120771927122,0.120769548479,0.120767169974,0.120764791608,0.120762413381,0.120760035292,0.120757657342,0.120755279531,0.120752901858,0.120750524324,0.120748146928,0.120745769671,0.120743392552,0.120741015572,0.12073863873,0.120736262027,0.120733885462,0.120731509036,0.120729132748,0.120726756599,0.120724380588,0.120722004715,0.120719628981,0.120717253386,0.120714877928,0.120712502609,0.120710127428,0.120707752386,0.120705377482,0.120703002716,0.120700628089,0.1206982536,0.120695879249,0.120693505036,0.120691130962,0.120688757025,0.120686383227,0.120684009568,0.120681636046,0.120679262663,0.120676889417,0.12067451631,0.120672143341,0.12066977051,0.120667397818,0.120665025263,0.120662652846,0.120660280568,0.120657908427,0.120655536425,0.12065316456,0.120650792834,0.120648421246,0.120646049795,0.120643678483,0.120641307308,0.120638936272,0.120636565373,0.120634194613,0.12063182399,0.120629453505,0.120627083158,0.120624712949,0.120622342878,0.120619972944,0.120617603149,0.120615233491,0.120612863971,0.120610494589,0.120608125344,0.120605756238,0.120603387269,0.120601018438,0.120598649744,0.120596281188,0.12059391277,0.12059154449,0.120589176347,0.120586808342,0.120584440475,0.120582072745,0.120579705153,0.120577337699,0.120574970382,0.120572603203,0.120570236161,0.120567869257,0.12056550249,0.120563135861,0.120560769369,0.120558403015,0.120556036799,0.120553670719,0.120551304778,0.120548938974,0.120546573307,0.120544207777,0.120541842385,0.120539477131,0.120537112014,0.120534747034,0.120532382191,0.120530017486,0.120527652919,0.120525288488,0.120522924195,0.120520560039,0.120518196021,0.120515832139,0.120513468395,0.120511104788,0.120508741319,0.120506377987,0.120504014791,0.120501651733,0.120499288813,0.120496926029,0.120494563382,0.120492200873,0.120489838501,0.120487476266,0.120485114168,0.120482752207,0.120480390383,0.120478028696,0.120475667146,0.120473305734,0.120470944458,0.120468583319,0.120466222318,0.120463861453,0.120461500725,0.120459140134,0.120456779681,0.120454419364,0.120452059184,0.12044969914,0.120447339234,0.120444979465,0.120442619832,0.120440260337,0.120437900978,0.120435541756,0.120433182671,0.120430823722,0.120428464911,0.120426106236,0.120423747698,0.120421389297,0.120419031032,0.120416672904,0.120414314913,0.120411957059,0.120409599341,0.12040724176,0.120404884315,0.120402527008,0.120400169837,0.120397812802,0.120395455904,0.120393099143,0.120390742518,0.12038838603,0.120386029678,0.120383673463,0.120381317384,0.120378961442,0.120376605637,0.120374249968,0.120371894435,0.120369539039,0.120367183779,0.120364828656,0.120362473669,0.120360118818,0.120357764104,0.120355409527,0.120353055085,0.120350700781,0.120348346612,0.12034599258,0.120343638684,0.120341284924,0.120338931301,0.120336577814,0.120334224463,0.120331871248,0.12032951817,0.120327165228,0.120324812422,0.120322459752,0.120320107219,0.120317754822,0.120315402561,0.120313050436,0.120310698447,0.120308346594,0.120305994878,0.120303643297,0.120301291853,0.120298940544,0.120296589372,0.120294238336,0.120291887436,0.120289536671,0.120287186043,0.120284835551,0.120282485195,0.120280134975,0.12027778489,0.120275434942,0.12027308513,0.120270735453,0.120268385913,0.120266036508,0.120263687239,0.120261338106,0.120258989109,0.120256640248,0.120254291523,0.120251942933,0.120249594479,0.120247246161,0.120244897979,0.120242549933,0.120240202022,0.120237854247,0.120235506608,0.120233159104,0.120230811737,0.120228464504,0.120226117408,0.120223770447,0.120221423622,0.120219076933,0.120216730379,0.12021438396,0.120212037678,0.120209691531,0.120207345519,0.120204999643,0.120202653903,0.120200308298,0.120197962829,0.120195617495,0.120193272296,0.120190927234,0.120188582306,0.120186237514,0.120183892858,0.120181548337,0.120179203951,0.120176859701,0.120174515586,0.120172171606,0.120169827762,0.120167484054,0.12016514048,0.120162797042,0.120160453739,0.120158110572,0.12015576754,0.120153424643,0.120151081881,0.120148739255,0.120146396764,0.120144054408,0.120141712187,0.120139370101,0.120137028151,0.120134686336,0.120132344656,0.120130003111,0.120127661701,0.120125320427,0.120122979287,0.120120638283,0.120118297414,0.12011595668,0.12011361608,0.120111275616,0.120108935287,0.120106595093,0.120104255034,0.12010191511,0.120099575321,0.120097235667,0.120094896148,0.120092556764,0.120090217515,0.1200878784,0.120085539421,0.120083200576,0.120080861867,0.120078523292,0.120076184852,0.120073846547,0.120071508377,0.120069170341,0.120066832441,0.120064494675,0.120062157044,0.120059819547,0.120057482186,0.120055144959,0.120052807867,0.12005047091,0.120048134087,0.120045797399,0.120043460846,0.120041124427,0.120038788143,0.120036451994,0.120034115979,0.120031780099,0.120029444354,0.120027108743,0.120024773267,0.120022437925,0.120020102718,0.120017767645,0.120015432707,0.120013097903,0.120010763234,0.120008428699,0.120006094299,0.120003760034,0.120001425902,0.119999091906,0.119996758043,0.119994424315,0.119992090722,0.119989757262,0.119987423938,0.119985090747,0.119982757691,0.119980424769,0.119978091982,0.119975759329,0.11997342681,0.119971094425,0.119968762175,0.119966430059,0.119964098077,0.11996176623,0.119959434516,0.119957102937,0.119954771492,0.119952440181,0.119950109005,0.119947777962,0.119945447054,0.11994311628,0.11994078564,0.119938455134,0.119936124762,0.119933794525,0.119931464421,0.119929134451,0.119926804616,0.119924474914,0.119922145347,0.119919815913,0.119917486614,0.119915157448,0.119912828417,0.119910499519,0.119908170755,0.119905842126,0.11990351363,0.119901185268,0.11989885704,0.119896528946,0.119894200985,0.119891873159,0.119889545466,0.119887217908,0.119884890483,0.119882563192,0.119880236034,0.119877909011,0.119875582121,0.119873255365,0.119870928743,0.119868602254,0.119866275899,0.119863949678,0.119861623591,0.119859297637,0.119856971817,0.11985464613,0.119852320577,0.119849995158,0.119847669873,0.119845344721,0.119843019702,0.119840694817,0.119838370066,0.119836045449,0.119833720964,0.119831396614,0.119829072397,0.119826748313,0.119824424363,0.119822100546,0.119819776863,0.119817453313,0.119815129897,0.119812806614,0.119810483465,0.119808160449,0.119805837566,0.119803514817,0.119801192201,0.119798869718,0.119796547369,0.119794225153,0.119791903071,0.119789581122,0.119787259306,0.119784937623,0.119782616073,0.119780294657,0.119777973374,0.119775652224,0.119773331208,0.119771010325,0.119768689575,0.119766368958,0.119764048474,0.119761728123,0.119759407906,0.119757087821,0.11975476787,0.119752448052,0.119750128367,0.119747808815,0.119745489396,0.11974317011,0.119740850957,0.119738531937,0.119736213051,0.119733894297,0.119731575676,0.119729257188,0.119726938833,0.119724620611,0.119722302522,0.119719984566,0.119717666743,0.119715349053,0.119713031496,0.119710714071,0.119708396779,0.119706079621,0.119703762595,0.119701445702,0.119699128941,0.119696812314,0.119694495819,0.119692179457,0.119689863228,0.119687547132,0.119685231168,0.119682915337,0.119680599639,0.119678284073,0.119675968641,0.11967365334,0.119671338173,0.119669023138,0.119666708236,0.119664393466,0.119662078829,0.119659764325,0.119657449953,0.119655135714,0.119652821607,0.119650507633,0.119648193792,0.119645880083,0.119643566506,0.119641253062,0.119638939751,0.119636626572,0.119634313525,0.119632000611,0.119629687829,0.11962737518,0.119625062663,0.119622750279,0.119620438027,0.119618125907,0.11961581392,0.119613502065,0.119611190342,0.119608878752,0.119606567294,0.119604255969,0.119601944775,0.119599633714,0.119597322785,0.119595011989,0.119592701324,0.119590390792,0.119588080393,0.119585770125,0.119583459989,0.119581149986,0.119578840115,0.119576530376,0.119574220769,0.119571911294,0.119569601952,0.119567292741,0.119564983663,0.119562674717,0.119560365902,0.11955805722,0.11955574867,0.119553440252,0.119551131966,0.119548823812,0.11954651579,0.1195442079,0.119541900142,0.119539592515,0.119537285021,0.119534977659,0.119532670428,0.11953036333,0.119528056363,0.119525749529,0.119523442826,0.119521136255,0.119518829816,0.119516523508,0.119514217333,0.119511911289,0.119509605377,0.119507299597,0.119504993949,0.119502688432,0.119500383048,0.119498077795,0.119495772673,0.119493467684,0.119491162826,0.119488858099,0.119486553505,0.119484249042,0.119481944711,0.119479640511,0.119477336443,0.119475032506,0.119472728702,0.119470425028,0.119468121487,0.119465818077,0.119463514798,0.119461211651,0.119458908636,0.119456605752,0.119454302999,0.119452000378,0.119449697889,0.119447395531,0.119445093304,0.119442791209,0.119440489245,0.119438187413,0.119435885712,0.119433584143,0.119431282704,0.119428981398,0.119426680222,0.119424379178,0.119422078265,0.119419777484,0.119417476834,0.119415176315,0.119412875927,0.119410575671,0.119408275546,0.119405975552,0.11940367569,0.119401375958,0.119399076358,0.119396776889,0.119394477551,0.119392178345,0.119389879269,0.119387580325,0.119385281512,0.11938298283,0.119380684279,0.119378385859,0.11937608757,0.119373789413,0.119371491386,0.11936919349,0.119366895726,0.119364598092,0.11936230059,0.119360003218,0.119357705978,0.119355408868,0.119353111889,0.119350815042,0.119348518325,0.119346221739,0.119343925284,0.11934162896,0.119339332767,0.119337036705,0.119334740773,0.119332444973,0.119330149303,0.119327853764,0.119325558356,0.119323263079,0.119320967933,0.119318672917,0.119316378032,0.119314083278,0.119311788654,0.119309494162,0.1193071998,0.119304905568,0.119302611468,0.119300317498,0.119298023659,0.11929572995,0.119293436372,0.119291142925,0.119288849608,0.119286556422,0.119284263367,0.119281970442,0.119279677647,0.119277384984,0.11927509245,0.119272800048,0.119270507775,0.119268215634,0.119265923622,0.119263631742,0.119261339992,0.119259048372,0.119256756882,0.119254465524,0.119252174295,0.119249883197,0.119247592229,0.119245301392,0.119243010685,0.119240720109,0.119238429663,0.119236139347,0.119233849161,0.119231559106,0.119229269181,0.119226979386,0.119224689722,0.119222400188,0.119220110784,0.119217821511,0.119215532367,0.119213243354,0.119210954471,0.119208665719,0.119206377096,0.119204088604,0.119201800241,0.119199512009,0.119197223907,0.119194935935,0.119192648094,0.119190360382,0.119188072801,0.119185785349,0.119183498028,0.119181210836,0.119178923775,0.119176636844,0.119174350042,0.119172063371,0.11916977683,0.119167490418,0.119165204137,0.119162917986,0.119160631964,0.119158346072,0.119156060311,0.119153774679,0.119151489177,0.119149203805,0.119146918563,0.119144633451,0.119142348468,0.119140063616,0.119137778893,0.1191354943,0.119133209837,0.119130925503,0.1191286413,0.119126357226,0.119124073282,0.119121789467,0.119119505783,0.119117222228,0.119114938802,0.119112655507,0.119110372341,0.119108089305,0.119105806398,0.119103523621,0.119101240974,0.119098958456,0.119096676068,0.119094393809,0.11909211168,0.119089829681,0.119087547811,0.11908526607,0.11908298446,0.119080702978,0.119078421627,0.119076140404,0.119073859311,0.119071578348,0.119069297514,0.11906701681,0.119064736235,0.119062455789,0.119060175473,0.119057895286,0.119055615229,0.119053335301,0.119051055502,0.119048775832,0.119046496292,0.119044216882,0.1190419376,0.119039658448,0.119037379425,0.119035100532,0.119032821768,0.119030543133,0.119028264627,0.11902598625,0.119023708003,0.119021429885,0.119019151896,0.119016874036,0.119014596306,0.119012318704,0.119010041232,0.119007763889,0.119005486675,0.11900320959,0.119000932634,0.118998655807,0.118996379109,0.118994102541,0.118991826101,0.11898954979,0.118987273609,0.118984997556,0.118982721633,0.118980445838,0.118978170173,0.118975894636,0.118973619228,0.11897134395,0.1189690688,0.118966793779,0.118964518887,0.118962244124,0.11895996949,0.118957694984,0.118955420608,0.11895314636,0.118950872241,0.118948598251,0.11894632439,0.118944050658,0.118941777054,0.118939503579,0.118937230233,0.118934957016,0.118932683927,0.118930410967,0.118928138136,0.118925865434,0.11892359286,0.118921320415,0.118919048098,0.11891677591,0.118914503851,0.118912231921,0.118909960119,0.118907688445,0.118905416901,0.118903145485,0.118900874197,0.118898603038,0.118896332007,0.118894061106,0.118891790332,0.118889519687,0.118887249171,0.118884978783,0.118882708523,0.118880438392,0.11887816839,0.118875898516,0.11887362877,0.118871359153,0.118869089664,0.118866820303,0.118864551071,0.118862281967,0.118860012992,0.118857744145,0.118855475426,0.118853206836,0.118850938374,0.11884867004,0.118846401834,0.118844133757,0.118841865808,0.118839597987,0.118837330295,0.11883506273,0.118832795294,0.118830527986,0.118828260807,0.118825993755,0.118823726832,0.118821460036,0.118819193369,0.11881692683,0.118814660419,0.118812394137,0.118810127982,0.118807861955,0.118805596057,0.118803330286,0.118801064644,0.11879879913,0.118796533743,0.118794268485,0.118792003355,0.118789738352,0.118787473478,0.118785208731,0.118782944113,0.118780679622,0.11877841526,0.118776151025,0.118773886918,0.118771622939,0.118769359088,0.118767095365,0.11876483177,0.118762568302,0.118760304963,0.118758041751,0.118755778667,0.118753515711,0.118751252882,0.118748990181,0.118746727609,0.118744465163,0.118742202846,0.118739940656,0.118737678594,0.11873541666,0.118733154853,0.118730893174,0.118728631623,0.118726370199,0.118724108904,0.118721847735,0.118719586694,0.118717325781,0.118715064996,0.118712804338,0.118710543807,0.118708283405,0.118706023129,0.118703762982,0.118701502961,0.118699243069,0.118696983304,0.118694723666,0.118692464156,0.118690204773,0.118687945517,0.11868568639,0.118683427389,0.118681168516,0.118678909771,0.118676651152,0.118674392661,0.118672134298,0.118669876062,0.118667617953,0.118665359972,0.118663102117,0.118660844391,0.118658586791,0.118656329319,0.118654071974,0.118651814756,0.118649557666,0.118647300703,0.118645043867,0.118642787158,0.118640530576,0.118638274122,0.118636017795,0.118633761595,0.118631505522,0.118629249576,0.118626993758,0.118624738066,0.118622482502,0.118620227065,0.118617971755,0.118615716572,0.118613461516,0.118611206587,0.118608951785,0.11860669711,0.118604442562,0.118602188141,0.118599933847,0.11859767968,0.118595425641,0.118593171728,0.118590917942,0.118588664282,0.11858641075,0.118584157345,0.118581904067,0.118579650915,0.118577397891,0.118575144993,0.118572892222,0.118570639578,0.118568387061,0.118566134671,0.118563882407,0.11856163027,0.11855937826,0.118557126377,0.118554874621,0.118552622991,0.118550371488,0.118548120112,0.118545868862,0.118543617739,0.118541366743,0.118539115874,0.118536865131,0.118534614515,0.118532364025,0.118530113662,0.118527863426,0.118525613317,0.118523363334,0.118521113477,0.118518863747,0.118516614144,0.118514364667,0.118512115317,0.118509866093,0.118507616996,0.118505368025,0.118503119181,0.118500870463,0.118498621872,0.118496373407,0.118494125069,0.118491876857,0.118489628771,0.118487380812,0.118485132979,0.118482885273,0.118480637693,0.11847839024,0.118476142912,0.118473895711,0.118471648637,0.118469401689,0.118467154867,0.118464908171,0.118462661602,0.118460415159,0.118458168842,0.118455922651,0.118453676587,0.118451430649,0.118449184837,0.118446939151,0.118444693591,0.118442448158,0.118440202851,0.11843795767,0.118435712615,0.118433467686,0.118431222883,0.118428978207,0.118426733656,0.118424489232,0.118422244933,0.118420000761,0.118417756715,0.118415512795,0.118413269001,0.118411025333,0.11840878179,0.118406538374,0.118404295084,0.11840205192,0.118399808882,0.118397565969,0.118395323183,0.118393080523,0.118390837988,0.118388595579,0.118386353297,0.11838411114,0.118381869109,0.118379627203,0.118377385424,0.118375143771,0.118372902243,0.118370660841,0.118368419565,0.118366178415,0.11836393739,0.118361696491,0.118359455718,0.118357215071,0.118354974549,0.118352734154,0.118350493883,0.118348253739,0.11834601372,0.118343773827,0.11834153406,0.118339294418,0.118337054901,0.118334815511,0.118332576246,0.118330337107,0.118328098093,0.118325859204,0.118323620442,0.118321381805,0.118319143293,0.118316904907,0.118314666647,0.118312428512,0.118310190502,0.118307952618,0.118305714859,0.118303477226,0.118301239719,0.118299002336,0.118296765079,0.118294527948,0.118292290942,0.118290054061,0.118287817306,0.118285580676,0.118283344172,0.118281107793,0.118278871539,0.11827663541,0.118274399407,0.118272163529,0.118269927776,0.118267692149,0.118265456647,0.11826322127,0.118260986018,0.118258750892,0.118256515891,0.118254281015,0.118252046264,0.118249811638,0.118247577138,0.118245342763,0.118243108513,0.118240874388,0.118238640388,0.118236406513,0.118234172763,0.118231939139,0.118229705639,0.118227472265,0.118225239015,0.118223005891,0.118220772892,0.118218540017,0.118216307268,0.118214074644,0.118211842145,0.11820960977,0.118207377521,0.118205145397,0.118202913397,0.118200681523,0.118198449773,0.118196218148,0.118193986648,0.118191755274,0.118189524023,0.118187292898,0.118185061898,0.118182831022,0.118180600272,0.118178369646,0.118176139145,0.118173908768,0.118171678517,0.11816944839,0.118167218388,0.118164988511,0.118162758758,0.118160529131,0.118158299628,0.118156070249,0.118153840996,0.118151611867,0.118149382862,0.118147153983,0.118144925228,0.118142696597,0.118140468091,0.11813823971,0.118136011454,0.118133783321,0.118131555314,0.118129327431,0.118127099673,0.118124872039,0.11812264453,0.118120417145,0.118118189885,0.118115962749,0.118113735738,0.118111508851,0.118109282089,0.118107055451,0.118104828937,0.118102602548,0.118100376284,0.118098150144,0.118095924128,0.118093698236,0.118091472469,0.118089246827,0.118087021308,0.118084795914,0.118082570644,0.118080345499,0.118078120478,0.118075895581,0.118073670809,0.11807144616,0.118069221636,0.118066997237,0.118064772961,0.11806254881,0.118060324783,0.11805810088,0.118055877101,0.118053653446,0.118051429916,0.11804920651,0.118046983228,0.11804476007,0.118042537036,0.118040314126,0.11803809134,0.118035868679,0.118033646141,0.118031423728,0.118029201438,0.118026979273,0.118024757232,0.118022535314,0.118020313521,0.118018091852,0.118015870307,0.118013648885,0.118011427588,0.118009206414,0.118006985365,0.118004764439,0.118002543638,0.11800032296,0.117998102406,0.117995881976,0.11799366167,0.117991441487,0.117989221429,0.117987001494,0.117984781684,0.117982561997,0.117980342433,0.117978122994,0.117975903678,0.117973684487,0.117971465418,0.117969246474,0.117967027653,0.117964808957,0.117962590383,0.117960371934,0.117958153608,0.117955935406,0.117953717327,0.117951499373,0.117949281541,0.117947063834,0.11794484625,0.11794262879,0.117940411453,0.11793819424,0.11793597715,0.117933760184,0.117931543342,0.117929326623,0.117927110027,0.117924893556,0.117922677207,0.117920460982,0.117918244881,0.117916028903,0.117913813049,0.117911597318,0.11790938171,0.117907166226,0.117904950865,0.117902735628,0.117900520514,0.117898305523,0.117896090656,0.117893875912,0.117891661292,0.117889446795,0.117887232421,0.11788501817,0.117882804043,0.117880590039,0.117878376159,0.117876162401,0.117873948767,0.117871735256,0.117869521869,0.117867308604,0.117865095463,0.117862882445,0.11786066955,0.117858456779,0.11785624413,0.117854031605,0.117851819203,0.117849606924,0.117847394768,0.117845182735,0.117842970826,0.117840759039,0.117838547375,0.117836335835,0.117834124418,0.117831913123,0.117829701952,0.117827490904,0.117825279978,0.117823069176,0.117820858497,0.11781864794,0.117816437507,0.117814227197,0.117812017009,0.117809806945,0.117807597003,0.117805387184,0.117803177489,0.117800967916,0.117798758466,0.117796549138,0.117794339934,0.117792130853,0.117789921894,0.117787713058,0.117785504345,0.117783295755,0.117781087287,0.117778878943,0.117776670721,0.117774462622,0.117772254645,0.117770046792,0.117767839061,0.117765631452,0.117763423967,0.117761216604,0.117759009364,0.117756802246,0.117754595251,0.117752388379,0.117750181629,0.117747975002,0.117745768498,0.117743562116,0.117741355857,0.11773914972,0.117736943706,0.117734737815,0.117732532046,0.117730326399,0.117728120875,0.117725915474,0.117723710195,0.117721505038,0.117719300004,0.117717095093,0.117714890304,0.117712685637,0.117710481093,0.117708276671,0.117706072372,0.117703868195,0.11770166414,0.117699460208,0.117697256398,0.11769505271,0.117692849145,0.117690645702,0.117688442382,0.117686239183,0.117684036108,0.117681833154,0.117679630322,0.117677427613,0.117675225026,0.117673022562,0.117670820219,0.117668617999,0.117666415901,0.117664213925,0.117662012072,0.11765981034,0.117657608731,0.117655407244,0.117653205879,0.117651004636,0.117648803515,0.117646602516,0.11764440164,0.117642200885,0.117640000253,0.117637799743,0.117635599354,0.117633399088,0.117631198944,0.117628998922,0.117626799021,0.117624599243,0.117622399587,0.117620200053,0.11761800064,0.11761580135,0.117613602181,0.117611403135,0.11760920421,0.117607005408,0.117604806727,0.117602608168,0.117600409731,0.117598211416,0.117596013222,0.117593815151,0.117591617201,0.117589419374,0.117587221667,0.117585024083,0.117582826621,0.11758062928,0.117578432061,0.117576234964,0.117574037989,0.117571841135,0.117569644403,0.117567447793,0.117565251304,0.117563054937,0.117560858692,0.117558662568,0.117556466566,0.117554270686,0.117552074927,0.11754987929,0.117547683775,0.117545488381,0.117543293109,0.117541097958,0.117538902929,0.117536708021,0.117534513235,0.117532318571,0.117530124028,0.117527929606,0.117525735306,0.117523541128,0.117521347071,0.117519153135,0.117516959321,0.117514765628,0.117512572057,0.117510378607,0.117508185279,0.117505992072,0.117503798986,0.117501606022,0.117499413179,0.117497220457,0.117495027857,0.117492835378,0.117490643021,0.117488450784,0.117486258669,0.117484066676,0.117481874803,0.117479683052,0.117477491422,0.117475299914,0.117473108526,0.11747091726,0.117468726115,0.117466535091,0.117464344189,0.117462153407,0.117459962747,0.117457772208,0.11745558179,0.117453391493,0.117451201317,0.117449011263,0.117446821329,0.117444631517,0.117442441826,0.117440252255,0.117438062806,0.117435873478,0.117433684271,0.117431495185,0.117429306219,0.117427117375,0.117424928652,0.11742274005,0.117420551569,0.117418363209,0.117416174969,0.117413986851,0.117411798854,0.117409610977,0.117407423221,0.117405235587,0.117403048073,0.11740086068,0.117398673408,0.117396486256,0.117394299226,0.117392112316,0.117389925528,0.11738773886,0.117385552312,0.117383365886,0.11738117958,0.117378993395,0.117376807331,0.117374621388,0.117372435565,0.117370249863,0.117368064282,0.117365878821,0.117363693481,0.117361508262,0.117359323164,0.117357138186,0.117354953328,0.117352768592,0.117350583976,0.11734839948,0.117346215105,0.117344030851,0.117341846718,0.117339662704,0.117337478812,0.11733529504,0.117333111388,0.117330927857,0.117328744447,0.117326561157,0.117324377987,0.117322194938,0.11732001201,0.117317829202,0.117315646514,0.117313463947,0.1173112815,0.117309099174,0.117306916968,0.117304734882,0.117302552917,0.117300371072,0.117298189347,0.117296007743,0.117293826259,0.117291644896,0.117289463653,0.11728728253,0.117285101527,0.117282920645,0.117280739883,0.117278559241,0.117276378719,0.117274198318,0.117272018037,0.117269837876,0.117267657835,0.117265477914,0.117263298114,0.117261118434,0.117258938874,0.117256759434,0.117254580114,0.117252400915,0.117250221835,0.117248042876,0.117245864037,0.117243685317,0.117241506718,0.117239328239,0.11723714988,0.117234971641,0.117232793522,0.117230615523,0.117228437644,0.117226259885,0.117224082246,0.117221904727,0.117219727328,0.117217550049,0.11721537289,0.117213195851,0.117211018932,0.117208842132,0.117206665453,0.117204488893,0.117202312453,0.117200136134,0.117197959934,0.117195783853,0.117193607893,0.117191432053,0.117189256332,0.117187080731,0.11718490525,0.117182729889,0.117180554647,0.117178379525,0.117176204523,0.117174029641,0.117171854878,0.117169680235,0.117167505712,0.117165331309,0.117163157025,0.117160982861,0.117158808816,0.117156634892,0.117154461086,0.117152287401,0.117150113835,0.117147940389,0.117145767062,0.117143593855,0.117141420767,0.117139247799,0.11713707495,0.117134902222,0.117132729612,0.117130557122,0.117128384752,0.117126212501,0.11712404037,0.117121868358,0.117119696465,0.117117524692,0.117115353039,0.117113181505,0.11711101009,0.117108838795,0.117106667619,0.117104496562,0.117102325625,0.117100154807,0.117097984109,0.11709581353,0.11709364307,0.11709147273,0.117089302509,0.117087132407,0.117084962424,0.117082792561,0.117080622817,0.117078453193,0.117076283687,0.117074114301,0.117071945034,0.117069775886,0.117067606858,0.117065437948,0.117063269158,0.117061100487,0.117058931935,0.117056763503,0.117054595189,0.117052426995,0.11705025892,0.117048090963,0.117045923126,0.117043755408,0.117041587809,0.11703942033,0.117037252969,0.117035085727,0.117032918604,0.117030751601,0.117028584716,0.11702641795,0.117024251304,0.117022084776,0.117019918367,0.117017752078,0.117015585907,0.117013419855,0.117011253922,0.117009088108,0.117006922413,0.117004756837,0.117002591379,0.117000426041,0.116998260821,0.11699609572,0.116993930738,0.116991765875,0.116989601131,0.116987436506,0.116985271999,0.116983107611,0.116980943342,0.116978779192,0.11697661516,0.116974451247,0.116972287453,0.116970123778,0.116967960221,0.116965796783,0.116963633464,0.116961470263,0.116959307181,0.116957144218,0.116954981374,0.116952818648,0.11695065604,0.116948493552,0.116946331181,0.11694416893,0.116942006797,0.116939844783,0.116937682887,0.11693552111,0.116933359451,0.116931197911,0.116929036489,0.116926875186,0.116924714001,0.116922552935,0.116920391987,0.116918231158,0.116916070447,0.116913909855,0.116911749381,0.116909589026,0.116907428789,0.11690526867,0.11690310867,0.116900948788,0.116898789024,0.116896629379,0.116894469853,0.116892310444,0.116890151154,0.116887991982,0.116885832929,0.116883673994,0.116881515177,0.116879356478,0.116877197898,0.116875039436,0.116872881092,0.116870722866,0.116868564759,0.116866406769,0.116864248898,0.116862091146,0.116859933511,0.116857775994,0.116855618596,0.116853461316,0.116851304154,0.11684914711,0.116846990184,0.116844833377,0.116842676687,0.116840520116,0.116838363662,0.116836207327,0.116834051109,0.11683189501,0.116829739029,0.116827583166,0.116825427421,0.116823271793,0.116821116284,0.116818960893,0.11681680562,0.116814650464,0.116812495427,0.116810340508,0.116808185706,0.116806031022,0.116803876457,0.116801722009,0.116799567679,0.116797413467,0.116795259373,0.116793105396,0.116790951538,0.116788797797,0.116786644174,0.116784490669,0.116782337282,0.116780184013,0.116778030861,0.116775877827,0.116773724911,0.116771572113,0.116769419432,0.116767266869,0.116765114424,0.116762962096,0.116760809886,0.116758657794,0.11675650582,0.116754353963,0.116752202224,0.116750050602,0.116747899098,0.116745747712,0.116743596443,0.116741445292,0.116739294259,0.116737143343,0.116734992544,0.116732841863,0.1167306913,0.116728540854,0.116726390526,0.116724240316,0.116722090222,0.116719940247,0.116717790388,0.116715640648,0.116713491024,0.116711341519,0.11670919213,0.116707042859,0.116704893706,0.116702744669,0.116700595751,0.116698446949,0.116696298265,0.116694149699,0.116692001249,0.116689852917,0.116687704703,0.116685556606,0.116683408626,0.116681260763,0.116679113018,0.116676965389,0.116674817879,0.116672670485,0.116670523209,0.11666837605,0.116666229008,0.116664082083,0.116661935276,0.116659788585,0.116657642012,0.116655495557,0.116653349218,0.116651202996,0.116649056892,0.116646910905,0.116644765035,0.116642619281,0.116640473646,0.116638328127,0.116636182725,0.11663403744,0.116631892273,0.116629747222,0.116627602288,0.116625457472,0.116623312772,0.11662116819,0.116619023724,0.116616879376,0.116614735144,0.11661259103,0.116610447032,0.116608303152,0.116606159388,0.116604015741,0.116601872211,0.116599728798,0.116597585502,0.116595442323,0.116593299261,0.116591156315,0.116589013486,0.116586870775,0.11658472818,0.116582585701,0.11658044334,0.116578301096,0.116576158968,0.116574016957,0.116571875063,0.116569733285,0.116567591624,0.116565450081,0.116563308653,0.116561167343,0.116559026149,0.116556885072,0.116554744111,0.116552603268,0.11655046254,0.11654832193,0.116546181436,0.116544041059,0.116541900798,0.116539760654,0.116537620627,0.116535480716,0.116533340922,0.116531201244,0.116529061683,0.116526922239,0.116524782911,0.116522643699,0.116520504604,0.116518365626,0.116516226764,0.116514088019,0.11651194939,0.116509810877,0.116507672481,0.116505534201,0.116503396038,0.116501257991,0.116499120061,0.116496982247,0.11649484455,0.116492706968,0.116490569504,0.116488432155,0.116486294923,0.116484157807,0.116482020808,0.116479883925,0.116477747158,0.116475610507,0.116473473973,0.116471337555,0.116469201253,0.116467065068,0.116464928999,0.116462793046,0.116460657209,0.116458521489,0.116456385884,0.116454250396,0.116452115024,0.116449979768,0.116447844629,0.116445709605,0.116443574698,0.116441439907,0.116439305232,0.116437170673,0.11643503623,0.116432901903,0.116430767693,0.116428633598,0.11642649962,0.116424365757,0.116422232011,0.11642009838,0.116417964866,0.116415831467,0.116413698185,0.116411565019,0.116409431968,0.116407299034,0.116405166215,0.116403033512,0.116400900926,0.116398768455,0.1163966361,0.116394503861,0.116392371738,0.116390239731,0.11638810784,0.116385976065,0.116383844405,0.116381712861,0.116379581433,0.116377450121,0.116375318925,0.116373187845,0.11637105688,0.116368926031,0.116366795298,0.116364664681,0.116362534179,0.116360403793,0.116358273523,0.116356143369,0.11635401333,0.116351883407,0.116349753599,0.116347623908,0.116345494332,0.116343364871,0.116341235527,0.116339106298,0.116336977184,0.116334848186,0.116332719304,0.116330590538,0.116328461887,0.116326333351,0.116324204931,0.116322076627,0.116319948438,0.116317820365,0.116315692407,0.116313564565,0.116311436838,0.116309309227,0.116307181731,0.116305054351,0.116302927086,0.116300799936,0.116298672902,0.116296545984,0.116294419181,0.116292292493,0.116290165921,0.116288039464,0.116285913122,0.116283786896,0.116281660785,0.11627953479,0.116277408909,0.116275283145,0.116273157495,0.116271031961,0.116268906542,0.116266781238,0.11626465605,0.116262530977,0.116260406019,0.116258281176,0.116256156449,0.116254031837,0.11625190734,0.116249782958,0.116247658692,0.11624553454,0.116243410504,0.116241286583,0.116239162777,0.116237039087,0.116234915511,0.116232792051,0.116230668705,0.116228545475,0.11622642236,0.11622429936,0.116222176475,0.116220053705,0.11621793105,0.11621580851,0.116213686085,0.116211563775,0.11620944158,0.116207319501,0.116205197536,0.116203075686,0.116200953951,0.116198832331,0.116196710826,0.116194589436,0.116192468161,0.116190347001,0.116188225955,0.116186105025,0.116183984209,0.116181863509,0.116179742923,0.116177622452,0.116175502096,0.116173381854,0.116171261728,0.116169141716,0.116167021819,0.116164902037,0.11616278237,0.116160662818,0.11615854338,0.116156424057,0.116154304849,0.116152185755,0.116150066776,0.116147947912,0.116145829163,0.116143710528,0.116141592008,0.116139473603,0.116137355312,0.116135237136,0.116133119075,0.116131001128,0.116128883296,0.116126765579,0.116124647976,0.116122530487,0.116120413114,0.116118295854,0.11611617871,0.11611406168,0.116111944764,0.116109827963,0.116107711277,0.116105594705,0.116103478247,0.116101361904,0.116099245675,0.116097129561,0.116095013562,0.116092897677,0.116090781906,0.11608866625,0.116086550708,0.11608443528,0.116082319967,0.116080204768,0.116078089684,0.116075974714,0.116073859859,0.116071745117,0.11606963049,0.116067515978,0.116065401579,0.116063287295,0.116061173126,0.11605905907,0.116056945129,0.116054831302,0.11605271759,0.116050603991,0.116048490507,0.116046377137,0.116044263881,0.11604215074,0.116040037713,0.116037924799,0.116035812,0.116033699316,0.116031586745,0.116029474288,0.116027361946,0.116025249718,0.116023137604,0.116021025604,0.116018913718,0.116016801946,0.116014690288,0.116012578744,0.116010467315,0.116008355999,0.116006244797,0.11600413371,0.116002022736,0.115999911876,0.115997801131,0.115995690499,0.115993579981,0.115991469578,0.115989359288,0.115987249112,0.11598513905,0.115983029102,0.115980919268,0.115978809548,0.115976699942,0.115974590449,0.115972481071,0.115970371806,0.115968262655,0.115966153618,0.115964044695,0.115961935886,0.11595982719,0.115957718609,0.115955610141,0.115953501786,0.115951393546,0.115949285419,0.115947177407,0.115945069507,0.115942961722,0.11594085405,0.115938746492,0.115936639048,0.115934531717,0.1159324245,0.115930317397,0.115928210407,0.115926103531,0.115923996769,0.11592189012,0.115919783585,0.115917677164,0.115915570856,0.115913464661,0.115911358581,0.115909252614,0.11590714676,0.11590504102,0.115902935393,0.11590082988,0.115898724481,0.115896619195,0.115894514022,0.115892408963,0.115890304018,0.115888199186,0.115886094467,0.115883989862,0.11588188537,0.115879780992,0.115877676727,0.115875572575,0.115873468537,0.115871364612,0.115869260801,0.115867157103,0.115865053518,0.115862950047,0.115860846689,0.115858743445,0.115856640313,0.115854537295,0.115852434391,0.115850331599,0.115848228921,0.115846126356,0.115844023904,0.115841921566,0.115839819341,0.115837717229,0.11583561523,0.115833513345,0.115831411572,0.115829309913,0.115827208367,0.115825106934,0.115823005615,0.115820904408,0.115818803315,0.115816702335,0.115814601468,0.115812500714,0.115810400073,0.115808299545,0.11580619913,0.115804098828,0.11580199864,0.115799898564,0.115797798601,0.115795698752,0.115793599015,0.115791499392,0.115789399881,0.115787300484,0.115785201199,0.115783102027,0.115781002969,0.115778904023,0.11577680519,0.11577470647,0.115772607863,0.115770509369,0.115768410988,0.11576631272,0.115764214564,0.115762116522,0.115760018592,0.115757920775,0.115755823071,0.11575372548,0.115751628002,0.115749530636,0.115747433384,0.115745336244,0.115743239216,0.115741142302,0.1157390455,0.115736948811,0.115734852235,0.115732755772,0.115730659421,0.115728563183,0.115726467057,0.115724371045,0.115722275145,0.115720179357,0.115718083683,0.11571598812,0.115713892671,0.115711797334,0.11570970211,0.115707606998,0.115705511999,0.115703417113,0.115701322339,0.115699227678,0.115697133129,0.115695038693,0.11569294437,0.115690850158,0.11568875606,0.115686662074,0.1156845682,0.115682474439,0.11568038079,0.115678287254,0.115676193831,0.115674100519,0.115672007321,0.115669914234,0.11566782126,0.115665728399,0.115663635649,0.115661543013,0.115659450488,0.115657358076,0.115655265777,0.115653173589,0.115651081514,0.115648989551,0.115646897701,0.115644805963,0.115642714337,0.115640622824,0.115638531423,0.115636440134,0.115634348957,0.115632257893,0.11563016694,0.115628076101,0.115625985373,0.115623894757,0.115621804254,0.115619713863,0.115617623584,0.115615533417,0.115613443362,0.11561135342,0.11560926359,0.115607173871,0.115605084265,0.115602994771,0.115600905389,0.115598816119,0.115596726962,0.115594637916,0.115592548983,0.115590460161,0.115588371451,0.115586282854,0.115584194369,0.115582105995,0.115580017734,0.115577929584,0.115575841547,0.115573753621,0.115571665808,0.115569578106,0.115567490516,0.115565403039,0.115563315673,0.115561228419,0.115559141277,0.115557054247,0.115554967329,0.115552880522,0.115550793828,0.115548707245,0.115546620774,0.115544534415,0.115542448168,0.115540362033,0.115538276009,0.115536190098,0.115534104298,0.11553201861,0.115529933033,0.115527847569,0.115525762216,0.115523676974,0.115521591845,0.115519506827,0.115517421921,0.115515337127,0.115513252444,0.115511167873,0.115509083414,0.115506999066,0.11550491483,0.115502830706,0.115500746693,0.115498662792,0.115496579003,0.115494495325,0.115492411758,0.115490328303,0.11548824496,0.115486161729,0.115484078609,0.1154819956,0.115479912703,0.115477829917,0.115475747243,0.115473664681,0.11547158223,0.11546949989,0.115467417662,0.115465335546,0.115463253541,0.115461171647,0.115459089864,0.115457008194,0.115454926634,0.115452845186,0.115450763849,0.115448682624,0.11544660151,0.115444520508,0.115442439616,0.115440358837,0.115438278168,0.115436197611,0.115434117165,0.11543203683,0.115429956607,0.115427876495,0.115425796494,0.115423716605,0.115421636826,0.115419557159,0.115417477604,0.115415398159,0.115413318826,0.115411239604,0.115409160493,0.115407081493,0.115405002604,0.115402923827,0.115400845161,0.115398766606,0.115396688162,0.115394609829,0.115392531607,0.115390453496,0.115388375497,0.115386297609,0.115384219831,0.115382142165,0.11538006461,0.115377987165,0.115375909832,0.11537383261,0.115371755499,0.115369678499,0.11536760161,0.115365524832,0.115363448165,0.115361371609,0.115359295163,0.115357218829,0.115355142606,0.115353066493,0.115350990492,0.115348914601,0.115346838822,0.115344763153,0.115342687595,0.115340612148,0.115338536812,0.115336461587,0.115334386472,0.115332311469,0.115330236576,0.115328161794,0.115326087123,0.115324012562,0.115321938113,0.115319863774,0.115317789546,0.115315715429,0.115313641422,0.115311567526,0.115309493741,0.115307420067,0.115305346503,0.11530327305,0.115301199708,0.115299126477,0.115297053356,0.115294980345,0.115292907446,0.115290834657,0.115288761978,0.115286689411,0.115284616954,0.115282544607,0.115280472371,0.115278400246,0.115276328231,0.115274256327,0.115272184533,0.11527011285,0.115268041277,0.115265969815,0.115263898464,0.115261827223,0.115259756092,0.115257685072,0.115255614163,0.115253543363,0.115251472675,0.115249402097,0.115247331629,0.115245261271,0.115243191024,0.115241120888,0.115239050862,0.115236980946,0.115234911141,0.115232841446,0.115230771861,0.115228702387,0.115226633023,0.115224563769,0.115222494626,0.115220425593,0.11521835667,0.115216287858,0.115214219155,0.115212150563,0.115210082082,0.115208013711,0.115205945449,0.115203877299,0.115201809258,0.115199741327,0.115197673507,0.115195605797,0.115193538197,0.115191470707,0.115189403328,0.115187336059,0.115185268899,0.11518320185,0.115181134911,0.115179068082,0.115177001364,0.115174934755,0.115172868256,0.115170801868,0.11516873559,0.115166669421,0.115164603363,0.115162537415,0.115160471576,0.115158405848,0.11515634023,0.115154274722,0.115152209323,0.115150144035,0.115148078857,0.115146013789,0.11514394883,0.115141883982,0.115139819243,0.115137754615,0.115135690096,0.115133625687,0.115131561389,0.1151294972,0.115127433121,0.115125369151,0.115123305292,0.115121241543,0.115119177903,0.115117114373,0.115115050953,0.115112987643,0.115110924442,0.115108861352,0.115106798371,0.1151047355,0.115102672738,0.115100610087,0.115098547545,0.115096485113,0.115094422791,0.115092360578,0.115090298475,0.115088236482,0.115086174598,0.115084112824,0.11508205116,0.115079989605,0.11507792816,0.115075866825,0.115073805599,0.115071744483,0.115069683477,0.11506762258,0.115065561793,0.115063501115,0.115061440547,0.115059380088,0.115057319739,0.1150552595,0.11505319937,0.115051139349,0.115049079438,0.115047019637,0.115044959945,0.115042900363,0.11504084089,0.115038781526,0.115036722272,0.115034663127,0.115032604092,0.115030545166,0.11502848635,0.115026427643,0.115024369046,0.115022310557,0.115020252179,0.115018193909,0.115016135749,0.115014077698,0.115012019757,0.115009961925,0.115007904202,0.115005846589,0.115003789085,0.11500173169,0.114999674404,0.114997617228,0.114995560161,0.114993503203,0.114991446355,0.114989389616,0.114987332986,0.114985276465,0.114983220053,0.114981163751,0.114979107557,0.114977051473,0.114974995498,0.114972939633,0.114970883876,0.114968828229,0.11496677269,0.114964717261,0.114962661941,0.11496060673,0.114958551628,0.114956496635,0.114954441751,0.114952386977,0.114950332311,0.114948277754,0.114946223307,0.114944168968,0.114942114739,0.114940060618,0.114938006607,0.114935952704,0.114933898911,0.114931845226,0.11492979165,0.114927738184,0.114925684826,0.114923631577,0.114921578437,0.114919525406,0.114917472484,0.114915419671,0.114913366967,0.114911314371,0.114909261884,0.114907209507,0.114905157238,0.114903105078,0.114901053026,0.114899001084,0.11489694925,0.114894897526,0.11489284591,0.114890794402,0.114888743004,0.114886691714,0.114884640533,0.114882589461,0.114880538497,0.114878487643,0.114876436896,0.114874386259,0.11487233573,0.11487028531,0.114868234999,0.114866184796,0.114864134702,0.114862084717,0.11486003484,0.114857985072,0.114855935413,0.114853885862,0.114851836419,0.114849787086,0.114847737861,0.114845688744,0.114843639736,0.114841590836,0.114839542046,0.114837493363,0.114835444789,0.114833396324,0.114831347967,0.114829299719,0.114827251579,0.114825203547,0.114823155625,0.11482110781,0.114819060104,0.114817012506,0.114814965017,0.114812917636,0.114810870364,0.1148088232,0.114806776145,0.114804729197,0.114802682359,0.114800635628,0.114798589006,0.114796542492,0.114794496087,0.11479244979,0.114790403601,0.114788357521,0.114786311548,0.114784265684,0.114782219929,0.114780174281,0.114778128742,0.114776083311,0.114774037989,0.114771992774,0.114769947668,0.11476790267,0.114765857781,0.114763812999,0.114761768326,0.11475972376,0.114757679303,0.114755634954,0.114753590714,0.114751546581,0.114749502557,0.11474745864,0.114745414832,0.114743371132,0.11474132754,0.114739284056,0.11473724068,0.114735197412,0.114733154252,0.1147311112,0.114729068256,0.114727025421,0.114724982693,0.114722940073,0.114720897561,0.114718855158,0.114716812862,0.114714770674,0.114712728594,0.114710686622,0.114708644758,0.114706603002,0.114704561354,0.114702519814,0.114700478382,0.114698437057,0.114696395841,0.114694354732,0.114692313731,0.114690272838,0.114688232053,0.114686191376,0.114684150806,0.114682110345,0.114680069991,0.114678029745,0.114675989607,0.114673949576,0.114671909653,0.114669869838,0.114667830131,0.114665790532,0.11466375104,0.114661711656,0.11465967238,0.114657633211,0.11465559415,0.114653555197,0.114651516351,0.114649477614,0.114647438983,0.114645400461,0.114643362046,0.114641323739,0.114639285539,0.114637247447,0.114635209462,0.114633171585,0.114631133816,0.114629096154,0.1146270586,0.114625021154,0.114622983815,0.114620946583,0.114618909459,0.114616872443,0.114614835534,0.114612798732,0.114610762038,0.114608725451,0.114606688972,0.114604652601,0.114602616337,0.11460058018,0.114598544131,0.114596508189,0.114594472354,0.114592436627,0.114590401008,0.114588365495,0.114586330091,0.114584294793,0.114582259603,0.11458022452,0.114578189545,0.114576154676,0.114574119916,0.114572085262,0.114570050716,0.114568016277,0.114565981945,0.114563947721,0.114561913604,0.114559879594,0.114557845692,0.114555811896,0.114553778208,0.114551744627,0.114549711153,0.114547677787,0.114545644528,0.114543611375,0.11454157833,0.114539545393,0.114537512562,0.114535479839,0.114533447222,0.114531414713,0.114529382311,0.114527350016,0.114525317828,0.114523285747,0.114521253773,0.114519221907,0.114517190147,0.114515158494,0.114513126949,0.11451109551,0.114509064179,0.114507032954,0.114505001837,0.114502970826,0.114500939923,0.114498909126,0.114496878437,0.114494847854,0.114492817379,0.11449078701,0.114488756748,0.114486726593,0.114484696545,0.114482666604,0.11448063677,0.114478607043,0.114476577422,0.114474547909,0.114472518502,0.114470489202,0.114468460009,0.114466430923,0.114464401944,0.114462373071,0.114460344306,0.114458315647,0.114456287095,0.114454258649,0.114452230311,0.114450202079,0.114448173954,0.114446145935,0.114444118024,0.114442090219,0.11444006252,0.114438034929,0.114436007444,0.114433980066,0.114431952794,0.11442992563,0.114427898571,0.11442587162,0.114423844775,0.114421818037,0.114419791405,0.11441776488,0.114415738462,0.11441371215,0.114411685944,0.114409659846,0.114407633854,0.114405607968,0.114403582189,0.114401556516,0.11439953095,0.114397505491,0.114395480138,0.114393454891,0.114391429751,0.114389404718,0.114387379791,0.11438535497,0.114383330256,0.114381305648,0.114379281147,0.114377256752,0.114375232464,0.114373208282,0.114371184206,0.114369160237,0.114367136374,0.114365112617,0.114363088967,0.114361065423,0.114359041986,0.114357018655,0.11435499543,0.114352972311,0.114350949299,0.114348926393,0.114346903594,0.1143448809,0.114342858313,0.114340835832,0.114338813458,0.114336791189,0.114334769027,0.114332746971,0.114330725022,0.114328703178,0.114326681441,0.11432465981,0.114322638285,0.114320616866,0.114318595553,0.114316574347,0.114314553246,0.114312532252,0.114310511364,0.114308490582,0.114306469906,0.114304449336,0.114302428873,0.114300408515,0.114298388264,0.114296368118,0.114294348079,0.114292328145,0.114290308318,0.114288288596,0.114286268981,0.114284249472,0.114282230068,0.114280210771,0.114278191579,0.114276172494,0.114274153514,0.114272134641,0.114270115873,0.114268097211,0.114266078656,0.114264060206,0.114262041862,0.114260023624,0.114258005491,0.114255987465,0.114253969545,0.11425195173,0.114249934021,0.114247916418,0.114245898921,0.11424388153,0.114241864244,0.114239847065,0.114237829991,0.114235813022,0.11423379616,0.114231779403,0.114229762753,0.114227746208,0.114225729768,0.114223713435,0.114221697207,0.114219681084,0.114217665068,0.114215649157,0.114213633352,0.114211617652,0.114209602059,0.11420758657,0.114205571188,0.114203555911,0.11420154074,0.114199525674,0.114197510714,0.11419549586,0.114193481111,0.114191466467,0.11418945193,0.114187437498,0.114185423171,0.11418340895,0.114181394834,0.114179380824,0.11417736692,0.114175353121,0.114173339428,0.11417132584,0.114169312357,0.11416729898,0.114165285708,0.114163272542,0.114161259482,0.114159246526,0.114157233677,0.114155220932,0.114153208293,0.11415119576,0.114149183331,0.114147171009,0.114145158791,0.114143146679,0.114141134672,0.114139122771,0.114137110975,0.114135099284,0.114133087699,0.114131076219,0.114129064844,0.114127053574,0.11412504241,0.114123031351,0.114121020397,0.114119009549,0.114116998806,0.114114988168,0.114112977635,0.114110967207,0.114108956885,0.114106946668,0.114104936556,0.114102926549,0.114100916648,0.114098906851,0.11409689716,0.114094887574,0.114092878093,0.114090868717,0.114088859446,0.11408685028,0.11408484122,0.114082832264,0.114080823414,0.114078814669,0.114076806029,0.114074797493,0.114072789063,0.114070780738,0.114068772518,0.114066764403,0.114064756393,0.114062748488,0.114060740688,0.114058732993,0.114056725402,0.114054717917,0.114052710537,0.114050703262,0.114048696092,0.114046689026,0.114044682066,0.11404267521,0.114040668459,0.114038661814,0.114036655273,0.114034648837,0.114032642506,0.114030636279,0.114028630158,0.114026624141,0.114024618229,0.114022612422,0.11402060672,0.114018601123,0.11401659563,0.114014590243,0.11401258496,0.114010579781,0.114008574708,0.114006569739,0.114004564875,0.114002560116,0.114000555461,0.113998550911,0.113996546466,0.113994542126,0.11399253789,0.113990533759,0.113988529732,0.113986525811,0.113984521993,0.113982518281,0.113980514673,0.11397851117,0.113976507771,0.113974504477,0.113972501288,0.113970498203,0.113968495223,0.113966492347,0.113964489576,0.113962486909,0.113960484347,0.11395848189,0.113956479537,0.113954477288,0.113952475144,0.113950473105,0.11394847117,0.113946469339,0.113944467613,0.113942465991,0.113940464474,0.113938463061,0.113936461753,0.113934460549,0.11393245945,0.113930458455,0.113928457564,0.113926456778,0.113924456096,0.113922455519,0.113920455045,0.113918454677,0.113916454412,0.113914454252,0.113912454196,0.113910454245,0.113908454398,0.113906454655,0.113904455016,0.113902455482,0.113900456052,0.113898456726,0.113896457504,0.113894458387,0.113892459374,0.113890460465,0.113888461661,0.11388646296,0.113884464364,0.113882465872,0.113880467484,0.113878469201,0.113876471021,0.113874472946,0.113872474975,0.113870477108,0.113868479345,0.113866481686,0.113864484131,0.113862486681,0.113860489334,0.113858492092,0.113856494953,0.113854497919,0.113852500989,0.113850504163,0.11384850744,0.113846510822,0.113844514308,0.113842517898,0.113840521592,0.11383852539,0.113836529292,0.113834533298,0.113832537408,0.113830541621,0.113828545939,0.113826550361,0.113824554887,0.113822559516,0.11382056425,0.113818569087,0.113816574028,0.113814579073,0.113812584222,0.113810589475,0.113808594832,0.113806600293,0.113804605857,0.113802611525,0.113800617297,0.113798623173,0.113796629153,0.113794635237,0.113792641424,0.113790647715,0.11378865411,0.113786660609,0.113784667211,0.113782673917,0.113780680727,0.113778687641,0.113776694658,0.113774701779,0.113772709004,0.113770716332,0.113768723764,0.1137667313,0.113764738939,0.113762746682,0.113760754529,0.113758762479,0.113756770533,0.113754778691,0.113752786952,0.113750795317,0.113748803786,0.113746812358,0.113744821033,0.113742829812,0.113740838695,0.113738847681,0.113736856771,0.113734865964,0.113732875261,0.113730884662,0.113728894166,0.113726903773,0.113724913484,0.113722923298,0.113720933216,0.113718943237,0.113716953362,0.11371496359,0.113712973922,0.113710984357,0.113708994895,0.113707005537,0.113705016282,0.113703027131,0.113701038083,0.113699049138,0.113697060297,0.113695071559,0.113693082925,0.113691094393,0.113689105966,0.113687117641,0.11368512942,0.113683141302,0.113681153287,0.113679165376,0.113677177568,0.113675189863,0.113673202261,0.113671214763,0.113669227368,0.113667240076,0.113665252887,0.113663265802,0.11366127882,0.113659291941,0.113657305165,0.113655318492,0.113653331923,0.113651345456,0.113649359093,0.113647372833,0.113645386676,0.113643400622,0.113641414672,0.113639428824,0.11363744308,0.113635457438,0.1136334719,0.113631486465,0.113629501133,0.113627515904,0.113625530778,0.113623545755,0.113621560835,0.113619576018,0.113617591304,0.113615606693,0.113613622185,0.11361163778,0.113609653478,0.113607669279,0.113605685183,0.11360370119,0.113601717299,0.113599733512,0.113597749828,0.113595766246,0.113593782768,0.113591799392,0.11358981612,0.11358783295,0.113585849883,0.113583866919,0.113581884057,0.113579901299,0.113577918643,0.113575936091,0.113573953641,0.113571971294,0.113569989049,0.113568006908,0.113566024869,0.113564042933,0.1135620611,0.113560079369,0.113558097741,0.113556116216,0.113554134794,0.113552153475,0.113550172258,0.113548191144,0.113546210132,0.113544229224,0.113542248417,0.113540267714,0.113538287113,0.113536306615,0.11353432622,0.113532345927,0.113530365737,0.113528385649,0.113526405664,0.113524425782,0.113522446002,0.113520466325,0.11351848675,0.113516507278,0.113514527909,0.113512548642,0.113510569477,0.113508590415,0.113506611456,0.113504632599,0.113502653845,0.113500675193,0.113498696644,0.113496718197,0.113494739852,0.11349276161,0.113490783471,0.113488805433,0.113486827499,0.113484849667,0.113482871937,0.113480894309,0.113478916784,0.113476939362,0.113474962041,0.113472984823,0.113471007708,0.113469030695,0.113467053784,0.113465076975,0.113463100269,0.113461123665,0.113459147164,0.113457170765,0.113455194468,0.113453218273,0.113451242181,0.113449266191,0.113447290303,0.113445314517,0.113443338834,0.113441363253,0.113439387774,0.113437412397,0.113435437122,0.11343346195,0.11343148688,0.113429511912,0.113427537046,0.113425562283,0.113423587621,0.113421613062,0.113419638605,0.11341766425,0.113415689997,0.113413715846,0.113411741797,0.11340976785,0.113407794006,0.113405820263,0.113403846623,0.113401873085,0.113399899648,0.113397926314,0.113395953082,0.113393979952,0.113392006924,0.113390033997,0.113388061173,0.113386088451,0.113384115831,0.113382143312,0.113380170896,0.113378198582,0.113376226369,0.113374254259,0.11337228225,0.113370310344,0.113368338539,0.113366366836,0.113364395235,0.113362423736,0.113360452339,0.113358481044,0.11335650985,0.113354538759,0.113352567769,0.113350596881,0.113348626095,0.113346655411,0.113344684828,0.113342714348,0.113340743969,0.113338773692,0.113336803516,0.113334833443,0.113332863471,0.113330893601,0.113328923833,0.113326954166,0.113324984601,0.113323015138,0.113321045777,0.113319076517,0.113317107359,0.113315138302,0.113313169348,0.113311200494,0.113309231743,0.113307263093,0.113305294545,0.113303326098,0.113301357754,0.11329938951,0.113297421368,0.113295453328,0.11329348539,0.113291517553,0.113289549817,0.113287582184,0.113285614651,0.11328364722,0.113281679891,0.113279712663,0.113277745537,0.113275778513,0.113273811589,0.113271844768,0.113269878047,0.113267911429,0.113265944911,0.113263978495,0.113262012181,0.113260045968,0.113258079856,0.113256113846,0.113254147937,0.11325218213,0.113250216424,0.11324825082,0.113246285316,0.113244319914,0.113242354614,0.113240389415,0.113238424317,0.113236459321,0.113234494425,0.113232529632,0.113230564939,0.113228600348,0.113226635858,0.113224671469,0.113222707182,0.113220742996,0.113218778911,0.113216814927,0.113214851045,0.113212887264,0.113210923584,0.113208960005,0.113206996528,0.113205033151,0.113203069876,0.113201106702,0.113199143629,0.113197180658,0.113195217787,0.113193255018,0.11319129235,0.113189329782,0.113187367317,0.113185404952,0.113183442688,0.113181480525,0.113179518464,0.113177556503,0.113175594644,0.113173632885,0.113171671228,0.113169709671,0.113167748216,0.113165786862,0.113163825609,0.113161864456,0.113159903405,0.113157942455,0.113155981606,0.113154020857,0.11315206021,0.113150099664,0.113148139218,0.113146178874,0.11314421863,0.113142258487,0.113140298446,0.113138338505,0.113136378665,0.113134418926,0.113132459287,0.11313049975,0.113128540314,0.113126580978,0.113124621743,0.113122662609,0.113120703576,0.113118744644,0.113116785812,0.113114827082,0.113112868452,0.113110909922,0.113108951494,0.113106993167,0.11310503494,0.113103076814,0.113101118788,0.113099160864,0.11309720304,0.113095245317,0.113093287694,0.113091330172,0.113089372751,0.113087415431,0.113085458211,0.113083501092,0.113081544074,0.113079587156,0.113077630339,0.113075673622,0.113073717006,0.113071760491,0.113069804076,0.113067847762,0.113065891549,0.113063935436,0.113061979423,0.113060023512,0.1130580677,0.11305611199,0.11305415638,0.11305220087,0.113050245461,0.113048290152,0.113046334944,0.113044379836,0.113042424829,0.113040469923,0.113038515117,0.113036560411,0.113034605806,0.113032651301,0.113030696896,0.113028742592,0.113026788389,0.113024834286,0.113022880283,0.11302092638,0.113018972579,0.113017018877,0.113015065276,0.113013111775,0.113011158374,0.113009205074,0.113007251874,0.113005298775,0.113003345775,0.113001392876,0.112999440078,0.11299748738,0.112995534781,0.112993582284,0.112991629886,0.112989677589,0.112987725392,0.112985773295,0.112983821299,0.112981869402,0.112979917606,0.11297796591,0.112976014314,0.112974062819,0.112972111424,0.112970160128,0.112968208933,0.112966257838,0.112964306844,0.112962355949,0.112960405155,0.11295845446,0.112956503866,0.112954553372,0.112952602978,0.112950652684,0.11294870249,0.112946752396,0.112944802402,0.112942852509,0.112940902715,0.112938953022,0.112937003428,0.112935053934,0.112933104541,0.112931155247,0.112929206054,0.11292725696,0.112925307966,0.112923359073,0.112921410279,0.112919461585,0.112917512992,0.112915564498,0.112913616104,0.11291166781,0.112909719616,0.112907771521,0.112905823527,0.112903875633,0.112901927838,0.112899980143,0.112898032549,0.112896085054,0.112894137658,0.112892190363,0.112890243168,0.112888296072,0.112886349076,0.11288440218,0.112882455384,0.112880508687,0.11287856209,0.112876615593,0.112874669196,0.112872722899,0.112870776701,0.112868830603,0.112866884605,0.112864938706,0.112862992907,0.112861047208,0.112859101609,0.112857156109,0.112855210709,0.112853265408,0.112851320208,0.112849375106,0.112847430105,0.112845485203,0.112843540401,0.112841595698,0.112839651095,0.112837706592,0.112835762188,0.112833817884,0.112831873679,0.112829929574,0.112827985568,0.112826041662,0.112824097856,0.112822154149,0.112820210541,0.112818267033,0.112816323625,0.112814380316,0.112812437106,0.112810493996,0.112808550986,0.112806608075,0.112804665263,0.112802722551,0.112800779938,0.112798837425,0.112796895011,0.112794952697,0.112793010482,0.112791068366,0.11278912635,0.112787184433,0.112785242615,0.112783300897,0.112781359279,0.112779417759,0.112777476339,0.112775535018,0.112773593797,0.112771652675,0.112769711652,0.112767770728,0.112765829904,0.112763889179,0.112761948553,0.112760008027,0.112758067599,0.112756127271,0.112754187043,0.112752246913,0.112750306883,0.112748366952,0.11274642712,0.112744487387,0.112742547754,0.11274060822,0.112738668784,0.112736729448,0.112734790212,0.112732851074,0.112730912035,0.112728973096,0.112727034256,0.112725095514,0.112723156872,0.112721218329,0.112719279885,0.112717341541,0.112715403295,0.112713465148,0.1127115271,0.112709589152,0.112707651302,0.112705713552,0.1127037759,0.112701838348,0.112699900894,0.11269796354,0.112696026284,0.112694089127,0.11269215207,0.112690215111,0.112688278252,0.112686341491,0.112684404829,0.112682468266,0.112680531802,0.112678595437,0.112676659171,0.112674723004,0.112672786935,0.112670850966,0.112668915095,0.112666979324,0.112665043651,0.112663108077,0.112661172601,0.112659237225,0.112657301948,0.112655366769,0.112653431689,0.112651496708,0.112649561825,0.112647627042,0.112645692357,0.112643757771,0.112641823283,0.112639888895,0.112637954605,0.112636020414,0.112634086322,0.112632152328,0.112630218433,0.112628284637,0.112626350939,0.11262441734,0.11262248384,0.112620550438,0.112618617135,0.112616683931,0.112614750825,0.112612817818,0.11261088491,0.1126089521,0.112607019389,0.112605086776,0.112603154262,0.112601221847,0.11259928953,0.112597357312,0.112595425192,0.112593493171,0.112591561248,0.112589629424,0.112587697699,0.112585766071,0.112583834543,0.112581903113,0.112579971781,0.112578040548,0.112576109413,0.112574178377,0.112572247439,0.1125703166,0.112568385859,0.112566455217,0.112564524673,0.112562594227,0.11256066388,0.112558733631,0.112556803481,0.112554873429,0.112552943475,0.11255101362,0.112549083863,0.112547154204,0.112545224644,0.112543295182,0.112541365819,0.112539436553,0.112537507386,0.112535578318,0.112533649347,0.112531720475,0.112529791701,0.112527863026,0.112525934449,0.112524005969,0.112522077589,0.112520149306,0.112518221122,0.112516293036,0.112514365048,0.112512437158,0.112510509366,0.112508581673,0.112506654078,0.112504726581,0.112502799182,0.112500871881,0.112498944678,0.112497017574,0.112495090568,0.112493163659,0.112491236849,0.112489310137,0.112487383523,0.112485457008,0.11248353059,0.11248160427,0.112479678049,0.112477751925,0.112475825899,0.112473899972,0.112471974142,0.112470048411,0.112468122777,0.112466197242,0.112464271805,0.112462346465,0.112460421224,0.11245849608,0.112456571034,0.112454646087,0.112452721237,0.112450796485,0.112448871832,0.112446947276,0.112445022818,0.112443098458,0.112441174196,0.112439250031,0.112437325965,0.112435401996,0.112433478126,0.112431554353,0.112429630678,0.112427707101,0.112425783621,0.11242386024,0.112421936956,0.11242001377,0.112418090682,0.112416167692,0.112414244799,0.112412322004,0.112410399307,0.112408476708,0.112406554207,0.112404631803,0.112402709497,0.112400787289,0.112398865178,0.112396943165,0.11239502125,0.112393099433,0.112391177713,0.112389256091,0.112387334566,0.112385413139,0.11238349181,0.112381570579,0.112379649445,0.112377728408,0.11237580747,0.112373886629,0.112371965885,0.112370045239,0.112368124691,0.11236620424,0.112364283887,0.112362363632,0.112360443474,0.112358523413,0.11235660345,0.112354683585,0.112352763817,0.112350844147,0.112348924574,0.112347005098,0.11234508572,0.11234316644,0.112341247257,0.112339328172,0.112337409183,0.112335490293,0.1123335715,0.112331652804,0.112329734206,0.112327815705,0.112325897301,0.112323978995,0.112322060786,0.112320142675,0.112318224661,0.112316306744,0.112314388925,0.112312471203,0.112310553579,0.112308636052,0.112306718622,0.112304801289,0.112302884054,0.112300966916,0.112299049875,0.112297132932,0.112295216085,0.112293299337,0.112291382685,0.112289466131,0.112287549673,0.112285633313,0.112283717051,0.112281800885,0.112279884817,0.112277968846,0.112276052972,0.112274137195,0.112272221516,0.112270305934,0.112268390448,0.11226647506,0.11226455977,0.112262644576,0.112260729479,0.11225881448,0.112256899577,0.112254984772,0.112253070064,0.112251155453,0.112249240939,0.112247326522,0.112245412202,0.112243497979,0.112241583853,0.112239669824,0.112237755893,0.112235842058,0.11223392832,0.11223201468,0.112230101136,0.112228187689,0.11222627434,0.112224361087,0.112222447931,0.112220534872,0.11221862191,0.112216709045,0.112214796277,0.112212883606,0.112210971032,0.112209058555,0.112207146175,0.112205233891,0.112203321705,0.112201409615,0.112199497622,0.112197585726,0.112195673927,0.112193762225,0.112191850619,0.112189939111,0.112188027699,0.112186116384,0.112184205166,0.112182294044,0.11218038302,0.112178472092,0.112176561261,0.112174650527,0.112172739889,0.112170829348,0.112168918904,0.112167008557,0.112165098306,0.112163188153,0.112161278095,0.112159368135,0.112157458271,0.112155548504,0.112153638834,0.11215172926,0.112149819783,0.112147910402,0.112146001119,0.112144091931,0.112142182841,0.112140273847,0.11213836495,0.112136456149,0.112134547445,0.112132638837,0.112130730326,0.112128821912,0.112126913594,0.112125005373,0.112123097248,0.11212118922,0.112119281288,0.112117373453,0.112115465714,0.112113558072,0.112111650527,0.112109743078,0.112107835725,0.112105928469,0.112104021309,0.112102114246,0.112100207279,0.112098300409,0.112096393635,0.112094486957,0.112092580376,0.112090673891,0.112088767503,0.112086861211,0.112084955016,0.112083048917,0.112081142914,0.112079237007,0.112077331197,0.112075425484,0.112073519866,0.112071614345,0.112069708921,0.112067803592,0.11206589836,0.112063993224,0.112062088185,0.112060183241,0.112058278394,0.112056373644,0.112054468989,0.112052564431,0.112050659969,0.112048755603,0.112046851334,0.112044947161,0.112043043084,0.112041139103,0.112039235218,0.11203733143,0.112035427737,0.112033524141,0.112031620641,0.112029717237,0.112027813929,0.112025910718,0.112024007602,0.112022104583,0.11202020166,0.112018298833,0.112016396102,0.112014493467,0.112012590928,0.112010688485,0.112008786139,0.112006883888,0.112004981733,0.112003079675,0.112001177712,0.111999275846,0.111997374075,0.111995472401,0.111993570822,0.11199166934,0.111989767953,0.111987866663,0.111985965468,0.11198406437,0.111982163367,0.11198026246,0.11197836165,0.111976460935,0.111974560316,0.111972659793,0.111970759366,0.111968859035,0.111966958799,0.11196505866,0.111963158616,0.111961258669,0.111959358817,0.111957459061,0.111955559401,0.111953659836,0.111951760368,0.111949860995,0.111947961718,0.111946062537,0.111944163452,0.111942264462,0.111940365569,0.111938466771,0.111936568069,0.111934669462,0.111932770951,0.111930872537,0.111928974217,0.111927075994,0.111925177866,0.111923279834,0.111921381898,0.111919484057,0.111917586312,0.111915688663,0.111913791109,0.111911893651,0.111909996289,0.111908099022,0.111906201851,0.111904304775,0.111902407795,0.111900510911,0.111898614123,0.111896717429,0.111894820832,0.11189292433,0.111891027924,0.111889131613,0.111887235398,0.111885339278,0.111883443254,0.111881547326,0.111879651493,0.111877755755,0.111875860113,0.111873964566,0.111872069115,0.11187017376,0.1118682785,0.111866383335,0.111864488266,0.111862593292,0.111860698414,0.111858803631,0.111856908944,0.111855014352,0.111853119855,0.111851225454,0.111849331148,0.111847436938,0.111845542823,0.111843648803,0.111841754879,0.11183986105,0.111837967317,0.111836073678,0.111834180136,0.111832286688,0.111830393336,0.111828500079,0.111826606917,0.111824713851,0.11182282088,0.111820928004,0.111819035224,0.111817142538,0.111815249948,0.111813357454,0.111811465054,0.11180957275,0.111807680541,0.111805788427,0.111803896408,0.111802004485,0.111800112656,0.111798220923,0.111796329286,0.111794437743,0.111792546295,0.111790654943,0.111788763686,0.111786872523,0.111784981456,0.111783090484,0.111781199608,0.111779308826,0.111777418139,0.111775527548,0.111773637052,0.11177174665,0.111769856344,0.111767966133,0.111766076017,0.111764185995,0.111762296069,0.111760406238,0.111758516502,0.111756626861,0.111754737315,0.111752847864,0.111750958508,0.111749069247,0.111747180081,0.11174529101,0.111743402033,0.111741513152,0.111739624366,0.111737735674,0.111735847078,0.111733958576,0.11173207017,0.111730181858,0.111728293641,0.111726405519,0.111724517492,0.11172262956,0.111720741722,0.11171885398,0.111716966332,0.111715078779,0.111713191321,0.111711303958,0.11170941669,0.111707529516,0.111705642437,0.111703755453,0.111701868564,0.11169998177,0.11169809507,0.111696208465,0.111694321955,0.11169243554,0.111690549219,0.111688662993,0.111686776862,0.111684890825,0.111683004883,0.111681119036,0.111679233284,0.111677347626,0.111675462063,0.111673576594,0.111671691221,0.111669805941,0.111667920757,0.111666035667,0.111664150672,0.111662265771,0.111660380965,0.111658496254,0.111656611637,0.111654727115,0.111652842687,0.111650958354,0.111649074115,0.111647189971,0.111645305922,0.111643421967,0.111641538107,0.111639654341,0.11163777067,0.111635887093,0.111634003611,0.111632120223,0.111630236929,0.11162835373,0.111626470626,0.111624587616,0.111622704701,0.11162082188,0.111618939153,0.111617056521,0.111615173983,0.11161329154,0.111611409191,0.111609526936,0.111607644776,0.11160576271,0.111603880739,0.111601998862,0.111600117079,0.111598235391,0.111596353797,0.111594472297,0.111592590892,0.111590709581,0.111588828364,0.111586947242,0.111585066214,0.11158318528,0.111581304441,0.111579423695,0.111577543044,0.111575662488,0.111573782025,0.111571901657,0.111570021383,0.111568141203,0.111566261118,0.111564381126,0.111562501229,0.111560621426,0.111558741718,0.111556862103,0.111554982583,0.111553103156,0.111551223824,0.111549344586,0.111547465443,0.111545586393,0.111543707437,0.111541828576,0.111539949809,0.111538071135,0.111536192556,0.111534314071,0.11153243568,0.111530557384,0.111528679181,0.111526801072,0.111524923057,0.111523045137,0.11152116731,0.111519289577,0.111517411939,0.111515534394,0.111513656944,0.111511779587,0.111509902324,0.111508025156,0.111506148081,0.1115042711,0.111502394214,0.111500517421,0.111498640722,0.111496764117,0.111494887606,0.111493011189,0.111491134866,0.111489258637,0.111487382501,0.11148550646,0.111483630512,0.111481754658,0.111479878898,0.111478003232,0.11147612766,0.111474252182,0.111472376797,0.111470501507,0.11146862631,0.111466751207,0.111464876197,0.111463001282,0.11146112646,0.111459251732,0.111457377098,0.111455502558,0.111453628111,0.111451753758,0.111449879499,0.111448005333,0.111446131262,0.111444257284,0.111442383399,0.111440509609,0.111438635912,0.111436762309,0.111434888799,0.111433015383,0.111431142061,0.111429268832,0.111427395698,0.111425522656,0.111423649709,0.111421776855,0.111419904094,0.111418031427,0.111416158854,0.111414286374,0.111412413988,0.111410541696,0.111408669497,0.111406797392,0.11140492538,0.111403053462,0.111401181637,0.111399309906,0.111397438268,0.111395566724,0.111393695273,0.111391823916,0.111389952653,0.111388081482,0.111386210406,0.111384339423,0.111382468533,0.111380597737,0.111378727034,0.111376856424,0.111374985908,0.111373115486,0.111371245157,0.111369374921,0.111367504778,0.111365634729,0.111363764774,0.111361894912,0.111360025143,0.111358155467,0.111356285885,0.111354416396,0.111352547001,0.111350677699,0.11134880849,0.111346939375,0.111345070352,0.111343201424,0.111341332588,0.111339463846,0.111337595197,0.111335726641,0.111333858178,0.111331989809,0.111330121533,0.11132825335,0.111326385261,0.111324517265,0.111322649361,0.111320781552,0.111318913835,0.111317046211,0.111315178681,0.111313311244,0.1113114439,0.111309576649,0.111307709491,0.111305842427,0.111303975456,0.111302108577,0.111300241792,0.1112983751,0.111296508501,0.111294641995,0.111292775583,0.111290909263,0.111289043037,0.111287176903,0.111285310863,0.111283444915,0.111281579061,0.1112797133,0.111277847631,0.111275982056,0.111274116574,0.111272251185,0.111270385888,0.111268520685,0.111266655575,0.111264790558,0.111262925633,0.111261060802,0.111259196064,0.111257331418,0.111255466866,0.111253602406,0.11125173804,0.111249873766,0.111248009585,0.111246145497,0.111244281502,0.1112424176,0.111240553791,0.111238690074,0.111236826451,0.11123496292,0.111233099482,0.111231236137,0.111229372885,0.111227509726,0.111225646659,0.111223783686,0.111221920805,0.111220058017,0.111218195321,0.111216332719,0.111214470209,0.111212607792,0.111210745468,0.111208883236,0.111207021097,0.111205159051,0.111203297098,0.111201435238,0.11119957347,0.111197711794,0.111195850212,0.111193988722,0.111192127325,0.111190266021,0.111188404809,0.11118654369,0.111184682663,0.111182821729,0.111180960888,0.11117910014,0.111177239484,0.11117537892,0.11117351845,0.111171658071,0.111169797786,0.111167937593,0.111166077493,0.111164217485,0.111162357569,0.111160497747,0.111158638016,0.111156778379,0.111154918834,0.111153059381,0.111151200021,0.111149340753,0.111147481578,0.111145622496,0.111143763506,0.111141904608,0.111140045803,0.11113818709,0.11113632847,0.111134469942,0.111132611507,0.111130753164,0.111128894913,0.111127036755,0.111125178689,0.111123320716,0.111121462835,0.111119605046,0.11111774735,0.111115889746,0.111114032235,0.111112174816,0.111110317489,0.111108460255,0.111106603113,0.111104746063,0.111102889106,0.11110103224,0.111099175468,0.111097318787,0.111095462199,0.111093605703,0.111091749299,0.111089892988,0.111088036768,0.111086180642,0.111084324607,0.111082468664,0.111080612814,0.111078757056,0.11107690139,0.111075045817,0.111073190335,0.111071334946,0.111069479649,0.111067624444,0.111065769331,0.111063914311,0.111062059382,0.111060204546,0.111058349802,0.11105649515,0.11105464059,0.111052786122,0.111050931747,0.111049077463,0.111047223272,0.111045369172,0.111043515165,0.11104166125,0.111039807426,0.111037953695,0.111036100056,0.111034246509,0.111032393054,0.111030539691,0.11102868642,0.111026833241,0.111024980154,0.111023127159,0.111021274256,0.111019421445,0.111017568726,0.111015716098,0.111013863563,0.11101201112,0.111010158769,0.111008306509,0.111006454342,0.111004602266,0.111002750283,0.111000898391,0.110999046591,0.110997194883,0.110995343267,0.110993491743,0.110991640311,0.11098978897,0.110987937721,0.110986086565,0.1109842355,0.110982384527,0.110980533645,0.110978682856,0.110976832158,0.110974981552,0.110973131038,0.110971280616,0.110969430285,0.110967580046,0.110965729899,0.110963879844,0.11096202988,0.110960180009,0.110958330228,0.11095648054,0.110954630943,0.110952781439,0.110950932025,0.110949082704,0.110947233474,0.110945384336,0.110943535289,0.110941686334,0.110939837471,0.110937988699,0.110936140019,0.110934291431,0.110932442934,0.110930594529,0.110928746216,0.110926897994,0.110925049864,0.110923201825,0.110921353878,0.110919506022,0.110917658258,0.110915810586,0.110913963005,0.110912115516,0.110910268118,0.110908420812,0.110906573597,0.110904726474,0.110902879442,0.110901032502,0.110899185653,0.110897338896,0.11089549223,0.110893645656,0.110891799173,0.110889952781,0.110888106481,0.110886260273,0.110884414156,0.11088256813,0.110880722196,0.110878876353,0.110877030601,0.110875184941,0.110873339372,0.110871493895,0.110869648509,0.110867803214,0.110865958011,0.110864112899,0.110862267879,0.11086042295,0.110858578112,0.110856733365,0.11085488871,0.110853044146,0.110851199673,0.110849355292,0.110847511002,0.110845666803,0.110843822695,0.110841978679,0.110840134754,0.11083829092,0.110836447177,0.110834603526,0.110832759966,0.110830916497,0.110829073119,0.110827229832,0.110825386637,0.110823543533,0.11082170052,0.110819857598,0.110818014767,0.110816172028,0.11081432938,0.110812486822,0.110810644356,0.110808801981,0.110806959697,0.110805117505,0.110803275403,0.110801433392,0.110799591473,0.110797749644,0.110795907907,0.110794066261,0.110792224706,0.110790383242,0.110788541868,0.110786700586,0.110784859395,0.110783018295,0.110781177286,0.110779336368,0.110777495541,0.110775654805,0.11077381416,0.110771973606,0.110770133143,0.110768292771,0.110766452489,0.110764612299,0.1107627722,0.110760932191,0.110759092274,0.110757252447,0.110755412712,0.110753573067,0.110751733513,0.11074989405,0.110748054678,0.110746215397,0.110744376207,0.110742537107,0.110740698098,0.110738859181,0.110737020354,0.110735181618,0.110733342972,0.110731504418,0.110729665954,0.110727827581,0.110725989299,0.110724151108,0.110722313007,0.110720474997,0.110718637078,0.11071679925,0.110714961512,0.110713123865,0.110711286309,0.110709448844,0.110707611469,0.110705774185,0.110703936992,0.11070209989,0.110700262878,0.110698425957,0.110696589126,0.110694752386,0.110692915737,0.110691079179,0.110689242711,0.110687406333,0.110685570047,0.110683733851,0.110681897745,0.110680061731,0.110678225806,0.110676389973,0.11067455423,0.110672718577,0.110670883015,0.110669047544,0.110667212163,0.110665376873,0.110663541673,0.110661706564,0.110659871546,0.110658036618,0.11065620178,0.110654367033,0.110652532376,0.11065069781,0.110648863334,0.110647028949,0.110645194655,0.11064336045,0.110641526337,0.110639692313,0.11063785838,0.110636024538,0.110634190786,0.110632357124,0.110630523553,0.110628690072,0.110626856681,0.110625023381,0.110623190172,0.110621357052,0.110619524023,0.110617691085,0.110615858236,0.110614025479,0.110612192811,0.110610360234,0.110608527747,0.11060669535,0.110604863044,0.110603030828,0.110601198702,0.110599366666,0.110597534721,0.110595702866,0.110593871102,0.110592039427,0.110590207843,0.110588376349,0.110586544945,0.110584713632,0.110582882409,0.110581051276,0.110579220233,0.11057738928,0.110575558418,0.110573727645,0.110571896963,0.110570066371,0.110568235869,0.110566405458,0.110564575136,0.110562744905,0.110560914763,0.110559084712,0.110557254751,0.11055542488,0.110553595099,0.110551765409,0.110549935808,0.110548106298,0.110546276877,0.110544447547,0.110542618306,0.110540789156,0.110538960095,0.110537131125,0.110535302245,0.110533473455,0.110531644755,0.110529816144,0.110527987624,0.110526159194,0.110524330854,0.110522502603,0.110520674443,0.110518846373,0.110517018392,0.110515190502,0.110513362701,0.110511534991,0.11050970737,0.110507879839,0.110506052398,0.110504225047,0.110502397786,0.110500570615,0.110498743534,0.110496916542,0.110495089641,0.110493262829,0.110491436107,0.110489609475,0.110487782933,0.110485956481,0.110484130118,0.110482303846,0.110480477663,0.11047865157,0.110476825566,0.110474999653,0.110473173829,0.110471348095,0.110469522451,0.110467696896,0.110465871432,0.110464046057,0.110462220771,0.110460395576,0.11045857047,0.110456745454,0.110454920528,0.110453095691,0.110451270944,0.110449446287,0.110447621719,0.110445797241,0.110443972853,0.110442148554,0.110440324345,0.110438500226,0.110436676196,0.110434852256,0.110433028405,0.110431204644,0.110429380973,0.110427557391,0.110425733899,0.110423910497,0.110422087184,0.11042026396,0.110418440827,0.110416617782,0.110414794828,0.110412971962,0.110411149187,0.110409326501,0.110407503904,0.110405681397,0.110403858979,0.110402036651,0.110400214412,0.110398392263,0.110396570204,0.110394748233,0.110392926353,0.110391104561,0.110389282859,0.110387461247,0.110385639724,0.11038381829,0.110381996946,0.110380175691,0.110378354526,0.11037653345,0.110374712463,0.110372891566,0.110371070758,0.11036925004,0.110367429411,0.110365608871,0.11036378842,0.110361968059,0.110360147788,0.110358327605,0.110356507512,0.110354687508,0.110352867594,0.110351047768,0.110349228032,0.110347408386,0.110345588828,0.11034376936,0.110341949981,0.110340130692,0.110338311491,0.11033649238,0.110334673358,0.110332854425,0.110331035582,0.110329216827,0.110327398162,0.110325579586,0.1103237611,0.110321942702,0.110320124394,0.110318306174,0.110316488044,0.110314670003,0.110312852052,0.110311034189,0.110309216415,0.110307398731,0.110305581136,0.110303763629,0.110301946212,0.110300128884,0.110298311645,0.110296494495,0.110294677434,0.110292860463,0.11029104358,0.110289226786,0.110287410082,0.110285593466,0.11028377694,0.110281960502,0.110280144153,0.110278327894,0.110276511723,0.110274695642,0.110272879649,0.110271063746,0.110269247931,0.110267432205,0.110265616569,0.110263801021,0.110261985562,0.110260170192,0.110258354911,0.110256539719,0.110254724616,0.110252909602,0.110251094676,0.11024927984,0.110247465092,0.110245650434,0.110243835864,0.110242021383,0.110240206991,0.110238392687,0.110236578473,0.110234764347,0.11023295031,0.110231136362,0.110229322503,0.110227508733,0.110225695051,0.110223881458,0.110222067954,0.110220254539,0.110218441213,0.110216627975,0.110214814826,0.110213001766,0.110211188794,0.110209375911,0.110207563117,0.110205750412,0.110203937795,0.110202125268,0.110200312828,0.110198500478,0.110196688216,0.110194876043,0.110193063958,0.110191251962,0.110189440055,0.110187628237,0.110185816507,0.110184004865,0.110182193313,0.110180381849,0.110178570473,0.110176759186,0.110174947988,0.110173136878,0.110171325857,0.110169514925,0.110167704081,0.110165893325,0.110164082658,0.11016227208,0.11016046159,0.110158651189,0.110156840876,0.110155030652,0.110153220517,0.110151410469,0.110149600511,0.11014779064,0.110145980859,0.110144171165,0.110142361561,0.110140552044,0.110138742616,0.110136933277,0.110135124026,0.110133314863,0.110131505789,0.110129696803,0.110127887906,0.110126079097,0.110124270377,0.110122461744,0.110120653201,0.110118844745,0.110117036378,0.1101152281,0.110113419909,0.110111611807,0.110109803794,0.110107995868,0.110106188031,0.110104380283,0.110102572622,0.11010076505,0.110098957566,0.110097150171,0.110095342864,0.110093535645,0.110091728514,0.110089921472,0.110088114518,0.110086307652,0.110084500874,0.110082694185,0.110080887583,0.11007908107,0.110077274646,0.110075468309,0.110073662061,0.1100718559,0.110070049828,0.110068243844,0.110066437949,0.110064632141,0.110062826422,0.110061020791,0.110059215247,0.110057409792,0.110055604426,0.110053799147,0.110051993956,0.110050188854,0.110048383839,0.110046578913,0.110044774075,0.110042969325,0.110041164662,0.110039360088,0.110037555602,0.110035751204,0.110033946894,0.110032142673,0.110030338539,0.110028534493,0.110026730535,0.110024926665,0.110023122883,0.11002131919,0.110019515584,0.110017712066,0.110015908636,0.110014105294,0.11001230204,0.110010498874,0.110008695796,0.110006892806,0.110005089903,0.110003287089,0.110001484362,0.109999681724,0.109997879173,0.109996076711,0.109994274336,0.109992472049,0.10999066985,0.109988867738,0.109987065715,0.109985263779,0.109983461932,0.109981660172,0.1099798585,0.109978056916,0.109976255419,0.10997445401,0.10997265269,0.109970851457,0.109969050311,0.109967249254,0.109965448284,0.109963647402,0.109961846608,0.109960045902,0.109958245283,0.109956444752,0.109954644309,0.109952843953,0.109951043685,0.109949243505,0.109947443413,0.109945643408,0.109943843491,0.109942043662,0.10994024392,0.109938444266,0.1099366447,0.109934845221,0.10993304583,0.109931246527,0.109929447311,0.109927648183,0.109925849142,0.109924050189,0.109922251324,0.109920452546,0.109918653856,0.109916855254,0.109915056739,0.109913258311,0.109911459971,0.109909661719,0.109907863554,0.109906065477,0.109904267487,0.109902469585,0.10990067177,0.109898874043,0.109897076403,0.109895278851,0.109893481387,0.109891684009,0.10988988672,0.109888089517,0.109886292403,0.109884495375,0.109882698435,0.109880901583,0.109879104818,0.10987730814,0.10987551155,0.109873715047,0.109871918632,0.109870122304,0.109868326063,0.10986652991,0.109864733844,0.109862937866,0.109861141975,0.109859346171,0.109857550455,0.109855754825,0.109853959284,0.109852163829,0.109850368462,0.109848573183,0.10984677799,0.109844982885,0.109843187867,0.109841392936,0.109839598093,0.109837803337,0.109836008668,0.109834214087,0.109832419593,0.109830625186,0.109828830866,0.109827036633,0.109825242488,0.10982344843,0.109821654459,0.109819860575,0.109818066779,0.109816273069,0.109814479447,0.109812685912,0.109810892464,0.109809099104,0.10980730583,0.109805512644,0.109803719545,0.109801926533,0.109800133608,0.10979834077,0.109796548019,0.109794755355,0.109792962779,0.109791170289,0.109789377887,0.109787585572,0.109785793343,0.109784001202,0.109782209148,0.109780417181,0.109778625301,0.109776833508,0.109775041802,0.109773250183,0.109771458651,0.109769667206,0.109767875848,0.109766084577,0.109764293394,0.109762502297,0.109760711287,0.109758920363,0.109757129527,0.109755338778,0.109753548116,0.109751757541,0.109749967052,0.109748176651,0.109746386337,0.109744596109,0.109742805968,0.109741015915,0.109739225948,0.109737436068,0.109735646274,0.109733856568,0.109732066949,0.109730277416,0.10972848797,0.109726698612,0.109724909339,0.109723120154,0.109721331056,0.109719542044,0.109717753119,0.109715964281,0.10971417553,0.109712386866,0.109710598288,0.109708809797,0.109707021393,0.109705233076,0.109703444845,0.109701656701,0.109699868644,0.109698080674,0.10969629279,0.109694504993,0.109692717283,0.109690929659,0.109689142122,0.109687354672,0.109685567308,0.109683780032,0.109681992841,0.109680205738,0.109678418721,0.109676631791,0.109674844947,0.10967305819,0.10967127152,0.109669484936,0.109667698439,0.109665912028,0.109664125704,0.109662339467,0.109660553316,0.109658767252,0.109656981274,0.109655195383,0.109653409579,0.109651623861,0.10964983823,0.109648052685,0.109646267226,0.109644481854,0.109642696569,0.10964091137,0.109639126258,0.109637341232,0.109635556293,0.10963377144,0.109631986674,0.109630201994,0.1096284174,0.109626632893,0.109624848472,0.109623064138,0.109621279891,0.109619495729,0.109617711654,0.109615927666,0.109614143764,0.109612359948,0.109610576219,0.109608792576,0.109607009019,0.109605225549,0.109603442165,0.109601658868,0.109599875657,0.109598092532,0.109596309494,0.109594526542,0.109592743676,0.109590960896,0.109589178203,0.109587395596,0.109585613076,0.109583830641,0.109582048293,0.109580266032,0.109578483856,0.109576701767,0.109574919764,0.109573137847,0.109571356017,0.109569574272,0.109567792614,0.109566011042,0.109564229557,0.109562448157,0.109560666844,0.109558885617,0.109557104476,0.109555323422,0.109553542453,0.109551761571,0.109549980774,0.109548200064,0.10954641944,0.109544638903,0.109542858451,0.109541078085,0.109539297806,0.109537517613,0.109535737506,0.109533957484,0.109532177549,0.1095303977,0.109528617937,0.109526838261,0.10952505867,0.109523279165,0.109521499746,0.109519720414,0.109517941167,0.109516162007,0.109514382932,0.109512603943,0.109510825041,0.109509046224,0.109507267494,0.109505488849,0.10950371029,0.109501931818,0.109500153431,0.10949837513,0.109496596915,0.109494818787,0.109493040744,0.109491262787,0.109489484916,0.10948770713,0.109485929431,0.109484151818,0.10948237429,0.109480596849,0.109478819493,0.109477042223,0.109475265039,0.109473487941,0.109471710929,0.109469934003,0.109468157162,0.109466380407,0.109464603738,0.109462827155,0.109461050658,0.109459274246,0.109457497921,0.109455721681,0.109453945527,0.109452169458,0.109450393476,0.109448617579,0.109446841768,0.109445066043,0.109443290403,0.109441514849,0.109439739381,0.109437963999,0.109436188702,0.109434413491,0.109432638366,0.109430863326,0.109429088373,0.109427313504,0.109425538722,0.109423764025,0.109421989414,0.109420214888,0.109418440448,0.109416666094,0.109414891826,0.109413117643,0.109411343545,0.109409569533,0.109407795607,0.109406021767,0.109404248012,0.109402474342,0.109400700759,0.10939892726,0.109397153848,0.109395380521,0.109393607279,0.109391834123,0.109390061053,0.109388288068,0.109386515168,0.109384742354,0.109382969626,0.109381196983,0.109379424426,0.109377651954,0.109375879567,0.109374107267,0.109372335051,0.109370562921,0.109368790876,0.109367018917,0.109365247044,0.109363475256,0.109361703553,0.109359931935,0.109358160403,0.109356388957,0.109354617596,0.10935284632,0.10935107513,0.109349304025,0.109347533005,0.109345762071,0.109343991222,0.109342220458,0.10934044978,0.109338679187,0.10933690868,0.109335138258,0.109333367921,0.109331597669,0.109329827503,0.109328057422,0.109326287426,0.109324517516,0.109322747691,0.109320977951,0.109319208297,0.109317438727,0.109315669243,0.109313899845,0.109312130531,0.109310361303,0.10930859216,0.109306823102,0.109305054129,0.109303285242,0.109301516439,0.109299747722,0.10929797909,0.109296210544,0.109294442082,0.109292673706,0.109290905415,0.109289137209,0.109287369088,0.109285601052,0.109283833101,0.109282065236,0.109280297455,0.10927852976,0.10927676215,0.109274994625,0.109273227185,0.10927145983,0.10926969256,0.109267925375,0.109266158276,0.109264391261,0.109262624331,0.109260857487,0.109259090727,0.109257324053,0.109255557463,0.109253790959,0.109252024539,0.109250258205,0.109248491955,0.109246725791,0.109244959711,0.109243193717,0.109241427807,0.109239661983,0.109237896243,0.109236130588,0.109234365019,0.109232599534,0.109230834134,0.109229068819,0.109227303589,0.109225538444,0.109223773384,0.109222008408,0.109220243518,0.109218478712,0.109216713992,0.109214949356,0.109213184805,0.109211420339,0.109209655957,0.109207891661,0.109206127449,0.109204363323,0.109202599281,0.109200835323,0.109199071451,0.109197307664,0.109195543961,0.109193780343,0.10919201681,0.109190253361,0.109188489998,0.109186726719,0.109184963525,0.109183200415,0.109181437391,0.109179674451,0.109177911596,0.109176148825,0.109174386139,0.109172623538,0.109170861022,0.10916909859,0.109167336243,0.109165573981,0.109163811803,0.10916204971,0.109160287702,0.109158525779,0.10915676394,0.109155002185,0.109153240515,0.10915147893,0.10914971743,0.109147956014,0.109146194683,0.109144433436,0.109142672274,0.109140911197,0.109139150204,0.109137389295,0.109135628471,0.109133867732,0.109132107078,0.109130346507,0.109128586022,0.109126825621,0.109125065304,0.109123305072,0.109121544925,0.109119784862,0.109118024883,0.109116264989,0.10911450518,0.109112745455,0.109110985814,0.109109226258,0.109107466786,0.109105707399,0.109103948096,0.109102188878,0.109100429744,0.109098670695,0.10909691173,0.109095152849,0.109093394053,0.109091635341,0.109089876714,0.10908811817,0.109086359712,0.109084601337,0.109082843048,0.109081084842,0.109079326721,0.109077568684,0.109075810731,0.109074052863,0.109072295079,0.10907053738,0.109068779764,0.109067022233,0.109065264787,0.109063507424,0.109061750146,0.109059992952,0.109058235843,0.109056478817,0.109054721876,0.10905296502,0.109051208247,0.109049451559,0.109047694955,0.109045938435,0.109044181999,0.109042425648,0.109040669381,0.109038913198,0.109037157099,0.109035401084,0.109033645154,0.109031889307,0.109030133545,0.109028377867,0.109026622273,0.109024866764,0.109023111338,0.109021355997,0.109019600739,0.109017845566,0.109016090477,0.109014335472,0.109012580551,0.109010825715,0.109009070962,0.109007316293,0.109005561709,0.109003807208,0.109002052792,0.109000298459,0.108998544211,0.108996790047,0.108995035967,0.10899328197,0.108991528058,0.10898977423,0.108988020486,0.108986266826,0.108984513249,0.108982759757,0.108981006349,0.108979253025,0.108977499784,0.108975746628,0.108973993555,0.108972240567,0.108970487662,0.108968734842,0.108966982105,0.108965229452,0.108963476884,0.108961724399,0.108959971998,0.108958219681,0.108956467447,0.108954715298,0.108952963232,0.108951211251,0.108949459353,0.108947707539,0.108945955809,0.108944204163,0.1089424526,0.108940701122,0.108938949727,0.108937198416,0.108935447189,0.108933696045,0.108931944986,0.10893019401,0.108928443118,0.10892669231,0.108924941585,0.108923190945,0.108921440388,0.108919689914,0.108917939525,0.108916189219,0.108914438997,0.108912688859,0.108910938804,0.108909188833,0.108907438946,0.108905689143,0.108903939423,0.108902189787,0.108900440234,0.108898690766,0.108896941381,0.108895192079,0.108893442861,0.108891693727,0.108889944677,0.10888819571,0.108886446826,0.108884698027,0.108882949311,0.108881200678,0.108879452129,0.108877703664,0.108875955282,0.108874206984,0.108872458769,0.108870710638,0.108868962591,0.108867214627,0.108865466747,0.10886371895,0.108861971237,0.108860223607,0.108858476061,0.108856728598,0.108854981219,0.108853233923,0.10885148671,0.108849739582,0.108847992536,0.108846245575,0.108844498696,0.108842751901,0.10884100519,0.108839258562,0.108837512017,0.108835765556,0.108834019178,0.108832272884,0.108830526673,0.108828780545,0.108827034501,0.10882528854,0.108823542663,0.108821796869,0.108820051158,0.108818305531,0.108816559987,0.108814814527,0.10881306915,0.108811323856,0.108809578645,0.108807833518,0.108806088474,0.108804343514,0.108802598636,0.108800853842,0.108799109132,0.108797364504,0.10879561996,0.108793875499,0.108792131122,0.108790386827,0.108788642616,0.108786898488,0.108785154444,0.108783410482,0.108781666604,0.108779922809,0.108778179098,0.108776435469,0.108774691924,0.108772948462,0.108771205083,0.108769461787,0.108767718575,0.108765975445,0.108764232399,0.108762489436,0.108760746556,0.108759003759,0.108757261045,0.108755518415,0.108753775867,0.108752033403,0.108750291022,0.108748548724,0.108746806509,0.108745064377,0.108743322328,0.108741580362,0.10873983848,0.10873809668,0.108736354963,0.10873461333,0.108732871779,0.108731130312,0.108729388928,0.108727647626,0.108725906408,0.108724165272,0.10872242422,0.108720683251,0.108718942364,0.108717201561,0.108715460841,0.108713720203,0.108711979649,0.108710239177,0.108708498788,0.108706758483,0.10870501826,0.10870327812,0.108701538064,0.10869979809,0.108698058199,0.108696318391,0.108694578665,0.108692839023,0.108691099464,0.108689359987,0.108687620593,0.108685881283,0.108684142055,0.108682402909,0.108680663847,0.108678924868,0.108677185971,0.108675447157,0.108673708427,0.108671969778,0.108670231213,0.108668492731,0.108666754331,0.108665016014,0.10866327778,0.108661539628,0.10865980156,0.108658063574,0.108656325671,0.108654587851,0.108652850113,0.108651112458,0.108649374886,0.108647637397,0.10864589999,0.108644162666,0.108642425425,0.108640688266,0.10863895119,0.108637214197,0.108635477286,0.108633740459,0.108632003713,0.108630267051,0.108628530471,0.108626793974,0.108625057559,0.108623321227,0.108621584978,0.108619848811,0.108618112727,0.108616376726,0.108614640807,0.10861290497,0.108611169217,0.108609433545,0.108607697957,0.108605962451,0.108604227027,0.108602491686,0.108600756428,0.108599021252,0.108597286159,0.108595551148,0.10859381622,0.108592081374,0.108590346611,0.10858861193,0.108586877332,0.108585142817,0.108583408383,0.108581674033,0.108579939764,0.108578205578,0.108576471475,0.108574737454,0.108573003516,0.10857126966,0.108569535886,0.108567802195,0.108566068586,0.10856433506,0.108562601616,0.108560868254,0.108559134975,0.108557401778,0.108555668664,0.108553935632,0.108552202682,0.108550469815,0.10854873703,0.108547004328,0.108545271707,0.108543539169,0.108541806714,0.108540074341,0.10853834205,0.108536609841,0.108534877715,0.108533145671,0.108531413709,0.108529681829,0.108527950032,0.108526218317,0.108524486685,0.108522755134,0.108521023666,0.10851929228,0.108517560976,0.108515829755,0.108514098616,0.108512367559,0.108510636584,0.108508905691,0.108507174881,0.108505444153,0.108503713507,0.108501982943,0.108500252461,0.108498522062,0.108496791744,0.108495061509,0.108493331356,0.108491601285,0.108489871297,0.10848814139,0.108486411565,0.108484681823,0.108482952163,0.108481222585,0.108479493088,0.108477763674,0.108476034342,0.108474305093,0.108472575925,0.108470846839,0.108469117836,0.108467388914,0.108465660074,0.108463931317,0.108462202641,0.108460474048,0.108458745536,0.108457017107,0.10845528876,0.108453560494,0.108451832311,0.108450104209,0.10844837619,0.108446648252,0.108444920397,0.108443192623,0.108441464931,0.108439737322,0.108438009794,0.108436282348,0.108434554984,0.108432827702,0.108431100502,0.108429373384,0.108427646348,0.108425919394,0.108424192521,0.108422465731,0.108420739022,0.108419012395,0.10841728585,0.108415559387,0.108413833006,0.108412106707,0.108410380489,0.108408654353,0.1084069283,0.108405202327,0.108403476437,0.108401750629,0.108400024902,0.108398299257,0.108396573694,0.108394848213,0.108393122813,0.108391397496,0.10838967226,0.108387947105,0.108386222033,0.108384497042,0.108382772133,0.108381047306,0.10837932256,0.108377597897,0.108375873314,0.108374148814,0.108372424395,0.108370700058,0.108368975803,0.108367251629,0.108365527537,0.108363803527,0.108362079598,0.108360355751,0.108358631986,0.108356908302,0.1083551847,0.10835346118,0.108351737741,0.108350014384,0.108348291108,0.108346567914,0.108344844802,0.108343121771,0.108341398822,0.108339675954,0.108337953168,0.108336230463,0.10833450784,0.108332785299,0.108331062839,0.108329340461,0.108327618164,0.108325895948,0.108324173815,0.108322451762,0.108320729792,0.108319007902,0.108317286095,0.108315564368,0.108313842724,0.10831212116,0.108310399678,0.108308678278,0.108306956959,0.108305235722,0.108303514566,0.108301793491,0.108300072498,0.108298351586,0.108296630756,0.108294910007,0.108293189339,0.108291468753,0.108289748249,0.108288027825,0.108286307483,0.108284587223,0.108282867043,0.108281146946,0.108279426929,0.108277706994,0.10827598714,0.108274267368,0.108272547676,0.108270828067,0.108269108538,0.108267389091,0.108265669725,0.10826395044,0.108262231237,0.108260512115,0.108258793074,0.108257074115,0.108255355237,0.10825363644,0.108251917724,0.108250199089,0.108248480536,0.108246762064,0.108245043673,0.108243325364,0.108241607136,0.108239888988,0.108238170923,0.108236452938,0.108234735034,0.108233017212,0.108231299471,0.108229581811,0.108227864232,0.108226146734,0.108224429318,0.108222711982,0.108220994728,0.108219277555,0.108217560463,0.108215843452,0.108214126522,0.108212409674,0.108210692906,0.10820897622,0.108207259614,0.10820554309,0.108203826647,0.108202110285,0.108200394003,0.108198677803,0.108196961684,0.108195245647,0.10819352969,0.108191813814,0.108190098019,0.108188382305,0.108186666672,0.10818495112,0.10818323565,0.10818152026,0.108179804951,0.108178089723,0.108176374576,0.10817465951,0.108172944525,0.108171229621,0.108169514798,0.108167800056,0.108166085395,0.108164370815,0.108162656315,0.108160941897,0.10815922756,0.108157513303,0.108155799127,0.108154085032,0.108152371019,0.108150657086,0.108148943233,0.108147229462,0.108145515772,0.108143802162,0.108142088633,0.108140375185,0.108138661818,0.108136948532,0.108135235327,0.108133522202,0.108131809158,0.108130096195,0.108128383313,0.108126670512,0.108124957791,0.108123245151,0.108121532592,0.108119820114,0.108118107717,0.1081163954,0.108114683164,0.108112971008,0.108111258934,0.10810954694,0.108107835027,0.108106123195,0.108104411443,0.108102699772,0.108100988182,0.108099276672,0.108097565243,0.108095853895,0.108094142627,0.108092431441,0.108090720334,0.108089009309,0.108087298364,0.1080855875,0.108083876716,0.108082166013,0.10808045539,0.108078744849,0.108077034387,0.108075324007,0.108073613707,0.108071903488,0.108070193349,0.10806848329,0.108066773313,0.108065063416,0.108063353599,0.108061643863,0.108059934208,0.108058224633,0.108056515138,0.108054805725,0.108053096391,0.108051387139,0.108049677966,0.108047968874,0.108046259863,0.108044550932,0.108042842082,0.108041133312,0.108039424623,0.108037716014,0.108036007485,0.108034299037,0.10803259067,0.108030882383,0.108029174176,0.10802746605,0.108025758004,0.108024050039,0.108022342154,0.108020634349,0.108018926625,0.108017218981,0.108015511417,0.108013803934,0.108012096532,0.108010389209,0.108008681967,0.108006974806,0.108005267725,0.108003560724,0.108001853803,0.108000146963,0.107998440203,0.107996733523,0.107995026924,0.107993320405,0.107991613966,0.107989907608,0.107988201329,0.107986495132,0.107984789014,0.107983082977,0.10798137702,0.107979671143,0.107977965346,0.10797625963,0.107974553994,0.107972848438,0.107971142962,0.107969437567,0.107967732251,0.107966027016,0.107964321862,0.107962616787,0.107960911792,0.107959206878,0.107957502044,0.10795579729,0.107954092616,0.107952388023,0.107950683509,0.107948979076,0.107947274723,0.10794557045,0.107943866257,0.107942162144,0.107940458111,0.107938754158,0.107937050286,0.107935346493,0.107933642781,0.107931939149,0.107930235596,0.107928532124,0.107926828732,0.10792512542,0.107923422188,0.107921719036,0.107920015964,0.107918312972,0.10791661006,0.107914907229,0.107913204477,0.107911501805,0.107909799213,0.107908096701,0.107906394269,0.107904691917,0.107902989646,0.107901287454,0.107899585342,0.10789788331,0.107896181357,0.107894479485,0.107892777693,0.107891075981,0.107889374348,0.107887672796,0.107885971323,0.107884269931,0.107882568618,0.107880867385,0.107879166232,0.107877465159,0.107875764166,0.107874063253,0.107872362419,0.107870661666,0.107868960992,0.107867260398,0.107865559884,0.107863859449,0.107862159095,0.10786045882,0.107858758626,0.107857058511,0.107855358475,0.10785365852,0.107851958644,0.107850258848,0.107848559132,0.107846859496,0.10784515994,0.107843460463,0.107841761066,0.107840061748,0.107838362511,0.107836663353,0.107834964275,0.107833265277,0.107831566358,0.107829867519,0.10782816876,0.10782647008,0.10782477148,0.10782307296,0.107821374519,0.107819676159,0.107817977877,0.107816279676,0.107814581554,0.107812883512,0.107811185549,0.107809487666,0.107807789863,0.107806092139,0.107804394495,0.10780269693,0.107800999446,0.10779930204,0.107797604715,0.107795907468,0.107794210302,0.107792513215,0.107790816207,0.10778911928,0.107787422431,0.107785725662,0.107784028973,0.107782332363,0.107780635833,0.107778939383,0.107777243012,0.10777554672,0.107773850508,0.107772154375,0.107770458322,0.107768762348,0.107767066454,0.10776537064,0.107763674904,0.107761979248,0.107760283672,0.107758588175,0.107756892758,0.10775519742,0.107753502161,0.107751806982,0.107750111882,0.107748416862,0.107746721921,0.10774502706,0.107743332278,0.107741637575,0.107739942951,0.107738248407,0.107736553943,0.107734859558,0.107733165252,0.107731471025,0.107729776878,0.10772808281,0.107726388822,0.107724694912,0.107723001082,0.107721307332,0.107719613661,0.107717920069,0.107716226556,0.107714533122,0.107712839768,0.107711146494,0.107709453298,0.107707760182,0.107706067145,0.107704374187,0.107702681308,0.107700988509,0.107699295789,0.107697603148,0.107695910586,0.107694218104,0.107692525701,0.107690833377,0.107689141132,0.107687448966,0.10768575688,0.107684064873,0.107682372944,0.107680681096,0.107678989326,0.107677297635,0.107675606024,0.107673914491,0.107672223038,0.107670531664,0.107668840369,0.107667149153,0.107665458017,0.107663766959,0.10766207598,0.107660385081,0.107658694261,0.107657003519,0.107655312857,0.107653622274,0.10765193177,0.107650241345,0.107648550999,0.107646860732,0.107645170544,0.107643480436,0.107641790406,0.107640100455,0.107638410583,0.10763672079,0.107635031077,0.107633341442,0.107631651886,0.107629962409,0.107628273011,0.107626583692,0.107624894453,0.107623205292,0.10762151621,0.107619827207,0.107618138282,0.107616449437,0.107614760671,0.107613071984,0.107611383375,0.107609694846,0.107608006395,0.107606318023,0.107604629731,0.107602941517,0.107601253382,0.107599565326,0.107597877348,0.10759618945,0.10759450163,0.107592813889,0.107591126227,0.107589438644,0.10758775114,0.107586063715,0.107584376368,0.1075826891,0.107581001911,0.107579314801,0.10757762777,0.107575940817,0.107574253943,0.107572567148,0.107570880432,0.107569193795,0.107567507236,0.107565820756,0.107564134354,0.107562448032,0.107560761788,0.107559075623,0.107557389537,0.107555703529,0.1075540176,0.10755233175,0.107550645978,0.107548960286,0.107547274671,0.107545589136,0.107543903679,0.107542218301,0.107540533002,0.107538847781,0.107537162639,0.107535477575,0.10753379259,0.107532107684,0.107530422856,0.107528738107,0.107527053437,0.107525368845,0.107523684332,0.107521999897,0.107520315541,0.107518631264,0.107516947065,0.107515262944,0.107513578903,0.10751189494,0.107510211055,0.107508527249,0.107506843521,0.107505159872,0.107503476302,0.10750179281,0.107500109396,0.107498426061,0.107496742805,0.107495059627,0.107493376527,0.107491693506,0.107490010564,0.1074883277,0.107486644914,0.107484962207,0.107483279579,0.107481597028,0.107479914557,0.107478232163,0.107476549848,0.107474867612,0.107473185454,0.107471503374,0.107469821373,0.10746813945,0.107466457606,0.10746477584,0.107463094152,0.107461412543,0.107459731012,0.107458049559,0.107456368185,0.107454686889,0.107453005672,0.107451324532,0.107449643472,0.107447962489,0.107446281585,0.107444600759,0.107442920011,0.107441239342,0.107439558751,0.107437878239,0.107436197804,0.107434517448,0.10743283717,0.107431156971,0.107429476849,0.107427796806,0.107426116841,0.107424436955,0.107422757147,0.107421077416,0.107419397765,0.107417718191,0.107416038696,0.107414359278,0.107412679939,0.107411000678,0.107409321496,0.107407642391,0.107405963365,0.107404284417,0.107402605547,0.107400926755,0.107399248041,0.107397569406,0.107395890849,0.107394212369,0.107392533968,0.107390855645,0.1073891774,0.107387499234,0.107385821145,0.107384143135,0.107382465202,0.107380787348,0.107379109571,0.107377431873,0.107375754253,0.107374076711,0.107372399247};

      // Linear interpolation
      output(indGrid) = linInterp(x(indGrid), xGrid, yGrid, equalSpaces);

    } else {

      Rcpp::stop("nu must be 0 or 1 to use interpolation");

    }

  }

  // Asymptotic approximation: equations 10.40.1 and 10.17.1 in NIST
  if (indAsymp.n_elem > 0) {

    // Bessel approximation
    arma::vec bIApprox = arma::zeros(indAsymp.n_elem);

    // Common variables used in the series expansion
    arma::vec x8 = 8 * x(indAsymp);
    double dnu = 4 * nu * nu;

    // Series coefficients
    arma::vec ak = arma::ones(indAsymp.n_elem);

    // Series
    for (int k = 1; k <= maxK; k++) {

      // Factorial-like coefficient with switching sign
      ak %= -(dnu - (2 * k - 1) * (2 * k - 1)) / (k * x8);

      // Sum
      bIApprox += ak;

    }

    // Scale
    output(indAsymp) = (bIApprox + arma::ones(indAsymp.n_elem)) / sqrt(2 * arma::datum::pi * x(indAsymp));

  }

  return output;

}

// [[Rcpp::export(name = ".dTvmCpp")]]
arma::vec dTvmCpp(arma::mat x, arma::mat K, arma::mat M, arma::vec alpha, bool besselInterp = false, double l2pi = 0) {

  // Create result
  arma::mat dens = arma::zeros(x.n_rows, K.n_rows);

  // For storing M.row(h - 1)
  arma::rowvec Mh1 = arma::zeros(1, M.n_cols);

  // If l2pi is not provided
  if (l2pi == 0) {

    l2pi = -(x.n_cols * log(2 * arma::datum::pi));

  }

  // Loop on the mixture components
  for (arma::uword h = 0; h < K.n_rows; h++) {

    // Concentrations and means on h-th mixture
    arma::vec Kh = arma::conv_to<arma::vec>::from(K.row(h));
    arma::rowvec Mh = M.row(h);

    // Sum of logs of Bessel functions (exponentially scaled)
    double sumLogBesselExponScaled = 0;
    if (besselInterp) {

      sumLogBesselExponScaled = sum(log(besselIExponScaled(Kh)));

    } else {

      // Loop on the dimension
      for (arma::uword j = 0; j < K.n_cols; j++) {

        // Codification: bessel_i(x, nu, expon.scaled + 1)
        sumLogBesselExponScaled += log(R::bessel_i(Kh(j), 0, 2));

      }

    }

    // Recenter data
    x.each_row() -= Mh - Mh1;
    Mh1 = Mh;

    // Density of the weighted component
    dens.col(h) = alpha(h) * exp(l2pi - sumLogBesselExponScaled + (cos(x) - 1) * Kh);

  }

  return sum(dens, 1);

}

// [[Rcpp::export(name = ".clusterProbsTvm")]]
arma::mat clusterProbsTvm(arma::mat cosData, arma::mat sinData, const arma::mat M, const arma::mat K, arma::rowvec alpha, double l2pi, bool besselInterp = true) {

  // cos and sin of means times concentrations
  arma::mat cosMK = cos(M) % K ;
  arma::mat sinMK = sin(M) % K;

  // Compute Bessels
  arma::rowvec sumLogBesselExponScaled = arma::zeros(1, K.n_rows);
  if (besselInterp) {

    sumLogBesselExponScaled = sum(reshape(log(besselIExponScaled(vectorise(K))), K.n_rows, K.n_cols), 1).t();

  } else {

    for (arma::uword h = 0; h < M.n_rows; h++) {
      for(arma::uword j = 0; j < K.n_cols; j++) {

        // Codification: bessel_i(x, nu, expon.scaled + 1)
        sumLogBesselExponScaled(h) += log(R::bessel_i(K(h, j), 0, 2));

      }
    }

  }

  // Compute weights
  arma::mat weights = l2pi + cosData * cosMK.t() + sinData * sinMK.t();
  weights.each_row() -= sumLogBesselExponScaled + sum(K, 1).t();
  weights = exp(weights);

  // Weight by mixture weights
  weights.each_row() %= alpha;
  weights.each_col() /= sum(weights, 1);

  return weights;

}

// [[Rcpp::export(name = ".weightedMuKappa")]]
Rcpp::List weightedMuKappa(arma::mat cosData, arma::mat sinData, arma::mat weights, double kappaMax = 250, bool isotropic = false) {

  // Dimension and number of sets of weights
  arma::uword p = cosData.n_cols;
  arma::uword k = weights.n_cols;

  // Replace by transpose
  inplace_trans(weights);

  // Cartesian weighted means, matrices of sizes c(k, p)
  arma::mat meanCos = weights * cosData;
  arma::mat meanSin = weights * sinData;

  // Circular means - atan2 is not vectorized yet!
  arma::mat M = arma::zeros(k, p);
  for (arma::uword i = 0; i < k; i++) {
    for (arma::uword j = 0; j < p; j++) {

      M(i, j) = atan2(meanSin(i, j), meanCos(i, j));

    }
  }

  // Mean resultant lengths
  arma::mat K = sqrt(square(meanCos) + square(meanSin));

  // rh = c(rh1, ..., rhp)
  K.each_col() /= sum(weights, 1);

  // Approximation to A1Inv as in Banerjee et al. (2005)
  K += K / (1 - square(K));

  // Clamp to (0, kappaMax)
  return Rcpp::List::create(Rcpp::Named("M") = M, Rcpp::Named("K") = clamp(K, 0.0, kappaMax));

}
