% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ptable_to_syntax.R
\name{ptable_to_syntax}
\alias{ptable_to_syntax}
\alias{compare_ptables}
\title{Convert a 'lavaan' Parameter
Table to a 'lavaan' Model Syntax}
\usage{
ptable_to_syntax(object, allow_incomplete = FALSE)

compare_ptables(object1, object2)
}
\arguments{
\item{object}{If set to a \code{lavaan}
object, such
as the output of \code{\link[lavaan:sem]{lavaan::sem()}}
or \code{\link[lavaan:cfa]{lavaan::cfa()}}, the parameter
table will be extracted
from it by \code{\link[lavaan:lavParTable]{lavaan::parameterTable()}}.
If set to a parameter table, it will
be used to generate the model syntax.
It can also the output of
\code{\link[lavaan:model.syntax]{lavaan::lavParseModelString()}} with
\code{as.data.frame. = TRUE}, if
\code{allow_incomplete} is set to \code{TRUE}.
Note that \code{allow_incomplete} is set
to \code{FALSE} by default because
\code{\link[lavaan:model.syntax]{lavaan::lavParseModelString()}} only
parses the model syntax and there is
no guarantee that the model defined
is valid.}

\item{allow_incomplete}{Whether
incomplete parameter table formed
by \code{\link[lavaan:model.syntax]{lavaan::lavParseModelString()}}
with \code{as.data.frame. = TRUE} is
allowed. Default if \code{FALSE}.}

\item{object1}{The first \code{lavaan}
parameter table, to be compared with
\code{object2}. If it is set to a
\code{lavaan} object (e.g., the output
of \code{\link[lavaan:sem]{lavaan::sem()}} or
\code{\link[lavaan:cfa]{lavaan::cfa()}}), then the
parameter table will be extracted
from it.}

\item{object2}{The second \code{lavaan}
parameter table, to be compared with
\code{object1}. If it is set to a
\code{lavaan} object (e.g., the output
of \code{\link[lavaan:sem]{lavaan::sem()}} or
\code{\link[lavaan:cfa]{lavaan::cfa()}}), then the
parameter table will be extracted
from it.}
}
\value{
\code{\link[=ptable_to_syntax]{ptable_to_syntax()}} returns a
length-one character vector that stores
the generated \code{lavaan} model syntax.

\code{\link[=compare_ptables]{compare_ptables()}} returns a
length-one logical vector. \code{TRUE}
if the two models are identical
in form. \code{FALSE} if they are
not identical.
}
\description{
It tries to generate a
'lavaan' model syntax from a
\code{lavaan} parameter table.
}
\details{
This function tries to convert
a \code{lavaan} parameter table to a
text representation of the \code{lavaan}
model specified in model syntax.

When users call \code{\link[lavaan:sem]{lavaan::sem()}},
in addition to the model syntax,
other arguments not stored in the
syntax are also used to produce
the final model (e.g.,
\code{meanstructure}, \code{fixed.x}, and
\code{std.lv}). To produce exactly the
same model, these arguments are also
needed to be specified, which is
difficult to generate using only
the parameter table.

Therefore, the model syntax produced
will state all aspects of a model
explicitly, even for those aspects
that usually can be omitted due to
the default values of these arguments.
This approach requires users to call
\code{\link[lavaan:lavaan]{lavaan::lavaan()}} directly, instead
of its wrappers (e.g, \code{\link[lavaan:sem]{lavaan::sem()}}),
to produce the same parameter table.

The model syntax produced this way
is more difficult to read. However,
it ensures that original model can
be reproduced, without the need to
know the arguments to set.

Due to the nearly unlimited possibilities
in the form of a model, it is
recommended to compare the model
generated by the model syntax with
the original parameter table using
\code{\link[=compare_ptables]{compare_ptables()}}. It
only compares the forms of the
two models, including user starting
values, if any. It does not compare
parameter estimates and standard
errors.
\subsection{Raw Specification From \code{lavaan::lavParseModelString()}}{

There may be cases in which the
parameter table is the "incomplete"
table generated by \code{\link[lavaan:model.syntax]{lavaan::lavParseModelString()}},
with \code{as.data.frame. = TRUE}.
This table is "incomplete" because
it is formed merely by parsing the
model syntax. There is no guarantee
that the model is valid.

The function \code{\link[=ptable_to_syntax]{ptable_to_syntax()}}
has basic support for this kind
of tables but it is disabled by
default. To process an incomplete
parameter table formed by
\code{\link[lavaan:model.syntax]{lavaan::lavParseModelString()}},
set \code{allow_incomplete} to \code{TRUE}.
}

\subsection{Limitations}{

The function \code{\link[=ptable_to_syntax]{ptable_to_syntax()}}
does not yet support the following
models:
\itemize{
\item Multiple-group models.
\item Multilevel models.
\item Models with categorical variables.
\item Models with user-specified lower or
upper bounds.
\item Models with the operator \verb{<~}.
\item Models with constraints imposed by
\code{equal()}.
\item Models with labels having spaces.
\item Models with labels having syntax
operators (e.g., \code{~}, \verb{=~}, etc.).
}
}
}
\section{Functions}{
\itemize{
\item \code{ptable_to_syntax()}: Convert
a lavaan parameter a lavaan model
syntax.

\item \code{compare_ptables()}: Compare two lavaan parameter tables.

}}
\examples{

library(lavaan)

mod <-
"
visual  =~ x3 + x1 + x2
textual =~ x4 + x6 + x5
speed   =~ x7 + x8 + x9 + start(0.1) * x6
visual ~ a*textual
speed ~ b*visual
ab: = a * b
"

fit <- sem(mod, data = HolzingerSwineford1939)

mod_chk <- ptable_to_syntax(fit)
cat(mod_chk, sep = "\n")
# Need to call lavaan() directly
fit_chk <- lavaan(mod_chk, data = HolzingerSwineford1939)
fit_chk
fit
# Compare the parameter table:
(ptable1 <- parameterTable(fit))
(ptable2 <- parameterTable(fit_chk))
compare_ptables(ptable1, ptable2)


}
\seealso{
\code{\link[lavaan:lavaan]{lavaan::lavaan()}}, \code{\link[lavaan:lavParTable]{lavaan::parameterTable()}}
}
\author{
Shu Fai Cheung \url{https://orcid.org/0000-0002-9871-9448}.
This function is inspired by
a discussion at the Google Group
\url{https://groups.google.com/g/lavaan/c/1ueFiue9qLM/m/cJhxDoqeBAAJ}.
}
