% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/select_counts.R
\name{select_counts}
\alias{select_counts}
\title{Subsample the rows and columns of a count matrix.}
\usage{
select_counts(mat, nsamp = ncol(mat), ngene = nrow(mat),
  gselect = c("random", "max", "mean_max", "custom"), gvec = NULL,
  filter_first = FALSE, nskip = 0L)
}
\arguments{
\item{mat}{A numeric matrix of RNA-seq counts. The rows index the genes
and the columns index the samples.}

\item{nsamp}{The number of samples (columns) to select from \code{mat}.}

\item{ngene}{The number of genes (rows) to select from \code{mat}.}

\item{gselect}{How should we select the subset of genes? Options include:
\describe{
  \item{\code{random}}{Randomly select the genes, with each gene having
      an equal probability of being included in the subsampled matrix.}
  \item{\code{max}}{Choose the \code{ngene} most median-expressed genes.
      Ties are broken by mean-expression.}
  \item{\code{mean_max}}{Choose the \code{ngene} most mean-expressed
      genes.}
  \item{\code{custom}}{A user-specified list of genes. If
      \code{gselect = "custom"} then \code{gvec} needs to be
      non-\code{NULL}.}
}}

\item{gvec}{A logical vector of length \code{nrow(mat)}. A \code{TRUE}
in position \eqn{i} indicates inclusion into the smaller dataset.
Hence, \code{sum(gvec)} should equal \code{ngene}.}

\item{filter_first}{Should we first filter genes by the method of
Chen et al. (2016) (\code{TRUE}) or not (\code{FALSE})? If
\code{TRUE} then the \code{edgeR} package should be installed.}

\item{nskip}{The number of median-maximally expressed genes to skip.
Not used if \code{gselect = "custom"}.}
}
\value{
A numeric matrix, which is a \code{ngene} by \code{nsamp} sub-matrix
    of \code{mat}. If \code{rownames(mat)} is \code{NULL}, then the
    row names of the returned matrix are the indices in \code{mat} of the
    selected genes. If \code{colnames(mat)} is \code{NULL}, then the
    column names of the returned matrix are the indices in \code{mat} of
    the selected samples.
}
\description{
It is a good idea to subsample (each iteration) the genes and samples from
a real RNA-seq dataset prior to applying \code{\link{thin_diff}}
(and related functions) so that your conclusions are not dependent on the
specific structure of your dataset. This function is designed to efficiently
do this for you.
}
\details{
The samples (columns) are chosen randomly, with each sample having
an equal probability of being in the sub-matrix. The genes are selected
according to one of four schemes (see the description of the \code{gselect}
argument).

If you have edgeR installed, then some functionality is provided for
filtering out the lowest expressed genes prior to applying subsampling
(see the \code{filter_first} argument).
This filtering scheme is described in Chen et al. (2016).
If you want more control over this filtering, you should use
the \code{\link[edgeR]{filterByExpr}} function from edgeR directly. You
can install edgeR by following instructions here:
\url{https://doi.org/doi:10.18129/B9.bioc.edgeR}.
}
\examples{
## Simulate data from given matrix of counts
## In practice, you would obtain mat from a real dataset, not simulate it.
set.seed(1)
n   <- 100
p   <- 1000
mat <- matrix(stats::rpois(n * p, lambda = 50), nrow = p)

## Subsample the matrix, then feed it into a thinning function
submat <- select_counts(mat = mat, nsamp = 10, ngene = 100)
thout  <- thin_2group(mat = submat, prop_null = 0.5)

## The rownames and colnames (if NULL in mat) tell you which genes/samples
## were selected.
rownames(submat)
colnames(submat)

}
\references{
\itemize{
  \item{Chen, Yunshun, Aaron TL Lun, and Gordon K. Smyth. "From reads to genes to pathways: differential expression analysis of RNA-Seq experiments using Rsubread and the edgeR quasi-likelihood pipeline." F1000Research 5 (2016).}
}
}
\author{
David Gerard
}
