#' sharp: Stability-enHanced Approaches using Resampling Procedures
#'
#' In stability selection, resampling techniques are used to enhance the
#' reliability of the results. In this package, hyper-parameters are calibrated
#' by maximising model stability, which is measured by the negative
#' log-likelihood under the null hypothesis that all selection probabilities are
#' identical. Functions are readily implemented for the use of LASSO regression,
#' sparse PCA, sparse (group) PLS or graphical LASSO.
#'
#' \tabular{ll}{ Package: \tab sharp\cr Type: \tab Package\cr Version: \tab
#' 1.2.1 \cr Date: \tab 2022-12-08 \cr License: \tab GPL (>= 3)\cr Maintainer:
#' \tab Barbara Bodinier \email{b.bodinier@@imperial.ac.uk}}
#'
#' @references \insertRef{ourstabilityselection}{sharp}
#'
#'   \insertRef{stabilityselectionMB}{sharp}
#'
#'   \insertRef{ConsensusClustering}{sharp}
#'
#' @docType package
#' @name sharp-package
#'
#' @importFrom Rdpack reprompt
#' @importFrom mclust mclustBIC
#'
#' @examples
#' \donttest{
#' oldpar <- par(no.readonly = TRUE)
#' par(mar = c(5, 5, 5, 5))
#'
#' ## Regression models
#'
#' # Data simulation
#' set.seed(1)
#' simul <- SimulateRegression(n = 100, pk = 50)
#'
#' # Stability selection
#' stab <- VariableSelection(xdata = simul$xdata, ydata = simul$ydata)
#' CalibrationPlot(stab)
#' summary(stab)
#' SelectedVariables(stab)
#'
#'
#' ## Graphical models
#'
#' # Data simulation
#' set.seed(1)
#' simul <- SimulateGraphical(n = 100, pk = 20, topology = "scale-free")
#'
#' # Stability selection
#' stab <- GraphicalModel(xdata = simul$data)
#' CalibrationPlot(stab)
#' summary(stab)
#' plot(stab)
#'
#'
#' ## PCA models
#'
#' # Data simulation
#' set.seed(1)
#' simul <- SimulateComponents(pk = c(5, 3, 4))
#' plot(simul)
#'
#' # Stability selection
#' stab <- BiSelection(
#'   xdata = simul$data,
#'   ncomp = 3,
#'   implementation = SparsePCA
#' )
#' CalibrationPlot(stab)
#' summary(stab)
#' SelectedVariables(stab)
#'
#' par(oldpar)
#' }
NULL
