## __________________________________________________________
##
## SIMoNe
##
## INPUTS
##	X  : n*p data matrix
##	Q  : max number of classes to estimate
##	
## OUTPUT
##   a list with
##	classes : classification results of the last iteration
##	edges : edge estimation results of the last iteration
##	edges.perfect : Edge estimation results using
##                      theoretical classes information
##                      (only if cl.theo is given)
##
## Hybrid IE + IC algorithm
## __________________________________________________________
##
simone <- function (X, Q, ... ) {
  
  ## defaults for hidden optional arguments ##
  multipliers <- sub.param("multipliers", list(init=1,intra=1,inter=1.2,dust=1.5), ...)
  rho      <- sub.param("rho", 2/(BaseLambdaValue(X)*nrow(X)), ...)
  num.iter <- sub.param("num.iter", 5     , ...)
  eps      <- sub.param("eps"     , 1e-4 , ...)	
  cl.theo  <- sub.param("cl.theo" , NULL  , ...)	
  verbose  <- sub.param("verbose" , FALSE  , ...)
  
  ## parameters setup ##
  p          <- ncol(X)
  Sigma.hat  <- NULL
  resIE      <- NULL
  K.perfect  <- NULL
  resIC      <- NULL
  threshold  <- eps * sum(abs(var(X)))/p
  param      <- list()
  #param$mu   <- matrix(0,Q,Q)
  
  ## ---------------- ##
  ## 'perfect SIMoNe' ##
  ## ---------------- ##
  ## estimate edges with theoretic class knowledge 
  if (!is.null(cl.theo)) {
    ## use base.penalty with cl.theo and multipliers
    Rho <- Penalty(X, rho = rho,
                   classes = cl.theo,
                   multipliers = multipliers)
    K.perfect <- InferEdges(X, Rho, maxIt = 500, ...)$K.hat
  }
  
  ## --------- ##
  ## MAIN LOOP ##
  ## --------- ##

  delta     <- c(Inf)
  loop      <- TRUE
  iteration <- 1
  
  if (verbose) {
    cat("\n=============================== \n")
    cat(" STARTING SIMoNe ALGORITHM")
  }

  while (loop) {

    if (verbose) { 
      cat("\n=============================== \n")
      cat(" ITERATION",iteration,"on",num.iter," \n") 
    }

    ### EDGES ESTIMATION ##
    ## ----------------- ##
    if (verbose==TRUE) { cat("\n  M STEP (edges estimation) \n") }
    if (iteration>1){
      Rho <- Penalty(X, rho=rho,
                     classes = outIC$cl,
                     multipliers = multipliers)
    } else {
      Rho <- Penalty(X, rho=rho*multipliers$init)     
    }
    outIE <- InferEdges(X, Rho, Sigma.hat = Sigma.hat, ...)
    
    ## CLASSES ESTIMATION ##
    ## ------------------ ##
    if (verbose==TRUE) { cat("\n  E STEP (classes estimation) \n") }    
   
    outIC <- InferClasses(outIE$K.hat,
                          Q,
                          verbose = verbose,
                          param.default=param, ...)
    
    ## CONVERGENCE CHECK ##
    ## ----------------- ##
    ## if two consecutive estimated covariances matrices are identical
    ## according to the l1-norm, then estimation has converged.
    if (iteration > 1) {
      delta[iteration] <- sum ( abs(outIE$Sigma.hat - Sigma.hat) )
    }

    ## preparing the next iterate
    Sigma.hat <- outIE$Sigma.hat
    loop <- ( delta[iteration] > threshold ) && ( iteration < num.iter )
    iteration <- iteration + 1
  }

  ## RETURN RESULT ##
  ## ------------- ##
  res.simone <- list(cl            = factor(outIC$cl), # last vertices classification
                     K.hat         = outIE$K.hat, # last edges estimation
                     K.hat.perfect = K.perfect,   # estimation with theo classif 
                     delta	   = delta[-1],   # convergence
                     iteration     = iteration-1) # last iterate
  
  class(res.simone) <- "simone"
  return(res.simone)
}

## __________________________________________________________
##
## summary.simone
##
## INPUT :
##	res : an object of class 'simone', generated by 
##		function simone()
##
## ADDITIONAL INPUT '...' :
##	theo.Data : a list containing data about a 
##		theoretic graph, for use with simulations
##		$cl.theo : theoretic classification
##		$K.theo : theoretic precision matrix
##
## OUTPUT : to-screen summary
##
## Default summary method for an object of class 'simone'
## __________________________________________________________
##
summary.simone <- function ( object, ...)  {

  if (class(object)!="simone") { 
    cat("\n Trying to summary.simone a non-simone object : aborted \n")
    return(NULL)
  }

  theo.Data <- sub.param("theo.Data", NULL, ...)

  cat("\n ===================================== \n")
  cat(  " Summary of 'simone' algorithm results \n")
  cat(  " ===================================== \n")

  ## convergence ##
  cat("\n - Number of iterations :  \t", length(object$iteration))
  cat("\n - Convergence criteria : \n")
  print(object$delta)

  ## graph topology ##
  cat("\n - Number of nodes in the estimated graph : \t", length(object$cl) )
  if (!is.null(theo.Data$K.theo)) {
    cat("\n - Number of edges in the theoretic graph : \t",sum(abs(theo.Data$K.theo)[upper.tri(theo.Data$K.theo)]))
  }
  cat("\n - Number of estimated edges in final iteration : \t",
      sum(abs(object$K.hat)[upper.tri(object$K.hat)]))

  ## graph colours ##
  if (!is.null(theo.Data$cl.theo)) {
      cat("\n - Theoretic classification summary : \n ")
      print(summary(theo.Data$cl.theo))
  }

  cat("\n - Final estimated classification summary : \n")
  print(summary(object$cl))
  cat("\n")
}

## __________________________________________________________
##
##	sub.param
##
## Check for a specified parameter in the additional 
## parameters "..." and return it if it exists, or return
## a default value if not.
##
## INPUT : 
##	param : string designating the parameter to find
##	default : a default value to return if parameter 
##		is inexistant
##
## OUTPUT :
##	an object
## __________________________________________________________
##
sub.param <- function (param, default=NULL, ... ) {

  if (missing(param)) { return(default) }

  l <- list(...)
  res <- l[[  which( names(l) %in% param)[1] ]] 

  if (is.null(res)) { res <- default }

  return ( res )
}
