% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sim_fixed_n.R
\name{sim_fixed_n}
\alias{sim_fixed_n}
\title{Simulation of fixed sample size design for time-to-event endpoint}
\usage{
sim_fixed_n(
  n_sim = 1000,
  sample_size = 500,
  target_event = 350,
  stratum = data.frame(stratum = "All", p = 1),
  enroll_rate = data.frame(duration = c(2, 2, 10), rate = c(3, 6, 9)),
  fail_rate = data.frame(stratum = "All", duration = c(3, 100), fail_rate = log(2)/c(9,
    18), hr = c(0.9, 0.6), dropout_rate = rep(0.001, 2)),
  total_duration = 30,
  block = rep(c("experimental", "control"), 2),
  timing_type = 1:5,
  rho_gamma = data.frame(rho = 0, gamma = 0)
)
}
\arguments{
\item{n_sim}{Number of simulations to perform.}

\item{sample_size}{Total sample size per simulation.}

\item{target_event}{Targeted event count for analysis.}

\item{stratum}{A data frame with stratum specified in \code{stratum},
probability (incidence) of each stratum in \code{p}.}

\item{enroll_rate}{Piecewise constant enrollment rates by time period.
Note that these are overall population enrollment rates and the \code{stratum}
argument controls the random distribution between stratum.}

\item{fail_rate}{Piecewise constant control group failure rates, hazard ratio
for experimental vs. control, and dropout rates by stratum and time period.}

\item{total_duration}{Total follow-up from start of enrollment to data cutoff.}

\item{block}{As in \code{\link[=sim_pw_surv]{sim_pw_surv()}}. Vector of treatments to be included
in each block.}

\item{timing_type}{A numeric vector determining data cutoffs used;
see details. Default is to include all available cutoff methods.}

\item{rho_gamma}{A data frame with variables
\code{rho} and \code{gamma}, both greater than equal to zero,
to specify one Fleming-Harrington weighted logrank test per row.}
}
\value{
A data frame including columns:
\itemize{
\item \code{event}: Event count.
\item \code{ln_hr}: Log-hazard ratio.
\item \code{z}: Normal test statistic; < 0 favors experimental.
\item \code{cut}: Text describing cutoff used.
\item \code{duration}: Duration of trial at cutoff for analysis.
\item \code{sim}: Sequential simulation ID.
}

One row per simulated dataset per cutoff specified in \code{timing_type},
per test statistic specified.
If multiple Fleming-Harrington tests are specified in \code{rho_gamma},
then columns \code{rho} and \code{gamma} are also included.
}
\description{
\code{sim_fixed_n()} provides simulations of a single endpoint two-arm trial
where the enrollment, hazard ratio, and failure and dropout rates change
over time.
}
\details{
\code{timing_type} has up to 5 elements indicating different options
for data cutoff:
\itemize{
\item \code{1}: Uses the planned study duration.
\item \code{2}: The time the targeted event count is achieved.
\item \code{3}: The planned minimum follow-up after enrollment is complete.
\item \code{4}: The maximum of planned study duration and targeted event count cuts
(1 and 2).
\item \code{5}: The maximum of targeted event count and minimum follow-up cuts
(2 and 3).
}
}
\examples{
\dontshow{if (requireNamespace("dplyr", quietly = TRUE)) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
library(dplyr)
library(future)

# Example 1: logrank test ----
x <- sim_fixed_n(n_sim = 10, timing_type = 1, rho_gamma = data.frame(rho = 0, gamma = 0))
# Get power approximation
mean(x$z <= qnorm(.025))

# Example 2: WLR with FH(0,1) ----
sim_fixed_n(n_sim = 1, timing_type = 1, rho_gamma = data.frame(rho = 0, gamma = 1))
# Get power approximation
mean(x$z <= qnorm(.025))

\donttest{
# Example 3: MaxCombo, i.e., WLR-FH(0,0)+ WLR-FH(0,1)
# Power by test
# Only use cuts for events, events + min follow-up
x <- sim_fixed_n(
  n_sim = 10,
  timing_type = 2,
  rho_gamma = data.frame(rho = 0, gamma = c(0, 1))
)

# Get power approximation
x |>
  group_by(sim) |>
  filter(row_number() == 1) |>
  ungroup() |>
  summarize(power = mean(p_value < .025))

# Example 4
# Use two cores
set.seed(2023)
plan("multisession", workers = 2)
sim_fixed_n(n_sim = 10)
plan("sequential")

# Example 5: Stratified design with two strata
sim_fixed_n(
  n_sim = 100,
  sample_size = 500,
  target_event = 350,
  stratum = data.frame(stratum = c("Biomarker-positive", "Biomarker-negative"),
                       p = c(0.5, 0.5)),
  enroll_rate = data.frame(stratum = c("Biomarker positive", "Biomarker negative"),
                           duration = c(12, 12), rate = c(250 / 12, 250 / 12)),
  fail_rate = data.frame(stratum = c(rep("Biomarker-positive", 2), rep("Biomarker-negative", 2)),
                         duration = c(3, Inf, 3, Inf),
                         fail_rate = log(2) / c(12, 12, 8, 8),
                         hr = c(1, 0.6, 1, 0.7),
                         dropout_rate = rep(0.001, 4)),
  total_duration = 30,
  block = rep(c("experimental", "control"), 2),
  timing_type = 1:5,
  rho_gamma = data.frame(rho = 0, gamma = 0)
  )
}
\dontshow{\}) # examplesIf}
}
