% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/optimize.R
\name{optimizePars}
\alias{optimizePars}
\title{Optimize parameters for acoustic analysis}
\usage{
optimizePars(
  myfolder,
  key,
  myfun,
  pars,
  bounds = NULL,
  fitnessPar,
  fitnessFun = function(x) 1 - cor(x, key, use = "pairwise.complete.obs"),
  nIter = 10,
  init = NULL,
  initSD = 0.2,
  control = list(maxit = 50, reltol = 0.01, trace = 0),
  otherPars = list(plot = FALSE, verbose = FALSE),
  mygrid = NULL,
  verbose = TRUE
)
}
\arguments{
\item{myfolder}{path to where the .wav files live}

\item{key}{a vector containing the "correct" measurement that we are aiming
to reproduce}

\item{myfun}{the function being optimized: either 'segmentFolder' or
'analyzeFolder' (in quotes)}

\item{pars}{names of arguments to \code{myfun} that should be
optimized}

\item{bounds}{a list setting the lower and upper boundaries for possible
values of optimized parameters. For ex., if we optimize \code{smooth}
and \code{smoothOverlap}, reasonable bounds might be list(low = c(5,
0), high = c(500, 95))}

\item{fitnessPar}{the name of output variable that we are comparing with the
key, e.g. 'nBursts' or 'pitch_median'}

\item{fitnessFun}{the function used to evaluate how well the output of
\code{myfun} fits the key. Defaults to 1 - Pearson's correlation (i.e. 0 is
perfect fit, 1 is awful fit). For pitch, log scale is more meaningful, so a
good fitness criterion is "function(x) 1 - cor(log(x), log(key), use =
'pairwise.complete.obs')"}

\item{nIter}{repeat the optimization several times to check convergence}

\item{init}{initial values of optimized parameters (if NULL, the default
values are taken from the definition of \code{myfun})}

\item{initSD}{each optimization begins with a random seed, and
\code{initSD} specifies the SD of normal distribution used to generate
random deviation of initial values from the defaults}

\item{control}{a list of control parameters passed on to
\code{\link[stats]{optim}}. The method used is "Nelder-Mead"}

\item{otherPars}{a list of additional arguments to \code{myfun}}

\item{mygrid}{a dataframe with one column per parameter to optimize, with
each row specifying the values to try. If not NULL, \code{optimizePars}
simply evaluates each combination of parameter values, without calling
\code{\link[stats]{optim}} (see examples)}

\item{verbose}{if TRUE, reports the values of parameters evaluated and fitness}
}
\value{
Returns a matrix with one row per iteration with fitness in the first
  column and the best values of each of the optimized parameters in the
  remaining columns.
}
\description{
This customized wrapper for \code{\link[stats]{optim}} attempts to optimize the
parameters of \code{\link{segmentFolder}} or \code{\link{analyzeFolder}} by
comparing the results with a manually annotated "key". This optimization
function uses a single measurement per audio file (e.g., median pitch or the
number of syllables). For other purposes, you may want to adapt the
optimization function so that the key specifies the exact timing of
syllables, their median length, frame-by-frame pitch values, or any other
characteristic that you want to optimize for. The general idea remains the
same, however: we want to tune function parameters to fit our type of audio
and research priorities. The default settings of \code{\link{segmentFolder}}
and \code{\link{analyzeFolder}} have been optimized for human non-linguistic
vocalizations.
}
\details{
If your sounds are very different from human non-linguistic vocalizations,
you may want to change the default values of other arguments to speed up
convergence. Adapt the code to enforce suitable constraints, depending
on your data.
}
\examples{
\dontrun{
# Download 260 sounds from the supplements in Anikin & Persson (2017)
# - see http://cogsci.se/publications.html
# Unzip them into a folder, say '~/Downloads/temp'
myfolder = '~/Downloads/temp'  # 260 .wav files live here

# Optimization of SEGMENTATION
# Import manual counts of syllables in 260 sounds from
# Anikin & Persson (2017) (our "key")
key = segmentManual  # a vector of 260 integers

# Run optimization loop several times with random initial values
# to check convergence
# NB: with 260 sounds and default settings, this might take ~20 min per iteration!
res = optimizePars(myfolder = myfolder, myfun = 'segmentFolder', key = key,
  pars = c('shortestSyl', 'shortestPause', 'sylThres'),
  fitnessPar = 'nBursts',
  nIter = 3, control = list(maxit = 50, reltol = .01, trace = 0))

# Examine the results
print(res)
for (c in 2:ncol(res)) {
  plot(res[, c], res[, 1], main = colnames(res)[c])
}
pars = as.list(res[1, 2:ncol(res)])  # top candidate (best pars)
s = do.call(segmentFolder, c(myfolder, pars))  # segment with best pars
cor(key, as.numeric(s[, fitnessPar]))
boxplot(as.numeric(s[, fitnessPar]) ~ as.integer(key), xlab='key')
abline(a=0, b=1, col='red')

# Try a grid with particular parameter values instead of formal optimization
res = optimizePars(myfolder = myfolder, myfun = 'segmentFolder', key = segment_manual,
  pars = c('shortestSyl', 'shortestPause'),
  fitnessPar = 'nBursts',
  mygrid = expand.grid(shortestSyl = c(30, 40),
                       shortestPause = c(30, 40, 50)))
1 - res$fit  # correlations with key

# Optimization of PITCH TRACKING (takes several hours!)
res = optimizePars(myfolder = myfolder,
                   myfun = 'analyzeFolder',
                   key = log(pitchManual),  # log-scale better for pitch
                   pars = c('specThres', 'specSmooth'),
                   bounds = list(low = c(0, 0), high = c(1, Inf)),
                   fitnessPar = 'pitch_median',
                   nIter = 2,
                   otherPars = list(plot = FALSE, verbose = FALSE, step = 50,
                                    pitchMethods = 'spec'),
                   fitnessFun = function(x) {
                     1 - cor(log(x), key, use = 'pairwise.complete.obs') *
                       (1 - mean(is.na(x) & !is.na(key)))  # penalize failing to detect F0
                     })
}
}
