% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/filters.R
\name{bandpass}
\alias{bandpass}
\title{Bandpass/stop filters}
\usage{
bandpass(
  x,
  samplingRate = NULL,
  lwr = NULL,
  upr = NULL,
  action = c("pass", "stop")[1],
  dB = Inf,
  na.rm = TRUE,
  from = NULL,
  to = NULL,
  normalize = FALSE,
  reportEvery = NULL,
  saveAudio = NULL,
  plot = FALSE,
  savePlots = NULL,
  width = 900,
  height = 500,
  units = "px",
  res = NA,
  ...
)
}
\arguments{
\item{x}{path to a folder, one or more wav or mp3 files c('file1.wav',
'file2.mp3'), Wave object, numeric vector, or a list of Wave objects or
numeric vectors}

\item{samplingRate}{sampling rate of \code{x} (only needed if \code{x} is a
numeric vector)}

\item{lwr, upr}{cutoff frequencies, Hz. Specifying just lwr gives a low-pass
filter, just upr high-pass filter with action = 'pass' (or vice versa with
action = 'stop'). Specifying both lwr and upr a bandpass/bandstop filter,
depending on 'action'}

\item{action}{"pass" = preserve the selected frequency range (bandpass),
"stop" = remove the selected frequency range (bandstop)}

\item{dB}{a positive number giving the strength of effect in dB (defaults to
Int - complete removal of selected frequencies)}

\item{na.rm}{if TRUE, NAs are interpolated, otherwise they are preserved in
the output}

\item{from}{if NULL (default), analyzes the whole sound, otherwise
from...to (s)}

\item{to}{if NULL (default), analyzes the whole sound, otherwise
from...to (s)}

\item{normalize}{if TRUE, resets the output to the original scale (otherwise
filtering often reduces the amplitude)}

\item{reportEvery}{when processing multiple inputs, report estimated time
left every ... iterations (NULL = default, NA = don't report)}

\item{saveAudio}{full path to the folder in which to save audio files (one
per detected syllable)}

\item{plot}{should a spectrogram be plotted? TRUE / FALSE}

\item{savePlots}{full path to the folder in which to save the plots (NULL =
don't save, '' = same folder as audio)}

\item{width}{graphical parameters for saving plots passed to
\code{\link[grDevices]{png}}}

\item{height}{graphical parameters for saving plots passed to
\code{\link[grDevices]{png}}}

\item{units}{graphical parameters for saving plots passed to
\code{\link[grDevices]{png}}}

\item{res}{graphical parameters for saving plots passed to
\code{\link[grDevices]{png}}}

\item{...}{other graphical parameters}
}
\description{
Filtering in the frequency domain with FFT-iFFT: low-pass, high-pass,
bandpass, and bandstop filters. Similar to \code{\link[seewave]{ffilter}},
but here we use FFT instead of STFT - that is, the entire sound is processed
at once. This works best for relatively short sounds (seconds), but gives us
maximum precision (e.g., for precise notch filtering) and doesn't affect the
attack and decay. NAs are accepted and can be interpolated or preserved in
the output. Because we don't do STFT, arbitrarily short vectors are also fine
as input - for example, we can apply a low-pass filter prior to decimation
when changing the sampling rate without aliasing. Note that, unlike
\code{\link{pitchSmoothPraat}}, \code{bandpass} applies an abrupt cutoff
instead of a smooth gaussian filter.
}
\details{
Algorithm: fill in NAs with constant interpolation at the edges and linear
interpolation in the middle; perform FFT; set the frequency ranges to be
filtered out to 0; perform inverse FFT; set to the original scale; put the
NAs back in.
}
\examples{
# Filter white noise
s1 = fade(c(runif(2000, -1, 1)))
bandpass(s1, 16000, upr = 2000, plot = TRUE)    # low-pass
bandpass(s1, 16000, lwr = 2000, dB = 40, plot = TRUE)  # high-pass by 40 dB
bandpass(s1, 16000, lwr = 1000, upr = 1100, action = 'stop', plot = TRUE) # bandstop
s2 = bandpass(s1, 16000, lwr = 2000, upr = 2100, plot = TRUE) # bandpass
# playme(rep(s2, 5))
spectrogram(s2, 16000)  # more accurate than plotting the spectrum with plot = TRUE

# a short vector with some NAs
x = rnorm(150, 10) + 3 * sin((1:50) / 5)
x[sample(1:length(x), 50)] = NA
plot(x, type = 'l')
points(bandpass(x, samplingRate = 100, upr = 10), type = 'l', col = 'blue')

\dontrun{
# precise notch filtering is possible, even in low frequencies
whiteNoise = runif(8000, -1, 1)
s3 = bandpass(whiteNoise, 16000, lwr = 30, upr = 40,
              plot = TRUE, xlim = c(0, 500))
playme(rep(s3, 5))
spectrogram(s3, 16000, windowLength = 150, yScale = 'log')

# compare the same with STFT
s4 = seewave::ffilter(whiteNoise, f = 16000, from = 30, to = 40)
spectrogram(s4, 16000, windowLength = 150, yScale = 'log')
# (note: works better as wl approaches length(s4))

# high-pass all audio files in a folder
bandpass('~/Downloads/temp', saveAudio = '~/Downloads/temp/hp2000/',
         lwr = 2000, savePlots = '~/Downloads/temp/hp2000/')
}
}
