% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bernoulli_ARL.R
\name{bernoulli_ARL}
\alias{bernoulli_ARL}
\title{Average Run Length for Bernoulli CUSUM}
\usage{
bernoulli_ARL(h, n_grid, glmmod, theta, theta_true, p0, p1, method = c("MC",
  "SPRT"), smooth_prob = FALSE)
}
\arguments{
\item{h}{Control limit for the Bernoulli CUSUM}

\item{n_grid}{Number of state spaces used to discretize the outcome space (when \code{method = "MC"})
or number of grid points used for trapezoidal integration (when \code{method = "SPRT"}).
Increasing this number improves accuracy, but can also significantly increase computation time.}

\item{glmmod}{Generalized linear regression model used for risk-adjustment as produced by
the function \code{\link[stats:glm]{glm()}}. Suggested: \cr
\code{glm(as.formula("(survtime <= followup) & (censorid == 1) ~ covariates"), data = data)}. \cr
Alternatively, a list containing the following elements:
\describe{
\item{\code{formula}:}{a \code{\link[stats:formula]{formula()}} in the form \code{~ covariates};}
\item{\code{coefficients}:}{a named vector specifying risk adjustment coefficients
for covariates. Names must be the same as in \code{formula} and colnames of \code{data}.}
}}

\item{theta}{The \eqn{\theta}{\theta} value used to specify the odds ratio
\eqn{e^\theta}{e^\theta} under the alternative hypothesis.
If \eqn{\theta >= 0}{\theta >= 0}, the average run length for the upper one-sided
Bernoulli CUSUM will be determined. If \eqn{\theta < 0}{\theta < 0},
the average run length for the lower one-sided CUSUM will be determined.
Note that \deqn{p_1 = \frac{p_0 e^\theta}{1-p_0 +p_0 e^\theta}.}{p1 = (p0 * e^\theta)/(1-p0+p0 * e^\theta).}}

\item{theta_true}{The true log odds ratio \eqn{\theta}{\theta}, describing the
true increase in failure rate from the null-hypothesis. Default = log(1), indicating
no increase in failure rate.}

\item{p0}{The baseline failure probability at \code{entrytime + followup} for individuals.}

\item{p1}{The alternative hypothesis failure probability at \code{entrytime + followup} for individuals.}

\item{method}{The method used to obtain the average run length. Either "MC" for Markov Chain
or "SPRT" for SPRT methodology. Default = "MC".}

\item{smooth_prob}{Should the probability distribution of failure under the null distribution be smoothed?
Useful for small samples. Can only be TRUE when \code{glmmod} is supplied. Default = FALSE.}
}
\value{
A list containing:
\itemize{
\item \code{ARL_0}: A numeric value indicating the average run length in
number of outcomes when starting from state E_0.
\item \code{ARL}: A \code{data.frame} containing the average run length (#outcomes)
depending on the state in which the process starts \eqn{(E_0, E_1, \ldots, E_{n_{grid}-1})}{(E_0, E_1, ..., E_{n_grid-1})}
\describe{
\item{\code{start_val}:}{Starting value of the CUSUM, corresponding to the
discretized state spaces \eqn{E_{i}}{E_i};}
\item{\code{#outcomes}:}{ARL for the CUSUM with
initial value \code{start_val};}
}
\item \code{R}: A transition probability \code{matrix} containing the transition
probabilities between states \eqn{E_0, \ldots, E_{t-1}}{E_0, ..., E_{n_grid-1}}.
\eqn{R_{i,j}}{R_{i,j}} is the transition probability from state i to state j.
\item \code{h}: Value of the control limit.
} The value of \code{ARL_0} will be printed to the console.
}
\description{
This function allows to estimate the Average Run Length (ARL)
of the risk-adjusted Bernoulli CUSUM (see \code{\link[success:bernoulli_cusum]{bernoulli_cusum()}})
through a Markov Chain Approach (Brook & Evans(1972) & Steiner et al. (2000)) or
exploiting the relationship with the Sequential Probability Ratio Test (Kemp (1971)).
The function requires the specification of one of the following combinations of parameters
as arguments to the function:
\itemize{
\item \code{glmmod} & \code{theta}
\item \code{p0} & \code{theta}
\item \code{p0} & \code{p1}
} Average run length of lower-sided Bernoulli CUSUM charts can be determined
by specifying \code{theta} < 0.
}
\details{
The average run length of a CUSUM chart \eqn{S_n}{S_n} is given by
\eqn{E[\tau_n],}{E[\tau_n],} where \eqn{\tau_n}{\tau_n} is defined as:
\deqn{\tau_n = \inf\{n \geq 0: S_n \geq h\}.}{\tau_n = inf(n >= 0: S_n >= h).}

When \code{method = "MC"}, the average run length will be determined by
the Markov Chain approach described
in Brook & Evans (1972), using the risk-adjustment correction proposed in
Steiner et al. (2000). The idea is to discretize the domain (0, h) into \eqn{n_{grid} -1}{n_grid-1}
state spaces, with \eqn{E_0}{E_0} of width \eqn{w/2}{w/2}
and \eqn{E_1, \ldots, E_{n_{grid}-1}}{E_1, ..., E_{n_grid-1}} of width \eqn{w}{w}, such that
\eqn{E_{n_{grid}}}{E_{n_grid}} is an absorbing state.  This is done using the following steps:
\itemize{
\item \eqn{w}{w} is determined using the relationship \eqn{\frac{2h}{2t-1}}{2h/(2t-1)}.
\item Transition probabilities between the states are determined and
'transition matrix' \eqn{R}{R} is constructed.
\item The equation \eqn{(\bm{I}-\bm{R}) \bm{ARL} = \bm{1}}{(I-R) ARL = 1} is
solved to find the ARL starting from each of the states.
}

When \code{method = "SPRT"}, the average run length will be determined by
the relationship between the SPRT and CUSUM described in Kemp (1971), using the risk-adjustment
correction proposed in Steiner et al. (2000).
If N is the run length of a SPRT, P(0) the probability of
a SPRT terminating on the lower boundary of zero and R the run length of
a CUSUM, then: \deqn{E[R] = \frac{E[N]}{1 - P(0)}.}{E[R] = E[N]/(1-P(0)).}
\eqn{E[N]}{E[N]} and \eqn{P(0)}{P(0)} are completely determined by
\deqn{G_n(z) = \int_0^h F(z-w) dG_{n-1}(w)}{G_n(z) = \int_0^h F(z-w) dG_{n-1}(w)}
with \eqn{F(x)}{F(x)} the cdf of the singletons \eqn{W_n}{Wn}. The integral can be
approximated using the generalized trapezoidal quadrature rule:
\deqn{G_n(z) = \sum_{i=0}^{n_{grid}-1} \frac{F(z-x_{i+1}) + F(z-x_{i})}{2} \left(G_{n-1}(x_{i+1}) - G_{n-1}(x_{i})  \right)}{Gn(z) = \sum_{i=0}^{n_grid-1} (F(z-x_{i+1}) + F(z-x_{i}))/2 * (G_{n-1}(x_{i+1}) - G_{n-1}(x_{i}))}
}
\examples{
#Determine a risk-adjustment model using a generalized linear model.
#Outcome (failure within 100 days) is regressed on the available covariates:
glmmodber <- glm((survtime <= 100) & (censorid == 1)~ age + sex + BMI,
                  data = surgerydat, family = binomial(link = "logit"))
#Determine the Average Run Length in number of outcomes for
#control limit h = 2.5 with (0, h) divided into n_grid = 200 segments
ARL <- bernoulli_ARL(h = 2.5, n_grid = 200, glmmod = glmmodber, theta = log(2))
#Calculate ARL, but now exploiting connection between SPRT and CUSUM:
#n_grid now decides the accuracy of the Trapezoidal rule for integral approximation
ARLSPRT <- bernoulli_ARL(h = 2.5, n_grid = 200, glmmod = glmmodber,
theta = log(2), method = "SPRT")



}
\references{
Brook, D., & Evans, D. A. (1972). An Approach to the Probability
Distribution of CUSUM Run Length. Biometrika, 59(3), 539-549.
\doi{10.2307/2334805}

Steiner, S. H., Cook, R. J., Farewell, V. T., & Treasure, T. (2000).
Monitoring surgical performance using risk-adjusted cumulative sum charts.
Biostatistics, 1(4), 441-452. \doi{10.1093/biostatistics/1.4.441}

Kemp, K. W. (1971). Formal Expressions which Can Be Applied to Cusum Charts.
Journal of the Royal Statistical Society. Series B (Methodological), 33(3),
331-360. \doi{10.1111/j.2517-6161.1971.tb01521.x}
}
\seealso{
\code{\link[success]{bernoulli_cusum}}, \code{\link[success]{bernoulli_control_limit}}
}
\author{
Daniel Gomon
}
\concept{average run length}
