% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/time_seq_id.R
\name{time_seq_id}
\alias{time_seq_id}
\title{Generate a unique identifier for a regular time sequence with gaps}
\usage{
time_seq_id(
  x,
  timespan = granularity(x),
  threshold = 1,
  g = NULL,
  na_skip = TRUE,
  rolling = TRUE,
  switch_on_boundary = FALSE
)
}
\arguments{
\item{x}{Time vector. \cr
E.g. a \code{Date}, \code{POSIXt}, \code{numeric} or any time-based vector.}

\item{timespan}{\link{timespan}.}

\item{threshold}{Threshold such that when the time elapsed
exceeds this, the sequence ID is incremented by 1.
For example, if \code{timespan = "days"} and \code{threshold = 2},
then when 2 days have passed, a new ID is created.
Furthermore, \code{threshold} generally need not be supplied as \cr
\code{timespan = "3 days"} & \code{threshold = 1} \cr
is identical to \cr
\code{timespan = "days"} & \code{threshold = 3}. \cr}

\item{g}{Object used for grouping x.
This can for example be a vector or data frame.
\code{g} is passed directly to \code{collapse::GRP()}.}

\item{na_skip}{Should \code{NA} values be skipped? Default is \code{TRUE}.}

\item{rolling}{When this is \code{FALSE}, a new ID is created every time
a cumulative amount of time has passed. Once that amount of time has passed,
a new ID is created, the clock "resets" and we start counting from that point.}

\item{switch_on_boundary}{When an exact amount of time
(specified in \code{time_by}) has passed, should there an increment in ID?
The default is \code{FALSE}. For example, if \code{time_by = "days"} and
\code{switch_on_boundary = FALSE}, \code{>} 1 day must have passed, otherwise
\code{>=} 1 day must have passed.}
}
\value{
An integer vector of \code{length(x)}.
}
\description{
A unique identifier is created every time a specified amount of
time has passed, or in the case of regular sequences, when there is a gap
in time.
}
\details{
\code{time_seq_id()} Assumes \code{x} is regular and in
ascending or descending order.
To check this condition formally, use \code{time_is_regular()}.
}
\examples{
library(dplyr)
library(timeplyr)
library(lubridate)
\dontshow{
.n_dt_threads <- data.table::getDTthreads()
.n_collapse_threads <- collapse::get_collapse()$nthreads
data.table::setDTthreads(threads = 2L)
collapse::set_collapse(nthreads = 1L)
}
# Weekly sequence, with 2 gaps in between
x <- time_seq(today(), length.out = 10, time = "week")
x <- x[-c(3, 7)]
# A new ID when more than a week has passed since the last time point
time_seq_id(x)
# A new ID when >= 2 weeks has passed since the last time point
time_seq_id(x, threshold = 2, switch_on_boundary = TRUE)
# A new ID when at least 4 cumulative weeks have passed
time_seq_id(x, timespan = "4 weeks",
            switch_on_boundary = TRUE, rolling = FALSE)
# A new ID when more than 4 cumulative weeks have passed
time_seq_id(x, timespan = "4 weeks",
            switch_on_boundary = FALSE, rolling = FALSE)
\dontshow{
data.table::setDTthreads(threads = .n_dt_threads)
collapse::set_collapse(nthreads = .n_collapse_threads)
}
}
