## ----setup, echo=FALSE, message=FALSE, results="hide"-------------------------
knitr::opts_chunk$set(size = "small", prompt = TRUE, comment = NA,
                      out.width=".9\\linewidth")
knitr::knit_hooks$set(
  document = function(x) {sub('\\usepackage[]{color}', '\\usepackage{xcolor}',
                              x, fixed = TRUE)}
)
oldpar <- par(no.readonly = TRUE) ## NOTE: for setting back at the end
oldopt <- options()
options(prompt = "R> ", continue = "+  ")
options(width = 80, digits = 3)

## Dependencies
library("tramME")
library("ggplot2")
library("lme4")
library("survival")

## Functions
mycolors <- function(nr, type = "line") {
  cols <- list()
  cols[[1]] <- c(red = 0, green = 84, blue = 150)
  cols[[2]] <- c(red = 202, green = 108, blue = 24)
  out <- as.list(cols[[nr]])
  out$alpha <- switch(type, line = 255L, fill = 140L)
  out$maxColorValue <- 255
  do.call("rgb", out)
}

plot_cicomp <- function(est1, est2, labels, mnames, vline = 0, ylab = "",
                        col1 = mycolors(1), col2 = mycolors(2), xlas = 0,
                        legend_pos = "topleft") {
  stopifnot(all(dim(est1) == dim(est2)))
  nr <- nrow(est1)
  xs <- matrix(rep(c(-0.2, 0.2), nr) + rep(1:nr, each = 2), nrow = 2)
  plot(0, 0, type = "n", xlab = ylab, ylab = "",
       xlim = range(est1[, -1], est2[, -1]) + c(-0.2, 0.2),
       ylim = rev(range(xs) + c(-0.3, 0.3)), yaxt = "n", yaxs = "i")
  abline(h = seq(1.5, nr-0.5, by = 1), col = "grey", lty = 2)
  abline(v = vline)
  points(c(rbind(est1[, 1], est2[, 1])), c(xs),
         col = rep(c(col1, col2), nr), pch = 20)
  segments(x0 = est1[, 2], y0 = xs[1, ], x1 = est1[, 3], col = col1, lwd = 2)
  segments(x0 = est2[, 2], y0 = xs[2, ], x1 = est2[, 3], col = col2, lwd = 2)
  axis(2, at = 1:nr, labels = labels, tick = FALSE, las = xlas)
  if (!is.null(legend_pos)) {
    legend(legend_pos, mnames, col = c(col1, col2),
           pch = 20, lwd = 2, cex = 0.95, bg = "white")
  }
}

## ggplot theme
ggth <- theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
              panel.background = element_blank(),
              strip.background = element_rect(colour = "black", fill = "white"),
              axis.line = element_line(colour = "black"),
              legend.title = element_blank(),
              legend.key = element_rect(fill = "transparent", colour = "transparent"))

## ----sleeptudy-data, echo=FALSE, fig.width=7, fig.height=4--------------------
data("sleepstudy", package = "lme4")
dat <- within(sleepstudy, Subject <- paste0("Subject = ", Subject))
ggplot(dat, aes(x = Days, y = Reaction)) +
  facet_wrap(~ Subject, nrow = 3) +
  geom_point(col = "black", size = 0.75) +
  geom_line(col = "black") +
  scale_x_continuous("Days of sleep deprivation",
                     breaks = c(0, 2, 4, 6, 8)) +
  ylab("Average reaction time (ms)") +
  ggth

## ----message=FALSE------------------------------------------------------------
library("tramME")
sleep_lmME <- LmME(Reaction ~ Days + (Days | Subject), data = sleepstudy)
logLik(sleep_lmME)

## ----message=FALSE------------------------------------------------------------
library("lme4")
sleep_lmer <- lmer(Reaction ~ Days + (Days | Subject), data = sleepstudy,
                   REML = FALSE)
logLik(sleep_lmer)

## -----------------------------------------------------------------------------
cbind(coef = coef(sleep_lmME, as.lm = TRUE),
      se = sqrt(diag(vcov(sleep_lmME, as.lm = TRUE, pargroup = "fixef"))))

## -----------------------------------------------------------------------------
summary(sleep_lmer)$coefficients

## -----------------------------------------------------------------------------
VarCorr(sleep_lmME, as.lm = TRUE) ## random effects
sigma(sleep_lmME) ## residual SD
VarCorr(sleep_lmer)

## -----------------------------------------------------------------------------
library("survival")
ub <- ceiling(sleepstudy$Reaction / 50) * 50
lb <- floor(sleepstudy$Reaction / 50) * 50
lb[ub == 200] <- 0
sleepstudy$Reaction_ic <- Surv(lb, ub, type = "interval2")
head(sleepstudy$Reaction_ic)

## -----------------------------------------------------------------------------
sleep_lmME2 <- LmME(Reaction_ic ~ Days + (Days | Subject), data = sleepstudy)
logLik(sleep_lmME2)

## -----------------------------------------------------------------------------
cbind(coef = coef(sleep_lmME2, as.lm = TRUE),
      se = sqrt(diag(vcov(sleep_lmME2, as.lm = TRUE, pargroup = "fixef"))))
sigma(sleep_lmME2)
VarCorr(sleep_lmME2, as.lm = TRUE)

## -----------------------------------------------------------------------------
sleep_lmME3 <- LmME(Reaction_ic ~ Days + (Days || Subject), data = sleepstudy)
logLik(sleep_lmME3)

## -----------------------------------------------------------------------------
anova(sleep_lmME2, sleep_lmME3)

## -----------------------------------------------------------------------------
sleep_bc <- BoxCoxME(Reaction ~ Days + (Days | Subject), data = sleepstudy)
logLik(sleep_bc)

## ----sleepstudy-trafo, echo=FALSE,fig.width=4.5, fig.height=4, out.width="0.7\\linewidth"----
par(mfrow = c(1, 1), cex = 0.8)
tr_bc <- trafo(sleep_bc, type = "trafo", confidence = "none", K = 100)
plot(tr_bc, lty = 1, xlab = "Average reaction time (ms)", ylab = expression(h(y)),
     lwd = 2)
tr_lm <- trafo(sleep_lmME, type = "trafo", confidence = "none", K = 100)
plot(tr_lm, lty = 2, lwd = 2, add = TRUE)
rug(sleepstudy$Reaction, col = rgb(.1, .1, .1, .1))
legend("topleft", c("BoxCoxME", "LmME"), lty = c(1, 2), lwd = 2,
       bty = "n", cex = 0.8)

## ----eval=FALSE---------------------------------------------------------------
#  ndraws <- 1000
#  nd <- expand.grid(
#    Reaction = seq(min(sleepstudy$Reaction), max(sleepstudy$Reaction), length.out = 100),
#    Days = 0:9,
#    Subject = 1)
#  re <- simulate(sleep_bc, newdata = nd, nsim = ndraws, what = "ranef", seed = 100)
#  cp <- parallel::mclapply(re, function(x) {
#    predict(sleep_bc, newdata = nd, ranef = x, type = "distribution")
#  }, mc.cores = 8)
#  cp <- array(unlist(cp), dim = c(100, 10, ndraws))
#  mp_bc <- apply(cp, c(1, 2), mean)

## ----margdist-bc, echo=FALSE--------------------------------------------------
## Saved for faster execution
nd <- expand.grid(
  Reaction = seq(min(sleepstudy$Reaction), max(sleepstudy$Reaction), length.out = 100),
  Days = 0:9,
  Subject = 1)
mp_bc <-
structure(c(0.0077542315483954414, 0.010784884400333159, 0.014810066359617775,
0.020079340808101431, 0.026878959217792875, 0.035528618437561783,
0.046374066124735797, 0.059774654067956519, 0.076085591488462681,
0.095635631658505568, 0.11870197993879497, 0.14548499566053535,
0.17608549532439818, 0.21048707798703031, 0.24854506200733764,
0.28981980953409131, 0.33217157958146803, 0.37451409545635161,
0.41626623581475591, 0.45694673824953169, 0.49617320648448382,
0.53365636222345869, 0.56919143053165056, 0.60264799945187852,
0.63395929767698023, 0.66311154929596172, 0.69013385579714825,
0.71508889802288578, 0.73806462989338273, 0.75916704292277926,
0.77851401056360103, 0.7962301704721777, 0.81244276779004354,
0.82727836072203842, 0.84086027853590828, 0.85330671931801949,
0.86472937835336638, 0.87523250608137315, 0.88491230571302104,
0.8938565935480437, 0.90214465881689254, 0.90984727371333751,
0.91702681756534909, 0.92373749131227856, 0.93002560917904797,
0.93592996325664646, 0.94148226321957063, 0.94670765726482298,
0.95162534123648579, 0.95624926063018389, 0.96058890480243608,
0.96465018460997476, 0.96843637463480048, 0.97194909028213394,
0.97518925987902794, 0.97815804412440988, 0.98085765141868275,
0.98329199889147156, 0.98546717581041954, 0.98739167807326467,
0.98907639832780359, 0.99053437378663767, 0.99178031022284507,
0.99282991268002174, 0.9937003527064403, 0.99445535314437694,
0.99512923721199542, 0.99572939063158783, 0.99626269053302441,
0.99673552441759117, 0.99715380959677846, 0.99752301297189816,
0.99784817106405621, 0.99813391023409237, 0.99838446704962591,
0.99860370876299442, 0.99879515386173123, 0.9989619926446579,
0.99910710776407108, 0.99923309466018861, 0.99934228180002893,
0.99943675062094894, 0.99951835507044728, 0.99958874062942249,
0.99964936270628224, 0.99970150429417948, 0.99974629279288274,
0.99978471590981388, 0.99981763657080258, 0.99984580678923463,
0.99986988046154623, 0.99989042507650661, 0.99990793234457165,
0.99992282777103059, 0.9999354792120998, 0.9999462044660955,
0.99995527796206818, 0.99996293661568958, 0.99996938492678888,
0.99997479939489886, 0.0038670740811471531, 0.0055413020091851064,
0.0078363767821907279, 0.010935814938944223, 0.015060562150321533,
0.0204712747529581, 0.027468208009479911, 0.036387372035951236,
0.047591715345428609, 0.061456597895374838, 0.078349685560898372,
0.09860645570047176, 0.12250347052077139, 0.15023217251105989,
0.1818760126950841, 0.217251573918185, 0.25461699311880276, 0.29299880766991993,
0.33181356216891961, 0.37053904485852346, 0.40872422027927324,
0.4459927167385544, 0.48204182614188035, 0.51663853522465542,
0.54961370342729676, 0.5808551865789674, 0.61030047692835709,
0.63792926899246438, 0.66375624537306899, 0.6878242902908297,
0.71019827041107819, 0.73095946678341694, 0.75020069572755488,
0.76802211925093122, 0.78452771668399734, 0.79982236832062681,
0.81400948840805565, 0.82718913804975969, 0.83945654748250575,
0.85090098068524578, 0.86160488228262366, 0.87164325618013916,
0.88108323634969254, 0.88998382178671154, 0.89839575905006475,
0.90636156616467267, 0.91391570020901314, 0.92108487682609175,
0.92788855243051838, 0.93433957840462767, 0.94044503070267382,
0.94620720804273717, 0.95162477785913557, 0.95669403269718889,
0.96141020273813971, 0.96576875517717442, 0.9697666009892838,
0.97340312666344675, 0.97668097434374856, 0.97960650865683363,
0.98218993077904426, 0.98444502674705714, 0.98638856300804378,
0.98803936222531752, 0.98941916166440524, 0.99062504141803709,
0.99170950191759577, 0.99268270525769076, 0.99355419952764723,
0.99433293194287342, 0.99502726439001998, 0.99564499091518988,
0.99619335676031073, 0.99667907862194483, 0.99710836586699125,
0.99748694248954939, 0.99782006963213776, 0.99811256852277264,
0.99836884369806078, 0.99859290639292686, 0.99878839798169916,
0.99895861335499236, 0.99910652411414946, 0.99923480146174926,
0.99934583866442506, 0.99944177296417658, 0.99952450681732519,
0.99959572834668275, 0.99965693090243291, 0.99970943164039161,
0.99975438904219338, 0.99979281931983077, 0.99982561166604, 0.9998535423314312,
0.99987728752818794, 0.99989743517788754, 0.9999144955369077,
0.99992891074654056, 0.99994106336603217, 0.99995128395516397,
0.0020313431504307995, 0.002982333567236632, 0.004317636575617549,
0.0061643081848569209, 0.0086804231988383065, 0.012059299443387357,
0.016532925375230344, 0.022373528925393831, 0.029891984781906033,
0.039431815100717804, 0.051357968423325136, 0.066040302819788316,
0.083832570495308026, 0.10504847316996781, 0.1299368479811143,
0.15854204935026753, 0.18958816672097625, 0.2223171269323343,
0.25624153280759232, 0.29089086877176695, 0.32582809136961838,
0.36066048410685592, 0.39504583727666781, 0.42869500419869633,
0.46137175727510116, 0.49289070264355861, 0.52311385455757142,
0.55194633784027425, 0.57933158176554955, 0.60524628732545449,
0.62969538578134887, 0.65270715407005753, 0.67432860818341922,
0.69462125698213639, 0.71365726530653906, 0.73151604674095472,
0.74828128328242194, 0.76403835170472745, 0.77887212460790722,
0.79286510773485497, 0.80609587360608181, 0.81863775417180562,
0.83055776118621649, 0.84191571147149102, 0.85276354419799871,
0.86314482773889412, 0.87309446346721253, 0.8826386018855924,
0.891794791504121, 0.90057238173465926, 0.90897319674108823,
0.91699248702322389, 0.92462014945450954, 0.93184218528982199,
0.93864234102789468, 0.94500385165024492, 0.95091118314105627,
0.95635165513298559, 0.96131681851440665, 0.96580346926105187,
0.96981419912511524, 0.97335741413289334, 0.9764467882736203,
0.97910015463403566, 0.98134122433026083, 0.98331933680287387,
0.98511574707548033, 0.98674372009859324, 0.98821593750612491,
0.98954448243731719, 0.99074082957007192, 0.99181583997730383,
0.99277976043943894, 0.99364222686110293, 0.99441227144856981,
0.99509833330757314, 0.99570827212004087, 0.99624938455492551,
0.99672842306436704, 0.99715161671367769, 0.99752469369353969,
0.99785290516650305, 0.99814105010808041, 0.99839350081576461,
0.99861422877705786, 0.99880683060965125, 0.9989745538125353,
0.99912032209516566, 0.99924676008186664, 0.99935621721944612,
0.99945079074657206, 0.99953234761301168, 0.99960254526468495,
0.99966285123612031, 0.9997145615150056, 0.99975881766390651,
0.99979662270185177, 0.9998288557634234, 0.99985628556540207,
0.999879582721098, 0.0011641703273082228, 0.0017418485745337574,
0.0025657957599043192, 0.0037232349041215825, 0.0053252351848746852,
0.0075108633166159578, 0.010451209006580767, 0.014352692596769688,
0.019458788565575404, 0.026049137791324041, 0.034435096132149158,
0.044951128621887861, 0.057942038937319022, 0.073746644697612454,
0.092678964613490553, 0.11491688128387936, 0.13959080186204972,
0.16617565467987877, 0.19432704607855439, 0.22368698061273481,
0.2538993234052756, 0.28462213991967966, 0.31553709095779736,
0.34635614148774824, 0.37682588698121711, 0.4067298300188657,
0.43588895311674825, 0.46416092907141887, 0.49143828925375171,
0.51764583782007301, 0.54273756109005822, 0.56669324076738015,
0.5895149401606441, 0.61122349571008006, 0.631855112728268, 0.65145813471914604,
0.67009003021709568, 0.68781462001981686, 0.70469955116303029,
0.72081401209683782, 0.73622667623068949, 0.751003858112685,
0.7652078676287638, 0.77889555221198203, 0.79211702443883281,
0.80491458167850571, 0.81732183457793495, 0.8293630708289228,
0.84105288839426307, 0.85239613654684143, 0.86338820206283307,
0.87401567026585447, 0.88425737538546301, 0.89408583172869482,
0.90346900744433112, 0.91237236845911729, 0.92076108500032083,
0.92860226140513791, 0.93586702633148622, 0.94253230912230535,
0.94858213153528692, 0.9540082625622498, 0.95881011471020361,
0.96299379612676306, 0.96657567716642245, 0.96977851191118691,
0.97272465871037661, 0.97542911400455168, 0.97790655512331004,
0.98017129316161189, 0.98223723116377382, 0.98411782737435149,
0.98582606338658008, 0.98737441707592566, 0.98877484024432905,
0.99003874091794797, 0.99117697023779949, 0.99219981386077061,
0.99311698775156176, 0.99393763819881664, 0.99467034583601333,
0.99532313339465206, 0.99590347686845915, 0.99641831972646278,
0.99687408978266379, 0.99727671831218756, 0.99763166099875944,
0.99794392030553714, 0.99821806887936559, 0.99845827362541417,
0.99866832012254403, 0.99885163708720393, 0.9990113206328145,
0.99915015811043295, 0.99927065135334925, 0.99937503918198589,
0.99946531905536318, 0.99954326778119973, 0.99961046121857011,
0.99966829292531212, 0.00075093515071660965, 0.001135137815199426,
0.0016873744215252907, 0.0024692241588464025, 0.0035602218451490824,
0.0050613737282430744, 0.0070987198861287293, 0.0098266116869074587,
0.013430203638775604, 0.018126508387498092, 0.024163313075076603,
0.031815379922539778, 0.041377669416048229, 0.053155750097596918,
0.067453932776498096, 0.084491379096954686, 0.103683281721554,
0.12468551150753872, 0.14728045871839227, 0.17122713766119721,
0.1962717067248797, 0.22215663556039209, 0.24862863049215617,
0.27544536606920972, 0.30238101115611271, 0.32923052287465776,
0.35581270866000381, 0.38197210582985958, 0.40757977911077936,
0.43253317620409271, 0.45675520419783561, 0.48019269564264522,
0.50281442566603496, 0.5246088248384897, 0.54558151061930849,
0.56575273632433354, 0.58515483303764182, 0.60382969835786304,
0.62182636737954766, 0.63919868650036848, 0.65600309986037808,
0.67229655158782009, 0.68813450450375735, 0.70356907732805207,
0.71864730737546967, 0.73340955368733118, 0.74788806572853583,
0.7621057541136611, 0.77607521088323306, 0.78979803583988706,
0.80326453026498801, 0.81645381769940972, 0.82933444124682054,
0.84186546645856641, 0.85399808777107733, 0.86567769574258202,
0.87684631491277076, 0.88744527276008833, 0.89741791501872603,
0.90671214783215182, 0.91528256796250129, 0.92309194095762093,
0.93011180223897139, 0.93632198104078723, 0.94171708369180807,
0.9466097676293117, 0.95117426339768885, 0.95542473523800298,
0.95937542080271354, 0.96304055920721332, 0.9664343253290768,
0.9695707699514764, 0.97246376535937418, 0.97512695603190036,
0.9775737141234877, 0.97981709948192575, 0.98186982400633682,
0.98374422019629981, 0.98545221378066017, 0.98700530033841771,
0.98841452583365585, 0.98969047098247787, 0.99084323935439755,
0.99188244908648193, 0.99281722805925243, 0.99365621235244572,
0.99440754776953832, 0.99507889419519258, 0.99567743253148222,
0.99620987394802418, 0.99668247117823794, 0.99710103159833441,
0.99747093183612723, 0.99779713367176892, 0.99808420101020623,
0.9983363177237351, 0.99855730618090333, 0.99875064629389387,
0.99891949492964016, 0.99906670553991028, 0.00055784083816549771,
0.00084212207644364552, 0.0012497918948209698, 0.0018263751320818182,
0.0026311762727933377, 0.0037400612180356844, 0.0052482202957687199,
0.0072727232245752031, 0.0099546319605513592, 0.013460370726373112,
0.017981986005835907, 0.023735911292762486, 0.030959929491598554,
0.039908207411363056, 0.05084450624559126, 0.063979310297375191,
0.078907118142143601, 0.095400515834723143, 0.11332594702382419,
0.13252722134911349, 0.15283256453926952, 0.17406093121006658,
0.19602764119758431, 0.21854940298983158, 0.24144874952783987,
0.26455787341715853, 0.28772182863309664, 0.31080106670563101,
0.33367329188930028, 0.35623464442043046, 0.37840024683552853,
0.40010417061660647, 0.42129889639582413, 0.4419543496253534,
0.46205659531969079, 0.48160627133890194, 0.50061683124904932,
0.51911265673533569, 0.53712708743826931, 0.55470040431865375,
0.57187779236590963, 0.58870730051562714, 0.60523781165795631,
0.62151703399400093, 0.6375895269168792, 0.65349478002439443,
0.66926537252831864, 0.68492525161782947, 0.70048818127712864,
0.7159564261920599, 0.73131974672486111, 0.74655478800357999,
0.76162494611619169, 0.77648078431935097, 0.79106104960234969,
0.80529430349021536, 0.81910113097096571, 0.83239683052010016,
0.84509442146252167, 0.85710773950367392, 0.86835433523553029,
0.87875785088859704, 0.88824953135121565, 0.89676852449068067,
0.90427213450780519, 0.91116799684762628, 0.9176873369712556,
0.92384063677303907, 0.92963894650829526, 0.93509379111140323,
0.94021708060439158, 0.94502102504101038, 0.94951805434152703,
0.95372074328724965, 0.95764174186109741, 0.96129371104180938,
0.96468926408509703, 0.96784091325595245, 0.97076102191321423,
0.97346176179121768, 0.97595507527465652, 0.97825264242227594,
0.98036585246309493, 0.98230577946564268, 0.98408316186611344,
0.98570838553499485, 0.9871914700630785, 0.98854205795604499,
0.98976940644118538, 0.99088238160930731, 0.99188945463847278,
0.99279869987288683, 0.99361779455893473, 0.9943540200699974,
0.99501426448117392, 0.9956050263833659, 0.99613241985228873,
0.99660218051090166, 0.99701967264260949, 0.99738989732664574,
0.00047260153879679284, 0.00070776254208849033, 0.0010415886018056029,
0.0015093469794246856, 0.0021569704877916823, 0.0030432990074853723,
0.0042422550778044811, 0.0058447607530018476, 0.0079602274245390466,
0.010717498892954822, 0.014265176870239602, 0.018771281790558255,
0.024422184709989182, 0.031420691363972471, 0.039983091335223521,
0.050292065700296172, 0.062053163248106682, 0.075112253561352982,
0.089388263550982922, 0.10478028803946053, 0.12117218938497841,
0.13843713757796405, 0.15644193422211955, 0.17505100885911012,
0.19413001872924446, 0.21354901541663068, 0.23318516252045812,
0.25292500032549242, 0.27266626043964637, 0.29231923879337152,
0.31180774103191145, 0.33106962058335521, 0.35005693607604277,
0.36873576046903334, 0.38708567844023928, 0.40509901070026794,
0.4227798037691412, 0.44014262151611494, 0.45721117085088542,
0.47401678899549271, 0.49059681449827031, 0.50699285934805649,
0.52324899596169572, 0.53940987114109473, 0.55551875992166611,
0.57161557602059065, 0.58773486261002073, 0.60390379738679023,
0.62014025900843761, 0.63645101703238838, 0.65283012301974164,
0.66925759423115061, 0.68569849045510789, 0.70210248558562094,
0.71840402513144697, 0.7345231359233555, 0.75036691314755177,
0.76583165264530961, 0.78080552570773687, 0.79517161422860116,
0.8088110426825289, 0.82160586719272777, 0.83344131710207425,
0.84420693397220625, 0.85381092697743155, 0.86274568715360922,
0.87129607781750151, 0.87946629335575099, 0.88726149495860551,
0.89468773763697007, 0.90175189661027055, 0.9084615931942096,
0.9148251203289971, 0.92085136793439204, 0.92654974834892612,
0.93193012219658211, 0.93700272511364191, 0.94177809585046468,
0.9462670063277322, 0.9504803942662442, 0.95442929901814699,
0.95812480120279309, 0.9615779666924118, 0.96479979540424143,
0.96780117524201126, 0.97059284139766389, 0.97318534108228272,
0.97558900361201184, 0.97781391563887665, 0.97986990119547679,
0.98176650612280336, 0.98351298637646611, 0.98511829966094888,
0.98659109982468329, 0.9879397334594221, 0.98917223818260591,
0.99029634213693907, 0.99131946431209583, 0.99224871537388193,
0.9930908987707463, 0.0004417596556922092, 0.00065511387710996184,
0.00095473680177844092, 0.0013696946135349109, 0.0019371529296417275,
0.0027040404003192209, 0.0037287340305225533, 0.0050826020964284315,
0.0068512293123319514, 0.0091351920990321144, 0.012050345898123477,
0.015727693996981817, 0.020312965459280116, 0.025965981410683219,
0.032859723200193695, 0.041144349457226587, 0.050592649990531487,
0.061094767842535438, 0.072601355301504383, 0.085047370321446505,
0.098354577847288668, 0.11243440452437341, 0.12719099037663306,
0.14252425668907578, 0.15833283649155358, 0.17451676127592558,
0.19097984273727306, 0.20763172163719248, 0.22438957648851759,
0.24117949597687099, 0.25793752487839083, 0.27461039670340065,
0.29115596899371704, 0.30754337962484035, 0.32375294448910846,
0.33977581822525804, 0.35561343992207639, 0.37127678483127091,
0.38678544114000896, 0.40216652800825159, 0.41745346773252801,
0.43268462151300563, 0.44790179539769837, 0.46314862112507627,
0.47846881638159361, 0.49390433103704906, 0.50949339078850631,
0.52526845780231579, 0.54125413965444424, 0.55746509303010761,
0.57390398659491348, 0.59055960676138342, 0.60740520836666712,
0.62439722615113868, 0.64147446810785336, 0.65855790355057797,
0.67555113276406131, 0.69234157839594535, 0.7088023708590323,
0.72479481372293375, 0.74017121644315143, 0.75477777911660826,
0.76845711581204124, 0.78104991553567227, 0.79241278756039213,
0.80310032575360724, 0.81344028296506732, 0.82343004616770077,
0.83306811806950565, 0.84235406692172454, 0.85128847462785129,
0.85987288390206706, 0.86810974501965799, 0.87600236250248842,
0.88355484190173794, 0.89077203668839333, 0.89765949514731558,
0.90422340709836979, 0.91047055024029278, 0.9164082359287864,
0.92204425425594572, 0.92738681838705461, 0.93244450822440972,
0.93722621359591651, 0.94174107729780898, 0.94599843844480236,
0.95000777668702008, 0.95377865793212502, 0.957320682256387,
0.96064343469551938, 0.9637564395734165, 0.96666911895568752,
0.96939075570909516, 0.97193046151401208, 0.97429715002307382,
0.97649951519477074, 0.97854601466581304, 0.98044485787047786,
0.98220399847771567, 0.98383113060495553, 0.0004428610445394649,
0.00064964407316614578, 0.00093751665412552675, 0.0013320882551858947,
0.0018650397825651773, 0.0025751312301625132, 0.0035093138360919548,
0.004723886796523925, 0.0062855898397638794, 0.0082725024893414729,
0.01077464998994682, 0.0138942900550441, 0.017745936088813457,
0.022456204702420983, 0.02816351506707284, 0.034989193819245347,
0.042748046379494017, 0.051356432317689833, 0.06078269188114499,
0.07098358173945811, 0.081905597174750058, 0.093486599949826152,
0.10565772308915544, 0.1183454586486332, 0.13147381161992133,
0.14496641089366658, 0.15874849135821487, 0.17274868774073612,
0.18690060398071631, 0.20114413941551276, 0.21542656512037867,
0.22970335169768924, 0.24393875503357509, 0.25810617004407044,
0.27218826478209218, 0.28617690869873919, 0.3000729093918631,
0.31388557182351084, 0.32763209276990707, 0.3413368012810255,
0.35503025335487109, 0.36874818613966437, 0.38253033410484816,
0.39641910717649226, 0.41045812928526559, 0.4246906356549397,
0.43915772904076666, 0.45389649959904571, 0.46893802067612161,
0.48430524394799612, 0.50001083213196418, 0.51605498555651808,
0.5324233391300669, 0.54908502666805925, 0.56599102699224024,
0.58307291645622272, 0.60024215044792739, 0.61738997653715078,
0.63438803947885847, 0.65108967030401876, 0.66733175836283976,
0.68293699046686673, 0.69771611307882886, 0.71146974218144476,
0.72400866641467287, 0.73591945549279358, 0.74755644246698572,
0.75891090221486313, 0.76997514174582748, 0.78074250044515792,
0.79120734337054888, 0.80136504804953179, 0.81121198531075356,
0.82074549474993364, 0.82996385547935148, 0.83886625283590111,
0.84745274172569873, 0.8557242072629756, 0.86368232331880013,
0.87132950953370869, 0.87866888727158177, 0.88570423490514905,
0.89243994273229266, 0.89888096773306136, 0.90503278829637546,
0.91090135897835978, 0.91649306530590136, 0.92181467861255073,
0.92687331089090841, 0.93167636966579614, 0.93623151293361107,
0.94054660427136239, 0.94462966828860351, 0.94848884667035849,
0.95213235513228045, 0.95556844167370814, 0.95880534656365135,
0.96185126452370961, 0.96471430957665905, 0.96740248300782483,
0.00046785016485904266, 0.00067809984217098295, 0.00096838914337345498,
0.0013623446211294696, 0.0018880953556370185, 0.0025787069201502716,
0.0034727528756908856, 0.0046150609263250972, 0.0060576206608422885,
0.0078605826698282637, 0.010093235508834559, 0.012834835215953326,
0.016175186301195917, 0.020214918899510499, 0.025065445459539783,
0.030824755048922133, 0.037335828439023624, 0.04453172558991636,
0.052390807202392212, 0.060882827450239381, 0.069969860084527921,
0.07960734723014487, 0.089745290573164826, 0.10032957973263892,
0.11130342789674944, 0.1226088675276485, 0.13418825101637186,
0.14598570145746592, 0.15794846483580163, 0.17002812437154652,
0.18218164845847942, 0.19437225404544053, 0.20657007650468223,
0.21875264451565055, 0.23090516410664563, 0.24302061978662126,
0.25509970282828864, 0.26715057744941739, 0.27918849511852867,
0.29123526572431258, 0.30331859213705276, 0.31547127200546726,
0.32773026774432257, 0.34013564288166048, 0.35272936060265797,
0.36555393886250415, 0.37865095631881296, 0.39205940508399956,
0.40581389047096195, 0.41994268501363913, 0.43446565446134866,
0.44939208726533997, 0.46471847592880472, 0.48042631744365916,
0.49648001899848554, 0.51282501134056524, 0.52938618176416274,
0.5460667370574811, 0.56274758892005949, 0.57928731575239034,
0.59552269190747043, 0.61126968717558328, 0.62632472706989006,
0.64046587346580286, 0.6534738090264729, 0.66593754648391013,
0.67822085462591464, 0.69031196071987744, 0.70219970196762449,
0.71387355802743191, 0.72532368132743796, 0.7365409249384246,
0.74751686771499193, 0.75824383636700865, 0.76871492409336084,
0.77892400540202944, 0.78886574675699728, 0.79853561273427986,
0.80792986743520179, 0.81704557099165165, 0.8258805711004974,
0.83443348963630692, 0.84270370450600873, 0.85069132701875716,
0.85839717514235225, 0.86582274309808505, 0.87297016780458947,
0.87984219271544539, 0.88644212960433488, 0.89277381883676332,
0.89884158863177221, 0.90465021376521992, 0.91020487410353912,
0.91551111328933121, 0.92057479783363794, 0.92540207680948539,
0.92999934229167436, 0.93437319065173907, 0.93853038479603068,
0.94247781742892578), .Dim = c(100L, 10L))

## ----sleepstudy-margdist,echo=FALSE,message=FALSE,fig.width=7,fig.height=4,cache=TRUE----
## --- Lm
## re <- simulate(sleep_lmME, newdata = nd, nsim = ndraws, what = "ranef", seed = 100)
## cp <- parallel::mclapply(re, function(x) {
##   predict(sleep_lmME, newdata = nd, ranef = x, type = "distribution")
## }, mc.cores = 8)
## cp <- array(unlist(cp), dim = c(100, 10, ndraws))
## mp_lmME <- apply(cp, c(1, 2), mean)

## Saved for faster execution
mp_lmME <-
structure(c(0.051502692515938503, 0.060316536603317317, 0.07027481726447779,
0.081457512341298013, 0.093938425290456595, 0.10778298714503767,
0.12304602287438188, 0.13976955150130374, 0.1579806952902186,
0.17768977641467501, 0.19888867934414553, 0.22154955351185338,
0.24562392357372387, 0.27104226387210256, 0.29771407989803433,
0.32552852310370112, 0.35435554700997618, 0.38404759296535673,
0.41444177401053345, 0.44536250599891902, 0.47662451732672256,
0.50803615319694084, 0.53940287804719445, 0.57053087124437618,
0.60123060685117913, 0.63132030846224041, 0.66062917483170047,
0.68900028108067013, 0.71629307326754132, 0.74238539040241891,
0.76717496678760533, 0.79058038793353369, 0.81254149421871535,
0.83301924689995488, 0.85199509004786678, 0.86946985860257298,
0.88546229629661566, 0.90000725716127339, 0.91315367043286855,
0.92496235086009693, 0.93550373486369542, 0.94485561810071783,
0.95310096228552721, 0.96032582928017785, 0.96661748920977653,
0.97206273741078975, 0.9767464430657552, 0.98075034101602632,
0.98415206796120269, 0.98702443540557394, 0.98943492451891935,
0.99144538263319359, 0.99311189737585248, 0.99448482232983204,
0.99560892742568696, 0.99652364778537994, 0.99726340620093545,
0.99785798659470226, 0.99833293843572546, 0.99870999496844892,
0.99900749106738607, 0.99924076942109663, 0.99942256646206473,
0.99956337191955669, 0.99967175803261354, 0.99975467629665304,
0.99981772112606315, 0.99986536100828105, 0.99990113862493935,
0.99992784205312746, 0.99994764956952353, 0.99996225079896328,
0.99997294701357975, 0.99998073333389037, 0.99998636544102526,
0.99999041320778304, 0.99999330341919168, 0.99999535350014757,
0.99999679791358842, 0.99999780864862609, 0.99999851099167447,
0.99999899556944338, 0.99999932747289222, 0.99999955311619582,
0.99999970535349869, 0.99999980726687476, 0.99999987494916798,
0.99999991953270373, 0.99999994865673791, 0.99999996752053033,
0.99999997963295895, 0.99999998734171691, 0.99999999220374103,
0.99999999524224981, 0.99999999712350929, 0.99999999827727848,
0.99999999897810665, 0.9999999993996771, 0.99999999965077424,
0.99999999979884746, 0.029743714263824821, 0.035288473433720727,
0.041658275179856805, 0.048933691147462002, 0.057195419715225508,
0.066522768630955284, 0.076991965618111211, 0.088674328137865827,
0.10163433273201421, 0.11592763287101479, 0.13159908148961835,
0.14868081994017693, 0.16719049852027471, 0.18712969471161142,
0.20848259358242108, 0.23121499035838025, 0.25527366798817835,
0.28058619278072461, 0.30706115915773741, 0.3345889006494342,
0.36304266896448584, 0.39228026687315537, 0.42214610440105049,
0.45247363212381247, 0.48308809087352006, 0.51380950458830699,
0.54445583297678346, 0.57484619365989731, 0.60480405990521069,
0.63416034024023693, 0.66275625020320306, 0.69044589414836854,
0.71709848605722448, 0.74260015221526576, 0.76685527472884607,
0.78978735238099362, 0.81133937338144568, 0.83147371225337874,
0.85017157955202449, 0.86743206755698832, 0.88327084689258417,
0.8977185777623502, 0.91081910489280671, 0.92262750733443477,
0.93320807313024345, 0.94263226487147656, 0.95097673578799091,
0.95832144782932938, 0.96474793378605783, 0.97033773548163116,
0.97517103999336507, 0.97932552622489966, 0.98287542534623629,
0.98589079093122189, 0.98843696824081317, 0.99057424710916675,
0.99235767928734042, 0.9938370388141724, 0.9950569028904791,
0.99605683066999917, 0.99687161816758829, 0.9975316089349846,
0.99806304208486407, 0.99848842148588346, 0.99882689235481004,
0.9990946139086857, 0.99930511910605846, 0.99946965472111637,
0.99959749700021705, 0.99969623990960077, 0.99977205447703965,
0.99982991895518125, 0.99987382049926665, 0.99990692977503259,
0.99993175041896609, 0.99995024559172785, 0.99996394402740785,
0.99997402801713908, 0.99998140570474225, 0.99998676994109903,
0.99999064576609931, 0.99999342838181005, 0.99999541326368435,
0.9999968198402851, 0.99999780996492771, 0.99999850221077546,
0.99999898284779276, 0.99999931420719868, 0.99999954100688537,
0.99999969509883657, 0.99999979900543179, 0.99999986853375711,
0.9999999146936559, 0.99999994509416745, 0.9999999649523067,
0.99999997781604066, 0.99999998607825802, 0.99999999133914319,
0.99999999465951661, 0.99999999673641937, 0.018915708609449636,
0.02254159542100705, 0.026738331486430286, 0.031570452766962562,
0.037105083585634968, 0.0434111737063997, 0.050558582167302933,
0.058617012074670856, 0.067654806071835646, 0.077737618017924545,
0.088926982351487688, 0.10127880845006863, 0.11484183279451789,
0.12965606666124133, 0.14575128115013816, 0.16314557436993551,
0.18184406731993613, 0.20183777522659849, 0.22310269965043658,
0.24559918345850132, 0.26927156570865779, 0.29404816663845579,
0.31984162440517777, 0.34654959519051937, 0.37405581706537155,
0.40223152600139006, 0.43093720009050396, 0.46002459592628792,
0.48933902978217719, 0.51872184626392159, 0.54801300905832973,
0.57705374271924303, 0.60568915149272051, 0.63377074122020327,
0.66115877344707752, 0.68772438691060056, 0.71335143032028725,
0.73793796135126599, 0.76139737948765007, 0.78365917412084718,
0.80466928341459354, 0.82439007317658985, 0.84279995765280524,
0.85989269520117395, 0.87567640075046183, 0.89017232350385633,
0.90341344236643251, 0.91544293308407876, 0.92631256024828768,
0.9360810444358032, 0.94481245019021887, 0.95257463473918924,
0.95943779071395685, 0.9654731091164267, 0.97075158173752329,
0.9753429554819103, 0.97931484484041464, 0.98273200324273913,
0.98565574932191358, 0.9881435402726414, 0.99024868148517409,
0.99202015944595312, 0.99350258345275555, 0.99473622091680047,
0.99575711083053553, 0.99659724027805929, 0.99728476956659973,
0.99784429257646257, 0.9982971201822316, 0.99866157601416372,
0.9989532953364183, 0.99918551935587363, 0.99936937878747478,
0.99951416194346732, 0.99962756394754049, 0.99971591487307343,
0.99978438564867311, 0.99983717145406259, 0.99987765304313536,
0.99990853698362103, 0.99993197620500784, 0.99994967251319233,
0.99996296287976794, 0.9999728913654905, 0.99998026851115729,
0.99998571994407903, 0.99998972582211354, 0.99999265258531556,
0.99999477832060213, 0.9999963128777557, 0.99999741371337403,
0.99999819828843028, 0.99999875370831881, 0.99999914417331592,
0.99999941670267811, 0.99999960550658717, 0.99999973530564945,
0.99999982383614971, 0.99999988372905668, 0.99999992391021808,
0.013602785971022139, 0.016167725126909682, 0.019135854483643636,
0.022554654508571775, 0.026474346623675215, 0.030947564839391182,
0.03602893453935576, 0.041774555308553903, 0.048241386716717807,
0.055486538244484337, 0.06356646705047353, 0.072536090000201989,
0.08244781928614224, 0.093350534028217816, 0.10528850340846378,
0.11830028009592017, 0.1324175858662846, 0.14766421429409804,
0.16405497804153477, 0.18159473040797303, 0.20027749224005589,
0.22008571582810035, 0.24098971683757958, 0.2629473034809085,
0.2859036289132082, 0.30979128819507651, 0.33453067515016605,
0.3600306071947344, 0.38618921796543076, 0.41289510864429407,
0.44002873967344319, 0.46746403551383897, 0.49507016670212128,
0.52271346615627012, 0.55025943089151019, 0.5775747563765965,
0.60452934893294052, 0.63099826198362619, 0.65686350459734189,
0.6820156755165897, 0.70635538246321106, 0.72979441462723216,
0.7522566454388343, 0.77367865251972578, 0.79401005161452065,
0.81321355083511282, 0.83126474027729524, 0.84815163962543272,
0.86387403247411532, 0.87844262059244083, 0.89187803416813871,
0.90420973522577552, 0.91547485103424597, 0.92571697259348495,
0.93498495045935981, 0.94333171649913838, 0.9508131559460179,
0.95748704960765241, 0.96341210152215873, 0.96864706295014413,
0.97324995950355109, 0.97727742455588507, 0.98078413892336624,
0.98382237418206508, 0.98644163488790915, 0.98868839336445857,
0.99060590956964667, 0.9922341277893425, 0.99360964147124031,
0.99476571734786123, 0.9957323700487053, 0.99653647862317174,
0.99720193675085134, 0.99774982887560748, 0.9981986250428998,
0.99856438782959644, 0.99886098541865265, 0.99910030557523899,
0.99929246601353239, 0.99944601739048378, 0.99956813590885407,
0.9996648032397788, 0.99974097216771673, 0.99980071700111972,
0.99984736836574817, 0.99988363249249346, 0.99991169552010006,
0.99993331365199634, 0.99994989023676983, 0.99996254098902937,
0.9999721486403006, 0.99997940831944743, 0.9999848649215699,
0.99998894364634494, 0.9999919747837287, 0.99999421370784558,
0.99999585691796533, 0.99999705484588997, 0.99999792203695437,
0.99999854521041165, 0.010993845381845346, 0.012969102505081218,
0.015242451046749473, 0.017848296604537069, 0.020823240752817947,
0.024205946365580218, 0.028036953233428474, 0.032358440654153275,
0.037213934234839548, 0.042647954846985373, 0.048705608538754032,
0.05543211724079318, 0.062872291309345935, 0.071069946331411879,
0.080067268162737959, 0.089904131863381642, 0.10061738201173719,
0.11224008378120907, 0.1248007561094848, 0.13832260022348522,
0.15282273863755791, 0.16831148144076852, 0.18479163814229355,
0.20225789445264189, 0.22069627403557346, 0.24008370536077228,
0.26038771321352006, 0.28156625308003452, 0.30356770445389009,
0.32633103606247099, 0.34978615210081854, 0.37385442384682016,
0.39844940564098508, 0.42347772833335695, 0.44884015717470072,
0.47443279505131591, 0.500148406254458, 0.525877830965372, 0.55151145663529455,
0.57694070971108524, 0.6020595298961029, 0.62676578944791184,
0.65096262190415632, 0.67455962799641134, 0.69747393116436318,
0.71963106074480099, 0.7409656472517403, 0.76142192082820392,
0.78095401058189162, 0.7995260487908411, 0.81711208960798176,
0.83369585669894153, 0.84927033808314922, 0.86383724925482885,
0.87740638744333466, 0.88999490068992027, 0.90162649536680384,
0.91233060496422047, 0.92214154155276007, 0.9310976494233989,
0.93924047814300948, 0.94661398975703082, 0.95326381223264778,
0.95923654856517637, 0.96457914835872394, 0.96933834622144199,
0.97356016905710185, 0.97728951234641781, 0.98056978383577498,
0.98344261171171354, 0.98594761334072989, 0.98812221998009409,
0.99000155248312982, 0.99161834288464135, 0.99300289680309273,
0.99418309177729491, 0.99518440691144427, 0.99602997948638172,
0.99674068447175646, 0.9973352331219314, 0.99783028705006227,
0.99824058435370999, 0.9985790745241544, 0.99885705902769761,
0.99908433461902568, 0.99926933664993023, 0.99941927988195134,
0.9995402946026265, 0.9996375561787616, 0.99971540654682911,
0.99977746652577959, 0.99982673822395762, 0.99986569718162266,
0.99989637422745337, 0.99992042731816533, 0.99993920386603719,
0.99995379423541675, 0.99996507720645689, 0.99997375826658974,
0.99998040160481905, 0.0097711873376043808, 0.011407681305625784,
0.01327487049277723, 0.015397918542939083, 0.017803612773407057,
0.020520313893969015, 0.023577880090507215, 0.027007562867561057,
0.030841872012720033, 0.035114407113556012, 0.039859653260899998,
0.045112738940400667, 0.050909154666716014, 0.057284431656652188,
0.064273780758655447, 0.071911692930392485, 0.08023150374472808,
0.089264925659033684, 0.099041553051230385, 0.10958834625723851,
0.12092910199355286, 0.13308391857965926, 0.14606866526282852,
0.15989446567737325, 0.17456720603402567, 0.19008707902857438,
0.20644817467707666, 0.22363812931735341, 0.24163784384415804,
0.26042128183966018, 0.27995535758416545, 0.30019992294134368,
0.32110786076475134, 0.34262529073269737, 0.36469189136665187,
0.38724133942853645, 0.4102018649592471, 0.43349691698552523,
0.45704593149301048, 0.4807651897835688, 0.50456875197696394,
0.52836944737050384, 0.55207990082897029, 0.57561357252064549,
0.59888578729282227, 0.62181472989369879, 0.64432238313951462,
0.66633538797131708, 0.68778580705236592, 0.70861177697055011,
0.72875803802757499, 0.74817633478498324, 0.7668256847544086,
0.78467251663748994, 0.80169068314414449, 0.81786135649963365,
0.8331728172012306, 0.84762014837318234, 0.86120484921610696,
0.87393438161781312, 0.88582166407777541, 0.89688452679880026,
0.9071451412147401, 0.91662943643476202, 0.92536651415321802,
0.93338807253654943, 0.94072784847208668, 0.94742108635235533,
0.95350404027290314, 0.95901351514689093, 0.96398645080254963,
0.9684595516622948, 0.97246896315207443, 0.97604999461507103,
0.97923688726825886, 0.98206262470290018, 0.98455878263801166,
0.98675541411725365, 0.98868096609941114, 0.99036222341115954,
0.99182427626686875, 0.99309050795605924, 0.99418259978800083,
0.99512055089721341, 0.9959227109926625, 0.99660582452994795,
0.99718508506993442, 0.99767419874820018, 0.99808545582441044,
0.99842980923095781, 0.99871695892759982, 0.998955440729561,
0.99915271814610951, 0.99931527567517497, 0.99944871196888752,
0.99955783132660625, 0.99964673208734744, 0.99971889067466613,
0.99977724027911252, 0.99982424342717346, 0.009311599908882388,
0.010745727952448731, 0.012365056508567214, 0.014188251044984283,
0.016235163515630036, 0.018526820655979308, 0.021085398531025198,
0.02393418148124871, 0.027097503453712514, 0.030600669590979498,
0.034469855911513643, 0.038731984982550711, 0.043414575686445356,
0.048545565531858598, 0.054153104467683934, 0.060265319812455261,
0.066910052693461439, 0.074114567263972447, 0.081905234890991677,
0.09030719643226956, 0.099344006602858914, 0.1090372652262416,
0.11940624084001387, 0.1304674926597664, 0.14223449728776424,
0.15471728678718003, 0.16792210483842906, 0.18185108766721667,
0.19650197630110336, 0.21186786648714379, 0.22793700229667041,
0.24469261905699813, 0.26211284077909575, 0.28017063668472403,
0.29883384076217157, 0.31806523748207083, 0.33782271587198853,
0.3580594930747088, 0.37872440730281937, 0.39976227876426212,
0.42111433569395912, 0.44271870112058154, 0.46451093447048913,
0.48642462061630587, 0.50839199757511266, 0.530344612812781,
0.55221399707784768, 0.57393234392700743, 0.59543318266298417,
0.61665203232024057, 0.63752702462562838, 0.65799948453295587,
0.67801445796661641, 0.69752117777491962, 0.71647346053505512,
0.73483002869995451, 0.75255475455210719, 0.76961682444376867,
0.78599082376975138, 0.80165674495669215, 0.81659992239204593,
0.83081089960441179, 0.84428523511213771, 0.85702325416870195,
0.86902975416131845, 0.88031367169072783, 0.89088771941472011,
0.90076800062071827, 0.90997360924792281, 0.91852622274435469,
0.92644969474447192, 0.93376965410137436, 0.94051311630333223,
0.94670811273715283, 0.95238334261533464, 0.95756785164603953,
0.96229074068795628, 0.96658090670186314, 0.97046681730828332,
0.9739763192228732, 0.97713647981810003, 0.9799734601094563,
0.98251241664635691, 0.98477742915511945, 0.98679145037382832,
0.98857627435722928, 0.99015251961272621, 0.99153962373214022,
0.99275584666433625, 0.99381828037241093, 0.99474286326922656,
0.99554439845966725, 0.9962365753771657, 0.99683199483916562,
0.99734219783203037, 0.99777769846065367, 0.99814802147049397,
0.99846174459471537, 0.99872654573279762, 0.99894925467172335,
0.0092998607782912111, 0.010607850761829364, 0.012068586536096307,
0.013696103747475574, 0.015505322762666873, 0.01751205511400011,
0.019733001295843041, 0.022185738662256514, 0.024888698122700729,
0.027861128279016947, 0.031123045603469797, 0.034695169239024784,
0.038598839026981677, 0.042855915452387323, 0.047488660361849136,
0.052519597565200471, 0.057971352789616903, 0.063866472912009656,
0.070227224943721855, 0.077075375862552323, 0.084431955054728239,
0.092317001811165408, 0.10074930098213085, 0.10974611049571892,
0.11932288495455395, 0.12949299991386393, 0.14026748169275779,
0.15165474766924411, 0.16366036195866948, 0.17628681118496439,
0.18953330474258331, 0.20339560353812905, 0.21786588072068258,
0.23293261738457946, 0.24858053568004332, 0.26479057121244043,
0.28153988605986052, 0.29880192319457544, 0.3165465025543544,
0.3347399584690286, 0.35334531759983628, 0.37232251598982341,
0.3916286532537066, 0.41121828136187594, 0.43104372490932213,
0.45105542922529196, 0.47120233219610957, 0.49143225526461654,
0.51169230875476512, 0.53192930646207992, 0.55209018435433377,
0.57212241823649101, 0.59197443533539207, 0.61159601493226379,
0.63093867339135501, 0.64995602917926931, 0.66860414372595722,
0.68684183423808254, 0.7046309548424402, 0.72193664272502789,
0.73872752626161653, 0.75497589253237263, 0.77065781209857076,
0.78575321950813226, 0.8002459486917477, 0.81412372320124804,
0.82737810210113405, 0.8400043832151024, 0.85200146630659845,
0.86337167958850936, 0.87412057366845064, 0.8842566876083654,
0.893791292188585, 0.90273811570986928, 0.91111305774823881,
0.91893389621349708, 0.92621999287635837, 0.93299200224600831,
0.93927158832212188, 0.94508115332956633, 0.9504435820800119,
0.95538200509571836, 0.95991958307591307, 0.96407931468347718,
0.96788386898001544, 0.97135544314775901, 0.9745156454225572,
0.97738540244782568, 0.97998488957708563, 0.98233348204011672,
0.98444972438327227, 0.9863513152330462, 0.98805510424004184,
0.98957709805231986, 0.99093247234202952, 0.99213558725090523,
0.99320000409758602, 0.99413850175990315, 0.99496309175778719,
0.99568503166435629, 0.0095623510503293884, 0.010787464619134216,
0.012141044148200475, 0.013633785828106645, 0.015277067715644323,
0.017082964415061182, 0.019064255050257214, 0.021234423635404839,
0.023607651043369416, 0.026198797855965107, 0.029023377448451734,
0.032097518706908766, 0.035437917799728227, 0.039061778427090202,
0.042986739963979115, 0.047230792907153277, 0.051812181052631523,
0.056749289888159478, 0.062060520805342988, 0.067764150936655068,
0.07387817871598866, 0.080420155652472528, 0.08740700529046358,
0.094854830887295502, 0.10277871394651723, 0.11119250636016377,
0.12010861949408687, 0.12953781404747827, 0.13948899488517594,
0.14996901523986067, 0.16098249468339063, 0.17253165506058588,
0.18461617817099601, 0.19723308839901557, 0.21037666277076963,
0.22403837011033195, 0.2382068401373684, 0.25286786255262822,
0.26800441544952941, 0.28359672180982426, 0.29962233241287822,
0.31605623321687493, 0.33287097514398228, 0.35003682419269855,
0.36752192987187687, 0.3852925100606287, 0.40331305050716121,
0.42154651725611036, 0.43995458031883478, 0.45849784686926476,
0.47713610216812286, 0.49582855631084827, 0.51453409478642909,
0.53321153075428329, 0.55181985691844482, 0.57031849491780906,
0.58866754026050883, 0.6068280009985525, 0.62476202854233409,
0.64243313922149947, 0.65980642537393275, 0.67684875485619089,
0.69352895789408286, 0.70981800012264706, 0.72568914050936451,
0.74111807263882945, 0.75608304760180578, 0.77056497652647549,
0.78454751066691031, 0.79801709697036893, 0.81096300721526826,
0.82337733916198552, 0.83525498868403558, 0.84659359252166866,
0.85739344207916712, 0.86765736951425632, 0.87739060818075509,
0.88660063022394908, 0.89529696474175091, 0.90349100037786445,
0.91119577648810079, 0.91842576711781365, 0.92519666196294192,
0.93152514828642441, 0.93742869745885382, 0.9429253594205268,
0.94803356795075766, 0.95277195920112356, 0.9571592055152458,
0.96121386612366644, 0.96495425586727379, 0.96839833266264752,
0.97156360397423835, 0.97446705210170104, 0.97712507763171419,
0.97955345995455045, 0.98176733332455879, 0.98378117657284792,
0.98560881428346536, 0.9872634270433327, 0.0099955792942231208,
0.011162480802793201, 0.012439012138297472, 0.013833432908993405,
0.015354545084896355, 0.017011711648771112, 0.018814869506506477,
0.020774535816768757, 0.02290180706348845, 0.025208350380234345,
0.027706386826773052, 0.030408666496832094, 0.033328435483796832,
0.036479394831372421, 0.039875651637279005, 0.043531662454747773,
0.047462169052418693, 0.051682126461040097, 0.05620662307713982,
0.061050792439173295, 0.066229716175086148, 0.071758317577365782,
0.077651245324773674, 0.083922747063359018, 0.090586532895156738,
0.097655629297927121, 0.10514222459310116, 0.11305750775468948,
0.12141150305777798, 0.13021290373964522, 0.13946890842344667,
0.14918506447029758, 0.15936512262650379, 0.17001090728079829,
0.18112220632504009, 0.19269668402843745, 0.20472981952212685,
0.21721487250263594, 0.230142876671942, 0.24350266032226833,
0.25728089243221397, 0.27146215174843075, 0.28602901565140088,
0.30096216519168034, 0.31624050255592429, 0.33184127737416269,
0.34774021867878618, 0.36391166991612311, 0.38032872512129101,
0.39696336511523728, 0.41378659328879464, 0.4307685711303833,
0.44787875407656208, 0.46508602848557085, 0.48235885054622663,
0.49966538775675112, 0.51697366328122341, 0.53425170307276681,
0.55146768520816869, 0.56859009047364117, 0.58558785293187288,
0.60243050902531869, 0.61908834374705857, 0.63553253253138753,
0.65173527775222184, 0.66766993902084459, 0.68331115678750165,
0.69863496901428701, 0.71361892084762801, 0.72824216724170521,
0.74248556835540624, 0.75633177727624379, 0.76976531925025593,
0.78277266117251021, 0.79534226968556421, 0.80746465591323002,
0.8191324046867563, 0.83034018614609495, 0.84108474784271248,
0.85136488592771287, 0.86118139464841281, 0.87053699414371388,
0.87943623735420984, 0.88788539767138275, 0.89589233967007653,
0.90346637584106926, 0.91061811262729575, 0.9173592892524477,
0.92370261282196586, 0.92966159300122286, 0.93525037927515686,
0.94048360341532622, 0.94537622937065136, 0.94994341239572078,
0.95420036886202064, 0.95816225787551601, 0.96184407554804019,
0.96526056252911074, 0.96842612518217763, 0.9713547705669453), .Dim = c(100L,
10L))

dat <- nd
dat$Days <- paste0("Days = ", dat$Days)
dat$mp_lm <- c(mp_lmME)
dat$mp_bc <- c(mp_bc)
dat2 <- sleepstudy
dat2$Days <- paste0("Days = ", dat2$Days)

ggplot(dat, aes(x = Reaction)) +
  facet_wrap(~ Days, nrow = 2) +
  geom_line(aes(y = mp_bc, colour = "BoxCoxME")) +
  geom_line(aes(y = mp_lm, colour = "LmME")) +
  stat_ecdf(aes(x = Reaction, colour = "ECDF"), data = dat2, geom = "step") +
  xlab("Average reaction time (ms)") +
  ylab("Marginal distribution") +
  ggth +
  theme(legend.position = "bottom") +
  scale_color_manual(
    values = c(rgb(0, 84, 150, maxColorValue = 255),
               rgb(202, 108, 24, maxColorValue = 255),
               rgb(.5, .5, .5, .5)),
    breaks = c("BoxCoxME", "LmME", "ECDF"))

## ----neckpain-data, echo=FALSE, fig.width=6, fig.height=3---------------------
data("neck_pain", package = "ordinalCont")
dat <- within(neck_pain, {
  trt <- ifelse(laser == 1, "Active", "Placebo")
  t   <- ifelse(time == 1, "Baseline",
         ifelse(time == 2, "Week 7", "Week 12"))
})
ggplot(dat, aes(t, vas, group = id)) +
  facet_wrap(~ trt) +
  geom_line(col = "grey", alpha = 0.5) +
  geom_point(col = "grey", alpha = 0.5, size = 0.75) +
  scale_x_discrete(limits = c("Baseline", "Week 7", "Week 12")) +
  xlab("Follow-up times") +
  ylab("Pain level (VAS)") +
  ggth

## -----------------------------------------------------------------------------
neck_tr <- ColrME(vas ~ laser * time + (1 | id), data = neck_pain,
                  bounds = c(0, 1), support = c(0, 1))

## ----results="hide", message=FALSE, cache=TRUE--------------------------------
library("ordinalCont")
neck_ocm <- ocm(vas ~ laser * time + (1 | id), data = neck_pain, scale = c(0, 1))

## ----neckpain-compestim, echo=FALSE,fig.width=7, fig.height=3.5---------------
par(mfrow = c(1, 2), cex = 0.8)
## -- baseline
h_tr <- trafo(neck_tr, type = "trafo", confidence = "interval", K = 100)
h_ocm <- ordinalCont:::get_gfun.ocm(neck_ocm)
plot(h_tr, col = mycolors(1), fill = mycolors(1, "fill"),
     xlab = "Pain level (VAS)", ylab = expression(h(y)), lwd = 2)
polygon(c(h_ocm[, 1], rev(h_ocm[, 1])), c(h_ocm[, 3], rev(h_ocm[, 4])), border = NA,
        col = mycolors(2, "fill"))
lines(h_ocm[, 1], h_ocm[, 2], col = mycolors(2), lwd = 2)
legend("topleft", c("tramME", "ordinalCont"),
       col = c(mycolors(1), mycolors(2)),
       lty = 1, lwd = 2, bty = "n")
rug(neck_pain$vas, col = rgb(.1, .1, .1, .1))
## --- lORs
idx_ocm <- 2:6
ci_ocm <- cbind(coef(neck_ocm)[idx_ocm], confint(neck_ocm)[idx_ocm, ])
ci_tr <- cbind(coef(neck_tr), confint(neck_tr, pargroup = "shift"))
labs <- c(expression(beta[Active]), expression(beta["7w"]), expression(beta["12w"]),
          expression(beta["7w, Active"]), expression(beta["12w, Active"]))
#par(cex = 0.8, mar = c(4, 10, 2, 2))
plot_cicomp(ci_tr, ci_ocm, labels = labs, xlas = 1,
            mnames = c("tramME", "ordinalCont"), ylab = "Log-odds ratios",
            legend_pos = NULL)

## -----------------------------------------------------------------------------
exp(coef(neck_tr))

## ----eval=FALSE---------------------------------------------------------------
#  jointCDF <- function(re, nd, mod) {
#    nd <- nd[rep(1, length(re)), ]
#    nd$id <- seq(nrow(nd)) ## to take vector-valued REs
#    pr <- predict(mod, newdata = nd, ranef = re, type = "distribution") *
#      dnorm(re, 0, sd = sqrt(varcov(mod)[[1]][1, 1]))
#    c(pr)
#  }
#  marginalCDF <- function(nd, mod) {
#    nd$cdf <- integrate(jointCDF, lower = -Inf, upper = Inf, nd = nd, mod = mod)$value
#    nd
#  }
#  nd <- expand.grid(vas = seq(0, 1, length.out = 100),
#                    time = unique(neck_pain$time),
#                    laser = unique(neck_pain$laser))
#  mp_colr <- parallel::mclapply(split(nd, seq(nrow(nd))),
#                                marginalCDF, mod = neck_tr, mc.cores = 8)
#  mp_colr <- do.call("rbind", mp_colr)

## ----margdist-colr, echo=FALSE------------------------------------------------
## Saved for faster execution
mp_colr <-
structure(list(vas = c(0, 0.010101010101010102, 0.020202020202020204,
0.030303030303030304, 0.040404040404040407, 0.050505050505050511,
0.060606060606060608, 0.070707070707070718, 0.080808080808080815,
0.090909090909090912, 0.10101010101010102, 0.11111111111111112,
0.12121212121212122, 0.13131313131313133, 0.14141414141414144,
0.15151515151515152, 0.16161616161616163, 0.17171717171717174,
0.18181818181818182, 0.19191919191919193, 0.20202020202020204,
0.21212121212121213, 0.22222222222222224, 0.23232323232323235,
0.24242424242424243, 0.25252525252525254, 0.26262626262626265,
0.27272727272727276, 0.28282828282828287, 0.29292929292929293,
0.30303030303030304, 0.31313131313131315, 0.32323232323232326,
0.33333333333333337, 0.34343434343434348, 0.35353535353535359,
0.36363636363636365, 0.37373737373737376, 0.38383838383838387,
0.39393939393939398, 0.40404040404040409, 0.4141414141414142,
0.42424242424242425, 0.43434343434343436, 0.44444444444444448,
0.45454545454545459, 0.4646464646464647, 0.47474747474747481,
0.48484848484848486, 0.49494949494949497, 0.50505050505050508,
0.51515151515151525, 0.5252525252525253, 0.53535353535353536,
0.54545454545454553, 0.55555555555555558, 0.56565656565656575,
0.5757575757575758, 0.58585858585858586, 0.59595959595959602,
0.60606060606060608, 0.61616161616161624, 0.6262626262626263,
0.63636363636363646, 0.64646464646464652, 0.65656565656565657,
0.66666666666666674, 0.6767676767676768, 0.68686868686868696,
0.69696969696969702, 0.70707070707070718, 0.71717171717171724,
0.72727272727272729, 0.73737373737373746, 0.74747474747474751,
0.75757575757575768, 0.76767676767676774, 0.77777777777777779,
0.78787878787878796, 0.79797979797979801, 0.80808080808080818,
0.81818181818181823, 0.8282828282828284, 0.83838383838383845,
0.84848484848484851, 0.85858585858585867, 0.86868686868686873,
0.8787878787878789, 0.88888888888888895, 0.89898989898989912,
0.90909090909090917, 0.91919191919191923, 0.92929292929292939,
0.93939393939393945, 0.94949494949494961, 0.95959595959595967,
0.96969696969696972, 0.97979797979797989, 0.98989898989898994,
1, 0, 0.010101010101010102, 0.020202020202020204, 0.030303030303030304,
0.040404040404040407, 0.050505050505050511, 0.060606060606060608,
0.070707070707070718, 0.080808080808080815, 0.090909090909090912,
0.10101010101010102, 0.11111111111111112, 0.12121212121212122,
0.13131313131313133, 0.14141414141414144, 0.15151515151515152,
0.16161616161616163, 0.17171717171717174, 0.18181818181818182,
0.19191919191919193, 0.20202020202020204, 0.21212121212121213,
0.22222222222222224, 0.23232323232323235, 0.24242424242424243,
0.25252525252525254, 0.26262626262626265, 0.27272727272727276,
0.28282828282828287, 0.29292929292929293, 0.30303030303030304,
0.31313131313131315, 0.32323232323232326, 0.33333333333333337,
0.34343434343434348, 0.35353535353535359, 0.36363636363636365,
0.37373737373737376, 0.38383838383838387, 0.39393939393939398,
0.40404040404040409, 0.4141414141414142, 0.42424242424242425,
0.43434343434343436, 0.44444444444444448, 0.45454545454545459,
0.4646464646464647, 0.47474747474747481, 0.48484848484848486,
0.49494949494949497, 0.50505050505050508, 0.51515151515151525,
0.5252525252525253, 0.53535353535353536, 0.54545454545454553,
0.55555555555555558, 0.56565656565656575, 0.5757575757575758,
0.58585858585858586, 0.59595959595959602, 0.60606060606060608,
0.61616161616161624, 0.6262626262626263, 0.63636363636363646,
0.64646464646464652, 0.65656565656565657, 0.66666666666666674,
0.6767676767676768, 0.68686868686868696, 0.69696969696969702,
0.70707070707070718, 0.71717171717171724, 0.72727272727272729,
0.73737373737373746, 0.74747474747474751, 0.75757575757575768,
0.76767676767676774, 0.77777777777777779, 0.78787878787878796,
0.79797979797979801, 0.80808080808080818, 0.81818181818181823,
0.8282828282828284, 0.83838383838383845, 0.84848484848484851,
0.85858585858585867, 0.86868686868686873, 0.8787878787878789,
0.88888888888888895, 0.89898989898989912, 0.90909090909090917,
0.91919191919191923, 0.92929292929292939, 0.93939393939393945,
0.94949494949494961, 0.95959595959595967, 0.96969696969696972,
0.97979797979797989, 0.98989898989898994, 1, 0, 0.010101010101010102,
0.020202020202020204, 0.030303030303030304, 0.040404040404040407,
0.050505050505050511, 0.060606060606060608, 0.070707070707070718,
0.080808080808080815, 0.090909090909090912, 0.10101010101010102,
0.11111111111111112, 0.12121212121212122, 0.13131313131313133,
0.14141414141414144, 0.15151515151515152, 0.16161616161616163,
0.17171717171717174, 0.18181818181818182, 0.19191919191919193,
0.20202020202020204, 0.21212121212121213, 0.22222222222222224,
0.23232323232323235, 0.24242424242424243, 0.25252525252525254,
0.26262626262626265, 0.27272727272727276, 0.28282828282828287,
0.29292929292929293, 0.30303030303030304, 0.31313131313131315,
0.32323232323232326, 0.33333333333333337, 0.34343434343434348,
0.35353535353535359, 0.36363636363636365, 0.37373737373737376,
0.38383838383838387, 0.39393939393939398, 0.40404040404040409,
0.4141414141414142, 0.42424242424242425, 0.43434343434343436,
0.44444444444444448, 0.45454545454545459, 0.4646464646464647,
0.47474747474747481, 0.48484848484848486, 0.49494949494949497,
0.50505050505050508, 0.51515151515151525, 0.5252525252525253,
0.53535353535353536, 0.54545454545454553, 0.55555555555555558,
0.56565656565656575, 0.5757575757575758, 0.58585858585858586,
0.59595959595959602, 0.60606060606060608, 0.61616161616161624,
0.6262626262626263, 0.63636363636363646, 0.64646464646464652,
0.65656565656565657, 0.66666666666666674, 0.6767676767676768,
0.68686868686868696, 0.69696969696969702, 0.70707070707070718,
0.71717171717171724, 0.72727272727272729, 0.73737373737373746,
0.74747474747474751, 0.75757575757575768, 0.76767676767676774,
0.77777777777777779, 0.78787878787878796, 0.79797979797979801,
0.80808080808080818, 0.81818181818181823, 0.8282828282828284,
0.83838383838383845, 0.84848484848484851, 0.85858585858585867,
0.86868686868686873, 0.8787878787878789, 0.88888888888888895,
0.89898989898989912, 0.90909090909090917, 0.91919191919191923,
0.92929292929292939, 0.93939393939393945, 0.94949494949494961,
0.95959595959595967, 0.96969696969696972, 0.97979797979797989,
0.98989898989898994, 1, 0, 0.010101010101010102, 0.020202020202020204,
0.030303030303030304, 0.040404040404040407, 0.050505050505050511,
0.060606060606060608, 0.070707070707070718, 0.080808080808080815,
0.090909090909090912, 0.10101010101010102, 0.11111111111111112,
0.12121212121212122, 0.13131313131313133, 0.14141414141414144,
0.15151515151515152, 0.16161616161616163, 0.17171717171717174,
0.18181818181818182, 0.19191919191919193, 0.20202020202020204,
0.21212121212121213, 0.22222222222222224, 0.23232323232323235,
0.24242424242424243, 0.25252525252525254, 0.26262626262626265,
0.27272727272727276, 0.28282828282828287, 0.29292929292929293,
0.30303030303030304, 0.31313131313131315, 0.32323232323232326,
0.33333333333333337, 0.34343434343434348, 0.35353535353535359,
0.36363636363636365, 0.37373737373737376, 0.38383838383838387,
0.39393939393939398, 0.40404040404040409, 0.4141414141414142,
0.42424242424242425, 0.43434343434343436, 0.44444444444444448,
0.45454545454545459, 0.4646464646464647, 0.47474747474747481,
0.48484848484848486, 0.49494949494949497, 0.50505050505050508,
0.51515151515151525, 0.5252525252525253, 0.53535353535353536,
0.54545454545454553, 0.55555555555555558, 0.56565656565656575,
0.5757575757575758, 0.58585858585858586, 0.59595959595959602,
0.60606060606060608, 0.61616161616161624, 0.6262626262626263,
0.63636363636363646, 0.64646464646464652, 0.65656565656565657,
0.66666666666666674, 0.6767676767676768, 0.68686868686868696,
0.69696969696969702, 0.70707070707070718, 0.71717171717171724,
0.72727272727272729, 0.73737373737373746, 0.74747474747474751,
0.75757575757575768, 0.76767676767676774, 0.77777777777777779,
0.78787878787878796, 0.79797979797979801, 0.80808080808080818,
0.81818181818181823, 0.8282828282828284, 0.83838383838383845,
0.84848484848484851, 0.85858585858585867, 0.86868686868686873,
0.8787878787878789, 0.88888888888888895, 0.89898989898989912,
0.90909090909090917, 0.91919191919191923, 0.92929292929292939,
0.93939393939393945, 0.94949494949494961, 0.95959595959595967,
0.96969696969696972, 0.97979797979797989, 0.98989898989898994,
1, 0, 0.010101010101010102, 0.020202020202020204, 0.030303030303030304,
0.040404040404040407, 0.050505050505050511, 0.060606060606060608,
0.070707070707070718, 0.080808080808080815, 0.090909090909090912,
0.10101010101010102, 0.11111111111111112, 0.12121212121212122,
0.13131313131313133, 0.14141414141414144, 0.15151515151515152,
0.16161616161616163, 0.17171717171717174, 0.18181818181818182,
0.19191919191919193, 0.20202020202020204, 0.21212121212121213,
0.22222222222222224, 0.23232323232323235, 0.24242424242424243,
0.25252525252525254, 0.26262626262626265, 0.27272727272727276,
0.28282828282828287, 0.29292929292929293, 0.30303030303030304,
0.31313131313131315, 0.32323232323232326, 0.33333333333333337,
0.34343434343434348, 0.35353535353535359, 0.36363636363636365,
0.37373737373737376, 0.38383838383838387, 0.39393939393939398,
0.40404040404040409, 0.4141414141414142, 0.42424242424242425,
0.43434343434343436, 0.44444444444444448, 0.45454545454545459,
0.4646464646464647, 0.47474747474747481, 0.48484848484848486,
0.49494949494949497, 0.50505050505050508, 0.51515151515151525,
0.5252525252525253, 0.53535353535353536, 0.54545454545454553,
0.55555555555555558, 0.56565656565656575, 0.5757575757575758,
0.58585858585858586, 0.59595959595959602, 0.60606060606060608,
0.61616161616161624, 0.6262626262626263, 0.63636363636363646,
0.64646464646464652, 0.65656565656565657, 0.66666666666666674,
0.6767676767676768, 0.68686868686868696, 0.69696969696969702,
0.70707070707070718, 0.71717171717171724, 0.72727272727272729,
0.73737373737373746, 0.74747474747474751, 0.75757575757575768,
0.76767676767676774, 0.77777777777777779, 0.78787878787878796,
0.79797979797979801, 0.80808080808080818, 0.81818181818181823,
0.8282828282828284, 0.83838383838383845, 0.84848484848484851,
0.85858585858585867, 0.86868686868686873, 0.8787878787878789,
0.88888888888888895, 0.89898989898989912, 0.90909090909090917,
0.91919191919191923, 0.92929292929292939, 0.93939393939393945,
0.94949494949494961, 0.95959595959595967, 0.96969696969696972,
0.97979797979797989, 0.98989898989898994, 1, 0, 0.010101010101010102,
0.020202020202020204, 0.030303030303030304, 0.040404040404040407,
0.050505050505050511, 0.060606060606060608, 0.070707070707070718,
0.080808080808080815, 0.090909090909090912, 0.10101010101010102,
0.11111111111111112, 0.12121212121212122, 0.13131313131313133,
0.14141414141414144, 0.15151515151515152, 0.16161616161616163,
0.17171717171717174, 0.18181818181818182, 0.19191919191919193,
0.20202020202020204, 0.21212121212121213, 0.22222222222222224,
0.23232323232323235, 0.24242424242424243, 0.25252525252525254,
0.26262626262626265, 0.27272727272727276, 0.28282828282828287,
0.29292929292929293, 0.30303030303030304, 0.31313131313131315,
0.32323232323232326, 0.33333333333333337, 0.34343434343434348,
0.35353535353535359, 0.36363636363636365, 0.37373737373737376,
0.38383838383838387, 0.39393939393939398, 0.40404040404040409,
0.4141414141414142, 0.42424242424242425, 0.43434343434343436,
0.44444444444444448, 0.45454545454545459, 0.4646464646464647,
0.47474747474747481, 0.48484848484848486, 0.49494949494949497,
0.50505050505050508, 0.51515151515151525, 0.5252525252525253,
0.53535353535353536, 0.54545454545454553, 0.55555555555555558,
0.56565656565656575, 0.5757575757575758, 0.58585858585858586,
0.59595959595959602, 0.60606060606060608, 0.61616161616161624,
0.6262626262626263, 0.63636363636363646, 0.64646464646464652,
0.65656565656565657, 0.66666666666666674, 0.6767676767676768,
0.68686868686868696, 0.69696969696969702, 0.70707070707070718,
0.71717171717171724, 0.72727272727272729, 0.73737373737373746,
0.74747474747474751, 0.75757575757575768, 0.76767676767676774,
0.77777777777777779, 0.78787878787878796, 0.79797979797979801,
0.80808080808080818, 0.81818181818181823, 0.8282828282828284,
0.83838383838383845, 0.84848484848484851, 0.85858585858585867,
0.86868686868686873, 0.8787878787878789, 0.88888888888888895,
0.89898989898989912, 0.90909090909090917, 0.91919191919191923,
0.92929292929292939, 0.93939393939393945, 0.94949494949494961,
0.95959595959595967, 0.96969696969696972, 0.97979797979797989,
0.98989898989898994, 1), time = structure(c(1L, 1L, 1L, 1L, 1L,
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L,
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L,
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L,
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L,
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L,
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 2L,
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L,
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L,
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L,
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L,
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L,
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L,
2L, 2L, 2L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L,
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L,
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L,
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L,
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L,
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L,
3L, 3L, 3L, 3L, 3L, 3L, 3L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L,
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L,
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L,
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L,
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L,
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L,
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 2L, 2L, 2L, 2L, 2L,
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L,
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L,
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L,
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L,
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L,
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 3L,
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L,
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L,
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L,
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L,
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L,
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L,
3L, 3L, 3L), .Label = c("1", "2", "3"), class = "factor"), laser = structure(c(2L,
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L,
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L,
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L,
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L,
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L,
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L,
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L,
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L,
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L,
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L,
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L,
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L,
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L,
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L,
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L,
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L,
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L,
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L,
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 1L, 1L, 1L, 1L, 1L,
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L,
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L,
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L,
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L,
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L,
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L,
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L,
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L,
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L,
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L,
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L,
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L,
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L,
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L,
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L,
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L,
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L,
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L,
1L, 1L, 1L, 1L, 1L, 1L, 1L), .Label = c("2", "1"), class = "factor"),
    cdf = c(0, 0.0030303595531766276, 0.0040492276931370459,
    0.0053053014799513036, 0.0068250590587671696, 0.0086324595369397517,
    0.010748293558096941, 0.013189740125380425, 0.015970135218596779,
    0.019098939163028929, 0.022581876145450076, 0.026421211091123022,
    0.030616125862331599, 0.035163157476039958, 0.040056664629635426,
    0.04528929415252634, 0.050852425106046012, 0.056736574393250937,
    0.06293175340339173, 0.069427770100199918, 0.076214474933130189,
    0.083281951993284045, 0.090620659018493435, 0.098221521288238189,
    0.10607598527127143, 0.11417603823050576, 0.12251419997376972,
    0.13108349267164662, 0.13987739423184842, 0.14888978019169766,
    0.15811485851737481, 0.16754710111704527, 0.17718117530926045,
    0.18701187795325278, 0.19703407445208043, 0.20724264438625123,
    0.21763243512439009, 0.22819822438662513, 0.23893469240250337,
    0.2498364040049067, 0.26089780073119678, 0.27211320275964607,
    0.28347682029057059, 0.29498277378550641, 0.30662512230278349,
    0.31839789901290344, 0.33029515284147082, 0.34231099507046342,
    0.35443964962982727, 0.36667550573011298, 0.37901317142233915,
    0.39144752662247334, 0.40397377410353608, 0.41658748693684045,
    0.4292846508534609, 0.44206169999577305, 0.45491554453480215,
    0.46784358864028414, 0.48084373730513397, 0.49391439054324049,
    0.50705442349866969, 0.5202631510259047, 0.53354027532629178,
    0.54688581525861302, 0.56030001598661561, 0.573783237690626,
    0.58733582216379265, 0.60095793624865301, 0.61464939126252649,
    0.62840943783004466, 0.64223653591105412, 0.65612810030887547,
    0.67008022259761768, 0.68408737125063446, 0.69814207281940566,
    0.71223457833635762, 0.72635252072572698, 0.74048057092430197,
    0.75460010264482913, 0.76868887824257026, 0.78272077092075232,
    0.79666554143990453, 0.81048869042811411, 0.82415141010154058,
    0.83761066139122253, 0.85081940373588782, 0.86372700465241614,
    0.87627985406905562, 0.88842220368934699, 0.90009724374817446,
    0.911248417929726, 0.92182096167192507, 0.93176362969636539,
    0.94103055604508168, 0.94958316556426525, 0.95739203187721766,
    0.9644385565139254, 0.97071633079149577, 0.97623204036726874,
    1.0000000000000002, 0, 0.067410773914296146, 0.082082781253483214,
    0.098224049754717893, 0.11570185615659816, 0.13436053873714293,
    0.15403025367379791, 0.17453502593018813, 0.19569967133893629,
    0.21735537554150763, 0.23934388411315893, 0.26152037983141402,
    0.28375519983190595, 0.30593458451617794, 0.32796066063792578,
    0.34975085198716493, 0.37123689025615714, 0.39236357204022321,
    0.41308737980778987, 0.43337505786536135, 0.45320221044818976,
    0.47255196881745914, 0.49141375778231067, 0.50978217917011581,
    0.52765602002621725, 0.54503738624061571, 0.56193095737411636,
    0.57834335522819791, 0.59428261676835936, 0.60975776103332113,
    0.62477843937044286, 0.6393546585181864, 0.65349656654692034,
    0.66721429234919716, 0.68051783015257306, 0.69341696135090136,
    0.70592120677243897, 0.7180398032983385, 0.72978169949701732,
    0.74115556563983531, 0.75216981410771944, 0.76283262678646013,
    0.77315198658183937, 0.7831357106671446, 0.79279148350805884,
    0.80212688809656085, 0.81114943416947949, 0.81986658249158528,
    0.82828576455034963, 0.83641439724226418, 0.84425989233132326,
    0.85182966063131649, 0.85913111100725681, 0.86617164441003458,
    0.87295864325468631, 0.87949945652904504, 0.88580138107870632,
    0.89180456004878816, 0.89771735557656818, 0.90334552657064648,
    0.90876299499479585, 0.91397641838278143, 0.91899223888807235,
    0.92381665290371295, 0.92845558138464879, 0.93291464151490155,
    0.93719912038668185, 0.94131395138763252, 0.94526369402499821,
    0.94905251794968282, 0.95268419197586807, 0.95616207891880978,
    0.95948913708903005, 0.96266792927834233, 0.96570064004368694,
    0.96858910202908188, 0.97133483195370796, 0.97393907672449076,
    0.97640286989419878, 0.97872709766600186, 0.98091257819222355,
    0.98296014260238063, 0.98487073036173989, 0.9866454862107813,
    0.98828586011390218, 0.98979370603687333, 0.99117137546371037,
    0.99242180172503902, 0.99354856556493132, 0.99455594920203261,
    0.99544895758603136, 0.99623331542369287, 0.99691543360037516,
    0.99750232867033595, 0.99800156594654232, 0.99842106133050912,
    0.9987689922298677, 0.99905360076598471, 0.99928302471197672,
    1.0000000000000002, 0, 0.042713552752672634, 0.052964381830369361,
    0.064495922848309004, 0.077253853313680246, 0.091159178625131107,
    0.10611368899911769, 0.12200563198071029, 0.13871515616066338,
    0.15611920919472469, 0.17409569735764499, 0.19252682090275697,
    0.2113015828906622, 0.23031752775590547, 0.24948180200873324,
    0.26871164720140756, 0.28793443923999901, 0.30708738265298113,
    0.32611695722505335, 0.34497820033368665, 0.3636338934475925,
    0.38205370692420343, 0.40021334428646366, 0.41809371596603756,
    0.43568016318470559, 0.45296174514073106, 0.46993059680056476,
    0.48658136014714948, 0.5029106884703255, 0.51891682097833902,
    0.53459922346055, 0.54995828976653716, 0.56499509834145112,
    0.57971121785484026, 0.59410855598808177, 0.60818924563283705,
    0.62195556304590949, 0.63540987286478556, 0.64855459528424664,
    0.66139219110742875, 0.67392516080052589, 0.68615605408950964,
    0.69808748703356149, 0.70972216388939591, 0.72106290144078689,
    0.73211265380686219, 0.74287453606015941, 0.75335184528042953,
    0.76354807794244495, 0.77346694278539818, 0.78311236853793431,
    0.79248850607662102, 0.80159972477718322, 0.81045060297783023,
    0.8190459126134978, 0.82739059820016481, 0.83548975045129614,
    0.84334857489590309, 0.85097235594208021, 0.85836641689374782,
    0.86553607648453434, 0.8724866025442376, 0.87922316346307905,
    0.88575077816996339, 0.89200706112374739, 0.89819819305482662,
    0.90412682864394922, 0.90986409164332127, 0.91541350898359652,
    0.92077817474611867, 0.92596071536185731, 0.93096326167854737,
    0.93578742936295223, 0.94043430918819426, 0.94490446881287737,
    0.94919796767472608, 0.95331438657939005, 0.95725287344513943,
    0.96101220644492769, 0.96459087544654798, 0.96798718216843027,
    0.97119935882565256, 0.97422570422750188, 0.9770647353047156,
    0.97971535020107103, 0.98217700274868491, 0.9844498720059689,
    0.98653503396930242, 0.98843461671518273, 0.99015193422293502,
    0.99169158878477748, 0.99305953322883711, 0.99426308094241078,
    0.99531086817051884, 0.99621274967508378, 0.99697964053831845,
    0.99762328846409742, 0.99815606252444322, 0.99859057057481859,
    1.0000000000000002, 0, 0.019025837765481321, 0.024257815928250097,
    0.03034627350689497, 0.037307227271984482, 0.045139384463306106,
    0.053825445876648455, 0.063334094448268485, 0.073622414355032847,
    0.084638503203525742, 0.096324077162512151, 0.10861691568652958,
    0.12145304067156146, 0.13476856891781941, 0.1485012134953268,
    0.1625914377965014, 0.17698328586160117, 0.19162492495129271,
    0.20646894268651036, 0.22147244284035733, 0.23659698240924995,
    0.25180838906670616, 0.26707649342949596, 0.28237480542404075,
    0.29768015890595712, 0.31297234386318068, 0.32823374119842019,
    0.3434489713199555, 0.35860456458586371, 0.37368865901871479,
    0.38869072858459508, 0.40360134364951972, 0.41841196392248792,
    0.43311476320480274, 0.44770248453263128, 0.46216832377415884,
    0.4765058393817071, 0.49070888576778487, 0.50477156764385323,
    0.51868821260900044, 0.53245335928499571, 0.54606175835054449,
    0.55950838392030411, 0.57278845283505675, 0.58589744957185685,
    0.59883115464175174, 0.61158567451346324, 0.62415747128055243,
    0.63654339047373443, 0.64874068560626874, 0.6607470382259597,
    0.67256057242987799, 0.6841798629752377, 0.69560393629015449,
    0.70683226384982134, 0.71786474753602181, 0.72870169674046592,
    0.73934379710534026, 0.7497920709185929, 0.76004782929828718,
    0.77011261641210149, 0.77998814608754874, 0.78967623127923936,
    0.79917870697557436, 0.8084973472530822, 0.81763377732702724,
    0.8265893816067732, 0.83536520894838984, 0.84396187650936039,
    0.85237947385417878, 0.86061746923690674, 0.86867462029693232,
    0.87654889174386652, 0.88423738296941357, 0.89173626889593838,
    0.89904075773494851, 0.90614506965828345, 0.91304244064664131,
    0.91972515593139481, 0.92618461743453939, 0.93241144937767451,
    0.93839564570724743, 0.94412676210011648, 0.94959415400412439,
    0.95478726037751593, 0.95969593048625845, 0.96431078830383066,
    0.96862362579087924, 0.97262781273860366, 0.97631870715015012,
    0.97969404590607478, 0.98275429746052023, 0.98550294160214258,
    0.98794666572152923, 0.99009544406148642, 0.9919624854689344,
    0.99356403702172236, 0.99491903795849335, 0.99604864200639187,
    1.0000000000000002, 0, 0.011762649815007674, 0.015213009849163817,
    0.019298645745041688, 0.024049679883259324, 0.029484275579229866,
    0.035608711739192084, 0.042418067863455283, 0.049897377676773172,
    0.05802309784043802, 0.066764744765181294, 0.076086571583854984,
    0.085949183149629502, 0.09631101477749035, 0.10712962693039493,
    0.11836279105001417, 0.12996936022523131, 0.14190993217699413,
    0.15414732145139223, 0.1666468633893792, 0.17937657513608374,
    0.19230719941667973, 0.20541215571483296, 0.2186674214079298,
    0.23205136278401295, 0.24554453301779935, 0.259129451347474,
    0.27279037501959114, 0.28651307314351715, 0.30028460946089008,
    0.31409313919919291, 0.32792772363071443, 0.34177816467556876,
    0.35563486084054652, 0.3694886849429137, 0.38333088339919602,
    0.3971529963352588, 0.41094679737072343, 0.42470425162661407,
    0.43841749028234189, 0.45207879985215188, 0.4656806242502759,
    0.47921557765887868, 0.49267646619601652, 0.50605631639621973,
    0.51934840855907505, 0.53254631308714262, 0.54564392802007056,
    0.55863551607353856, 0.57151573960657653, 0.58427969206588071,
    0.59692292458814955, 0.60944146657848219, 0.62183183922203444,
    0.63409106102519575, 0.64621664461971595, 0.65820658419712241,
    0.67005933307067245, 0.68177377098792868, 0.69334916093951937,
    0.70478509533034184, 0.71608143150086334, 0.72723821671169198,
    0.73825560283865432, 0.74913375117351955, 0.75987272789346227,
    0.77047239095725994, 0.78093226941553462, 0.79125143639372997,
    0.80142837732745897, 0.81146085540706037, 0.82134577662684161,
    0.83107905733719245, 0.840655497763375, 0.85006866557683858,
    0.85931079426975576, 0.86837270176816506, 0.87724373538929756,
    0.88591174985603582, 0.89436312556199915, 0.9025828345530309,
    0.91055456165606252, 0.91826088772971759, 0.92568354100647066,
    0.93280372081432794, 0.93960249548712849, 0.94606127290496589,
    0.9521623378102676, 0.95788944486236316, 0.96322845047906025,
    0.96816796016766171, 0.97269996173898166, 0.97682040917282476,
    0.98052971706282366, 0.98383312877352691, 0.98674090821403793,
    0.98926833556294835, 0.99143547287540257, 0.99326669536190315,
    1.0000000000000002, 0, 0.016379425894658069, 0.020980288144023203,
    0.026364467348958736, 0.032553973340516497, 0.039555229110835886,
    0.047359904382965848, 0.055946426483212582, 0.065281951789566264,
    0.075324586853625863, 0.08602567411696932, 0.097331994124552998,
    0.1091877768248922, 0.12153645349407013, 0.13432211463297961,
    0.14749066622293147, 0.16099069665368829, 0.1747740799889278,
    0.18879634900763953, 0.20301687483627101, 0.2173988901299691,
    0.23190939070636757, 0.24651894713381359, 0.26120145368567116,
    0.27593383777152541, 0.29069574876956733, 0.30546924131449144,
    0.32023846465164058, 0.33498936669436691, 0.34970941891648172,
    0.36438736614623779, 0.37901300365809981, 0.3935769826326061,
    0.4080706440211242, 0.42248588006238275, 0.4368150221074043,
    0.45105075298075331, 0.46518604180676421, 0.47921409903318168,
    0.49312834927006749, 0.50692241951190875, 0.52059014031222917,
    0.53412555752215674, 0.5475229522791456, 0.56077686703283636,
    0.57388213551659417, 0.58683391471118607, 0.59962771699750794,
    0.61225944085486872, 0.6247253986269542, 0.63702234004634006,
    0.64914747037764275, 0.66109846220667112, 0.67287346006616222,
    0.68447107724611223, 0.69589038428708994, 0.70713088879748065,
    0.71819250637023846, 0.72907552250197361, 0.73978054553830919,
    0.75030845078644559, 0.76066031605152296, 0.77083734897116563,
    0.78084080664673305, 0.79067190820504329, 0.80033174107592875,
    0.80982116194445486, 0.81914069353750563, 0.82829041863794306,
    0.83726987299013456, 0.84607793907143236, 0.85471274305624667,
    0.86317155769085097, 0.87145071422211662, 0.87954552697103483,
    0.88745023459451633, 0.89515796251105206, 0.90266071134247328,
    0.90994937649977836, 0.91701380415921241, 0.92384288876770615,
    0.93042471680708128, 0.93674676075040819, 0.94279612587712724,
    0.94855985079972605, 0.95402526013760924, 0.95918036473290058,
    0.96401430117231268, 0.9685177982678701, 0.97268365376095445,
    0.97650720016249348, 0.97998673405023595, 0.98312388519623872,
    0.98592388579011148, 0.98839572568874479, 0.99056007542693814,
    0.99240954662259051, 0.99398752887678909, 0.99530853297233224,
    1.0000000000000002)), out.attrs = list(dim = c(vas = 100L,
time = 3L, laser = 2L), dimnames = list(vas = c("vas=0.0000",
"vas=0.0101", "vas=0.0202", "vas=0.0303", "vas=0.0404", "vas=0.0505",
"vas=0.0606", "vas=0.0707", "vas=0.0808", "vas=0.0909", "vas=0.1010",
"vas=0.1111", "vas=0.1212", "vas=0.1313", "vas=0.1414", "vas=0.1515",
"vas=0.1616", "vas=0.1717", "vas=0.1818", "vas=0.1919", "vas=0.2020",
"vas=0.2121", "vas=0.2222", "vas=0.2323", "vas=0.2424", "vas=0.2525",
"vas=0.2626", "vas=0.2727", "vas=0.2828", "vas=0.2929", "vas=0.3030",
"vas=0.3131", "vas=0.3232", "vas=0.3333", "vas=0.3434", "vas=0.3535",
"vas=0.3636", "vas=0.3737", "vas=0.3838", "vas=0.3939", "vas=0.4040",
"vas=0.4141", "vas=0.4242", "vas=0.4343", "vas=0.4444", "vas=0.4545",
"vas=0.4646", "vas=0.4747", "vas=0.4848", "vas=0.4949", "vas=0.5051",
"vas=0.5152", "vas=0.5253", "vas=0.5354", "vas=0.5455", "vas=0.5556",
"vas=0.5657", "vas=0.5758", "vas=0.5859", "vas=0.5960", "vas=0.6061",
"vas=0.6162", "vas=0.6263", "vas=0.6364", "vas=0.6465", "vas=0.6566",
"vas=0.6667", "vas=0.6768", "vas=0.6869", "vas=0.6970", "vas=0.7071",
"vas=0.7172", "vas=0.7273", "vas=0.7374", "vas=0.7475", "vas=0.7576",
"vas=0.7677", "vas=0.7778", "vas=0.7879", "vas=0.7980", "vas=0.8081",
"vas=0.8182", "vas=0.8283", "vas=0.8384", "vas=0.8485", "vas=0.8586",
"vas=0.8687", "vas=0.8788", "vas=0.8889", "vas=0.8990", "vas=0.9091",
"vas=0.9192", "vas=0.9293", "vas=0.9394", "vas=0.9495", "vas=0.9596",
"vas=0.9697", "vas=0.9798", "vas=0.9899", "vas=1.0000"), time = c("time=1",
"time=2", "time=3"), laser = c("laser=1", "laser=2"))), row.names = c("1",
"2", "3", "4", "5", "6", "7", "8", "9", "10", "11", "12", "13",
"14", "15", "16", "17", "18", "19", "20", "21", "22", "23", "24",
"25", "26", "27", "28", "29", "30", "31", "32", "33", "34", "35",
"36", "37", "38", "39", "40", "41", "42", "43", "44", "45", "46",
"47", "48", "49", "50", "51", "52", "53", "54", "55", "56", "57",
"58", "59", "60", "61", "62", "63", "64", "65", "66", "67", "68",
"69", "70", "71", "72", "73", "74", "75", "76", "77", "78", "79",
"80", "81", "82", "83", "84", "85", "86", "87", "88", "89", "90",
"91", "92", "93", "94", "95", "96", "97", "98", "99", "100",
"101", "102", "103", "104", "105", "106", "107", "108", "109",
"110", "111", "112", "113", "114", "115", "116", "117", "118",
"119", "120", "121", "122", "123", "124", "125", "126", "127",
"128", "129", "130", "131", "132", "133", "134", "135", "136",
"137", "138", "139", "140", "141", "142", "143", "144", "145",
"146", "147", "148", "149", "150", "151", "152", "153", "154",
"155", "156", "157", "158", "159", "160", "161", "162", "163",
"164", "165", "166", "167", "168", "169", "170", "171", "172",
"173", "174", "175", "176", "177", "178", "179", "180", "181",
"182", "183", "184", "185", "186", "187", "188", "189", "190",
"191", "192", "193", "194", "195", "196", "197", "198", "199",
"200", "201", "202", "203", "204", "205", "206", "207", "208",
"209", "210", "211", "212", "213", "214", "215", "216", "217",
"218", "219", "220", "221", "222", "223", "224", "225", "226",
"227", "228", "229", "230", "231", "232", "233", "234", "235",
"236", "237", "238", "239", "240", "241", "242", "243", "244",
"245", "246", "247", "248", "249", "250", "251", "252", "253",
"254", "255", "256", "257", "258", "259", "260", "261", "262",
"263", "264", "265", "266", "267", "268", "269", "270", "271",
"272", "273", "274", "275", "276", "277", "278", "279", "280",
"281", "282", "283", "284", "285", "286", "287", "288", "289",
"290", "291", "292", "293", "294", "295", "296", "297", "298",
"299", "300", "301", "302", "303", "304", "305", "306", "307",
"308", "309", "310", "311", "312", "313", "314", "315", "316",
"317", "318", "319", "320", "321", "322", "323", "324", "325",
"326", "327", "328", "329", "330", "331", "332", "333", "334",
"335", "336", "337", "338", "339", "340", "341", "342", "343",
"344", "345", "346", "347", "348", "349", "350", "351", "352",
"353", "354", "355", "356", "357", "358", "359", "360", "361",
"362", "363", "364", "365", "366", "367", "368", "369", "370",
"371", "372", "373", "374", "375", "376", "377", "378", "379",
"380", "381", "382", "383", "384", "385", "386", "387", "388",
"389", "390", "391", "392", "393", "394", "395", "396", "397",
"398", "399", "400", "401", "402", "403", "404", "405", "406",
"407", "408", "409", "410", "411", "412", "413", "414", "415",
"416", "417", "418", "419", "420", "421", "422", "423", "424",
"425", "426", "427", "428", "429", "430", "431", "432", "433",
"434", "435", "436", "437", "438", "439", "440", "441", "442",
"443", "444", "445", "446", "447", "448", "449", "450", "451",
"452", "453", "454", "455", "456", "457", "458", "459", "460",
"461", "462", "463", "464", "465", "466", "467", "468", "469",
"470", "471", "472", "473", "474", "475", "476", "477", "478",
"479", "480", "481", "482", "483", "484", "485", "486", "487",
"488", "489", "490", "491", "492", "493", "494", "495", "496",
"497", "498", "499", "500", "501", "502", "503", "504", "505",
"506", "507", "508", "509", "510", "511", "512", "513", "514",
"515", "516", "517", "518", "519", "520", "521", "522", "523",
"524", "525", "526", "527", "528", "529", "530", "531", "532",
"533", "534", "535", "536", "537", "538", "539", "540", "541",
"542", "543", "544", "545", "546", "547", "548", "549", "550",
"551", "552", "553", "554", "555", "556", "557", "558", "559",
"560", "561", "562", "563", "564", "565", "566", "567", "568",
"569", "570", "571", "572", "573", "574", "575", "576", "577",
"578", "579", "580", "581", "582", "583", "584", "585", "586",
"587", "588", "589", "590", "591", "592", "593", "594", "595",
"596", "597", "598", "599", "600"), class = "data.frame")

## ----neckpain-margdist,echo=FALSE,fig.width=7,fig.height=3--------------------
dat <- within(mp_colr, {
  trt <- factor(ifelse(laser == 1, "Active", "Placebo"), levels = c("Active", "Placebo"))
  t   <- factor(ifelse(time == 1, "Baseline",
                ifelse(time == 2, "Week 7", "Week 12")),
                levels = c("Baseline", "Week 7", "Week 12"))
})
dat2 <- within(neck_pain, {
  trt <- factor(ifelse(laser == 1, "Active", "Placebo"), levels = c("Active", "Placebo"))
  t   <- factor(ifelse(time == 1, "Baseline",
                ifelse(time == 2, "Week 7", "Week 12")),
                levels = c("Baseline", "Week 7", "Week 12"))
})
ggplot() +
  facet_wrap(~ t) +
  geom_line(data = dat, aes(vas, cdf, color = trt)) +
  stat_ecdf(aes(x = vas, color = laser, group = laser), data = dat2, geom = "step",
            size = 0.3, alpha = 0.5) +
  xlab("Pain level (VAS)") +
  ylab("Marginal distribution") +
  ggth +
  theme(legend.position = "bottom") +
  scale_color_manual(
    values = c(rgb(0, 84, 150, maxColorValue = 255),
               rgb(202, 108, 24, maxColorValue = 255),
               rgb(0, 84, 150, maxColorValue = 255),
               rgb(202, 108, 24, maxColorValue = 255)),
    breaks = c("Active", "Placebo"))

## ----cache=TRUE---------------------------------------------------------------
data("eortc", package = "coxme")
eortc$trt <- factor(eortc$trt, levels = c(0, 1))
eortc_cp <- CoxphME(Surv(y, uncens) ~ trt + (1 | center/trt), data = eortc,
                    log_first = TRUE, order = 10)

## -----------------------------------------------------------------------------
exp(confint(eortc_cp, parm = "trt1", estimate = TRUE))

## ----cache=TRUE---------------------------------------------------------------
exp(confint(eortc_cp, pargroup = "ranef", type = "profile", estimate = TRUE,
            ncpus = 2, parallel = "multicore"))

## ----cache=TRUE, warning=FALSE------------------------------------------------
eortc_cp2 <- CoxphME(Surv(y, uncens) | 0 + trt ~ 0 + (1 | center/trt), data = eortc,
                     log_first = TRUE, order = 10)
tr <- trafo(eortc_cp2, confidence = "interval")

## ----eortc-trafo, echo=FALSE, fig.width=4.5, fig.height=4, warning=FALSE, out.width="0.75\\linewidth"----
x <- log(as.numeric(rownames(tr[[1]]))[-1])
par(cex = 0.8, mfrow = c(1, 1))
plot(0, type = "n", xlim = range(x), ylim = range(tr[[1]], tr[[2]], na.rm = TRUE),
     xlab = "log-time", ylab = expression(h(y)), cex.axis = 0.8, cex.lab = 0.8)
polygon(c(x, rev(x)), c(tr[[1]][-1, 2], rev(tr[[1]][-1, 3])), border = NA,
        col = mycolors(1, "fill"))
lines(x, tr[[1]][-1, 1], lwd = 2, col = mycolors(1))
polygon(c(x, rev(x)), c(tr[[2]][-1, 2], rev(tr[[2]][-1, 3])), border = NA,
        col = mycolors(2, "fill"))
lines(x, tr[[2]][-1, 1], lwd = 2, col = mycolors(2))
rug(log(eortc$y), col = rgb(.1, .1, .1, .1))
legend("topleft", c("treatment", "control"),
       col = c(mycolors(1), mycolors(2)), lwd = 2, bty = "n", cex = 0.8)

## -----------------------------------------------------------------------------
eortc_w <- SurvregME(Surv(y, uncens) ~ trt + (1 | center/trt), data = eortc,
                     dist = "weibull")

## ----cache=TRUE---------------------------------------------------------------
## --- CoxphME
c(coef = coef(eortc_cp), se = sqrt(diag(vcov(eortc_cp, pargroup = "shift"))))
VarCorr(eortc_cp)
## --- SurvregME
c(coef = -coef(eortc_w), se = sqrt(diag(vcov(eortc_w, pargroup = "shift"))))
VarCorr(eortc_w)

## -----------------------------------------------------------------------------
c(logLik(eortc_cp), logLik(eortc_w))

## ----message=FALSE, cache=TRUE------------------------------------------------
library("coxme")
eortc_cm <- coxme(Surv(y, uncens) ~ trt + (1 | center/trt), data = eortc)
summary(eortc_cm)

## ----soup-data, echo=FALSE, fig.width = 6, fig.height=3.5---------------------
data("soup", package = "ordinal")
dat <- within(soup, {
  prod <- factor(ifelse(PROD == "Ref", "Reference", "Test"), levels = c("Reference", "Test"))
})
ggplot(data = dat, aes(x = prod, fill = SURENESS)) +
  facet_wrap(~ SOUPFREQ) +
  geom_bar(position = "fill") +
  scale_fill_grey() +
  xlab("Product") +
  ylab("Proportion") +
  ggth +
  theme(axis.ticks.x = element_blank(), legend.position = "bottom",
        legend.title = element_text()) +
  guides(fill = guide_legend(nrow = 1, title = "Sureness"))

## ----cache=TRUE---------------------------------------------------------------
soup_pr <- PolrME(SURENESS ~ PROD + SOUPFREQ + (1 | RESP/PROD),
                  data = soup, method = "probit")
logLik(soup_pr)

## ----cache=TRUE, message=FALSE------------------------------------------------
library("ordinal")
soup_or <- clmm(SURENESS ~ PROD + SOUPFREQ + (1 | RESP/PROD), data = soup,
                link = "probit")
logLik(soup_or)

## -----------------------------------------------------------------------------
max(abs(coef(soup_or) - coef(soup_pr, with_baseline = TRUE)))

## ----cache=TRUE---------------------------------------------------------------
soup_pr2 <- PolrME(SURENESS | SOUPFREQ ~ PROD + (1 | RESP/PROD),
                   data = soup, method = "probit")
logLik(soup_pr2)

## -----------------------------------------------------------------------------
anova(soup_pr, soup_pr2)

## ----include=FALSE------------------------------------------------------------
## NOTE: reset user's options and pars (as per CRAN submission request)
options(oldopt)
par(oldpar)

